/**
  *********************************************************************************
  *
  * @file    usbd_hid_keyb.c
  * @brief   USB HID keyboard device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbd_hid_keyb.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_Keyb Keyboard
  * @brief Device Keyboard driver
  * @{
  */
/** @defgroup Device_Keyb_Private_Variables Private Variables
  * @{
  */
/**
  * @brief Configuration descriptor header
  */
static uint8_t __keyb_desc[] = {
	9,                       /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION, /**< Type of the descriptor */
	USBShort(34),            /**< Total size of the descriptor */
	1,                       /**< Number of the inerfaces */
	1,                       /**< Unique value for this configuration */
	5,                       /**< Index of the string */
	USB_CONF_ATTR_SELF_PWR,  /**< Type of the power */
	250,                     /**< Maximum power in 2mA increments */
};

/**
  * @brief HID interface descriptor
  */
static uint8_t __hid_interface[HID_INTERFACE_SIZE] = {
	9,                     /**< Size of the interface */
	USB_DTYPE_INTERFACE,   /**< Type of the interface */
	0,                     /**< Index of the interface */
	0,                     /**< Alternate setting */
	1,                     /**< Number of endpoints in the interface */
	USB_CLASS_HID,         /**< Interface class */
	USB_HID_SCLASS_BOOT,   /**< Interface sub-class */
	USB_HID_PROTOCOL_KEYB, /**< Interface protocol */
	4,                     /**< Index of the string */
};

/**
  * @brief HID IN endpoint descriptor
  */
static const uint8_t __hid_in_ep[HID_IN_ENDPOINT_SIZE] = {
	7,                                 /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                /**< Type is an endpoint */
	USB_EP_DESC_IN | KEYB_IN_ENDPOINT, /**< Endpoint direction */
	USB_EP_ATTR_INT,                   /**< Endpoint type */
	USBShort(64),                      /**< Maximum packet size */
	16,                                /**< Polling interval */
};

/**
  * @brief HID OUT endpoint descriptor
  */
static const uint8_t __hid_out_ep[HID_OUT_ENDPOINT_SIZE] = {
	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_OUT | KEYB_OUT_ENDPOINT, /**< Endpoint direction */
	USB_EP_ATTR_INT,                     /**< Endpoint type */
	USBShort(64),                        /**< Maximum packet size */
	16,                                  /**< Polling interval */
};

/**
  * @brief Keyboad report descriptor
  */
static const uint8_t __keyb_report_desc[] = {
	USAGE_PAGE(USB_HID_GENERIC_DESKTOP),
	USAGE(USB_HID_KEYBOARD),
	COLLECTION(USB_HID_APPLICATION),

		/* Modifier keys.
		 * 8 - 1 bit values indicating the modifier keys (ctrl, shift...)
		 */
		REPORT_SIZE(1),
		REPORT_COUNT(8),
		USAGE_PAGE(USB_HID_USAGE_KEYCODES),
		USAGE_MIN(224),
		USAGE_MAX(231),
		LOGICAL_MIN(0),
		LOGICAL_MAX(1),
		INPUT(USB_HID_INPUT_DATA | USB_HID_INPUT_VARIABLE | USB_HID_INPUT_ABS),

		/* One byte of rsvd data required by HID spec. */
		REPORT_COUNT(1),
		REPORT_SIZE(8),
		INPUT(USB_HID_INPUT_CONSTANT),

		/* Keyboard LEDs, 5 - 1 bit values. */
		REPORT_COUNT(5),
		REPORT_SIZE(1),
		USAGE_PAGE(USB_HID_USAGE_LEDS),
		USAGE_MIN(1),
		USAGE_MAX(5),
		OUTPUT(USB_HID_OUTPUT_DATA | USB_HID_OUTPUT_VARIABLE | USB_HID_OUTPUT_ABS),
		/* 1 - 3 bit value to pad out to a full byte. */
		REPORT_COUNT(1),
		REPORT_SIZE(3),
		OUTPUT(USB_HID_OUTPUT_CONSTANT),

		/* The Key buffer.
		 * 6 - 8 bit values to store the current key state.
		 */
		REPORT_COUNT(6),
		REPORT_SIZE(8),
		LOGICAL_MIN(0),
		LOGICAL_MAX(101),
		USAGE_PAGE(USB_HID_USAGE_KEYCODES),
		USAGE_MIN(0),
		USAGE_MAX(101),
		INPUT(USB_HID_INPUT_DATA | USB_HID_INPUT_ARRAY),
	END_COLLECTION
};

/**
  * @brief HID descriptor
  */
static const hid_desc_t __keyb_hid_desc = {
	9,                 /**< Size of the endpoint descriptor */
	USB_HID_DTYPE_HID, /**< Type is an endpoint */
	0x111,             /**< Version 1.11 */
	0,                 /**< Country code */
	1,                 /**< Number of the descriptors */
	{
		{
			USB_HID_DTYPE_REPORT,      /**< Report descriptor */
			sizeof(__keyb_report_desc) /**< Size of thr report descriptor */
		}
	}
};

/**
  * @brief HID configuration section
  */
static const config_section_t __hid_config_sec = {
	sizeof(__keyb_desc),
	__keyb_desc
};

/**
  * @brief HID interface section
  */
static const config_section_t __hid_interface_sec = {
	sizeof(__hid_interface),
	__hid_interface
};

/**
  * @brief HID IN endpoint section
  */
static const config_section_t __hid_in_ep_sec = {
	sizeof(__hid_in_ep),
	__hid_in_ep
};

/**
  * @brief HID OUT endpoint section
  */
static const config_section_t __hid_out_ep_sec = {
	sizeof(__hid_out_ep),
	__hid_out_ep
};

/**
  * @brief HID descriptor section
  */
static config_section_t __hid_desc_sec = {
	sizeof(__keyb_hid_desc),
	(const uint8_t *)&__keyb_hid_desc
};

/**
  * @brief HID section
  */
static const config_section_t *__hid_sec[] = {
	&__hid_config_sec,
	&__hid_interface_sec,
	&__hid_desc_sec,
	&__hid_in_ep_sec,
	&__hid_out_ep_sec
};

/**
  * @brief Number of the section
  */
#define NUM_HID_SECTIONS	((sizeof(__hid_sec) / sizeof(__hid_sec[0])) - 1)

/**
  * @brief HID configuration header
  */
static config_head_t __hid_config_head = {
	NUM_HID_SECTIONS,
	__hid_sec
};

/**
  * @brief HID configuration descriptor
  */
static const config_head_t * const __hid_config_desc[] = {
	&__hid_config_head
};

/**
  * @brief Keyboad class descriptor
  */
static const uint8_t * const __keyb_class_desc[] = {
	__keyb_report_desc
};
/**
  * @}
  */

/** @defgroup Device_Keyb_Private_Functions Private Functions
  * @{
  */
static uint32_t hid_keyb_rx_handler(void *device, uint32_t event, uint32_t param, void *data);
static uint32_t hid_keyb_tx_handler(void *device, uint32_t event, uint32_t param, void *data);

/**
  * @brief  Handle HID device class event.
  * @param  device: Keyboard device.
  * @param  event: Event.
  * @param  param: Event-specific value.
  * @param  data: Event-specific data.
  * @retval Status.
  */
static uint32_t hid_keyb_rx_handler(void *device, uint32_t event, uint32_t param, void *data)
{
	hid_keyb_inst_t *inst;
	usbd_hid_keyb_dev_t *dev;

	assert_param(device);

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_keyb_dev_t *)device;
	inst = &dev->inst;

	/* Which event? */
	switch (event) {
	case USB_EVENT_CONNECTED:
		inst->config = 1;
		/* Send a CONNECTED event to the client */
		dev->cbk(dev->arg, USB_EVENT_CONNECTED, 0, NULL);
		break;

	case USB_EVENT_DISCONNECTED:
		inst->config = 0;
		/* Send a DISCONNECTED event to the client */
		dev->cbk(dev->arg, USB_EVENT_DISCONNECTED, 0, NULL);
		break;

	case USBD_HID_EVENT_IDLE_TIMEOUT:
	case USBD_HID_EVENT_GET_REPORT:
		/* Set the report pointer */
		*(uint8_t **)data = inst->report;
		/* Return the size of the report */
		return KEYB_IN_REPORT_SIZE;

	case USBD_HID_EVENT_REPORT_SENT:
		break;

	case USBD_HID_EVENT_GET_REPORT_BUFFER:
		/* Return a pointer to a buffer large enough to receive the report */
		if ((uint32_t)data == KEYB_OUT_REPORT_SIZE)
			return ((uint32_t)inst->data);
		else
			return 0;

	case USBD_HID_EVENT_SET_REPORT:
		/* If the LED state have changed, notify the information to the client */
		if (inst->led_state != inst->data[0]) {
			/* Records the new LED state */
			inst->led_state = inst->data[0];
			/* Send a SET_LEDS event to the client */
			dev->cbk(dev->arg, USBD_HID_KEYB_EVENT_SET_LEDS, inst->data[0], NULL);
		}
		break;

	case USBD_HID_EVENT_SET_PROTOCOL:
		/* Set the report protocol */
		inst->protocol = param;
		break;

	case USBD_HID_EVENT_GET_PROTOCOL:
		/* Get the report protocol */
		return inst->protocol;

	case USB_EVENT_ERROR:
	case USB_EVENT_SUSPEND:
	case USB_EVENT_RESUME:
	case USB_EVENT_LPM_RESUME:
	case USB_EVENT_LPM_SLEEP:
	case USB_EVENT_LPM_ERROR:
		return dev->cbk(dev->arg, event, param, data);

	default:
		break;
	}

	return 0;
}

/**
  * @brief  Handle HID device class transmit event.
  * @param  device: Keyboard device.
  * @param  event: Event.
  * @param  param: Event-specific value.
  * @param  data: Event-specific data.
  * @retval Status.
  */
static uint32_t hid_keyb_tx_handler(void *device, uint32_t event, uint32_t param, void *data)
{
	uint32_t len;
	hid_keyb_inst_t *inst;
	usbd_hid_keyb_dev_t *dev;
	usbd_hid_dev_t *_dev;

	assert_param(device);

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_keyb_dev_t *)device;
	inst = &dev->inst;
	_dev = &inst->dev;

	/* Which event? */
	switch (event) {
	case USB_EVENT_TX_COMPLETE:
		/* Do it have a pending report needing transmitted */
		if (inst->change) {
			/* Send the report immediately */
			len = usbd_hid_report_write((void *)_dev, inst->report, KEYB_IN_REPORT_SIZE, 1);

			/* Clear the change flag */
			if (len != 0)
				inst->change = 0;
		}
		else {
			/* It have nothing to send */
			inst->state = HID_KEYB_STATE_IDLE;
		}

		/* Send the TX_COMPLETE event to the client */
		dev->cbk(dev->arg, USB_EVENT_TX_COMPLETE, param, NULL);
		break;

	default:
		break;
	}

	return 0;
}

/**
  * @brief  Add the supplied usage code to the list.
  * @param  inst: Device instance.
  * @param  usage: HID usage code.
  * @retval Status.
  */
static uint8_t press_list_add(hid_keyb_inst_t *inst, uint8_t usage)
{
	uint32_t i;

	/* Check if the new one is already there */
	for (i = 0; i < (uint32_t)inst->key_cnt; ++i) {
		if (usage == inst->press[i])
			break;
	}

	/* Is already there */
	if (i < inst->key_cnt)
		return 1;

	if (inst->key_cnt < KEYB_MAX_CHARS_PER_REPORT) {
		/* Store the new key press in the list */
		inst->press[inst->key_cnt] = usage;
		inst->key_cnt++;

		return 1;
	}

	return 0;
}

/**
  * @brief  Delete the supplied usage code from the list.
  * @param  inst: Device instance.
  * @param  usage: HID usage code.
  * @retval Status.
  */
static uint8_t press_list_del(hid_keyb_inst_t *inst, uint8_t usage)
{
	uint32_t i, pos = 0;

	/* Find the usage code in the current list */
	for (i = 0; i < KEYB_MAX_CHARS_PER_REPORT; ++i) {
		if (inst->press[i] == usage) {
			pos = i;
			break;
		}
	}

	/* Not find */
	if (i == KEYB_MAX_CHARS_PER_REPORT)
		return 0;

	/* Remove the usage code */
	for (i = pos + 1; i < KEYB_MAX_CHARS_PER_REPORT; ++i)
		inst->press[i - 1] = inst->press[i];

	/* Clear the last entry in the array */
	inst->press[KEYB_MAX_CHARS_PER_REPORT - 1] = HID_KEYB_USAGE_RESERVED;
	inst->key_cnt--;

	return 1;
}
/**
  * @}
  */

/** @defgroup Device_Keyb_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes HID keyboard device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Keyboard device.
  * @retval Device structure.
  */
void *usbd_hid_keyb_init(uint32_t idx, usbd_hid_keyb_dev_t *dev)
{
	void *ret;
	usbd_hid_dev_t *_dev;
	config_desc_t *desc;

	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->cbk);

	/* Create a pointer to the device */
	_dev = &dev->inst.dev;
	/* Composite initialization */
	ret  = usbd_hid_keyb_init_comp(idx, dev, 0);
	/* Fill the configuration descriptor */
	desc = (config_desc_t *)__keyb_desc;
	desc->bmAttributes = dev->attr_pwr;
	desc->bMaxPower =  (uint8_t)(dev->max_power >> 1);

	if (ret) {
		/* Initialze device controler */
		usbd_hid_init(idx, _dev);
		return (void *)dev;
	}

	return NULL;
}

/**
  * @brief  Initializes HID keyboard device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Keyboard device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_hid_keyb_init_comp(uint32_t idx, usbd_hid_keyb_dev_t *dev, comp_entry_t *entry)
{
	uint32_t i;
	hid_keyb_inst_t *inst;
	usbd_hid_dev_t *_dev;

	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->cbk);

	/* Create a pointer to the device instance */
	inst = &dev->inst;
	/* Initialize the device instance information */
	inst->config      = 0;
	inst->protocol    = USB_HID_PROTOCOL_REPORT;
	inst->idle.dur4ms = 125;
	inst->idle.id     = 0;
	inst->idle.since  = 0;
	inst->idle.next   = 0;
	inst->led_state   = 0;
	inst->key_cnt     = 0;
	inst->state       = HID_KEYB_STATE_UNCONFIG;

	/* Initialize the press array */
	for (i = 0; i < KEYB_MAX_CHARS_PER_REPORT; ++i)
		inst->press[i] = HID_KEYB_USAGE_RESERVED;

	/* Initialize the device information */
	_dev = &inst->dev;
	_dev->vid             = dev->vid;
	_dev->pid             = dev->pid;
	_dev->max_power       = dev->max_power;
	_dev->attr_pwr        = dev->attr_pwr;
	_dev->sub_class       = USB_HID_SCLASS_BOOT;
	_dev->protocol        = USB_HID_PROTOCOL_KEYB;
	_dev->nr_input_report = 1;
	_dev->idle            = &inst->idle;
	_dev->rx_cbk          = hid_keyb_rx_handler;
	_dev->rx_arg          = (void *)dev;
	_dev->tx_cbk          = hid_keyb_tx_handler;
	_dev->tx_arg          = (void *)dev;
	_dev->use_oep         = 0,
	_dev->desc_hid        = &__keyb_hid_desc;
	_dev->desc_class      = __keyb_class_desc;
	_dev->desc_str        = dev->desc_str;;
	_dev->num_str         = dev->num_str;
	_dev->desc_config     = __hid_config_desc;

	/* Call the lower layer composite initialization */
	return usbd_hid_init_comp(idx, _dev, entry);
}

/**
  * @brief  Terminal the HID keyboard device.
  * @param  device: Keyboard device.
  * @retval None
  */
void usbd_hid_keyb_term(void *device)
{
	usbd_hid_keyb_dev_t *dev;
	usbd_hid_dev_t *_dev;

	assert_param(device);

	/* Create a pointer to the device */
	dev  = (usbd_hid_keyb_dev_t *)device;
	_dev = &dev->inst.dev;
	/* Clear the flag */
	dev->inst.config = 0;

	/* Call the lower layer function */
	usbd_hid_term(_dev);
	return;
}

/**
  * @brief  Set callback parameter.
  * @param  device: Keyboard device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_hid_keyb_set_param(void *device, void *arg)
{
	void *old;
	usbd_hid_keyb_dev_t *dev;

	assert_param(device);

	/* Create a pointer to the device */
	dev = (usbd_hid_keyb_dev_t *)device;
	/* Set the parameter */
	old = dev->arg;
	dev->arg = arg;

    return old;
}

/**
  * @brief  Reports a key state change to the USB host.
  * @param  device: Keyboard device.
  * @param  modify: States of each of the keyboard modifiers.
  * @param  usage: Usage code of the key.
  * @param  press: Pressed/Released.
  * @retval Status.
  */
uint32_t usbd_hid_keyb_state_change(void *device, uint8_t modify, uint8_t usage, uint8_t press)
{
	uint8_t ret = 1;
	uint32_t i, cnt;
	hid_keyb_inst_t *inst;
	usbd_hid_keyb_dev_t *dev;
	usbd_hid_dev_t *_dev;

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_keyb_dev_t *)device;
	_dev = &dev->inst.dev;
	inst = &dev->inst;

	/* Record the function key */
	inst->report[0] = modify;
	inst->report[1] = 0;

	if (usage != HID_KEYB_USAGE_RESERVED) {
		if (press) {
			/* Add the usage into the press array */
			ret = press_list_add(inst, usage);
		}
		else {
			/* Delete the usage from the press array */
			ret = press_list_del(inst, usage);

			if (!ret) {
				return KEYB_ERR_NOT_FOUND;
			}
		}

		/* Build the report from the press array */
		for (i = 0; i < KEYB_MAX_CHARS_PER_REPORT; ++i)
			inst->report[2 + i] = (ret ? inst->press[i] : HID_KEYB_USAGE_ROLLOVER);
	}

	/* Check the flag */
	if (!inst->config)
		return KEYB_ERR_NOT_CONFIGURED;

	/* If the transmitter is free, send the report */
	if (usbd_hid_tx_packet_avail((void *)_dev)) {
		/* Set the state */
		inst->state = HID_KEYB_STATE_SEND;
		/* Send the report to the host */
		cnt = usbd_hid_report_write((void *)_dev, inst->report, KEYB_IN_REPORT_SIZE, true);

		if (!cnt)
			return KEYB_ERR_TX_ERROR;
	}
	else {
		/* It can't send the report immediately, so set the
		 * flag in order to send next time the transmitter
		 * is free.
		 */
		inst->change = 1;
	}

	return (ret ? KEYB_SUCCESS : KEYB_ERR_TOO_MANY_KEYS);
}

/**
  * @brief  Set power status.
  * @param  device: Keyboard device.
  * @param  power: Status of the power.
  * @retval None
  */
void usbd_hid_keyb_power_status_set(void *device, uint8_t power)
{
	usbd_hid_keyb_dev_t *dev;
	usbd_hid_dev_t *_dev;

	assert_param(device);

	/* Create a pointer to the device */
	dev  = (usbd_hid_keyb_dev_t *)device;
	_dev = &dev->inst.dev;
	/* Set the status of the power */
	usbd_hid_power_status_set((void *)_dev, power);
}

/**
  * @brief  Requests a remote wake up to resume communication.
  * @param  device: Keyboard device.
  * @retval Status.
  */
int usbd_hid_keyb_remote_wakeup_req(void *device)
{
	usbd_hid_keyb_dev_t *dev;
	usbd_hid_dev_t *_dev;

	assert_param(device);

	/* Create a pointer to the device */
	dev  = (usbd_hid_keyb_dev_t *)device;
	_dev = &dev->inst.dev;

	/* Call the lower layer function */
	return usbd_hid_remote_wakeup_req((void *)_dev);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
