/**
  *********************************************************************************
  *
  * @file    usbd_hid_game.h
  * @brief   Header file for the HID game pad driver[Device].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_HID_GAME_H__
#define __USBD_HID_GAME_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_hid.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_Game
  * @{
  */
/**
  * @defgroup Device_Game_Public_Macros Public Macros
  * @{
  */
#define USBD_GAME_SUCCESS	0U
#define USBD_GAME_TX_ERROR	1U
#define USBD_GAME_NOT_CONFIG	2U
#define GAME_IN_ENDPOINT	HID_GAME_EP_IN
/**
  * @}
  */

/**
  * @defgroup Device_Game_Public_Types Public Types
  * @{
  */
/**
  * @brief HID game pad status
  */
typedef enum {
	HID_GAME_STATE_DISCONN = 0,	/**< Not configured */
	HID_GAME_STATE_IDLE    = 1,	/**< Idle */
	HID_GAME_STATE_SEND    = 2,	/**< Waiting on data to be sent */
} game_state_t;

/**
  * @brief Game pad instance information
  */
typedef struct {
	game_state_t state;	/**< Status */
	hid_report_idle_t idle;	/**< Report idle */
	usbd_hid_dev_t dev;	/**< HID device structure */
} hid_game_inst_t;

/**
  * @brief define operating parameters for the HID game device
  */
typedef struct {
	const uint16_t vid;			/**< Vendor ID */
	const uint16_t pid;			/**< Product ID */
	const uint16_t max_power;		/**< Maximum power consumption */
	const uint8_t attr_pwr;			/**< Attribute of the power */
	const usb_cbk cbk;			/**< Callback function */
	void *arg;				/**< Parameter */
	const uint8_t * const *desc_str;	/**< String descriptor */
	const uint32_t num_str;			/**< Number of the string descriptors */
	const uint8_t *desc_report;		/**< Report descriptor */
	const uint32_t report_size;		/**< Size of the report */
	hid_game_inst_t inst;			/**< Device instance */
} usbd_hid_game_dev_t;

/**
  * @brief define game report structure
  */
typedef struct {
	int8_t x;	/**< X position */
	int8_t y;	/**< Y position */
	int8_t z;	/**< Z position */
	uint8_t button;	/**< Map of button */
} PACKED game_report_t;
/**
  * @}
  */

/** @addtogroup Device_Game_Public_Functions
  * @{
  */
extern void *usbd_hid_game_init(uint32_t idx, usbd_hid_game_dev_t *dev);
extern void *usbd_hid_game_init_comp(uint32_t idx, usbd_hid_game_dev_t *dev, comp_entry_t *entry);
extern void usbd_hid_game_term(usbd_hid_game_dev_t *dev);
extern uint32_t usbd_hid_game_report_send(usbd_hid_game_dev_t *dev, void *report, uint32_t size);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
