/**
  *********************************************************************************
  *
  * @file    usbd_hid_game.c
  * @brief   USB HID game pad device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbd_hid_game.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_Game GamePad
  * @brief Device Game driver
  * @{
  */
/** @defgroup Device_Game_Private_Variables Private Variables
  * @{
  */
/**
  * @brief Configuration descriptor header
  */
static uint8_t __game_desc[] = {
	9,                       /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION, /**< Type of the descriptor */
	USBShort(24),            /**< Total size of the descriptor */
	1,                       /**< Number of the inerfaces */
	1,                       /**< Unique value for this configuration */
	5,                       /**< Index of the string */
	USB_CONF_ATTR_SELF_PWR,  /**< Type of the power */
	0,                     /**< Maximum power in 2mA increments */
};

/**
  * @brief Interface descriptor
  */
static uint8_t __hid_interface[HID_INTERFACE_SIZE] = {
	9,                   /**< Size of the interface */
	USB_DTYPE_INTERFACE, /**< Type of the interface */
	0,                   /**< Index of the interface */
	0,                   /**< Alternate setting */
	1,                   /**< Number of endpoints in the interface */
	USB_CLASS_HID,       /**< Interface class */
	0,                   /**< Interface sub-class */
	0,                   /**< Interface protocol */
	4,                   /**< Index of the string */
};

/**
  * @brief Endpoint descriptor
  */
static const uint8_t __hid_in_ep[HID_IN_ENDPOINT_SIZE] = {
	7,                                  /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                 /**< Type is an endpoint */
	USB_EP_DESC_IN | GAME_IN_ENDPOINT,  /**< Endpoint direction */
	USB_EP_ATTR_INT,                    /**< Endpoint type */
	USBShort(64),                       /**< Maximum packet size */
	1,                                  /**< Polling interval */
};

/**
  * @brief HID report structure
  */
static const uint8_t __game_report_desc[] = {
	USAGE_PAGE(USB_HID_GENERIC_DESKTOP),
	USAGE(USB_HID_JOYSTICK),
	COLLECTION(USB_HID_APPLICATION),

		USAGE_PAGE(USB_HID_GENERIC_DESKTOP),
		USAGE(USB_HID_POINTER),
		COLLECTION(USB_HID_PHYSICAL),

			USAGE(USB_HID_X),
			USAGE(USB_HID_Y),
			USAGE(USB_HID_Z),

			REPORT_SIZE(8),
			REPORT_COUNT(3),
			INPUT(USB_HID_INPUT_DATA | USB_HID_INPUT_VARIABLE | USB_HID_INPUT_ABS),

			USAGE_PAGE(USB_HID_BUTTONS),
			USAGE_MIN(1),
			USAGE_MAX(8),
			LOGICAL_MIN(0),
			LOGICAL_MAX(1),
			PHYSICAL_MIN(0),
			PHYSICAL_MAX(1),

			REPORT_SIZE(1),
			REPORT_COUNT(8),
			INPUT(USB_HID_INPUT_DATA | USB_HID_INPUT_VARIABLE | USB_HID_INPUT_ABS),

		END_COLLECTION,
	END_COLLECTION
};

/**
  * @brief HID descriptor
  */
static hid_desc_t __game_hid_desc = {
	9,                                             /**< Length of the descriptor */
	USB_HID_DTYPE_HID,                             /**< Type of the descriptor */
	0x111,                                         /**< Version 1.11 in BCD code */
	0,                                             /**< Country code */
	1,                                             /**< Number of descriptor */
	{
		{
			USB_HID_DTYPE_REPORT,          /**< Report descriptor */
			sizeof(__game_report_desc)     /**< Size of the report descriptor */
		}
	}
};

/**
  * @brief HID configuration section
  */
static const config_section_t __hid_config_sec = {
	sizeof(__game_desc),
	__game_desc
};

/**
  * @brief HID interface section
  */
static const config_section_t hid_interface_sec = {
	sizeof(__hid_interface),
	__hid_interface
};

/**
  * @brief HID IN endpoint section
  */
static const config_section_t __hid_in_ep_sec = {
	sizeof(__hid_in_ep),
	__hid_in_ep
};

/**
  * @brief HID IN endpoint section
  */
static config_section_t __hid_desc_sec = {
	sizeof(__game_hid_desc),
	(const uint8_t *)&__game_hid_desc
};

/**
  * @brief HID section
  */
static const config_section_t *__hid_sec[] = {
	&__hid_config_sec,
	&hid_interface_sec,
	&__hid_desc_sec,
	&__hid_in_ep_sec,
};

/**
  * @brief Number of the section
  */
#define NUM_HID_SECTIONS	(sizeof(__hid_sec) / sizeof(config_section_t *))

/**
  * @brief HID configuration header
  */
static config_head_t __hid_config_head = {
	NUM_HID_SECTIONS,
	__hid_sec
};

/**
  * @brief HID configuration descriptor
  */
static const config_head_t * const __hid_config_desc[] = {
	&__hid_config_head
};

/**
  * @brief Game configuration descriptor
  */
static const uint8_t *__game_class_desc[] = {
	__game_report_desc
};
/**
  * @}
  */

/** @defgroup Device_Gmae_Private_Functions Private Functions
  * @{
  */
/**
  * @brief  handle HID gamepad transmit channel event.
  * @param  device: Gamepad device.
  * @param  event: Event.
  * @param  param: Event-specific value.
  * @param  data: Event-specific data.
  * @retval Status.
  */
static uint32_t hid_game_tx_handler(void *device, uint32_t event, uint32_t param, void *data)
{
	hid_game_inst_t *inst;
	usbd_hid_game_dev_t *dev;

	assert_param(device);

	/* Create a pointer to the device instance */
	dev = (usbd_hid_game_dev_t *)device;
	inst = &dev->inst;

	/* Which event? */
	switch (event) {
	case USB_EVENT_TX_COMPLETE:
		/* TX complete, Set STATE to idle */
		inst->state = HID_GAME_STATE_IDLE;
		/* Send a TX_COMPLETE event to the client */
		dev->cbk(dev->arg, USB_EVENT_TX_COMPLETE, param, NULL);
		break;

	default:
		break;
	}

	return 0;
}

/**
  * @brief  handle HID gamepad receive channel event.
  * @param  device: Gamepad device.
  * @param  event: Event.
  * @param  param: Event-specific value.
  * @param  data: Event-specific data.
  * @retval Status.
  */
static uint32_t hid_game_rx_handler(void *device, uint32_t event, uint32_t param, void *data)
{
	uint32_t ret = 0;
	hid_game_inst_t *inst;
	usbd_hid_game_dev_t *dev;

	assert_param(device);

	/* Create a pointer to the device instance */
	dev = (usbd_hid_game_dev_t *)device;
	inst = &dev->inst;

	/* Which event? */
	switch (event) {
	case USB_EVENT_CONNECTED:
		/* TX complete, Set STATE to idle */
		inst->state = HID_GAME_STATE_IDLE;
		/* Send a CONNECTED event to the client */
		dev->cbk(dev->arg, USB_EVENT_CONNECTED, 0, NULL);

		break;
	case USB_EVENT_DISCONNECTED:
		/* TX complete, Set STATE to disconnected */
		inst->state = HID_GAME_STATE_DISCONN;
		/* Send a DISCONNECTED event to the client */
		ret = dev->cbk(dev->arg, USB_EVENT_DISCONNECTED, 0, NULL);

		break;
	case USBD_HID_EVENT_IDLE_TIMEOUT:
		/* Get the pointer to the idle report structure */
		*(void **)data = (void *)&inst->idle;
		/* Get the size of the report */
		ret = sizeof(inst->idle);

		break;
	case USBD_HID_EVENT_GET_REPORT:
		/* If it's an IN request, then send it to the client */
		if (param == USB_HID_REPORT_IN)
			ret = dev->cbk(dev->arg, USBD_HID_EVENT_GET_REPORT, 0, data);

		break;

	case USBD_HID_EVENT_REPORT_SENT:
		break;

	case USB_EVENT_ERROR:
	case USB_EVENT_SUSPEND:
	case USB_EVENT_RESUME:
	case USB_EVENT_LPM_RESUME:
	case USB_EVENT_LPM_SLEEP:
	case USB_EVENT_LPM_ERROR:
		ret = dev->cbk(dev->arg, event, param, data);
		break;

	case USBD_HID_EVENT_GET_REPORT_BUFFER:
		break;

	default:
		break;
	}

	return ret;
}
/**
  * @}
  */

/** @defgroup Device_Game_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes HID gamepad device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Gamepad device.
  * @retval Device structure.
  */
void *usbd_hid_game_init(uint32_t idx, usbd_hid_game_dev_t *dev)
{
	void *ret;
	usbd_hid_dev_t *_dev;
	config_desc_t *desc;

	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->cbk);

	/* Create a pointer to the device */
	_dev = &dev->inst.dev;
	/* Composite initialization */
	ret  = usbd_hid_game_init_comp(idx, dev, 0);

	/* Fill the device descriptor */
	desc = (config_desc_t *)__game_desc;
	desc->bmAttributes = dev->attr_pwr;
	desc->bMaxPower    = dev->max_power >> 1;

	if (ret) {
		/* Initialze hid */
		usbd_hid_init(idx, _dev);
		return (void *)dev;
	}

	return NULL;
}

/**
  * @brief  Initializes HID device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Gamepad device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_hid_game_init_comp(uint32_t idx, usbd_hid_game_dev_t *dev, comp_entry_t *entry)
{
	hid_game_inst_t *inst;
	usbd_hid_dev_t *_dev;

	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->cbk);

	/* Create a pointer to the device instance */
	inst = &dev->inst;
	/* Set the state as default state */
	inst->state = HID_GAME_STATE_DISCONN;

	/* Initialize the idle structrue */
	inst->idle.dur4ms = 125;
	inst->idle.id     = 0;
	inst->idle.since  = 0;
	inst->idle.next   = 0;

	/* Fill the device structure */
	_dev                  = &inst->dev;
	_dev->vid             = dev->vid;
	_dev->pid             = dev->pid;
	_dev->max_power       = dev->max_power;
	_dev->attr_pwr        = dev->attr_pwr;
	_dev->sub_class       = 0;
	_dev->protocol        = 0;
	_dev->nr_input_report = 1;
	_dev->idle            = &inst->idle;
	_dev->tx_cbk          = hid_game_tx_handler;
	_dev->tx_arg          = (void *)dev;
	_dev->rx_cbk          = hid_game_rx_handler;
	_dev->rx_arg          = (void *)dev;
	_dev->use_oep         = 0;
	_dev->desc_hid        = &__game_hid_desc;
	_dev->desc_class      = __game_class_desc;
	_dev->desc_str        = dev->desc_str;
	_dev->num_str         = dev->num_str;
	_dev->desc_config     = __hid_config_desc;

	if (dev->desc_report) {
		/* Save the report descriptor */
		__game_class_desc[0]        = dev->desc_report;
		__game_hid_desc.desc[0].len = dev->report_size;
	}

	/* Initialize the lower layer hid composite */
	return usbd_hid_init_comp(idx, _dev, entry);
}

/**
  * @brief  Terminal the HID device.
  * @param  dev: Gamepad device.
  * @retval None
  */
void usbd_hid_game_term(usbd_hid_game_dev_t *dev)
{
	usbd_hid_dev_t *_dev;

	assert_param(dev);

	/* Create a pointer to the device */
	_dev = &dev->inst.dev;
	/* Set the state as default state */
	dev->inst.state = HID_GAME_STATE_DISCONN;
	/* Terminal the HID device */
	usbd_hid_term(_dev);
}

/**
  * @brief  Schedule a report to be sent.
  * @param  dev: Gamepad device.
  * @param  report: Data to send to the host.
  * @param  size: Size of the buffer.
  * @retval Status.
  */
uint32_t usbd_hid_game_report_send(usbd_hid_game_dev_t *dev, void *report, uint32_t size)
{
	uint32_t cnt;
	hid_game_inst_t *inst;
	usbd_hid_dev_t *_dev;

	/* Create a pointer to the device instance */
	inst = &dev->inst;
	/* Create a pointer to the device */
	_dev = &dev->inst.dev;

	/* Check the state */
	if (inst->state == HID_GAME_STATE_DISCONN)
		return	USBD_GAME_NOT_CONFIG;
	/* Check the TX_FIFO */
	if ((usbd_hid_tx_packet_avail((void *)_dev)) == 0)
		return USBD_GAME_TX_ERROR;

	/* Set the state */
	inst->state = HID_GAME_STATE_SEND;
	/* Send a report to the host */
	cnt = usbd_hid_report_write((void *)_dev, report, size, 1);

	return cnt ? USBD_GAME_SUCCESS : USBD_GAME_TX_ERROR;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
