/**
  *********************************************************************************
  *
  * @file    usbd_hid.h
  * @brief   Header file for the HID driver[Device].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_HID_H__
#define __USBD_HID_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_core.h"
#include "usb_hid.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_HID
  * @{
  */
/**
  * @defgroup Device_HID_Public_Macros Public Macros
  * @{
  */
#define HID_INTERFACE_SIZE	9
#define HID_IN_ENDPOINT_SIZE	7
#define HID_OUT_ENDPOINT_SIZE	7
#define HID_DESCRIPTOR_SIZE	9
#define COMPOSITE_DHID_SIZE	32

#define USAGE(x)          	0x09, ((x) & 0xff)
#define USAGE_PAGE(x)		0x05, ((x) & 0xff)
#define USAGE_VENDOR(x)		0x0A, ((x) & 0xFF), (((x) >> 8) & 0xFF)
#define USAGE_PAGE_VENDOR(x)	0x06, ((x) & 0xFF), (((x) >> 8) & 0xFF)
#define USAGE_MIN(x)		0x19, ((x) & 0xff)
#define USAGE_MAX(x)		0x29, ((x) & 0xff)
#define LOGICAL_MIN(x)		0x15, ((x) & 0xff)
#define LOGICAL_MAX(x)		0x25, ((x) & 0xff)
#define PHYSICAL_MIN(x)		0x36, ((x) & 0xFF), (((x) >> 8) & 0xFF)
#define PHYSICAL_MAX(x)		0x46, ((x) & 0xFF), (((x) >> 8) & 0xFF)
#define COLLECTION(x)		0xa1, ((x) & 0xff)
#define END_COLLECTION		0xc0
#define REPORT_COUNT(x)		0x95, ((x) & 0xff)
#define REPORT_ID(x)		0x85, ((x) & 0xff)
#define REPORT_SIZE(x)		0x75, ((x) & 0xff)
#define INPUT(x)		0x81, ((x) & 0xff)
#define INPUT2(x)		0x82, ((x) & 0xff), (((x) >> 8) & 0xFF)
#define FEATURE(x)		0xB1, ((x) & 0xff)
#define FEATURE2(x)		0xB2, ((x) & 0xff), (((x) >> 8) & 0xFF)
#define OUTPUT(x)		0x91, ((x) & 0xff)
#define OUTPUT2(x)		0x92, ((x) & 0xff), (((x) >> 8) & 0xFF)
#define UNIT_EXPONENT(x)	0x55, ((x) & 0x0f)
#define UNIT(x)			0x67, ((x) & 0x0f), (((x) >> 8) & 0xFF), \
				(((x) >> 16) & 0xFF), (((x) >> 24) & 0xFF)
#define UNIT_DISTANCE_CM	0x66, 0x11, 0x00
#define UNIT_DISTANCE_I		0x66, 0x13, 0x00
#define UNIT_ROTATION_DEG	0x66, 0x14, 0x00
#define UNIT_ROTATION_RAD	0x66, 0x12, 0x00
#define UNIT_MASS_G		0x66, 0x01, 0x01
#define UNIT_TIME_S		0x66, 0x01, 0x10
#define UNIT_TEMP_K		0x67, 0x01, 0x00, 0x01, 0x00
#define UNIT_TEMP_F		0x67, 0x03, 0x00, 0x01, 0x00
#define UNIT_VELOCITY_SI	0x66, 0x11, 0xF0
#define UNIT_MOMENTUM_SI	0x66, 0x11, 0xF1
#define UNIT_ACCELERATION_SI	0x66, 0x11, 0xE0
#define UNIT_FORCE_SI		0x66, 0x11, 0xE1
#define UNIT_ENERGY_SI		0x66, 0x21, 0xE1
#define UNIT_ANG_ACCEL_SI	0x66, 0x12, 0xE0
#define UNIT_VOLTAGE		0x67, 0x21, 0xD1, 0xF0, 0x00
#define UNIT_CURRENT_A		0x67, 0x01, 0x00, 0x10, 0x00
#define USBD_HID_PACKET_MAX	64


#define USB_RX_ERROR_FLAGS	(USBERR_DEV_RX_DATA_ERROR | \
                                 USBERR_DEV_RX_OVERRUN    | \
                                 USBERR_DEV_RX_FIFO_FULL)
#define HID_NOT_FOUND		0xFFFFFFFFU
#define HID_DO_PACKET_RX	5
#define HID_DO_SEND_IDLE_REPORT	6
#define INT_IN_ENDPOINT		USB_EP_3
#define INT_OUT_ENDPOINT	USB_EP_3

#define USBD_HID_EVENT_GET_REPORT		(USBD_HID_EVENT_BASE + 0)
#define USBD_HID_EVENT_REPORT_SENT		(USBD_HID_EVENT_BASE + 1)
#define USBD_HID_EVENT_GET_REPORT_BUFFER	(USBD_HID_EVENT_BASE + 2)
#define USBD_HID_EVENT_SET_REPORT		(USBD_HID_EVENT_BASE + 3)
#define USBD_HID_EVENT_GET_PROTOCOL		(USBD_HID_EVENT_BASE + 4)
#define USBD_HID_EVENT_SET_PROTOCOL		(USBD_HID_EVENT_BASE + 5)
#define USBD_HID_EVENT_IDLE_TIMEOUT		(USBD_HID_EVENT_BASE + 6)
/**
  * @}
  */

/**
  * @defgroup Device_HID_Public_Types Public Types
  * @{
  */
/**
  * @brief HID status
  */
typedef enum {
	HID_STATE_UNCONFIG  = 0,	/**< Unconfigure */
	HID_STATE_IDLE      = 1,	/**< Idle */
	HID_STATE_WAIT_DATA = 2,	/**< Waiting data */
} hid_state_t;

/**
  * @brief Track idle time for reports
  */
typedef struct {
	uint8_t dur4ms;		/**< Idle duration for the report expressed */
	uint8_t id;		/**< ID of the report */
	uint16_t next;		/**< Number of milliseconds before we need to send a report */
	uint32_t since;		/**< Number of milliseconds since the last time this report was sent */
} hid_report_idle_t;

/**
  * @brief Define the private instance data.
  */
typedef struct {
	uint32_t usb_idx;		/**< Index of the USB controller */
	device_info_t info;		/**< Device information */
	volatile hid_state_t rx_state;	/**< State of the HID receive */
	volatile hid_state_t tx_state;	/**< State of the HID transmit */
	volatile uint16_t flag;		/**< Flags */
	uint16_t in_size;		/**< Size of the HID IN report */
	uint16_t in_idx;		/**< Index of the IN report */
	uint16_t out_size;		/**< Size of the HID OUT report */
	uint8_t *in_data;		/**< IN report data */
	uint8_t *out_data;		/**< OUT report data */
	volatile uint8_t conn;		/**< Connection status */
	volatile uint8_t sending;	/**< IN transaction is in process */
	uint8_t pending;		/**< Pending */
	uint8_t iep;			/**< IN endpoint */
	uint8_t oep;			/**< OUT endpoint */
	uint8_t interface;		/**< Interface */
} hid_inst_t;

/**
  * @brief Define operating parameters for the HID device
  */
typedef struct {
	uint16_t vid;					/**< Vendor ID */
	uint16_t pid;					/**< Product ID */
	uint16_t max_power;				/**< Maximum power consumption */
	uint8_t attr_pwr;				/**< Attribute of the power */
	uint8_t sub_class;				/**< Sub-class */
	uint8_t protocol;				/**< Protocol */
	uint8_t nr_input_report;			/**< Number of Input reports */
	hid_report_idle_t *idle;			/**< Pointer to report idle structure */
	usb_cbk rx_cbk;					/**< Callback function */
	void *rx_arg;					/**< Parameter */
	usb_cbk tx_cbk;					/**< Callback function */
	void *tx_arg;					/**< Parameter */
	uint8_t use_oep;				/**< Whether to use OUT endpoint */
	const hid_desc_t *desc_hid;			/**< HID descriptor */
	const uint8_t * const *desc_class;		/**< Class descriptor */
	const uint8_t * const *desc_str;		/**< String descriptor */
	uint32_t num_str;				/**< Number of the string descriptors */
	const config_head_t * const *desc_config;	/**< Configuration descriptor */
	hid_inst_t inst;				/**< Device instance */
} usbd_hid_dev_t;
/**
  * @}
  */

/** @addtogroup Device_HID_Public_Functions
  * @{
  */
extern void *usbd_hid_init(uint32_t idx, usbd_hid_dev_t *dev);
extern void *usbd_hid_init_comp(uint32_t idx, usbd_hid_dev_t *dev, comp_entry_t *entry);
extern void usbd_hid_term(void *device);
extern void *usbd_hid_set_rx_param(void *device, void *arg);
extern void *usbd_hid_set_tx_param(void *device, void *arg);
extern uint32_t usbd_hid_report_write(void *device, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_hid_packet_read(void *device, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_hid_tx_packet_avail(void *device);
extern uint32_t usbd_hid_rx_packet_avail(void *device);
extern void usbd_hid_power_status_set(void *device, uint8_t power);
extern int8_t usbd_hid_remote_wakeup_req(void *device);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
