/**
  *********************************************************************************
  *
  * @file    usbd_hid.c
  * @brief   USB HID device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbd_hid.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_HID HID
  * @brief Device HID driver
  * @{
  */
/** @defgroup Device_HID_Private_Functions Private Functions
  * @{
  */
static void handle_get_desc(void *device, usb_request_t *req);
static void handle_request(void *device, usb_request_t *req);
static void handle_config_change(void *device, uint32_t info);
static void handle_ep0_recv(void *device, uint32_t size);
static void handle_ep0_send(void *device, uint32_t info);
static void handle_reset(void *device);
static void handle_suspend(void *device);
static void handle_resume(void *device);
static void handle_disconnect(void *device);
static void handle_ep(void *device, uint32_t status);
static void handle_device(void *device, uint32_t req, void *data);
/**
  * @}
  */

/** @defgroup Device_HID_Private_Variables Private Variables
  * @{
  */
/**
  * @brief Device descriptor
  */
uint8_t __hid_device_desc[] = {
	18,                       /**< Size of the descriptor */
	USB_DTYPE_DEVICE,         /**< Type of the descriptor */
	USBShort(0x110),          /**< Version 1.1 */
	USB_CLASS_DEVICE,         /**< USB device class */
	0,                        /**< USB device sub-class */
	USB_HID_PROTOCOL_NONE,    /**< USB device protocol */
	USBD_HID_PACKET_MAX,      /**< Maximum packet size */
	USBShort(0),              /**< VID */
	USBShort(0),              /**< PID */
	USBShort(0x100),          /**< Device version */
	1,                        /**< Manufacturer string */
	2,                        /**< Product string */
	3,                        /**< Manufacturer serial number */
	1                         /**< number of the configurations */
};

/**
  * @brief HID handler
  */
const base_handler_t __hid_handler = {
	handle_get_desc,      /**< Get descriptor */
	handle_request,       /**< Request handler */
	0,                    /**< Inerface change */
	handle_config_change, /**< Configuration change */
	handle_ep0_recv,      /**< Data receive */
	handle_ep0_send,      /**< Data sent callback */
	handle_reset,         /**< Reset handler */
	handle_suspend,       /**< Suspend handler */
	handle_resume,        /**< Resume handler */
	handle_disconnect,    /**< Disconnect handler */
	handle_ep,            /**< Endpoint handler */
	handle_device         /**< Device handler */
};
/**
  * @}
  */

/** @addtogroup Device_HID_Private_Functions
  * @{
  */
/**
  * @brief  Set/Reset the flag.
  * @param  flag: Flag.
  * @param  bit: indicates which bit number is to be set or reset.
  * @param  set: Set/Reset.
  * @retval None
  */
static void config_hid_flag(volatile uint16_t *flag, uint16_t bit, uint8_t set)
{
	/* Set/Reset the flag bit */
	if (set)
		SET_BIT(*flag, 1u << bit);
	else
		CLEAR_BIT(*flag, 1u << bit);

	return;
}

/**
  * @brief  Clear the counter used to keep track of the time elapsed.
  * @param  dev: HID device structure.
  * @param  id: Report ID.
  * @retval None
  */
static void clear_report_timer(const usbd_hid_dev_t *dev, uint8_t id)
{
	uint32_t i;

	if (dev->nr_input_report > 1) {
		/* Scan the table it was provided to fine the entry */
		for (i = 0; i < dev->nr_input_report; ++i) {
			if (dev->idle[i].id == id)
				break;
		}
	}
	else {
		i = 0;
	}

	/* If it was found so clear its timer */
	if (i < dev->nr_input_report)
		dev->idle[i].since = 0;

	return;
}

/**
  * @brief  Clear the idle period timers.
  * @param  dev: HID device structure.
  * @retval None
  */
static void clear_idle_timer(const usbd_hid_dev_t *dev)
{
	uint32_t i;

	/* Clear the time fot next report */
	for (i = 0; i < dev->nr_input_report; ++i)
		dev->idle[i].next = dev->idle[i].dur4ms << 2;

	return;
}

/**
  * @brief  Process the report idle timers.
  * @param  dev: HID device structure.
  * @param  elapse: number of milliseconds that have elapsed.
  * @retval None
  */
static void process_idle_timer(const usbd_hid_dev_t *dev, uint32_t elapse)
{
	uint32_t i, size;
	void *report;
	hid_inst_t *inst;
	uint8_t state = 0;

	/* Create a pointer to the device instance */
	inst = &((usbd_hid_dev_t *)dev)->inst;

	/* Look at each input report */
	for (i = 0; i < dev->nr_input_report; ++i) {
		/* Update the time since the last report was sent */
		dev->idle[i].since += elapse;

		/* Timer is running? */
		if (dev->idle[i].dur4ms) {
			/* It's about to expire */
			if (dev->idle[i].next <= elapse) {
				/* Timer is about to expire and send a report now */
				if ((inst->tx_state == HID_STATE_IDLE) && (inst->sending == 0)) {
					/* Call the callback function */
					size = dev->rx_cbk(dev->rx_arg, USBD_HID_EVENT_IDLE_TIMEOUT,
									dev->idle[i].id, &report);
					/* Send the report */
					usbd_hid_report_write((void *)dev, report, size, 1);
					/* Reload the timer value */
					dev->idle[i].next = dev->idle[i].dur4ms << 2;
				}
				else {
					/* It can't send the report */
					dev->idle[i].next = 0;
					state = 1;
				}
			}
			else {
				/* Update the time till the next report */
				dev->idle[i].next -= elapse;
			}
		}
	}

	/* Records this so that it can process as soon as possible */
	config_hid_flag(&inst->flag, HID_DO_SEND_IDLE_REPORT, state);
	return;
}

/**
  * @brief  Set the report idle timers.
  * @param  dev: HID device structure.
  * @param  id: Report ID.
  * @param  timeout: Timeout.
  * @retval None
  */
static void set_idle_timeout(const usbd_hid_dev_t *dev, uint8_t id, uint8_t timeout)
{
	uint32_t i;
	uint8_t flag = 0;
	hid_report_idle_t *idle;

	/* Look at each input report */
	for (i = 0; i < dev->nr_input_report; ++i) {
		idle = &dev->idle[i];

		/* Check the report ID */
		if (!id || (id == idle->id)) {
			/* Save the duration for the idle timer */
			idle->dur4ms = timeout;

			/* If is it enable the idle timer? */
			if (timeout) {
				/* Determine what the timeout is for this
				 * report given the time since the last
				 * report of this type was sent.
				 */
				if (idle->since >= (timeout << 2)) {
					idle->next = 0;
					flag       = 1;
				}
				else {
					idle->next = ((timeout << 2) - idle->since);
				}
			}
		}
	}

	/* It need to send at least one of the
	 * input reports immediately.
	 */
	if (flag)
		process_idle_timer(dev, 0);

	return;
}

/**
  * @brief  Get the report idle timers.
  * @param  dev: HID device structure.
  * @param  id: Report ID.
  * @retval Timeout.
  */
static uint32_t get_idle_timeout(const usbd_hid_dev_t *dev, uint8_t id)
{
	uint32_t i;
	hid_report_idle_t *idle;

	/* Look at each input report */
	for (i = 0; i < dev->nr_input_report; ++i) {
		idle = &dev->idle[i];

		if (!id || (id == idle->id))
			return idle->dur4ms;
	}

	return HID_NOT_FOUND;
}

/**
  * @brief  Find the HID descriptor of a given type.
  * @param  dev: HID device structure.
  * @param  type: Type of the HID descriptor.
  * @param  idx: Index of of the descriptor.
  * @param  len: Length.
  * @retval Index of the descriptor.
  */
static uint32_t find_hid_desc(const usbd_hid_dev_t *dev, uint8_t type, uint32_t idx, uint32_t *len)
{
	uint8_t find = 0;
	uint32_t i, cnt = 0, last = 0;
	const hid_class_desc_info_t *desc;

	/* Look at each HID descpritor */
	for (i = 0; i < dev->desc_hid->nr; ++i) {
		desc = &(dev->desc_hid->desc[i]);

		if (desc->type == type) {
			find = 1;

			if (cnt == idx) {
				/* Find it */
				*len = desc->len;
				return i;
			}
			else {
				/* Update the count and keep looking */
				++cnt;
				last = i;
			}
		}
	}

	/* Don't find the requested descpritor, return a physical descpritor */
	if ((type == USB_HID_DTYPE_PHYSICAL) && find) {
		/* Get the length of the last descriptor */
		desc = &(dev->desc_hid->desc[last]);
		*len = desc->len;

		return last;
	}
	else {
		return HID_NOT_FOUND;
	}
}

/**
  * @brief  Schedule transmission of the next packet.
  * @param  inst: HID device instance.
  * @retval Status.
  */
static int32_t schedule_report_send(hid_inst_t *inst)
{
	uint8_t *data;
	int32_t nr, ret;

	/* Set the number of the report */
	nr = (uint32_t)(inst->in_size - inst->in_idx);

	/* Adjust the number */
	if (nr > USBD_HID_PACKET_MAX)
		nr = USBD_HID_PACKET_MAX;

	/* Get the beginning pointer */
	data = inst->in_data + inst->in_idx;
	/* Put the data into the FIFO */
	ret  = map_usb_ep_data_put(inst->iep, data, nr);

	if (ret != -1) {
		/* Update the count */
		inst->in_idx += nr;
		/* Send the data to host */
		ret = map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
	}

	return ret;
}

/**
  * @brief  Process receives notifications from the host.
  * @param  dev: HID device stucture.
  * @param  state: Status.
  * @retval Status.
  */
static int8_t process_data_from_host(usbd_hid_dev_t *dev, uint32_t state)
{
	uint32_t _status, size;
	hid_inst_t *inst;

	/* Create a pointer to the device instance */
	inst    = &dev->inst;
	/* Get the endpoint state */
	_status = map_usb_ep_status(inst->oep);
	/* Clear the endpoint status */
	map_usb_dev_ep_status_clear(inst->oep, _status);

	/* Received a packet? */
	if (_status & USB_DEV_RX_PKT_RDY) {
		/* Set the flag */
		config_hid_flag(&inst->flag, HID_DO_PACKET_RX, 1);
		/* Get the size of the packet */
		size = map_usb_ep_data_avail(inst->oep);
		/* Send a RX_AVAILABLE event to the  client */
		dev->rx_cbk(dev->rx_arg, USB_EVENT_RX_AVAILABLE, size, NULL);
	}
	else {
		/* Send a ERROR event to the client */
		if (_status & USB_RX_ERROR_FLAGS)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_ERROR, (_status & USB_RX_ERROR_FLAGS), NULL);

		return -1;
	}

	return 0;
}

/**
  * @brief  Process send data to the host.
  * @param  dev: HID device stucture.
  * @param  state: Status.
  * @retval Status.
  */
static int8_t process_data_to_host(usbd_hid_dev_t *dev, uint32_t state)
{
	uint32_t _status;
	hid_inst_t *inst;

	/* Create a pointer to the device instance */
	inst    = &dev->inst;
	/* Get the endpoint state */
	_status = map_usb_ep_status(inst->iep);
	/* Clear the endpoint status */
	map_usb_dev_ep_status_clear(inst->iep, _status);

	/* Last packet was transmitted successfully */
	if (inst->in_size == inst->in_idx) {
		/* Set the TX_STATE to idle */
		inst->tx_state = HID_STATE_IDLE;
		/* Call the callback function */
		dev->tx_cbk(dev->tx_arg, USB_EVENT_TX_COMPLETE, inst->in_size, NULL);

		/* Need to send a report? */
		if (inst->flag & (1 << HID_DO_SEND_IDLE_REPORT))
			process_idle_timer(dev, 0);
	}
	else {
		/* Waiting more data or a zero-length packet */
		schedule_report_send(inst);
	}

	return 0;
}

/**
  * @brief  Handle the interrupts on the endpoints.
  * @param  device: HID device.
  * @param  status: Current status.
  * @retval None
  */
static void handle_ep(void *device, uint32_t status)
{
	hid_inst_t *inst;

	assert_param(device != 0);

	/* Create a pointer to the device instance */
	inst = &((usbd_hid_dev_t *)device)->inst;

	/* Handler the interrupt OUT endpoint */
	if (status & (0x10000 << inst->oep))
		process_data_from_host(device, status);

	/* Handler the interrupt IN endpoint */
	if (status & (1 << inst->iep))
		process_data_to_host(device, status);

	return;
}

/**
  * @brief  Handle device configuration changes.
  * @param  device: HID device.
  * @param  info: New value.
  * @retval None
  */
static void handle_config_change(void *device, uint32_t info)
{
	hid_inst_t *inst;
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_dev_t *)device;
	inst = &dev->inst;

	/* Default the state */
	inst->rx_state = HID_STATE_IDLE;
	inst->tx_state = HID_STATE_IDLE;

	/* Send a CONNECTED event to the client */
	if (!inst->conn)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_CONNECTED, 0, NULL);

	/* Clear the idle timers */
	clear_idle_timer(dev);
	/* Records the connected state */
	inst->conn = 1;

	return;
}

/**
  * @brief  Handle device envent.
  * @param  device: HID device.
  * @param  req: Request.
  * @param  data: Data buffer.
  * @retval None
  */
static void handle_device(void *device, uint32_t req, void *data)
{
	uint8_t *buf;
	hid_inst_t *inst;
	usbd_hid_dev_t *dev;

	/* Create a pointer to the data */
	buf  = (uint8_t *)data;
	/* Create a pointer to the device instance */
	dev  = (usbd_hid_dev_t *)device;
	inst = &dev->inst;

	switch (req) {
	case USB_EVENT_COMP_IFACE_CHANGE:
		/* Handle the interface change event */
		inst->interface = buf[1];
		break;

	case USB_EVENT_COMP_EP_CHANGE:
		/* Handle the endpoint change event */
		if (buf[0] & USB_EP_DESC_IN)
			inst->iep = buf[1] & 0x7f;
		else
			inst->oep = buf[1] & 0x7f;

		break;
	case USB_EVENT_LPM_RESUME:
		/* Send a LPM_RESUME event to the client */
		if (dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_RESUME, 0, NULL);

		break;
	case USB_EVENT_LPM_SLEEP:
		/* Send a LPM_SLEEP event to the client */
		if(dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_SLEEP, 0, NULL);

		break;
	case USB_EVENT_LPM_ERROR:
		/* Send a LPM_ERROR event to the client */
		if(dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_ERROR, 0, NULL);
		break;

	default:
		break;
	}

	return;
}

/**
  * @brief  Handle disconnect envent.
  * @param  device: HID device.
  * @retval None
  */
static void handle_disconnect(void *device)
{
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	dev = (usbd_hid_dev_t *)device;

	/* Send a DISCONNECTED event to the client */
	if (dev->inst.conn)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_DISCONNECTED, 0, NULL);

	/* Records the connected state */
	dev->inst.conn = 0;
	return;
}

/**
  * @brief  Handle get descriptor.
  * @param  device: HID device.
  * @param  req: Non-standard request.
  * @retval None
  */
static void handle_get_desc(void *device, usb_request_t *req)
{
	uint32_t size, _desc;
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	/* Create a pointer to the device */
	dev = (usbd_hid_dev_t *)device;

	/* Check the type of class descriptor */
	switch (req->wValue >> 8) {
	case USB_HID_DTYPE_REPORT:
	case USB_HID_DTYPE_PHYSICAL:
		/* Find the descriptor that is being requested */
		size = 0;
		_desc = find_hid_desc(dev, req->wValue >> 8, req->wValue & 0xFF, &size);

		/* No find? */
		if (_desc == HID_NOT_FOUND) {
			usb_dcd_ep0_stall(0);
			return;
		}

		/* Adjust the size */
		if (size > req->wLength)
			size = req->wLength;

		/* Send the data */
		usb_dcd_ep0_data_send(0, (uint8_t *)dev->desc_class[_desc], size);
		break;

	case USB_HID_DTYPE_HID:
		/* Get the size of the HID descriptor */
		size = dev->desc_hid->len;

		/* Adjust the size */
		if (size > req->wLength)
			size = req->wLength;

		/* Send the data */
		usb_dcd_ep0_data_send(0, (uint8_t *)dev->desc_hid, size);
		break;

	default:
		usb_dcd_ep0_stall(0);
		break;
	}
}

/**
  * @brief  Handle non-standard request.
  * @param  device: HID device.
  * @param  req: Non-standard request.
  * @retval None
  */
static void handle_request(void *device, usb_request_t *req)
{
	uint8_t protocol, *report;
	uint32_t size, timeout;
	hid_inst_t *inst;
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_dev_t *)device;
	inst = &dev->inst;

	/* Check the interface */
	if (req->wIndex != inst->interface)
		return;

	/* Check the type of request */
	switch (req->bRequest) {
	case USBREQ_SET_REPORT:
		/* Alloc a buffer large enough to hold the report */
		inst->out_size = req->wLength;
		inst->out_data = (uint8_t *)dev->rx_cbk(dev->rx_arg, USBD_HID_EVENT_GET_REPORT_BUFFER,
							req->wValue, (void *)(uint32_t)(req->wLength));

		/* Check the buffer pointer */
		if (!inst->out_data) {
			usb_dcd_ep0_stall(0);
		}
		else {
			/* Set the RX_STATE */
			inst->rx_state = HID_STATE_WAIT_DATA;
			/* Read the payload of the request now */
			usb_dcd_ep0_data_req(0, inst->out_data, (uint32_t)req->wLength);
			/* ACK the data */
			map_usb_dev_ep_data_ack(USB_EP_0, false);
		}

		break;
	case USBREQ_GET_REPORT:
		/* Get the latest report from user */
		size = dev->rx_cbk(dev->rx_arg, USBD_HID_EVENT_GET_REPORT, req->wValue, &report);
		/* ACK the data */
		map_usb_dev_ep_data_ack(USB_EP_0, true);
		/* Send back the requested report */
		inst->pending = 1;
		usb_dcd_ep0_data_send(0, report, size);

		break;
	case USBREQ_SET_IDLE:
		/* Set the idle timerout */
		set_idle_timeout(dev, req->wValue & 0xFF, (req->wValue >> 8) & 0xFF);
		/* ACK the data */
		map_usb_dev_ep_data_ack(USB_EP_0, true);

		break;
	case USBREQ_GET_IDLE:
		/* Get the idle timerout */
		timeout = get_idle_timeout(dev, req->wValue);

		/* Check the timeout */
		if (timeout != HID_NOT_FOUND) {
			/* ACK the data */
			map_usb_dev_ep_data_ack(USB_EP_0, true);
			/* Send response via EP0 */
			usb_dcd_ep0_data_send(0, (uint8_t *)&timeout, 1);
		}
		else {
			usb_dcd_ep0_stall(0);
		}

		break;
	case USBREQ_SET_PROTOCOL:
		/* Check the class */
		if (dev->sub_class == USB_HID_SCLASS_BOOT) {
			/* ACK the data */
			map_usb_dev_ep_data_ack(USB_EP_0, true);
			/* Send a SET_PROTOCOL event to the client */
			dev->rx_cbk(dev->rx_arg, USBD_HID_EVENT_SET_PROTOCOL, req->wValue, NULL);
		}
		else {
			usb_dcd_ep0_stall(0);
		}

		break;
	case USBREQ_GET_PROTOCOL:
		if (dev->sub_class == USB_HID_SCLASS_BOOT) {
			/* ACK the data */
			map_usb_dev_ep_data_ack(USB_EP_0, true);
			/* Get the latest protocol */
			protocol = (uint8_t)dev->rx_cbk(dev->rx_arg, USBD_HID_EVENT_GET_PROTOCOL, 0, NULL);
			/* Send response via EP0 */
			usb_dcd_ep0_data_send(0, (uint8_t *)&protocol, 1);
		}
		else {
			usb_dcd_ep0_stall(0);
		}

		break;
	default:
		usb_dcd_ep0_stall(0);
		break;
	}

	return;
}

/**
  * @brief  Data requested on endpoint zero is received.
  * @param  device: HID device.
  * @param  size: Size of the data.
  * @retval None
  */
static void handle_ep0_recv(void *device, uint32_t size)
{
	hid_inst_t *inst;
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	/* Check the size */
	if(size == 0)
		return;

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_dev_t *)device;
	inst = &dev->inst;

	/* Check the RX_STATE */
	if (inst->rx_state != HID_STATE_WAIT_DATA)
		return;

	/* Set the RX_STATE to idle */
	inst->rx_state = HID_STATE_IDLE;
	/* Send a SET_REPORT event to the client */
	dev->rx_cbk(dev->rx_arg, USBD_HID_EVENT_SET_REPORT, inst->out_size, inst->out_data);

	return;
}

/**
  * @brief  Data sent on endpoint zero is received a acknowledged.
  * @param  device: HID device.
  * @param  info: Information.
  * @retval None
  */
static void handle_ep0_send(void *device, uint32_t info)
{
	hid_inst_t *inst;
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_dev_t *)device;
	inst = &dev->inst;

	/* If it just sent a report, send an event to the client */
	if (inst->pending) {
		/* Clear the flag */
		inst->pending = 0;
		/* Send a REPORT_SENT event to the client */
		dev->rx_cbk(dev->rx_arg, USBD_HID_EVENT_REPORT_SENT, 0, NULL);
	}

	return;
}

/**
  * @brief  Called whenever the device is reset.
  * @param  device: HID device.
  * @retval None
  */
static void handle_reset(void *device)
{
	assert_param(device != 0);

	/* Handle disconnect */
	handle_disconnect(device);
	return;
}

/**
  * @brief  Called whenever the device is suspend.
  * @param  device: HID device.
  * @retval None
  */
static void handle_suspend(void *device)
{
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	/* Create a pointer to the device */
	dev = (usbd_hid_dev_t *)device;
	/* Send a SUSPEND event to the client */
	dev->rx_cbk(dev->rx_arg, USB_EVENT_SUSPEND, 0, NULL);
	/* Records the connected state */
	dev->inst.conn = 0;
	return;
}

/**
  * @brief  Called whenever the device is resume.
  * @param  device: HID device.
  * @retval None
  */
static void handle_resume(void *device)
{
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	/* Create a pointer to the device */
	dev = (usbd_hid_dev_t *)device;
	/* Send a RESUME event to the client */
	dev->rx_cbk(dev->rx_arg, USB_EVENT_RESUME, 0, NULL);
	return;
}

/**
  * @brief  Provides us with a time reference.
  * @param  device: HID device.
  * @param  time: Elapsed time in milliseconds.
  * @retval None
  */
static void hid_tick_handler(void *device, uint32_t time)
{
	uint32_t size;
	hid_inst_t *inst;
	usbd_hid_dev_t *dev;

	assert_param(device != 0);

	/* Create a pointer to the device instance */
	dev  = (usbd_hid_dev_t *)device;
	inst = &dev->inst;

	/* If it's connected, process the idle timers */
	if (inst->conn)
		process_idle_timer(dev, time);

	/* Do it have a deferred receive waiting */
	if (inst->flag & (1 << HID_DO_PACKET_RX)) {
		/* Get the size of packet */
		size = map_usb_ep_data_avail(inst->oep);
		/* Send a RX_AVAILABLE event to the client */
		dev->rx_cbk(dev->rx_arg, USB_EVENT_RX_AVAILABLE, size, NULL);
	}

	return;
}
/**
  * @}
  */

/** @defgroup Device_HID_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes HID device.
  * @param  idx: Index of the USB controller.
  * @param  dev: HID device.
  * @retval Device structure.
  */
void *usbd_hid_init(uint32_t idx, usbd_hid_dev_t *dev)
{
	device_desc_t *desc;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->rx_cbk);
	assert_param(dev->tx_cbk);
	assert_param(dev->desc_class);
	assert_param(dev->desc_hid);
	assert_param((dev->nr_input_report == 0) || dev->idle);

	/* Composite initialization */
	usbd_hid_init_comp(idx, dev, 0);

	/* Fill the device descriptor */
	desc = (device_desc_t *)dev->inst.info.device_desc;
	desc->idVendor  = dev->vid;
	desc->idProduct = dev->pid;

	/* Initialze device controler */
	usb_dcd_init(idx, &dev->inst.info, (void *)dev);
	return (void *)dev;
}

/**
  * @brief  Initializes HID device.
  * @param  idx: Index of the USB controller.
  * @param  dev: HID device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_hid_init_comp(uint32_t idx, usbd_hid_dev_t *dev, comp_entry_t *entry)
{
	hid_inst_t *inst;
	endpoint_desc_t *ep;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_config);
	assert_param(dev->desc_str);
	assert_param(dev->rx_cbk);
	assert_param(dev->tx_cbk);
	assert_param(dev->desc_class);
	assert_param(dev->desc_hid);
	assert_param((dev->nr_input_report == 0) || dev->idle);

	/* Create a pointer to the device instance */
	inst = &dev->inst;

	/* Initialize the device information */
	inst->info.cbk            = &__hid_handler;
	inst->info.device_desc    = __hid_device_desc;
	inst->info.config_desc    = dev->desc_config;
	inst->info.string_desc    = dev->desc_str;
	inst->info.nr_string_desc = dev->num_str;

	/* Initialize the device instance structure */
	inst->interface = 0;
	inst->iep       = 0;
	inst->oep       = 0;

	/* Get a pointer to the endpoint */
	ep = usb_dcd_endpoint_get(dev->desc_config[0], inst->interface, 0, 0);

	/* Records the endpoint number */
	if (ep) {
		if (ep->bEndpointAddress & 0x80)
			inst->iep = ep->bEndpointAddress & 0x7F;
		else
			inst->oep = ep->bEndpointAddress;
	}

	/* Get a pointer to the endpoint */
	ep = usb_dcd_endpoint_get(dev->desc_config[0], inst->interface, 0, 1);

	/* Records the endpoint number */
	if (ep) {
		if (ep->bEndpointAddress & 0x80)
			inst->iep = ep->bEndpointAddress & 0x7F;
		else
			inst->oep = ep->bEndpointAddress;
	}

	/* Check the IN endponit */
	if (inst->iep == 0)
		return NULL;

	/* Initialze the composite entry */
	if (entry != 0) {
		entry->info = &inst->info;
		entry->inst = (void *)dev;
	}

	/* Initialize the device information */
	inst->usb_idx  = 0;
	inst->rx_state = HID_STATE_UNCONFIG;
	inst->tx_state = HID_STATE_UNCONFIG;
	inst->flag     = 0;
	inst->conn     = 0;
	inst->pending  = 0;
	inst->sending  = 0;
	inst->in_idx   = 0;
	inst->in_size  = 0;
	inst->in_data  = NULL;
	inst->out_size = 0;
	inst->out_data = NULL;

	/* Initialize the device information */
	usb_device_info_init(0, &inst->info);
	/* Reset the idle timer */
	clear_idle_timer(dev);
	/* Initialze tick */
	usb_tick_init();
	/* Register tick callback fucntion */
	usb_tick_handler_register(hid_tick_handler, (void *)dev);

	return (void *)dev;
}

/**
  * @brief  Terminal the HID device.
  * @param  device: HID device.
  * @retval None
  */
void usbd_hid_term(void *device)
{
	hid_inst_t *inst;

	assert_param(device);

	/* Create a pointer to the device instance */
	inst = &((usbd_hid_dev_t *)device)->inst;
	/* Terminate the device controler */
	usb_dcd_term(inst->usb_idx);
	inst->usb_idx = 0;

	return;
}

/**
  * @brief  Set RX callback parameter.
  * @param  device: HID device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_hid_set_rx_param(void *device, void *arg)
{
	void *old;

	assert_param(device);

	/* Set the RX callback parameter */
	old = ((usbd_hid_dev_t *)device)->rx_arg;
	((usbd_hid_dev_t *)device)->rx_arg = arg;

	return old;
}

/**
  * @brief  Set TX callback parameter.
  * @param  device: HID device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_hid_set_tx_param(void *device, void *arg)
{
	void *old;

	assert_param(device);

	/* Set the TX callback parameter */
	old = ((usbd_hid_dev_t *)device)->tx_arg;
	((usbd_hid_dev_t *)device)->tx_arg = arg;

	return old;
}

/**
  * @brief  Transmit a HID device report to the USB host.
  * @param  device: HID device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Ignored in this implementation.
  * @retval Length.
  */
uint32_t usbd_hid_report_write(void *device, uint8_t *data, uint32_t len, uint8_t last)
{
	hid_inst_t *inst;
	int32_t ret;

	assert_param(device);

	/* Create a pointer to the device instance */
	inst = &((usbd_hid_dev_t *)device)->inst;
	/* Set the flag indicating that it's in the process of sending a report */
	inst->sending = 1;

	/* Check the TX_STATE */
	if (inst->tx_state != HID_STATE_IDLE) {
		inst->sending = 0;
		return 0;
	}

	/* Clear the elapsed time */
	if (len)
		clear_report_timer(device, *data);

	/* Zero-copy */
	inst->in_idx  = 0;
	inst->in_size = len;
	inst->in_data = data;

	/* Set the TX_STATE */
	inst->tx_state = HID_STATE_WAIT_DATA;
	/* Schedule transmission of the report */
	ret = schedule_report_send(inst);
	/* Clear the flag */
	inst->sending = 0;

	if (ret != -1)
		return len;

	return 0;
}

/**
  * @brief  Read a packet of data received from the USB host.
  * @param  device: HID device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Ignored in this implementation.
  * @retval Length.
  */
uint32_t usbd_hid_packet_read(void *device, uint8_t *data, uint32_t len, uint8_t last)
{
	int32_t ret;
	uint32_t status, cnt, pkt;
	hid_inst_t *inst;

	assert_param(device);

	/* Create a pointer to the device instance */
	inst   = &((usbd_hid_dev_t *)device)->inst;
	/* Get the endpoint status */
	status = map_usb_ep_status(inst->oep);

	/* Has it received a packet? */
	if (status & USB_DEV_RX_PKT_RDY) {
		/* Get the length of the packet */
		pkt = map_usb_ep_data_avail(inst->oep);
		cnt = len;
		/* Get the data of the packet */
		ret = map_usb_ep_data_get(inst->oep, data, &cnt);

		/* Check if it read the last segment */
		if (cnt == pkt) {
			/* Clear the endpoint state */
			map_usb_dev_ep_status_clear(inst->oep, status);
			/* ACK the packet */
			map_usb_dev_ep_data_ack(inst->oep, true);
			/* Clear the flag indicating that the packet has been received */
			config_hid_flag(&inst->flag, HID_DO_PACKET_RX, false);
		}

		/* Check the return value */
		if (ret != -1)
			return cnt;
	}

	return 0;
}

/**
  * @brief  Get number of free bytes in the transmit buffer.
  * @param  device: HID device.
  * @retval Length.
  */
uint32_t usbd_hid_tx_packet_avail(void *device)
{
	hid_inst_t *inst;

	assert_param(device);

	/* Create a pointer to the device instance */
	inst = &((usbd_hid_dev_t *)device)->inst;

	/* Check the TX_STATE */
	if (inst->tx_state == HID_STATE_IDLE)
		return USBD_HID_PACKET_MAX;

	return 0;
}

/**
  * @brief  Determine whether a packet is available.
  * @param  device: HID device.
  * @retval Length.
  */
uint32_t usbd_hid_rx_packet_avail(void *device)
{
	uint32_t status, size;
	hid_inst_t *inst;

	assert_param(device);

	/* Create a pointer to the device instance */
	inst   = &((usbd_hid_dev_t *)device)->inst;
	/* Get the endpoint status */
	status = map_usb_ep_status(inst->oep);

	if (status & USB_DEV_RX_PKT_RDY) {
		/* Get the size of the packet */
		size = map_usb_ep_data_avail(inst->oep);
		return size;
	}

	return 0;
}

/**
  * @brief  Set power status.
  * @param  device: HID device.
  * @param  power: Status of the power.
  * @retval None
  */
void usbd_hid_power_status_set(void *device, uint8_t power)
{
	assert_param(device);

	usb_dcd_power_status_set(0, power);
	return;
}

/**
  * @brief  Requests a remote wake up to resume communication.
  * @param  device: HID device.
  * @retval Status.
  */
int8_t usbd_hid_remote_wakeup_req(void *device)
{
	assert_param(device);
	return usb_dcd_remote_wakeup_req(0);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
