/**
  *********************************************************************************
  *
  * @file    usbd_core.h
  * @brief   Header file for the CORE[Device].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_CORE_H__
#define __USBD_CORE_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usb_utils.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_Core
  * @{
  */
/**
  * @defgroup Device_Core_Public_Macros Public Macros
  * @{
  */
#define USB_EP_IN			0U
#define USB_EP_OUT			1U
#define USB_MAX_INTERFACES_PER_DEVICE	8U
#define EP0_MAX_PACKET_SIZE		64U
#define DEV_ADDR_PENDING		0x80000000U
#define DEFAULT_CONFIG_ID		1U
#define REMOTE_WAKEUP_PULSE_MS		10U
#define REMOTE_WAKEUP_READY_MS		20U
#define USBLIB_LPM_STATE_DISABLED	0x0U
#define USBLIB_LPM_STATE_AWAKE		0x1U
#define USBLIB_LPM_STATE_SLEEP		0x2U
/**
  * @}
  */

/**
  * @defgroup Device_Core_Public_Types Public Types
  * @{
  */
/**
  * @brief Endpoint 0 status
  */
typedef enum {
	USB_EP0_STATE_IDLE      = 0x0,	/**< Idle status */
	USB_EP0_STATE_TX        = 0x1,	/**< Send data status */
	USB_EP0_STATE_TX_CONFIG = 0x2,	/**< Send configure descriptor status*/
	USB_EP0_STATE_RX        = 0x3,	/**< Receive status */
	USB_EP0_STATE_STATUS    = 0x4,	/**< Before idle status */
	USB_EP0_STATE_STALL     = 0x5,	/**< Stall status */
} ep0_state_t;

/**
  * @brief Endpoint information
  */
typedef struct {
	uint32_t size[2];	/**< Size information */
} usb_ep_info_t;

/**
  * @brief Device information
  */
typedef struct {
	const base_handler_t *cbk;			/**< Callback function */
	const uint8_t *device_desc;			/**< Device descriptor */
	const config_head_t * const *config_desc;	/**< Configure descriptor */
	const uint8_t * const *string_desc;		/**< String descriptor */
	uint32_t nr_string_desc;			/**< Number of the String descriptor */
} device_info_t;

/**
  * @brief Composite device entry structure
  */
typedef struct {
	const device_info_t *info;	/**< Information */
	void *inst;			/**< Instance */
	uint32_t space;			/**< Space */
} comp_entry_t;

/**
  * @brief Device instance structure
  */
typedef struct {
    volatile ep0_state_t ep0_state;		/**< EP0 status */
    volatile uint32_t dev_addr;			/**< Device address */
    uint32_t config;				/**< Configure */
    uint32_t config_default;			/**< Default configure */
    uint8_t alt[USB_MAX_INTERFACES_PER_DEVICE];	/**< Alternate setting */
    uint8_t *ep0_data;				/**< EP0 data */
    volatile uint32_t ep0_remain;		/**< EP0 remaining length */
    uint32_t size_out;				/**< Size of OUT transmit */
    uint8_t status;				/**< Status */
    uint8_t halt[2][USBLIB_NUM_EP - 1];		/**< Halt array */
    uint8_t config_sec;				/**< Configure sections */
    uint16_t offset_sec;			/**< Offset sections */
    uint8_t config_idx;				/**< Index of the configure */
    uint8_t power_set;				/**< Power paramter */
    uint8_t remote_wakeup;			/**< Remote wakeup feature */
    uint8_t cnt_wakeup;				/**< Count of wakeup */
    void *arg;					/**< Parameter */
    uint32_t lpm_state;				/**< Low power management */
    uint32_t feature;				/**< Feature */
} dcd_instance_t;
/**
  * @}
  */
/** @addtogroup Device_Core_Public_Variables
  * @{
  */
extern dcd_instance_t __dcd_inst[];
extern device_info_t *__dev_info[];
/**
  * @}
  */

/** @addtogroup Device_Core_Public_Functions
  * @{
  */
extern void usb_device_info_init(uint32_t idx, device_info_t *dev);
extern void usb_dcd_init(uint32_t idx, device_info_t *dev, void *arg);
extern void usb_dcd_term(uint32_t idx);
extern void usb_dcd_ep0_stall(uint32_t idx);
extern void usb_dcd_ep0_data_req(uint32_t idx, uint8_t *data, uint32_t size);
extern void usb_dcd_ep0_data_send(uint32_t idx, uint8_t *data, uint32_t size);
extern endpoint_desc_t *usb_dcd_endpoint_get(const config_head_t *config, uint32_t nr, uint32_t alt, uint32_t idx);
extern void usb_dcd_config_set_default(uint32_t idx, uint32_t config);
extern void usb_dcd_power_status_set(uint32_t idx, uint8_t status);
extern uint8_t usb_dcd_remote_wakeup_req(uint32_t idx);
extern uint8_t usb_dcd_remote_wakeup_lpm(uint32_t idx);
extern void usb0_device_int_handler(void);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
