/**
  *********************************************************************************
  *
  * @file    usbd_core.c
  * @brief   Functions related to device mode.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbd_core.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @defgroup DEVICE Device
  * @brief Device mode driver
  * @{
  */
/** @defgroup Device_Core Core
  * @brief Device core driver
  * @{
  */
/** @defgroup Device_Core_Private_Functions Private Functions
  * @{
  */
static void usbd_get_status(void *inst, usb_request_t *req);
static void usbd_clear_feature(void *inst, usb_request_t *req);
static void usbd_set_feature(void *inst, usb_request_t *req);
static void usbd_set_address(void *inst, usb_request_t *req);
static void usbd_get_desc(void *inst, usb_request_t *req);
static void usbd_set_desc(void *inst, usb_request_t *req);
static void usbd_get_config(void *inst, usb_request_t *req);
static void usbd_set_config(void *inst, usb_request_t *req);
static void usbd_get_interface(void *inst, usb_request_t *req);
static void usbd_set_interface(void *inst, usb_request_t *req);
static void usbd_sync_frame(void *inst, usb_request_t *req);
/**
  * @}
  */

/** @defgroup Device_Core_Public_Variables Public Variables
  * @{
  */
dcd_instance_t __dcd_inst[1];
device_info_t *__dev_info[1];
/**
  * @}
  */
/** @defgroup Device_Core_Private_Variables Private Variables
  * @{
  */
static uint8_t __data_in[EP0_MAX_PACKET_SIZE];

static const std_request __usb_std_request[] = {
	usbd_get_status,
	usbd_clear_feature,
	0,
	usbd_set_feature,
	0,
	usbd_set_address,
	usbd_get_desc,
	usbd_set_desc,
	usbd_get_config,
	usbd_set_config,
	usbd_get_interface,
	usbd_set_interface,
	usbd_sync_frame
};
/**
  * @}
  */

/** @addtogroup Device_Core_Private_Functions
  * @{
  */
/**
  * @brief  Get the next configure descriptor.
  * @param  config: Configure descriptor.
  * @param  sec: Section.
  * @param  desc: Head of descriptor.
  * @retval Head of the next descriptor.
  */
static desc_head_t *get_next_config_desc(const config_head_t *config, uint32_t *sec, desc_head_t *desc)
{
	/* Get the next descriptor */
	desc = NEXT_USB_DESCRIPTOR(desc);

	/* This is the end of the section */
	if ((uint8_t *)desc >= (config->section[*sec]->data + config->section[*sec]->size)) {
		/* Move to the next section */
		(*sec)++;

		/* If it's within the descriptor? */
		if (*sec < config->nr_section)
			desc = (desc_head_t *)config->section[*sec]->data;
		else
			desc = NULL;
	}

	return desc;
}

/**
  * @brief  Get the alternate descriptor from the configure descriptor.
  * @param  config: Configure descriptor.
  * @param  nr: Number of the descriptor.
  * @param  idx: Index of the descriptor.
  * @param  sec: Section.
  * @retval Alternate descriptor.
  */
static interface_desc_t *get_alt_interface_from_config(const config_head_t *config,
						uint8_t nr, uint32_t idx, uint32_t *sec)
{
	desc_head_t *desc;
	uint32_t cnt, __sec;

	/* Initialize the counting loop */
	desc  = (desc_head_t *)config->section[0]->data;
	cnt   = 0;
	__sec = 0;

	/* Until it reach the end */
	while (desc) {
		/* The descriptor match the type */
		if ((desc->bDescriptorType == USB_DTYPE_INTERFACE)
					&& (((interface_desc_t *)desc)->bInterfaceNumber == nr)) {
			/* Find it */
			if (cnt++ == idx) {
				*sec = __sec;
				return (interface_desc_t *)desc;
			}
		}

		/* Move to the next descriptor */
		desc = get_next_config_desc(config, &__sec, desc);
	}

	return NULL;
}

/**
  * @brief  Get the size of the configure descriptor.
  * @param  config: Configure descriptor.
  * @retval Size of the descriptor.
  */
static  uint32_t usb_dcd_desc_size(const config_head_t *config)
{
	uint32_t i, len = 0;

	/* Calculate the total length of the descriptor */
	for (i = 0; i < config->nr_section; ++i)
		len += config->section[i]->size;

	return len;
}

/**
  * @brief  Get the number of the configure descriptor.
  * @param  config: Configure descriptor.
  * @param  type: Type of the descriptor.
  * @retval Number of the descriptor.
  */
static uint32_t usb_dcd_desc_num(const config_head_t *config, uint32_t type)
{
	uint32_t i, nr = 0;

	/* Calculate the numbers of the descriptor of the spicified type */
	for (i = 0; i < config->nr_section;  ++i)
		nr += usb_desc_get_nr((desc_head_t *)config->section[i]->data, config->section[i]->size, type);

	return nr;
}

/**
  * @brief  Get the descriptor.
  * @param  config: Configure descriptor.
  * @param  type: Type of the descriptor.
  * @param  idx: Index.
  * @param  sec: Section.
  * @retval Descriptor.
  */
static desc_head_t *usb_dcd_desc_get(const config_head_t *config, uint32_t type, uint32_t idx, uint32_t *sec)
{
	uint32_t i, nr, cnt = 0;

	/* Find the descriptor according the specified type */
	for (i = 0; i < (uint32_t)config->nr_section; ++i) {
		/* Get the numbers of the sections */
		nr = usb_desc_get_nr((desc_head_t *)config->section[i]->data, config->section[i]->size, type);
		/* Find the section */
		if ((cnt + nr) > idx) {
			/* Save the section */
			*sec = i;
			/* Find the specified descriptor */
			return usb_desc_get((desc_head_t *)config->section[i]->data,
							config->section[i]->size, type, idx - cnt);
		}

		/* Move to the next section */
		cnt += nr;
	}

	return NULL;
}

/**
  * @brief  Get the interface descriptor.
  * @param  config: Configure descriptor.
  * @param  idx: Index.
  * @param  alt: Alternate setting.
  * @param  sec: Section.
  * @retval Interface descriptor.
  */
static interface_desc_t *usb_dcd_interface_get(const config_head_t *config, uint32_t idx, uint32_t alt, uint32_t *sec)
{
	/* If ignore the alternate configuration */
	if (alt == USB_DESC_ANY)
		return (interface_desc_t *)usb_dcd_desc_get(config, USB_DTYPE_INTERFACE, idx, sec);
	else
		return get_alt_interface_from_config(config, idx, alt, sec);
}

/**
  * @brief  Get the FIFO size of the specified endpoint.
  * @param  max: Maximum size.
  * @param  used: Used size.
  * @retval Size.
  */
static uint32_t get_ep_fifo_size(uint32_t max, uint32_t *used)
{
	uint32_t i, size;

	/* Find the endpoint supported size */
	for (i = USB_FIFO_SZ_8; i <= USB_FIFO_SZ_2048; ++i) {
		size = 8 << i;

		/* Is the large enough */
		if (size >= max) {
			*used = size;
			return i;
		}
	}

	/* It can't support size requested */
	*used = 0;

	return USB_FIFO_SZ_8;
}

/**
  * @brief  Get the type of the specified endpoint.
  * @param  endpoint: Endpoint descriptor.
  * @param  idx: Index.
  * @param  max: Maximum size.
  * @param  flag: Return the type of the endpoint.
  * @retval None
  */
static void get_ep_desc_type(endpoint_desc_t *endpoint, uint32_t *idx, uint32_t *max, uint32_t *flag)
{
	/* Get the endpoint index */
	*idx  = endpoint->bEndpointAddress & USB_EP_DESC_NUM_M;
	/* Get the maximum packet size */
	*max  = endpoint->wMaxPacketSize & USB_EP_MAX_PACKET_COUNT_M;
	/* Get the direction */
	*flag = (endpoint->bEndpointAddress & USB_EP_DESC_IN) ? USB_EP_DEV_IN : USB_EP_DEV_OUT;

	/* Set the endpoint mode */
	switch (endpoint->bmAttributes & USB_EP_ATTR_TYPE_M) {
	case USB_EP_ATTR_CONTROL:
		*flag |= USB_EP_MODE_CTRL;
		break;
	case USB_EP_ATTR_BULK:
		*flag |= USB_EP_MODE_BULK;
		break;
	case USB_EP_ATTR_INT:
		*flag |= USB_EP_MODE_INT;
		break;
	case USB_EP_ATTR_ISOC:
		*flag |= USB_EP_MODE_ISOC;
		break;
	default:
		break;
	}
}

/**
  * @brief  Configure the USB controller appropriately.
  * @param  inst: Device instance.
  * @param  config: configuration descriptor.
  * @retval Status.
  */
static uint8_t usb_device_config(dcd_instance_t *inst, const config_head_t *config)
{
	uint32_t i, cnt, idx, type, max, flag;
	uint32_t nr_if, nr_ep, used, sec;
	interface_desc_t *interface;
	endpoint_desc_t *endpoint;
	usb_ep_info_t ep_info[NUM_USB_EP - 1];

	assert_param(inst);
	assert_param(config);

	/* Clear the infomation */
	for (i = 0; i < (NUM_USB_EP - 1); ++i) {
		ep_info[i].size[USB_EP_IN]  = 0;
		ep_info[i].size[USB_EP_OUT] = 0;
	}

	/* Get the number endpoints in the descriptor */
	nr_ep = usb_dcd_desc_num(config, USB_DTYPE_ENDPOINT);
	/* Get the number interfaces in the descriptor */
	nr_if = usb_dcd_desc_num(config, USB_DTYPE_INTERFACE);

	/* Configure the each endpoint */
	for (i = 0; i < nr_ep; ++i) {
		/* Get a pointer to the endpoint descriptor */
		endpoint = (endpoint_desc_t *)usb_dcd_desc_get(config, USB_DTYPE_ENDPOINT, i, &sec);
		/* Get the index endpoint */
		idx = (uint32_t)endpoint->bEndpointAddress & USB_EP_DESC_NUM_M;
		/* Get the direction */
		type = (endpoint->bEndpointAddress & USB_EP_DESC_IN) ? USB_EP_IN : USB_EP_OUT;

		/* Check the index */
		if ((idx >= NUM_USB_EP) || (idx == 0))
			return 1;

		/* Get the maximum packet size */
		if (endpoint->wMaxPacketSize > ep_info[idx - 1].size[type])
			ep_info[idx - 1].size[type] = endpoint->wMaxPacketSize;
	}

	/* Configure the each interface */
	for (i = 0; i < nr_if; ++i) {
		/* Get the next interface descriptor */
		interface = usb_dcd_interface_get(config, i, USB_DESC_ANY, &sec);

		/* If it's the default interface */
		if (interface && (interface->bAlternateSetting == 0)) {
			/* Get the number of the endpoint */
			nr_ep = (uint32_t)interface->bNumEndpoints;

			/* Configure the descriptor */
			for (cnt = 0; cnt < nr_ep; ++cnt) {
				/* Get a pointer to the endpoint descriptor */
				endpoint = usb_dcd_endpoint_get(config, interface->bInterfaceNumber,
								interface->bAlternateSetting, cnt);

				/* Make sure the pointer is not NULL */
				if (endpoint) {
					/* Get information for the descriptor */
					get_ep_desc_type(endpoint, &idx, &max, &flag);

					/* Check the endpoint is not EP0 */
					if (idx == 0)
						return 1;

					/* Configure the endpoint */
					map_usb_dev_ep_config(idx, max, flag);
				}
			}
		}
	}

	/* Get maximum size of EP0 */
	cnt = MAX_PACKET_SIZE_EP0;

	/* Configure the FIFO */
	for (i = 1; i < NUM_USB_EP; ++i) {
		/* Configure the IN endpoint */
		if (ep_info[i - 1].size[USB_EP_IN]) {
			/* Get the maximum size */
			max = get_ep_fifo_size(ep_info[i - 1].size[USB_EP_IN], &used);

			/* The FIFO space can be used? */
			if (used == 0)
				return 1;

			/* Configure the FIFO */
			map_usb_fifo_config_set(i, cnt, max, USB_EP_DEV_IN);
			cnt += used;
		}

		/* Configure the OUT endpoint */
		if (ep_info[i - 1].size[USB_EP_OUT]) {
			/* Get the maximum size */
			max = get_ep_fifo_size(ep_info[i - 1].size[USB_EP_OUT], &used);

			/* The FIFO space can be used? */
			if (used == 0)
				return 1;

			/* Configure the FIFO */
			map_usb_fifo_config_set(i, cnt, max, USB_EP_DEV_OUT);
			cnt += used;
		}
	}

	return 0;
}

/**
  * @brief  Configure the affected USB endpoints appropriately.
  * @param  inst: Device instance.
  * @param  config: configuration descriptor.
  * @param  nr: Number of the interface.
  * @param  alt: Alternate setting number.
  * @retval Status.
  */
static uint8_t usb_device_config_alt(dcd_instance_t *inst, const config_head_t *config, uint8_t nr, uint8_t alt)
{
	uint32_t i, cnt, nr_if, nr_ep,
		 max, flag, sec, idx;
	interface_desc_t *interface;
	endpoint_desc_t *endpoint;

	/* Get the number interfaces in the descriptor */
	nr_if = usb_dcd_desc_num(config, USB_DTYPE_INTERFACE);

	/* Find the interface descriptor and alternate setting numbers */
	for (i = 0; i < nr_if; ++i) {
		/* Get the next interface descriptor */
		interface = usb_dcd_interface_get(config, i, USB_DESC_ANY, &sec);

		/* Check if is the default interface */
		if (interface && (interface->bInterfaceNumber == nr) && (interface->bAlternateSetting == alt)) {
			/* Get the numbers of the endpoint */ 
			nr_ep = interface->bNumEndpoints;

			/* Configure each endpoint */
			for (cnt = 0; cnt < nr_ep; ++cnt) {
				/* Get a pointer to the endpoint */
				endpoint = usb_dcd_endpoint_get(config, interface->bInterfaceNumber,
								interface->bAlternateSetting, cnt);

				/* Check the pointer is not NULL */
				if (endpoint) {
					/* Get the information of the endpoint */
					get_ep_desc_type(endpoint, &idx, &max, &flag);

					/* Check the index of endpoint */
					if (idx == 0)
						return 1;

					/* Configure the endpoint */
					map_usb_dev_ep_config(idx, max, flag);
				}
			}

			return 0;
		}
	}

	return 1;
}

/**
  * @brief  Determines which string descriptor to send.
  * @param  lang: String language ID.
  * @param  idx: String descriptor index.
  * @retval Status.
  */
static int32_t usbd_string_idx_from_req(uint16_t lang, uint16_t idx)
{
	string0_desc_t *p;
	uint32_t nr_lang, nr_string, i;

	/* Ensure it has a string table */
	if((__dev_info[0] == 0) || (__dev_info[0]->string_desc == 0))
		return -1;

	/* Check the index */
	if (idx == 0)
		return 0;

	/* Get the number of the languages */
	nr_lang = (__dev_info[0]->string_desc[0][0] - 2) / 2;
	if (nr_lang == 0)
		return -1;

	/* Get the number of the string */
	nr_string = ((__dev_info[0]->nr_string_desc - 1) / nr_lang);
	/* Check the size */
	if ((1 + (nr_string * nr_lang)) != __dev_info[0]->nr_string_desc)
		return -1;

	/* Create a pointer to string descriptor */
	p = (string0_desc_t *)(__dev_info[0]->string_desc[0]);

	/* Find the position */
	for (i = 0; i < nr_lang; ++i) {
		if (p->wLANGID[i] == lang)
			return (nr_string * i) + idx;
	}

	return -1;
}

/**
  * @brief  Send data on endpoint zero.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
static void usbd_ep0_tx(uint32_t idx)
{
	uint8_t *buf;
	uint32_t len;

	assert_param(idx == 0);

	/* Set EP0 state as TX */
	__dcd_inst[0].ep0_state = USB_EP0_STATE_TX;
	/* Set the length */
	len = __dcd_inst[0].ep0_remain;
	/* Adjust the length */
/**issue**/	len = len > EP0_MAX_PACKET_SIZE ? EP0_MAX_PACKET_SIZE : len;
	/* Create a pointer to the data */
	buf = (uint8_t *)__dcd_inst[0].ep0_data;

	/* Update the data pointer and position */
	__dcd_inst[0].ep0_remain -= len;
	__dcd_inst[0].ep0_data   += len;
	/* Put the data into the FIFO */
	map_usb_ep_data_put(USB_EP_0, buf, len);

/**issue**/
	/* Check the length */
	if (len == EP0_MAX_PACKET_SIZE) {
		/* Send the data */
		map_usb_ep_data_send(USB_EP_0, USB_TRANS_IN);
	}
	else {
		/* Set the EP0 state as STATUS */
		__dcd_inst[0].ep0_state = USB_EP0_STATE_STATUS;
		/* Send last packet */
		map_usb_ep_data_send(USB_EP_0, USB_TRANS_IN_LAST);

		/* Call callback function */
		if ((__dev_info[0]->cbk->data_send) && (__dcd_inst[0].size_out != 0)) {
			__dev_info[0]->cbk->data_send(__dcd_inst[0].arg, __dcd_inst[0].size_out);
			__dcd_inst[0].size_out = 0;
		}
	}
}

/**
  * @brief  Send the configuration descriptor on endpoint zero.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
static void usbd_ep0_tx_config(uint32_t idx)
{
	uint8_t *buf;
	uint32_t len, slen, __len;
	config_desc_t desc;
	const config_head_t *hdr;
	const config_section_t *sec;

	assert_param(idx == 0);

	/* Set the EP0 state as TX_CONFIG */
	__dcd_inst[0].ep0_state = USB_EP0_STATE_TX_CONFIG;
	/* Create a pointer to configuration descriptor */
	hdr = __dev_info[0]->config_desc[__dcd_inst[0].config_idx];
	/* Set the length */
	len = __dcd_inst[0].ep0_remain;
	/* Adjust the length */
	len = len > EP0_MAX_PACKET_SIZE ? EP0_MAX_PACKET_SIZE : len;

	/* If this is the first call */
	if ((__dcd_inst[0].offset_sec == 0) && (__dcd_inst[0].config_sec == 0)) {
		/* Set the pointer */
		desc = *(config_desc_t *)__dcd_inst[0].ep0_data;
		/* Update the length */
		desc.wTotalLength = (uint16_t)usb_dcd_desc_size(hdr);
		/* Adjust the size */
		__len = len < sizeof(config_desc_t) ? len : sizeof(config_desc_t);
		/* Put the data into FIFO */
		map_usb_ep_data_put(USB_EP_0, (uint8_t *)&desc, __len);

		/* If it's the end of the first section */
		if (hdr->section[0]->size == __len) {
			/* Update the pointer to the next section */
			__dcd_inst[0].offset_sec = 0;
			__dcd_inst[0].config_sec = 1;
		}
		else {
			/* It has sent some bytes of the descriptor */
			__dcd_inst[0].offset_sec = (uint8_t)__len;
		}

		/* Update the size */
		__len = len - __len;
	}
	else {
		/* Update the size */
		__len = len;
	}

	/* Until the sending process is complete */
	while (__len) {
		/* Create the pointer to current section */
		sec  = hdr->section[__dcd_inst[0].config_sec];
		/* Calculate the position */
		slen = (uint32_t)(sec->size - __dcd_inst[0].offset_sec);
		/* Save the position */
		buf  = (uint8_t *)sec->data + __dcd_inst[0].offset_sec;

		/* Adjust the size */
		if (slen > __len)
			slen = __len;

		/* Put the data into the FIFO */
		map_usb_ep_data_put(USB_EP_0, buf, slen);

		/* Fix up pointer for next iteration */
		__len -= slen;
		__dcd_inst[0].offset_sec += (uint8_t)slen;

		/* If it's the end of the section */
		if (__dcd_inst[0].offset_sec == sec->size){
			/* Move to the next section */
			__dcd_inst[0].config_sec++;
			__dcd_inst[0].offset_sec = 0;
		}
	}

	/* Update the remain pointer */
	__dcd_inst[0].ep0_remain -= len;

	/* Check the number of the section */
	if (hdr->nr_section <= __dcd_inst[0].config_sec)
		__dcd_inst[0].ep0_remain = 0;

	/* If there is no more data don't keep looking */
	if (__dcd_inst[0].ep0_remain != 0){
		buf = (uint8_t *)hdr->section[__dcd_inst[0].config_sec]->data;
		__len = __dcd_inst[0].offset_sec;
		__dcd_inst[0].ep0_data = (buf + __len);
	}

	/* If it's the last packet */
	if (len == EP0_MAX_PACKET_SIZE) {
		/* Send the data */
		map_usb_ep_data_send(USB_EP_0, USB_TRANS_IN);
	}
	else {
		/* Send the last packet */
		map_usb_ep_data_send(USB_EP_0, USB_TRANS_IN_LAST);
		
		/* Call the callback function */
		if ((__dev_info[0]->cbk->data_send) && (__dcd_inst[0].size_out != 0)) {
			__dev_info[0]->cbk->data_send(__dcd_inst[0].arg, __dcd_inst[0].size_out);
			__dcd_inst[0].size_out = 0;
		}

		/* Set the EP0 state as STATUS */
		__dcd_inst[0].ep0_state = USB_EP0_STATE_STATUS;
	}

	return;
}

/**
  * @brief  Handle the GET_STATUS standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_get_status(void *inst, usb_request_t *req)
{
	uint16_t data, idx;
	uint32_t dir;
	dcd_instance_t *__inst;

	assert_param(req != 0);
	assert_param(inst != 0);

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, false);

	switch (req->bmRequestType & USB_RTYPE_RECIPIENT_M) {
	case USB_RTYPE_DEVICE:	/* Handle the device status request */
		/* Return the current status */
		data = __inst->status;
		break;

	case USB_RTYPE_INTERFACE:	/* Handle the interface status request */
		/* Return 0 */
		data = 0;
		break;

	case USB_RTYPE_ENDPOINT:	/* Handle the endpoint status request */
		/* Get the index of the endpoint */
		idx = req->wIndex & USB_REQ_EP_NUM_M;

		/* Check the index */
		if ((idx == 0) || (idx >= NUM_USB_EP)) {
			usb_dcd_ep0_stall(0);
			return;
		}
		else {
			/* Get the direction */
			dir = ((req->wIndex & USB_REQ_EP_DIR_M) ==
					USB_REQ_EP_DIR_IN) ? USB_EP_IN : USB_EP_OUT;
			/* Get the current halt status */
			data = __inst->halt[dir][idx - 1];
		}
		break;

	default:	/* Handle an unknown request */
		/* Stall the EP0 */
		usb_dcd_ep0_stall(0);
		return;
	}

	/* Send the status response */
	__inst->ep0_remain = 2;
	__inst->ep0_data   = (uint8_t *)&data;
	usbd_ep0_tx(0);

	return;
}

/**
  * @brief  Handle the CLEAR_FEATURE standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_clear_feature(void *inst, usb_request_t *req)
{
	uint16_t idx;
	uint32_t dir;
	dcd_instance_t *__inst;

	assert_param(req != 0);
	assert_param(inst != 0);

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, true);

	switch (req->bmRequestType & USB_RTYPE_RECIPIENT_M) {
	case USB_RTYPE_DEVICE:	/* Handle the device */
		/* We only clear remote wakeup */
		if(USB_FEATURE_REMOTE_WAKE & req->wValue)
			__inst->status &= ~USB_STATUS_REMOTE_WAKE;
		else
			usb_dcd_ep0_stall(0);

		break;
	case USB_RTYPE_ENDPOINT:	/* Handle the endpoint */
		/* Get the index of the endpoint */
		idx = req->wIndex & USB_REQ_EP_NUM_M;

		/* Check the index */
		if ((idx == 0) || (idx > NUM_USB_EP)) {
			usb_dcd_ep0_stall(0);
			return;
		}

		/* Only the halt feature is supported */
		if (USB_FEATURE_EP_HALT != req->wValue) {
			usb_dcd_ep0_stall(0);
			return;
		}

		/* Get the direction */
		dir = ((req->wIndex & USB_REQ_EP_DIR_M) ==
				USB_REQ_EP_DIR_IN) ? USB_EP_IN : USB_EP_OUT;
		/* Clear the halt condition */
		__inst->halt[dir][idx - 1] = 0;

		if (dir == USB_EP_IN)
			map_usb_dev_ep_stall_clear(idx, USB_EP_DEV_IN);
		else
			map_usb_dev_ep_stall_clear(idx, USB_EP_DEV_OUT);
		break;

	default:	/* Handle an nuknown request */
		/* Stall the EP0 */
		usb_dcd_ep0_stall(0);
		return;
	}

	return;
}

/**
  * @brief  Handle the SET_FEATURE standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_set_feature(void *inst, usb_request_t *req)
{
	uint16_t idx;
	uint32_t dir;
	dcd_instance_t *__inst;

	assert_param(req != 0);
	assert_param(inst != 0);

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, true);

	switch (req->bmRequestType & USB_RTYPE_RECIPIENT_M) {
	case USB_RTYPE_DEVICE:	/* Handle the device */
		/* We only clear remote wakeup */
		if(USB_FEATURE_REMOTE_WAKE & req->wValue)
			__inst->status |= USB_STATUS_REMOTE_WAKE;
		else
			usb_dcd_ep0_stall(0);
		break;

	case USB_RTYPE_ENDPOINT:	/* Handle the endpoint */
		/* Get the index of the endpoint */
		idx = req->wIndex & USB_REQ_EP_NUM_M;

		/* Check the index */
		if ((idx == 0) || (idx >= NUM_USB_EP)) {
			usb_dcd_ep0_stall(0);
			return;
		}

		/* Only the halt feature is supported */
		if (USB_FEATURE_EP_HALT != req->wValue) {
			usb_dcd_ep0_stall(0);
			return;
		}

		/* Get the direction */
		dir = ((req->wIndex & USB_REQ_EP_DIR_M) ==
				USB_REQ_EP_DIR_IN) ? USB_EP_IN : USB_EP_OUT;
		/* Clear the halt condition */
		__inst->halt[dir][idx - 1] = 1;
		break;

	default:	/* Handle an nuknown request */
		/* Stall the EP0 */
		usb_dcd_ep0_stall(0);
		return;
	}

	return;
}

/**
  * @brief  Handle the SET_ADDRESS standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_set_address(void *inst, usb_request_t *req)
{
	dcd_instance_t *__inst;

	assert_param(req != 0);
	assert_param(inst != 0);

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, true);
	/* Save the device address */
	__inst->dev_addr = req->wValue | DEV_ADDR_PENDING;
	/* Set the state as status */
	__inst->ep0_state = USB_EP0_STATE_STATUS;

	return;
}

/**
  * @brief  Handle the GET_DESCRIPTOR standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_get_desc(void *inst, usb_request_t *req)
{
	uint8_t conf = 0, idx;
	int32_t __idx;
	dcd_instance_t *__inst;
	device_info_t *dev;
	const config_head_t *config;
	const device_desc_t *dev_desc;

	assert_param(req != 0);
	assert_param(inst != 0);

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	dev = __dev_info[0];
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, false);

	switch (req->wValue >> 8) {
	case USB_DTYPE_DEVICE:	/* device_desc_t descriptor request */
		/* Return the device descriptor */
		__inst->ep0_data   = (uint8_t *)dev->device_desc;
		/* Set the remain size */
		__inst->ep0_remain = dev->device_desc[0];

		break;
	case USB_DTYPE_CONFIGURATION:	/* Configuration descriptor request */
		/* Get the index of the configuration */
		idx = (uint8_t)(req->wValue & 0xFF);
		/* Create a pointer */
		dev_desc = (const device_desc_t *)dev->device_desc;

		/* Check the index */
		if (idx >= dev_desc->bNumConfigurations) {
			/* Invalid configuration index */
			usb_dcd_ep0_stall(0);
			__inst->ep0_data   = 0;
			__inst->ep0_remain = 0;
		}
		else {
			/* Get the specified configuration descriptor */
			config = dev->config_desc[idx];

			__inst->config_sec = 0;
			__inst->offset_sec = 0;
			__inst->config_idx = idx;
			/* Set the sending data address */
			__inst->ep0_data   = (uint8_t *)config->section[0]->data;
			/* Set the remain size */
			__inst->ep0_remain = usb_dcd_desc_size(config);
			conf = 1;
			
			printf_e("data at 0x%X, length = %d\r\n", __inst->ep0_data, __inst->ep0_remain);
			printf_e("0x%X 0x%X 0x%X 0x%X 0x%X 0x%X\r\n", *(__inst->ep0_data), *(__inst->ep0_data + 1), *(__inst->ep0_data + 2), 
															*(__inst->ep0_data + 3), *(__inst->ep0_data + 4), *(__inst->ep0_data + 5));
		}

		break;
	case USB_DTYPE_STRING:	/* String descriptor request */
		/* Get the index of the string descriptor */
		__idx = usbd_string_idx_from_req(req->wIndex, req->wValue & 0xFF);

		/* Check the index */
		if (__idx == -1) {
			usb_dcd_ep0_stall(0);
			break;
		}

		/* Set the sending data address */
		__inst->ep0_data   = (uint8_t *)dev->string_desc[__idx];
		/* Set the remain size */
		__inst->ep0_remain = dev->string_desc[__idx][0];

		break;

	default:	/* Other request is not handled in this function */
		/* Call the callback function */
		if(dev->cbk->get_desc)
			dev->cbk->get_desc(__dcd_inst[0].arg, req);
		else
			usb_dcd_ep0_stall(0);

		return;
	}

	/* Send the data for this request */
	if (__inst->ep0_data) {
		/* Adjust the total size */
		if (__inst->ep0_remain > req->wLength)
			__inst->ep0_remain = req->wLength;

		/* If it's a configuration descriptor or not? */
		if (conf)
			usbd_ep0_tx_config(0);
		else
			usbd_ep0_tx(0);
	}
}

/**
  * @brief  Handle the SET_DESCRIPTOR standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_set_desc(void *inst, usb_request_t *req)
{
	/* ACK the EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, false);
	/* ACK the data on EP0 */
	usb_dcd_ep0_stall(0);

	return;
}

/**
  * @brief  Handle the GET_CONFIGURATION standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_get_config(void *inst, usb_request_t *req)
{
	uint8_t value;
	dcd_instance_t *__inst;

	assert_param(req != 0);
	assert_param(inst != 0);

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, 0);

	/* If it's a pending address, then the device is not configured */
	if (__inst->dev_addr & DEV_ADDR_PENDING)
		value = 0;
	else
		value = (uint8_t)__inst->config;

	/* Send the response */
	__inst->ep0_data   = &value;
	__inst->ep0_remain = 1;
	usbd_ep0_tx(0);

	return;
}

/**
  * @brief  Handle the SET_CONFIGURATION standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_set_config(void *inst, usb_request_t *req)
{
	dcd_instance_t *__inst;
	device_info_t *dev;
	const config_head_t *hdr;
	const config_desc_t *desc;

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	dev    = __dev_info[0];
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, true);

	/* Check the device descriptor */
	if (req->wValue > dev->device_desc[17]) {
		usb_dcd_ep0_stall(0);
		return;
	}

	/* Save the configuration */
	__inst->config = req->wValue;

	if (__inst->config) {
		/*  Get a pointer to the configuration descriptor */
		hdr = dev->config_desc[req->wValue - 1];
		desc = (const config_desc_t *)(hdr->section[0]->data);

		/* Remember the new self- or bus-powered state if the user has not
		 * already called us to tell us the state to report.
		 */
		if (!__inst->power_set) {
			if ((desc->bmAttributes & USB_CONF_ATTR_PWR_M) == USB_CONF_ATTR_SELF_PWR)
				__inst->status |= USB_STATUS_SELF_PWR;
			else
				__inst->status &= ~USB_STATUS_SELF_PWR;
		}

		/* Configure the new descriptor */
		usb_device_config(__inst, dev->config_desc[req->wValue - 1]);
		printf_e("\rSet configure: %d\n", __inst->config);
	}

	/* Call the callback function */
	if (dev->cbk->config_change)
		dev->cbk->config_change(__dcd_inst[0].arg, __inst->config);

	return;
}

/**
  * @brief  Handle the GET_INTERFACE standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_get_interface(void *inst, usb_request_t *req)
{
	uint8_t value;
	dcd_instance_t *__inst;

	assert_param(req != 0);
	assert_param(inst != 0);

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, false);

	/* Check the devie address */
	if (__inst->dev_addr & DEV_ADDR_PENDING) {
		value = 0;
	}
	else {
		/* Check the interface number */
		if (req->wIndex < USB_MAX_INTERFACES_PER_DEVICE){
			/* Get the current alternate setting */
			value = __inst->alt[req->wIndex];
		}
		else {
			/* Stall the EP0 */
			usb_dcd_ep0_stall(0);
			return;
		}
	}

	/* Send the response */
	__inst->ep0_data   = &value;
	__inst->ep0_remain = 1;
	usbd_ep0_tx(0);

	return;
}

/**
  * @brief  Handle the SET_INTERFACE standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_set_interface(void *inst, usb_request_t *req)
{
	uint8_t idx, ret;
	uint32_t i, sec, nr;
	const config_head_t *hdr;
	interface_desc_t *desc;
	dcd_instance_t *__inst;
	device_info_t *dev;

	assert_param(req != 0);
	assert_param(inst != 0);

	/* Create the device infomation pointer */
	__inst = (dcd_instance_t *)inst;
	dev    = __dev_info[0];
	hdr    = dev->config_desc[__inst->config - 1];
	nr     = usb_dcd_desc_num(hdr, USB_DTYPE_INTERFACE);
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, true);

	/* Find the specified interface descriptor */
	for (i = 0; i < nr; ++i) {
		/* Get the next interface descriptor */
		desc = usb_dcd_interface_get(hdr, i, USB_DESC_ANY, &sec);

		/* If find it */
		if (desc && (desc->bInterfaceNumber == req->wIndex) && (desc->bAlternateSetting == req->wValue)) {
			idx = desc->bInterfaceNumber;

			assert_param(idx < USB_MAX_INTERFACES_PER_DEVICE);

			/* Save the setting */
			__inst->alt[idx] = desc->bAlternateSetting;
			/* Reconfigure the endpoints to match the request */
			ret = usb_device_config_alt(__inst, hdr, idx, desc->bAlternateSetting);
			/* Call callback function */
			if (ret == 0 && dev->cbk->interface_change)
				dev->cbk->interface_change(__dcd_inst[0].arg, req->wIndex, req->wValue);

			return;
		}
	}

	/* If we drop out the request, need to stall the EP0 */
	usb_dcd_ep0_stall(0);
	return;
}

/**
  * @brief  Handle the SYNC_FRAME standard USB request.
  * @param  inst: Device instance.
  * @param  req: Standard USB request.
  * @retval None
  */
static void usbd_sync_frame(void *inst, usb_request_t *req)
{
	/* ACK the data on EP0 */
	map_usb_dev_ep_data_ack(USB_EP_0, true);
	/* Stall the EP0 */
	usb_dcd_ep0_stall(0);

	return;
}

/**
  * @brief  SOF interrupt to process any outstanding remote wake up requests.
  * @param  inst: Device instance.
  * @retval None
  */
static void usbd_tick_resume_handler(dcd_instance_t *inst)
{
	if (__dcd_inst[0].remote_wakeup == 0)
		return;

	/* Increment the counter */
	++__dcd_inst[0].cnt_wakeup;

	/* If reached the 10ms, it needs to send the resume signal */
	if (__dcd_inst[0].cnt_wakeup == REMOTE_WAKEUP_PULSE_MS)
		map_usb_host_resume(0);

	/* It reached the point at which it can tell the bus has resumed? */
	if (__dcd_inst[0].cnt_wakeup == REMOTE_WAKEUP_READY_MS) {
		__dcd_inst[0].remote_wakeup = 0;

		/* Call the resume callback function */
		if (__dev_info[0]->cbk->resume_handler)
			__dev_info[0]->cbk->resume_handler(__dcd_inst[0].arg);
	}
}

/**
  * @brief  Read a request packet and dispatches it to a standard request handler.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
static void usbd_distribute_request(uint32_t idx)
{
	uint32_t len;
	usb_request_t *req;

	req = (usb_request_t *)__data_in;
	len = EP0_MAX_PACKET_SIZE;
	/* Get the data from the EP0 */
	map_usb_ep_data_get(USB_EP_0, __data_in, &len);
	
	printf_e("request: 0x%.2x, 0x%.4X, len = %d\r\n", req->bRequest, req->wValue, len);
	
	if(len == 0)
		return;
	
	/* If it is a standard request or not? */
	if ((req->bmRequestType & USB_RTYPE_TYPE_M) != USB_RTYPE_STANDARD) {
		if(__dev_info[0]->cbk->request_handler)
			__dev_info[0]->cbk->request_handler(__dcd_inst[0].arg, req);
		else
			usb_dcd_ep0_stall(0);
	}
	else {
		/* Jump the table to the appropriate handler */
		if ((req->bRequest < (sizeof(__usb_std_request) / sizeof(std_request)))
						&& (__usb_std_request[req->bRequest] != 0))
			__usb_std_request[req->bRequest](&__dcd_inst[0], req);
		else
			usb_dcd_ep0_stall(0);
	}
}

/**
  * @brief  Interrupt handler for endpoint zero.
  * @param  inst: Device instance.
  * @retval None
  */
static void usbd_enum_handler(dcd_instance_t *inst)
{
	uint32_t status, len;

	/* Get the EP0 status */
	status = map_usb_ep_status(USB_EP_0);

	switch(inst->ep0_state) {
	case USB_EP0_STATE_STATUS:	/* Handle the state status */
		printf_e("state status: 0x%.2x\r\n", status);
		/* Set the state as IDLE */
		inst->ep0_state = USB_EP0_STATE_IDLE;

		/* Check there is a pending address */
		if (inst->dev_addr & DEV_ADDR_PENDING) {
			/* Clear the pending address, and then set the address */
			inst->dev_addr &= ~DEV_ADDR_PENDING;
			map_usb_dev_set_addr(inst->dev_addr);
			printf_e("address: 0x%x\r\n", inst->dev_addr);
			//need to send 0 size pack first
		}
		/* If a new packet is arrived, we need to read it */
		if (status & USB_DEV_EP0_OUT_PKTRDY)
			usbd_distribute_request(0);

		break;
	case USB_EP0_STATE_IDLE:	/* Handle the IDLE state */
		printf_e("state idle: 0x%.2x\r\n", status);
		/* If a new packet is arrived, we need to read it */
		if(status & USB_DEV_EP0_OUT_PKTRDY)
			usbd_distribute_request(0);

		break;
	case USB_EP0_STATE_TX:	/* Handle the TX state */
		printf_e("state tx\r\n");
		usbd_ep0_tx(0);
		break;

	case USB_EP0_STATE_TX_CONFIG:	/* Handle the RX_CONFIG state */
		printf_e("state tx config\r\n");
		usbd_ep0_tx_config(0);
		break;

	case USB_EP0_STATE_RX:	/* Handle the RX state */
		printf_e("state rx\r\n");
		/* Set the number of bytes to get out of this next packet */
		if (inst->ep0_remain > EP0_MAX_PACKET_SIZE)
			len = EP0_MAX_PACKET_SIZE;
		else
			len = inst->ep0_remain;

		/* Get the data from EP0 */
		map_usb_ep_data_get(USB_EP_0, inst->ep0_data, &len);

		/* If there we not more that EP0_MAX_PACKET_SIZE or more bytes
		 * remaining then this transfer is complete.  If there were exactly
		 * EP0_MAX_PACKET_SIZE remaining then there still needs to be
		 * null packet sent before this is complete.
		 */
		if (inst->ep0_remain < EP0_MAX_PACKET_SIZE) {
			inst->ep0_state = USB_EP0_STATE_STATUS;

			/* Call the receive callback function */
			if ((__dev_info[0]->cbk->data_recv) && (inst->size_out != 0)) {
				__dev_info[0]->cbk->data_recv(__dcd_inst[0].arg, inst->size_out);
				inst->size_out = 0;
			}

			/* ACK the data on EP0, this is the last packet */
			map_usb_dev_ep_data_ack(USB_EP_0, 1);
		}
		else {
			/* ACK the data on EP0, more data is coming */
			map_usb_dev_ep_data_ack(USB_EP_0, 0);
		}

		inst->ep0_data   += len;
		inst->ep0_remain -= len;

		break;
	case USB_EP0_STATE_STALL:	/* Handle the STALL state */
		/* If we send a stall signal, then entry this interrupt */
		if (status & USB_DEV_EP0_SENT_STALL) {
			/* Clear the EP0 status */
			map_usb_dev_ep_status_clear(USB_EP_0, USB_DEV_EP0_SENT_STALL);
			inst->ep0_state = USB_EP0_STATE_IDLE;
		}

		break;
	default:
		break;
	}

	return;
}

/**
  * @brief  Handle bus reset notifications.
  * @param  inst: Device instance.
  * @retval None
  */
static void usbd_enum_reset_handler(dcd_instance_t *inst)
{
	uint32_t i;

	/* Disable remote wakeup signal */
	inst->status &= ~USB_STATUS_REMOTE_WAKE;
	inst->remote_wakeup = 0;

	/* Call the reset callback function */
	if (__dev_info[0]->cbk->reset_handler)
		__dev_info[0]->cbk->reset_handler(__dcd_inst[0].arg);

	/* Default the configuration identifier */
	inst->config = inst->config_default;

	for (i = 0; i < USB_MAX_INTERFACES_PER_DEVICE; ++i)
		inst->alt[i] = 0x0;
	
	return;
}
/**
  * @}
  */

/** @defgroup Device_Core_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initialize an instance of the __dcd_inst.
  * @param  idx: Index of the USB controller.
  * @param  dev: Device information.
  * @retval None
  */
void usb_device_info_init(uint32_t idx, device_info_t *dev)
{
	assert_param(idx == 0);
	assert_param(dev);

	__dcd_inst[0].lpm_state      = 0;
	__dcd_inst[0].config         = DEFAULT_CONFIG_ID;
	__dcd_inst[0].config_default = DEFAULT_CONFIG_ID;
	__dcd_inst[0].ep0_state      = USB_EP0_STATE_IDLE;
	__dcd_inst[0].status         = 0;
	__dcd_inst[0].remote_wakeup  = 0;
	__dcd_inst[0].power_set      = 0;

	return;
}

/**
  * @brief  Initialize the USB device control driver for the USB controller.
  * @param  idx: Index of the USB controller.
  * @param  dev: Device information.
  * @param  arg: Parameter.
  * @retval None
  */
void usb_dcd_init(uint32_t idx, device_info_t *dev, void *arg)
{
	const config_head_t *hdr;
	const config_desc_t *desc;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(__usb_mode != USB_LIB_MODE_HOST);

	__dev_info[0]     = dev;
	__dcd_inst[0].arg = arg;
	
	/* Initialize the device infomation structure */
	usb_device_info_init(idx, dev);

	/* Default the device mode */
	if (__usb_mode == USB_LIB_MODE_NONE)
		__usb_mode = USB_LIB_MODE_DEVICE;

	if (__usb_mode != USB_LIB_MODE_OTG) {
		__usb_mode = USB_LIB_MODE_DEVICE;
		
		map_usb_device_components_init();
		
		if (__dcd_inst[0].feature & USB_FEATURE_LPM_EN) {
			map_usb_dev_lpm_config(USB_DEV_LPM_EN);
			map_usb_lpm_int_enable(USB_INTLPM_RESUME | USB_INTLPM_ERROR |
					   USB_INTLPM_ACK | USB_INTLPM_NYET);
			map_usb_dev_lpm_enable();
			__dcd_inst[0].lpm_state = USBLIB_LPM_STATE_AWAKE;
		}
		else {
			map_usb_dev_lpm_disable();
			map_usb_dev_lpm_config(USB_DEV_LPM_NONE);
			__dcd_inst[0].lpm_state = USBLIB_LPM_STATE_DISABLED;
		}
	}

	/* Initialize the USB dma module */
	usb_dma_init(idx);
	/* Initialize the tick module */
	usb_tick_init();

	/* Get a pointer to the configuration descriptor */
	hdr = dev->config_desc[__dcd_inst[0].config_default - 1];
	desc = (const config_desc_t *)(hdr->section[0]->data);

	if ((desc->bmAttributes & USB_CONF_ATTR_PWR_M) == USB_CONF_ATTR_SELF_PWR)
		__dcd_inst[0].status |= USB_STATUS_SELF_PWR;
	else
		__dcd_inst[0].status &= ~USB_STATUS_SELF_PWR;

	/* Get the current interrupt status and clear all interrupts */
	map_usb_int_status_get();
	map_usb_int_status_ep_get();
	/* Enable USB interrupts */
	map_usb_int_enable(USB_INTCTRL_RESET | USB_INTCTRL_DISCONNECT | USB_INTCTRL_RESUME |
						USB_INTCTRL_SUSPEND | USB_INTCTRL_SOF);
	map_usb_int_enable_ep(USB_INTEP_ALL);
	/* Enable software connect */
	map_usb_dev_connect();
	/* Enable USB interrupts */
	map_usb_int_register();

	return;
}

/**
  * @brief  Release the USB device control driver for the USB controller.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
void usb_dcd_term(uint32_t idx)
{
	assert_param(idx == 0);

	/* Disable interrupts */
	map_usb_int_unregister();
	/* Reset tick module */
	usb_tick_reset();
	/* Empty active device */
	__dev_info[0] = 0;
	/* Disable interrupt */
	map_usb_int_disable(USB_INTCTRL_ALL);
	map_usb_int_disable_ep(USB_INTEP_ALL);
	/* Disable software connect */
	map_usb_dev_disconnect();
	/* Clear all pending interrupts */
	map_usb_int_status_get();
	map_usb_int_status_ep_get();

#ifdef USB_DRIVER_MD
	map_usb_controller_reset();
	map_usb_clk_phy_disable();
	map_usb_controller_disable();
#endif
	return;
}

/**
  * @brief  Generate a stall condition on endpoint zero.
  * @param  idx: Index of the USB controller.
  * @retval None
  */
void usb_dcd_ep0_stall(uint32_t idx)
{
	assert_param(idx == 0);

	/* Stall the endpoint */
	map_usb_dev_ep_stall(USB_EP_0, USB_EP_DEV_OUT);
	/* Set state as stall */
	__dcd_inst[0].ep0_state = USB_EP0_STATE_STALL;
}

/**
  * @brief  Start the request for data from the host on endpoint zero.
  * @param  idx: Index of the USB controller.
  * @param  data: Data buffer.
  * @param  size: Size of the buffer.
  * @retval None
  */
void usb_dcd_ep0_data_req(uint32_t idx, uint8_t *data, uint32_t size)
{
	assert_param(idx == 0);

	__dcd_inst[0].ep0_state  = USB_EP0_STATE_RX;
	__dcd_inst[0].ep0_data   = data;
	__dcd_inst[0].size_out   = size;
	__dcd_inst[0].ep0_remain = size;

	return;
}

/**
  * @brief  Request transfer of data to the host on endpoint zero.
  * @param  idx: Index of the USB controller.
  * @param  data: Data buffer.
  * @param  size: Size of the buffer.
  * @retval None
  */
void usb_dcd_ep0_data_send(uint32_t idx, uint8_t *data, uint32_t size)
{
	assert_param(idx == 0);

	__dcd_inst[0].ep0_data   = data;
	__dcd_inst[0].size_out   = size;
	__dcd_inst[0].ep0_remain = size;
	usbd_ep0_tx(0);

	return;
}

/**
  * @brief  Get a endpoint descriptor from particular configuration descriptor.
  * @param  config: Configuration descriptor.
  * @param  nr: Interface number.
  * @param  alt: Alternate setting number.
  * @param  idx: Index of the endpoint.
  * @retval Endpoint descriptor.
  */
endpoint_desc_t *usb_dcd_endpoint_get(const config_head_t *config, uint32_t nr, uint32_t alt, uint32_t idx)
{
	interface_desc_t *interface;
	desc_head_t *endpoint;
	uint32_t sec, cnt = 0;

	/* Find the requested interface descriptor */
	interface = usb_dcd_interface_get(config, nr, alt, &sec);

	/* Check the pointer is not empty */
	if (interface == NULL)
		return NULL;
	/* Check the index of the endpoint */
	if (idx >= interface->bNumEndpoints)
		return NULL;

	/* Create a pointer to the interface */
	endpoint = (desc_head_t *)interface;

	while (endpoint) {
		if (endpoint->bDescriptorType == USB_DTYPE_ENDPOINT) {
			/* Find the endpoint descriptor */
			if (cnt++ == idx)
				return ((endpoint_desc_t *)endpoint);
		}

		/* Move to the next descriptor */
		endpoint = get_next_config_desc(config, &sec, endpoint);
	}

	return NULL;

}

/**
  * @brief  Set default configuration descriptor.
  * @param  idx: Index of the USB controller.
  * @param  config: Configuration descriptor.
  * @retval Endpoint descriptor.
  */
void usb_dcd_config_set_default(uint32_t idx, uint32_t config)
{
    assert_param(idx == 0);

    __dcd_inst[0].config_default = config;
}

/**
  * @brief  Reports the device power status.
  * @param  idx: Index of the USB controller.
  * @param  status: Indicate the current power status.
  * @retval None
  */
void usb_dcd_power_status_set(uint32_t idx, uint8_t status)
{
	assert_param(idx == 0);

	if ((status != USB_STATUS_BUS_PWR) && (status != USB_STATUS_SELF_PWR))
		return;

	/* Update the device status */
	__dcd_inst[0].power_set = 1;
	__dcd_inst[0].status &= ~USB_STATUS_PWR_M;
	__dcd_inst[0].status |= status;
}

/**
  * @brief  Request a remote wake up to resume communication.
  * @param  idx: Index of the USB controller.
  * @retval Status.
  */
uint8_t usb_dcd_remote_wakeup_req(uint32_t idx)
{
	assert_param(idx == 0);

	if (__dcd_inst[0].status & USB_STATUS_REMOTE_WAKE) {
		if (!__dcd_inst[0].remote_wakeup) {
			__dcd_inst[0].cnt_wakeup    = 0;
			__dcd_inst[0].remote_wakeup = 1;
			map_usb_host_resume(1);
			return 0;
		}
	}

	return 1;
}

/**
  * @brief  Request an LPM remote wake up to resume communication.
  * @param  idx: Index of the USB controller.
  * @retval Status.
  */
uint8_t usb_dcd_remote_wakeup_lpm(uint32_t idx)
{
	assert_param(idx == 0);

	if (map_usb_lpm_remote_wake_is_enable()) {
		map_usb_dev_lpm_remote_wake();
		return 0;
	}

	return 1;
}

/**
  * @brief  Enable/Disable features of the USB stack.
  * @param  idx: Index of the USB controller.
  * @param  feature: Feature.
  * @param  arg: Parameter.
  * @retval Status.
  */
uint8_t usb_dcd_feature_set(uint32_t idx, uint32_t feature, void *arg)
{
	uint8_t ret = 0;
	lpm_feature_t *lpm_feature;

	switch(feature) {
	case USB_FEATURE_LPM:
		/* Save LPM setting */
		lpm_feature = (lpm_feature_t *)arg;

		if (lpm_feature->feature & USB_FEATURE_LPM_EN)
			__dcd_inst[0].feature |= USB_FEATURE_LPM_EN;
		else
			__dcd_inst[0].feature &= ~USB_FEATURE_LPM_EN;

		break;
	case USB_FEATURE_POWER:
		/* Update the device status */
		__dcd_inst[0].power_set = 1;
		__dcd_inst[0].status &= ~USB_FEATURE_POWER_SELF;
		__dcd_inst[0].status |= (uint8_t)(*(uint32_t *)arg);

		break;
	default:
		ret = 1;
		break;
	}

	return ret;
}

/**
  * @brief  Get features of the USB stack.
  * @param  idx: Index of the USB controller.
  * @param  feature: Feature.
  * @param  arg: Parameter.
  * @retval Always 0.
  */
uint8_t usb_dcd_feature_get(uint32_t idx, uint32_t feature, void *arg)
{
    return 0;
}

/**
  * @brief  USB device interrupt handler.
  * @param  idx: Index of the USB controller.
  * @param  status: Current interrupt status.
  * @retval None
  */
void _usb_device_int_handler(uint32_t idx, uint32_t status)
{
	static uint32_t sof = 0;
	void *arg;
	uint32_t lpm_state;

	/* If device is inactive, then disconnect from USB bus */
	if (__dev_info[0] == 0) {
		map_usb_dev_disconnect();
		return;
	}

	arg = __dcd_inst[0].arg;

	/* Receive a reset signal from the USB bus */
	if (status & USB_INTCTRL_RESET)
	{
		usbd_enum_reset_handler(&__dcd_inst[0]);
	}
	/* Receive a suspend signal from the USB bus */
	if (status & USB_INTCTRL_SUSPEND){
		/* Call the callback function */
		if (__dev_info[0]->cbk->suspend_handler)
			__dev_info[0]->cbk->suspend_handler(arg);
	}
	/* Receive a resume signal from the USB bus */
	if (status & USB_INTCTRL_RESUME) {
		/* Call the callback function */
		if (__dev_info[0]->cbk->resume_handler)
			__dev_info[0]->cbk->resume_handler(arg);
	}
	/* The device is disconnected */
	if (status & USB_INTCTRL_DISCONNECT) {
		/* Call the callback function */
		if (__dev_info[0]->cbk->disconnect_handler)
			__dev_info[0]->cbk->disconnect_handler(arg);
	}

	/* Receive a SOF */
	if (status & USB_INTCTRL_SOF) {
		++__usb_sof_count;
		++sof;
		usbd_tick_resume_handler(&__dcd_inst[0]);

		/* Call the tick function? */
		if (sof == USB_SOF_TICK_DIVIDE) {
			sof = 0;
			usb_tick_inc(USB_SOF_TICK_DIVIDE);
		}
	}

	/* Handle LPM interrupt */
	lpm_state = map_usb_lpm_int_status_get();

	/* Host LPM resume request has been ack, allow the device to sleep */
	if ((__dcd_inst[0].lpm_state == USBLIB_LPM_STATE_SLEEP) && 
				((lpm_state & (USB_INTLPM_ACK | USB_INTLPM_RESUME)) == USB_INTLPM_RESUME)) {
		if (__dev_info[0]->cbk->device_handler)
			__dev_info[0]->cbk->device_handler(arg, USB_EVENT_LPM_RESUME, NULL);

		__dcd_inst[0].lpm_state = USBLIB_LPM_STATE_AWAKE;
		map_usb_dev_lpm_enable();
	}
	/* Host LPM sleep request has been ack, allow the device to sleep */
	else if ((__dcd_inst[0].lpm_state == USBLIB_LPM_STATE_AWAKE) &&
				((lpm_state & (USB_INTLPM_ACK | USB_INTLPM_RESUME)) == USB_INTLPM_ACK)) {
		if (__dev_info[0]->cbk->device_handler)
			__dev_info[0]->cbk->device_handler(arg, USB_EVENT_LPM_SLEEP, NULL);

		__dcd_inst[0].lpm_state = USBLIB_LPM_STATE_SLEEP;
	}
	else if (lpm_state & USB_INTLPM_NYET) {
		if (__dev_info[0]->cbk->device_handler)
			__dev_info[0]->cbk->device_handler(arg, USB_EVENT_LPM_ERROR, NULL);
	}

	/* Get the endpoint interrupt status */
	status = map_usb_int_status_ep_get();
	
	/* Handle EP0 interrupt */
	if (status & USB_INTEP_0) {
		usbd_enum_handler(&__dcd_inst[0]);
		status &= ~USB_INTEP_0;
	}

	/* Call the endpoint callback function */
	if ((__dev_info[0]->cbk->ep_handler) && (status != 0))
		__dev_info[0]->cbk->ep_handler(arg, status);

	return;
}

/**
  * @brief  USB0 device interrupt handler.
  * @retval None
  */
void usb0_device_int_handler(void)
{
	uint32_t status;

	/* Get the USB interrupt status */
	status = map_usb_int_status_get();
	/* Handle the USB interrupt */
	_usb_device_int_handler(0, status);

	return;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
