/**
  *********************************************************************************
  *
  * @file    usbd_comp.c
  * @brief   USB composite device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbd_comp.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_COMP Composite
  * @brief Device Composite driver
  * @{
  */
/** @defgroup Device_COMP_Private_Variables Private Variables
  * @{
  */
/**
  * @brief Configuration descriptor
  */
config_head_t *__g_comp_config_desc[1];

/**
  * @brief Device descriptor
  */
static uint8_t __comp_device_desc[] = {
	18,                       /**< Size of the descriptor */
	USB_DTYPE_DEVICE,         /**< Type of the descriptor */
	USBShort(0x110),          /**< Version 1.1 */
	USB_CLASS_MISC,           /**< USB device class */
	USB_MISC_SUBCLASS_COMMON, /**< USB device sub-class */
	USB_MISC_PROTOCOL_IAD,    /**< USB device protocol */
	64,                       /**< Maximum packet size */
	USBShort(0),              /**< VID */
	USBShort(0),              /**< PID */
	USBShort(0x100),          /**< Device version */
	1,                        /**< Manufacturer string */
	2,                        /**< Product string */
	3,                        /**< Manufacturer serial number */
	1,                        /**< number of the configurations */
};

/**
  * @brief Configuration descriptor header
  */
static const uint8_t __comp_config_desc[] = {
	9,                        /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION,  /**< Type of the descriptor */
	USBShort(0),              /**< Total size of the descriptor */
	0,                        /**< Number of the inerfaces */
	1,                        /**< Unique value for this configuration */
	0,                        /**< Index of the string */
	USB_CONF_ATTR_BUS_PWR,    /**< Type of the power */
	250,                      /**< Maximum power in 2mA increments */
};
/**
  * @}
  */

/** @defgroup Device_COMP_Private_Functions Private Functions
  * @{
  */
static void handle_desc_get(void *device, usb_request_t *request);
static void handle_request(void *device, usb_request_t *request);
static void handle_interface_change(void *device, uint8_t interface, uint8_t alt);
static void handle_config_change(void *device, uint32_t value);
static void handle_data_recv(void *device, uint32_t info);
static void handle_data_sent(void *device, uint32_t info);
static void handle_reset(void *device);
static void handle_suspend(void *device);
static void handle_resume(void *device);
static void handle_disconnect(void *device);
static void handle_ep(void *device, uint32_t status);
static void handle_device(void *device, uint32_t request, void *data);
/**
  * @}
  */

/** @addtogroup Device_COMP_Private_Variables
  * @{
  */
/**
  * @brief Device handler
  */
const base_handler_t comp_handler = {
	handle_desc_get,         /**< Get descriptor */
	handle_request,          /**< Request handler */
	handle_interface_change, /**< Inerface change */
	handle_config_change,    /**< Configuration change */
	handle_data_recv,        /**< Data receive */
	handle_data_sent,        /**< Data sent callback */
	handle_reset,            /**< Reset handler */
	handle_suspend,          /**< Suspend handler */
	handle_resume,           /**< Resume handler */
	handle_disconnect,       /**< Disconnect handler */
	handle_ep,               /**< Endpoint handler */
	handle_device,           /**< Device handler */
};
/**
  * @}
  */

/** @addtogroup Device_COMP_Private_Functions
  * @{
  */
/**
  * @brief  Determine which device to call from the interface.
  * @param  dev: Composite device structure.
  * @param  interface: Interface.
  * @retval Index.
  */
static uint32_t interface_to_idx(usbd_comp_dev_t *dev, uint32_t interface)
{
	uint32_t i, tmp;

	for (i = 0; i < dev->nr_device; ++i) {
		/* Get the lookup value from the device */
		tmp = dev->device[i].space;
		tmp = (tmp >> (8 * LOOKUP_INTERFACE_BYTE)) & 0xff;

		/* Find the desired device */
		if (interface < tmp)
			return i;
	}

	/* Return an invalid idex */
	return INVALID_DEVICE_INDEX;
}

/**
  * @brief  Determine which device to call from the endpoint.
  * @param  dev: Composite device structure.
  * @param  ep: Endpoint.
  * @param  in: Direction, IN/OUT.
  * @retval Index.
  */
static uint32_t ep_to_idx(usbd_comp_dev_t *dev, uint32_t ep, uint8_t in)
{
	uint32_t i, k, tmp;

	/* Make sure the direction */
	k = in ? LOOKUP_IN_END_BYTE : LOOKUP_OUT_END_BYTE;

	for (i = 0; i < dev->nr_device; ++i) {
		/* Get the lookup byte from the device */
		tmp = dev->device[i].space;
		tmp = (tmp >> (k * 8)) & 0xff;

		/* Find the desired device */
		if (ep < tmp)
			return i;
	}

	/* Return an invalid idex */
	return(INVALID_DEVICE_INDEX);
}

/**
  * @brief  Check if any device classes need a get descriptor.
  * @param  device: Composite device structure.
  * @param  request: USB request.
  * @retval None
  */
static void handle_desc_get(void *device, usb_request_t *request)
{
	uint32_t idx;
	const device_info_t *info;
	usbd_comp_dev_t *dev;

	/* Create the pointer to the device */
	dev = (usbd_comp_dev_t *)device;

	/* Determine the type of request */
	switch (request->bmRequestType & USB_RTYPE_RECIPIENT_M) {
	case USB_RTYPE_INTERFACE:
		idx = interface_to_idx(dev, (request->wIndex & 0xFF));
		break;

	case USB_RTYPE_ENDPOINT:
		idx = ep_to_idx(dev, (request->wIndex & 0x0F), (request->wIndex & 0x80) ? 1 : 0);
		break;

	/* If your composite device has some specific descriptor,
	 * you should add some codes here.
	 */
	case USB_RTYPE_DEVICE:
	case USB_RTYPE_OTHER:
	default:
		idx = INVALID_DEVICE_INDEX;
		break;
	}

	/* Check the index */
	if (idx == INVALID_DEVICE_INDEX) {
		usb_dcd_ep0_stall(dev->inst.usb_idx);
		return;
	}

	/* Create a pointer to the device instance */
	info = dev->device[idx].info;

	/* Call the callback function */
	if (info->cbk->get_desc) {
		dev->inst.ep0_owner = idx;
		info->cbk->get_desc(dev->device[idx].inst, request);
		return;
	}

	/* Stall the EP0 */
	usb_dcd_ep0_stall(dev->inst.usb_idx);
	return;
}

/**
  * @brief  Handle a non-standard request.
  * @param  device: Composite device structure.
  * @param  request: USB request.
  * @retval None
  */
static void handle_request(void *device, usb_request_t *request)
{
	uint32_t idx;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Determine the type of this request */
	switch (request->bmRequestType & USB_RTYPE_RECIPIENT_M) {
	case USB_RTYPE_INTERFACE:
		idx = interface_to_idx(dev, (request->wIndex & 0xFF));
		break;

	case USB_RTYPE_ENDPOINT:
		idx = ep_to_idx(dev, (request->wIndex & 0x0F), (request->wIndex & 0x80) ? 1 : 0);
		break;

	/* If your composite device has some specific request,
	 * you need add some codes here.
	 */
	case USB_RTYPE_DEVICE:
	case USB_RTYPE_OTHER:
	default:
		idx = INVALID_DEVICE_INDEX;
		break;
	}

	/* Check the index */
	if (idx == INVALID_DEVICE_INDEX) {
		usb_dcd_ep0_stall(dev->inst.usb_idx);
		return;
	}

	/* Create a pointer to the device instance */
	info = dev->device[idx].info;

	/* Call the callback function */
	if (info->cbk->request_handler) {
		dev->inst.ep0_owner = idx;
		info->cbk->request_handler(dev->device[idx].inst, request);
		return;
	}

	/* Stall the EP0 */
	usb_dcd_ep0_stall(dev->inst.usb_idx);
	return;
}

/**
  * @brief  Handle the device interface changes.
  * @param  device: Composite device structure.
  * @param  interface: Interface.
  * @param  alt: Alternate setting.
  * @retval None
  */
static void handle_interface_change(void *device, uint8_t interface, uint8_t alt)
{
	uint32_t i;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	for (i = 0; i < dev->nr_device; ++i) {
		info = dev->device[i].info;

		/* Call the callback function */
		if (info->cbk->interface_change)
			info->cbk->interface_change(dev->device[i].inst, interface, alt);
	}
}

/**
  * @brief  Handle the device configure changes.
  * @param  device: Composite device structure.
  * @param  value: Value.
  * @retval None
  */
static void handle_config_change(void *device, uint32_t value)
{
	uint32_t i;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Call the callback function */
	if (dev->cbk) {
		dev->cbk(device, USB_EVENT_CONFIG_CHANGE, value, NULL);
	}

	for (i = 0; i < dev->nr_device; ++i){
		info = dev->device[i].info;

		if (info->cbk->config_change)
			info->cbk->config_change(dev->device[i].inst, value);
	}
}

/**
  * @brief  Handle data being received back from the host.
  * @param  device: Composite device structure.
  * @param  info: Information.
  * @retval None
  */
static void handle_data_recv(void *device, uint32_t info)
{
	uint32_t idx;
	const device_info_t *_info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Get the device which last handled a transfer on EP0 */
	idx   = dev->inst.ep0_owner;
	_info = dev->device[idx].info;

	/* Check index */
	if (idx == INVALID_DEVICE_INDEX)
		return;

	/* Call the callback function */
	if (_info->cbk->data_recv)
		_info->cbk->data_recv(dev->device[idx].inst, info);

	return;
}

/**
  * @brief  Handle data being set to the host.
  * @param  device: Composite device structure.
  * @param  info: Information.
  * @retval None
  */
static void handle_data_sent(void *device, uint32_t info)
{
	uint32_t idx;
	const device_info_t *_info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Get the device which last handled a transfer on EP0 */
	idx   = dev->inst.ep0_owner;
	_info = dev->device[idx].info;

	/* Check index */
	if (idx == INVALID_DEVICE_INDEX)
		return;

	/* Call the callback function */
	if (_info->cbk->data_send)
		_info->cbk->data_send(dev->device[idx].inst, info);

	return;
}

/**
  * @brief  Handle reset.
  * @param  device: Composite device structure.
  * @retval None
  */
static void handle_reset(void *device)
{
	uint32_t i;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Call the callback function */
	if (dev->cbk)
		dev->cbk(device, USB_EVENT_CONNECTED, 0, NULL);

	for (i = 0; i < dev->nr_device; ++i) {
		info = dev->device[i].info;

		if (info->cbk->reset_handler)
			info->cbk->reset_handler(dev->device[i].inst);
	}

	return;
}

/**
  * @brief  Handle suspend.
  * @param  device: Composite device structure.
  * @retval None
  */
static void handle_suspend(void *device)
{
	uint32_t i;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Call the callback function */
	if (dev->cbk)
		dev->cbk(device, USB_EVENT_SUSPEND, 0, NULL);

	for (i = 0; i < dev->nr_device; ++i) {
		info = dev->device[i].info;

		if (info->cbk->suspend_handler)
			info->cbk->suspend_handler(dev->device[i].inst);
	}

	return;
}

/**
  * @brief  Handle resume.
  * @param  device: Composite device structure.
  * @retval None
  */
static void handle_resume(void *device)
{
	uint32_t i;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Call the callback function */
	if (dev->cbk)
		dev->cbk(device, USB_EVENT_RESUME, 0, NULL);

	for (i = 0; i < dev->nr_device; ++i) {
		info = dev->device[i].info;

		if (info->cbk->resume_handler)
			info->cbk->resume_handler(dev->device[i].inst);
	}

	return;
}

/**
  * @brief  Handle disconnect.
  * @param  device: Composite device structure.
  * @retval None
  */
static void handle_disconnect(void *device)
{
	uint32_t i;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Check the callback function */
	if (dev->cbk)
		dev->cbk(device, USB_EVENT_DISCONNECTED, 0, NULL);

	for (i = 0; i < dev->nr_device; ++i) {
		info = dev->device[i].info;

		if (info->cbk->disconnect_handler)
			info->cbk->disconnect_handler(dev->device[i].inst);
	}

	return;
}

/**
  * @brief  Handle endpoint callback.
  * @param  device: Composite device structure.
  * @param  status: Current status.
  * @retval None
  */
static void handle_ep(void *device, uint32_t status)
{
	uint32_t i;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	/* Call each endpoint handlers */
	for (i = 0; i < dev->nr_device; ++i) {
		info = dev->device[i].info;

		if (info->cbk->ep_handler)
			info->cbk->ep_handler(dev->device[i].inst, status);
	}

	return;
}

/**
  * @brief  Handle device envent.
  * @param  device: Composite device structure.
  * @param  request: Request.
  * @param  data: Data buffer.
  * @retval None
  */
static void handle_device(void *device, uint32_t request, void *data)
{
	uint32_t i;
	const device_info_t *info;
	usbd_comp_dev_t *dev = (usbd_comp_dev_t *)device;

	for (i = 0; i < dev->nr_device; ++i) {
		info = dev->device[i].info;

		if (info->cbk->device_handler)
			info->cbk->device_handler(dev->device[i].inst, request, data);
	}

	/* Check callback function */
	if (dev->cbk == NULL)
		return;
       
	switch (request) {
	case USB_EVENT_LPM_RESUME:
		/* Send LPM resume event */
		dev->cbk(dev, USB_EVENT_LPM_RESUME, 0, NULL);
		break;

	case USB_EVENT_LPM_SLEEP:
		/* Send LPM sleep event */
		dev->cbk(dev, USB_EVENT_LPM_SLEEP, 0, NULL);
		break;

	case USB_EVENT_LPM_ERROR:
		/* Send LPM error event */
		dev->cbk(dev, USB_EVENT_LPM_ERROR, 0, NULL);
		break;

	default:
		break;
	}

	return;
}

/**
  * @brief  Handle the device interface changes.
  * @param  entry: Composite entry array.
  * @param  _old: Old interface.
  * @param  _new: New interface.
  * @retval None
  */
static void comp_interface_change(comp_entry_t *entry, uint8_t _old, uint8_t _new)
{
	uint8_t interface[2];

	/* Check the callback function */
	if (entry->info->cbk->device_handler == NULL)
		return;       

	/* Save the data */
	interface[0] = _old;
	interface[1] = _new;

	/* Call the callback function */
	entry->info->cbk->device_handler( entry->inst, USB_EVENT_COMP_IFACE_CHANGE, (void *)interface);
	return;
}

/**
  * @brief  Handle the device endpoint changes.
  * @param  entry: Composite entry array.
  * @param  _old: Old endpoint.
  * @param  _new: New endpoint.
  * @retval None
  */
static void comp_ep_change(comp_entry_t *entry, uint8_t _old, uint8_t _new)
{
	uint8_t interface[2];

	/* Check the callback function */
	if (entry->info->cbk->device_handler == NULL)
		return;       

	/* Save the data */
	interface[0] = _old;
	interface[1] = _new;

	/* Call the callback function */
	entry->info->cbk->device_handler(entry->inst, USB_EVENT_COMP_EP_CHANGE, (void *)interface);
}

/**
  * @brief  Handle the device endpoint changes.
  * @param  dev: Composite device structure.
  * @retval Status.
  */
static uint32_t build_comp_desc(usbd_comp_dev_t *dev)
{
	uint32_t i = 0, offset = 0, k, fix = 0, idx = 0;
	uint16_t total_len = 9, len;
	uint8_t interface = 0, in_ep = 1, out_ep = 1;
	uint8_t *data, *tmp;
	const config_head_t *config_hdr;
	desc_head_t *hdr;
	const uint8_t *desc;
	interface_desc_t *iface;
	endpoint_desc_t *ep;
	const device_info_t *_dev;

	/* Initialize the structure */
	dev->inst.p_sec[0] = &dev->inst.sec[0];
	dev->inst.p_sec[0]->data = (uint8_t *)&dev->inst.desc_config;
	dev->inst.p_sec[0]->size = dev->inst.desc_config.bLength;
	dev->inst.p_sec[1] = &dev->inst.sec[1];
	dev->inst.p_sec[1]->size = 0;
	dev->inst.p_sec[1]->data = dev->inst.data;
	/* Create a pointer to the data */
	data = dev->inst.data;

	/* Iterate through each device */
	while (idx < dev->nr_device) {
		/* Save the current starting address */
		tmp = data + offset;
		/* Create a pointer to the header */
		_dev = dev->device[idx].info;
		config_hdr = _dev->config_desc[0];

		/* Iterate through each section */
		for (i = 0; i < config_hdr->nr_section; ++i) {
			/* Initialize the local offset */
			if (i) {
				len = 0;
			}
			else {
				/* First section so skip the header */
				len = 9;
				/* If this section includes only
				 * configuration descriptor, skip it
				 */
				if (config_hdr->section[i]->size <= len)
					continue;
			}

			/* Get a pointer to the descriptor */
			desc = config_hdr->section[i]->data;

			/* Check the allocated space */
			if (offset > dev->inst.size)
				return 1;

			/* Copy the descriptor */
			for (k = 0; k < config_hdr->section[i]->size; ++k)
				data[k + offset] = desc[k];

			/* Read out the descriptors */
			while (len < config_hdr->section[i]->size) {
				/* Create a pointer to the header */
				hdr = (desc_head_t *)&data[offset + len];

				/* Check the interface descriptors */
				if (hdr->bDescriptorType == USB_DTYPE_INTERFACE) {
					iface = (interface_desc_t *)hdr;

					/* If it's an alternate setting */
					if (iface->bAlternateSetting != 0) {
						iface->bInterfaceNumber = interface - 1;
					}
					else {
						/* Interface number has changed */
						comp_interface_change(&dev->device[idx], iface->bInterfaceNumber, interface);
						/* Save the value and move to the next number */
						iface->bInterfaceNumber = interface;
						iface->iInterface = 0;
						++interface;
					}
				}
				else if (hdr->bDescriptorType == USB_DTYPE_ENDPOINT){
					ep = (endpoint_desc_t *)hdr;

					/* If it's IN or OUT endpoint */
					if (ep->bEndpointAddress & USB_RTYPE_DIR_IN) {
						/* If it's the fixed interrupt endpoint */
						if (((ep->bmAttributes & USB_EP_ATTR_TYPE_M) == USB_EP_ATTR_INT) &&
								(dev->pid == USB_PID_COMP_SERIAL)) {
							/* If the fixed endpoint has been set */
							if (fix == 0)
								fix = in_ep++;

							comp_ep_change(&dev->device[idx], ep->bEndpointAddress, fix);
							ep->bEndpointAddress = fix | USB_RTYPE_DIR_IN;
						}
						else {
							/* It's interface number has changed */
							comp_ep_change(&dev->device[idx], ep->bEndpointAddress, in_ep);
							ep->bEndpointAddress = in_ep++ | USB_RTYPE_DIR_IN;
						}
					}
					else {
						/* It's interface number has changed */
						comp_ep_change(&dev->device[idx], ep->bEndpointAddress, out_ep);
						ep->bEndpointAddress = out_ep++;
					}
				}

				/* Move to the next descriptor */
				len += hdr->bLength;
			}

			offset += config_hdr->section[i]->size;
			total_len += len;
		}

		/* Call the callback function */
		dev->device[idx].info->cbk->device_handler(dev->device[idx].inst, USB_EVENT_COMP_CONFIG, (void *)tmp);
		/* Add an entry into the device workspace array */
		dev->device[idx].space = (idx << (LOOKUP_INDEX_BYTE * 8)) | (interface << (LOOKUP_INTERFACE_BYTE * 8)) |
					 (out_ep << (LOOKUP_OUT_END_BYTE * 8)) | (in_ep << (LOOKUP_IN_END_BYTE * 8));

		/* Move to the next device */
		++idx;
	}

	/* Modify the configuration descriptor to match
	 * the number of interface, and save the new total size
	 */
	dev->inst.hdr.nr_section = 2;
	dev->inst.p_sec[1]->size = offset;
	dev->inst.desc_config.bNumInterfaces = interface;
	dev->inst.desc_config.wTotalLength   = total_len;

	return 0;
}
/**
  * @}
  */

/** @defgroup Device_COMP_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes the composite class device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Composite device.
  * @param  size: Size of the buffer.
  * @param  data: Data buffer.
  * @retval Device structure.
  */
void *usbd_comp_init(uint32_t idx, usbd_comp_dev_t *dev, uint32_t size, uint8_t *data)
{
	int32_t i;
	uint8_t *tmp;
	comp_inst_t *inst;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_str);

	/* Initialize the instance structure */
	inst                      = &dev->inst;
	inst->size                = size;
	inst->data                = data;
	inst->usb_idx             = idx;
	inst->ep0_owner           = INVALID_DEVICE_INDEX;
	inst->info.cbk            = &comp_handler;
	inst->info.device_desc    = __comp_device_desc;
	inst->info.config_desc    = (const config_head_t * const *)__g_comp_config_desc;
	inst->info.string_desc    = 0;
	inst->info.nr_string_desc = 0;

	/* Initialize the device infomation structure */
	usb_device_info_init(0, &inst->info);

	__g_comp_config_desc[0] = &inst->hdr;
	__g_comp_config_desc[0]->nr_section = 0;
	__g_comp_config_desc[0]->section = (const config_section_t * const *)inst->p_sec;

	/* Create a pointer to the descriptor */
	tmp = (uint8_t *)&inst->desc_config;

	/* Copy the descriptor into the instance data */
	for (i = 0; i < __comp_config_desc[0]; ++i)
		tmp[i] = __comp_config_desc[i];

	/* Create a pointer to the descriptor */
	tmp = (uint8_t *)&inst->desc_device;

	/* Copy the descriptor into the instance data */
	for (i = 0; i < __comp_device_desc[0]; ++i)
		tmp[i] = __comp_device_desc[i];

	/* Fill the instance structure */
	inst->desc_device.idVendor     = dev->vid;
	inst->desc_device.idProduct    = dev->pid;
	inst->desc_config.bmAttributes = dev->attr_pwr;
	inst->desc_config.bMaxPower    = (uint8_t)(dev->max_power >> 1);
	inst->info.device_desc         = (const uint8_t *)&inst->desc_device;
	inst->info.string_desc         = dev->desc_str;
	inst->info.nr_string_desc      = dev->num_str;

	/* Create a combined descriptor */
	if (build_comp_desc(dev))
		return 0;

	/* Initialize the device controller */
	usb_dcd_init(idx, &inst->info, (void *)dev);

	return (void *)dev;
}

/**
  * @brief  Terminal the composite device.
  * @param  dev: Composite device.
  * @retval None
  */
void usbd_comp_term(usbd_comp_dev_t *dev)
{
	return;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
