/**
  *********************************************************************************
  *
  * @file    usbd_cdc.h
  * @brief   Header file for the CDC driver[Device].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_CDC_H__
#define __USBD_CDC_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_core.h"
#include "usb_cdc.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_CDC
  * @{
  */
/**
  * @defgroup Device_CDC_Public_Macros Public Macros
  * @{
  */
#define SER_DESCRIPTOR_SIZE		8
#define SER_COMM_INTERFACE_SIZE		35
#define SER_DATA_INTERFACE_SIZE		23
#define COMPOSITE_CDC_SIZE		66
#define USBD_CDC_EVENT_SEND_BREAK	(USBD_CDC_EVENT_BASE + 0)
#define USBD_CDC_EVENT_CLEAR_BREAK	(USBD_CDC_EVENT_BASE + 1)
#define USBD_CDC_EVENT_SET_CTRL_STATE	(USBD_CDC_EVENT_BASE + 2)
#define USBD_CDC_EVENT_SET_LINE_CODING	(USBD_CDC_EVENT_BASE + 3)
#define USBD_CDC_EVENT_GET_LINE_CODING	(USBD_CDC_EVENT_BASE + 4)

#define USB_RX_ERROR_FLAGS		(USBERR_DEV_RX_DATA_ERROR |\
                                         USBERR_DEV_RX_OVERRUN    |\
                                         USBERR_DEV_RX_FIFO_FULL)
#define USB_CDC_SERIAL_ERRORS		(USB_CDC_SERIAL_STATE_OVERRUN |\
                                         USB_CDC_SERIAL_STATE_PARITY  |\
                                         USB_CDC_SERIAL_STATE_FRAMING)
#define RX_BLOCK_OPS			((1 << CDC_DO_SEND_BREAK)         |\
                                         (1 << CDC_DO_LINE_CODING_CHANGE) |\
                                         (1 << CDC_DO_LINE_STATE_CHANGE))

#define MAX_REQUEST_DATA_SIZE		sizeof(line_coding_t)
#define CDC_DO_SERIAL_STATE_CHANGE	0
#define CDC_DO_SEND_BREAK		1
#define CDC_DO_CLEAR_BREAK		2
#define CDC_DO_LINE_CODING_CHANGE	3
#define CDC_DO_LINE_STATE_CHANGE	4
#define CDC_DO_PACKET_RX		5

#define SERIAL_INTERFACE_CONTROL	0
#define SERIAL_INTERFACE_DATA		1
#define DATA_IN_EP_MAX_SIZE		64
#define DATA_OUT_EP_MAX_SIZE		64
#define DATA_IN_EP_MAX_SIZE_HS		64
#define DATA_OUT_EP_MAX_SIZE_HS		64
#define CTL_IN_EP_MAX_SIZE		16
#define CONTROL_ENDPOINT		CDC0_EP_CTRL
#define DATA_IN_ENDPOINT		CDC0_EP_IN
#define DATA_OUT_ENDPOINT		CDC0_EP_OUT
#define CONTROL_ENDPOINT_1		CDC1_EP_CTRL
#define DATA_IN_ENDPOINT_1		CDC1_EP_IN
#define DATA_OUT_ENDPOINT_1		CDC1_EP_OUT
/**
  * @}
  */

/**
  * @defgroup Device_CDC_Public_Types Public Types
  * @{
  */
/**
  * @brief CDC status
  */
typedef enum {
	CDC_STATE_UNCONFIG  = 0,	/**< Unconfigure */
	CDC_STATE_IDLE      = 1,	/**< Idle */
	CDC_STATE_WAIT_DATA = 2,	/**< Waiting data */
} cdc_state_t;

/**
  * @brief Define the private instance data.
  */
typedef struct {
	uint32_t usb_idx;		/**< Index of the USB controller */
	device_info_t info;		/**< Device information */
	volatile cdc_state_t rx_state;	/**< State of the CDC receive */
	volatile cdc_state_t tx_state;	/**< State of the CDC transmit */
	volatile cdc_state_t req_state;	/**< Request status */
	volatile cdc_state_t it_state;	/**< Interrupt status */
	volatile uint8_t pending;	/**< Pending */
	uint16_t dur;			/**< Break duration */
	uint16_t ctrl_state;		/**< Line control status */
	uint16_t serial_state;		/**< Serial status */
	volatile uint16_t flag;		/**< Flag */
	uint16_t size;			/**< Size of the last transmit */
	line_coding_t coding;		/**< Current serial line coding */
	volatile uint8_t rx_block;	/**< Serial port receive is blocked */
	volatile uint8_t ctrl_block;	/**< Serial control port is blocked */
	volatile uint8_t conn;		/**< connection status */
	uint8_t ctrl_ep;		/**< Control endpoint */
	uint8_t iep;			/**< IN endpoint */
	uint8_t oep;			/**< OUT endpoint */
	uint8_t interface_ctrl;		/**< Interface number for the control interface */
	uint8_t interface_data;		/**< Interface number for the data interface */
} cdc_inst_t;

/**
  * @brief Define operating parameters for the HID device
  */
typedef struct {
	const uint16_t vid;		/**< Vendor ID */
	const uint16_t pid;		/**< Product ID */
	const uint16_t max_power;	/**< Maximum power consumption */
	const uint8_t attr_pwr;		/**< Attribute of the power */
	const usb_cbk ctrl_cbk;		/**< Callback for control endpoint */
	void *ctrl_arg;			/**< Parameter */
	const usb_cbk rx_cbk;		/**< Callback for OUT endpoint */
	void *rx_arg;			/**< Parameter */
	const usb_cbk tx_cbk;		/**< Callback for IN endpoint */
	void *tx_arg;			/**< Parameter */
	const uint8_t * const *desc_str;/**< String descriptor */
	const uint32_t num_str;		/**< Number of the string descriptors */
	cdc_inst_t inst;		/**< Device instance */
} usbd_cdc_dev_t;
/**
  * @}
  */

/** @addtogroup Device_CDC_Public_Functions
  * @{
  */
extern void *usbd_cdc_init(uint32_t idx, usbd_cdc_dev_t *dev);
extern void *usbd_cdc_init_comp(uint32_t idx, usbd_cdc_dev_t *dev, comp_entry_t *entry, uint8_t second);
extern void usbd_cdc_term(usbd_cdc_dev_t *dev);
extern void *usbd_cdc_set_ctrl_param(usbd_cdc_dev_t *dev, void *arg);
extern void *usbd_cdc_set_rx_param(usbd_cdc_dev_t *dev, void *arg);
extern void *usbd_cdc_set_tx_param(usbd_cdc_dev_t *dev, void *arg);
extern uint32_t usbd_cdc_packet_write(usbd_cdc_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_cdc_packet_read(usbd_cdc_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_cdc_tx_packet_avail(usbd_cdc_dev_t *dev);
extern uint32_t usbd_cdc_rx_packet_avail(usbd_cdc_dev_t *dev);
extern void usbd_cdc_serial_state_change(usbd_cdc_dev_t *dev, uint16_t state);
extern int usbd_cdc_remote_wakeup_req(usbd_cdc_dev_t *dev);
extern void usbd_cdc_power_status_set(usbd_cdc_dev_t *dev, uint8_t power);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
