/**
  *********************************************************************************
  *
  * @file    usbd_cdc.c
  * @brief   USB CDC device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbd_cdc.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_CDC CDC
  * @brief Device CDC driver
  * @{
  */
/** @defgroup Device_CDC_Private_Variables Private Variables
  * @{
  */
/**
  * @brief Maximum packet size
  */
static uint16_t __max_packet_size = 64;

/**
  * @brief Device descriptor
  */
uint8_t __cdc_device_desc[] = {
	18,                       /**< Size of the descriptor */
	USB_DTYPE_DEVICE,         /**< Type of the descriptor */
#ifndef USBD_CDC_HS
	USBShort(0x110),          /**< Version 1.1 */
#else
	USBShort(0x200),          /**< Version 2.0 */
#endif
	USB_CLASS_CDC,            /**< USB device class */
	0,                        /**< USB device sub-class */
	USB_CDC_PROTOCOL_NONE,    /**< USB device protocol */
	64,                       /**< Maximum packet size */
	USBShort(0),              /**< VID */
	USBShort(0),              /**< PID */
	USBShort(0x100),          /**< Device version */
	1,                        /**< Manufacturer string */
	2,                        /**< Product string */
	3,                        /**< Manufacturer serial number */
	1                         /**< number of the configurations */
};

/**
  * @brief Configuration descriptor header
  */
uint8_t __cdc_desc[] = {
	9,                       /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION, /**< Type of the descriptor */
	USBShort(9),             /**< Total size of the descriptor */
	2,                       /**< Number of the inerfaces */
	1,                       /**< Unique value for this configuration */
	5,                       /**< Index of the string */
	USB_CONF_ATTR_SELF_PWR,  /**< Type of the power */
	250,                     /**< Maximum power in 2mA increments */
};

/**
  * @brief CDC configuration section
  */
const config_section_t __cdc_config_sec = {
	sizeof(__cdc_desc),
	__cdc_desc
};

/**
  * @brief Interface Association Descriptor
  */
uint8_t __iad_desc[SER_DESCRIPTOR_SIZE] = {
	8,                               /**< Size of the descriptor */
	USB_DTYPE_INTERFACE_ASC,         /**< Type of the descriptor */
	0x0,                             /**< Starting interface */
	0x2,                             /**< Number of the interface */
	USB_CLASS_CDC,                   /**< Class for this association */
	USB_CDC_SUBCLASS_ABSTRACT_MODEL, /**< Sub-class for this association */
	USB_CDC_PROTOCOL_V25TER,         /**< Protocol for this association */
	0                                /**< Index of the string for this association */
};

/**
  * @brief IAD configuration section
  */
const config_section_t __iad_config_sec = {
	sizeof(__iad_desc),
	__iad_desc
};

/**
  * @brief Control interface descriptor
  */
const uint8_t __cdc_comm_interface[SER_COMM_INTERFACE_SIZE] = {
	9,                                    /**< Size of the descriptor */
	USB_DTYPE_INTERFACE,                  /**< Type of the descriptor */
	SERIAL_INTERFACE_CONTROL,             /**< Index for this interface */
	0,                                    /**< Alternate setting */
	1,                                    /**< Number of endpoint */
	USB_CLASS_CDC,                        /**< Class of the interface */
	USB_CDC_SUBCLASS_ABSTRACT_MODEL,      /**< Sub-class of the interface */
	USB_CDC_PROTOCOL_V25TER,              /**< Protocol of the interface */
	4,                                    /**< Index of the string */

	5,                                    /**< Size of the functional descriptor */
	USB_CDC_CS_INTERFACE,                 /**< Type of the functional descriptor */
	USB_CDC_FD_SUBTYPE_HEADER,            /**< Header functional descriptor */
	USBShort(0x110),                      /**< Version 1.1 */

	4,                                    /**< Size of the functional descriptor */
	USB_CDC_CS_INTERFACE,                 /**< Type of the functional descriptor */
	USB_CDC_FD_SUBTYPE_ABSTRACT_CTL_MGMT, /**< Sub-type */
	USB_CDC_ACM_SUPPORTS_LINE_PARAMS | USB_CDC_ACM_SUPPORTS_SEND_BREAK, /**< Line parameters */

	5,                                    /**< Size of the functional descriptor */
	USB_CDC_CS_INTERFACE,                 /**< Type of the functional descriptor */
	USB_CDC_FD_SUBTYPE_UNION,             /**< Sub-type */
	SERIAL_INTERFACE_CONTROL,             /**< Number of the control interface */
	SERIAL_INTERFACE_DATA,                /**< Number of the data interface */

	5,                                    /**< Size of the functional descriptor */
	USB_CDC_CS_INTERFACE,                 /**< Type of the functional descriptor */
	USB_CDC_FD_SUBTYPE_CALL_MGMT,         /**< Sub-type */
	USB_CDC_CALL_MGMT_HANDLED,            /**< Call MGMT handled */
	SERIAL_INTERFACE_DATA,                /**< Number of the data interface */

	7,                                    /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                   /**< Type of the endpoint descriptor */
	USB_EP_DESC_IN | CONTROL_ENDPOINT,    /**< Endpoint direction */
	USB_EP_ATTR_INT,                      /**< Endpoint type */
	USBShort(CTL_IN_EP_MAX_SIZE),         /**< Maximum packet size */
	1                                     /**< Polling interval */
};

/**
  * @brief Communication interface section
  */
const config_section_t __cdc_comm_interface_sec = {
	sizeof(__cdc_comm_interface),
	__cdc_comm_interface
};

/**
  * @brief CDC data interface descriptor
  */
const uint8_t __cdc_data_interface[SER_DATA_INTERFACE_SIZE] = {
	9,                                   /**< Size of the interface */
	USB_DTYPE_INTERFACE,                 /**< Type of the interface */
	SERIAL_INTERFACE_DATA,               /**< Index of the interface */
	0,                                   /**< Alternate setting */
	2,                                   /**< Number of endpoints in the interface */
	USB_CLASS_CDC_DATA,                  /**< Interface class */
	0,                                   /**< Interface sub-class */
	USB_CDC_PROTOCOL_NONE,               /**< Interface protocol */
	0,                                   /**< Index of the string */

	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_IN | DATA_IN_ENDPOINT,   /**< Endpoint direction */
	USB_EP_ATTR_BULK,                    /**< Endpoint type */
	USBShort(DATA_IN_EP_MAX_SIZE),       /**< Maximum packet size */
	0,                                   /**< Polling interval */

	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_OUT | DATA_OUT_ENDPOINT, /**< Endpoint direction */
	USB_EP_ATTR_BULK,                    /**< Endpoint type */
	USBShort(DATA_OUT_EP_MAX_SIZE),      /**< Maximum packet size */
	0,                                   /**< Polling interval */
};

/**
  * @brief CDC data interface section
  */
const config_section_t __cdc_data_interface_sec = {
	sizeof(__cdc_data_interface),
	__cdc_data_interface
};

/**
  * @brief CDC data interface descriptor for High-Speed mode
  */
const uint8_t __cdc_data_interface_hs[SER_DATA_INTERFACE_SIZE] = {
	9,                                   /**< Size of the interface */
	USB_DTYPE_INTERFACE,                 /**< Type of the interface */
	SERIAL_INTERFACE_DATA,               /**< Index of the interface */
	0,                                   /**< Alternate setting */
	2,                                   /**< Number of endpoints in the interface */
	USB_CLASS_CDC_DATA,                  /**< Interface class */
	0,                                   /**< Interface sub-class */
	USB_CDC_PROTOCOL_NONE,               /**< Interface protocol */
	0,                                   /**< Index of the string */

	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_IN | DATA_IN_ENDPOINT,   /**< Endpoint direction */
	USB_EP_ATTR_BULK,                    /**< Endpoint type */
	USBShort(DATA_IN_EP_MAX_SIZE_HS),    /**< Maximum packet size */
	0,                                   /**< Polling interval */

	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_OUT | DATA_OUT_ENDPOINT, /**< Endpoint direction */
	USB_EP_ATTR_BULK,                    /**< Endpoint type */
	USBShort(DATA_OUT_EP_MAX_SIZE_HS),   /**< Maximum packet size */
	0,                                   /**< Polling interval */
};

/**
  * @brief CDC data interface section for High-Speed mode
  */
const config_section_t __cdc_data_interface_sec_hs = {
	sizeof(__cdc_data_interface_hs),
	__cdc_data_interface_hs
};

/**
  * @brief CDC section
  */
const config_section_t *__cdc_sec[] = {
	&__cdc_config_sec,
	&__cdc_comm_interface_sec,
	&__cdc_data_interface_sec,
};

/**
  * @brief CDC section for High-Speed mode
  */
const config_section_t *__cdc_sec_hs[] = {
	&__cdc_config_sec,
	&__cdc_comm_interface_sec,
	&__cdc_data_interface_sec_hs,
};

/**
  * @brief Number of the section
  */
#define NUM_CDCSER_SECTIONS	(sizeof(__cdc_sec) / sizeof(__cdc_sec[0]))

/**
  * @brief CDC configuration header
  */
const config_head_t __cdc_config_head = {
	NUM_CDCSER_SECTIONS,
	__cdc_sec
};

/**
  * @brief CDC configuration header for High-Speed mode
  */
const config_head_t __cdc_config_head_hs = {
	NUM_CDCSER_SECTIONS,
	__cdc_sec_hs
};

/**
  * @brief CDC Communication section
  */
const config_section_t *__cdc_comp_sec[] = {
	&__cdc_config_sec,
	&__iad_config_sec,
	&__cdc_comm_interface_sec,
	&__cdc_data_interface_sec,
};

/**
  * @brief CDC Communication section for High-Speed mode
  */
const config_section_t *__cdc_comp_sec_hs[] = {
	&__cdc_config_sec,
	&__iad_config_sec,
	&__cdc_comm_interface_sec,
	&__cdc_data_interface_sec_hs,
};

/**
  * @brief Number of the section
  */
#define NUM_COMP_CDCSER_SECTIONS (sizeof(__cdc_comp_sec) / sizeof(__cdc_comp_sec[0]))

/**
  * @brief CDC Communication configuration head
  */
const config_head_t __cdc_comp_config_head = {
	NUM_COMP_CDCSER_SECTIONS,
	__cdc_comp_sec
};

/**
  * @brief CDC Communication configuration head for High-Speed mode
  */
const config_head_t __cdc_comp_config_head_hs = {
	NUM_COMP_CDCSER_SECTIONS,
	__cdc_comp_sec_hs
};

/**
  * @brief CDC configuration descriptor
  */
const config_head_t * const __cdc_config_desc[] = {
	&__cdc_config_head
};

/**
  * @brief CDC configuration descriptor for High-Speed mode
  */
const config_head_t * const __cdc_config_desc_hs[] = {
	&__cdc_config_head_hs
};

/**
  * @brief CDC Communication configuration descriptor
  */
const config_head_t * const __cdc_comp_config_desc[] = {
	&__cdc_comp_config_head
};

/**
  * @brief CDC Communication configuration descriptor for High-Speed mode
  */
const config_head_t * const __cdc_comp_config_desc_hs[] = {
	&__cdc_comp_config_head_hs
};

/**
  * @brief Control interface descriptor
  */
const uint8_t __cdc_comm_interface1[SER_COMM_INTERFACE_SIZE] = {
	9,                                    /**< Size of the descriptor */
	USB_DTYPE_INTERFACE,                  /**< Type of the descriptor */
	SERIAL_INTERFACE_CONTROL,             /**< Index for this interface */
	0,                                    /**< Alternate setting */
	1,                                    /**< Number of endpoint */
	USB_CLASS_CDC,                        /**< Class of the interface */
	USB_CDC_SUBCLASS_ABSTRACT_MODEL,      /**< Sub-class of the interface */
	USB_CDC_PROTOCOL_V25TER,              /**< Protocol of the interface */
	4,                                    /**< Index of the string */

	5,                                    /**< Size of the functional descriptor */
	USB_CDC_CS_INTERFACE,                 /**< Type of the functional descriptor */
	USB_CDC_FD_SUBTYPE_HEADER,           /**< Header functional descriptor */
	USBShort(0x110),                      /**< Version 1.1 */

	4,                                    /**< Size of the functional descriptor */
	USB_CDC_CS_INTERFACE,                 /**< Type of the functional descriptor */
	USB_CDC_FD_SUBTYPE_ABSTRACT_CTL_MGMT, /**< Sub-type */
	USB_CDC_ACM_SUPPORTS_LINE_PARAMS | USB_CDC_ACM_SUPPORTS_SEND_BREAK, /**< Line parameters */

	5,                                    /**< Size of the functional descriptor */
	USB_CDC_CS_INTERFACE,                 /**< Type of the functional descriptor */
	USB_CDC_FD_SUBTYPE_UNION,             /**< Sub-type */
	SERIAL_INTERFACE_CONTROL,             /**< Number of the control interface */
	SERIAL_INTERFACE_DATA,                /**< Number of the data interface */

	5,                                    /**< Size of the functional descriptor */
	USB_CDC_CS_INTERFACE,                 /**< Type of the functional descriptor */
	USB_CDC_FD_SUBTYPE_CALL_MGMT,         /**< Sub-type */
	USB_CDC_CALL_MGMT_HANDLED,            /**< Call MGMT handled */
	SERIAL_INTERFACE_DATA,                /**< Number of the data interface */

	7,                                    /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                   /**< Type of the endpoint descriptor */
	USB_EP_DESC_IN | CONTROL_ENDPOINT_1,  /**< Endpoint direction */
	USB_EP_ATTR_INT,                      /**< Endpoint type */
	USBShort(CTL_IN_EP_MAX_SIZE),         /**< Maximum packet size */
	1                                     /**< Polling interval */
};

/**
  * @brief Communication interface section
  */
const config_section_t __cdc_comm_interface1_sec = {
	sizeof(__cdc_comm_interface1),
	__cdc_comm_interface1
};

/**
  * @brief CDC data interface descriptor
  */
const uint8_t __cdc_data_interface1[SER_DATA_INTERFACE_SIZE] = {
	9,                                     /**< Size of the interface */
	USB_DTYPE_INTERFACE,                   /**< Type of the interface */
	SERIAL_INTERFACE_DATA,                 /**< Index of the interface */
	0,                                     /**< Alternate setting */
	2,                                     /**< Number of endpoints in the interface */
	USB_CLASS_CDC_DATA,                    /**< Interface class */
	0,                                     /**< Interface sub-class */
	USB_CDC_PROTOCOL_NONE,                 /**< Interface protocol */
	0,                                     /**< Index of the string */

	7,                                     /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                    /**< Type is an endpoint */
	USB_EP_DESC_IN | DATA_IN_ENDPOINT_1,   /**< Endpoint direction */
	USB_EP_ATTR_BULK,                      /**< Endpoint type */
	USBShort(DATA_IN_EP_MAX_SIZE),         /**< Maximum packet size */
	0,                                     /**< Polling interval */

	7,                                     /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                    /**< Type is an endpoint */
	USB_EP_DESC_OUT | DATA_OUT_ENDPOINT_1, /**< Endpoint direction */
	USB_EP_ATTR_BULK,                      /**< Endpoint type */
	USBShort(DATA_OUT_EP_MAX_SIZE),        /**< Maximum packet size */
	0,                                     /**< Polling interval */
};

/**
  * @brief CDC data interface section
  */
const config_section_t __cdc_data_interface1_sec = {
	sizeof(__cdc_data_interface1),
	__cdc_data_interface1
};

/**
  * @brief CDC data interface descriptor for High-Speed mode
  */
const uint8_t __cdc_data_interface1_hs[SER_DATA_INTERFACE_SIZE] = {
	9,                                     /**< Size of the interface */
	USB_DTYPE_INTERFACE,                   /**< Type of the interface */
	SERIAL_INTERFACE_DATA,                 /**< Index of the interface */
	0,                                     /**< Alternate setting */
	2,                                     /**< Number of endpoints in the interface */
	USB_CLASS_CDC_DATA,                    /**< Interface class */
	0,                                     /**< Interface sub-class */
	USB_CDC_PROTOCOL_NONE,                 /**< Interface protocol */
	0,                                     /**< Index of the string */

	7,                                     /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                    /**< Type is an endpoint */
	USB_EP_DESC_IN | DATA_IN_ENDPOINT_1,   /**< Endpoint direction */
	USB_EP_ATTR_BULK,                      /**< Endpoint type */
	USBShort(DATA_IN_EP_MAX_SIZE_HS),      /**< Maximum packet size */
	0,                                     /**< Polling interval */

	7,                                     /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                    /**< Type is an endpoint */
	USB_EP_DESC_OUT | DATA_OUT_ENDPOINT_1, /**< Endpoint direction */
	USB_EP_ATTR_BULK,                      /**< Endpoint type */
	USBShort(DATA_OUT_EP_MAX_SIZE_HS),     /**< Maximum packet size */
	0,                                     /**< Polling interval */
};

/**
  * @brief CDC data interface section for High-Speed mode
  */
const config_section_t __cdc_data_interface1_sec_hs = {
	sizeof(__cdc_data_interface1_hs),
	__cdc_data_interface1_hs
};

/**
  * @brief CDC section
  */
const config_section_t *__cdc_comp1_sec[] = {
	&__cdc_config_sec,
	&__iad_config_sec,
	&__cdc_comm_interface1_sec,
	&__cdc_data_interface1_sec,
};

/**
  * @brief CDC section for High-Speed mode
  */
const config_section_t *__cdc_comp1_sec_hs[] = {
	&__cdc_config_sec,
	&__iad_config_sec,
	&__cdc_comm_interface1_sec,
	&__cdc_data_interface1_sec_hs,
};

/**
  * @brief Number of the section
  */
#define NUM_COMP1_CDCSER_SECTIONS (sizeof(__cdc_comp1_sec) / sizeof(__cdc_comp1_sec[0]))

/**
  * @brief CDC Communication configuration head
  */
const config_head_t __cdc_comp_config1_head = {
	NUM_COMP1_CDCSER_SECTIONS,
	__cdc_comp1_sec
};

/**
  * @brief CDC Communication configuration head for High-Speed mode
  */
const config_head_t __cdc_comp_config1_head_hs = {
	NUM_COMP1_CDCSER_SECTIONS,
	__cdc_comp1_sec_hs
};

/**
  * @brief CDC Communication configuration descriptor
  */
const config_head_t * const __cdc_comp_config1_desc[] = {
	&__cdc_comp_config1_head
};

/**
  * @brief CDC Communication configuration descriptor for High-Speed mode
  */
const config_head_t * const __cdc_comp_config1_desc_hs[] = {
	&__cdc_comp_config1_head_hs
};
/**
  * @}
  */

/** @defgroup Device_CDC_Private_Functions Private Functions
  * @{
  */
static void handle_request(void *device, usb_request_t *request);
static void handle_config_change(void *device, uint32_t info);
static void handle_ep0_data(void *device, uint32_t size);
static void handle_disconnect(void *device);
static void handle_ep(void *device, uint32_t status);
static void handle_suspend(void *device);
static void handle_resume(void *device);
static void handle_device(void *device, uint32_t request, void *data);
/**
  * @}
  */

/** @addtogroup Device_CDC_Private_Variables
  * @{
  */
const base_handler_t cdc_handler = {
	0,                    /**< Get descriptor */
	handle_request,       /**< Request handler */
	0,                    /**< Inerface change */
	handle_config_change, /**< Configuration change */
	handle_ep0_data,      /**< Data receive */
	0,                    /**< Data sent callback */
	0,                    /**< Reset handler */
	handle_suspend,       /**< Suspend handler */
	handle_resume,        /**< Resume handler */
	handle_disconnect,    /**< Disconnect handler */
	handle_ep,            /**< Endpoint handler */
	handle_device         /**< Device handler */
};
/**
  * @}
  */

/** @addtogroup Device_CDC_Private_Functions
  * @{
  */
/**
  * @brief  Set/Reset the flag.
  * @param  flag: Flag.
  * @param  bit: indicates which bit number is to be set or reset.
  * @param  set: Set/Reset.
  * @retval None
  */
static void config_cdc_flag(volatile uint16_t *flag, uint16_t bit, uint8_t set)
{
	/* Set/Reset the flag bit */
	if (set)
		SET_BIT(*flag, 1u << bit);
	else
		CLEAR_BIT(*flag, 1u << bit);

	return;
}

/**
  * @brief  Determines whether or not the client has consumed all received data.
  * @param  dev: CDC device.
  * @retval Status.
  */
static uint8_t consume_all_data(const usbd_cdc_dev_t *dev)
{
	uint32_t remain;

	/* Send a event to the client that how many bytes still has to process */
	remain = dev->rx_cbk(dev->rx_arg, USB_EVENT_DATA_REMAINING, 0, NULL);

	return remain ? 0 : 1;
}

/**
  * @brief  Set/Reset a break condition.
  * @param  dev: CDC device.
  * @param  send: Set/Reset.
  * @retval None
  */
static void send_break(usbd_cdc_dev_t *dev, uint8_t send)
{
	config_cdc_flag(&dev->inst.flag, CDC_DO_SEND_BREAK, 0);
	config_cdc_flag(&dev->inst.flag, CDC_DO_CLEAR_BREAK, send);

	/* Tell the client to start/stop sending the break */
	dev->ctrl_cbk(dev->ctrl_arg, (send ? USBD_CDC_EVENT_SEND_BREAK :
                                     USBD_CDC_EVENT_CLEAR_BREAK), 0, NULL);

	return;
}

/**
  * @brief  Request to set the serial communication parameters.
  * @param  dev: CDC device.
  * @retval None
  */
static void send_line_coding_change(usbd_cdc_dev_t *dev)
{
	/* Clear the flag */
	config_cdc_flag(&dev->inst.flag, CDC_DO_LINE_CODING_CHANGE, 0);
	/* Notify the client to update the serial parameters */
	dev->ctrl_cbk(dev->ctrl_arg, USBD_CDC_EVENT_SET_LINE_CODING, 0, &dev->inst.coding);
}

/**
  * @brief  Request to set the RTS and DTR handshake line states.
  * @param  dev: CDC device.
  * @retval None
  */
static void send_line_state_change(usbd_cdc_dev_t *dev)
{
	/* Clear the flag */
	config_cdc_flag(&dev->inst.flag, CDC_DO_LINE_STATE_CHANGE, 0);
	/* Notify the client to update the serial parameters */
	dev->ctrl_cbk(dev->ctrl_arg, USBD_CDC_EVENT_SET_CTRL_STATE,
					dev->inst.ctrl_state, NULL);

	return;
}

/**
  * @brief  Check if no data remains to be processed.
  * @param  dev: CDC device.
  * @param  dur: Duration.
  * @retval Status.
  */
static uint8_t check_send_break(usbd_cdc_dev_t *dev, uint16_t dur)
{
	uint8_t ret;

	/* Check if the client has consumed all data from the host */
	ret = consume_all_data(dev);

	/* Send the break request */
	if (ret)
		send_break(dev, (dur ? 1 : 0));

	return ret;
}

/**
  * @brief  Check if no data remains to be processed.
  * @param  dev: CDC device.
  * @retval Status.
  */
static uint8_t check_send_line_coding_change(usbd_cdc_dev_t *dev)
{
	uint8_t ret;

	/* Check if the client has consumed all data from the host */
	ret = consume_all_data(dev);

	/* Send the line coding change request */
	if (ret)
		send_line_coding_change(dev);

	return ret;
}

/**
  * @brief  Check if no data remains to be processed.
  * @param  dev: CDC device.
  * @retval Status.
  */
static uint8_t check_send_line_state_change(usbd_cdc_dev_t *dev)
{
	uint8_t ret;

	/* Check if the client has consumed all data from the host */
	ret = consume_all_data(dev);

	/* Send the line state change request */
	if (ret)
		send_line_state_change(dev);

	return ret;
}

/**
  * @brief  Notifies the client of a change in the serial line state.
  * @param  dev: CDC device.
  * @retval Status.
  */
static uint8_t send_serial_state(usbd_cdc_dev_t *dev)
{
	uint16_t state;
	int32_t ret;
	cdc_inst_t *inst;
	usb_request_t request;

	/* Create a pointer to device instance */
	inst = &dev->inst;
	/* Set the interrupt state */
	inst->it_state = CDC_STATE_WAIT_DATA;
	/* Clear the flag */
	config_cdc_flag(&inst->flag, CDC_DO_SERIAL_STATE_CHANGE, 0);
	/* Take a snapshot of the serial state */
	state = inst->serial_state;

	/* Fill the request */
	request.bmRequestType = (USB_RTYPE_DIR_IN | USB_RTYPE_CLASS | USB_RTYPE_INTERFACE);
	request.bRequest      = USB_CDC_NOTIFY_SERIAL_STATE;
	request.wValue        = 0;
	request.wIndex        = 0;
	request.wLength       = USB_CDC_NOTIFY_SERIAL_STATE_SIZE;

	/* Write the request to the FIFO */
	ret = map_usb_ep_data_put(inst->ctrl_ep, (uint8_t *)&request, sizeof(usb_request_t));
	ret = map_usb_ep_data_put(inst->ctrl_ep, (uint8_t *)&state, USB_CDC_NOTIFY_SERIAL_STATE_SIZE);

	if (ret != -1)
		ret = map_usb_ep_data_send(inst->ctrl_ep, USB_TRANS_IN);

	if (ret == -1) {
		inst->it_state = CDC_STATE_IDLE;
		return 0;
	}

	/* Clear the error bits */
	inst->serial_state &= ~(state & USB_CDC_SERIAL_ERRORS);
	return 1;
}

/**
  * @brief  Process receive data from host.
  * @param  dev: CDC device.
  * @param  status: Interrupt status.
  * @retval Status.
  */
static uint8_t process_data_from_host(usbd_cdc_dev_t *dev, uint32_t status)
{
	uint32_t ep_status, size;
	cdc_inst_t *inst;

	/* Create a pointer to device instance */
	inst = &dev->inst;
	/* Get the endpoint status */
	ep_status = map_usb_ep_status(inst->oep);
	/* Clear the endpoint status bits */
	map_usb_dev_ep_status_clear(inst->oep, ep_status);

	/* Has a packet been received */
	if (ep_status & USB_DEV_RX_PKT_RDY) {
		/* Set the flag to indicate that a packet read is pending */
		config_cdc_flag(&inst->flag, CDC_DO_PACKET_RX, true);

		/* Is the receive channel currently blocked */
		if (!inst->ctrl_block && !inst->rx_block) {
			/* Get the size of the packet */
			size = map_usb_ep_data_avail(inst->oep);
			/* Notify the event to the client */
			dev->rx_cbk(dev->rx_arg, USB_EVENT_RX_AVAILABLE, size, NULL);
		}
	}
	else {
		/* No packet was received */
		if (ep_status & USB_RX_ERROR_FLAGS)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_ERROR, (ep_status & USB_RX_ERROR_FLAGS), NULL);

		return 0;
	}

	return 1;
}

/**
  * @brief  Receives notifications related to interrupt messages sent to the host.
  * @param  dev: CDC device.
  * @param  status: Interrupt status.
  * @retval Status.
  */
static uint8_t process_notify_to_host(usbd_cdc_dev_t *dev, uint32_t status)
{
	uint32_t ep_status;
	cdc_inst_t *inst;

	/* Create a pointer to device instance */
	inst = &dev->inst;
	/* Get the endpoint status */
	ep_status = map_usb_ep_status(inst->ctrl_ep);
	/* Clear the endpoint status bits */
	map_usb_dev_ep_status_clear(inst->ctrl_ep, ep_status);

	/* Did the state change while we were waiting for
	 * the previous notification to complete?
	 * If the state changed, we were waiting
	 * so we need to schedule another notification immediately.
	 */
	if (inst->flag & (1 << CDC_DO_SERIAL_STATE_CHANGE))
		return send_serial_state(dev);

	/* Set the interrupt state */
	inst->it_state = CDC_STATE_IDLE;
	return 1;
}

/**
  * @brief  Process send data to host.
  * @param  dev: CDC device.
  * @param  status: Interrupt status.
  * @retval Status.
  */
static uint8_t process_data_to_host(usbd_cdc_dev_t *dev, uint32_t status)
{
	uint8_t full;
	uint32_t ep_status, size;
	cdc_inst_t *inst;

	/* Create a pointer to device instance */
	inst = &dev->inst;
	/* Get the endpoint status */
	ep_status = map_usb_ep_status(inst->iep);
	/* Clear the endpoint status bits */
	map_usb_dev_ep_status_clear(inst->iep, ep_status);
	/* Set the TX state */
	inst->tx_state = CDC_STATE_IDLE;

	/* Check the size */
	if (inst->size == 0)
		return 1;

	/* Have it sent a full packet */
	full = (inst->size == __max_packet_size) ? 1 : 0;
	size = inst->size;
	inst->size = 0;
	/* Notify the client that the last transmit completed */
	dev->tx_cbk(dev->tx_arg, USB_EVENT_TX_COMPLETE, size, NULL);

	/* Check if it had previously sent a full packet
	 * It need to send a zero length packet to indicate
	 * the end of the transfer */
	if (full && !inst->size) {
		/* Set the TX state */
		inst->tx_state = CDC_STATE_WAIT_DATA;
		/* Send a zero length packet */
		map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
	}

	return 0;
}

/**
  * @brief  Handle the interrupts on the endpoints.
  * @param  device: CDC device.
  * @param  status: Current status.
  * @retval None
  */
static void handle_ep(void *device, uint32_t status)
{
	usbd_cdc_dev_t *dev = (usbd_cdc_dev_t *)device;

	assert_param(device);

	/* Handle the interrupt IN control endpoint */
	if (status & (1 << dev->inst.ctrl_ep))
		process_notify_to_host(dev, status);

	/* Handle the OUT data endpoint */
	if (status & (0x10000 << dev->inst.oep))
		process_data_from_host(dev, status);

	/* Handle the IN data endpoint */
	if (status & (1 << dev->inst.iep))
		process_data_to_host(dev, status);

	return;
}

/**
  * @brief  Handle device configuration changes.
  * @param  device: CDC device.
  * @param  info: Information.
  * @retval None
  */
static void handle_config_change(void *device, uint32_t info)
{
	usbd_cdc_dev_t *dev = (usbd_cdc_dev_t *)device;

	assert_param(device);

	/* Set the endpoint state to idle */
	dev->inst.it_state  = CDC_STATE_IDLE;
	dev->inst.req_state = CDC_STATE_IDLE;
	dev->inst.rx_state  = CDC_STATE_IDLE;
	dev->inst.tx_state  = CDC_STATE_IDLE;

	/* Send a connected event to client */
	if (dev->inst.conn == 0)
		dev->ctrl_cbk(dev->ctrl_arg, USB_EVENT_CONNECTED, 0, NULL);

	/* Records the connect state */
	dev->inst.conn = 1;
	return;
}

/**
  * @brief  Data requested on endpoint zero is received.
  * @param  device: CDC device.
  * @param  size: Size of the data.
  * @retval None
  */
static void handle_ep0_data(void *device, uint32_t size)
{
	uint8_t ret;
	usbd_cdc_dev_t *dev;
	cdc_inst_t *inst;

	assert_param(device);

	/* Check the size */
	if (size == 0)
		return;

	/* Create a pointer to device instance */
	dev  = (usbd_cdc_dev_t *)device;
	inst = &dev->inst;

	/* Check the request state */
	if (inst->req_state != CDC_STATE_WAIT_DATA)
		return;

	/* Process the received data */
	switch (inst->pending) {
	case USB_CDC_SET_LINE_CODING:
		if (size != sizeof(line_coding_t)) {
			usb_dcd_ep0_stall(0);
		}
		else {
			/* Set the flag that it need to send a notification to the client */
			config_cdc_flag(&inst->flag, CDC_DO_LINE_CODING_CHANGE, 1);
			/* If it can send the notification immediately */
			ret = check_send_line_coding_change(dev);

			/* If it can't send the request to the client,
			 * block reception of more data from the host
			 * until previous data is processed and we send
			 * the change request.
			 */
			if (ret == 0)
				inst->rx_block = 1;
		}

		break;

	default:
		usb_dcd_ep0_stall(0);
		break;
	}

	inst->req_state = CDC_STATE_IDLE;
	return;
}

/**
  * @brief  Handle device envent.
  * @param  device: CDC device.
  * @param  request: Request.
  * @param  data: Data buffer.
  * @retval None
  */
static void handle_device(void *device, uint32_t request, void *data)
{
	uint8_t *buf;
	usbd_cdc_dev_t *dev;
	cdc_inst_t *inst;

	assert_param(device);

	/* Create a pointer to device instance */
	dev  = (usbd_cdc_dev_t *)device;
	inst = &dev->inst;
	/* Create a pointer to the data */
	buf  = (uint8_t *)data;

	switch (request) {
	case USB_EVENT_COMP_IFACE_CHANGE:
		/* Save the change to the appropriate interface number */
		if (buf[0] == SERIAL_INTERFACE_CONTROL)
			inst->interface_ctrl = buf[1];
		else if (buf[0] == SERIAL_INTERFACE_DATA)
			inst->interface_data = buf[1];
		break;

	case USB_EVENT_COMP_EP_CHANGE:
		/* If this is an IN or OUT endpoint that has changed */
		if (buf[0] & USB_EP_DESC_IN) {
			if ((buf[0] & 0x7F) == CONTROL_ENDPOINT)
				inst->ctrl_ep = buf[1] & 0x7F;
			else
				inst->iep = buf[1] & 0x7F;
		}
		else {
			/* Get the new endpoint number */
			inst->oep = buf[1] & 0x7F;
		}
		break;

	case USB_EVENT_COMP_CONFIG:
		/* Sets the control interface and data interface */
		buf[2]  = inst->interface_ctrl;
		buf[29] = inst->interface_ctrl;
		buf[30] = inst->interface_data;
		buf[35] = inst->interface_data;
		break;

	case USB_EVENT_LPM_RESUME:
		/* Send the LPM_RESUME event to client */
		dev->ctrl_cbk((void *)dev->ctrl_cbk, USB_EVENT_LPM_RESUME, 0, NULL);
		break;

	case USB_EVENT_LPM_SLEEP:
		/* Send the LPM_SLEEP event to client */
		dev->ctrl_cbk((void *)dev->ctrl_cbk, USB_EVENT_LPM_SLEEP, 0, NULL);
		break;

	case USB_EVENT_LPM_ERROR:
		/* Send the LPM_ERROR event to client */
		dev->ctrl_cbk((void *)dev->ctrl_cbk, USB_EVENT_LPM_ERROR, 0, NULL);
		break;

	default:
		break;
	}

	return;
}

/**
  * @brief  Handle non-standard request.
  * @param  device: CDC device.
  * @param  request: Non-standard request.
  * @retval None
  */
static void handle_request(void *device, usb_request_t *request)
{
	uint8_t ret;
	usbd_cdc_dev_t *dev;
	cdc_inst_t *inst;
	line_coding_t tmp;

	assert_param(device);

	/* Create a pointer to device instance */
	dev  = (usbd_cdc_dev_t *)device;
	inst = &dev->inst;

	/* Check the interface */
	if(request->wIndex != inst->interface_ctrl)
		return;

	switch (request->bRequest) {
	case USB_CDC_SET_LINE_CODING:
		/* Records the request the it is processing */
		inst->pending   = USB_CDC_SET_LINE_CODING;
		/* Set the state to indicate it's waiting for data */
		inst->req_state = CDC_STATE_WAIT_DATA;
		/* Read the data of the request */
		usb_dcd_ep0_data_req(0, (uint8_t *)&inst->coding, sizeof(line_coding_t));
		/* ACK the the packet */
		map_usb_dev_ep_data_ack(USB_EP_0, false);

		break;

	case USB_CDC_GET_LINE_CODING:
		/* ACK the the packet */
		map_usb_dev_ep_data_ack(USB_EP_0, false);
		/* Notify the event to the client */
		dev->ctrl_cbk(dev->ctrl_arg, USBD_CDC_EVENT_GET_LINE_CODING, 0, &tmp);
		/* Send the line coding parameters to the host */
		usb_dcd_ep0_data_send(0, (uint8_t *)&tmp, sizeof(line_coding_t));

		break;

	case USB_CDC_SET_CONTROL_LINE_STATE:
		/* ACK the the packet */
		map_usb_dev_ep_data_ack(USB_EP_0, false);
		/* Set the line state */
		inst->ctrl_state = request->wValue;
		/* Set the flag */
		config_cdc_flag(&inst->flag, CDC_DO_LINE_STATE_CHANGE, 1);
		/* If it can notify now */
		ret = check_send_line_state_change(dev);

		/* If it can't send the line state change request,
		 * block reception of more data from the host until
		 * previous data is processed.
		 */
		if (ret == 0)
			inst->rx_block = 1;

		break;

	case USB_CDC_SEND_BREAK:
		/* ACK the the packet */
		map_usb_dev_ep_data_ack(USB_EP_0, false);
		/* Save the requested break duration */
		inst->dur = request->wValue;
		/* Set the flag */
		config_cdc_flag(&inst->flag, CDC_DO_SEND_BREAK, true);
		/* Send the break request */
		ret = check_send_break(dev, request->wValue);

		/* If it can't send the line state change request,
		 * block reception of more data from the host until
		 * previous data is processed.
		 */
		if (ret == 0)
			inst->rx_block = 1;

		break;

	default:
		usb_dcd_ep0_stall(0);
		break;
	}
}

/**
  * @brief  Handle disconnect envent.
  * @param  device: CDC device.
  * @retval None
  */
static void handle_disconnect(void *device)
{
	usbd_cdc_dev_t *dev = (usbd_cdc_dev_t *)device;

	assert_param(device);

	/* Send the disconnected event to the client */
	if (dev->inst.conn)
		dev->ctrl_cbk(dev->ctrl_arg, USB_EVENT_DISCONNECTED, 0, NULL);

	/* Records the connected state */
	dev->inst.conn = 0;
	return;
}

/**
  * @brief  Called whenever the device is suspend.
  * @param  device: CDC device.
  * @retval None
  */
static void handle_suspend(void *device)
{
	usbd_cdc_dev_t *dev = (usbd_cdc_dev_t *)device;

	assert_param(device);

	/* Send the suspend event to the client */
	dev->ctrl_cbk(dev->ctrl_arg, USB_EVENT_SUSPEND, 0, NULL);
	return;
}

/**
  * @brief  Called whenever the device is resume.
  * @param  device: CDC device.
  * @retval None
  */
static void handle_resume(void *device)
{
	usbd_cdc_dev_t *dev = (usbd_cdc_dev_t *)device;

	assert_param(device);

	/* Send the resume event to the client */
	dev->ctrl_cbk(dev->ctrl_arg, USB_EVENT_RESUME, 0, NULL);
	return;
}

/**
  * @brief  Called periodically and provides a time reference.
  * @param  device: CDC device.
  * @param  time: Elapsed time in milliseconds since the last call to this function.
  * @retval None
  */
static void cdc_tick_handler(void *device, uint32_t time)
{
	uint8_t ret;
	usbd_cdc_dev_t *dev;
	cdc_inst_t *inst;

	assert_param(device);

	/* Create a pointer to the device instance */
	dev  = (usbd_cdc_dev_t *)device;
	inst = &dev->inst;

	/* Check the flag */
	if (inst->flag == 0)
		return;

	/* Check if it's time to turn off a break condition */
	if (inst->flag & (1 << CDC_DO_CLEAR_BREAK)) {
		if (inst->dur <= time)
			send_break(dev, 0);
		else
			inst->dur -= time;
	}

	/* Check if the client consume all data */
	ret = consume_all_data(dev);

	if (ret) {
		/* Start sending a break condition */
		if (inst->flag & (1 << CDC_DO_SEND_BREAK))
			send_break(dev, true);

		/* Need to set the RTS/CTS */
		if (inst->flag & (1 << CDC_DO_LINE_STATE_CHANGE))
			send_line_state_change(dev);

		/* Need to change the line coding parameters */
		if (inst->flag & (1 << CDC_DO_LINE_CODING_CHANGE))
			send_line_coding_change(dev);

		/* If it can unblock receive? */
		if (!(inst->flag & RX_BLOCK_OPS))
			inst->rx_block = 0;
	}

	return;
}
/**
  * @}
  */

/** @defgroup Device_CDC_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes CDC device.
  * @param  idx: Index of the USB controller.
  * @param  dev: CDC device.
  * @retval Device structure.
  */
void *usbd_cdc_init(uint32_t idx, usbd_cdc_dev_t *dev)
{
	void *ret;
	cdc_inst_t *inst;
	device_desc_t *dev_desc;
	config_desc_t *config_desc;

	/* Initialize the internal state */
	ret = usbd_cdc_init_comp(idx, dev, 0, 0);

	if (ret) {
		/* Fill the device descriptor */
		dev_desc = (device_desc_t *)__cdc_device_desc;
		dev_desc->idVendor = dev->vid;
		dev_desc->idProduct = dev->pid;
		/* Fill the configuration descriptor */
		config_desc = (config_desc_t *)__cdc_desc;
		config_desc->bmAttributes = dev->attr_pwr;
		config_desc->bMaxPower    = (uint8_t)(dev->max_power >> 1);

		/* Create a pointer to the device instance */
		inst = &dev->inst;
		/* Enable the interrupt control endpoint */
		inst->ctrl_ep = CONTROL_ENDPOINT;
		/* Use the configuration descriptor */
		inst->info.config_desc = __cdc_config_desc;

		/* Initialize the USB cotroller */
		usb_dcd_init(idx, &inst->info, (void *)dev);
	}

	return ret;
}

/**
  * @brief  Initializes CDC device.
  * @param  idx: Index of the USB controller.
  * @param  dev: CDC device.
  * @param  entry: Composite entry.
  * @param  second: First/Second.
  * @retval Device structure.
  */
void *usbd_cdc_init_comp(uint32_t idx, usbd_cdc_dev_t *dev, comp_entry_t *entry, uint8_t second)
{
	cdc_inst_t *inst;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->ctrl_cbk);
	assert_param(dev->rx_cbk);
	assert_param(dev->tx_cbk);

	/* Create a pointer to the device instance */
	inst = &dev->inst;

	/* Create the composite entry */
	if (entry != 0) {
		entry->info = &inst->info;
		entry->inst = (void *)dev;
	}

	/* Initialize the device information */
	inst->info.cbk = &cdc_handler;
	inst->info.device_desc = __cdc_device_desc;

	/* The configuration descpritor is different composite and alone device */
	if (entry == 0) {
		inst->info.config_desc = __cdc_config_desc;
#ifdef USBD_CDC_HS
		inst->info.config_desc = __cdc_config_desc_hs;
#endif
	}
	else {
		if (second == 0) {
			inst->info.config_desc = __cdc_comp_config_desc;
#ifdef USBD_CDC_HS
			inst->info.config_desc = __cdc_comp_config_desc_hs;
#endif
		}
		else {
			inst->info.config_desc = __cdc_comp_config1_desc;
#ifdef USBD_CDC_HS
			inst->info.config_desc = __cdc_comp_config1_desc_hs;
#endif			
		}
	}

	/* Set the endpoint */
	if (second == 0) {
		inst->iep     = DATA_IN_ENDPOINT;
		inst->oep     = DATA_OUT_ENDPOINT;
		inst->ctrl_ep = CONTROL_ENDPOINT;
	}
	else {
		inst->iep     = DATA_IN_ENDPOINT_1;
		inst->oep     = DATA_OUT_ENDPOINT_1;
		inst->ctrl_ep = CONTROL_ENDPOINT_1;
	}

	/* Initialize the instance structure */
	inst->info.string_desc    = 0;
	inst->info.nr_string_desc = 0;
	inst->interface_ctrl      = SERIAL_INTERFACE_CONTROL;
	inst->interface_data      = SERIAL_INTERFACE_DATA;
	inst->usb_idx             = 0;
	inst->rx_state            = CDC_STATE_UNCONFIG;
	inst->tx_state            = CDC_STATE_UNCONFIG;
	inst->it_state            = CDC_STATE_UNCONFIG;
	inst->req_state           = CDC_STATE_UNCONFIG;
	inst->pending             = 0;
	inst->dur                 = 0;
	inst->serial_state        = 0;
	inst->flag                = 0;
	inst->ctrl_state          = 0;
	inst->rx_block            = 0;
	inst->ctrl_block          = 0;
	inst->conn                = 0;

	/* Initialize the device information structure */
	usb_device_info_init(0, &inst->info);

	/* Set the string stable */
	inst->info.string_desc    = dev->desc_str;
	inst->info.nr_string_desc = dev->num_str;

	/* Initialize the USB tick module */
	usb_tick_init();
	/* Register this class tick handler */
	usb_tick_handler_register(cdc_tick_handler, (void *)dev);

	return (void *)dev;
}

/**
  * @brief  Terminal the CDC device.
  * @param  dev: CDC device.
  * @retval None
  */
void usbd_cdc_term(usbd_cdc_dev_t *dev)
{
	assert_param(dev);

	/* Terminate the requested instance */
	usb_dcd_term(dev->inst.usb_idx);
	dev->inst.usb_idx = 0;

	return;
}

/**
  * @brief  Set control callback parameter.
  * @param  dev: CDC device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_cdc_set_ctrl_param(usbd_cdc_dev_t *dev, void *arg)
{
	void *old;

	assert_param(dev);

	/* Set the callback parameter */
	old = dev->ctrl_arg;
	dev->ctrl_arg = arg;

	return old;
}

/**
  * @brief  Set RX callback parameter.
  * @param  dev: CDC device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_cdc_set_rx_param(usbd_cdc_dev_t *dev, void *arg)
{
	void *old;

	assert_param(dev);

	/* Set the callback parameter */
	old = dev->rx_arg;
	dev->rx_arg = arg;

	return old;
}

/**
  * @brief  Set TX callback parameter.
  * @param  dev: CDC device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_cdc_set_tx_param(usbd_cdc_dev_t *dev, void *arg)
{
	void *old;

	assert_param(dev);

	/* Set the callback parameter */
	old = dev->tx_arg;
	dev->tx_arg = arg;

	return old;
}

/**
  * @brief  Transmits a packet to the USB host.
  * @param  dev: CDC device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Indicates whether more data is to be written.
  * @retval Status.
  */
uint32_t usbd_cdc_packet_write(usbd_cdc_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last)
{
	int32_t ret;
	cdc_inst_t *inst = &dev->inst;

	assert_param(dev);

	/* Check the length and TX state */
	if ((len > __max_packet_size) || (inst->tx_state != CDC_STATE_IDLE))
		return 0;

	/* Write data into FIFO */
	ret = map_usb_ep_data_put(inst->iep, data, len);

	if (ret != -1) {
		/* Records the bytes it sent */
		inst->size += (uint16_t)len;

		/* If this is the last call for the packet */
		if (last) {
			/* Send the packet to the host */
			inst->tx_state = CDC_STATE_WAIT_DATA;
			ret = map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
		}
	}

	return ret == -1 ? 0 : len;
}

/**
  * @brief  Received a packet from the USB host.
  * @param  dev: CDC device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Indicates whether more data is to be written.
  * @retval Status.
  */
uint32_t usbd_cdc_packet_read(usbd_cdc_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last)
{
	int32_t ret;
	uint32_t ep_status, cnt, _len;
	cdc_inst_t *inst = &dev->inst;

	assert_param(dev);

	/* Get the endpoint status */
	ep_status = map_usb_ep_status(inst->oep);

	/* Check the packet */
	if ((ep_status & USB_DEV_RX_PKT_RDY) == 0)
		return 0;

	/* If receive a blocked or the buffer is too small,
	 * set the flag that it have a packet waiting.
	 */
	if (inst->rx_block || inst->ctrl_block) {
		config_cdc_flag(&inst->flag, CDC_DO_PACKET_RX, 1);
		return 0;
	}

	/* Get the size of the packet */
	_len = map_usb_ep_data_avail(inst->oep);

	/* Get the packet */
	cnt = len;
	ret = map_usb_ep_data_get(inst->oep, data, &cnt);

	/* If the last packet data? */
	if (cnt == _len) {
		/* Clear the endpoint status */
		map_usb_dev_ep_status_clear(inst->oep, ep_status);
		/* ACK the paceket */
		map_usb_dev_ep_data_ack(inst->oep, 1);
		/* Clear the flag */
		config_cdc_flag(&inst->flag, CDC_DO_PACKET_RX, 0);
	}

	return ret == -1 ? 0 : cnt;
}

/**
  * @brief  Get number of free bytes in the transmit buffer.
  * @param  dev: CDC device.
  * @retval Length.
  */
uint32_t usbd_cdc_tx_packet_avail(usbd_cdc_dev_t *dev)
{
	assert_param(dev);

	if (dev->inst.tx_state != CDC_STATE_IDLE)
		return 0;
	else
		return __max_packet_size;
}

/**
  * @brief  Determine whether a packet is available.
  * @param  dev: CDC device.
  * @retval Length.
  */
uint32_t usbd_cdc_rx_packet_avail(usbd_cdc_dev_t *dev)
{
	uint32_t ep_status;

	assert_param(dev);

	/* If receive a blocked or the buffer is too small,
	 * set the flag that it have a packet waiting.
	 */
	if (dev->inst.rx_block || dev->inst.ctrl_block)
		return 0;

	/* Get the endpoint status */
	ep_status = map_usb_ep_status(dev->inst.oep);

	/* Get the size of the packet */
	if (ep_status & USB_DEV_RX_PKT_RDY)
		return map_usb_ep_data_avail(dev->inst.oep);

	return 0;
}

/**
  * @brief  Informs the CDC module of changes in the serial control line states.
  * @param  dev: CDC device.
  * @param  state: States of the various control lines and any receive errors detected.
  * @retval Length.
  */
void usbd_cdc_serial_state_change(usbd_cdc_dev_t *dev, uint16_t state)
{
	assert_param(dev);

	/* Records the new state */
	dev->inst.serial_state = state;
	/* Set the flag indicating that a serial state change */
	config_cdc_flag(&dev->inst.flag, CDC_DO_SERIAL_STATE_CHANGE, 1);

	/* Send the state change immediately */
	if (dev->inst.it_state == CDC_STATE_IDLE) {
		/* Send the notification immediately */
		dev->inst.it_state = CDC_STATE_WAIT_DATA;
		send_serial_state(dev);
	}

	return;
}

/**
  * @brief  Requests a remote wake up to resume communication.
  * @param  dev: CDC device.
  * @retval Status.
  */
int usbd_cdc_remote_wakeup_req(usbd_cdc_dev_t *dev)
{
	assert_param(dev);

	return usb_dcd_remote_wakeup_req(0);
}

/**
  * @brief  Set power status.
  * @param  dev: CDC device.
  * @param  power: Status of the power.
  * @retval None
  */
void usbd_cdc_power_status_set(usbd_cdc_dev_t *dev, uint8_t power)
{
	assert_param(dev);

	usb_dcd_power_status_set(0, power);
	return;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
