/**
  *********************************************************************************
  *
  * @file    usbd_bulk.h
  * @brief   Header file for the BULK driver[Device].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_BULK_H__
#define __USBD_BULK_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_core.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_BULK
  * @{
  */
/**
  * @defgroup Device_BULK_Public_Macros Public Macros
  * @{
  */
#define USBD_BULK_EVENT_DMA_RX	(USBD_MSC_EVENT_BASE + 0)
#define USBD_BULK_EVENT_DMA_ERR	(USBD_MSC_EVENT_BASE + 1)

#define BULK_INTERFACE_SIZE	23
#define COMPOSITE_BULK_SIZE	BULK_INTERFACE_SIZE
#define USB_RX_ERROR_FLAGS	(USBERR_DEV_RX_DATA_ERROR | USBERR_DEV_RX_OVERRUN | USBERR_DEV_RX_FIFO_FULL)
#define BULK_DO_PACKET_RX	5
#define DATA_IN_ENDPOINT	BULK_EP_IN
#define DATA_OUT_ENDPOINT	BULK_EP_OUT
/**
  * @}
  */

/**
  * @defgroup Device_BULK_Public_Types Public Types
  * @{
  */
/**
  * @brief BULK status
  */
typedef enum {
	BULK_STATE_UNCONFIG    = 0,	/**< Unconfigure */
	BULK_STATE_IDLE        = 1,	/**< Idle */
	BULK_STATE_WAIT_DATA   = 2,	/**< Waiting data */
} bulk_state_t;

/**
  * @brief Define the private instance data.
  */
typedef struct {
	uint32_t usb_idx;		/**< Index of the USB controller */
	device_info_t info;		/**< Device information */
	volatile bulk_state_t rx_state;	/**< State of the BULK receive */
	volatile bulk_state_t tx_state;	/**< State of the BULK transmit */
	volatile uint16_t flag;		/**< Flags */
	uint16_t size;			/**< Size */
	volatile uint8_t conn;		/**< Connection status */
	uint8_t iep;			/**< IN endpoint */
	uint8_t oep;			/**< OUT endpoint */
	uint8_t idmach;                 /**< IN dma channel */
	uint8_t odmach;                 /**< OUT dma channel */
	uint8_t interface;		/**< Interface */
} bulk_inst_t;

/**
  * @brief Define operating parameters for the BULK device
  */
typedef struct {
	const uint16_t vid;		/**< Vendor ID */
	const uint16_t pid;		/**< Product ID */
	const uint16_t max_power;	/**< Maximum power consumption */
	const uint8_t attr_pwr;		/**< Attribute of the power */
	const usb_cbk rx_cbk;		/**< Callback function */
	void *rx_arg;			/**< Parameter */
	const usb_cbk tx_cbk;		/**< Callback function */
	void *tx_arg;			/**< Parameter */
	const uint8_t * const *desc_str;/**< String descriptor */
	const uint32_t num_str;		/**< Number of the string descriptors */
	bulk_inst_t inst;		/**< Device instance */
} usbd_bulk_dev_t;
/**
  * @}
  */

/** @addtogroup Device_BULK_Public_Functions
  * @{
  */
extern void *usbd_bulk_init(uint32_t idx, usbd_bulk_dev_t *dev);
extern void *usbd_bulk_init_comp(uint32_t idx, usbd_bulk_dev_t *dev, comp_entry_t *entry);
extern void usbd_bulk_term(usbd_bulk_dev_t *dev);
extern void *usbd_bulk_set_rx_param(usbd_bulk_dev_t *dev, void *arg);
extern void *usbd_bulk_set_tx_param(usbd_bulk_dev_t *dev, void *arg);
extern uint32_t usbd_bulk_packet_write(usbd_bulk_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_bulk_packet_read(usbd_bulk_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_bulk_packet_write_by_dma(usbd_bulk_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_bulk_packet_read_by_dma(usbd_bulk_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last);
extern uint32_t usbd_bulk_tx_packet_avail(usbd_bulk_dev_t *dev);
extern uint32_t usbd_bulk_rx_packet_avail(usbd_bulk_dev_t *dev);
extern int usbd_bulk_remote_wakeup_req(usbd_bulk_dev_t *dev);
extern void usbd_bulk_power_status_set(usbd_bulk_dev_t *dev, uint8_t power);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
