/**
  *********************************************************************************
  *
  * @file    usbd_bulk.c
  * @brief   USB BULK device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#include "usbd_bulk.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_BULK BULK
  * @brief Device BULK driver
  * @{
  */
/** @defgroup Device_BULK_Private_Variables Private Variables
  * @{
  */
static uint16_t __max_packet_size = 64;
static uint8_t __last;
static uint32_t __len;
static uint32_t __status;
#ifdef USB_DMA_N_SUPPORT
static usb_dma_callback_t bulk_tx_cbk;
static usb_dma_callback_t bulk_rx_cbk;
#else
static usb_dma_callback_t bulk_cbk;
#endif

/**
  * @brief Device descriptor
  */
uint8_t __bulk_device_desc[] = {
	18,                       /**< Size of the descriptor */
	USB_DTYPE_DEVICE,         /**< Type of the descriptor */
#ifndef USBD_BULK_HS
	USBShort(0x110),          /**< Version 1.1 */
#else
	USBShort(0x200),          /**< Version 2.0 */
#endif
	USB_CLASS_VEND_SPECIFIC,  /**< USB device class */
	0,                        /**< USB device sub-class */
	0,                        /**< USB device protocol */
	64,                       /**< Maximum packet size */
	USBShort(0),              /**< VID */
	USBShort(0),              /**< PID */
	USBShort(0x100),          /**< Device version */
	1,                        /**< Manufacturer string */
	2,                        /**< Product string */
	3,                        /**< Manufacturer serial number */
	1                         /**< number of the configurations */
};

/**
  * @brief Configuration descriptor header
  */
uint8_t __bulk_desc[] = {
	9,                       /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION, /**< Type of the descriptor */
	USBShort(32),            /**< Total size of the descriptor */
	1,                       /**< Number of the inerfaces */
	1,                       /**< Unique value for this configuration */
	5,                       /**< Index of the string */
	USB_CONF_ATTR_SELF_PWR,  /**< Type of the power */
	250,                     /**< Maximum power in 2mA increments */
};

/**
  * @brief BULK interface descriptor
  */
const uint8_t __bulk_interface[BULK_INTERFACE_SIZE] = {
	9,                                   /**< Size of the interface */
	USB_DTYPE_INTERFACE,                 /**< Type of the interface */
	0,                                   /**< Index of the interface */
	0,                                   /**< Alternate setting */
	2,                                   /**< Number of endpoints in the interface */
	USB_CLASS_VEND_SPECIFIC,             /**< Interface class */
	0,                                   /**< Interface sub-class */
	0,                                   /**< Interface protocol */
	4,                                   /**< Index of the string */
	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_IN | DATA_IN_ENDPOINT,   /**< Endpoint direction */
	USB_EP_ATTR_BULK,                    /**< Endpoint type */
	USBShort(64),                        /**< Maximum packet size */
	0,                                   /**< Polling interval */
	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_OUT | DATA_OUT_ENDPOINT, /**< Endpoint direction */
	USB_EP_ATTR_BULK,                    /**< Endpoint type */
	USBShort(64),                        /**< Maximum packet size */
	0,                                   /**< Polling interval */
};

/**
  * @brief BULK interface descriptor for High-Speed mode
  */
const uint8_t __bulk_interface_hs[BULK_INTERFACE_SIZE] = {
	9,                                   /**< Size of the interface */
	USB_DTYPE_INTERFACE,                 /**< Type of the interface */
	0,                                   /**< Index of the interface */
	0,                                   /**< Alternate setting */
	2,                                   /**< Number of endpoints in the interface */
	USB_CLASS_VEND_SPECIFIC,             /**< Interface class */
	0,                                   /**< Interface sub-class */
	0,                                   /**< Interface protocol */
	4,                                   /**< Index of the string */
	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_IN | DATA_IN_ENDPOINT,   /**< Endpoint direction */
	USB_EP_ATTR_BULK,                    /**< Endpoint type */
	USBShort(512),                       /**< Maximum packet size */
	0,                                   /**< Polling interval */
	7,                                   /**< Size of the endpoint descriptor */
	USB_DTYPE_ENDPOINT,                  /**< Type is an endpoint */
	USB_EP_DESC_OUT | DATA_OUT_ENDPOINT, /**< Endpoint direction */
	USB_EP_ATTR_BULK,                    /**< Endpoint type */
	USBShort(512),                       /**< Maximum packet size */
	0,                                   /**< Polling interval */
};

/**
  * @brief BULK configuration section
  */
const config_section_t __bulk_config_sec = {
	sizeof(__bulk_desc),
	__bulk_desc
};

/**
  * @brief BULK interface section
  */
const config_section_t __bulk_interface_sec = {
	sizeof(__bulk_interface),
	__bulk_interface
};

/**
  * @brief BULK High-Speed interface section
  */
const config_section_t __bulk_interface_hs_sec = {
	sizeof(__bulk_interface_hs),
	__bulk_interface_hs
};

/**
  * @brief BULK section
  */
const config_section_t *__bulk_sec[] = {
	&__bulk_config_sec,
	&__bulk_interface_sec
};

/**
  * @brief BULK High-Speed section
  */
const config_section_t *__bulk_hs_sec[] = {
	&__bulk_config_sec,
	&__bulk_interface_hs_sec
};

/**
  * @brief Number of the section
  */
#define NUM_BULK_SECTIONS	(sizeof(__bulk_sec) / sizeof(__bulk_sec[0]))

/**
  * @brief BULK configuration header
  */
const config_head_t __bulk_config_head = {
	NUM_BULK_SECTIONS,
	__bulk_sec
};

/**
  * @brief BULK High-Speed configuration header
  */
const config_head_t __bulk_config_head_hs = {
	NUM_BULK_SECTIONS,
	__bulk_hs_sec
};

/**
  * @brief BULK configuration descriptor
  */
const config_head_t * const __bulk_config_desc[] = {
	&__bulk_config_head
};

/**
  * @brief BULK High-Speed configuration descriptor
  */
const config_head_t * const __bulk_config_desc_hs[] = {
	&__bulk_config_head_hs
};
/**
  * @}
  */

/** @defgroup Device_BULK_Private_Functions Private Functions
  * @{
  */
static void handle_config_change(void *device, uint32_t info);
static void handle_disconnect(void *device);
static void handle_ep(void *device, uint32_t status);
static void handle_suspend(void *device);
static void handle_resume(void *device);
static void handle_device(void *device, uint32_t request, void *data);
/**
  * @}
  */

/** @addtogroup Device_BULK_Private_Variables
  * @{
  */
const base_handler_t bulk_handler = {
	0,                    /**< Get descriptor */
	0,                    /**< Request handler */
	0,                    /**< Inerface change */
	handle_config_change, /**< Configuration change */
	0,                    /**< Data receive */
	0,                    /**< Data sent callback */
	0,                    /**< Reset handler */
	handle_suspend,       /**< Suspend handler */
	handle_resume,        /**< Resume handler */
	handle_disconnect,    /**< Disconnect handler */
	handle_ep,            /**< Endpoint handler */
	handle_device         /**< Device handler */
};
/**
  * @}
  */

/** @addtogroup Device_BULK_Private_Functions
  * @{
  */
/**
  * @brief  Set/Reset the flag.
  * @param  flag: Flag.
  * @param  bit: indicates which bit number is to be set or reset.
  * @param  set: Set/Reset.
  * @retval None
  */
static void config_bulk_flag(volatile uint16_t *flag, uint16_t bit, uint8_t set)
{
	/* Set/Reset the flag bit */
	if (set)
		SET_BIT(*flag, 1u << bit);
	else
		CLEAR_BIT(*flag, 1u << bit);

	return;
}

#ifdef USB_DMA_N_SUPPORT
/**
  * @brief  Callback function when external DMA transfer complete.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_bulk_dma_tx_cplt(void *arg)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)arg;
	bulk_inst_t *inst = &dev->inst;

	inst->size += __len;

	if (__last) {
		inst->tx_state = BULK_STATE_WAIT_DATA;
		map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
	}

	return;
}

/**
  * @brief  Callback function when external DMA transfer complete.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_bulk_dma_rx_cplt(void *arg)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)arg;
	bulk_inst_t *inst = &dev->inst;

	if (__last) {
		map_usb_dev_ep_status_clear(inst->oep, __status);
		map_usb_dev_ep_data_ack(inst->oep, true);
		config_bulk_flag(&inst->flag, BULK_DO_PACKET_RX, 0);
	}

	dev->rx_cbk(dev->rx_arg, USBD_BULK_EVENT_DMA_RX, __len, NULL);
	return;
}

/**
  * @brief  Callback function when external DMA send error.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_bulk_dma_e_err(void *arg)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)arg;

	dev->rx_cbk(dev->rx_arg, USBD_BULK_EVENT_DMA_ERR, 0, NULL);
	return;
}
#else

/**
  * @brief  Callback function when DMA transfer complete.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_bulk_dma_cplt(uint8_t ch, void *arg)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)arg;
	bulk_inst_t *inst = &dev->inst;

	if (ch == inst->idmach) {
		inst->size += __len;
	
		if (__last) {
			inst->tx_state = BULK_STATE_WAIT_DATA;
			map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
		}
	}
	else {
		if (__last) {
			map_usb_dev_ep_status_clear(inst->oep, __status);
			map_usb_dev_ep_data_ack(inst->oep, true);
			config_bulk_flag(&inst->flag, BULK_DO_PACKET_RX, 0);
		}
	
		dev->rx_cbk(dev->rx_arg, USBD_BULK_EVENT_DMA_RX, __len, NULL);
	}

	return;
}

/**
  * @brief  Callback function when DMA send error.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_bulk_dma_err(uint8_t ch, void *arg)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)arg;

	dev->rx_cbk(dev->rx_arg, USBD_BULK_EVENT_DMA_ERR, 0, NULL);
	return;
}
#endif

/**
  * @brief  Process receives notifications from the host.
  * @param  dev: BULK device stucture.
  * @param  status: Status.
  * @retval Status.
  */
static int process_data_from_host(usbd_bulk_dev_t *dev, uint32_t status)
{
	uint32_t ep_status, size;
	bulk_inst_t *inst;

	/* Create a pointer to the device instance */
	inst      = &dev->inst;
	/* Get the endpoint status */
	ep_status = map_usb_ep_status(inst->oep);
	/* Clear the endpoint status */
	map_usb_dev_ep_status_clear(inst->oep, ep_status);

	/* Has receive a packet */
	if (ep_status & USB_DEV_RX_PKT_RDY) {
		/* Set flag bit */
		config_bulk_flag(&inst->flag, BULK_DO_PACKET_RX, 1);
		/* Get the size of the data */
		size = map_usb_ep_data_avail(inst->oep);
		/* Call the callback function */
		dev->rx_cbk(dev->rx_arg, USB_EVENT_RX_AVAILABLE, size, NULL);
	}
	else {
		/* No packet was received */
		if (ep_status & USB_RX_ERROR_FLAGS)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_ERROR, (ep_status & USB_RX_ERROR_FLAGS), NULL);

		return -1;
	}

	return 0;
}

/**
  * @brief  Process send data to the host.
  * @param  dev: BULK device stucture.
  * @param  status: Status.
  * @retval Status.
  */
static int process_data_to_host(usbd_bulk_dev_t *dev, uint32_t status)
{
	uint32_t ep_status, size;
	bulk_inst_t *inst;

	/* Create a pointer to the device instance */
	inst = &dev->inst;
	/* Get the endpoint status */
	ep_status = map_usb_ep_status(inst->iep);
	/* Clear the endpoint status */
	map_usb_dev_ep_status_clear(inst->iep, ep_status);
	/* Set the TX_STATE as IDLE */
	inst->tx_state = BULK_STATE_IDLE;

	/* Get the size */
	size = inst->size;
	inst->size = 0;

	/* Call the callback function */
	dev->tx_cbk(dev->tx_arg, USB_EVENT_TX_COMPLETE, size, NULL);
	return 0;
}

/**
  * @brief  Handle the interrupts on the endpoints.
  * @param  device: BULK device.
  * @param  status: Current status.
  * @retval None
  */
static void handle_ep(void *device, uint32_t status)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)device;

	assert_param(dev);

	/* Handle OUT data endpoint */
	if (status & (0x10000 << (dev->inst.oep)))
		process_data_from_host(dev, status);

	/* Handle IN data endpoint */
	if (status & (1 << (dev->inst.iep)))
		process_data_to_host(dev, status);

	return;
}

/**
  * @brief  Handle device configuration changes.
  * @param  device: BULK device.
  * @param  ui32Info: New value.
  * @retval None
  */
static void handle_config_change(void *device, uint32_t ui32Info)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)device;

	assert_param(dev);

	/* Set all endpoint state to IDLE */
	dev->inst.rx_state = BULK_STATE_IDLE;
	dev->inst.tx_state = BULK_STATE_IDLE;

	/* Call the callback function */
	if (dev->rx_cbk)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_CONNECTED, 0, NULL);

	/* Save the connected state */
	dev->inst.conn = 1;
	return;
}

/**
  * @brief  Handle device envent.
  * @param  device: BULK device.
  * @param  request: Request.
  * @param  data: Data buffer.
  * @retval None
  */
static void handle_device(void *device, uint32_t request, void *data)
{
	uint8_t *buf = (uint8_t *)data;
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)device;

	switch (request) {
	case USB_EVENT_COMP_IFACE_CHANGE:
		/* Interface change event */
		dev->inst.interface = buf[1];
		break;

	case USB_EVENT_COMP_EP_CHANGE:
		/* Endpint change event */
		if (buf[0] & USB_EP_DESC_IN)
			dev->inst.iep = buf[1] & 0x7f;
		else
			dev->inst.oep = buf[1] & 0x7f;

		break;
	case USB_EVENT_LPM_RESUME:
		/* LPM resume event */
		if (dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_RESUME, 0, NULL);

		break;
	case USB_EVENT_LPM_SLEEP:
		/* LPM sleep event */
		if (dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_SLEEP, 0, NULL);

		break;
	case USB_EVENT_LPM_ERROR:
		/* LPM error event */
		if (dev->rx_cbk)
			dev->rx_cbk(dev->rx_arg, USB_EVENT_LPM_ERROR, 0, NULL);

		break;
	default:
		break;
	}
}

/**
  * @brief  Handle disconnect envent.
  * @param  device: BULK device.
  * @retval None
  */
static void handle_disconnect(void *device)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)device;

	assert_param(device);

	/* Call the callback function */
	if (dev->inst.conn && dev->rx_cbk)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_DISCONNECTED, 0, NULL);

	/* Save the disconnected state */
	dev->inst.conn= 0;
}

/**
  * @brief  Called whenever the device is suspend.
  * @param  device: BULK device.
  * @retval None
  */
static void handle_suspend(void *device)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)device;

	assert_param(device);

	/* Call callback function */
	if (dev->rx_cbk)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_SUSPEND, 0, NULL);

	return;
}

/**
  * @brief  Called whenever the device is resume.
  * @param  device: BULK device.
  * @retval None
  */
static void handle_resume(void *device)
{
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)device;

	assert_param(device);

	/* Call callback function */
	if (dev->rx_cbk)
		dev->rx_cbk(dev->rx_arg, USB_EVENT_RESUME, 0, NULL);

	return;
}

/**
  * @brief  Provides us with a time reference.
  * @param  device: BULK device.
  * @param  time: Elapsed time in milliseconds.
  * @retval None
  */
static void bulk_tick_handler(void *device, uint32_t time)
{
	uint32_t size;
	usbd_bulk_dev_t *dev = (usbd_bulk_dev_t *)device;

	assert_param(dev);

	/* Have a deferred receive waiting */
	if ((dev->inst.flag & (1 << BULK_DO_PACKET_RX)) && (dev->rx_cbk))  {
		/* Get size of the data */
		size = map_usb_ep_data_avail(dev->inst.oep);
		/* Call the callback function */
		dev->rx_cbk(dev->rx_arg, USB_EVENT_RX_AVAILABLE, size, NULL);
	}

	return;
}
/**
  * @}
  */

/** @defgroup Device_BULK_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes BULK device.
  * @param  idx: Index of the USB controller.
  * @param  dev: BULK device.
  * @retval Device structure.
  */
void *usbd_bulk_init(uint32_t idx, usbd_bulk_dev_t *dev)
{
	void *ret;
	device_desc_t *device_desc;
	config_desc_t *config_desc;

	assert_param(idx == 0);
	assert_param(dev);

	/* Composite initialization */
	ret = usbd_bulk_init_comp(idx, dev, 0);

	if (ret) {
		/* Fill the device descriptor */
		device_desc = (device_desc_t *)__bulk_device_desc;
		device_desc->idVendor  = dev->vid;
		device_desc->idProduct = dev->pid;
		/* Fill the configuration descriptor */
		config_desc = (config_desc_t *)__bulk_desc;
		config_desc->bmAttributes = dev->attr_pwr;
		config_desc->bMaxPower    = dev->max_power >> 1;

		/* Initialze device controler */
		usb_dcd_init(idx, &dev->inst.info, (void *)dev);
	}

	return ret;
}

/**
  * @brief  Initializes BULK device.
  * @param  idx: Index of the USB controller.
  * @param  dev: BULK device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_bulk_init_comp(uint32_t idx, usbd_bulk_dev_t *dev, comp_entry_t *entry)
{
	bulk_inst_t *inst;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->rx_cbk);
	assert_param(dev->tx_cbk);

	/* Create a pointer to the device instance */
	inst = &dev->inst;

	/* Initialze the composite entry */
	if (entry != 0) {
		entry->info = &inst->info;
		entry->inst = (void *)dev;
	}

	/* Initialize the device information */
	inst->info.cbk         = &bulk_handler;
	inst->info.device_desc = __bulk_device_desc;
	inst->info.config_desc = __bulk_config_desc;
	__max_packet_size      = 64;
#ifdef USBD_BULK_HS
	inst->info.config_desc = __bulk_config_desc_hs;
	__max_packet_size      = 512;
#endif
	inst->info.string_desc    = 0;
	inst->info.nr_string_desc = 0;

	/* Initialize the device instance structure */
	inst->usb_idx  = 0;
	inst->rx_state = BULK_STATE_UNCONFIG;
	inst->tx_state = BULK_STATE_UNCONFIG;
	inst->flag     = 0;
	inst->conn     = 0;

	/* Initialize the device information */
	usb_device_info_init(0, &inst->info);
	/* Initialize device instance endpoint */
	inst->iep = DATA_IN_ENDPOINT;
	inst->oep = DATA_OUT_ENDPOINT;
	inst->interface = 0;
	/* Initialize the device instance structure */
	inst->info.string_desc    = dev->desc_str;
	inst->info.nr_string_desc = dev->num_str;

	/* Initialze DMA */
	usb_dma_init(0);
	/* Initialze tick */
	usb_tick_init();
	/* Register tick callback fucntion */
	usb_tick_handler_register(bulk_tick_handler, (void *)dev);

	/* Configure DMA structure */
#ifdef USB_DMA_N_SUPPORT
	bulk_tx_cbk.cplt_cbk = usbd_bulk_dma_tx_cplt;
	bulk_tx_cbk.err_cbk  = usbd_bulk_dma_e_err;
	bulk_tx_cbk.cplt_arg = dev;
	bulk_tx_cbk.err_arg  = dev;
	bulk_rx_cbk.cplt_cbk = usbd_bulk_dma_rx_cplt;
	bulk_rx_cbk.err_cbk  = usbd_bulk_dma_e_err;
	bulk_rx_cbk.cplt_arg = dev;
	bulk_rx_cbk.err_arg  = dev;

	usb_dma_channel_alloc(dev->inst.iep, USB_E_DMA_TYPE_BULK_TX, &bulk_tx_cbk);
	usb_dma_channel_alloc(dev->inst.oep, USB_E_DMA_TYPE_BULK_RX, &bulk_rx_cbk);
#else
	bulk_cbk.cplt_cbk = usbd_bulk_dma_cplt;
	bulk_cbk.err_cbk  = usbd_bulk_dma_err;
	bulk_cbk.cplt_arg = dev;
	bulk_cbk.err_arg  = dev;

	dev->inst.idmach = usb_dma_channel_alloc(dev->inst.iep, USB_DMA_EP_CFG_TX, &bulk_cbk);
	dev->inst.odmach = usb_dma_channel_alloc(dev->inst.oep, USB_DMA_EP_CFG_RX_DEV, &bulk_cbk);
#endif
	return (void *)dev;
}

/**
  * @brief  Terminal the BULK device.
  * @param  dev: BULK device.
  * @retval None
  */
void usbd_bulk_term(usbd_bulk_dev_t *dev)
{
	assert_param(dev);

	/* Terminate the device controler */
	usb_dcd_term(dev->inst.usb_idx);
	dev->inst.usb_idx = 0;

	return;
}

/**
  * @brief  Set RX callback parameter.
  * @param  dev: BULK device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_bulk_set_rx_param(usbd_bulk_dev_t *dev, void *arg)
{
	void *old;

	assert_param(dev);

	/* Set the RX callback parameter */
	old = dev->rx_arg;
	dev->rx_arg = arg;

	return old;
}

/**
  * @brief  Set TX callback parameter.
  * @param  dev: BULK device.
  * @param  arg: Parameter.
  * @retval Old parameter.
  */
void *usbd_bulk_set_tx_param(usbd_bulk_dev_t *dev, void *arg)
{
	void *old;

	assert_param(dev);

	/* Set the TX callback parameter */
	old = dev->tx_arg;
	dev->tx_arg = arg;

	return old;
}

/**
  * @brief  Transmits a packet to the USB host.
  * @param  dev: BULK device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Indicates whether more data is to be written.
  * @retval Length.
  */
uint32_t usbd_bulk_packet_write(usbd_bulk_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last)
{
	int32_t ret;
	bulk_inst_t *inst = &dev->inst;

	assert_param(dev);

	/* Check the length and state */
	if ((len > __max_packet_size) || (inst->tx_state != BULK_STATE_IDLE))
		return 0;

	/* Put the data into the FIFO */
	ret = map_usb_ep_data_put(inst->iep, data, len);

	if (ret == -1)
		return 0;

	/* Save sent bytes */
	inst->size += len;

	/* Last call for a packet */
	if (last) {
		/* Set the TX_STATE to WAIT_DATA */
		inst->tx_state = BULK_STATE_WAIT_DATA;
		/* Send the data */
		ret = map_usb_ep_data_send(inst->iep, USB_TRANS_IN);
	}

	return ret != -1 ? len : 0;
}

/**
  * @brief  Received a packet from the USB host.
  * @param  dev: BULK device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Indicates whether more data is to be written.
  * @retval Status.
  */
uint32_t usbd_bulk_packet_read(usbd_bulk_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last)
{
	int ret;
	uint32_t status, cnt, _len;
	bulk_inst_t *inst = &dev->inst;

	assert_param(dev);

	/* Get the endpoint status */
	status = map_usb_ep_status(inst->oep);

	/* Check had receive a pecket */
	if ((status & USB_DEV_RX_PKT_RDY) == 0)
		return 0;

	/* Save the length */
	cnt  = len;
	/* Get the size of the data */
	_len = map_usb_ep_data_avail(inst->oep);
	/* Get data from FIFO */
	ret  = map_usb_ep_data_get(inst->oep, data, &cnt);

	if (cnt == _len) {
		/* Clear endpoint status */
		map_usb_dev_ep_status_clear(inst->oep, status);
		/* ACK the data */
		map_usb_dev_ep_data_ack(inst->oep, true);
		/* Reset the flag bit */
		config_bulk_flag(&inst->flag, BULK_DO_PACKET_RX, 0);
	}

	return ret != -1 ? cnt : 0;
}

/**
  * @brief  Transmits a packet to the USB host via DMA.
  * @param  dev: BULK device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Indicates whether more data is to be written.
  * @retval Length.
  */
uint32_t usbd_bulk_packet_write_by_dma(usbd_bulk_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last)
{
	bulk_inst_t *inst = &dev->inst;

	assert_param(dev);

	/* Check the length and state */
	if ((len > __max_packet_size) || (inst->tx_state != BULK_STATE_IDLE))
		return 0;

	__last = last;
	__len  = len;
#ifdef USB_DMA_N_SUPPORT
	usb_dma_config(USB_E_DMA_TYPE_BULK_TX, (uint32_t)data, len);
#else
	usb_dma_config(dev->inst.idmach, (uint32_t)data, len);
#endif
	return len;
}

/**
  * @brief  Received a packet from the USB host via DMA.
  * @param  dev: BULK device.
  * @param  data: Data buffer.
  * @param  len: Size of the buffer.
  * @param  last: Indicates whether more data is to be written.
  * @retval Status.
  */
uint32_t usbd_bulk_packet_read_by_dma(usbd_bulk_dev_t *dev, uint8_t *data, uint32_t len, uint8_t last)
{
	uint32_t status, _len;
	bulk_inst_t *inst = &dev->inst;

	assert_param(dev);

	/* Get the endpoint status */
	status = map_usb_ep_status(inst->oep);

	/* Check had received a pecket */
	if ((status & USB_DEV_RX_PKT_RDY) == 0)
		return 0;

	__last   = last;
	_len     = map_usb_ep_data_avail(inst->oep);
	__len    = _len;
	__status = status;

#ifdef USB_DMA_N_SUPPORT
	usb_dma_config(USB_E_DMA_TYPE_BULK_RX, (uint32_t)data, _len);
#else
	usb_dma_config(inst->odmach, (uint32_t)data, _len);
#endif	
	return _len;
}

/**
  * @brief  Get number of free bytes in the transmit buffer.
  * @param  dev: BULK device.
  * @retval Length.
  */
uint32_t usbd_bulk_tx_packet_avail(usbd_bulk_dev_t *dev)
{
	assert_param(dev);

	/* Check the TX_STATE */
	if (dev->inst.tx_state == BULK_STATE_IDLE)
		return __max_packet_size;

	return 0;
}

/**
  * @brief  Determine whether a packet is available.
  * @param  dev: BULK device.
  * @retval Length.
  */
uint32_t usbd_bulk_rx_packet_avail(usbd_bulk_dev_t *dev)
{
	uint32_t status;

	assert_param(dev);

	/* Get the endpoint status */
	status = map_usb_ep_status(dev->inst.oep);

	/* Had received a packet? */
	if (status & USB_DEV_RX_PKT_RDY)
		return map_usb_ep_data_avail(dev->inst.oep);

	return 0;
}

/**
  * @brief  Requests a remote wake up to resume communication.
  * @param  dev: BULK device.
  * @retval Status.
  */
int usbd_bulk_remote_wakeup_req(usbd_bulk_dev_t *dev)
{
	assert_param(dev);

	return usb_dcd_remote_wakeup_req(0);
}

/**
  * @brief  Set power status.
  * @param  dev: BULK device.
  * @param  power: Status of the power.
  * @retval None
  */
void usbd_bulk_power_status_set(usbd_bulk_dev_t *dev, uint8_t power)
{
	assert_param(dev);

	usb_dcd_power_status_set(0, power);
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
