/**
  *********************************************************************************
  *
  * @file    usbd_audio.h
  * @brief   Header file for the audio driver[Device].
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */

#ifndef __USBD_AUDIO_H__
#define __USBD_AUDIO_H__

#ifdef __cplusplus
extern "C"
{
#endif

#include "usbd_core.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @addtogroup Device_Audio
  * @{
  */
/**
  * @defgroup Device_Audio_Public_Macros Public Macros
  * @{
  */
#define AUDIO_DESCRIPTOR_SIZE	8
#define CTRL_INTERFACE_SIZE	52
#define STREAM_INTERFACE_SIZE	52
#define COMP_DAUDIO_SIZE	112

#define AUDIO_IN_TERMINAL_ID	1
#define AUDIO_OUT_TERMINAL_ID	2
#define AUDIO_CONTROL_ID	3
#define AUDIO_INTERFACE_CTRL	0
#define AUDIO_INTERFACE_OUTPUT	1
#define ISOC_OUT_EP_MAX_SIZE	192
#define ISOC_OUT_ENDPOINT       AUDIO_EP_OUT

#define USBD_AUDIO_EVENT_IDLE		(USBD_AUDIO_EVENT_BASE + 0)
#define USBD_AUDIO_EVENT_ACTIVE		(USBD_AUDIO_EVENT_BASE + 1)
#define USBD_AUDIO_EVENT_DATAOUT	(USBD_AUDIO_EVENT_BASE + 2)
#define USBD_AUDIO_EVENT_VOLUME		(USBD_AUDIO_EVENT_BASE + 4)
#define USBD_AUDIO_EVENT_MUTE		(USBD_AUDIO_EVENT_BASE + 5)
#define USBD_AUDIO_EVENT_DMA_ERR	(USBD_AUDIO_EVENT_BASE + 6)
/**
  * @}
  */

/**
  * @defgroup Device_Audio_Public_Types Public Types
  * @{
  */
typedef void (* usb_audio_cbk)(void *buf, uint32_t param, uint32_t event);

/**
  * @brief Define the private instance data.
  */
typedef struct {
	uint32_t usb_idx;	/**< Index of the USB controller */
	device_info_t info;	/**< Device information */
	int16_t vol_max;	/**< Maximum volume */
	int16_t vol_min;	/**< Minimum volume */
	int16_t vol_step;	/**< Minimum volume step */
	void *data;		/**< Buffer provided by caller */
	uint32_t size;		/**< Size of the data area */
	uint32_t nr;		/**< Number of valid bytes */
	usb_audio_cbk cbk;	/**< Callback for the buffer */
	uint16_t req_type;	/**< Request type */
	uint8_t request;	/**< Request */
	uint16_t update;	/**< Update value */
	int16_t volume;		/**< Current Volume setting */
	uint8_t mute;		/**< Current Mute setting */
	uint32_t rate;		/**< Current Sample rate */
	uint8_t oep;		/**< OUT endpoint */
	uint8_t odmach;		/**< OUT DMA channel */
	uint8_t interface_ctrl;	/**< Control interface */
	uint8_t interface_audio;/**< Audio interface */
	uint8_t use_dma;	/**< Enable/Disable DMA transfer data */
} audio_inst_t;

/**
  * @brief Define operating parameters for the Audio device
  */
typedef struct {
	const uint16_t vid;		/**< Vendor ID */
	const uint16_t pid;		/**< Product ID */
	const char vendor[8];		/**< Vendor string */
	const char product[16];		/**< Product string */
	const char version[4];		/**< Version string */
	const uint16_t max_power;	/**< Maximum power consumption */
	const uint8_t attr_pwr;		/**< Attribute of the power */
	const usb_cbk cbk;		/**< Callback */
	const uint8_t * const *desc_str;/**< String descriptor */
	const uint32_t num_str;		/**< Number of the string descriptors */
	const int16_t vol_max;		/**< Maximum volume */
	const int16_t vol_min;		/**< Minimum volume */
	const int16_t vol_step;		/**< Minimum volume step */
	audio_inst_t inst;		/**< Device instance */
} usbd_audio_dev_t;
/**
  * @}
  */

/** @addtogroup Device_Audio_Public_Functions
  * @{
  */
extern void *usbd_audio_init(uint32_t idx, usbd_audio_dev_t *dev);
extern void *usbd_audio_init_comp(uint32_t idx, usbd_audio_dev_t *dev, comp_entry_t *entry);
extern void usbd_audio_term(usbd_audio_dev_t *dev);
extern int32_t usbd_audio_data_out(usbd_audio_dev_t *dev, void *data, uint32_t size, usb_audio_cbk cbk);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
