/**
  *********************************************************************************
  *
  * @file    usbd_audio.c
  * @brief   USB Audio device driver.
  *
  * @version V1.0
  * @date    30 Jul 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          30 Jul 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  *
  * THIS SOFTWARE IS PROVIDED "AS IS" AND WITH ALL FAULTS.
  * NO WARRANTIES, WHETHER EXPRESS, IMPLIED OR STATUTORY, INCLUDING, BUT
  * NOT LIMITED TO, IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR
  * A PARTICULAR PURPOSE APPLY TO THIS SOFTWARE. EASTSOFT SHALL NOT, UNDER ANY
  * CIRCUMSTANCES, BE LIABLE FOR SPECIAL, INCIDENTAL, OR CONSEQUENTIAL
  * DAMAGES, FOR ANY REASON WHATSOEVER.
  *
  *********************************************************************************
  */
#include "usb_audio.h"
#include "usbd_audio.h"


/** @addtogroup USB_LIBRARY
  * @{
  */
/** @addtogroup DEVICE
  * @{
  */
/** @defgroup Device_Audio Audio
  * @brief Device Audio driver
  * @{
  */
/** @defgroup Device_Audio_Private_Variables Private Variables
  * @{
  */
/**
  * @brief DMA callback
  */
usb_dma_callback_t audio_dma_cbk;

#if !defined(USB_AUDIO_OUT_IN) && !defined(USB_AUDIO_OUT) && !defined(USB_AUDIO_IN)
#define USB_AUDIO_OUT_IN
#endif

/**
  * @brief Device descriptor
  */
static uint8_t __audio_device_desc[] = {
	18,                /**< Size of the descriptor */
	USB_DTYPE_DEVICE,  /**< Type of the descriptor */
	USBShort(0x110),   /**< Version 1.1 */
	0,                 /**< USB device class */
	0,                 /**< USB device sub-class */
	0,                 /**< USB device protocol */
	64,                /**< Maximum packet size */
	USBShort(0),       /**< VID */
	USBShort(0),       /**< PID */
	USBShort(0x100),   /**< Device version */
	1,                 /**< Manufacturer string */
	2,                 /**< Product string */
	3,                 /**< Manufacturer serial number */
	1                  /**< number of the configurations */
};

/**
  * @brief Configuration descriptor header
  */
static uint8_t __audio_desc[] = {
	9,                        /**< Size of the descriptor */
	USB_DTYPE_CONFIGURATION,  /**< Type of the descriptor */
	USBShort(32),             /**< Total size of the descriptor */
	2,                        /**< Number of the inerfaces */
	1,                        /**< Unique value for this configuration */
	0,                        /**< Index of the string */
	USB_CONF_ATTR_BUS_PWR,    /**< Type of the power */
	250,                      /**< Maximum power in 2mA increments */
};

/**
  * @brief Interface association descriptor
  */
static uint8_t __iad_audio_desc[AUDIO_DESCRIPTOR_SIZE] = {
	8,                       /**< Size of the descriptor */
	USB_DTYPE_INTERFACE_ASC, /**< Type of the descriptor */
	0x0,                     /**< Default starting interface */
	0x2,                     /**< Number of interface */
	USB_CLASS_AUDIO,         /**< Device class */
	USB_SUBCLASS_UNDEFINED,  /**< Device sub-class */
	USB_PROTOCOL_UNDEFINED,  /**< Device protocol */
	0                        /**< Index of the string */
};

/**
  * @brief IAD configuration section
  */
const config_section_t __iad_audio_config_sec = {
	sizeof(__iad_audio_desc),
	__iad_audio_desc
};

/**
  * @brief Audio control interface descriptor
  */
const uint8_t __audio_ctrl_interface[CTRL_INTERFACE_SIZE] = {
	9,                                         /**< Size of the interface */
	USB_DTYPE_INTERFACE,                       /**< Type of the interface */
	AUDIO_INTERFACE_CTRL,                      /**< Index of the interface */
	0,                                         /**< Alternate setting */
	0,                                         /**< Number of endpoints in the interface */
	USB_CLASS_AUDIO,                           /**< Interface class */
	USB_ASC_AUDIO_CONTROL,                     /**< Interface sub-class */
	0,                                         /**< Interface protocol */
	0,                                         /**< Index of the string */

	9,                                         /**< Size of the interface */
	USB_DTYPE_CS_INTERFACE,                    /**< Type of the interface */
	USB_ACDSTYPE_HEADER,                       /**< Sub-type */
	USBShort(0x0100),                          /**< Version */
	USBShort((9 + 9 + 12 + 13 + 9)),           /**< Total bytes */
	1,                                         /**< Number of the streaming */
	1,                                         /**< Index of the Streaming */

	12,                                        /**< Size of the interface */
	USB_DTYPE_CS_INTERFACE,                    /**< Type of the interface */
	USB_ACDSTYPE_IN_TERMINAL,                  /**< Sub-type */
	AUDIO_IN_TERMINAL_ID,                      /**< Terminal ID */
	USBShort(USB_TTYPE_STREAMING),             /**< Streaming */
	0,                                         /**< Output terminal ID */
	2,                                         /**< Number of the logical output channels */
	USBShort((USB_CHANNEL_L | USB_CHANNEL_R)), /**< Logical channels */
	0,                                         /**< String index of the channel name */
	0,                                         /**< String index of the terminal name */

	13,                                        /**< Size of the interface */
	USB_DTYPE_CS_INTERFACE,                    /**< Type of the interface */
	USB_ACDSTYPE_FEATURE_UNIT,                 /**< Sub-type */
	AUDIO_CONTROL_ID,                          /**< Control ID */
	AUDIO_IN_TERMINAL_ID,                      /**< Terminal ID */
	2,                                         /**< Size of an element */
	USBShort(USB_ACONTROL_MUTE),               /**< Master mute control */
	USBShort(USB_ACONTROL_VOLUME),             /**< Left volume control */
	USBShort(USB_ACONTROL_VOLUME),             /**< Right volume control */
	0,                                         /**< String index of the feature */

	9,                                         /**< Size of the interface */
	USB_DTYPE_CS_INTERFACE,                    /**< Type of the interface */
	USB_ACDSTYPE_OUT_TERMINAL,                 /**< Sub-type */
	AUDIO_OUT_TERMINAL_ID,                     /**< OUT Terminal ID */
	USBShort(USB_ATTYPE_SPEAKER),              /**< Output type */
	AUDIO_IN_TERMINAL_ID,                      /**< OUT Terminal ID */
	AUDIO_CONTROL_ID,                          /**< Control ID */
	0,                                         /**< String index of the output terninal */
};

/**
  * @brief Audio stream interface descriptor
  */
const uint8_t __audio_stream_interface[STREAM_INTERFACE_SIZE] = {
	9,                        /**< Size of the interface */
	USB_DTYPE_INTERFACE,      /**< Size of the interface */
	AUDIO_INTERFACE_OUTPUT,   /**< Index of the interface */
	0,                        /**< Alternate setting */
	0,                        /**< Number of endpoints in the interface */
	USB_CLASS_AUDIO,          /**< Interface class */
	USB_ASC_AUDIO_STREAMING,  /**< Interface sub-class */
	0,                        /**< Unused */
	0,                        /**< String index for the interface */

	9,                        /**< Size of the interface */
	USB_DTYPE_INTERFACE,      /**< Type of the interface */
	1,                        /**< Index of the interface */
	1,                        /**< Alternate setting */
	1,                        /**< Number of the endpointer */
	USB_CLASS_AUDIO,          /**< Interface class */
	USB_ASC_AUDIO_STREAMING,  /**< Interface sub-class */
	0,                        /**< Unused */
	0,                        /**< String index for the interface */

	7,                        /**< Size of the interface */
	USB_DTYPE_CS_INTERFACE,   /**< Type of the interface */
	USB_ASDSTYPE_GENERAL,     /**< General information */
	AUDIO_IN_TERMINAL_ID,     /**< Terminal ID */
	1,                        /**< Frame delay */
	USBShort(USB_ADF_PCM),    /**< PCM */

	11,                       /**< Size of the interface */
	USB_DTYPE_CS_INTERFACE,   /**< Type of the interface */
	USB_ASDSTYPE_FORMAT_TYPE, /**< Type of the audio streaming format */
	USB_AF_TYPE_TYPE_I,       /**< I audio format type */
	2,                        /**< Audio channels */
	2,                        /**< Two bytes per sub-frame */
	16,                       /**< Sample rate */
	1,                        /**< Number of sample rate */
	USB3Byte(48000),          /**< sample rate */

	9,                        /**< Size of the interface */
	USB_DTYPE_ENDPOINT,       /**< Type of the interface */
	USB_EP_DESC_OUT | ISOC_OUT_ENDPOINT, /**< Endpoint */
	USB_EP_ATTR_ISOC | USB_EP_ATTR_ISOC_ADAPT | USB_EP_ATTR_USAGE_DATA, /**< Endpoint */
	USBShort(ISOC_OUT_EP_MAX_SIZE), /**< Maximum packet size */
	1,                        /**< Polling interval */
	0,                        /**< Unused */
	0,                        /**< Endpoint address */

	7,                        /**< Size of the interface */
	USB_ACSDT_ENDPOINT,       /**< Type of the interface */
	USB_ASDSTYPE_GENERAL,     /**< General descriptor */
	USB_EP_ATTR_ACG_SAMPLING, /**< Sampling frequency */
	USB_EP_LOCKDELAY_UNDEF,   /**< Undefined lock delay */
	USBShort(0),              /**< Not lock delay */
};

/**
  * @brief Audio configuration section
  */
const config_section_t __audio_config_sec = {
	sizeof(__audio_desc),
	__audio_desc
};

/**
  * @brief Audio stream interface section
  */
const config_section_t __audio_stream_interface_sec = {
	sizeof(__audio_stream_interface),
	__audio_stream_interface
};

/**
  * @brief Audio control interface section
  */
const config_section_t __audio_ctrl_interface_sec = {
	sizeof(__audio_ctrl_interface),
	__audio_ctrl_interface
};

/**
  * @brief Audio section
  */
const config_section_t *__audio_sec[] = {
	&__audio_config_sec,
	&__iad_audio_config_sec,
	&__audio_ctrl_interface_sec,
	&__audio_stream_interface_sec
};

/**
  * @brief Number of the section
  */
#define NUM_AUDIO_SECTIONS      (sizeof(__audio_sec) / sizeof(__audio_sec[0]))

/**
  * @brief Audio configuration head
  */
const config_head_t __audio_config_head = {
	NUM_AUDIO_SECTIONS,
	__audio_sec
};

/**
  * @brief Audio configuration descriptor
  */
const config_head_t * const __audio_config_desc[] = {
	&__audio_config_head
};
/**
  * @}
  */

/** @defgroup Device_Audio_Private_Functions Private Functions
  * @{
  */
static void handle_request(void *device, usb_request_t *request);
static void handle_interface_change(void *device, uint8_t interface, uint8_t alt);
static void handle_config_change(void *device, uint32_t value);
static void handle_ep0_data(void *device, uint32_t info);
static void handle_disconnect(void *device);
static void handle_ep(void *device, uint32_t status);
static void handle_device(void *device, uint32_t request, void *data);
/**
  * @}
  */

/** @addtogroup Device_Audio_Private_Variables
  * @{
  */
/**
  * @brief Device handler
  */
static const base_handler_t audio_handler = {
	0,                       /**< Get descriptor */
	handle_request,          /**< Request handler */
	handle_interface_change, /**< Inerface change */
	handle_config_change,    /**< Configuration change */
	handle_ep0_data,         /**< Data receive */
	0,                       /**< Data sent callback */
	0,                       /**< Reset handler */
	0,                       /**< Suspend handler */
	0,                       /**< Resume handler */
	handle_disconnect,       /**< Disconnect handler */
	handle_ep,               /**< Endpoint handler */
	handle_device            /**< Device handler */
};
/**
  * @}
  */

/** @addtogroup Device_Audio_Private_Functions
  * @{
  */
#ifdef USB_DMA_N_SUPPORT
/**
  * @brief  Callback function when DMA transfer complete.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_audio_dma_cplt(void *arg)
{
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)arg;
	audio_inst_t *inst = &dev->inst;

	map_usb_dev_ep_data_ack(inst->oep, 0);
	inst->cbk(inst->data, inst->size, USBD_AUDIO_EVENT_DATAOUT);

	return;
}

/**
  * @brief  Callback function when DMA send error.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_audio_dma_err(void *arg)
{
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)arg;

	dev->cbk(dev, USBD_AUDIO_EVENT_DMA_ERR, 0, NULL);
	return;
}
#else

/**
  * @brief  Callback function when DMA transfer complete.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_audio_dma_cplt(uint8_t ch, void *arg)
{
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)arg;
	audio_inst_t *inst = &dev->inst;

	map_usb_dev_ep_data_ack(inst->oep, 0);
	inst->cbk(inst->data, inst->size, USBD_AUDIO_EVENT_DATAOUT);

	return;
}

/**
  * @brief  Callback function when DMA send error.
  * @param  ch: DMA channel.
  * @param  arg: Parameter.
  * @retval None
  */
static void usbd_audio_dma_err(uint8_t ch, void *arg)
{
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)arg;

	dev->cbk(dev, USBD_AUDIO_EVENT_DMA_ERR, 0, NULL);
	return;
}
#endif

/**
  * @brief  Handle non-standard request.
  * @param  device: Audio device.
  * @param  request: Non-standard request.
  * @retval None
  */
static void handle_request(void *device, usb_request_t *request)
{
	uint32_t ctrl, rec, stall = 0;
	audio_inst_t *inst;
	usbd_audio_dev_t *dev;

	/* Create a pointer to the device */
	dev  = (usbd_audio_dev_t *)device;
	/* Create a pointer to the device instance */
	inst = &dev->inst;
	/* Get the request type */
	rec  = request->bmRequestType & USB_RTYPE_RECIPIENT_M;
	/* ACK the data */
	map_usb_dev_ep_data_ack(USB_EP_0, false);
	/* Save the request type and value */
	inst->req_type = request->bmRequestType;
	inst->request  = request->bRequest;

	/* If it's an endpoint request to the streaming endpoint */
	if ((rec == USB_RTYPE_ENDPOINT) && (request->wIndex == inst->oep)) {
		switch (inst->request) {
		case USB_AC_SET_CUR:
			if (request->wValue == SAMPLING_FREQ_CONTROL) {
				/* Retrieving the sample rate */
				usb_dcd_ep0_data_req(0, (uint8_t *)&inst->rate, 3);
				/* Save the updating */
				inst->update = SAMPLING_FREQ_CONTROL;
			}

			break;
		case USB_AC_GET_CUR:
			/* Send back the sample rate */
			if (request->wValue == SAMPLING_FREQ_CONTROL)
				usb_dcd_ep0_data_send(0, (uint8_t *)&inst->rate, 3);
			
			break;
		default:
			/* Don't handle unknown request */
			stall = 1;
			break;
		}
	}
	else if (rec == USB_RTYPE_INTERFACE) {
		/* Make sure this request is to the control interface */
		if ((uint8_t)request->wIndex != inst->interface_ctrl)
			return;
		if ((AUDIO_CONTROL_ID << 8) != (request->wIndex & USB_CS_CONTROL_M))
			return;

		/* Get the control value */
		ctrl = request->wValue & USB_CS_CONTROL_M;

		switch (inst->request) {
		case USB_AC_GET_MAX:
			/* Get the maximum volume setting */
			if (ctrl == VOLUME_CONTROL)
				usb_dcd_ep0_data_send(0, (uint8_t *)&inst->vol_max, 2);
			else
				stall = 1;

			break;
		case USB_AC_GET_MIN:
			/* Get the minimum volume setting */
			if(ctrl == VOLUME_CONTROL)
				usb_dcd_ep0_data_send(0, (uint8_t *)&inst->vol_min, 2);
			else
				stall = 1;

			break;
		case USB_AC_GET_RES:
			/* Get the volume step setting */
			if (ctrl == VOLUME_CONTROL)
				usb_dcd_ep0_data_send(0, (uint8_t *)&inst->vol_step, 2);
			else
				stall = 1;

			break;
		case USB_AC_GET_CUR:
			/* Get the current volume value */
			if(ctrl == VOLUME_CONTROL)
				usb_dcd_ep0_data_send(0, (uint8_t *)&inst->volume, 2);
			else if(ctrl == MUTE_CONTROL)
				usb_dcd_ep0_data_send(0, (uint8_t *)&inst->mute, 1);
			else
				stall = 1;

			break;
		case USB_AC_SET_CUR:
			/* Set the current volume value */
			if(ctrl == VOLUME_CONTROL) {
				usb_dcd_ep0_data_req(0, (uint8_t *)&inst->volume, 2);
				inst->update = VOLUME_CONTROL;
			}
			else if(ctrl == MUTE_CONTROL) {
				usb_dcd_ep0_data_req(0, (uint8_t *)&inst->mute, 1);
				inst->update = MUTE_CONTROL;
			}
			else {
				stall = 1;
			}
			break;
		case USB_AC_SET_RES:
			/* Set the current volume step */
			if(ctrl == VOLUME_CONTROL) {
				usb_dcd_ep0_data_req(0, (uint8_t *)&inst->vol_step, 2);
				inst->update = VOLUME_CONTROL;
			}
			else {
				stall = 1;
			}

			break;
		default:
			/* Don't handle unknown request */
			stall = 1;
			break;
		}
	}

	/* Stall all unknow requests */
	if (stall)
		usb_dcd_ep0_stall(0);

	return;
}

/**
  * @brief  Handle device interface changes.
  * @param  device: Audio device.
  * @param  interface: Interface.
  * @param  alt: Alternal setting.
  * @retval None
  */
static void handle_interface_change(void *device, uint8_t interface, uint8_t alt)
{
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)device;

	/* Which interface to change */
	if (alt == 0)
		dev->cbk(dev, USBD_AUDIO_EVENT_IDLE, 0, NULL);
	else
		dev->cbk(dev, USBD_AUDIO_EVENT_ACTIVE, 0, NULL);

	return;
}

/**
  * @brief  Handle device configuration changes.
  * @param  device: Audio device.
  * @param  value: New value.
  * @retval None
  */
static void handle_config_change(void *device, uint32_t value)
{
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)device;

	/* Call the callback function */
	dev->cbk(dev, USB_EVENT_CONNECTED, 0, NULL);
	return;
}

/**
  * @brief  Handle EP0 data.
  * @param  device: Audio device.
  * @param  info: Information.
  * @retval None
  */
static void handle_ep0_data(void *device, uint32_t info)
{
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)device;
	audio_inst_t *inst = &dev->inst;

	/* Check which value was set */
	if (inst->update && (inst->request == USB_AC_SET_CUR)) {
		/* Interface request */
		if ((inst->req_type & USB_RTYPE_RECIPIENT_M) == USB_RTYPE_INTERFACE) {
			if (inst->update == VOLUME_CONTROL)
				dev->cbk(dev, USBD_AUDIO_EVENT_VOLUME, inst->volume, NULL);
			else if (inst->update == MUTE_CONTROL)
				dev->cbk(dev, USBD_AUDIO_EVENT_MUTE, inst->mute, NULL);
		}
		/* Clear the update pending */
		inst->update = 0;
	}

	return;
}

/**
  * @brief  Handle disconnect envent.
  * @param  device: Audio device.
  * @retval None
  */
static void handle_disconnect(void *device)
{
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)device;

	/* Call the callback function */
	dev->cbk(dev, USB_EVENT_DISCONNECTED, 0, NULL);
	return;
}

/**
  * @brief  Handle the interrupts on the endpoints.
  * @param  device: Audio device.
  * @param  status: Current status.
  * @retval None
  */
static void handle_ep(void *device, uint32_t status)
{
	uint32_t ep_status;
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)device;
	audio_inst_t *inst = &dev->inst;

	/* Get the current endpoint status */
	ep_status = map_usb_ep_status(inst->oep);

	/* If it's a receive interrupt pending */
	if (status & (0x10000 << inst->oep)) {
		/* Get the size of data in the FIFO */
		inst->size = map_usb_ep_data_avail(inst->oep);
		/* Clear the status */
		map_usb_dev_ep_status_clear(inst->oep, ep_status);

		/* Use DMA transfer */
		if (inst->use_dma) {
#ifdef USB_DMA_N_SUPPORT
		usb_dma_config(USB_E_DMA_TYPE_AUDIO_RX, (uint32_t)inst->data, inst->size);
#else
		usb_dma_config(inst->odmach, (uint32_t)inst->data, inst->size);
#endif
		}
		else {
			/* Read data from FIFO */
			map_usb_ep_data_get(inst->oep, (uint8_t *)inst->data, &inst->size);
			/* ACK the data */
			map_usb_dev_ep_data_ack(inst->oep, 0);
			/* Call the callback function */
			inst->cbk(inst->data, inst->size, USBD_AUDIO_EVENT_DATAOUT);
		}
	}

	return;
}

/**
  * @brief  Handle device envent.
  * @param  device: Audio device.
  * @param  request: Request.
  * @param  data: Data buffer.
  * @retval None
  */
static void handle_device(void *device, uint32_t request, void *data)
{
	uint8_t *buf = (uint8_t *)data;
	usbd_audio_dev_t *dev = (usbd_audio_dev_t *)device;
	audio_inst_t *inst = &dev->inst;

	switch (request) {
	case USB_EVENT_COMP_IFACE_CHANGE:
		/* Interface change event */
		if (buf[0] == AUDIO_INTERFACE_CTRL)
			inst->interface_ctrl = buf[1];
		else if (buf[0] == AUDIO_INTERFACE_OUTPUT)
			inst->interface_audio = buf[1];

		break;
	case USB_EVENT_COMP_EP_CHANGE:
		/* Endpoint change event */
		if ((buf[0] & USB_EP_DESC_IN) == 0) {
			if (inst->odmach != 0)
				usb_dma_channel_free(dev->inst.oep, USB_DMA_EP_CFG_RX_DEV);

			inst->oep    = buf[1] & 0x7f;
#ifdef USB_DMA_N_SUPPORT
			usb_dma_channel_alloc(inst->oep, USB_E_DMA_TYPE_AUDIO_RX, &audio_dma_cbk);
#else
			inst->odmach = usb_dma_channel_alloc(inst->oep, USB_DMA_EP_CFG_RX_DEV, &audio_dma_cbk);
#endif
		}
		break;

	case USB_EVENT_COMP_CONFIG:
		/* Handle class specific */
		buf[2] = inst->interface_ctrl;
		break;
	case USB_EVENT_LPM_RESUME:
		/* LPM resume event */
		dev->cbk(dev, USB_EVENT_LPM_RESUME, 0, NULL);
		break;
	case USB_EVENT_LPM_SLEEP:
		/* LPM sleep event */
		dev->cbk(dev, USB_EVENT_LPM_SLEEP, 0, NULL);
		break;
	case USB_EVENT_LPM_ERROR:
		/* LPM error event */
		dev->cbk(dev, USB_EVENT_LPM_ERROR, 0, NULL);
		break;
	default:
		break;
	}

	return;
}
/**
  * @}
  */

/** @defgroup Device_Audio_Public_Functions Public Functions
  * @{
  */
/**
  * @brief  Initializes audio device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Audio device.
  * @retval Device structure.
  */
void *usbd_audio_init(uint32_t idx, usbd_audio_dev_t *dev)
{
	config_desc_t *config_desc;
	device_desc_t *device_desc;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->cbk);

	/* Composite initialization */
	usbd_audio_init_comp(idx, dev, 0);

	/* Fill the device descriptor */
	device_desc = (device_desc_t *)__audio_device_desc;
	device_desc->idVendor  = dev->vid;
	device_desc->idProduct = dev->pid;

	/* Fill the configuration descriptor */
	config_desc = (config_desc_t *)__audio_desc;
	config_desc->bmAttributes = dev->attr_pwr;
	config_desc->bMaxPower = (uint8_t)(dev->max_power >> 1);

	/* Initialize device controler */
	usb_dcd_init(idx, &dev->inst.info, (void *)dev);
	return (void *)dev;
}

/**
  * @brief  Initializes audio device.
  * @param  idx: Index of the USB controller.
  * @param  dev: Audio device.
  * @param  entry: Composite entry.
  * @retval Device structure.
  */
void *usbd_audio_init_comp(uint32_t idx, usbd_audio_dev_t *dev, comp_entry_t *entry)
{
	audio_inst_t *inst = &dev->inst;

	assert_param(idx == 0);
	assert_param(dev);
	assert_param(dev->desc_str);
	assert_param(dev->cbk);

	/* Initialize the composite entry */
	if (entry != 0) {
		entry->info = &inst->info;
		entry->inst = (void *)dev;
	}

	/* Initialize the device information */
	inst->info.cbk            = &audio_handler;
	inst->info.device_desc    = __audio_device_desc;
	inst->info.config_desc    = __audio_config_desc;
	inst->info.string_desc    = 0;
	inst->info.nr_string_desc = 0;

	/* Initialize the device */
	usb_device_info_init(0, &inst->info);

	/* Initialze the device instance */
	inst->interface_ctrl      = AUDIO_INTERFACE_CTRL;
	inst->interface_audio     = AUDIO_INTERFACE_OUTPUT;
	inst->oep                 = ISOC_OUT_ENDPOINT;
	inst->odmach              = 0;
	inst->data                = NULL;
	inst->vol_max             = dev->vol_max;
	inst->vol_min             = dev->vol_min;
	inst->vol_step            = dev->vol_step;
	inst->update              = 0;
	inst->info.string_desc    = dev->desc_str;
	inst->info.nr_string_desc = dev->num_str;
	inst->use_dma             = 1;

	/* Initialize the DMA */
	usb_dma_init(0);
	audio_dma_cbk.cplt_cbk = usbd_audio_dma_cplt;
	audio_dma_cbk.err_cbk  = usbd_audio_dma_err;
	audio_dma_cbk.cplt_arg = dev;
	audio_dma_cbk.err_arg  = dev;
#ifdef USB_DMA_N_SUPPORT
	usb_dma_channel_alloc(dev->inst.oep, USB_E_DMA_TYPE_AUDIO_RX, &audio_dma_cbk);
#else
	dev->inst.odmach = usb_dma_channel_alloc(dev->inst.oep, USB_DMA_EP_CFG_RX_DEV, &audio_dma_cbk);
#endif
	return (void *)dev;
}

/**
  * @brief  Terminal the audio device.
  * @param  dev: Audio device.
  * @retval None
  */
void usbd_audio_term(usbd_audio_dev_t *dev)
{
	assert_param(dev);

	/* Terminal the device controler */
	usb_dcd_term(dev->inst.usb_idx);
	return;
}

/**
  * @brief  Supply buffers to the audio class to be filled from the USB host device.
  * @param  dev: Audio device.
  * @param  data: Data buffer.
  * @param  size: Size of the buffer.
  * @param  cbk: Notification when this buffer has valid data.
  * @retval Always 0.
  */
int32_t usbd_audio_data_out(usbd_audio_dev_t *dev, void *data, uint32_t size, usb_audio_cbk cbk)
{
	assert_param(dev);
	assert_param(data);
	assert_param(cbk);
	assert_param(size >= ISOC_OUT_EP_MAX_SIZE);

	/* Initialize the device instance */
	dev->inst.data = data;
	dev->inst.size = size;
	dev->inst.nr   = 0;
	dev->inst.cbk  = cbk;

	return 0;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
