/**********************************************************************************
 *
 * @file    micro_boot.h
 * @brief   head file of micro_boot.c, describe the API of MicroBoot middleware
 *
 * @version V1.0
 * @date    22 Feb  2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          22 Feb  2023    shicc           version:1.0.1	Notes:Migrate the enumeration types and structure types to the header file micro_boot
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 **********************************************************************************
 */

#ifndef __MICRO_BOOT_H__
#define __MICRO_BOOT_H__


/* Includes ---------------------------------------------------------------- */
#include<stdint.h>
#include<string.h>   

/* Exported Macros ----------------------------------------------------------- */

#ifdef ES_MICRO_BOOT__USE_EX_FILE_CONFIG_INFO    
/*外部文件配置*/
#include "es_micro_boot_config_info.h"
#else
/*默认配置*/
#define FRAME_INTERVAL   500    /* the longest time interval between received cmd frame and execution result frame */
#define BUF_SIZE (260)   /* rx tx buf size*/                   

#define ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_READ_M       (1)

#define ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GO           (1) 

#define ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_WRITE_M      (1)    
#define ES_MICRO_BOOT_CONFIG__WRITE_M_CMD_OPTION__SUPPORT_RAM   (1)   /*支持写RAM*/ 
#define ES_MICRO_BOOT_CONFIG__BOOT_RAM_START_ADDR               (0x20000000)      /*可以被写的RAM区域的首地址*/
#define ES_MICRO_BOOT_CONFIG__WRITE_M_CMD_OPTION__CHECK_MEM     (1)   /*写ROM后，校验*/

#define ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_EX_ERASE     (1)

#define ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_GET_CRC32    (1)

#define ES_MICRO_BOOT_CONFIG__SUPPORT_CMD_CHECK_EMPTY  (1)

#endif

/* Exported Types ---------------------------------------------------------- */

/**
  *@brief: if the step of state machine procedure is successful
            then return PASS(0x00) else FAIL(0x01)
  */
typedef enum fsm_result_e
{
    FAIL = 0x01,
    PASS = 0x00
} fsm_result_t;

/**
  *@brief: if IAP operation is successful then return IAP_SUCCESS(0x01)
            else IAP_FAIL(0x00)
  */
typedef enum fsm_iap_result_e
{
    IAP_FAIL    = 0,
    IAP_SUCCESS = 1
} fsm_iap_result_t;

/**
  *@brief: the description of state machine
  */
typedef struct isp_data_s
{
    uint32_t            u_frame_timer;                                     /* the time interval for frame control */
    void (*p_send_func)(uint8_t *, uint32_t);                              /* the msg send function of state machine */
    void (*p_recv_func)(uint8_t *, uint32_t);                              /* the msg receive function of state machine */
    fsm_result_t (*p_is_send_over)(void);                           /* return PASS to indicate all data expected to send has been read out by upper,
                                                                              and FAIL to indicate process of send is not complete */

    fsm_result_t (*p_is_recv_over)(void);                           /* return PASS to indicate all data expected to send has been read out by upper,
                                                                              and FAIL to indicate process of send is not complete */
    fsm_result_t (*p_check_empty)(uint32_t *, uint32_t);                   /* the function of state machine to check whether specified address is empty */
    uint32_t (*p_get_crc32)(uint32_t *, uint32_t);                         /* the function of state machine to calculate crc32*/
    void (*p_read_memo)(uint8_t *, void *, uint32_t);                      /* the boot function of state machine to read memory */
    void (*p_go)(uint32_t);                                          /* the boot function of state machine to boot app to run from specified address */
    fsm_iap_result_t (*p_page_erase)(uint32_t);                            /* the page erase function of state machine */
    fsm_iap_result_t (*p_words_progrm)(uint32_t, uint32_t, uint32_t);      /* the words program function of state machine */
} isp_data_t;


/* Imported Variables ----------------------------------------------------------- */

extern isp_data_t g_isp_data;

/**
  *@brief: if the procedure of state machine is successful
            then return POS_ACK(0x79) else NEG_ACK(0x1F)
  */
typedef enum fsm_resp_e
{
    POS_ACK = 0x79,
    NEG_ACK = 0x1F
} fsm_resp_t;

/**
  *@brief enum definition of state machine procedure
  */
typedef enum
{
    IN_START_RX = 0,
    IN_START_HANDLE = 1,
    IN_START_CMD_HANDLE = 2,
    IN_CMD_HANDLE_FUNC = 3,
    IN_FSM_STEP_OTHER = 4,
} fsm_step_t;
               
typedef enum
{
    ES_CMD_STEP_0 = 0,
    ES_CMD_STEP_1 = 1,
    ES_CMD_STEP_2 = 2,
    ES_CMD_STEP_3 = 3,
    ES_CMD_STEP_4 = 4,
    ES_CMD_STEP_5 = 5,
    ES_CMD_STEP_6 = 6,
    ES_CMD_STEP_7 = 7,
    ES_CMD_STEP_8 = 8,
} es_cmd_step_t;

/**
  *@brief enum definition of cmd supported by state machine
  */
typedef enum
{
    NONE = 0x00,
    START = 0x3F,
    GET = 0x80,
    GET_V = 0x81,
    GET_ID = 0x82,
    READ_M = 0x91,
    GO = 0xA1,
    WRITE_M = 0xB1,
    EX_ERASE = 0xC4,
    GET_CRC32 = 0xD0,
    CHECK_EMPTY = 0xD1
} fsm_cmd_t;

/**
  *@brief union definition used for sort
  */
typedef union
{
    uint32_t b32;
    uint8_t b08[4];
} u_b32b08_t;

/**
  *@brief struct definition used for describing the operation of state machine
  */
typedef struct fsm_data_struct
{
    uint8_t      syn;
    fsm_step_t   step;
    fsm_cmd_t    cmd;
    uint8_t      trigger;            /* the trigger state of state machine */
    uint16_t     rx_len;             /* the byte length state machine needs to receive */
    uint16_t     tx_len;             /* the byte length state machine needs to send */
    u_b32b08_t   para1;              /* the 1th parameter received by state machine */
    u_b32b08_t   para2;              /* the 2th parameter received by state machine */
    uint8_t      rx_buf[BUF_SIZE];   /* the receive buffer of state machine */
    uint8_t      tx_buf[BUF_SIZE];   /* the send buffer of state machine */
} fsm_data_type_t;

/* Private Variables--------------------------------------------------------- */

/**
  *@brief struct variable definition used to operate state machine
  */
extern fsm_data_type_t    s_fsm_data;

/* Exported Constants -------------------------------------------------------- */


/* Exported Variables -------------------------------------------------------- */


/* Exported Functions Prototypes------------------------------------------------------ */

/**
  * @brief  only function clarification, it need to be complted by user
  *         it's used to assign specified functions completed by user which is designed to process the communication
  *         to state machine function pointer, it must be called before using void fsm_para_init(void).
  * @param  None
  * @retval None
  */
void fsm_comm_func_init(void);

/**
  * @brief  only function clarification, it need to be complted by user
  *         it's used to assign specified functions completed by user which is designed to process the execution steps of
  *         the state machine to state machine function pointer, it must be called before using void fsm_para_init(void).
  * @param  None
  * @retval None
  */
void fsm_exec_func_init(void);


/* Exported Functions ------------------------------------------------------ */

/**
  * @brief  the function is designed to init parameters of state machine, it can only be used after function
  *         void fsm_comm_func_init(void) and void fsm_exec_func_init(void) called.
  * @param  None
  * @retval None
  */
void fsm_para_init(void);

/**
  * @brief  the main function of state machine.
  * @param  None
  * @retval None
  */
void proc_fsm(void);

/**
  * @brief  the function is designed to detect whether state machine is triggered, it'll return 1 if triggered, else 0.
  * @param  None
  * @retval
  */
uint8_t fsm_is_trigger(void);

/**
  * @brief  the function is designed to process the subsequent logic of state machine and copy the received data to the data
  *         object of state machine and when receive enough bytes, it must be called when it has received all expected data.
  * @param  None
  * @retval None
  */
void fsm_recv_over(void);

/**
  * @brief  the function is designed to process the subsequent logic of state machine when send out all expected bytes,
  *         it must be called when it has sent over.
  * @param  None
  * @retval None
  */
void fsm_send_over(void);

#endif  /* */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

