/**
  *********************************************************************************
  *
  * @file    rbtree.h
  * @brief   rbtree interface
  *
  * @version V1.0
  * @date    05 Jun 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          05 Jun 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __RBTREE_H__
#define __RBTREE_H__

#include "type.h"


/** @addtogroup Micro-Kernel EASTSOFT Micro-Kernel
  * @{
  */

/** @addtogroup rb_tree rb_tree Management
  * @brief    red black tree module
  * @{
  */

/** @defgroup RBTREE_Public_Macros Public Macros
  * @{
  */
#define rb_entry(ptr, type, member) \
        ((type *)(uint32_t)((char *)(ptr) - (unsigned long)(&((type *)0)->member)))
/**
  * @}
  */

/** @defgroup RBTREE_Public_Types Public Types
  * @{
  */

/**
  * @brief Defines node's color.
  */
typedef enum {
	RB_RED   = 0,	/**< Red */
	RB_BLACK = 1,	/**< Black */
} __rb_color_t;

/**
  * @brief Defines node structure.
  */
typedef struct rb_node_s {
	struct rb_node_s *parent;	/**< Parent of the node */
	struct rb_node_s *left;		/**< Left son of the node */
	struct rb_node_s *right;	/**< Right son of the node */
	__rb_color_t color;		/**< Color of the node */
} rb_node_t;

/**
  * @brief Defines compare function.
  */
typedef int (*rb_node_cmp)(rb_node_t *node, void *key);

/**
  * @brief Defines root structure.
  */
typedef struct rb_root_s {
	rb_node_t *node;	/**< Pointer to the root */
} rb_root_t;

/**
  * @brief Defines rb_tree structure.
  */
typedef struct rb_tree_s {
	uint32_t size;		/**< Size of the rb_tree */
	rb_root_t root;		/**< Root of the rb_tree */
	rb_node_cmp cmp;	/**< Compare function */
} rb_tree_t;
/**
  * @}
  */

/** @addtogroup RBTREE_Public_Functions
  * @{
  */
extern void rb_tree_init(rb_tree_t *tree, rb_node_cmp cmp);
extern void rb_tree_insert(rb_tree_t *tree, rb_node_t *node, void *key);
extern rb_node_t *rb_tree_find(rb_tree_t *tree, void *key);
extern uint32_t rb_tree_get_size(rb_tree_t *tree);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#endif
