/**
  *********************************************************************************
  *
  * @file    ke.h
  * @brief   Kernel definition.
  *
  * @version V1.0
  * @date    02 Mar 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          02 Mar 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __KE_H__
#define __KE_H__

#ifdef __cplusplus
extern "C" {
#endif

#include <stdlib.h>
#include "type.h"
#include "list.h"
#include "utils.h"
#include "ald_uart.h"


/** @addtogroup Micro-Kernel EASTSOFT Micro-Kernel
  * @{
  */

/** @defgroup ke_base Kernel Base
  * @brief    kernel base module
  * @{
  */

/** @defgroup KE_Private_Macros Private Macros
  * @{
  */

#define KE_SHELL

#define TASK_PRIO_TMR		15
#define TASK_PRIO_SHELL		30
#define TASK_PRIO_IDLE		31
	
#define KE_ENTER_CRITICAL()		\
do {					\
	cpu_sr = __get_PRIMASK();	\
	__disable_irq();		\
} while (0)

#define KE_EXIT_CRITICAL()	__set_PRIMASK(cpu_sr)

#define KE_PANIC() 	\
do {			\
	while (1)	\
		;	\
} while (0)
/**
  * @}
  */

/** @defgroup KE_Public_Types Public Types
  * @{
  */
typedef void (*task_cbk_t)(void *arg);
typedef void (*tmr_cbk_t)(void *arg);

/**
  * @brief Defines semaphore structure.
  */
typedef struct ke_sem_s {
	uint16_t count;		/**< Counter */
	uint16_t task_prio;	/**< The priority of the task */
} ke_sem_t;

/**
  * @brief Defines queue structure.
  */
typedef struct ke_que_s {
	list_head_t list;	/**< Link for list */

	uint16_t nr;		/**< Number of the messages in the queue */
	uint16_t task_prio;	/**< The priority of the task */
} ke_que_t;

/**
  * @brief Defines message queue structure.
  */
typedef struct ke_msgq_s {
	list_head_t list;	/**< Link for list */
	uint16_t nr;		/**< Number of the messages in the queue */
} ke_msgq_t;

/**
  * @brief Defines message structure.
  */
typedef struct ke_msg_s {
	list_head_t link;	/**< Link for list */

	void *msg;		/**< Pointer to message */
	void *hook;		/**< Pointer to hook function */
	void *arg;		/**< Pointer to the parameter */
	uint32_t len;		/**< Length of buffer */
	uint32_t rsv;		/**< Reserve variable */
} ke_msg_t;

/**
  * @brief Defines timer structure.
  */
typedef struct ke_tmr_s {
	list_head_t link;	/**< Link for list */

	uint32_t expire;	/**< Delay in time units */
	uint32_t _expire;	/**< Temporary varible */
	uint32_t period;	/**< Period in period mode */
	tmr_cbk_t func;		/**< Callback function of the timer */
	void *arg;		/**< Parameter of the callback function */
} ke_tmr_t;
/**
  * @}
  */
/**
  * @}
  */

/* Initialize functions */
void ke_mem_init(uint8_t *addr, uint32_t size);
void ke_task_init(void);
void ke_tmr_init(void);
void ke_tmr_start(void);
#ifdef KE_SHELL
void ke_shell_init(uart_handle_t *hperh);
#endif

/* Internal functions */
void task_schedule(void);
void task_show(void);
void mem_show(void);
void tmr_show(void);

/* extern functions */
#if defined (__ICCARM__)
extern void ke_init(uint32_t ram_size, uart_handle_t *hperh, uint32_t heap_size);
#else
extern void ke_init(uint32_t ram_size, uart_handle_t *hperh);
#endif

extern void ke_start(void);
extern void ke_show(void);
__weak void idle_func(void *arg);	/* This function can be overwritten in user file. */
__weak void ke_10ms_irq_cbk(void);	/* This function can be overwritten in user file. */

extern int task_create(task_cbk_t func, void *arg, int prio);
extern int task_delete(int prio);
extern int task_suspend(int prio);
extern int task_resume(int prio);

extern ke_sem_t *sem_create(int prio);
extern void sem_delete(ke_sem_t *sem);
extern int sem_pend(ke_sem_t *sem);
extern void sem_post(ke_sem_t *sem);

extern ke_que_t *que_create(int prio);
extern void que_delete(ke_que_t *que);
extern void que_enq(ke_que_t *que, ke_msg_t *msg);
extern ke_msg_t *que_deq(ke_que_t *que);
extern uint32_t que_get_number(ke_que_t *que);
extern void msgq_init(ke_msgq_t *msgq);
extern void msgq_enq(ke_msgq_t *msgq, ke_msg_t *msg);
extern ke_msg_t *msgq_deq(ke_msgq_t *msgq);

extern ke_tmr_t *tmr_create(uint32_t expire, uint32_t period, tmr_cbk_t func, void *arg);
extern int tmr_modify(ke_tmr_t *tmr, uint32_t expire, uint32_t period, tmr_cbk_t func, void *arg);
extern void tmr_delete(ke_tmr_t *tmr);
extern void tmr_start(ke_tmr_t *tmr);
extern void tmr_stop(ke_tmr_t *tmr);
extern uint32_t get_kernel_tick(void);

extern void *ke_malloc(uint32_t size);
extern void ke_free(void *ptr);

#ifdef KE_SHELL
extern int32_t shell_cmd_insert(char *cmd, void *func, uint32_t nr_arg);
extern void shell_modify_password(char *pw);
extern void printf_s(const char *fmt, ...);
#else
#define printf_s(x, ...)
#endif

/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif /* __KE_H__ */
