/**********************************************************************************
 *
 * @file    pid_c.c
 * @brief   PID implement in C
 *
 * @date    24 Jan 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          24 Jan 2024     AE              the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


/* Includes ------------------------------------------------------------------*/
#include "es32_dsp.h"


/* Private typedef -----------------------------------------------------------*/
/* Private define ------------------------------------------------------------*/
/* Private macro -------------------------------------------------------------*/
/* Private variables ---------------------------------------------------------*/
int16_t PrevError = 0, IntTerm = 0;
int16_t PrevError_C = 0, IntTerm_C = 0;

/* Private function prototypes -----------------------------------------------*/
/* Private functions ---------------------------------------------------------*/



/**
  * @brief  PID in C, Error computed outside the routine
  * @param ror: difference between reference and measured value
  *   Coeff: pointer to the coefficient table
  * @retval : PID output (command)
  */
int16_t do_pid(int16_t Error, int16_t *Coeff)
{
  int16_t Kp, Ki, Kd, Output;

  Kp = Coeff[0];
  Ki = Coeff[1];
  Kd = Coeff[2];

  IntTerm_C += Ki*Error;
  Output = Kp * Error;
  Output += IntTerm_C;
  Output += Kd * (Error - PrevError_C);

  PrevError_C = Error;

  return (Output);
}



/**
  * @brief  PID in C, Error computed inside the routine
  * @param : Input (measured value)
  *   Ref: reference (target value)
  *   Coeff: pointer to the coefficient table
  * @retval : PID output (command)
  */
int16_t do_full_pid(int16_t In, int16_t Ref, int16_t *Coeff)
{
  int16_t Kp, Ki, Kd, Output, Error;

  Error = Ref - In;
  Kp = Coeff[0];
  Ki = Coeff[1];
  Kd = Coeff[2];

  IntTerm_C += Ki*Error;
  Output = Kp * Error;
  Output += IntTerm_C;
  Output += Kd * (Error - PrevError_C);

  PrevError_C = Error;

  return (Output);
}

void reset_pid_c(int16_t prev_err,int16_t int_term)
{
    PrevError_C=prev_err;
    IntTerm_C=int_term;
}
void reset_pid_asm(int16_t prev_err,int16_t int_term)
{
    PrevError = prev_err;
    IntTerm = int_term;
}
