/**
  *********************************************************************************
  * @file   boot_frame.c
  * @brief  boot frames module.
  *
  * @version V1.0
  * @date    31 Dec 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <string.h>
#include <stdio.h>
#include "boot_frame.h"
#include "boot_flash.h"
#include "boot_shell.h"


/** @addtogroup Bootloader
  * @{
  */
/** @defgroup Frame Frame
  * @brief    Bootloader Frame Module
  * @{
  */
/** @defgroup Frame_Public_Variables Public Variable
  * @brief    Frame Public Variable
  * @{
  */
boot_frame_env_t frame_env;
crc_handle_t h_crc;
/**
  * @}
  */

/** @defgroup Frame_Private_Functions Private Functions
  * @brief    Frame Private Functions
  * @{
  */
/**
  * @brief  Send command to tramsmitor
  * @param  cmd: Command.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_cmd_send(xmod_tx_cmd_t cmd)
{
    uint8_t _cmd = cmd;

    if ((ald_uart_send(frame_env.h_uart, &_cmd, 1, 100)) != OK)
    {
        frame_env.state = UART_SEND_ERROR;
        return -1;
    }

    return 0;
}

/**
  * @brief  Check frame index.
  * @param  idx: frame index.
  * @retval 0-success, other value indicates failed.
  */
static int check_idx(uint8_t idx)
{
    uint8_t _idx  = 0;
    uint8_t next_idx = 0;

    _idx = frame_env.idx;
    next_idx = _idx == 0xFF ? 0 : _idx + 1;

    if ((next_idx != idx) && (idx != _idx))
        return 1;

    return 0;
}

#ifdef XMODE_1K
/**
  * @brief  Receive data by 1K_XMODE frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_1K_recv(void)
{
    uint16_t crc, _crc;

    if ((ald_uart_recv(frame_env.h_uart, &frame_env.soh, 1, TIME_OUT)) == TIMEOUT)
    {
        frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if (frame_env.soh == XMODE_EOT)
    {
        frame_env.state = FRAME_SUCCESS;
        return 1;
    }

    if ((ald_uart_recv(frame_env.h_uart, frame_env.data, 1028, TIME_OUT)) == TIMEOUT)
    {
        frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if ((frame_env.data[0] ^ frame_env.data[1]) != 0xFF)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    if ((check_idx(frame_env.data[0])) != 0)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    CRC_RESET(&h_crc);
    _crc = ald_crc_calculate(&h_crc, &frame_env.data[2], 1024);
    crc = (frame_env.data[1026] << 8) | frame_env.data[1027];

    if (crc != _crc)
    {
        frame_env.state = FRAME_VERIFY_ERROR;
        return -3;
    }

    if (write_data_to_flash(frame_env.idx, &frame_env.data[2], crc))
    {
        frame_env.state = FLASH_WRITE_ERROR;
        return -4;
    }

    ++frame_env.idx;
    return 0;
}
#endif

#ifdef XMODE_128
/**
  * @brief  Receive data by XMODE frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_128_recv(void)
{
    uint16_t crc, _crc;
    uint8_t xmode_buf[132];
    uint8_t tmp = 0;
    uint16_t page = 0;

    memset(xmode_buf, 0, sizeof(xmode_buf));

    if ((ald_uart_recv(frame_env.h_uart, &frame_env.soh, 1, TIME_OUT)) == TIMEOUT)
    {
        frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if (frame_env.soh == XMODE_EOT)
    {
        frame_env.state = FRAME_SUCCESS;

        if (frame_env.idx % 8 == 7)
            return 1;

        CRC_RESET(&h_crc);
        crc = ald_crc_calculate(&h_crc, frame_env.data, 1024);

        page = frame_env.idx / 8;

        if (write_data_to_flash(page, frame_env.data, crc))
        {
            memset(frame_env.data, 0, sizeof(frame_env.data));
            frame_env.state = FLASH_WRITE_ERROR;
            return -4;
        }

        return 1;
    }

    if ((ald_uart_recv(frame_env.h_uart, xmode_buf, 132, TIME_OUT)) == TIMEOUT)
    {
        frame_env.state = FRAME_RECEIVE_TIMEOUT;
        return -1;
    }

    if ((xmode_buf[0] ^ xmode_buf[1]) != 0xFF)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    if ((check_idx(xmode_buf[0])) != 0)
    {
        frame_env.state = FRAME_INDEX_ERROR;
        return -2;
    }

    CRC_RESET(&h_crc);
    _crc = ald_crc_calculate(&h_crc, &xmode_buf[2], 128);
    crc = (xmode_buf[130] << 8) | xmode_buf[131];

    if (crc != _crc)
    {
        frame_env.state = FRAME_VERIFY_ERROR;
        return -3;
    }

    tmp = (frame_env.idx % 8);
    memcpy((char *)&frame_env.data[tmp * 128], (const char *)&xmode_buf[2], 128);

    if (tmp == 7)
    {
        CRC_RESET(&h_crc);
        crc = ald_crc_calculate(&h_crc, frame_env.data, 1024);

        page = frame_env.idx / 8;

        if (write_data_to_flash(page, frame_env.data, crc))
        {
            memset(frame_env.data, 0, sizeof(frame_env.data));
            frame_env.state = FLASH_WRITE_ERROR;
            return -4;
        }

        memset(frame_env.data, 0, sizeof(frame_env.data));
    }

    ++frame_env.idx;
    return 0;
}
#endif

/**
  * @brief  Receive a frame.
  * @retval 0-success, other value indicates failed.
  */
static int xmod_frame_recv(void)
{
    int result = 0;
#ifdef XMODE_128
    result = xmod_128_recv();
#endif

#ifdef XMODE_1K
    result = xmod_1K_recv();
#endif
    return result;
}

/**
  * @}
  */

/** @defgroup Frame_Public_Functions Public Functions
  * @brief    Frame Public Functions
  * @{
  */

/**
  * @brief  Show update info
  * @retval none.
  */
void boot_update_info(void)
{
    if (frame_env.state == FRAME_SUCCESS)
    {
        printf_e("\r\nUpdate program success !");
    }
    else if (frame_env.state == FRAME_INDEX_ERROR)
    {
        printf_e("\r\nUpdate program error: FRAME_INDEX_ERROR !!!\r\n");
    }
    else if (frame_env.state == FRAME_RECEIVE_TIMEOUT)
    {
        printf_e("\r\nUpdate program error: FRAME_RECEIVE_TIMEOUT !!!\r\n");
    }
    else if (frame_env.state == FRAME_VERIFY_ERROR)
    {
        printf_e("\r\nUpdate program error: FRAME_VERIFY_ERROR !!!\r\n");
    }
    else if (frame_env.state == UART_SEND_ERROR)
    {
        printf_e("\r\nUpdate program error: UART_SEND_ERROR !!!\r\n");
    }
    else if (frame_env.state == FLASH_WRITE_ERROR)
    {
        printf_e("\r\nUpdate program error: FLASH_WRITE_ERROR !!!\r\n");
    }
    else
    {
    }

    if (frame_env.state != FRAME_SUCCESS)
        flash_usr_page_erase();

    return;
}

/**
  * @brief  Setup CRC for xmodem.
  * @retval 0-success, other value indicates failed.
  */
void xmod_crc_setup()
{
    /* Clear crc_handle_t structure */
    memset(&h_crc, 0x0, sizeof(crc_handle_t));

    /* Initialize CRC */
    h_crc.perh          = CRC;
    h_crc.init.mode     = CRC_MODE_CCITT;
    h_crc.init.seed     = 0x0;
    h_crc.init.data_rev = DISABLE;
    h_crc.init.data_inv = DISABLE;
    h_crc.init.chs_rev  = DISABLE;
    h_crc.init.chs_inv  = DISABLE;
    ald_crc_init(&h_crc);
}

/**
  * @brief  Initialize frame module
  * @param  hperh: Pointer to uart_handle_t structure.
  * @retval None
  */
void boot_frame_init(uart_handle_t *hperh)
{
    if (hperh == NULL)
        return;

    memset(&frame_env, 0, sizeof(boot_frame_env_t));
    frame_env.h_uart = hperh;

    return;
}

/**
  * @brief  Check app flash and copy info data to last page
  * @retval 0 if success, others if error
  */
int check_app(void)
{
	//Check app
	uint32_t page=0;
	uint8_t* data;
	uint32_t dcrc=0;
	uint32_t dlen=0;
	uint32_t*info_data;
	
	#ifdef XMODE_128
		page=frame_env.idx / 8-1;
	#endif
	#ifdef XMODE_1K
		page = frame_env.idx - 1;
	#endif
	data = (uint8_t*)USR_FLASH_ADDER(page);
	if (!VALID_APP_ADDR((uint32_t)(data + 1024 - 4)))
	{
		return -1; /*User app page invalid (receive error)*/
	}
	info_data = (uint32_t*)(data + INFO_ADDR_IN_PAGE);
	dcrc = info_data[0];
	dlen = info_data[1];
	
	if (!VALID_APP_ADDR(APP_FLASH_S + dlen - 1))
	{
		return -3; /*User app length invalid */
	}
	
	if (page!=APP_LAST_PAGE)
	{
		/* We have to move app_info to the last page of app_flash */
		ald_iap_program_word(APP_FLASH_INFO_S, dcrc);
		ald_iap_program_word(APP_FLASH_INFO_S + 4, dlen);
	}
	printf_e("App length=%u\r\n",dlen);
	return check_app_crc();
}

/**
  * @brief  download bootloader program.
  * @retval 0-success, other value indicates failed.
  */
int  boot_image_update(void)
{
    int ret;
    xmod_crc_setup();

    frame_env.idx   = 0;
    frame_env.cnt   = 0;
    frame_env.state = FRAME_SUCCESS;
    xmod_cmd_send(XMOD_TX_CMD_START);
    while (1)
    {
        ret = xmod_frame_recv();

        if ((ret < 0) && (++frame_env.cnt < 10))
        {
            if (frame_env.idx == 0)
                xmod_cmd_send(XMOD_TX_CMD_START);
            else
                xmod_cmd_send(XMOD_TX_CMD_NCK);

            continue;
        }

        if (ret < 0)
        {
            xmod_cmd_send(XMOD_TX_CMD_TERM);
            return ret;
        }

        if (ret == 1)
        {
            xmod_cmd_send(XMOD_TX_CMD_ACK);
            break;
        }

        xmod_cmd_send(XMOD_TX_CMD_ACK);
    }
    
    if (frame_env.state == FRAME_SUCCESS)
    {
        if (check_app())
	{
	    frame_env.state = APP_VERIFY_ERROR;
	    return -5;
	}
    }
    
    return 0;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
