/**
  *********************************************************************************
  *
  * @file    boot_fatfs.c
  * @brief   Fatfs file for DEMO
  *
  * @version V1.0
  * @date    31 Dec 2019
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          31 Dec 2019     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "boot_fatfs.h"
#include "utils.h"
#include "string.h"
#include "ald_iap.h"
#include "ald_syscfg.h"

/** @addtogroup Bootloader Bootloader
  * @{
  */
/** @defgroup USB_Fatfs Fatfs
  * @brief    USB Fatfs Module
  * @{
  */
/** @defgroup Fatfs_Public_Variables Public Variables
  * @brief    Fatfs Public Variables
  * @{
  */
fat_type_e FATFS_TYPE = TYPE_UKNOW;
const uint8_t binName[11] = {'I', 'M', 'A', 'G', 'E', ' ', ' ', ' ', 'B', 'I', 'N'};
const uint8_t binName_S[11] = {'i', 'm', 'a', 'g', 'e', ' ', ' ', ' ', 'b', 'i', 'n'};
const uint8_t BS_jmpBoot[3] = {0xEB, 0x3C, 0x90};
const uint8_t BS_OEMName[8] = {0x4D, 0x53, 0x44, 0x4F, 0x53, 0x35, 0x2E, 0x30};
const uint8_t BPB_BytsPerSec[2] = {0x00, 0x04};
const uint8_t BPB_SecperClus[1] = {0x01};
const uint8_t BPB_RsvSecCnt[2] = {0x01, 0x00};
const uint8_t BPB_NumFATs[1] = {0x02};
const uint8_t BPB_RootEntCnt[2] = {0x00, 0x02};
#ifdef FATFS
const uint8_t BPB_TotSec16[2] = {0xE0, 0x01};
#else
const uint8_t BPB_TotSec16[2] = {0xF0, 0x00};
#endif
const uint8_t BPB_Media[1] = {0xF8};
const uint8_t BPB_FATSz16[2] = {0x01, 0x00};
const uint8_t BPB_SecPerTrk[2] = {0x01, 0x00};
const uint8_t BPB_NumHeads[2] = {0x00, 0x00};
const uint8_t BPB_HiddSec[4] = {0x00, 0x00, 0x00, 0x00};
const uint8_t BPB_TotSec32[4] = {0x00, 0x00, 0x00, 0x00};
const uint8_t BS_drvNum[1] = {0x80};
const uint8_t BS_Reserved1[1] = {0x00};
const uint8_t BS_BootSig[1] = {0x29};
const uint8_t BS_VolID[4] = {0x1C, 0x6B, 0x44, 0xBA};
const uint8_t BS_VolLab[11] = {0x4E, 0x4F, 0x20, 0x4E, 0x41, 0x4D, 0x45, 0x20, 0x20, 0x20, 0x20};
const uint8_t BS_FilSysType[8] = {0x46, 0x41, 0x54, 0x31, 0x32, 0x20, 0x20};
const uint8_t SingaTure[2] = {0x55, 0xAA};
const uint8_t FAT[3] = {0xF8, 0xFF, 0xFF};

static uint8_t tmp_buf_1k[1024] = {0};

#if defined(ALD_BIN_HDR) && defined(USBD_FATFS)
static uint32_t crc_table[256] = {0};
#endif
/**
  * @}
  */

/** @defgroup Fatfs_Private_Functions Private Functions
  * @brief    Fatfs Private Functions
  * @{
  */
/**
  * @brief  transform little port data to big port.
  * @param  dat: data source
  * @param  buf_len: data length
  * @retval None
  */
static uint32_t buf_lb2bb(uint8_t *dat, uint8_t buf_len)
{
    uint32_t temp = 0;
    uint32_t fact = 1;
    uint8_t i = 0;
    uint8_t len = buf_len;

    for (i = 0; i < len; i++)
    {
        temp += dat[i] * fact;
        fact *= 256;
    }

    return temp;
}

/**
  * @brief  get data cluster origination section.
  * @param  src: BDR section
  * @retval origination data section number
  */
static uint32_t get_dataclust_sec(void *src)
{
    uint32_t FirstDataSec   = 0;
    uint32_t RootDirSectors = 0;
    uint32_t RootEntCnt     = 0;
    uint32_t BytesPerSec    = 0;
    uint32_t FATSz          = 0;
    uint32_t RsvdSecCnt     = 0;
    uint32_t NumFATs        = 0;

    dbr_t *pDbr = (dbr_t *)src;
    fat32_dbr_t *pDbr32 = (fat32_dbr_t *)src;

    if (FATFS_TYPE == TYPE_UKNOW)
        return 0;

    RootEntCnt = buf_lb2bb(pDbr->BPB_RootEntCnt, 2);
    BytesPerSec = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);
    RsvdSecCnt = buf_lb2bb(pDbr->BPB_RsvSecCnt, 2);
    NumFATs = buf_lb2bb(pDbr->BPB_NumFATs, 1);

    if (FATFS_TYPE == TYPE_FAT32)
    {
        FATSz = buf_lb2bb(pDbr32->fat.BPB_FATSz32, 4);
    }
    else
    {
        FATSz = buf_lb2bb(pDbr->BPB_FATSz16, 2);
    }

    if (BytesPerSec == 0)
        return 0;

    RootDirSectors = ((RootEntCnt * 32) + (BytesPerSec - 1)) / BytesPerSec;
    FirstDataSec = RsvdSecCnt + (NumFATs * FATSz) + RootDirSectors;

    return FirstDataSec;
}

#if defined(ALD_BIN_HDR) && defined(USBD_FATFS)
/**
  * @brief  get specified cluster FAT.
  * @param  src: BDR section
  * @param  nClust: specified cluster number.
  * @retval specified cluster FAT value.
  */
static uint32_t get_curclust_fat(void *src, uint32_t nClust)
{
    uint32_t tmp;
    uint32_t FATClusEntryVal32 = 0;
    uint16_t FATClusEntryVal16 = 0;
    uint16_t FATClusEntryVal12 = 0;
    uint32_t ThisFATEntOffset  = 0;
    uint32_t FATOffset         = 0;
    uint32_t BytesPerSec       = 0;
    uint8_t *fat_buf           = NULL;

    dbr_t *pDbr = (dbr_t *)src;

    if (FATFS_TYPE == TYPE_UKNOW)
        return 0;

    if (FATFS_TYPE == TYPE_FAT32)
    {
        FATOffset = nClust * 4;
    }
    else if (FATFS_TYPE == TYPE_FAT16)
    {
        FATOffset = nClust * 2;
    }
    else
    {
        FATOffset = nClust + (nClust / 2);
    }

    BytesPerSec = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);

    if (BytesPerSec == 0)
        return 0;

    ThisFATEntOffset = (FATOffset % BytesPerSec);
    fat_buf = (uint8_t *)src + buf_lb2bb(pDbr->BPB_RsvSecCnt, 2) * BytesPerSec;

    if (FATFS_TYPE == TYPE_FAT32)
    {
        FATClusEntryVal32 = (*(uint32_t *) & fat_buf[ThisFATEntOffset]) & 0x0FFFFFF;
        tmp = buf_lb2bb((uint8_t *)&FATClusEntryVal32, 4) & 0x0FFFFFF;
        FATClusEntryVal32 = tmp;

        return FATClusEntryVal32;
    }
    else if (FATFS_TYPE == TYPE_FAT16)
    {
        FATClusEntryVal16 = (*(uint16_t *) & fat_buf[ThisFATEntOffset]);
        tmp = buf_lb2bb((uint8_t *)&FATClusEntryVal16, 2);
        FATClusEntryVal16 = tmp;

        return FATClusEntryVal16;
    }
    else
    {
        FATClusEntryVal12 = *(uint16_t *) & fat_buf[ThisFATEntOffset];
        tmp = buf_lb2bb((uint8_t *)&FATClusEntryVal12, 2);
        FATClusEntryVal12 = tmp;

        if (nClust & 0x0001)
        {
            FATClusEntryVal12 = FATClusEntryVal12 >> 4;
        }
        else
        {
            FATClusEntryVal12 = FATClusEntryVal12 & 0x0FFF;
        }
    }

    return FATClusEntryVal12;
}


/**
  * @brief  get specified cluster value.
  * @param  src: BDR section
  * @param  cruclu: specified cluster
  * @param  buf: loading specified cluster value
  * @retval None.
  */
static uint32_t get_curfat_val(void *src, uint32_t cruclu, uint8_t *buf)
{
    uint32_t clu_val = 0;
    uint32_t BytesPerSec = 0;
    uint8_t *pData   = NULL;
    dbr_t *pDbr = (dbr_t *)src;

    if (src == NULL)
        return clu_val;

    BytesPerSec = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);
    pData = (uint8_t *)src + get_dataclust_sec(src) * BytesPerSec;
    clu_val = get_curclust_fat(src, cruclu);

    if (curfat_is_bat(clu_val))
        return clu_val;

    if (cruclu >= 2)
        pData = pData + (cruclu - 2) * BytesPerSec;

    memcpy(buf, pData, BytesPerSec);

    return clu_val;
}

/**
  * @brief  bin file crc check algorithm.
  * @param  data: pointer of the data to be checked.
  * @param  size: length of the data to be checked.
  * @retval crc check value.
  */
static uint32_t bin_calc_crc(uint8_t *data, uint32_t size)
{
    uint32_t i, j;
    uint32_t r;
    uint32_t crc_poly = 0xEDB88320;
    uint32_t crc = 0xFFFFFFFF;
    uint8_t *p = data;

    for (i = 0; i < 256; i++)
    {
        r = i;

        for (j = 0; j < 8; ++j)
        {
            if (r & 1)
            {
                r = (r >> 1) ^ crc_poly;
            }
            else
            {
                r >>= 1;
            }
        }

        crc_table[i] = r;
    }

    for (; size > 0; --size, ++p)
    {
        crc = crc_table[((uint8_t)crc) ^ *p] ^ (crc >> 8);
    }

    return (crc ^ 0xFFFFFFFF);
}

#endif

#ifdef USBD_FATFS
/**
  * @brief  initialize fs BDR.
  * @param  buf: BDR section
  * @retval None.
  */
static void init_fat12_bdr(uint8_t *buf)
{
    if (buf == NULL)
        return ;

    memcpy(&buf[0], BS_jmpBoot, 3);
    memcpy(&buf[3], BS_OEMName, 8);
    memcpy(&buf[11], BPB_BytsPerSec, 2);
    memcpy(&buf[13], BPB_SecperClus, 1);
    memcpy(&buf[14], BPB_RsvSecCnt, 2);
    memcpy(&buf[16], BPB_NumFATs, 1);
    memcpy(&buf[17], BPB_RootEntCnt, 2);
    memcpy(&buf[19], BPB_TotSec16, 2);
    memcpy(&buf[21], BPB_Media, 1);
    memcpy(&buf[22], BPB_FATSz16, 2);
    memcpy(&buf[24], BPB_SecPerTrk, 2);
    memcpy(&buf[26], BPB_NumHeads, 2);
    memcpy(&buf[28], BPB_HiddSec, 4);
    memcpy(&buf[32], BPB_TotSec32, 4);
    memcpy(&buf[36], BS_drvNum, 1);
    memcpy(&buf[37], BS_Reserved1, 1);
    memcpy(&buf[38], BS_BootSig, 1);
    memcpy(&buf[39], BS_VolID, 4);
    memcpy(&buf[43], BS_VolLab, 11);
    memcpy(&buf[54], BS_FilSysType, 8);
    memcpy(&buf[510], SingaTure, 2);

    return ;
}

/**
  * @brief  initialize fs FAT.
  * @param  buf: FAT section
  * @retval None.
  */
static void init_fat12_fat(uint8_t *buf)
{
    if (buf == NULL)
        return ;

    memcpy(&buf[0], FAT, 3);

    return ;
}
#endif

/**
  * @}
  */

/** @defgroup Fatfs_Public_Functions Public Functions
  * @brief    Fatfs Public Functions
  * @verbatim
 ===============================================================================
              ##### USB flash functions #####
 ===============================================================================
   [..]   This section provides functions allowing to:
      (+) get file system type.
      (+) check specified FAT status.
      (+) Initialize FAT12 file system.
      (+) get specified file information.
      (+) read data from fs.
      (+) write data to fs.
      (+) Write data to app flash specified address.
      (+) Write data to app ram specified address.
      (+) get bin file header struct.
      (+) check bin crc value.
      (+) program software in ram.
      (+) program software in flash.
    @endverbatim
  * @{
  */
/**
  * @brief  get specified FAT Section.
  * @param  src: BDR section
  * @param  crufat: specified FAT
  * @retval specified FAT correspond Section.
  */
uint32_t get_curfat_sec(void *src, uint32_t crufat)
{
    uint32_t FirstDataSec;
    uint32_t curFatSec;
    dbr_t *pDbr = (dbr_t *)src;
    uint32_t SecPerClus;

    SecPerClus = buf_lb2bb(pDbr->BPB_SecperClus, 1);
    FirstDataSec = get_dataclust_sec(src);

    if (crufat > 2)
    {
        curFatSec = FirstDataSec + SecPerClus * (crufat - 2);
    }
    else
    {
        curFatSec = FirstDataSec;
    }

    return curFatSec;
}

/**
  * @brief  get root cluster section.
  * @param  src: BDR section
  * @retval section number of the root cluster.
  */
uint32_t get_rootclust_sec(void *src)
{
    uint32_t RootDirSectors = 0;
    uint32_t FATSz          = 0;
    uint32_t RsvdSecCnt     = 0;
    uint32_t NumFATs        = 0;

    dbr_t *pDbr = (dbr_t *)src;
    fat32_dbr_t *pDbr32 = (fat32_dbr_t *)src;

    if (FATFS_TYPE == TYPE_UKNOW)
        return 0;

    RsvdSecCnt = buf_lb2bb(pDbr->BPB_RsvSecCnt, 2);
    NumFATs = buf_lb2bb(pDbr->BPB_NumFATs, 1);

    if (FATFS_TYPE == TYPE_FAT32)
    {
        FATSz = buf_lb2bb(pDbr32->fat.BPB_FATSz32, 4);
    }
    else
    {
        FATSz = buf_lb2bb(pDbr->BPB_FATSz16, 2);
    }

    RootDirSectors = RsvdSecCnt + (NumFATs * FATSz);

    return RootDirSectors;
}

/**
  * @brief  get fatfs type.
  * @param  src: BDR section
  * @retval None
  */
void get_fatfs_type(void *src)
{
    uint32_t RootDirSectors = 0;
    uint32_t RootEntCnt     = 0;
    uint32_t BytesPerSec    = 0;
    uint32_t FATSz          = 0;
    uint32_t RsvdSecCnt     = 0;
    uint32_t NumFATs        = 0;
    uint32_t TotSec         = 0;
    uint32_t DataSec        = 0;
    uint32_t CntofClust     = 0;
    uint32_t SecPerClus     = 0;

    dbr_t *pDbr = (dbr_t *)src;
    fat32_dbr_t *pDbr32 = (fat32_dbr_t *)src;
    uint8_t *pflag = (uint8_t *)src;

    if (src == NULL)
    {
        FATFS_TYPE = TYPE_UKNOW;
        return;
    }

    if ((pflag[510] != 0x55) || (pflag[511] != 0xAA))
    {
        FATFS_TYPE = TYPE_UKNOW;
        return;
    }

    RootEntCnt = buf_lb2bb(pDbr->BPB_RootEntCnt, 2);
    BytesPerSec = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);

    if (BytesPerSec == 0)
        return;

    RootDirSectors = ((RootEntCnt * 32) + (BytesPerSec - 1)) / BytesPerSec;
    FATSz = buf_lb2bb(pDbr->BPB_FATSz16, 2);
    TotSec = buf_lb2bb(pDbr->BPB_TotSec16, 2);
    NumFATs = buf_lb2bb(pDbr->BPB_NumFATs, 1);
    SecPerClus = buf_lb2bb(pDbr->BPB_SecperClus, 1);
    RsvdSecCnt = buf_lb2bb(pDbr->BPB_RsvSecCnt, 2);

    if (FATSz == 0)
        FATSz = buf_lb2bb(pDbr32->fat.BPB_FATSz32, 4);

    if (TotSec == 0)
        TotSec = buf_lb2bb(pDbr->BPB_TotSec32, 4);

    DataSec = TotSec - (RsvdSecCnt + (NumFATs * FATSz) + RootDirSectors);

    if (SecPerClus == 0)
        return;

    CntofClust = DataSec / SecPerClus;

    if (CntofClust < 4085)
    {
        FATFS_TYPE = TYPE_FAT12;
    }
    else if (CntofClust < 65525)
    {
        FATFS_TYPE = TYPE_FAT16;
    }
    else
    {
        FATFS_TYPE = TYPE_FAT32;
    }

    return;
}

/**
  * @brief  check specified FAT if is eof.
  * @param  FATContent: specified FAT number.
  * @retval 0-success, other value indicates failed.
  */
type_bool_t curfat_is_eof(uint32_t FATContent)
{
    if (FATFS_TYPE == TYPE_FAT12)
    {
        if (FATContent >= 0x0FF8)
            return TRUE;
    }
    else if (FATFS_TYPE == TYPE_FAT16)
    {
        if (FATContent >= 0xFFF8)
            return TRUE;
    }
    else if (FATFS_TYPE == TYPE_FAT32)
    {
        if (FATContent >= 0x0FFFFFF8)
            return TRUE;
    }
    else
    {

    }

    return FALSE;
}
/**
  * @brief  check specified FAT if is bat.
  * @param  FATContent: specified FAT number.
  * @retval 0-success, other value indicates failed.
  */
type_bool_t curfat_is_bat(uint32_t FATContent)
{
    if (FATFS_TYPE == TYPE_FAT12)
    {
        if (FATContent == 0x0FF7)
            return TRUE;
    }
    else if (FATFS_TYPE == TYPE_FAT16)
    {
        if (FATContent == 0xFFF7)
            return TRUE;
    }
    else if (FATFS_TYPE == TYPE_FAT32)
    {
        if (FATContent == 0x0FFFFFF7)
            return TRUE;
    }
    else
    {

    }

    return FALSE;
}

/**
  * @brief  get specified file information.
  * @param  src: BDR section
  * @param  fName: specified file name
  * @param  f_len: loading specified file length
  * @retval specified file first FAT value.
  */
uint32_t get_dir_param(void *src, const uint8_t *fName, uint32_t *f_len)
{
    uint8_t *pRootDir = NULL;
    uint8_t tmp = 0;
    uint32_t start_fat = 0;
    uint16_t FileSz[2] = {0};
#if defined USBD_FATFS
    uint32_t RootClustSec = 0;
    uint32_t BytesPerSec = 0;
    dbr_t *pDbr = (dbr_t *)src;

    RootClustSec = get_rootclust_sec(src);
    BytesPerSec  = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);

    pRootDir = (uint8_t *)src + (BytesPerSec * RootClustSec);
#endif

#if defined USBH_FATFS
    pRootDir = src;
#endif

    while (*pRootDir != 0x00)
    {
        if (pRootDir[0] != 0xE5)
        {
            if ((pRootDir[11] & ATTR_ARCHIEVE) != ATTR_ARCHIEVE)
            {
                pRootDir += 32;
                continue;
            }
            else
            {
                for (tmp = 0; tmp < 11; tmp++)
                {
                    if ((pRootDir[tmp] != binName[tmp]) && (pRootDir[tmp] != binName_S[tmp]))
                    {
                        break;
                    }
                }

                if (tmp == 11)
                {
                    start_fat = pRootDir[27];
                    start_fat = (start_fat << 8) + pRootDir[26];
                    FileSz[0] = (pRootDir[31] << 8) + pRootDir[30];
                    FileSz[1] = (pRootDir[29] << 8) + pRootDir[28];
                    *f_len = (FileSz[0] << 16) + FileSz[1];

                    return start_fat;
                }
            }
        }

        pRootDir += 32;
    }

    *f_len = (FileSz[0] << 16) + FileSz[1];

    return start_fat;
}

#ifdef USBD_FATFS
/**
  * @brief  initialize a FAT12 fs.
  * @retval None.
  */
void init_fat12_fs(void)
{
    uint16_t i = 0;

    memset(tmp_buf_1k, 0, 1024);
    init_fat12_bdr(tmp_buf_1k);
    ald_iap_program_words(SECTION_ADDR(0), tmp_buf_1k, DISK_BLOCK_SIZE, ENABLE);

    memset(tmp_buf_1k, 0, sizeof(tmp_buf_1k));
    init_fat12_fat(tmp_buf_1k);
    ald_iap_program_words(SECTION_ADDR(1), tmp_buf_1k, DISK_BLOCK_SIZE, ENABLE);
    ald_iap_program_words(SECTION_ADDR(2), tmp_buf_1k, DISK_BLOCK_SIZE, ENABLE);

    memset(tmp_buf_1k, 0, sizeof(tmp_buf_1k));

    for (i = 3; i < DISK_BLOCK_NR; i++)
        ald_iap_program_words(SECTION_ADDR(i), tmp_buf_1k, DISK_BLOCK_SIZE, ENABLE);

    return;
}
/**
  * @brief  read data of fs section.
  * @param  buf: pointer of the data to be read.
  * @param  sector: first read section.
  * @param  num_block: length of the section to be read.
  * @retval None.
  */
void fs_flash_read(uint8_t *buf, uint32_t sector, uint32_t num_block)
{
    uint8_t *p;
    uint32_t i = 0;

    p = (uint8_t *)SECTION_ADDR(sector);

    for (i = 0; i < num_block * DISK_BLOCK_SIZE; i++)
    {
        buf[i] = (*p);
        p++;
    }

    return;
}
/**
  * @brief  write data to fs section.
  * @param  buf: pointer of the data to be write.
  * @param  sector: first write section.
  * @param  num_block: length of the section to be write.
  * @retval None.
  */
void fs_flash_write(uint8_t *buf, uint32_t sector, uint32_t num_block)
{
    ald_iap_program_words(SECTION_ADDR(sector), buf, DISK_BLOCK_SIZE * num_block, ENABLE);

    return;
}
#endif
/**
  * @brief  write data to the app ram.
  * @param  buf: pointer of the data to be write.
  * @param  len: length of the data to be write.
  * @retval None.
  */
void app_ram_write(uint8_t *buf, uint32_t len)
{
    uint8_t *p = (uint8_t *)APP_RAM_S;

    if ((buf == NULL) || (len == 0))
        return;

    memcpy(p, buf, len);

    return;
}
/**
  * @brief  write data to the app flash.
  * @param  buf: pointer of the data to be write.
  * @param  len: length of the data to be write.
  * @retval None.
  */
void app_flash_write(uint8_t *buf, uint32_t len)
{
    uint16_t page = 0;
    uint16_t remainder = len % 1024;
    uint8_t *p = buf;

    if ((buf == NULL) || (len == 0))
        return;

    memset(tmp_buf_1k, 0, 1024);
    page = len / 1024;

    ald_iap_program_words(APP_FLASH_S, p, 1024 * page, ENABLE);
    p = p + 1024 * page;

    if (remainder != 0)
    {
        memcpy(tmp_buf_1k, p, remainder);
        ald_iap_program_words(APP_FLASH_S + 1024 * page, tmp_buf_1k, 1024, ENABLE);
    }

    return;
}

/**
  * @brief  get bin file header struct.
  * @param  src: pointer of the fatfs DBR section.
  * @param  b_hdr: loader bin file header struct.
  * @param  StFAT: the bin file first FAT.
  * @retval pointer of the program bin address.
  */
uint8_t *get_bin_hdr(void *src, bin_hdr_t *b_hdr, uint32_t StFAT)
{

#if defined(ALD_BIN_HDR) && defined(USBD_FATFS)
    uint8_t *Sec0 = (uint8_t *)FAT_FLASH_S;
#endif
    uint32_t BytesPerSec = 0;
    uint8_t *pBin = NULL;
    dbr_t *pDbr = (dbr_t *)src;

    if (b_hdr == NULL)
        return pBin;

    BytesPerSec = buf_lb2bb(pDbr->BPB_BytsPerSec, 2);
    pBin = (uint8_t *)src + get_dataclust_sec(src) * BytesPerSec;

#if defined(ALD_BIN_HDR) && defined(USBD_FATFS)
    memset(tmp_buf_1k, 0, 1024);
    get_curfat_val(Sec0, StFAT, tmp_buf_1k);
    b_hdr->vid = buf_lb2bb(tmp_buf_1k, 4);
    b_hdr->pid = buf_lb2bb(tmp_buf_1k + 4, 4);
    b_hdr->addr = buf_lb2bb(tmp_buf_1k + 8, 4);
    b_hdr->len = buf_lb2bb(tmp_buf_1k + 12, 4);
    b_hdr->crc = buf_lb2bb(tmp_buf_1k + 16, 4);
    pBin += 32;
#endif

    return pBin;
}

#if defined(ALD_BIN_HDR) && defined(USBD_FATFS)
/**
  * @brief  check bin file crc value.
  * @param  data: pointer of the bin file to be checked.
  * @param  b_hdr: pointer of the check bin file header struct.
  * @retval 0-success, other value indicates failed.
  */
type_bool_t check_bin_crc(uint8_t *data, bin_hdr_t *b_hdr)
{
    uint32_t crc_val = 0;
    uint8_t *p = data;

    if ((p == NULL) || (b_hdr->len == 0))
        return FALSE;

    crc_val = bin_calc_crc(p, b_hdr->len);

    if (crc_val != b_hdr->crc)
        return FALSE;

    return TRUE;
}
#endif

/**
  * @brief  Run in flash
  * @retval None.
  */
void run_in_flash(void)
{
    uint32_t addr = 0;

    SYSCFG_UNLOCK();
    SYSCFG_BOOTFLASH_MAPPING_DISABLE();
    SYSCFG_LOCK();

    addr = *(uint32_t *)(APP_FLASH_S + 4) ;
    ald_vtor_config(APP_FLASH_S, ENABLE);
    __set_MSP(*(uint32_t *)APP_FLASH_S);

    ((void(*)(void))(addr))();

    return;
}
/**
  * @brief  Run in ram
  * @retval None.
  */
void run_in_ram(void)
{
    uint32_t addr = 0;

    addr = *(uint32_t *)(APP_RAM_S + 4) ;
    ald_vtor_config(APP_RAM_S, ENABLE);
	__set_MSP(*(uint32_t *)APP_RAM_S);
    ((void(*)(void))(addr))();

    return;
}

/**
  * @brief  Set run flag
  * @param  flag: run in flash or run
  * @param  size: App code size
  * @retval None.
  */
void set_run_flag(uint8_t flag, uint32_t size)
{
    memset(tmp_buf_1k, 0, 1024);

    memcpy(tmp_buf_1k, (uint8_t *)0x7DC00, 1024);
    tmp_buf_1k[1016] = flag;
    *(uint32_t *)&tmp_buf_1k[1020] = size;

    ald_iap_program_words(0x7DC00, tmp_buf_1k, 1024, ENABLE);

    return;
}

/**
  * @brief  Get Run flag
  * @param  size: App code size
  * @retval run flag.
  */
uint8_t get_run_flag(uint32_t *size)
{
    uint32_t *p = NULL;
    uint32_t flag = 0;

    p = (uint32_t *)0x7DC00;
    flag = (*(p + 254)) & 0xFF;
    *size = *(p + 255);

    return flag;
}
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
