/**********************************************************************************
 *
 * @file    md_rcu.c
 * @brief   md_rcu C file
 *
 * @date    09 Apr. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          09 Apr. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "system_es32m0502.h"
#include "md_rcu.h"
#include "md_fc.h"
#include "md_tick.h"

/* Private Variables --------------------------------------------------------- */

static uint32_t  TICKms;
static uint32_t  TICK100us;
static uint32_t  TICK10us;

/* Private Macros ------------------------------------------------------------ */

#define TICK_CLOCKSOURCE 16000000

/* Public Variables ---------------------------------------------------------- */

uint32_t    VCO_BUF[9]             = {2, 4, 6, 8, 12, 16, 24, 32, 64};

/* Private Constants --------------------------------------------------------- */
/* Private function prototypes ----------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @addtogroup MD_RCU
  * @{
  */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup MD_RCU_Public_Functions
  * @{
  */
/** @addtogroup RCU_Function
  * @{
  */
void  md_tick_init_rcu(void)
{
    TICKms = TICK_CLOCKSOURCE / 1000;
    TICK100us = TICK_CLOCKSOURCE / 10000;
    TICK10us = TICK_CLOCKSOURCE / 100000;

    md_tick_set_rvr_reload(TICK, ((1 << 24) - 1));            /*  Maximun ReLoad  */
    md_tick_set_cvr_current(TICK, 0);                         /*  Reset CVR  */
    md_tick_set_csr_clksrc(TICK, MD_SYSTICK_CLKSRC_HCLK);     /*  MCU Clock  */
    md_tick_enable_csr_enable(TICK);                          /*  Counter Enable  */
}

void  md_tick_waitms_rcu(uint8_t Unit, uint16_t msCnt)
{
    uint32_t  mstime;

    mstime = (1 << 24) - (Unit * TICKms);

    while (msCnt--)
    {
        md_tick_set_cvr_current(TICK, ((1 << 24) - 1)); /*  Reset CVR  */

        while (md_tick_get_cvr_current(TICK) > mstime);
    }
}

void  md_tick_wait100us_rcu(uint16_t Unit, uint16_t usCnt)
{
    uint32_t  ustime;

    ustime = (1 << 24) - (Unit * TICK100us);

    while (usCnt--)
    {
        md_tick_set_cvr_current(TICK, ((1 << 24) - 1)); /*  Reset CVR  */

        while (md_tick_get_cvr_current(TICK) > ustime);
    }
}

void  md_tick_wait10us_rcu(uint16_t Unit, uint16_t usCnt)
{
    uint32_t  ustime;

    ustime = (1 << 24) - (Unit * TICK10us);

    while (usCnt--)
    {
        md_tick_set_cvr_current(TICK, ((1 << 24) - 1)); /*  Reset CVR  */

        while (md_tick_get_cvr_current(TICK) > ustime);
    }
}

void md_rcu_check_hosc_ready(void)
{
    md_tick_init_rcu();

    md_tick_waitms_rcu(1, 1);

    while (md_rcu_is_active_flag_hosc_ready(RCU) == MD_RCU_HOSCRDY_NOT_READY)
    {
        md_rcu_disable_hosc(RCU);

        md_tick_waitms_rcu(1, 1);

        md_rcu_enable_hosc(RCU);

        md_tick_waitms_rcu(1, 1);

    }
}

void md_rcu_sys_init(RCU_TypeDef *rcu, md_rcu_init_typedef *RCU_InitStruct)
{
    uint32_t    PLL_FRQ;
    double      fration;

    md_fc_set_read_latency(FC, MD_FC_WAIT_BETWEEN_48MHz_AND_72Mhz);

    if (RCU_InitStruct->HS_Clock & RCU_CON_PLLON)
        md_rcu_enable_pll(rcu);
    else
        md_rcu_disable_pll(rcu);

    if (RCU_InitStruct->HS_Clock & RCU_CON_HOSCON)
        md_rcu_enable_hosc(rcu);
    else
        md_rcu_disable_hosc(rcu);

    if (RCU_InitStruct->HS_Clock & RCU_CON_HRCON)
        md_rcu_enable_hrc(rcu);
    else
        md_rcu_disable_hrc(rcu);

    if (RCU_InitStruct->LS_Clock & RCU_LCON_LRCON)
        md_rcu_enable_lrc(rcu);
    else
        md_rcu_disable_lrc(rcu);

    /* Need to check */
    /* make sure HOSC CLK Ready */
    if ((RCU_InitStruct->HS_Clock & RCU_CON_HOSCON))
        md_rcu_check_hosc_ready();

    while (
        ((RCU->CON  & RCU_CON_PLLON) && (!md_rcu_is_active_flag_pll_ready(rcu)))
        || ((RCU->CON  & RCU_CON_HOSCON) && (!md_rcu_is_active_flag_hosc_ready(rcu)))
        || ((RCU->CON  & RCU_CON_HRCON) && (!md_rcu_is_active_flag_hrc_ready(rcu)))
        || ((RCU->LCON & RCU_LCON_LRCON) && (!md_rcu_is_active_flag_lrc_ready(rcu)))
    );

    md_rcu_set_mco_div(rcu, RCU_InitStruct->Mpre);
    md_rcu_set_mco_source(rcu, RCU_InitStruct->Msw);
    md_rcu_set_pclk_div(rcu, RCU_InitStruct->Ppre);
    md_rcu_set_hclk_div(rcu, RCU_InitStruct->Hpre);
    md_rcu_set_system_clock_source(rcu, RCU_InitStruct->Sw);

    // FPLL Frequency
    fration  = (double)md_rcu_get_pll_fn(RCU) + ((double)md_rcu_get_pll_fk(RCU) / (1 << 19));
    PLL_FRQ = (uint32_t)(4000000 * fration / (VCO_BUF[md_rcu_get_pll_fm(RCU)]));

    /* System Frequency */
    switch (md_rcu_get_current_system_clock(rcu)) /* System clock switch(SYSCLK) */
    {
        case MD_RCU_SWS_SYSCLK_HRC: /*================= HRC selected as system clock*/
            SystemCoreClock = (uint32_t)(__HRC);
            break;

        case MD_RCU_SWS_SYSCLK_HOSC: /*================= HOSC selected as system clock*/
            SystemCoreClock = (uint32_t)(__HOSC);
            break;

        case MD_RCU_SWS_SYSCLK_PLL: /*================= PLL selected as system clock*/
            SystemCoreClock = PLL_FRQ;
            break;

        case MD_RCU_SWS_SYSCLK_LRC: /*================= LRC selected as system clock*/
            SystemCoreClock = (__LRC);
            break;

        default:
            SystemCoreClock = (uint32_t)(__HRC);
            break;
    }

    /* Core Frequency */
    SystemFrequency_SysClk = SystemCoreClock;

    /* PLL Frequency */
    PLLFrequency = PLL_FRQ;

    /* AHB Frequency */
    if ((md_rcu_get_hclk_div(rcu) >= 12))
        SystemFrequency_AHBClk = SystemCoreClock >> ((md_rcu_get_hclk_div(rcu) & 0x07) + 2);
    else if ((md_rcu_get_hclk_div(rcu)) >= 8)
        SystemFrequency_AHBClk = SystemCoreClock >> ((md_rcu_get_hclk_div(rcu) & 0x07) + 1);
    else
        SystemFrequency_AHBClk = SystemCoreClock;

    /* APB Frequency */
    if (md_rcu_get_pclk_div(rcu))
        SystemFrequency_APBClk = SystemFrequency_AHBClk >> ((md_rcu_get_pclk_div(rcu) & 0x03) + 1);
    else
        SystemFrequency_APBClk = SystemFrequency_AHBClk;

    if (RCU_InitStruct->HS_Clock & RCU_CON_CSSON)
        md_rcu_enable_hosc_css(rcu);
    else
        md_rcu_disable_hosc_css(rcu);

    md_rcu_set_current_system_frequency(RCU, (SystemFrequency_SysClk / 1000000));

    if (SystemFrequency_AHBClk / 1000000 > 48)
        md_fc_set_read_latency(FC, MD_FC_WAIT_BETWEEN_48MHz_AND_72Mhz);
    else if (SystemFrequency_AHBClk / 1000000 > 24)
        md_fc_set_read_latency(FC, MD_FC_WAIT_BETWEEN_24MHz_AND_48Mhz);
    else
        md_fc_set_read_latency(FC, MD_FC_WAIT_LESS_THAN_24MHz);
}

void md_rcu_pll_init(RCU_TypeDef *rcu, md_rcu_init_typedef *RCU_InitStruct)
{
    uint32_t    fpllin;
    uint32_t    fvco;

    if ((RCU_InitStruct->Sw == MD_RCU_SW_SYSCLK_PLL) && (RCU_InitStruct->PllSrc == MD_RCU_PLLSRC_HRC))
        md_rcu_enable_hrc(rcu);
    else
        md_rcu_disable_hrc(rcu);

    if ((RCU_InitStruct->Sw == MD_RCU_SW_SYSCLK_PLL) && (RCU_InitStruct->PllSrc == MD_RCU_PLLSRC_HOSC))
        md_rcu_enable_hosc(rcu);
    else
        md_rcu_disable_hosc(rcu);

    md_rcu_disable_pll(rcu);

    while (md_rcu_is_active_flag_pll_ready(rcu) == MD_RCU_PLLRDY_READY);

    md_rcu_set_pll_source(rcu, RCU_InitStruct->PllSrc);

    switch ((RCU_InitStruct->PllSrc))
    {
        case MD_RCU_PLLSRC_HRC :
            fpllin = (uint32_t)(__HRC);
            break;

        case MD_RCU_PLLSRC_HOSC :
            fpllin = (uint32_t)(__HOSC);
            break;

        default :
            fpllin = (uint32_t)(__HRC);
            break;
    }

    md_rcu_set_pll_prediv(rcu, ((fpllin / __PLL_CLKREF) - 1));

    if (RCU_InitStruct->Pllclk <= 72000000 && RCU_InitStruct->Pllclk >= 4000000)
    {
        if ((RCU_InitStruct->Pllclk / 1000000) >= 48)
            md_rcu_set_pll_fm(rcu, MD_RCU_FM_PLL_DIV_4);
        else if ((RCU_InitStruct->Pllclk / 1000000) >= 32)
            md_rcu_set_pll_fm(rcu, MD_RCU_FM_PLL_DIV_6);
        else if ((RCU_InitStruct->Pllclk / 1000000) >= 24)
            md_rcu_set_pll_fm(rcu, MD_RCU_FM_PLL_DIV_8);
        else if ((RCU_InitStruct->Pllclk / 1000000) >= 16)
            md_rcu_set_pll_fm(rcu, MD_RCU_FM_PLL_DIV_12);
        else if ((RCU_InitStruct->Pllclk / 1000000) >= 12)
            md_rcu_set_pll_fm(rcu, MD_RCU_FM_PLL_DIV_16);
        else if ((RCU_InitStruct->Pllclk / 1000000) >= 4)
            md_rcu_set_pll_fm(rcu, MD_RCU_FM_PLL_DIV_24);
        else
            md_rcu_set_pll_fm(rcu, MD_RCU_FM_PLL_DIV_64);

        fvco = RCU_InitStruct->Pllclk * VCO_BUF[md_rcu_get_pll_fm(RCU)];

        md_rcu_set_pll_fn(rcu, fvco / __PLL_CLKREF);

        md_rcu_set_pll_fk(rcu, (((long long)(fvco) << 19) / __PLL_CLKREF) & 0x7FFFF);

        md_rcu_enable_pll(rcu);

        while (md_rcu_is_active_flag_pll_ready(rcu) == 0);
    }
}

/**
  * @brief  Configure system clock using specified parameters
  * @param  clk: The parameter can be one of the following:
  *           @arg @ref MD_RCU_SW_SYSCLK_HRC  16MHz
  *           @arg @ref MD_RCU_SW_SYSCLK_HOSC 4MHz ~ 32MHz
  *           @arg @ref MD_RCU_SW_SYSCLK_PLL  32768Hz
  *           @arg @ref MD_RCU_SW_SYSCLK_LRC  4、8、12、16、24、32、36、40、48、64、72MHz
  * @param  clock: The clock which will be set. the value depends
  *         on the parameter of clk.
  * @retval The status of HAL.
  */
md_status_t md_rcu_clock_config(md_rcu_clock_t clk, uint32_t clock)
{
    uint32_t    PLL_FRQ;
    double      fration;

    md_fc_set_read_latency(FC, MD_FC_WAIT_BETWEEN_48MHz_AND_72Mhz);

    switch (clk)
    {
        case MD_RCU_SW_SYSCLK_HRC:
            md_rcu_enable_hrc(RCU);
            while (!md_rcu_is_active_flag_hrc_ready(RCU));
            md_rcu_set_system_clock_source(RCU, MD_RCU_SW_SYSCLK_HRC);
            break;

        case MD_RCU_SW_SYSCLK_HOSC:
            md_rcu_enable_hosc(RCU);
            md_rcu_check_hosc_ready();
            while (!md_rcu_is_active_flag_hosc_ready(RCU));
            md_rcu_set_system_clock_source(RCU, MD_RCU_SW_SYSCLK_HOSC);
            break;

        case MD_RCU_SW_SYSCLK_PLL:
            md_rcu_enable_pll(RCU);
            while (!md_rcu_is_active_flag_pll_ready(RCU));
            if((clock >= 48000000) && (md_rcu_get_hclk_div(RCU) == MD_RCU_HPRE_SYSCLK_DIV_1))
            {
                md_rcu_set_hclk_div(RCU, MD_RCU_HPRE_SYSCLK_DIV_2);
                md_rcu_set_system_clock_source(RCU, MD_RCU_SW_SYSCLK_PLL);
                SystemFrequency_AHBClk = clock >> 1;
                md_tick_init(MD_SYSTICK_CLKSRC_HCLK);
                md_tick_wait10us(1,1);
                md_rcu_set_hclk_div(RCU, MD_RCU_HPRE_SYSCLK_DIV_1);
            }
            md_rcu_set_system_clock_source(RCU, MD_RCU_SW_SYSCLK_PLL);
            break;

        case MD_RCU_SW_SYSCLK_LRC:
            md_rcu_enable_lrc(RCU);
            while (!md_rcu_is_active_flag_lrc_ready(RCU));
            md_rcu_set_system_clock_source(RCU, MD_RCU_SW_SYSCLK_LRC);
            break;

        default:
            break;
    }

    /* FPLL Frequency */
    fration  = (double)md_rcu_get_pll_fn(RCU) + ((double)md_rcu_get_pll_fk(RCU) / (1 << 19));
    PLL_FRQ = (uint32_t)(4000000 * fration / (VCO_BUF[md_rcu_get_pll_fm(RCU)]));

    /* System Frequency */
    SystemCoreClock = clock;

    /* Core Frequency */
    SystemFrequency_SysClk = SystemCoreClock;

    /* PLL Frequency */
    PLLFrequency = PLL_FRQ;

    /* AHB Frequency */
    if ((md_rcu_get_hclk_div(RCU) >= 12))
        SystemFrequency_AHBClk = SystemCoreClock >> ((md_rcu_get_hclk_div(RCU) & 0x07) + 2);
    else if ((md_rcu_get_hclk_div(RCU)) >= 8)
        SystemFrequency_AHBClk = SystemCoreClock >> ((md_rcu_get_hclk_div(RCU) & 0x07) + 1);
    else
        SystemFrequency_AHBClk = SystemCoreClock;

    /* APB Frequency */
    if (md_rcu_get_pclk_div(RCU))
        SystemFrequency_APBClk = SystemFrequency_AHBClk >> ((md_rcu_get_pclk_div(RCU) & 0x03) + 1);
    else
        SystemFrequency_APBClk = SystemFrequency_AHBClk;

    md_rcu_set_current_system_frequency(RCU, (SystemFrequency_SysClk / 1000000));

    if (SystemFrequency_AHBClk / 1000000 > 48)
        md_fc_set_read_latency(FC, MD_FC_WAIT_BETWEEN_48MHz_AND_72Mhz);
    else if (SystemFrequency_AHBClk / 1000000 > 24)
        md_fc_set_read_latency(FC, MD_FC_WAIT_BETWEEN_24MHz_AND_48Mhz);
    else
        md_fc_set_read_latency(FC, MD_FC_WAIT_LESS_THAN_24MHz);

    return MD_OK;
}

/**
  * @brief  Configure PLL using specified parameters.
  * @param  input: The input clock type.
  * @param  output: The output clock which can be 4、8、12、16、24、32、36、40、48、64、72MHz.
  * @retval None
  */
void md_rcu_pll_config(md_rcu_pll_source_t input, md_rcu_pll_output_t output)
{
    uint32_t    fpllin;
    uint32_t    fvco;

    if (input == MD_RCU_PLLSRC_HRC)
        md_rcu_enable_hrc(RCU);
    else
        md_rcu_enable_hosc(RCU);

    md_rcu_disable_pll(RCU);

    while (md_rcu_is_active_flag_pll_ready(RCU) == MD_RCU_PLLRDY_READY);

    md_rcu_set_pll_source(RCU, input);

    switch (input)
    {
        case MD_RCU_PLLSRC_HRC :
            fpllin = (uint32_t)(__HRC);
            break;

        case MD_RCU_PLLSRC_HOSC :
            fpllin = (uint32_t)(__HOSC);
            break;

        default :
            fpllin = (uint32_t)(__HRC);
            break;
    }

    md_rcu_set_pll_prediv(RCU, ((fpllin / __PLL_CLKREF) - 1));

    if (output <= 72000000 && output >= 4000000)
    {
        if ((output / 1000000) >= 48)
            md_rcu_set_pll_fm(RCU, MD_RCU_FM_PLL_DIV_4);
        else if ((output / 1000000) >= 32)
            md_rcu_set_pll_fm(RCU, MD_RCU_FM_PLL_DIV_6);
        else if ((output / 1000000) >= 24)
            md_rcu_set_pll_fm(RCU, MD_RCU_FM_PLL_DIV_8);
        else if ((output / 1000000) >= 16)
            md_rcu_set_pll_fm(RCU, MD_RCU_FM_PLL_DIV_12);
        else if ((output / 1000000) >= 12)
            md_rcu_set_pll_fm(RCU, MD_RCU_FM_PLL_DIV_16);
        else if ((output / 1000000) >= 4)
            md_rcu_set_pll_fm(RCU, MD_RCU_FM_PLL_DIV_24);
        else
            md_rcu_set_pll_fm(RCU, MD_RCU_FM_PLL_DIV_64);

        fvco = output * VCO_BUF[md_rcu_get_pll_fm(RCU)];

        md_rcu_set_pll_fn(RCU, fvco / __PLL_CLKREF);

        md_rcu_set_pll_fk(RCU, (((long long)(fvco) << 19) / __PLL_CLKREF) & 0x7FFFF);

        md_rcu_enable_pll(RCU);

        while (md_rcu_is_active_flag_pll_ready(RCU) == 0);
    }
}

/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
