/**********************************************************************************
 *
 * @file    md_iwdt.c
 * @brief   md_iwdt C file
 *
 * @date    06 Jun. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          06 Jun. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "md_iwdt.h"

/**@addtogroup MD
 * @{
 */

/**@defgroup IWDT IWDT
 * @brief IWDT module driver
 * @{
 */

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @defgroup IWDT_Public_Functions IWDT Public Functions
 * @{
 */
/**
 * @brief Initializes the IWDT according to the specified parameters.
 * @param prescale
 * @param load: iwdt counter reload value
 * @param win_enable: Window Enable or Disable
 * @param win: Window counter value
 * @retval None
 */
void md_iwdt_init(md_iwdt_prescaler_t prescale, uint32_t load)
{
    WRITE_REG(IWDT->BKKR, MD_IWDT_KEY_ENABLE);

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKKR, MD_IWDT_KEY_WR_ACCESS_ENABLE);

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKPR, prescale);

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKRLR, load);

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKKR, MD_IWDT_KEY_RELOAD);

    return;
}

void md_iwdt_window_set(uint32_t win)
{
    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKKR, MD_IWDT_KEY_WR_ACCESS_ENABLE);

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKWINR, win);
}

/**
 * @brief Get the counter value
 * @retval Value
 */
uint32_t md_iwdt_get_counter_value(void)
{
    uint32_t cnt;

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKSR, 1 << IWDT_BKSR_R_CNT_POS);

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    cnt = READ_REG(IWDT->BKSR);
    return cnt;
}

/**
 * @brief Get the window counter value
 * @retval Value
 */
uint32_t md_iwdt_get_win_counter_value(void)
{
    uint32_t cnt;

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKWINR, 1 << IWDT_BKWINR_R_WIN_POS);

    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    cnt = READ_REG(IWDT->BKWINR);
    return cnt;
}

/**
 * @brief Refreshes the WWDT
 * @retval None
 */
void md_iwdt_feed_dog(void)
{
    while ((uint32_t)(READ_BIT(IWDT->BKFR, IWDT_BKFR_BUSY) == (IWDT_BKFR_BUSY)));

    WRITE_REG(IWDT->BKKR, MD_IWDT_KEY_RELOAD);
}
/**
 * @}
 */

/**
 * @}
 */

/**
 * @}
 */

/******************* (C) COPYRIGHT Eastsoft Microelectronics END OF FILE****/

