/**********************************************************************************
 *
 * @file    md_iir.c
 * @brief   md_iir C file
 *
 * @date    11 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          11 Aug 2022     kevin zeng      the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include <math.h>
#include <string.h>
#include "md_rcu.h"
#include "md_iir.h"

/* Private Macros ------------------------------------------------------------ */

#define _PI acos(-1)

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @addtogroup MD_IIR
  * @{
  */

/** @addtogroup MD_IIR_Public_Functions
  * @{
  */
/** @addtogroup IIR_Function
  * @{
  */
void md_iir_init(IIR_TypeDef *iir, md_iir_init_typedef *iir_init)
{
    md_iir_set_mode(iir, iir_init->mode);
    md_iir_set_first_input_factor(iir, iir_init->scale_factor_0);
    md_iir_set_first_coefficient_b1(iir, iir_init->coeb_b1_0);
    md_iir_set_first_coefficient_b2(iir, iir_init->coeb_b2_0);
    md_iir_set_first_coefficient_b3(iir, iir_init->coeb_b3_0);
    md_iir_set_first_coefficient_a2(iir, iir_init->coeb_a2_0);
    md_iir_set_first_coefficient_a3(iir, iir_init->coeb_a3_0);
    md_iir_set_second_input_factor(iir, iir_init->scale_factor_1);
    md_iir_set_second_coefficient_b1(iir, iir_init->coeb_b1_1);
    md_iir_set_second_coefficient_b2(iir, iir_init->coeb_b2_1);
    md_iir_set_second_coefficient_b3(iir, iir_init->coeb_b3_1);
    md_iir_set_second_coefficient_a2(iir, iir_init->coeb_a2_1);
    md_iir_set_second_coefficient_a3(iir, iir_init->coeb_a3_1);
    md_iir_set_reset(iir);
}

void md_iir_coefficient_calculate(uint32_t *coef, double num)
{
    /* limit input protect */
    if (num >= 64)
        num = 63.9999999f;
    else if (num < -64)
        num = -64.0f;


    num = num * (1 << 19); /* num*2^19 */

    *coef = (int32_t)(num + 0.5); /* num rounding than write in coef */

}

void md_iir_data_calculate(uint32_t *coef, double num)
{
    /* limit input protect */
    if (num >= 2)
        num = 1.9999999f;
    else if (num < -2)
        num = -2.0f;


    num = num * (1 << 14); /* num*2^14 */

    *coef = (int32_t)(num + 0.5); /* num rounding than write in coef */
}

void md_iir_feedback_temp_calculate(uint32_t *coef, double num)
{
    /* limit input protect */
    if (num >= 256)
        num = 255.9999999f;
    else if (num < -256)
        num = -256.0f;

    num = num * (1 << 15); /* num*2^23 */

    *coef = (int32_t)(num + 0.5); /* num rounding than write in coef */
}

double md_iir_coefficient_reverse_calculate(uint32_t coef)
{
    return ((double)((int)coef) / (double)(1 << 19));
}

double md_iir_data_reverse_calculate(uint32_t coef)
{
    return ((double)((int)coef) / (double)(1 << 14));
}

double md_iir_feedback_temp_reverse_calculate(uint32_t coef)
{
    return ((double)((int)coef) / (double)(1 << 15));
}

void md_iir_coefficient_determines(IIR_TypeDef *iir, uint32_t sampling_frequency, uint32_t cut_off_frequency, md_iir_order_pass_filter_t order)
{
    md_iir_init_typedef iir_init;

    double c = (double)tan((double)_PI * ((double)cut_off_frequency / (double)sampling_frequency));

    memset(&iir_init, 0, sizeof(md_iir_init_typedef));

    iir_init.mode = MD_IIR_MODE_ONE_STAGE_BIQUAD;

    if (order == MD_IIR_1ST_ORDER_LOW_PASS_FILTER)
    {
        md_iir_coefficient_calculate(&iir_init.coeb_b1_0, 1);
        md_iir_coefficient_calculate(&iir_init.coeb_b2_0, 1);
        md_iir_coefficient_calculate(&iir_init.coeb_b3_0, 0);
        md_iir_coefficient_calculate(&iir_init.coeb_a2_0, ((c - 1) / (c + 1)));
        md_iir_coefficient_calculate(&iir_init.coeb_a3_0, 0);
        md_iir_coefficient_calculate(&iir_init.scale_factor_0, c / (c + 1));
    }
    else if (order == MD_IIR_2ST_ORDER_LOW_PASS_FILTER)
    {
        md_iir_coefficient_calculate(&iir_init.coeb_b1_0, 1);
        md_iir_coefficient_calculate(&iir_init.coeb_b2_0, 2);
        md_iir_coefficient_calculate(&iir_init.coeb_b3_0, 1);
        md_iir_coefficient_calculate(&iir_init.coeb_a2_0, ((2 * (pow(c, 2) - 1)) / (pow(c, 2) + sqrt(2)*c + 1)));
        md_iir_coefficient_calculate(&iir_init.coeb_a3_0, ((pow(c, 2) - sqrt(2)*c + 1) / (pow(c, 2) + sqrt(2)*c + 1)));
        md_iir_coefficient_calculate(&iir_init.scale_factor_0, (pow(c, 2) / (pow(c, 2) + sqrt(2)*c + 1)));
    }
    else if (order == MD_IIR_1ST_ORDER_HIGH_PASS_FILTER)
    {
        md_iir_coefficient_calculate(&iir_init.coeb_b1_0, 1);
        md_iir_coefficient_calculate(&iir_init.coeb_b2_0, -1);
        md_iir_coefficient_calculate(&iir_init.coeb_b3_0, 0);
        md_iir_coefficient_calculate(&iir_init.coeb_a2_0, ((c - 1) / (c + 1)));
        md_iir_coefficient_calculate(&iir_init.coeb_a3_0, 0);
        md_iir_coefficient_calculate(&iir_init.scale_factor_0, 1 / (c + 1));
    }
    else if (order == MD_IIR_2ST_ORDER_HIGH_PASS_FILTER)
    {
        md_iir_coefficient_calculate(&iir_init.coeb_b1_0, 1);
        md_iir_coefficient_calculate(&iir_init.coeb_b2_0, -2);
        md_iir_coefficient_calculate(&iir_init.coeb_b3_0, 1);
        md_iir_coefficient_calculate(&iir_init.coeb_a2_0, ((2 * (pow(c, 2) - 1)) / (pow(c, 2) + sqrt(2)*c + 1)));
        md_iir_coefficient_calculate(&iir_init.coeb_a3_0, ((pow(c, 2) - sqrt(2)*c + 1) / (pow(c, 2) + sqrt(2)*c + 1)));
        md_iir_coefficient_calculate(&iir_init.scale_factor_0, (1 / (pow(c, 2) + sqrt(2)*c + 1)));
    }

    md_iir_init(iir, &iir_init);
}

void md_iir_get_delay_register(IIR_TypeDef *iir, md_iir_delay_register_typedef *iir_intd)
{
    iir_intd->intd1_0 = md_iir_get_first_delay_register1(iir);
    iir_intd->intd1_1 = md_iir_get_second_delay_register1(iir);
    iir_intd->intd2_0 = md_iir_get_first_delay_register2(iir);
    iir_intd->intd2_1 = md_iir_get_second_delay_register2(iir);
}

void md_iir_set_delay_register(IIR_TypeDef *iir, md_iir_delay_register_typedef *iir_intd)
{
    md_iir_set_first_delay_register1(iir, iir_intd->intd1_0);
    md_iir_set_second_delay_register1(iir, iir_intd->intd1_1);
    md_iir_set_first_delay_register2(iir, iir_intd->intd2_0);
    md_iir_set_second_delay_register2(iir, iir_intd->intd2_1);
}
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/

