/**********************************************************************************
 *
 * @file    md_rcu.c
 * @brief   md_rcu C file
 *
 * @date    11 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          11 Aug 2022     Ginger         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/
#include "md_rcu.h"
#include "md_gpio.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @addtogroup MD_GPIO
  * @{
  */

/** @addtogroup MD_GPIO_Public_Functions
  * @{
  */
/** @addtogroup GPIO_Function
  * @{
  */

/**
  * @brief  Initialize the GPIOx peripheral according to the specified
  *         parameters in the gpio_init_t.
  * @param  GPIOx: Where x can be (A--C) to select the GPIO peripheral.
  * @param  pin: The pin which need to initialize.
  * @param  init: Pointer to a md_gpio_init_t structure that can contains
  *         the configuration information for the specified parameters.
  * @retval None
  */
void md_gpio_init(GPIO_TypeDef *GPIOx, uint16_t pin, md_gpio_init_t *init)
{
    if (POSITION_VAL(pin) <= 7)
    {
        md_gpio_set_pin_function0_7(GPIOx, pin, init->func);
    }
    else
    {
        md_gpio_set_pin_function8_15(GPIOx, pin, init->func);
    }

    md_gpio_set_pin_output_type(GPIOx, pin, init->odos);
    md_gpio_set_pin_pull(GPIOx, pin, init->pupd);
    md_gpio_set_pin_outputdriving(GPIOx, pin, init->odrv);
    md_gpio_set_pin_mode(GPIOx, pin, init->mode);
    md_gpio_set_pin_type(GPIOx, pin, init->type);
    MODIFY_REG(GPIOx->FIR, (GPIO_FIR_FIR0 << (POSITION_VAL(pin))), (init->flt << (POSITION_VAL(pin))));

    return;
}

/**
  * @brief  Initialize the GPIOx peripheral using the default parameters.
  * @param  init: Pointer to a md_gpio_init_t structure
  * @retval None
  */
void md_gpio_init_struct(md_gpio_init_t *init)
{
    /* Fill md_gpio_init_t structure with default parameter */
    init->mode  = MD_GPIO_MODE_OUTPUT;
    init->odos  = MD_GPIO_PUSH_PULL;
    init->pupd  = MD_GPIO_PUSH_UP;
    init->odrv  = MD_GPIO_DRIVING_8MA;
    init->flt   = MD_GPIO_FILTER_DISABLE;
    init->type  = MD_GPIO_TYPE_CMOS;
    init->func  = MD_GPIO_FUNC_0;

    return;
}

/**
  * @brief  Set or clear the select Pin data.
  * @param  GPIOx: Where x can be (A--B) to select the GPIO peripheral.
  * @param  pin: The specified pin to be written.
  * @param  val: The specifies value to be written.
  * @retval None
  */
void md_gpio_write_pin(GPIO_TypeDef *GPIOx, uint16_t pin, uint8_t val)
{
    if ((val & (0x01)) == 0x00)
        GPIOx->BSBR = pin << 16U;
    else
        GPIOx->BSBR = pin;

    return;
}

/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
