/**********************************************************************************
 *
 * @file    md_dma.c
 * @brief   DMA module driver.
 *
 * @date    24 Apr. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author       Notes
 *          24 Apr. 2024    Lisq         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */

#include "md_dma.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @addtogroup MD_DMA
  * @{
  */

/** @addtogroup MD_DMA_Public_Functions
  * @{
  */
/** @addtogroup MD_DMA_Public_Functions_Group1
  * @{
  */
/**
  * @brief  Reset the DMA register
  * @param  DMAx: Pointer to DMA peripheral
  * @retval None
  */
void md_dma_reset(DMA_TypeDef *DMAx)
{
    uint32_t i;

    WRITE_REG(DMAx->IDR, 0x3FFF);
    WRITE_REG(DMAx->ICR, 0x3FFF);

    for (i = 0; i < DMA_CHANNELS; ++i)
    {
        md_dma_disable_channel(DMAx, i);
        WRITE_REG(DMAx->CH[i].CON, 0x0);
        WRITE_REG(DMAx->CH[i].SAR, 0x0);
        WRITE_REG(DMAx->CH[i].DAR, 0x0);
        WRITE_REG(DMAx->CH[i].NDT, 0x0);
        WRITE_REG(DMA_MUX->CH_SELCON[i], 0x0);
    }

    return;
}

/**
  * @brief  DMA init
  * @param  ch: DMA channel
  * @param  config: DMA config
  * @retval None
  */
void md_dma_init(DMA_TypeDef *DMAx, uint32_t ch, md_dma_config_t *config)
{
    md_dma_set_source_addr(DMAx, ch, config->src);
    md_dma_set_source_width(DMAx, ch, config->src_data_width);
    md_dma_set_source_inc(DMAx, ch, config->src_inc);
    md_dma_set_dest_addr(DMAx, ch, config->dst);
    md_dma_set_dest_width(DMAx, ch, config->dst_data_width);
    md_dma_set_dest_inc(DMAx, ch, config->dst_inc);
    md_dma_set_circle_mode(DMAx, ch, config->circle_mode);
    md_dma_set_mem_to_mem(DMAx, ch, config->mem_to_mem);
    md_dma_set_priority(DMAx, ch, config->priority);
    md_dma_set_r_power(DMAx, ch, config->R_power);
    md_dma_set_transfer_size(DMAx, ch, config->size);
    md_dma_set_channel_msigsel(DMA, ch, config->msigsel);
    md_dma_set_data_dir(DMA, ch, config->dir);
    md_dma_enable_channel(DMAx, ch);
}

/**
  * @brief  Configure dma_config_t structure using default parameter.
  *         User can invoked this function, before configure dma_config_t
  * @param  hperh: Pointer to DMA_handle_t structure that contains
  *                configuration information for specified DMA channel.
  * @retval None
  */
void md_dma_config_struct(md_dma_config_t *config)
{
    config->src_data_width    = MD_DMA_DATA_SIZE_BYTE;
    config->src_inc           = MD_DMA_DATA_INC_ENABLE;
    config->dst_data_width    = MD_DMA_DATA_SIZE_BYTE;
    config->dst_inc           = MD_DMA_DATA_INC_ENABLE;
    config->R_power           = MD_DMA_R_POWER_1;
    config->priority          = MD_DMA_LOW_PRIORITY;
    config->dir               = MD_DMA_DIR_TO_SRAM;
    config->mem_to_mem        = ENABLE;
    config->circle_mode       = DISABLE;
    config->irq_tc            = DISABLE;
    config->irq_ht            = DISABLE;
    config->channel           = 0x0U;

    return;
}
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
