/**********************************************************************************
 *
 * @file    md_can.c
 * @brief   md_can C file
 *
 * @date    11 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          11 Aug 2022     kevin zeng      the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes -------------------------------------------------------------------*/

#include <math.h>
#include <string.h>
#include "md_rcu.h"
#include "md_can.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @addtogroup MD_CAN
  * @{
  */

/** @addtogroup MD_CAN_Public_Functions
  * @{
  */
/** @addtogroup CAN_Function 
  * @{
  */
void md_can_mode_config(CAN_TypeDef *CANx, md_can_mode_t mode)
{
    switch (mode)
    {
        case INITIAL_MODE:
            md_can_set_fltini(CANx, MD_CAN_FILTER_INITIAL_MODE);
            md_can_set_init_request(CANx, MD_CAN_INITIAL_REQUEST_ENTER);
            md_can_set_sleep_request(CANx, MD_CAN_SLEEP_REQUEST_LEAVE);

            while (md_can_is_active_flag_inistat(CANx) != MD_CAN_INITIAL_REQUEST_ENTER);

            while (md_can_is_active_flag_slpstat(CANx) != MD_CAN_SLEEP_REQUEST_LEAVE);

            break;

        case NORMAL_MODE:
            md_can_set_init_request(CANx, MD_CAN_INITIAL_REQUEST_LEAVE);
            md_can_set_sleep_request(CANx, MD_CAN_SLEEP_REQUEST_LEAVE);

            while (md_can_is_active_flag_inistat(CANx) != MD_CAN_INITIAL_REQUEST_LEAVE);

            while (md_can_is_active_flag_slpstat(CANx) != MD_CAN_SLEEP_REQUEST_LEAVE);

            md_can_set_fltini(CANx, MD_CAN_FILTER_WORK_MODE);
            break;

        case SLEEP_MODE:
            md_can_set_init_request(CANx, MD_CAN_INITIAL_REQUEST_LEAVE);
            md_can_set_sleep_request(CANx, MD_CAN_SLEEP_REQUEST_ENTER);

            while (md_can_is_active_flag_inistat(CANx) != MD_CAN_INITIAL_REQUEST_LEAVE);

            while (md_can_is_active_flag_slpstat(CANx) != MD_CAN_SLEEP_REQUEST_ENTER);

            break;
    }
}

void md_can_debugmode_config(CAN_TypeDef *CANx, md_can_debugmode_t mode)
{
    md_can_mode_config(CANx, INITIAL_MODE);

    switch (mode)
    {
        case NORMAL_DEBUGMODE:
            md_can_set_silent(CANx, MD_CAN_SILENT_NORMAL_MODE);
            md_can_disable_loopmode(CANx);
            break;

        case LOOPBACK_DEBUGMODE:
            md_can_set_silent(CANx, MD_CAN_SILENT_NORMAL_MODE);
            md_can_enable_loopmode(CANx);
            break;

        case SILENT_DEBUGMODE:
            md_can_set_silent(CANx, MD_CAN_SILENT_SILENCE_MODE);
            md_can_disable_loopmode(CANx);
            break;

        case SILENT_LOOPBACK_DEBUGMODE:
            md_can_set_silent(CANx, MD_CAN_SILENT_SILENCE_MODE);
            md_can_enable_loopmode(CANx);
            break;
    }

    md_can_mode_config(CANx, NORMAL_MODE);
}

void md_can_bittime_config(CAN_TypeDef *CANx, md_can_bittime_config_typedef *config)
{
    md_can_mode_config(CANx, INITIAL_MODE);
    md_can_set_bpsc(CANx, config->bpsc);
    md_can_set_seg1(CANx, config->seg1);
    md_can_set_seg2(CANx, config->seg2);
    md_can_set_resjw(CANx, config->resjw);
    md_can_mode_config(CANx, NORMAL_MODE);
}

static uint32_t filter32bit_id_calculate(uint32_t id, md_can_id_type_t id_type, md_frame_type_t frame_type)
{
    if (id_type == STANDARD_IDENTIFIER)
    {
        id &= 0x7FFU;
        return (uint32_t)((id << 21) | (id_type << 2) | (frame_type << 1));
    }
    else if (id_type == EXTENDED_IDENTIFIER)
    {
        id &= 0x1FFFFFFFU;
        return (uint32_t)((id << 3) | (id_type << 2) | (frame_type << 1));
    }

    return 0;
}

static uint16_t filter16bit_id_calculate(uint32_t id, md_can_id_type_t id_type, md_frame_type_t frame_type)
{
    if (id_type == STANDARD_IDENTIFIER)
    {
        id &= 0x7FFU;
        return (uint16_t)((id << 5) | (frame_type << 4) | (id_type << 3));
    }
    else if (id_type == EXTENDED_IDENTIFIER)
    {
        id &= 0x1FFFFFFFU;
        return (uint16_t)((((id & 0x1FFC0000U) >> 18) << 5) | (frame_type << 4) | (id_type << 3) | ((id & 0x38000U) >> 15));
    }

    return 0;
}
/**
  * @param  a 32-bit filter_number
  *         @arg @ref MD_CAN_FILTER_0
  *         @arg @ref MD_CAN_FILTER_1
  *         @arg @ref MD_CAN_FILTER_2
  *         @arg @ref MD_CAN_FILTER_3
  *         @arg @ref MD_CAN_FILTER_4
  *         @arg @ref MD_CAN_FILTER_5
  *         @arg @ref MD_CAN_FILTER_6
  *         @arg @ref MD_CAN_FILTER_7
  *         @arg @ref MD_CAN_FILTER_8
  *         @arg @ref MD_CAN_FILTER_9
  *         @arg @ref MD_CAN_FILTER_10
  *         @arg @ref MD_CAN_FILTER_11
  *         @arg @ref MD_CAN_FILTER_12
  *         @arg @ref MD_CAN_FILTER_13
*/
void md_can_filter_config(CAN_TypeDef *CANx, md_can_filter_list_t filter_number, md_can_filter_config_typedef *filter_config)
{
    void (*md_can_set_filter[][2])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_filter0_register1,
        md_can_set_filter0_register2,

        md_can_set_filter1_register1,
        md_can_set_filter1_register2,

        md_can_set_filter2_register1,
        md_can_set_filter2_register2,

        md_can_set_filter3_register1,
        md_can_set_filter3_register2,

        md_can_set_filter4_register1,
        md_can_set_filter4_register2,

        md_can_set_filter5_register1,
        md_can_set_filter5_register2,

        md_can_set_filter6_register1,
        md_can_set_filter6_register2,

        md_can_set_filter7_register1,
        md_can_set_filter7_register2,

        md_can_set_filter8_register1,
        md_can_set_filter8_register2,

        md_can_set_filter9_register1,
        md_can_set_filter9_register2,

        md_can_set_filter10_register1,
        md_can_set_filter10_register2,

        md_can_set_filter11_register1,
        md_can_set_filter11_register2,

        md_can_set_filter12_register1,
        md_can_set_filter12_register2,

        md_can_set_filter13_register1,
        md_can_set_filter13_register2
    };

    md_can_set_fltini(CANx, MD_CAN_FILTER_INITIAL_MODE);
    md_can_set_mod(CANx, filter_number, filter_config->filter_mode);
    md_can_set_sel(CANx, filter_number, filter_config->filter_width);
    md_can_set_assign(CANx, filter_number, filter_config->filter_fifo_assign);

    if (filter_config->filter_width == MD_CAN_FILTER_SEL_32BIT)
    {
        if (filter_config->filter_mode == MD_CAN_FILTER_MOD_MASK)
        {
            md_can_set_filter[filter_number][0](CANx, filter32bit_id_calculate(filter_config->filter_mode_select.identifier_32bit_mask.id, filter_config->filter_mode_select.identifier_32bit_mask.id_type, filter_config->filter_mode_select.identifier_32bit_mask.frame_type));
            md_can_set_filter[filter_number][1](CANx, filter_config->filter_mode_select.identifier_32bit_mask.mask);
        }
        else if (filter_config->filter_mode == MD_CAN_FILTER_MOD_LIST_OF_IDENTIFIERS)
        {
            md_can_set_filter[filter_number][0](CANx, filter32bit_id_calculate(filter_config->filter_mode_select.identifier_32bit_list.id1, filter_config->filter_mode_select.identifier_32bit_list.id1_type, filter_config->filter_mode_select.identifier_32bit_list.frame1_type));
            md_can_set_filter[filter_number][1](CANx, filter32bit_id_calculate(filter_config->filter_mode_select.identifier_32bit_list.id2, filter_config->filter_mode_select.identifier_32bit_list.id2_type, filter_config->filter_mode_select.identifier_32bit_list.frame2_type));
        }
    }
    else if (filter_config->filter_width == MD_CAN_FILTER_SEL_16BIT)
    {
        if (filter_config->filter_mode == MD_CAN_FILTER_MOD_MASK)
        {
            md_can_set_filter[filter_number][0](CANx, (uint32_t)(filter_config->filter_mode_select.identifier_16bit_mask.mask1 << 16)
                                                | (uint32_t)(filter16bit_id_calculate(filter_config->filter_mode_select.identifier_16bit_mask.id1, filter_config->filter_mode_select.identifier_16bit_mask.id1_type, filter_config->filter_mode_select.identifier_16bit_mask.frame1_type)));
            md_can_set_filter[filter_number][1](CANx, (uint32_t)(filter_config->filter_mode_select.identifier_16bit_mask.mask2 << 16)
                                                | (uint32_t)(filter16bit_id_calculate(filter_config->filter_mode_select.identifier_16bit_mask.id2, filter_config->filter_mode_select.identifier_16bit_mask.id2_type, filter_config->filter_mode_select.identifier_16bit_mask.frame2_type)));
        }
        else if (filter_config->filter_mode == MD_CAN_FILTER_MOD_LIST_OF_IDENTIFIERS)
        {
            md_can_set_filter[filter_number][0](CANx, (uint32_t)(filter16bit_id_calculate(filter_config->filter_mode_select.identifier_16bit_list.id1, filter_config->filter_mode_select.identifier_16bit_list.id1_type, filter_config->filter_mode_select.identifier_16bit_list.frame1_type) << 16)
                                                | (uint32_t)(filter16bit_id_calculate(filter_config->filter_mode_select.identifier_16bit_list.id2, filter_config->filter_mode_select.identifier_16bit_list.id2_type, filter_config->filter_mode_select.identifier_16bit_list.frame2_type)));
            md_can_set_filter[filter_number][1](CANx, (uint32_t)(filter16bit_id_calculate(filter_config->filter_mode_select.identifier_16bit_list.id3, filter_config->filter_mode_select.identifier_16bit_list.id3_type, filter_config->filter_mode_select.identifier_16bit_list.frame3_type) << 16)
                                                | (uint32_t)(filter16bit_id_calculate(filter_config->filter_mode_select.identifier_16bit_list.id4, filter_config->filter_mode_select.identifier_16bit_list.id4_type, filter_config->filter_mode_select.identifier_16bit_list.frame4_type)));
        }
    }

    md_can_set_fltini(CANx, MD_CAN_FILTER_WORK_MODE);
}

void md_can_init(CAN_TypeDef *CANx, md_can_init_typedef *init)
{
    md_can_mode_config(CANx, INITIAL_MODE);

    md_can_set_TXMP(CANx, init->transfer_mailbox_priority);
    md_can_set_RXFOPM(CANx, init->receiver_fifo_overflow_mode);

    if (init->auto_resend_disable)
        md_can_enable_artxdis(CANx);
    else
        md_can_disable_artxdis(CANx);

    if (init->auto_wakeup_enable)
        md_can_enable_auto_wake_up(CANx);
    else
        md_can_disable_auto_wake_up(CANx);

    if (init->auto_quit_busSwitch_enable)
        md_can_enable_aboffen(CANx);
    else
        md_can_disable_aboffen(CANx);

    if (init->timer_trigger_communication_enable)
        md_can_enable_timer_trigger_connect(CANx);
    else
        md_can_disable_timer_trigger_connect(CANx);

    md_can_bittime_config(CANx, &(init->bit_time));

    md_can_mode_config(CANx, NORMAL_MODE);
}

void md_can_send(CAN_TypeDef *CANx, md_can_txmailbox_typedef *message)
{
    uint32_t mailbox_num = 0;
    int i = 0;
    void (*md_can_set_txmailbox_standardid[])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_txmailbox0_standardid,
        md_can_set_txmailbox1_standardid,
        md_can_set_txmailbox2_standardid
    };

    void (*md_can_set_txmailbox_expandid[])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_txmailbox0_expandid,
        md_can_set_txmailbox1_expandid,
        md_can_set_txmailbox2_expandid
    };

    void (*md_can_set_txmailbox_ide[])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_txmailbox0_ide,
        md_can_set_txmailbox1_ide,
        md_can_set_txmailbox2_ide
    };

    void (*md_can_set_txmailbox_rtr[])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_txmailbox0_rtr,
        md_can_set_txmailbox1_rtr,
        md_can_set_txmailbox2_rtr
    };

    void (*md_can_enable_txmailbox_request[])(CAN_TypeDef * CANx) =
    {
        md_can_enable_txmailbox0_request,
        md_can_enable_txmailbox1_request,
        md_can_enable_txmailbox2_request
    };

    void (*md_can_set_txmailbox_stamp[])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_txmailbox0_stamp,
        md_can_set_txmailbox1_stamp,
        md_can_set_txmailbox2_stamp,
    };

    void (*md_can_set_txmailbox_txgt[])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_txmailbox0_txgt,
        md_can_set_txmailbox1_txgt,
        md_can_set_txmailbox2_txgt
    };

    void (*md_can_set_txmailbox_datalength[])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_txmailbox0_datalength,
        md_can_set_txmailbox1_datalength,
        md_can_set_txmailbox2_datalength
    };

    void (*md_can_set_txmailbox_byte[][8])(CAN_TypeDef * CANx, uint32_t value) =
    {
        md_can_set_txmailbox0_byte0,
        md_can_set_txmailbox0_byte1,
        md_can_set_txmailbox0_byte2,
        md_can_set_txmailbox0_byte3,
        md_can_set_txmailbox0_byte4,
        md_can_set_txmailbox0_byte5,
        md_can_set_txmailbox0_byte6,
        md_can_set_txmailbox0_byte7,
        md_can_set_txmailbox1_byte0,
        md_can_set_txmailbox1_byte1,
        md_can_set_txmailbox1_byte2,
        md_can_set_txmailbox1_byte3,
        md_can_set_txmailbox1_byte4,
        md_can_set_txmailbox1_byte5,
        md_can_set_txmailbox1_byte6,
        md_can_set_txmailbox1_byte7,
        md_can_set_txmailbox2_byte0,
        md_can_set_txmailbox2_byte1,
        md_can_set_txmailbox2_byte2,
        md_can_set_txmailbox2_byte3,
        md_can_set_txmailbox2_byte4,
        md_can_set_txmailbox2_byte5,
        md_can_set_txmailbox2_byte6,
        md_can_set_txmailbox2_byte7
    };

    uint32_t (*md_can_is_active_flag_txmef[])(CAN_TypeDef * CANx) =
    {
        md_can_is_active_flag_txm0ef,
        md_can_is_active_flag_txm1ef,
        md_can_is_active_flag_txm2ef
    };

    mailbox_num = md_can_get_next_free_mailbox(CANx);

    while (md_can_is_active_flag_txmef[mailbox_num](CANx) == 0);

    if (message->id_type == STANDARD_IDENTIFIER)
        md_can_set_txmailbox_standardid[mailbox_num](CANx, (message->id & 0x7FF));
    else if (message->id_type == EXTENDED_IDENTIFIER)
        md_can_set_txmailbox_expandid[mailbox_num](CANx, (message->id & 0x1FFFFFFF));

    md_can_set_txmailbox_ide[mailbox_num](CANx, message->id_type);
    md_can_set_txmailbox_rtr[mailbox_num](CANx, message->frame_type);
    md_can_set_txmailbox_stamp[mailbox_num](CANx, message->message_timestamp);
    md_can_set_txmailbox_txgt[mailbox_num](CANx, message->send_global_time_enable);
    md_can_set_txmailbox_datalength[mailbox_num](CANx, message->length);

    for (i = 0; i < 8; i++)
    {
        md_can_set_txmailbox_byte[mailbox_num][i](CANx, message->data[i]);
    }

    md_can_enable_txmailbox_request[mailbox_num](CANx);
}



void md_can_receive(CAN_TypeDef *CANx, md_can_rx_fifo_t fifo_num, md_can_rxmailbox_typedef *message)
{
    int i = 0;

    uint32_t (*md_can_get_rxfifo_standardid[])(CAN_TypeDef * CANx) =
    {
        md_can_get_rxfifo0_standardid,
        md_can_get_rxfifo1_standardid
    };

    uint32_t (*md_can_get_rxfifo_expandid[])(CAN_TypeDef * CANx) =
    {
        md_can_get_rxfifo0_expandid,
        md_can_get_rxfifo1_expandid
    };

    uint32_t (*md_can_get_rxfifo_ide[])(CAN_TypeDef * CANx) =
    {
        md_can_get_rxfifo0_ide,
        md_can_get_rxfifo1_ide
    };

    uint32_t (*md_can_get_rxfifo_rtr[])(CAN_TypeDef * CANx) =
    {
        md_can_get_rxfifo0_rtr,
        md_can_get_rxfifo1_rtr
    };

    uint32_t (*md_can_get_rxfifo_stamp[])(CAN_TypeDef * CANx) =
    {
        md_can_get_rxfifo0_stamp,
        md_can_get_rxfifo1_stamp
    };

    uint32_t (*md_can_get_rxfifo_fltidx[])(CAN_TypeDef * CANx) =
    {
        md_can_get_rxfifo0_fltidx,
        md_can_get_rxfifo1_fltidx
    };

    uint32_t (*md_can_get_rxfifo_datalength[])(CAN_TypeDef * CANx) =
    {
        md_can_get_rxfifo0_datalength,
        md_can_get_rxfifo1_datalength
    };

    uint32_t (*md_can_get_rxfifo_byte[][8])(CAN_TypeDef * CANx) =
    {
        md_can_get_rxfifo0_byte0,
        md_can_get_rxfifo0_byte1,
        md_can_get_rxfifo0_byte2,
        md_can_get_rxfifo0_byte3,
        md_can_get_rxfifo0_byte4,
        md_can_get_rxfifo0_byte5,
        md_can_get_rxfifo0_byte6,
        md_can_get_rxfifo0_byte7,
        md_can_get_rxfifo1_byte0,
        md_can_get_rxfifo1_byte1,
        md_can_get_rxfifo1_byte2,
        md_can_get_rxfifo1_byte3,
        md_can_get_rxfifo1_byte4,
        md_can_get_rxfifo1_byte5,
        md_can_get_rxfifo1_byte6,
        md_can_get_rxfifo1_byte7
    };

    message->id_type = (md_can_id_type_t)md_can_get_rxfifo_ide[fifo_num](CANx);

    if (message->id_type == STANDARD_IDENTIFIER)
    {
        message->id = md_can_get_rxfifo_standardid[fifo_num](CANx);
    }
    else if (message->id_type == EXTENDED_IDENTIFIER)
    {
        message->id = md_can_get_rxfifo_expandid[fifo_num](CANx);
    }

    message->frame_type = (md_frame_type_t)md_can_get_rxfifo_rtr[fifo_num](CANx);
    message->length = md_can_get_rxfifo_datalength[fifo_num](CANx);
    message->filter_match_index = md_can_get_rxfifo_fltidx[fifo_num](CANx);
    message->message_timestamp = md_can_get_rxfifo_stamp[fifo_num](CANx);

    for (i = 0; i < message->length; i++)
    {
        message->data[i] = md_can_get_rxfifo_byte[fifo_num][i](CANx);
    }

    if (fifo_num == FIFO0)
        md_can_set_rxfifo0_free(CANx, MD_CAN_RXFIFO_FREE_MESSAGE);
    else if (fifo_num == FIFO1)
        md_can_set_rxfifo1_free(CANx, MD_CAN_RXFIFO_FREE_MESSAGE);
}

/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
