/**********************************************************************************
 *
 * @file    md_adc.c
 * @brief   md_adc C file
 *
 * @date    11 Aug 2022
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          11 Aug 2022     Ginger         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Includes ------------------------------------------------------------------ */
#include "md_adc.h"
#include "md_tick.h"
#include "md_syscfg.h"
#include "md_rcu.h"

/* Private Macros ------------------------------------------------------------ */

/* Private Variables --------------------------------------------------------- */

/* Public Variables ---------------------------------------------------------- */

unsigned int ADC_KernelClock_Frequency;

/* Private Constants --------------------------------------------------------- */

/* Private function prototypes ----------------------------------------------- */

/* Private Function ---------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @addtogroup MD_ADC
  * @{
  */

/** @addtogroup MD_ADC_Public_Functions
  * @{
  */

/** @addtogroup ADC_Function
  * @{
  */
static void md_adc_close(ADC_TypeDef *adc)
{
    if (md_adc_get_start_regular(adc) == 1)
        md_adc_set_stop_regular(adc);

    while (md_adc_get_start_regular(adc) == 1);

    if (md_adc_get_start_injected(adc) == 1)
        md_adc_set_stop_injected(adc);

    while (md_adc_get_start_injected(adc) == 1);

    md_adc_disable_adcpower(adc);

    while (md_adc_is_enabled_adcpower(adc) == 1);
}

static void md_adc_open(ADC_TypeDef *adc)
{
    md_tick_init(MD_SYSTICK_CLKSRC_HCLK_DIV8);
    md_syscfg_enable_vdda(SYSCFG);
    md_tick_wait100us(1, 1);
    md_adc_enable_adcpower(adc);
}

/*
sampletime(ns)
*/
md_adc_sample_time_typedef md_adc_sampletime_calculate(ADC_TypeDef *adc, ADC_COMM_TypeDef *adc12_common, double sampletime)
{
    md_adc_sample_time_typedef feedback = {0};
    double temp = 0;
    uint32_t Tsmp = 0;
    double ADC_Frequency = 0;
    double resolution = 0;

    switch (md_adc_get_clock_divider(adc12_common))
    {
        case MD_ADC_CCR_PRESCALE_1:
            ADC_Frequency = ADC_KernelClock_Frequency;
            break;

        case MD_ADC_CCR_PRESCALE_2:
            ADC_Frequency = ADC_KernelClock_Frequency / 2;
            break;

        case MD_ADC_CCR_PRESCALE_4:
            ADC_Frequency = ADC_KernelClock_Frequency / 4;
            break;

        case MD_ADC_CCR_PRESCALE_6:
            ADC_Frequency = ADC_KernelClock_Frequency / 6;
            break;

        case MD_ADC_CCR_PRESCALE_8:
            ADC_Frequency = ADC_KernelClock_Frequency / 8;
            break;

        case MD_ADC_CCR_PRESCALE_10:
            ADC_Frequency = ADC_KernelClock_Frequency / 10;
            break;

        case MD_ADC_CCR_PRESCALE_12:
            ADC_Frequency = ADC_KernelClock_Frequency / 12;
            break;

        case MD_ADC_CCR_PRESCALE_16:
            ADC_Frequency = ADC_KernelClock_Frequency / 16;
            break;

        case MD_ADC_CCR_PRESCALE_32:
            ADC_Frequency = ADC_KernelClock_Frequency / 32;
            break;

        case MD_ADC_CCR_PRESCALE_64:
            ADC_Frequency = ADC_KernelClock_Frequency / 64;
            break;

        case MD_ADC_CCR_PRESCALE_128:
            ADC_Frequency = ADC_KernelClock_Frequency / 128;
            break;

        case MD_ADC_CCR_PRESCALE_256:
            ADC_Frequency = ADC_KernelClock_Frequency / 256;
            break;
    }

    temp = (sampletime) * (ADC_Frequency / (double)1000000000) - 3.5;

    if (0 <= temp && temp <= 63)
    {
        feedback.smp = 0 << 6;
        Tsmp = (int)(temp + 0.5);
        feedback.smp += ((int)(temp));
    }
    else if (64 <= temp && temp <= 190)
    {
        feedback.smp = 1 << 6;
        Tsmp = (int)((temp - 64) / 2) * 2 + 64;
        feedback.smp += ((int)((temp - 64) / 2 + 0.5));
    }
    else if (192 <= temp && temp <= 444)
    {
        feedback.smp = 2 << 6;
        Tsmp = ((int)((temp - 192) / 4)) * 4 + 192;
        feedback.smp += ((int)((temp - 192) / 4 + 0.5));
    }
    else if (448 <= temp && temp <= 952)
    {
        feedback.smp = 2 << 6;
        Tsmp = ((int)((temp - 448) / 4)) * 8 + 448;
        feedback.smp += ((int)((temp - 448) / 4 + 0.5));
    }
    else
    {
        feedback.conversion_time = -1;
        return feedback;
    }

    switch (md_adc_get_resolution_selection(adc))
    {
        case MD_ADC_CFG1_RESOL_12BIT:
            resolution = 13.5;
            break;

        case MD_ADC_CFG1_RESOL_10BIT:
            resolution = 11.5;
            break;

        case MD_ADC_CFG1_RESOL_8BIT:
            resolution = 9.5;
            break;

        case MD_ADC_CFG1_RESOL_6BIT:
            resolution = 7.5;
            break;
    }

    feedback.conversion_time = ((double)(3.5 + resolution + Tsmp) / ADC_Frequency) * 1000000000;

    return feedback;
}

//When jeos is triggered, you need to use this function to reset
void md_adc_inject_sequence_reset(ADC_TypeDef *adc)
{
    md_adc_set_start_injected(adc);
    md_adc_set_stop_injected(adc);

    while (md_adc_get_start_injected(adc) == 1);
}

void md_adc_init(ADC_TypeDef *adc, ADC_COMM_TypeDef *adc12_common, md_adc_init_typedef *adc_init)
{
    md_rcu_enable_adc(RCU);

    md_rcu_set_adc_kernel_source(RCU, MD_ADC_KERNEL_DISABLE);
    md_adc_close(adc);

    if (md_adc_is_enabled_adcpower(adc) == 0 &&
            md_adc_is_disabled_adcpower(adc) == 0 &&
            md_adc_get_start_injected(adc) == 0 &&
            md_adc_get_start_regular(adc) == 0 &&
            md_adc_get_stop_regular(adc) == 0 &&
            md_adc_get_stop_injected(adc) == 0)
    {
        md_adc_set_clock_divider(adc12_common, adc_init->clock_source_prescale);
        md_adc_set_dma_mode(adc12_common, adc_init->dual_mode.dma_mode);
        md_adc_set_delay_time(adc12_common, adc_init->dual_mode.delay_timr_between_samples);
        md_adc_set_dual_mode(adc12_common, adc_init->dual_mode.mode_select);

        md_rcu_set_adc_kernel_source(RCU, adc_init->clock_source);

        switch ((int)adc_init->clock_source)
        {
            case MD_ADC_KERNEL_PLLCLK:
                ADC_KernelClock_Frequency = PLLFrequency;
                break;

            case MD_ADC_KERNEL_SYSCLK:
                ADC_KernelClock_Frequency = SystemFrequency_SysClk;
                break;

            case MD_ADC_KERNEL_AHBCLK:
                ADC_KernelClock_Frequency = SystemFrequency_AHBClk;
                break;
        }

        if (adc_init->adc_lowpower_mode == MD_ADC_LOWPOWER_ENABLE)
            md_adc_enable_low_power_mode(adc);
        else
            md_adc_disable_low_power_mode(adc);

        md_adc_set_resolution_selection(adc, adc_init->data_resolution);
        md_adc_set_data_alignment(adc, adc_init->data_align);

        if (adc_init->dma_en == MD_ADC_DMA_ENABLE)
            md_adc_enable_dma_access(adc);
        else
            md_adc_disable_dma_access(adc);
    }

    md_adc_open(adc);
}

void md_adc_input_channel_config(ADC_TypeDef *adc, md_adc_input_channel_config_typedef *config)
{
    if (md_adc_get_start_regular(adc))
        md_adc_set_stop_regular(adc);

    if (md_adc_get_start_injected(adc))
        md_adc_set_stop_injected(adc);

    while (md_adc_get_start_regular(adc) == 1 || md_adc_get_start_injected(adc) == 1);

    if (config->sample_time_trigger_mode_en == MD_ADC_SAMPLE_TIME_TRIGGER_MODE_ENABLE)
        md_adc_enable_sample_time_trigger(adc);
    else
        md_adc_disable_sample_time_trigger(adc);

    if (config->bulb_mod_en == MD_ADC_BULB_MODE_ENABLE)
        md_adc_enable_bulb_mode(adc);
    else
        md_adc_disable_bulb_mode(adc);

    if (config->extra_increase_sample_time_en == MD_ADC_EXTRA_INCREASE_SAMPLE_TIME_ENABLE)
        md_adc_enable_add_sample_time(adc);
    else
        md_adc_disable_add_sample_time(adc);

    if (config->inject_auto_en == MD_ADC_INJECT_AUTO_ENABLE)
        md_adc_enable_injected_auto_convert(adc);
    else
        md_adc_disable_injected_auto_convert(adc);

    md_adc_set_regular_convsersion_mode(adc, config->regular_continute_mode);

    md_adc_set_injected_sequence_length(adc, config->inject_channel_length);
    md_adc_set_injected_sequence_selection_1st(adc, config->inject_channel_array.array_1);
    md_adc_set_injected_sequence_selection_2nd(adc, config->inject_channel_array.array_2);
    md_adc_set_injected_sequence_selection_3rd(adc, config->inject_channel_array.array_3);
    md_adc_set_injected_sequence_selection_4th(adc, config->inject_channel_array.array_4);

    md_adc_set_regular_sequence_length(adc, config->regular_channel_length);
    md_adc_set_regular_sequence_selection_1st(adc, config->regular_channel_array.array_1);
    md_adc_set_regular_sequence_selection_2nd(adc, config->regular_channel_array.array_2);
    md_adc_set_regular_sequence_selection_3rd(adc, config->regular_channel_array.array_3);
    md_adc_set_regular_sequence_selection_4th(adc, config->regular_channel_array.array_4);
    md_adc_set_regular_sequence_selection_5th(adc, config->regular_channel_array.array_5);
    md_adc_set_regular_sequence_selection_6th(adc, config->regular_channel_array.array_6);
    md_adc_set_regular_sequence_selection_7th(adc, config->regular_channel_array.array_7);
    md_adc_set_regular_sequence_selection_8th(adc, config->regular_channel_array.array_8);
    md_adc_set_regular_sequence_selection_9th(adc, config->regular_channel_array.array_9);
    md_adc_set_regular_sequence_selection_10th(adc, config->regular_channel_array.array_10);
    md_adc_set_regular_sequence_selection_11th(adc, config->regular_channel_array.array_11);
    md_adc_set_regular_sequence_selection_12th(adc, config->regular_channel_array.array_12);
    md_adc_set_regular_sequence_selection_13th(adc, config->regular_channel_array.array_13);
    md_adc_set_regular_sequence_selection_14th(adc, config->regular_channel_array.array_14);
    md_adc_set_regular_sequence_selection_15th(adc, config->regular_channel_array.array_15);
    md_adc_set_regular_sequence_selection_16th(adc, config->regular_channel_array.array_16);

    md_adc_set_sampletime_channel_0(adc, config->sample_time_config.ch0);
    md_adc_set_sampletime_channel_1(adc, config->sample_time_config.ch1);
    md_adc_set_sampletime_channel_2(adc, config->sample_time_config.ch2);
    md_adc_set_sampletime_channel_3(adc, config->sample_time_config.ch3);
    md_adc_set_sampletime_channel_4(adc, config->sample_time_config.ch4);
    md_adc_set_sampletime_channel_5(adc, config->sample_time_config.ch5);
    md_adc_set_sampletime_channel_6(adc, config->sample_time_config.ch6);
    md_adc_set_sampletime_channel_7(adc, config->sample_time_config.ch7);
    md_adc_set_sampletime_channel_8(adc, config->sample_time_config.ch8);
    md_adc_set_sampletime_channel_9(adc, config->sample_time_config.ch9);
    md_adc_set_sampletime_channel_10(adc, config->sample_time_config.ch10);
    md_adc_set_sampletime_channel_11(adc, config->sample_time_config.ch11);
    md_adc_set_sampletime_channel_12(adc, config->sample_time_config.ch12);
    md_adc_set_sampletime_channel_13(adc, config->sample_time_config.ch13);
    md_adc_set_sampletime_channel_14(adc, config->sample_time_config.ch14);
    md_adc_set_sampletime_channel_15(adc, config->sample_time_config.ch15);
    md_adc_set_sampletime_channel_16(adc, config->sample_time_config.ch16);
    md_adc_set_sampletime_channel_17(adc, config->sample_time_config.ch17);
    md_adc_set_sampletime_channel_18(adc, config->sample_time_config.ch18);
}


void md_adc_regular_trigger_config(ADC_TypeDef *adc, md_adc_regular_trigger_config_typedef *config)
{
    if (md_adc_get_start_regular(adc))
        md_adc_set_stop_regular(adc);

    while (md_adc_get_start_regular(adc) == 1);

    md_adc_set_regular_external_trigger_source(adc, config->regular_external_trigger_source);
    md_adc_set_regular_trigger_mode(adc, config->regular_trigger_mode);

    if (config->regular_discontinuous_mode_en == MD_ADC_REGULAR_DISCONTINUOUS_MODE_ENABLE)
        md_adc_enable_regular_discontinous(adc);
    else
        md_adc_disable_regular_discontinous(adc);

    md_adc_set_regular_discontinous_num(adc, config->regular_discontinuous_channel_length);
}

void md_adc_inject_trigger_config(ADC_TypeDef *adc, md_adc_inject_trigger_config_typedef *config)
{
    if (md_adc_get_start_injected(adc))
        md_adc_set_stop_injected(adc);

    while (md_adc_get_start_injected(adc) == 1);

    md_adc_set_injected_external_trigger_source(adc, config->inject_external_trigger_source);
    md_adc_set_injected_trigger_mode(adc, config->inject_trigger_mode);

    if (config->inject_discontinuous_mode_en == MD_ADC_INJECT_DISCONTINUOUS_MODE_ENABLE)
        md_adc_enable_injected_discontinous(adc);
    else
        md_adc_disable_injected_discontinous(adc);

    md_adc_set_jsqr_mode(adc, config->inject_discontinuous_channel_mode);

    if (config->inject_queue_en == MD_ADC_INJECT_QUEUE_ENABLE)
        md_adc_enable_injected_queue(adc);
    else
        md_adc_disable_injected_queue(adc);
}

void md_adc_oversample_offset_config(ADC_TypeDef *adc, md_adc_oversample_offset_config_typedef *config)
{
    if (md_adc_get_start_regular(adc))
        md_adc_set_stop_regular(adc);

    if (md_adc_get_start_injected(adc))
        md_adc_set_stop_injected(adc);

    while (md_adc_get_start_regular(adc) == 1 || md_adc_get_start_injected(adc) == 1);

    md_adc_set_regular_oversampling_mode(adc, config->regular_oversample_mode);
    md_adc_set_oversampling_trigger_mode(adc, config->oversample_trigger_mode);
    md_adc_set_oversampling_data_shift(adc, config->oversample_shift);
    md_adc_set_oversampling_rate(adc, config->oversample_ratio);

    if (config->regular_oversample_en == MD_ADC_REGULAR_OVERSAMPLE_ENABLE)
        md_adc_enable_regular_oversampling(adc);
    else
        md_adc_disable_regular_oversampling(adc);

    if (config->inject_oversample_en == MD_ADC_INJECT_OVERSAMPLE_ENABLE)
        md_adc_enable_injected_oversampling(adc);
    else
        md_adc_disable_injected_oversampling(adc);

    md_adc_set_compensation_value_1(adc, config->offset_cofig.offset_reg1);
    md_adc_set_compensation_polarity_1(adc, config->offset_cofig.positive_offset_reg1);
    md_adc_set_compensation_channel_selection_1(adc, config->offset_cofig.offset_reg1_channel);

    if (config->offset_cofig.data_saturation_reg1_en == MD_ADC_DATA_SATURATION_ENABLE)
        md_adc_enable_compensation_saturation_1(adc);
    else
        md_adc_disable_compensation_saturation_1(adc);

    if (config->offset_cofig.offset_reg1_en == MD_ADC_OFFSET_ENABLE)
        md_adc_enable_offset_compensation_1(adc);
    else
        md_adc_disable_offset_compensation_1(adc);

    md_adc_set_compensation_value_2(adc, config->offset_cofig.offset_reg2);
    md_adc_set_compensation_polarity_2(adc, config->offset_cofig.positive_offset_reg2);
    md_adc_set_compensation_channel_selection_2(adc, config->offset_cofig.offset_reg2_channel);

    if (config->offset_cofig.data_saturation_reg2_en == MD_ADC_DATA_SATURATION_ENABLE)
        md_adc_enable_compensation_saturation_2(adc);
    else
        md_adc_disable_compensation_saturation_2(adc);

    if (config->offset_cofig.offset_reg2_en == MD_ADC_OFFSET_ENABLE)
        md_adc_enable_offset_compensation_2(adc);
    else
        md_adc_disable_offset_compensation_2(adc);

    md_adc_set_compensation_value_3(adc, config->offset_cofig.offset_reg3);
    md_adc_set_compensation_polarity_3(adc, config->offset_cofig.positive_offset_reg3);
    md_adc_set_compensation_channel_selection_3(adc, config->offset_cofig.offset_reg3_channel);

    if (config->offset_cofig.data_saturation_reg3_en == MD_ADC_DATA_SATURATION_ENABLE)
        md_adc_enable_compensation_saturation_3(adc);
    else
        md_adc_disable_compensation_saturation_3(adc);

    if (config->offset_cofig.offset_reg3_en == MD_ADC_OFFSET_ENABLE)
        md_adc_enable_offset_compensation_3(adc);
    else
        md_adc_disable_offset_compensation_3(adc);

    md_adc_set_compensation_value_4(adc, config->offset_cofig.offset_reg4);
    md_adc_set_compensation_polarity_4(adc, config->offset_cofig.positive_offset_reg4);
    md_adc_set_compensation_channel_selection_4(adc, config->offset_cofig.offset_reg4_channel);

    if (config->offset_cofig.data_saturation_reg4_en == MD_ADC_DATA_SATURATION_ENABLE)
        md_adc_enable_compensation_saturation_4(adc);
    else
        md_adc_disable_compensation_saturation_4(adc);

    if (config->offset_cofig.offset_reg4_en == MD_ADC_OFFSET_ENABLE)
        md_adc_enable_offset_compensation_4(adc);
    else
        md_adc_disable_offset_compensation_4(adc);

}


void md_adc_watchdog_config(ADC_TypeDef *adc, md_adc_watchdog_config_typedef *config)
{
    if (md_adc_get_start_regular(adc))
        md_adc_set_stop_regular(adc);

    if (md_adc_get_start_injected(adc))
        md_adc_set_stop_injected(adc);

    while (md_adc_get_start_regular(adc) == 1 || md_adc_get_start_injected(adc) == 1);

    md_adc_set_awd1_channel_sel(adc, config->analog_watchdog1_channel_select);
    md_adc_set_awd1_filter(adc, config->analog_watchdog1_filter);
    md_adc_set_awd1_monitor_channel(adc, config->analog_watchdog1_channel);
    md_adc_set_awd1_high_threshold(adc, config->analog_watchdog1_channel_upper_limit);
    md_adc_set_awd1_low_threshold(adc, config->analog_watchdog1_channel_lower_limit);
    md_adc_set_awd2_channel(adc, config->analog_watchdog2_channel);
    md_adc_set_awd2_high_threshold(adc, config->analog_watchdog2_channel_upper_limit);
    md_adc_set_awd2_low_threshold(adc, config->analog_watchdog2_channel_lower_limit);
    md_adc_set_awd3_channel(adc, config->analog_watchdog3_channel);
    md_adc_set_awd3_high_threshold(adc, config->analog_watchdog3_channel_upper_limit);
    md_adc_set_awd3_low_threshold(adc, config->analog_watchdog3_channel_lower_limit);

    if (config->regular_analog_watchdog1_en == MD_ADC_REGULAR_ANALOG_WATCHDOG1_ENABLE)
        md_adc_enable_regular_awd(adc);
    else
        md_adc_disable_regular_awd(adc);

    if (config->inject_analog_watchdog1_en == MD_ADC_INJECT_ANALOG_WATCHDOG1_ENABLE)
        md_adc_enable_injected_awd(adc);
    else
        md_adc_disable_injected_awd(adc);

}
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
