/**********************************************************************************
 *
 * @file    md_vres.h
 * @brief   header file of md_vres.c
 *
 * @date    19 Apr 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Apr 2023     Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_VRES_H__
#define __MD_VRES_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32m0502.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_VRES VRES
  * @brief VRES micro driver
  * @{
  */

/** @defgroup MD_VRES_Public_Macros VRES Public Macros
  * @{
  */

/**
  * @brief VRES_CTRL VRES_CTRL Register
  */
typedef enum
{  
    MD_VRES_0V               = (0U) ,
    MD_VRES_1_BY_VDDA_DIV32  = (1U) ,
    MD_VRES_2_BY_VDDA_DIV32  = (2U) ,
    MD_VRES_3_BY_VDDA_DIV32  = (3U) ,
    MD_VRES_4_BY_VDDA_DIV32  = (4U) ,
    MD_VRES_5_BY_VDDA_DIV32  = (5U) ,
    MD_VRES_6_BY_VDDA_DIV32  = (6U) ,
    MD_VRES_7_BY_VDDA_DIV32  = (7U) ,
    MD_VRES_8_BY_VDDA_DIV32  = (8U) ,
    MD_VRES_9_BY_VDDA_DIV32  = (9U) ,
    MD_VRES_10_BY_VDDA_DIV32 = (10U),
    MD_VRES_11_BY_VDDA_DIV32 = (11U),
    MD_VRES_12_BY_VDDA_DIV32 = (12U),
    MD_VRES_13_BY_VDDA_DIV32 = (13U),
    MD_VRES_14_BY_VDDA_DIV32 = (14U),
    MD_VRES_15_BY_VDDA_DIV32 = (15U),
    MD_VRES_16_BY_VDDA_DIV32 = (16U),
    MD_VRES_17_BY_VDDA_DIV32 = (17U),
    MD_VRES_18_BY_VDDA_DIV32 = (18U),
    MD_VRES_19_BY_VDDA_DIV32 = (19U),
    MD_VRES_20_BY_VDDA_DIV32 = (20U),
    MD_VRES_21_BY_VDDA_DIV32 = (21U),
    MD_VRES_22_BY_VDDA_DIV32 = (22U),
    MD_VRES_23_BY_VDDA_DIV32 = (23U),
    MD_VRES_24_BY_VDDA_DIV32 = (24U),
    MD_VRES_25_BY_VDDA_DIV32 = (25U),
    MD_VRES_26_BY_VDDA_DIV32 = (26U),
    MD_VRES_27_BY_VDDA_DIV32 = (27U),
    MD_VRES_28_BY_VDDA_DIV32 = (28U),
    MD_VRES_29_BY_VDDA_DIV32 = (29U),
    MD_VRES_30_BY_VDDA_DIV32 = (30U),
    MD_VRES_31_BY_VDDA_DIV32 = (31U),
}md_vres_sel_t;
/**
  * @}
  */

/** @defgroup MD_VRES_Public_Functions VRES Public Functions
  * @{
  */

/** @defgroup CTRL
  * @{
  */

/**
  * @brief  resistor divider switch enable
  * @param  vres VRES Instance.
  * @retval None.
  */
__STATIC_INLINE void md_vres_enable_resistor_divider(VRES_TypeDef *vres)
{
    SET_BIT(vres->CTRL, VRES_CTRL_VRESEN);
}

/**
  * @}
  */
  
/** @defgroup VSEL
  * @{
  */

/**
  * @brief  Set VRES_VSEL register
  * @param  vres VRES Instance.
  * @param  vsel  
  * @retval none
  */
__STATIC_INLINE void md_vres_set_vsel(VRES_TypeDef *vres, uint32_t vsel)
{
    WRITE_REG(vres->VSEL, vsel);
}
/**
  * @brief  Get VRES_VSEL register
  * @param  vres VRES Instance.
  * @retval register value
  */
__STATIC_INLINE uint32_t md_vres_get_vsel(VRES_TypeDef *vres)
{
    return (uint32_t)(READ_REG(vres->VSEL));
}

/**
  * @brief  Set Resistor Divider VRES2 Output Selection
  * @param  vres VRES Instance.
  * @param  vres2sel This parameter can be one of the following values:
  *         @arg @ref MD_VRES_0V
  *         @arg @ref MD_VRES_1_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_2_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_3_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_4_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_5_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_6_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_7_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_8_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_9_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_10_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_11_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_12_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_13_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_14_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_15_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_16_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_17_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_18_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_19_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_20_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_21_BY_VDDA_DIV32  
  *         @arg @ref MD_VRES_22_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_23_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_24_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_25_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_26_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_27_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_28_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_29_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_30_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_31_BY_VDDA_DIV32
  * @retval none
  */
__STATIC_INLINE void md_vres_set_resistor_divider_vres2(VRES_TypeDef *vres, md_vres_sel_t vres2sel)
{
    MODIFY_REG(vres->VSEL, VRES_VSEL_VRES2SEL, vres2sel << VRES_VSEL_VRES2SEL_POSS);        
}
/**
  * @brief  Get Resistor Divider VRES2 Output Selection
  * @param  vres VRES Instance.
  * @retval mode This parameter can be one of the following values:
  *         @arg @ref MD_VRES_0V
  *         @arg @ref MD_VRES_1_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_2_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_3_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_4_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_5_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_6_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_7_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_8_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_9_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_10_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_11_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_12_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_13_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_14_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_15_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_16_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_17_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_18_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_19_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_20_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_21_BY_VDDA_DIV32  
  *         @arg @ref MD_VRES_22_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_23_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_24_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_25_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_26_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_27_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_28_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_29_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_30_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_31_BY_VDDA_DIV32
  */
__STATIC_INLINE uint32_t md_vres_get_resistor_divider_vres2(VRES_TypeDef *vres)
{
    return (uint32_t)(READ_BIT(vres->VSEL, VRES_VSEL_VRES2SEL) >> VRES_VSEL_VRES2SEL_POSS);     
}

/**
  * @brief  Set Resistor Divider VRES1 Output Selection
  * @param  vres VRES Instance.
  * @param  vres1sel This parameter can be one of the following values:
  *         @arg @ref MD_VRES_0V
  *         @arg @ref MD_VRES_1_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_2_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_3_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_4_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_5_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_6_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_7_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_8_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_9_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_10_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_11_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_12_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_13_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_14_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_15_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_16_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_17_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_18_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_19_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_20_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_21_BY_VDDA_DIV32  
  *         @arg @ref MD_VRES_22_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_23_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_24_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_25_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_26_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_27_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_28_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_29_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_30_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_31_BY_VDDA_DIV32
  * @retval none
  */
__STATIC_INLINE void md_vres_set_resistor_divider_vres1(VRES_TypeDef *vres, md_vres_sel_t vres1sel)
{
    MODIFY_REG(vres->VSEL, VRES_VSEL_VRES1SEL, vres1sel << VRES_VSEL_VRES1SEL_POSS);        
}
/**
  * @brief  Get Resistor Divider VRES1 Output Selection
  * @param  vres VRES Instance.
  * @retval mode This parameter can be one of the following values:
  *         @arg @ref MD_VRES_0V
  *         @arg @ref MD_VRES_1_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_2_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_3_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_4_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_5_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_6_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_7_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_8_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_9_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_10_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_11_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_12_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_13_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_14_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_15_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_16_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_17_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_18_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_19_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_20_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_21_BY_VDDA_DIV32  
  *         @arg @ref MD_VRES_22_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_23_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_24_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_25_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_26_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_27_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_28_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_29_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_30_BY_VDDA_DIV32
  *         @arg @ref MD_VRES_31_BY_VDDA_DIV32
  */
__STATIC_INLINE uint32_t md_vres_get_resistor_divider_vres1(VRES_TypeDef *vres)
{
    return (uint32_t)(READ_BIT(vres->VSEL, VRES_VSEL_VRES1SEL) >> VRES_VSEL_VRES1SEL_POSS);     
}

/**
  * @}
  */  
  
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif
#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
