/**********************************************************************************
 *
 * @file    md_timer.h
 * @brief   header file of md_timer.c
 *
 * @date    07 May 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          07 May 2024     Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_TIMER_H_
#define __MD_TIMER_H_

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes ------------------------------------------------------------------ */
#include "md_utils.h"

/* Exported Constants -------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */

/* Exported Types ------------------------------------------------------------ */

/** @addtogroup Micro_Driver
  * @{
  */
  
/** @defgroup MD_TIMER TIMER
  * @brief TIMER micro driver
  * @{
  */  

/** @defgroup MD_TIMER_Public_Macros TIMER Public Macros
  * @{
  */

/* Exported Macros ----------------------------------------------------------- */

/* TIMER_CON1 macros define */
#define MD_TIMER_DEBUGSELECT_INPUT          (0) /*!<  Channel input*/
#define MD_TIMER_DEBUGSELECT_OUTPUT         (1) /*!<  Channel output*/

/** @defgroup MD_TIMER_Private_Macros TIMER Private Macros
  * @{
  */

#define MD_IS_TIMER_CC2_INSTANCE(x)  (((x) == AD16C6T1) || \
                                      ((x) == AD16C6T2) || \
                                      ((x) == GP32C4T1) || \
                                      ((x) == GP32C4T2) || \
                                      ((x) == GP16C2T1) || \
                                      ((x) == GP16C2T2))
#define MD_IS_TIMER_REPETITION_COUNTER_INSTANCE(x) (((x) == AD16C6T1) || \
                                                    ((x) == AD16C6T2))
#define MD_IS_TIMER_CLOCK_DIVISION_INSTANCE(x) MD_IS_TIMER_CC2_INSTANCE(x)

/** @defgroup MD_TIMER_Public_Types TIMER Public Types
  * @{
  */
/**
  * @brief TIMER counter mode
  */
typedef enum
{
    MD_TIMER_CNT_MODE_UP      = 0U, /**< Counter mode up */
    MD_TIMER_CNT_MODE_DOWN    = 1U, /**< Counter mode down */
    MD_TIMER_CNT_MODE_CENTER1 = 2U, /**< Counter mode center1 */
    MD_TIMER_CNT_MODE_CENTER2 = 3U, /**< Counter mode center2 */
    MD_TIMER_CNT_MODE_CENTER3 = 4U, /**< Counter mode center3 */
} md_timer_cnt_mode_t;

/**
  * @brief TIMER clock division
  */
typedef enum
{
    MD_TIMER_CLOCK_DIV1 = 0U, /**< No prescaler is used */
    MD_TIMER_CLOCK_DIV2 = 1U, /** Clock is divided by 2 */
    MD_TIMER_CLOCK_DIV4 = 2U, /** Clock is divided by 4 */
} md_timer_clock_division_t;

/**
  * @brief TIMER output compare and PWM modes
  */
typedef enum
{
    MD_TIMER_OC_MODE_TIMERING       = 0U, /**< Output compare mode is timering */
    MD_TIMER_OC_MODE_ACTIVE         = 1U, /**< Output compare mode is active */
    MD_TIMER_OC_MODE_INACTIVE       = 2U, /**< Output compare mode is inactive */
    MD_TIMER_OC_MODE_TOGGLE         = 3U, /**< Output compare mode is toggle */
    MD_TIMER_OC_MODE_FORCE_INACTIVE = 4U, /**< Output compare mode is force inactive */
    MD_TIMER_OC_MODE_FORCE_ACTIVE   = 5U, /**< Output compare mode is force active */
    MD_TIMER_OC_MODE_PWM1           = 6U, /**< Output compare mode is pwm1 */
    MD_TIMER_OC_MODE_PWM2           = 7U, /**< Output compare mode is pwm2 */
    MD_TIMER_OC_MODE_MULTI_PULSE1   = 8U, /**< Output compare mode is multi pulse1 */
    MD_TIMER_OC_MODE_MULTI_PULSE2   = 9U, /**< Output compare mode is multi pulse2 */
    MD_TIMER_OC_MODE_COMBINED_PWM1  = 12U,/**< Output compare mode is combined pwm1 */
    MD_TIMER_OC_MODE_COMBINED_PWM2  = 13U,/**< Output compare mode is combined pwm2 */
    MD_TIMER_OC_MODE_ASY_C2_PWM1    = 14U,/**< Output compare mode is asymmetric C2 pwm1 */
    MD_TIMER_OC_MODE_ASY_C2_PWM2    = 15U,/**< Output compare mode is asymmetric C2 pwm2 */
    MD_TIMER_OC_MODE_ASY_C5_PWM1    = 16U,/**< Output compare mode is asymmetric C5 pwm1 */
    MD_TIMER_OC_MODE_ASY_C5_PWM2    = 17U,/**< Output compare mode is asymmetric C5 pwm2 */
} md_timer_oc_mode_t;

/**
  * @brief TIMER output compare polarity
  */
typedef enum
{
    MD_TIMER_OC_POLARITY_HIGH = 0U, /**< Output compare polarity is high */
    MD_TIMER_OC_POLARITY_LOW  = 1U, /**< Output compare polarity is low */
} md_timer_oc_polarity_t;

/**
  * @brief TIMER channel
  */
typedef enum
{
    MD_TIMER_CHANNEL_1   = 0U,   /**< Channel 1 */
    MD_TIMER_CHANNEL_2   = 1U,   /**< Channel 2 */
    MD_TIMER_CHANNEL_3   = 2U,   /**< Channel 3 */
    MD_TIMER_CHANNEL_4   = 3U,   /**< Channel 4 */
} md_timer_channel_t;

/**
  * @brief TIMER complementary output compare polarity
  */
typedef enum
{
    MD_TIMER_OCN_POLARITY_HIGH = 0U,    /**< Complementary output compare polarity is high */
    MD_TIMER_OCN_POLARITY_LOW  = 1U,    /**< Complementary output compare polarity is low */
} md_timer_ocn_polarity_t;

/**
  * @brief TIMER output compare idle state
  */
typedef enum
{
    MD_TIMER_OC_IDLE_RESET = 0U,    /**< Output compare idle state is reset */
    MD_TIMER_OC_IDLE_SET   = 1U,    /**< Output compare idle state is set */
} md_timer_oc_idle_t;

/**
  * @brief TIMER complementary output compare idle state
  */
typedef enum
{
    MD_TIMER_OCN_IDLE_RESET = 0U,   /**< Complementary output compare idle state is reset */
    MD_TIMER_OCN_IDLE_SET   = 1U,   /**< Complementary output compare idle state is set */
} md_timer_ocn_idle_t;

/**
  * @brief TIMER output compare state
  */
typedef enum
{
    MD_TIMER_OCSTATE_DISABLE = 0U,  /**< OCx is not active */
    MD_TIMER_OCSTATE_ENABLE  = 1U,  /**< OCx signal is output on the corresponding output pin */
} md_timer_oc_state_t;

/**
  * @brief TIMER encoder mode
  */
typedef enum
{
    MD_TIMER_ENC_MODE_TI1  = 1U, /**< encoder mode 1 */
    MD_TIMER_ENC_MODE_TI2  = 2U, /**< encoder mode 2 */
    MD_TIMER_ENC_MODE_TI12 = 3U, /**< encoder mode 3 */
} md_timer_encoder_mode_t;

/**
  * @brief TIMER input capture polarity
  */
typedef enum
{
    MD_TIMER_IC_POLARITY_RISE = 0U, /**< Input capture polarity rising */
    MD_TIMER_IC_POLARITY_FALL = 1U, /**< Input capture polarity falling */
    MD_TIMER_IC_POLARITY_BOTH = 5U, /**< Input capture polarity rising and falling */
} md_timer_ic_polarity_t;

/**
  *@brief TIMER input capture selection
  */
typedef enum
{
    MD_TIMER_IC_SEL_DIRECT   = 1U, /**< IC1 -- TI1 */
    MD_TIMER_IC_SEL_INDIRECT = 2U, /**< IC1 -- TI2 */
    MD_TIMER_IC_SEL_TRC      = 3U, /**< IC1 -- TRC */
} md_timer_ic_select_t;

/**
  * @brief TIMER input capture prescaler
  */
typedef enum
{
    MD_TIMER_IC_PSC_DIV1 = 0U, /**< Capture performed once every 1 events */
    MD_TIMER_IC_PSC_DIV2 = 1U, /**< Capture performed once every 2 events */
    MD_TIMER_IC_PSC_DIV4 = 2U, /**< Capture performed once every 4 events */
    MD_TIMER_IC_PSC_DIV8 = 3U, /**< Capture performed once every 4 events */
} md_timer_ic_prescaler_t;

/**
  * @brief TIMER lock level
  */
typedef enum
{
    MD_TIMER_LOCK_LEVEL_OFF = 0U, /**< Lock off */
    MD_TIMER_LOCK_LEVEL_1   = 1U, /**< Lock level 1 */
    MD_TIMER_LOCK_LEVEL_2   = 2U, /**< Lock level 2 */
    MD_TIMER_LOCK_LEVEL_3   = 3U, /**< Lock level 3 */
} md_timer_lock_level_t;

/**
  * @brief TIMER break polarity
  */
typedef enum
{
    MD_TIMER_BREAK_POLARITY_LOW  = 0U, /**< LOW */
    MD_TIMER_BREAK_POLARITY_HIGH = 1U, /**< HIGH */
} md_timer_break_polarity_t;

/**
  * @}
  */

typedef enum
{
    MD_TIMER_CLOCKDIVSION_DIV1=(0),   /*!<  tDTS=tCK_INT*/
    MD_TIMER_CLOCKDIVSION_DIV2=(1),   /*!<  tDTS=2*tCK_INT*/
    MD_TIMER_CLOCKDIVSION_DIV4=(2)    /*!<  tDTS=4*tCK_INT*/
}md_timer_clockdivsion_div_t;

typedef enum
{
    MD_TIMER_CENTERALIGNED_DISABLE=(0),   /*!<  Edge-aligned mode*/
    MD_TIMER_CENTERALIGNED_DOWN=(1),      /*!<  Center-aligned mode 1*/
    MD_TIMER_CENTERALIGNED_UP=(2),        /*!<  Center-aligned mode 2*/
    MD_TIMER_CENTERALIGNED_BOTH=(3)       /*!<  Center-aligned mode 3*/
}md_timer_centeraligned_list_t;

typedef enum
{
    MD_TIMER_UPCOUNTER=(0),   /*!<  Counter used as upcounter*/
    MD_TIMER_DOWNCOUNTER=(1)  /*!<  Counter used as downcounter*/
}md_timer_counter_direction_t;

typedef enum
{
    MD_TIMER_UPDATESOURCE_NORMAL=(0), /*!<  Counter overflow/underflow, Setting the UG bit, Update generation through the slave mode controller generate an update interrupt*/
    MD_TIMER_UPDATESOURCE_COUNTER=(1) /*!<  Only counter overflow/underflow generates an update interrupt or DMA request if Enabled*/
}md_timer_update_source_t;

/* TIMER_CON2 macros define*/
typedef enum
{   
    MD_TIMER_OISS4_IDLEOUTPUT_LOW=(0),      /*!<  OISS4=0 (after a dead-time if CH4ON is implemented) when GOEN=0*/
    MD_TIMER_OISS4_IDLEOUTPUT_HIGH=(1)      /*!<  OISS4=1 (after a dead-time if CH4ON is implemented) when GOEN=0*/
}md_timer_oiss4_idleoutput_t;

typedef enum
{
    MD_TIMER_OISS3N_IDLEOUTPUT_LOW=(0),   /*!<  OISS3N=0 (after a dead-time if CH3ON is implemented) when GOEN=0*/
    MD_TIMER_OISS3N_IDLEOUTPUT_HIGH=(1)   /*!<  OISS3N=1 (after a dead-time if CH3ON is implemented) when GOEN=0*/
}md_timer_oiss3n_idleoutput_t;

typedef enum
{
    MD_TIMER_OISS3_IDLEOUTPUT_LOW=(0),    /*!<  OISS3=0 (after a dead-time if CH3ON is implemented) when GOEN=0*/
    MD_TIMER_OISS3_IDLEOUTPUT_HIGH=(1)    /*!<  OISS3=1 (after a dead-time if CH3ON is implemented) when GOEN=0*/
}md_timer_oiss3_idleoutput_t;

typedef enum
{
    MD_TIMER_OISS2N_IDLEOUTPUT_LOW=(0),   /*!<  OISS2N=0 (after a dead-time if CH2ON is implemented) when GOEN=0*/
    MD_TIMER_OISS2N_IDLEOUTPUT_HIGH=(1)   /*!<  OISS2N=1 (after a dead-time if CH2ON is implemented) when GOEN=0*/
}md_timer_oiss2n_idleoutput_t;

typedef enum
{
    MD_TIMER_OISS2_IDLEOUTPUT_LOW=(0),    /*!<  OISS2=0 (after a dead-time if CH2ON is implemented) when GOEN=0*/
    MD_TIMER_OISS2_IDLEOUTPUT_HIGH=(1)    /*!<  OISS2=1 (after a dead-time if CH2ON is implemented) when GOEN=0*/
}md_timer_oiss2_idleoutput_t;

typedef enum
{
    MD_TIMER_OISS1N_IDLEOUTPUT_LOW=(0),     /*!<  OISS1N=0 (after a dead-time if CH1ON is implemented) when GOEN=0*/
    MD_TIMER_OISS1N_IDLEOUTPUT_HIGH=(1)     /*!<  OISS1N=1 (after a dead-time if CH1ON is implemented) when GOEN=0*/
}md_timer_oiss1n_idleoutput_t;

typedef enum
{
    MD_TIMER_OISS1_IDLEOUTPUT_LOW=(0),    /*!<  OISS1=0 (after a dead-time if CH1ON is implemented) when GOEN=0*/
    MD_TIMER_OISS1_IDLEOUTPUT_HIGH=(1)    /*!<  OISS1=1 (after a dead-time if CH1ON is implemented) when GOEN=0*/
}md_timer_oiss1_idleoutput_t;

typedef enum
{
    MD_TIMER_TI1INPUT_CH1=(0),    /*!<  The TIMERx_CH1 pin is connected to TI1 input*/
    MD_TIMER_TI1INPUT_XOR=(1)     /*!<  The TIMERx_CH1, CH2 and CH3 pins are connected to the TI1 input (XOR combination)*/
}md_timer_ti1input_list_t;

typedef enum
{
    MD_TIMER_MASTERMODE_RESET=(0),                    /*!<  Reset mode*/
    MD_TIMER_MASTERMODE_ENABLE=(1),                   /*!<  Enable mode*/
    MD_TIMER_MASTERMODE_UPDATE=(2),                   /*!<  Update mode*/
    MD_TIMER_MASTERMODE_COMPAREPULSE=(3),             /*!<  Compare Pulse mode*/
    MD_TIMER_MASTERMODE_COMPARE1=(4),                 /*!<  Compare 1 mode*/
    MD_TIMER_MASTERMODE_COMPARE2=(5),                 /*!<  Compare 2 mode*/
    MD_TIMER_MASTERMODE_COMPARE3=(6),                 /*!<  Compare 3 mode*/
    MD_TIMER_MASTERMODE_COMPARE4=(7),                 /*!<  Compare 4 mode*/
    MD_TIMER_MASTERMODE_ENCODER_CLOCK=(8),            /*!<  encoder clock mode*/
    MD_TIMER_MASTERMODE_MULTI_POINT_TRIGGER=(9)       /*!<  multi point trigger mode*/
}md_timer_mastermode_list_t;

typedef enum
{
    MD_TIMER_DMASELECTION_COMPARE=(0),    /*!<  CCx DMA request sent when CCx event occurs*/
    MD_TIMER_DMASELECTION_UPDATE=(1)      /*!<  CCx DMA requests sent when update event occurs*/
}md_timer_dmaselection_t;

typedef enum
{
    MD_TIMER_UPDATESELECTION_COMG=(0),    /*!<  When capture/compare control bits are preloaded (CCPC=1), they are updated by setting the COMG bit only*/
    MD_TIMER_UPDATESELECTION_BOTH=(1)     /*!<  When capture/compare control bits are preloaded (CCPC=1), they are updated by setting the COMG bit or when an rising edge occurs on TRGI*/
}md_timer_updateselection_t;

/* Exported Macros ----------------------------------------------------------- */

/* TIMER_SMCON macros define*/
#define MD_TIMER_TRIGGER_IT0                0x0    /*!<  Internal Trigger 0*/
#define MD_TIMER_TRIGGER_IT1                0x1    /*!<  Internal Trigger 1*/
#define MD_TIMER_TRIGGER_IT2                0x2    /*!<  Internal Trigger 2*/
#define MD_TIMER_TRIGGER_IT3                0x3    /*!<  Internal Trigger 3*/
#define MD_TIMER_TRIGGER_I1F_ED             0x4    /*!<  TI1 Edge Detector*/
#define MD_TIMER_TRIGGER_I1F                0x5    /*!<  Filtered Timer Input 1*/
#define MD_TIMER_TRIGGER_I2F                0x6    /*!<  Filtered Timer Input 2*/
#define MD_TIMER_TRIGGER_ETF                0x7    /*!<  External Trigger input*/
#define MD_TIMER_TRIGGER_IT4                0x8    /*!<  Internal Trigger 4*/
#define MD_TIMER_TRIGGER_IT5                0x9    /*!<  Internal Trigger 5*/
#define MD_TIMER_TRIGGER_IT6                0xA    /*!<  Internal Trigger 6*/
#define MD_TIMER_TRIGGER_IT7                0xB    /*!<  Internal Trigger 7*/
#define MD_TIMER_TRIGGER_IT8                0xC    /*!<  Internal Trigger 8*/

typedef enum
{
    MD_TIMER_ETRPOLARITY_NONINVERTED=(0),   /*!<  ETR is non-inverted*/
    MD_TIMER_ETRPOLARITY_INVERTED=(1)       /*!<  ETR is inverted*/
}md_timer_etrpolarity_t;

typedef enum
{
    MD_TIMER_ETRPRESCALER_DISABLE=(0),    /*!<  Prescaler OFF*/
    MD_TIMER_ETRPRESCALER_DIV2=(1),       /*!<  ETR frequency divided by 2*/
    MD_TIMER_ETRPRESCALER_DIV4=(2),       /*!<  ETR frequency divided by 4*/
    MD_TIMER_ETRPRESCALER_DIV8=(3)        /*!<  ETR frequency divided by 8*/
}md_timer_etrprescaler_t;

typedef enum
{
    MD_TIMER_ETRFILTER_FDIV1=(0),       /*!<  No filter*/
    MD_TIMER_ETRFILTER_FDIV1N2=(1),     /*!<  fSAMPLING = fCK_INT, N = 2*/
    MD_TIMER_ETRFILTER_FDIV1N4=(2),     /*!<  fSAMPLING = fCK_INT, N = 4*/
    MD_TIMER_ETRFILTER_FDIV1N8=(3),     /*!<  fSAMPLING = fCK_INT, N = 8*/
    MD_TIMER_ETRFILTER_FDIV2N6=(4),     /*!<  fSAMPLING = fDTS / 2, N = 6*/
    MD_TIMER_ETRFILTER_FDIV2N8=(5),     /*!<  fSAMPLING = fDTS / 2, N = 8*/
    MD_TIMER_ETRFILTER_FDIV4N6=(6),     /*!<  fSAMPLING = fDTS / 4, N = 6*/
    MD_TIMER_ETRFILTER_FDIV4N8=(7),     /*!<  fSAMPLING = fDTS / 4, N = 8*/
    MD_TIMER_ETRFILTER_FDIV8N6=(8),     /*!<  fSAMPLING = fDTS / 8, N = 6*/
    MD_TIMER_ETRFILTER_FDIV8N8=(9),     /*!<  fSAMPLING = fDTS / 8, N = 8*/
    MD_TIMER_ETRFILTER_FDIV16N5=(10),   /*!<  fSAMPLING = fDTS / 16, N = 5*/
    MD_TIMER_ETRFILTER_FDIV16N6=(11),   /*!<  fSAMPLING = fDTS / 16, N = 6*/
    MD_TIMER_ETRFILTER_FDIV16N8=(12),   /*!<  fSAMPLING = fDTS / 16, N = 8*/
    MD_TIMER_ETRFILTER_FDIV32N5=(13),   /*!<  fSAMPLING = fDTS / 32, N = 5*/
    MD_TIMER_ETRFILTER_FDIV32N6=(14),   /*!<  fSAMPLING = fDTS / 32, N = 6*/
    MD_TIMER_ETRFILTER_FDIV32N8=(15)    /*!<  fSAMPLING = fDTS / 32, N = 8*/
}md_timer_etrfilter_t;

typedef enum
{
    MD_TIMER_OCLRS_OCLR=(0),      /*!<  OCLR*/
    MD_TIMER_OCLRS_ETP=(1)        /*!<  ETP*/
}md_timer_oclrs_t;

typedef enum
{
    MD_TIMER_SLAVEMODE_DISABLE=(0),                                   /*!<  Slave mode disabled*/
    MD_TIMER_SLAVEMODE_ENCODER1=(1),                                  /*!<  Encoder mode 1*/
    MD_TIMER_SLAVEMODE_ENCODER2=(2),                                  /*!<  Encoder mode 2*/
    MD_TIMER_SLAVEMODE_ENCODER3=(3),                                  /*!<  Encoder mode 3*/
    MD_TIMER_SLAVEMODE_RESET=(4),                                     /*!<  Reset Mode*/
    MD_TIMER_SLAVEMODE_GATED=(5),                                     /*!<  Gated Mode*/
    MD_TIMER_SLAVEMODE_TRIGGER=(6),                                   /*!<  Trigger Mode*/
    MD_TIMER_SLAVEMODE_EXTERNALCLOCK=(7),                             /*!<  External Clock Mode 1*/
    MD_TIMER_SLAVEMODE_RESET_TRIGGER=(8),                             /*!<  Reset & Trigger Mode*/
    MD_TIMER_SLAVEMODE_GATED_RESET=(9),                               /*!<  Gated & Reset Mode*/
    MD_TIMER_SLAVEMODE_CLOCK_PLUS_DIRECTION_ENCODER_MODE1=(10),       /*!<  Clock Plus Direction Encoder Mode 1*/
    MD_TIMER_SLAVEMODE_CLOCK_PLUS_DIRECTION_ENCODER_MODE2=(11),       /*!<  Clock Plus Direction Encoder Mode 2*/
    MD_TIMER_SLAVEMODE_DIRECTIONAL_CLOCK_ENCODER_MODE1=(12),          /*!<  Directional Clock Encoder Mode 1*/
    MD_TIMER_SLAVEMODE_DIRECTIONAL_CLOCK_ENCODER_MODE2=(13),          /*!<  Directional Clock Encoder Mode 2*/
    MD_TIMER_SLAVEMODE_ENCODER4=(14),                                 /*!<  Encoder mode 4*/
    MD_TIMER_SLAVEMODE_ENCODER5=(15)                                  /*!<  Encoder mode 5*/
}md_timer_slavemode_list_t;


/* TIMER_CHMR1 and TIMER_CHMR2 output macros define*/
typedef enum
{
    MD_TIMER_OUTPUTMODE_DISABLE=(0),                            /*!<  Frozen*/
    MD_TIMER_OUTPUTMODE_HIGHONMSTCH=(1),                        /*!<  Set channel 1 to active level on match*/
    MD_TIMER_OUTPUTMODE_LOWONMSTCH=(2),                         /*!<  Set channel 1 to inactive level on match*/
    MD_TIMER_OUTPUTMODE_TOGGLE=(3),                             /*!<  Toggle*/
    MD_TIMER_OUTPUTMODE_FORCELOW=(4),                           /*!<  Force inactive level*/
    MD_TIMER_OUTPUTMODE_FORCEHIGH=(5),                          /*!<  Force active level*/
    MD_TIMER_OUTPUTMODE_PWMMODE1=(6),                           /*!<  PWM mode 1*/
    MD_TIMER_OUTPUTMODE_PWMMODE2=(7),                           /*!<  PWM mode 2*/
    MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1=(8),   /*!<  Multi-trigger single pulse mode 1*/
    MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2=(9),   /*!<  Multi-trigger single pulse mode 2*/
    MD_TIMER_OUTPUTMODE_COMPARE_PULSE_MODE=(10),                /*!<  Compare Pulse Mode*/
    MD_TIMER_OUTPUTMODE_DIRECTION_OUTPUT_MODE=(11),             /*!<  Direction output mode*/
    MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1=(12),                /*!<  Combined PWM mode 1*/
    MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2=(13),                /*!<  Combined PWM mode 2*/
    MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1=(14),           /*!<  Asymmetric C2 PWM Mode 1*/
    MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2=(15),           /*!<  Asymmetric C2 PWM Mode 2*/
    MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1=(16),           /*!<  Asymmetric C5 PWM Mode 1*/
    MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2=(17)            /*!<  Asymmetric C5 PWM Mode 2*/
}md_timer_outputmode_list_t;

/* TIMER_CHMR1 and TIMER_CHMR2 input macros define*/
typedef enum
{
    MD_TIMER_INPUTFILTER_FDIV1=(0),     /*!<  No filter*/
    MD_TIMER_INPUTFILTER_FDIV1N2=(1),   /*!<  fSAMPLING = fCK_INT, N = 2*/
    MD_TIMER_INPUTFILTER_FDIV1N4=(2),   /*!<  fSAMPLING = fCK_INT, N = 4*/
    MD_TIMER_INPUTFILTER_FDIV1N8=(3),   /*!<  fSAMPLING = fCK_INT, N = 8*/
    MD_TIMER_INPUTFILTER_FDIV2N6=(4),   /*!<  fSAMPLING = fDTS / 2, N = 6*/
    MD_TIMER_INPUTFILTER_FDIV2N8=(5),   /*!<  fSAMPLING = fDTS / 2, N = 8*/
    MD_TIMER_INPUTFILTER_FDIV4N6=(6),   /*!<  fSAMPLING = fDTS / 4, N = 6*/
    MD_TIMER_INPUTFILTER_FDIV4N8=(7),   /*!<  fSAMPLING = fDTS / 4, N = 8*/
    MD_TIMER_INPUTFILTER_FDIV8N6=(8),   /*!<  fSAMPLING = fDTS / 8, N = 6*/
    MD_TIMER_INPUTFILTER_FDIV8N8=(9),   /*!<  fSAMPLING = fDTS / 8, N = 8*/
    MD_TIMER_INPUTFILTER_FDIV16N5=(10), /*!<  fSAMPLING = fDTS / 16, N = 5*/
    MD_TIMER_INPUTFILTER_FDIV16N6=(11), /*!<  fSAMPLING = fDTS / 16, N = 6*/
    MD_TIMER_INPUTFILTER_FDIV16N8=(12), /*!<  fSAMPLING = fDTS / 16, N = 8*/
    MD_TIMER_INPUTFILTER_FDIV32N5=(13), /*!<  fSAMPLING = fDTS / 32, N = 5*/
    MD_TIMER_INPUTFILTER_FDIV32N6=(14), /*!<  fSAMPLING = fDTS / 32, N = 6*/
    MD_TIMER_INPUTFILTER_FDIV32N8=(15)  /*!<  fSAMPLING = fDTS / 32, N = 8*/
}md_timer_inputfilter_list_t;

typedef enum
{
    MD_TIMER_INPUTPRESCALE_DIV1=(0),    /*!<  no prescaler*/
    MD_TIMER_INPUTPRESCALE_DIV2=(1),    /*!<  capture is done once every 2 events*/
    MD_TIMER_INPUTPRESCALE_DIV4=(2),    /*!<  capture is done once every 4 events*/
    MD_TIMER_INPUTPRESCALE_DIV8=(3)     /*!<  capture is done once every 8 events*/
}md_timer_inputprescale_t;

/* TIMER_CHMR1/CHMR2 input/output macros define*/
typedef enum
{
    MD_TIMER_CHMODE_OUTPUT=(0),           /*!<  CCx channel is configured as output*/
    MD_TIMER_CHMODE_INPUT_DIRECT=(1),     /*!<  CCx channel is configured as input, ICx is mapped direct*/
    MD_TIMER_CHMODE_INPUT_INDIRECT=(2),   /*!<  CCx channel is configured as input, ICx is mapped indirect*/
    MD_TIMER_CHMODE_INPUT_TRC=(3)         /*!<  CCx channel is configured as input, ICx is mapped TRC*/
}md_timer_chmode_t;

/* TIMER_CCEP input macros define*/
typedef enum
{
    MD_TIMER_OUTPUTPOLARITY_HIGH=(0),   /*!<  active high*/
    MD_TIMER_OUTPUTPOLARITY_LOW=(1)     /*!<  active low*/
}md_timer_polarity_t;

/* TIMER_BDCFG input macros define*/
typedef enum
{
    MD_TIMER_BREAKPOLARITY_LOW=(0),   /*!<  Break input BRK is active low*/
    MD_TIMER_BREAKPOLARITY_HIGH=(1)   /*!<  Break input BRK is active high */
}md_timer_breakpolarity_t;

typedef enum
{
    MD_TIMER_OFFSTATERUN_DISABLE=(0),   /*!<  This bit is used when MOE=1, when inactive, OC/OCN outputs are disabled*/
    MD_TIMER_OFFSTATERUN_ENABLE=(1)     /*!<  This bit is used when MOE=1, when inactive, OC/OCN outputs are enabled*/
}md_timer_offstaterun_t;

typedef enum
{
    MD_TIMER_OFFSTATEIDLE_DISABLE=(0),  /*!<  This bit is used when MOE=0, when inactive, OC/OCN outputs are disabled*/
    MD_TIMER_OFFSTATEIDLE_ENABLE=(1)    /*!<  This bit is used when MOE=0, when inactive, OC/OCN outputs are forced*/
}md_timer_offstateidle_t;

typedef enum
{
    MD_TIMER_LOCKLEVEL_0=(0),   /*!<  LOCK OFF*/
    MD_TIMER_LOCKLEVEL_1=(1),   /*!<  LOCK Level 1*/
    MD_TIMER_LOCKLEVEL_2=(2),   /*!<  LOCK Level 2*/
    MD_TIMER_LOCKLEVEL_3=(3)    /*!<  LOCK Level 3*/
}md_timer_locklevel_t;

/* TIMER_OPTR input macros define*/
typedef enum
{
    MD_TIMER_ETRRMP_ETR=(0),      /*!<  Timer ETR*/
    MD_TIMER_ETRRMP_CMP1=(1),     /*!<  CMP1 output*/
    MD_TIMER_ETRRMP_CMP2=(2),     /*!<  CMP2 output*/
    MD_TIMER_ETRRMP_ADCAWD=(3)    /*!<  ADC watchdog*/
}md_timer_etrrmp_t;

#define MD_TIMER_CH4RMP_CH4                 (0)     /*!<  Timer channel 4*/

#define MD_TIMER_CH3RMP_CH3                 (0)     /*!<  Timer channel 3*/

#define MD_TIMER_CH2RMP_CH2                 (0)     /*!<  Timer channel 2*/
#define MD_TIMER_CH2RMP_CMP2                (1)     /*!<  CMP2 output*/

#define MD_TIMER_CH1RMP_CH1                 (0)     /*!<  Timer channel 1*/
#define MD_TIMER_CH1RMP_CMP1                (1)     /*!<  CMP1 output*/

typedef enum
{
    MD_TIMER_BRKFLT_FREQ_FDIV1=(0),       /*!<  No filter*/
    MD_TIMER_BRKFLT_FREQ_FDIV1N2=(1),     /*!<  fSAMPLING = fCK_INT, N = 2*/
    MD_TIMER_BRKFLT_FREQ_FDIV1N4=(2),     /*!<  fSAMPLING = fCK_INT, N = 4*/
    MD_TIMER_BRKFLT_FREQ_FDIV1N8=(3),     /*!<  fSAMPLING = fCK_INT, N = 8*/
    MD_TIMER_BRKFLT_FREQ_FDIV2N6=(4),     /*!<  fSAMPLING = fDTS / 2, N = 6*/
    MD_TIMER_BRKFLT_FREQ_FDIV2N8=(5),     /*!<  fSAMPLING = fDTS / 2, N = 8*/
    MD_TIMER_BRKFLT_FREQ_FDIV4N6=(6),     /*!<  fSAMPLING = fDTS / 4, N = 6*/
    MD_TIMER_BRKFLT_FREQ_FDIV4N8=(7),     /*!<  fSAMPLING = fDTS / 4, N = 8*/
    MD_TIMER_BRKFLT_FREQ_FDIV8N6=(8),     /*!<  fSAMPLING = fDTS / 8, N = 6*/
    MD_TIMER_BRKFLT_FREQ_FDIV8N8=(9),     /*!<  fSAMPLING = fDTS / 8, N = 8*/
    MD_TIMER_BRKFLT_FREQ_FDIV16N5=(10),   /*!<  fSAMPLING = fDTS / 16, N = 5*/
    MD_TIMER_BRKFLT_FREQ_FDIV16N6=(11),   /*!<  fSAMPLING = fDTS / 16, N = 6*/
    MD_TIMER_BRKFLT_FREQ_FDIV16N8=(12),   /*!<  fSAMPLING = fDTS / 16, N = 8*/
    MD_TIMER_BRKFLT_FREQ_FDIV32N5=(13),   /*!<  fSAMPLING = fDTS / 32, N = 5*/
    MD_TIMER_BRKFLT_FREQ_FDIV32N6=(14),   /*!<  fSAMPLING = fDTS / 32, N = 6*/
    MD_TIMER_BRKFLT_FREQ_FDIV32N8=(15)    /*!<  fSAMPLING = fDTS / 32, N = 8*/
}md_timer_brkflt_freq_t;

/**
  * @brief  TIMER time base configuration structure definition
  */
typedef struct
{
    uint32_t prescaler;                /**< Specifies the prescaler value used to divide the TIMER clock. */
    md_timer_cnt_mode_t mode;          /**< Specifies the counter mode. */
    uint32_t period;                   /**< Specifies the period value to be loaded into ARR at the next update event. */
    md_timer_clock_division_t clk_div; /**< Specifies the clock division.*/
    uint32_t re_cnt;                   /**< Specifies the repetition counter value. */
} md_timer_base_init_t;

/**
  * @brief  TIMER output compare configuration structure definition
  */
typedef struct
{
    md_timer_oc_state_t ocstate;
    md_timer_oc_state_t ocnstate;
    md_timer_oc_mode_t oc_mode;           /**< Specifies the TIMER mode. */
    uint32_t pulse;                       /**< Specifies the pulse value to be loaded into the Capture Compare Register. */
    md_timer_oc_polarity_t oc_polarity;   /**< Specifies the output polarity. */
    md_timer_ocn_polarity_t ocn_polarity; /**< Specifies the complementary output polarity. */
    type_func_t oc_fast_en;               /**< Specifies the Fast mode state. */
    md_timer_oc_idle_t oc_idle;           /**< Specifies the TIMER Output Compare pin state during Idle state. */
    md_timer_ocn_idle_t ocn_idle;         /**< Specifies the TIMER Output Compare pin state during Idle state. */
} md_timer_oc_init_t;

/**
  * @brief  TIMER input capture configuration structure definition
  */
typedef struct
{
    md_timer_ic_polarity_t polarity; /**< Specifies the active edge of the input signal */
    md_timer_ic_select_t sel;        /**< Specifies the input */
    md_timer_ic_prescaler_t psc;     /**< Specifies the Input Capture Prescaler */
    uint32_t filter;                 /**< Specifies the input capture filter */
} md_timer_ic_init_t;

/**
  * @brief  TIMER hall sensor configuretion structure definition
  */
typedef struct
{
    md_timer_ic_polarity_t polarity; /**< Specifies the active edge of the input signal */
    md_timer_ic_prescaler_t psc;     /**< Specifies the Input Capture Prescaler */
    uint32_t filter;                 /**< Specifies the input capture filter [0x0, 0xF] */
    uint32_t delay;                  /**< Specifies the pulse value to be loaded into the register [0x0, 0xFFFF] */
} md_timer_hall_sensor_init_t;

/**
  * @brief TIMER encoder configuration structure definition
  */
typedef struct
{
    md_timer_slavemode_list_t mode;        /**< Specifies the encoder mode */
    md_timer_ic_polarity_t ic1_polarity; /**< Specifies the active edge of the input signal */
    md_timer_ic_select_t ic1_sel;        /**< Specifies the input */
    md_timer_ic_prescaler_t ic1_psc;     /**< Specifies the Input Capture Prescaler */
    uint32_t ic1_filter;                 /**< Specifies the input capture filter */
    md_timer_ic_polarity_t ic2_polarity; /**< Specifies the active edge of the input signal */
    md_timer_ic_select_t ic2_sel;        /**< Specifies the input */
    md_timer_ic_prescaler_t ic2_psc;     /**< Specifies the Input Capture Prescaler */
    uint32_t ic2_filter;                 /**< Specifies the input capture filter */
} md_timer_encoder_init_t;

/**
  * @brief  TIMER break and dead time configuretion structure definition
  */
typedef struct
{
    type_func_t off_run;                /**< Enalbe/Disable off state in run mode */
    type_func_t off_idle;               /**< Enalbe/Disable off state in idle mode */
    md_timer_lock_level_t lock_level;   /**< Lock level */
    uint32_t dead_time;                 /**< Dead time, [0x0, 0xFF] */
    type_func_t break_state;            /**< Break state */
    md_timer_break_polarity_t polarity; /**< Break input polarity */
    type_func_t auto_out;               /**< Enalbe/Disable automatic output */
} md_timer_break_dead_time_t;

/* Exported Functions -------------------------------------------------------- */

/**
  * @}
  */

/** @defgroup MD_TIMER_Public_Functions TIMER Public Functions
  * @{
  */

/** @defgroup CON1
  * @{
  */
/**
  * @brief  Timer CON1 setup.
  * @param  TIMERx TIMER instance
  * @param  value (DBGSEL | DFCKSEL | ARPEN | CMSEL | DIRSEL | SPMEN | UERSEL | DISUE | CNTEN)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_con1(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CON1, value);
}

/**
  * @brief  GET Timer CON1 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CON1 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_con1(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CON1));
}

/**
  * @brief  Timer enable DITHEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_dithering_mode(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CON1, TIMER_CON1_DITHEN);
}

/**
  * @brief  Timer disable DITHEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_dithering_mode(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CON1, TIMER_CON1_DITHEN);
}

/**
  * @brief  Timer is enabled DITHEN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_dithering_mode(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CON1, TIMER_CON1_DITHEN) == (TIMER_CON1_DITHEN));
}


/**
  * @brief  Timer enable UPDREMAP
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_upd_remapping(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CON1, TIMER_CON1_UPDREMAP);
}

/**
  * @brief  Timer disable UPDREMAP
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_upd_remapping(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CON1, TIMER_CON1_UPDREMAP);
}

/**
  * @brief  Timer is enabled UPDREMAP
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_upd_remapping(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CON1, TIMER_CON1_UPDREMAP) == (TIMER_CON1_UPDREMAP));
}

/**
  * @brief  Timer clock division setup.
  * @param  TIMERx TIMER instance
  * @param  ClockDivision Clock division
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV1
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV2
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV4
  * @retval None
  */

__STATIC_INLINE void md_timer_set_df_clock_dfcksel(TIMER_TypeDef *TIMERx, md_timer_clockdivsion_div_t ClockDivision)
{
    MODIFY_REG(TIMERx->CON1, TIMER_CON1_DFCKSEL, ClockDivision << TIMER_CON1_DFCKSEL_POSS);
}

/**
  * @brief  Get timer Clock division.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV1
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV2
            @arg @ref MD_TIMER_CLOCKDIVSION_DIV4
  */

__STATIC_INLINE md_timer_clockdivsion_div_t md_timer_get_df_clock_dfcksel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_clockdivsion_div_t)(READ_BIT(TIMERx->CON1, TIMER_CON1_DFCKSEL) >> TIMER_CON1_DFCKSEL_POSS);
}

/**
  * @brief  Timer auto-reload preload enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_autoreload_preload_arpen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CON1, TIMER_CON1_ARPEN);
}

/**
  * @brief  Timer auto-reload preload disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_autoreload_preload_arpen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CON1, TIMER_CON1_ARPEN);
}

/**
  * @brief  Indicates whether the timer auto-reload preload is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_autoreload_preload_arpen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CON1, TIMER_CON1_ARPEN) == (TIMER_CON1_ARPEN));
}

/**
  * @brief  Timer center-aligned mode selection setup.
  * @param  TIMERx TIMER instance
  * @param  CenterAlignedMode center-aligned mode selection
            @arg @ref MD_TIMER_CENTERALIGNED_DISABLE
            @arg @ref MD_TIMER_CENTERALIGNED_DOWN
            @arg @ref MD_TIMER_CENTERALIGNED_UP
            @arg @ref MD_TIMER_CENTERALIGNED_BOTH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_counter_aligned_mode_cmsel(TIMER_TypeDef *TIMERx, md_timer_centeraligned_list_t CenterAlignedMode)
{
    MODIFY_REG(TIMERx->CON1, TIMER_CON1_CMSEL, CenterAlignedMode << TIMER_CON1_CMSEL_POSS);
}

/**
  * @brief  Get timer center-aligned mode selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_CENTERALIGNED_DISABLE
            @arg @ref MD_TIMER_CENTERALIGNED_DOWN
            @arg @ref MD_TIMER_CENTERALIGNED_UP
            @arg @ref MD_TIMER_CENTERALIGNED_BOTH
  */

__STATIC_INLINE md_timer_centeraligned_list_t md_timer_get_counter_aligned_mode_cmsel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_centeraligned_list_t)(READ_BIT(TIMERx->CON1, TIMER_CON1_CMSEL) >> TIMER_CON1_CMSEL_POSS);
}

/**
  * @brief  Timer counting direction setup.
  * @param  TIMERx TIMER instance
  * @param  direction Counting direction
            @arg @ref MD_TIMER_UPCOUNTER
            @arg @ref MD_TIMER_DOWNCOUNTER
  * @retval None
  */

__STATIC_INLINE void md_timer_set_counter_direction_dirsel(TIMER_TypeDef *TIMERx, md_timer_counter_direction_t direction)
{
    MODIFY_REG(TIMERx->CON1, TIMER_CON1_DIRSEL, direction << TIMER_CON1_DIRSEL_POS);
}

/**
  * @brief  Get timer counting direction.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_UPCOUNTER
            @arg @ref MD_TIMER_DOWNCOUNTER
  */

__STATIC_INLINE md_timer_counter_direction_t md_timer_get_counter_direction_dirsel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_counter_direction_t)(READ_BIT(TIMERx->CON1, TIMER_CON1_DIRSEL) >> TIMER_CON1_DIRSEL_POS);
}

/**
  * @brief  Timer one pulse mode enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_onepulse_spmen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CON1, TIMER_CON1_SPMEN);
}

/**
  * @brief  Timer one pulse mode disable.
  * @param  TIMERx TIMER instance
  * @retval None.
  */

__STATIC_INLINE void md_timer_disable_onepulse_spmen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CON1, TIMER_CON1_SPMEN);
}

/**
  * @brief  Indicates whether the timer one pulse mode is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_onepulse_spmen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CON1, TIMER_CON1_SPMEN) == (TIMER_CON1_SPMEN));
}

/**
  * @brief  Timer update request source setup.
  * @param  TIMERx TIMER instance
  * @param  UpdateSource Update request source select
            @arg @ref MD_TIMER_UPDATESOURCE_NORMAL
            @arg @ref MD_TIMER_UPDATESOURCE_COUNTER
  * @retval None
  */

__STATIC_INLINE void md_timer_set_update_source_uersel(TIMER_TypeDef *TIMERx, md_timer_update_source_t UpdateSource)
{
    MODIFY_REG(TIMERx->CON1, TIMER_CON1_UERSEL, UpdateSource << TIMER_CON1_UERSEL_POS);
}

/**
  * @brief  Get timer update request source.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_UPDATESOURCE_NORMAL
            @arg @ref MD_TIMER_UPDATESOURCE_COUNTER
  */

__STATIC_INLINE md_timer_update_source_t md_timer_get_update_source_uersel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_update_source_t)(READ_BIT(TIMERx->CON1, TIMER_CON1_UERSEL) >> TIMER_CON1_UERSEL_POS);
}

/**
  * @brief  Timer update event enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_update_disue(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CON1, TIMER_CON1_DISUE);
}

/**
  * @brief  Timer update event disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_update_disue(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CON1, TIMER_CON1_DISUE);
}

/**
  * @brief  Indicates whether the timer update event is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_update_disue(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CON1, TIMER_CON1_DISUE) == (TIMER_CON1_DISUE));
}

/**
  * @brief  Timer counter enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_counter_cnten(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CON1, TIMER_CON1_CNTEN);
}

/**
  * @brief  Timer counter disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_counter_cnten(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CON1, TIMER_CON1_CNTEN);
}

/**
  * @brief  Indicates whether the timer counter is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_counter_cnten(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CON1, TIMER_CON1_CNTEN) == (TIMER_CON1_CNTEN));
}
/**
  * @}
  */

/** @defgroup CON2
  * @{
  */
/**
  * @brief  Timer CON2 setup.
  * @param  TIMERx TIMER instance
  * @param  value (OISS4 | OISS3N | OISS3 | OISS2N | OISS2 | OISS1N | OISS1 | I1SEL | MMSEL | CCDMASEL | CCUSEL | CCPCEN)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_con2(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CON2, value);
}

/**
  * @brief  GET Timer CON2 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CON2 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_con2(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CON2));
}

/**
  * @brief  Set TIMER MMSELTGO2
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_master_mode_select_2(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_MMSELTGO2, value << TIMER_CON2_MMSELTGO2_POSS);
}

/**
  * @brief  Get TIMER MMSELTGO2
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_master_mode_select_2(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_MMSELTGO2) >> TIMER_CON2_MMSELTGO2_POSS);
}

/**
  * @brief  Set TIMER OISS6
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch6o_idle_select_oiss6(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS6, value << TIMER_CON2_OISS6_POS);
}

/**
  * @brief  Get TIMER OISS6
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch6o_idle_select_oiss6(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS6) >> TIMER_CON2_OISS6_POS);
}

/**
  * @brief  Set TIMER OISS5
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch5o_idle_select_oiss5(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS5, value << TIMER_CON2_OISS5_POS);
}

/**
  * @brief  Get TIMER OISS5
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch5o_idle_select_oiss5(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS5) >> TIMER_CON2_OISS5_POS);
}

/**
  * @brief  Set TIMER OISS4N
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch4on_idle_select_oiss4n(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS4N, value << TIMER_CON2_OISS4N_POS);
}

/**
  * @brief  Get TIMER OISS4N
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch4on_idle_select_oiss4n(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS4N) >> TIMER_CON2_OISS4N_POS);
}

/**
  * @brief  Timer output idle state 4 setup.
  * @param  TIMERx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_OISS4_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS4_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch4o_idle_select_oiss4(TIMER_TypeDef *TIMERx, md_timer_oiss4_idleoutput_t IdleOutput)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS4, IdleOutput << TIMER_CON2_OISS4_POS);
}

/**
  * @brief  Get timer output idle state 4.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OISS4_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS4_IDLEOUTPUT_HIGH
  */

__STATIC_INLINE md_timer_oiss4_idleoutput_t md_timer_get_ch4o_idle_select_oiss4(TIMER_TypeDef *TIMERx)
{
    return (md_timer_oiss4_idleoutput_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS4) >> TIMER_CON2_OISS4_POS);
}

/**
  * @brief  Timer output idle state 3N setup.
  * @param  TIMERx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_OISS3N_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS3N_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch3on_idle_select_oiss3n(TIMER_TypeDef *TIMERx, md_timer_oiss3n_idleoutput_t IdleOutput)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS3N, IdleOutput << TIMER_CON2_OISS3N_POS);
}

/**
  * @brief  Get timer output idle state 3N.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OISS3N_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS3N_IDLEOUTPUT_HIGH
  */

__STATIC_INLINE md_timer_oiss3n_idleoutput_t md_timer_get_ch3on_idle_select_oiss3n(TIMER_TypeDef *TIMERx)
{
    return (md_timer_oiss3n_idleoutput_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS3N) >> TIMER_CON2_OISS3N_POS);
}

/**
  * @brief  Timer output idle state 3 setup.
  * @param  TIMERx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_OISS3_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS3_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch3o_idle_select_oiss3(TIMER_TypeDef *TIMERx, md_timer_oiss3_idleoutput_t IdleOutput)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS3, IdleOutput << TIMER_CON2_OISS3_POS);
}

/**
  * @brief  Get timer output idle state 3.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OISS3_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS3_IDLEOUTPUT_HIGH
  */

__STATIC_INLINE md_timer_oiss3_idleoutput_t md_timer_get_ch3o_idle_select_oiss3(TIMER_TypeDef *TIMERx)
{
    return (md_timer_oiss3_idleoutput_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS3) >> TIMER_CON2_OISS3_POS);
}

/**
  * @brief  Timer output idle state 2N setup.
  * @param  TIMERx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_OISS2N_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS2N_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch2on_idle_select_oiss2n(TIMER_TypeDef *TIMERx, md_timer_oiss2n_idleoutput_t IdleOutput)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS2N, IdleOutput << TIMER_CON2_OISS2N_POS);
}

/**
  * @brief  Get timer output idle state 2N.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OISS2N_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS2N_IDLEOUTPUT_HIGH
  */

__STATIC_INLINE md_timer_oiss2n_idleoutput_t md_timer_get_ch2on_idle_select_oiss2n(TIMER_TypeDef *TIMERx)
{
    return (md_timer_oiss2n_idleoutput_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS2N) >> TIMER_CON2_OISS2N_POS);
}

/**
  * @brief  Timer output idle state 2 setup.
  * @param  TIMERx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_OISS2_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS2_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch2o_idle_select_oiss2(TIMER_TypeDef *TIMERx, md_timer_oiss2_idleoutput_t IdleOutput)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS2, IdleOutput << TIMER_CON2_OISS2_POS);
}

/**
  * @brief  Get timer output idle state 2.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OISS2_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS2_IDLEOUTPUT_HIGH
  */

__STATIC_INLINE md_timer_oiss2_idleoutput_t md_timer_get_ch2o_idle_select_oiss2(TIMER_TypeDef *TIMERx)
{
    return (md_timer_oiss2_idleoutput_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS2) >> TIMER_CON2_OISS2_POS);
}

/**
  * @brief  Timer output idle state 1N setup.
  * @param  TIMERx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_OISS1N_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS1N_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch1on_idle_select_oiss1n(TIMER_TypeDef *TIMERx, md_timer_oiss1n_idleoutput_t IdleOutput)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS1N, IdleOutput << TIMER_CON2_OISS1N_POS);
}

/**
  * @brief  Get timer output idle state 1N.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OISS1N_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS1N_IDLEOUTPUT_HIGH
  */

__STATIC_INLINE md_timer_oiss1n_idleoutput_t md_timer_get_ch1on_idle_select_oiss1n(TIMER_TypeDef *TIMERx)
{
    return (md_timer_oiss1n_idleoutput_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS1N) >> TIMER_CON2_OISS1N_POS);
}

/**
  * @brief  Timer output idle state 1 setup.
  * @param  TIMERx TIMER instance
  * @param  IdleOutput Output idle state
            @arg @ref MD_TIMER_OISS1_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS1_IDLEOUTPUT_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ch1o_idle_select_oiss1(TIMER_TypeDef *TIMERx, md_timer_oiss1_idleoutput_t IdleOutput)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_OISS1, IdleOutput << TIMER_CON2_OISS1_POS);
}

/**
  * @brief  Get timer output idle state 1.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OISS1_IDLEOUTPUT_LOW
            @arg @ref MD_TIMER_OISS1_IDLEOUTPUT_HIGH
  */

__STATIC_INLINE md_timer_oiss1_idleoutput_t md_timer_get_ch1o_idle_select_oiss1(TIMER_TypeDef *TIMERx)
{
    return (md_timer_oiss1_idleoutput_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_OISS1) >> TIMER_CON2_OISS1_POS);
}

/**
  * @brief  Timer TI1 selection setup.
  * @param  TIMERx TIMER instance
  * @param  TI1Input TI1 input select
            @arg @ref MD_TIMER_TI1INPUT_CH1
            @arg @ref MD_TIMER_TI1INPUT_XOR
  * @retval None
  */

__STATIC_INLINE void md_timer_set_i1_func_select_i1sel(TIMER_TypeDef *TIMERx, md_timer_ti1input_list_t TI1Input)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_I1SEL, TI1Input << TIMER_CON2_I1SEL_POS);
}

/**
  * @brief  Get timer TI1 selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_TI1INPUT_CH1
            @arg @ref MD_TIMER_TI1INPUT_XOR
  */

__STATIC_INLINE md_timer_ti1input_list_t md_timer_get_i1_func_select_i1sel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_ti1input_list_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_I1SEL) >> TIMER_CON2_I1SEL_POS);
}

/**
  * @brief  Timer master mode selection setup.
  * @param  TIMERx TIMER instance
  * @param  MasterMode Master mode selection
            @arg @ref MD_TIMER_MASTERMODE_RESET
            @arg @ref MD_TIMER_MASTERMODE_ENABLE
            @arg @ref MD_TIMER_MASTERMODE_UPDATE
            @arg @ref MD_TIMER_MASTERMODE_COMPAREPULSE
            @arg @ref MD_TIMER_MASTERMODE_COMPARE1
            @arg @ref MD_TIMER_MASTERMODE_COMPARE2
            @arg @ref MD_TIMER_MASTERMODE_COMPARE3
            @arg @ref MD_TIMER_MASTERMODE_COMPARE4
            @arg @ref MD_TIMER_MASTERMODE_ENCODER_CLOCK
            @arg @ref MD_TIMER_MASTERMODE_MULTI_POINT_TRIGGER
  * @retval None
  */

__STATIC_INLINE void md_timer_set_master_trgout_select_mmsel(TIMER_TypeDef *TIMERx, md_timer_mastermode_list_t MasterMode)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_MMSEL, (MasterMode & 0x7UL) << TIMER_CON2_MMSEL_POSS);
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_MMSEL2, ((MasterMode & 0x8UL) >> 3UL) << TIMER_CON2_MMSEL2_POS);
}

/**
  * @brief  Get timer master mode selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_MASTERMODE_RESET
            @arg @ref MD_TIMER_MASTERMODE_ENABLE
            @arg @ref MD_TIMER_MASTERMODE_UPDATE
            @arg @ref MD_TIMER_MASTERMODE_COMPAREPULSE
            @arg @ref MD_TIMER_MASTERMODE_COMPARE1
            @arg @ref MD_TIMER_MASTERMODE_COMPARE2
            @arg @ref MD_TIMER_MASTERMODE_COMPARE3
            @arg @ref MD_TIMER_MASTERMODE_COMPARE4
            @arg @ref MD_TIMER_MASTERMODE_ENCODER_CLOCK
            @arg @ref MD_TIMER_MASTERMODE_MULTI_POINT_TRIGGER
  */

__STATIC_INLINE md_timer_mastermode_list_t md_timer_get_master_trgout_select_mmsel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_mastermode_list_t)(((READ_BIT(TIMERx->CON2, TIMER_CON2_MMSEL2) >> TIMER_CON2_MMSEL2_POS) << 3UL) | (READ_BIT(TIMERx->CON2, TIMER_CON2_MMSEL) >> TIMER_CON2_MMSEL_POSS));
}

/**
  * @brief  Timer capture/compare DMA selection setup.
  * @param  TIMERx TIMER instance
  * @param  DMASelection Capture/compare DMA selection
            @arg @ref MD_TIMER_DMASELECTION_COMPARE
            @arg @ref MD_TIMER_DMASELECTION_UPDATE
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc_dma_select_ccdmasel(TIMER_TypeDef *TIMERx, md_timer_dmaselection_t DMASelection)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_CCDMASEL, DMASelection << TIMER_CON2_CCDMASEL_POS);
}

/**
  * @brief  Get timer capture/compare DMA selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_DMASELECTION_COMPARE
            @arg @ref MD_TIMER_DMASELECTION_UPDATE
  */

__STATIC_INLINE md_timer_dmaselection_t md_timer_get_cc_dma_select_ccdmasel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_dmaselection_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_CCDMASEL) >> TIMER_CON2_CCDMASEL_POS);
}

/**
  * @brief  Timer capture/compare control update selection setup.
  * @param  TIMERx TIMER instance
  * @param  UpdateSelection Capture/compare control update selection
            @arg @ref MD_TIMER_UPDATESELECTION_COMG
            @arg @ref MD_TIMER_UPDATESELECTION_BOTH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc_update_select_ccusel(TIMER_TypeDef *TIMERx, md_timer_updateselection_t UpdateSelection)
{
    MODIFY_REG(TIMERx->CON2, TIMER_CON2_CCUSEL, UpdateSelection << TIMER_CON2_CCUSEL_POS);
}

/**
  * @brief  Get timer capture/compare control update selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_UPDATESELECTION_COMG
            @arg @ref MD_TIMER_UPDATESELECTION_BOTH
 */

__STATIC_INLINE md_timer_updateselection_t md_timer_get_cc_update_select_ccusel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_updateselection_t)(READ_BIT(TIMERx->CON2, TIMER_CON2_CCUSEL) >> TIMER_CON2_CCUSEL_POS);
}

/**
  * @brief  Timer capture/compare preloaded control enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc_preload_ccpcen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CON2, TIMER_CON2_CCPCEN);
}

/**
  * @brief  Timer capture/compare preloaded control disable.
  * @param  TIMERx TIMER instance
  * @retval None.
  */

__STATIC_INLINE void md_timer_disable_cc_preload_ccpcen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CON2, TIMER_CON2_CCPCEN);
}

/**
  * @brief  Indicates whether the timer capture/compare preloaded control is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc_preload_ccpcen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CON2, TIMER_CON2_CCPCEN) == (TIMER_CON2_CCPCEN));
}
/**
  * @}
  */

/** @defgroup SMCON
  * @{
  */
/**
  * @brief  Timer SMCON setup.
  * @param  TIMERx TIMER instance
  * @param  value (TSSEL2 | ETPOL | ECM2EN | ETPRES | ETFLT | MSCFG | TSSEL1 | SMODS)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_smcon(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->SMCON, value);
}

/**
  * @brief  GET Timer SMCON register value.
  * @param  TIMERx TIMER instance
  * @retval Timer SMCON register value.
  */

__STATIC_INLINE uint32_t md_timer_get_smcon(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->SMCON));
}

/**
  * @brief  Set TIMER SMODPS
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_smodps(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_SMODPS, value << TIMER_SMCON_SMODPS_POS);
}

/**
  * @brief  Get TIMER SMODPS
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_smodps(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->SMCON, TIMER_SMCON_SMODPS) >> TIMER_SMCON_SMODPS_POS);
}

/**
  * @brief  Set TIMER SMODPE
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_smodpe(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_SMODPE, value << TIMER_SMCON_SMODPE_POS);
}

/**
  * @brief  Get TIMER SMODPE
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_smodpe(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->SMCON, TIMER_SMCON_SMODPE) >> TIMER_SMCON_SMODPE_POS);
}

/**
  * @brief  Timer trigger selection setup.
  * @param  TIMERx TIMER instance
  * @param  TrigerSelect External trigger selection
            @arg @ref MD_TIMER_TRIGGER_IT0
            @arg @ref MD_TIMER_TRIGGER_IT1
            @arg @ref MD_TIMER_TRIGGER_IT2
            @arg @ref MD_TIMER_TRIGGER_IT3
            @arg @ref MD_TIMER_TRIGGER_I1F_ED
            @arg @ref MD_TIMER_TRIGGER_I1F
            @arg @ref MD_TIMER_TRIGGER_I2F
            @arg @ref MD_TIMER_TRIGGER_ETF
            @arg @ref MD_TIMER_TRIGGER_IT4
            @arg @ref MD_TIMER_TRIGGER_IT5
            @arg @ref MD_TIMER_TRIGGER_IT6
            @arg @ref MD_TIMER_TRIGGER_IT7
            @arg @ref MD_TIMER_TRIGGER_IT8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_slave_trigger_tssel(TIMER_TypeDef *TIMERx, uint32_t TrigerSelect)
{
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_TSSEL1, (TrigerSelect & 0x7) << TIMER_SMCON_TSSEL1_POSS);
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_TSSEL2, (TrigerSelect >> 3) << TIMER_SMCON_TSSEL2_POSS);
}

/**
  * @brief  Get timer trigger selection setup.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_TRIGGER_IT0
            @arg @ref MD_TIMER_TRIGGER_IT1
            @arg @ref MD_TIMER_TRIGGER_IT2
            @arg @ref MD_TIMER_TRIGGER_IT3
            @arg @ref MD_TIMER_TRIGGER_I1F_ED
            @arg @ref MD_TIMER_TRIGGER_I1F
            @arg @ref MD_TIMER_TRIGGER_I2F
            @arg @ref MD_TIMER_TRIGGER_ETF
            @arg @ref MD_TIMER_TRIGGER_IT4
            @arg @ref MD_TIMER_TRIGGER_IT5
            @arg @ref MD_TIMER_TRIGGER_IT6
            @arg @ref MD_TIMER_TRIGGER_IT7
            @arg @ref MD_TIMER_TRIGGER_IT8
  */

__STATIC_INLINE uint32_t md_timer_get_slave_trigger_tssel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->SMCON, (TIMER_SMCON_TSSEL2 | TIMER_SMCON_TSSEL1)));
}

/**
  * @brief  Timer external trigger polarity setup.
  * @param  TIMERx TIMER instance
  * @param  ETRPolarity External trigger polarity
            @arg @ref MD_TIMER_ETRPOLARITY_NONINVERTED
            @arg @ref MD_TIMER_ETRPOLARITY_INVERTED
  * @retval None
  */

__STATIC_INLINE void md_timer_set_external_trigger_polarity_etpol(TIMER_TypeDef *TIMERx, md_timer_etrpolarity_t ETRPolarity)
{
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_ETPOL, ETRPolarity << TIMER_SMCON_ETPOL_POS);
}

/**
  * @brief  Timer external trigger polarity setup.
  * @param  TIMERx TIMER instance
  * @param  ETRPolarity External trigger polarity
  * @retval None
  */

__STATIC_INLINE md_timer_etrpolarity_t md_timer_get_external_trigger_polarity_etpol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_etrpolarity_t)(READ_BIT(TIMERx->SMCON, TIMER_SMCON_ETPOL) >> TIMER_SMCON_ETPOL_POS);
}

/**
  * @brief  Timer external clock enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_external_clk2mode_ecm2en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SMCON, TIMER_SMCON_ECM2EN);
}


/**
  * @brief  Timer external clock disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_external_clk2mode_ecm2en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->SMCON, TIMER_SMCON_ECM2EN);
}

/**
  * @brief  Indicates whether the timer external clock is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_external_clk2mode_ecm2en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->SMCON, TIMER_SMCON_ECM2EN) == (TIMER_SMCON_ECM2EN));
}

/**
  * @brief  Timer external trigger prescaler setup.
  * @param  TIMERx TIMER instance
  * @param  ETRPrescale External trigger prescaler
            @arg @ref MD_TIMER_ETRPRESCALER_DISABLE
            @arg @ref MD_TIMER_ETRPRESCALER_DIV2
            @arg @ref MD_TIMER_ETRPRESCALER_DIV4
            @arg @ref MD_TIMER_ETRPRESCALER_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_external_trigger_prescaler_etpres(TIMER_TypeDef *TIMERx, md_timer_etrprescaler_t ETRPrescale)
{
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_ETPRES, ETRPrescale << TIMER_SMCON_ETPRES_POSS);
}

/**
  * @brief  Get Timer external trigger prescaler setup.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_ETRPRESCALER_DISABLE
            @arg @ref MD_TIMER_ETRPRESCALER_DIV2
            @arg @ref MD_TIMER_ETRPRESCALER_DIV4
            @arg @ref MD_TIMER_ETRPRESCALER_DIV8
  */

__STATIC_INLINE md_timer_etrprescaler_t md_timer_get_external_trigger_prescaler_etpres(TIMER_TypeDef *TIMERx)
{
    return (md_timer_etrprescaler_t)(READ_BIT(TIMERx->SMCON, TIMER_SMCON_ETPRES) >> TIMER_SMCON_ETPRES_POSS);
}

/**
  * @brief  Timer external trigger filter setup.
  * @param  TIMERx TIMER instance
  * @param  ETRFilter External trigger filter
            @arg @ref MD_TIMER_ETRFILTER_FDIV1
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N2
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N4
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV2N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV2N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV4N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV4N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV8N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV8N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N5
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N5
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_external_trigger_filter_etflt(TIMER_TypeDef *TIMERx, md_timer_etrfilter_t ETRFilter)
{
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_ETFLT, ETRFilter << TIMER_SMCON_ETFLT_POSS);
}

/**
  * @brief  Get timer external trigger filter.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_ETRFILTER_FDIV1
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N2
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N4
            @arg @ref MD_TIMER_ETRFILTER_FDIV1N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV2N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV2N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV4N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV4N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV8N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV8N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N5
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV16N8
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N5
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N6
            @arg @ref MD_TIMER_ETRFILTER_FDIV32N8
  */

__STATIC_INLINE md_timer_etrfilter_t md_timer_get_external_trigger_filter_etflt(TIMER_TypeDef *TIMERx)
{
    return (md_timer_etrfilter_t)(READ_BIT(TIMERx->SMCON, TIMER_SMCON_ETFLT) >> TIMER_SMCON_ETFLT_POSS);
}

/**
  * @brief  Timer master/slave mode synchronize enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_master_mode_mscfg(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SMCON, TIMER_SMCON_MSCFG);
}

/**
  * @brief  Timer master/slave mode synchronize disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_master_mode_mscfg(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->SMCON, TIMER_SMCON_MSCFG);
}

/**
  * @brief  Get timer master/slave mode.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_master_mode_mscfg(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->SMCON, TIMER_SMCON_MSCFG) == TIMER_SMCON_MSCFG);
}

/**
  * @brief  Output channel clear selection setup.
  * @param  TIMERx TIMER instance
  * @param  selection clear selection
            @arg @ref MD_TIMER_OCLRS_OCLR
            @arg @ref MD_TIMER_OCLRS_ETP
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_channel_clear_oclrs(TIMER_TypeDef *TIMERx, md_timer_oclrs_t selection)
{
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_OCLRS, selection << TIMER_SMCON_OCLRS_POS);
}

/**
  * @brief  Get timer slave mode selection.
  * @param  TIMERx TIMER instance
  * @retval selection clear selection
            @arg @ref MD_TIMER_OCLRS_OCLR
            @arg @ref MD_TIMER_OCLRS_ETP
  */

__STATIC_INLINE md_timer_oclrs_t md_timer_get_output_channel_clear_oclrs(TIMER_TypeDef *TIMERx)
{
    return (md_timer_oclrs_t)(READ_BIT(TIMERx->SMCON, TIMER_SMCON_OCLRS) >> TIMER_SMCON_OCLRS_POS);
}

/**
  * @brief  Timer slave mode selection setup.
  * @param  TIMERx TIMER instance
  * @param  SlaveMode Slave mode selection
            @arg @ref MD_TIMER_SLAVEMODE_DISABLE
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER1
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER2
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER3
            @arg @ref MD_TIMER_SLAVEMODE_RESET
            @arg @ref MD_TIMER_SLAVEMODE_GATED
            @arg @ref MD_TIMER_SLAVEMODE_TRIGGER
            @arg @ref MD_TIMER_SLAVEMODE_EXTERNALCLOCK
            @arg @ref MD_TIMER_SLAVEMODE_RESET&TRIGGER
            @arg @ref MD_TIMER_SLAVEMODE_GATED&RESET
            @arg @ref MD_TIMER_SLAVEMODE_CLOCK_PLUS_DIRECTION_ENCODER_MODE1
            @arg @ref MD_TIMER_SLAVEMODE_CLOCK_PLUS_DIRECTION_ENCODER_MODE2
            @arg @ref MD_TIMER_SLAVEMODE_DIRECTIONAL_CLOCK_ENCODER_MODE1
            @arg @ref MD_TIMER_SLAVEMODE_DIRECTIONAL_CLOCK_ENCODER_MODE2
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER4
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER5
  * @retval None
  */

__STATIC_INLINE void md_timer_set_slave_mode_smods(TIMER_TypeDef *TIMERx, md_timer_slavemode_list_t SlaveMode)
{
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_SMODS, (SlaveMode & 0x7UL) << TIMER_SMCON_SMODS_POSS);
    MODIFY_REG(TIMERx->SMCON, TIMER_SMCON_SMODS2, ((SlaveMode & 0x8UL) >> 3UL) << TIMER_SMCON_SMODS2_POS);
}

/**
  * @brief  Get timer slave mode selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_SLAVEMODE_DISABLE
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER1
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER2
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER3
            @arg @ref MD_TIMER_SLAVEMODE_RESET
            @arg @ref MD_TIMER_SLAVEMODE_GATED
            @arg @ref MD_TIMER_SLAVEMODE_TRIGGER
            @arg @ref MD_TIMER_SLAVEMODE_EXTERNALCLOCK
            @arg @ref MD_TIMER_SLAVEMODE_RESET&TRIGGER
            @arg @ref MD_TIMER_SLAVEMODE_GATED&RESET
            @arg @ref MD_TIMER_SLAVEMODE_CLOCK_PLUS_DIRECTION_ENCODER_MODE1
            @arg @ref MD_TIMER_SLAVEMODE_CLOCK_PLUS_DIRECTION_ENCODER_MODE2
            @arg @ref MD_TIMER_SLAVEMODE_DIRECTIONAL_CLOCK_ENCODER_MODE1
            @arg @ref MD_TIMER_SLAVEMODE_DIRECTIONAL_CLOCK_ENCODER_MODE2
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER4
            @arg @ref MD_TIMER_SLAVEMODE_ENCODER5
  */

__STATIC_INLINE md_timer_slavemode_list_t md_timer_get_slave_mode_smods(TIMER_TypeDef *TIMERx)
{
    return (md_timer_slavemode_list_t)(((READ_BIT(TIMERx->SMCON, TIMER_SMCON_SMODS2) >> TIMER_SMCON_SMODS2_POS) << 3UL) | (READ_BIT(TIMERx->SMCON, TIMER_SMCON_SMODS) >> TIMER_SMCON_SMODS_POSS));
}
/**
  * @}
  */

/** @defgroup IER
  * @{
  */
/**
  * @brief  Timer IER setup.
  * @param  TIMERx TIMER instance
  * @param  value (CH4OV | CH3OV | CH2OV | CH1OV | BRK | TRGI | COM | CH4 | CH3 | CH2 | CH1 | UPD)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ier(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->IER, value);
}


/**
  * @brief  TIMER TERR interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_it_conversion_error(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_TERR);
}

/**
  * @brief  TIMER IERR interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_it_index_error(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_IERR);
}

/**
  * @brief  TIMER DIR interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_it_direction_change(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_DIR);
}

/**
  * @brief  TIMER IDX interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_it_effective_index(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_IDX);
}

/**
  * @brief  TIMER CH6 interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_it_ch6(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH6);
}

/**
  * @brief  TIMER CH5 interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_it_ch5(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH5);
}

/**
  * @brief  TIMER SBRK interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_it_system_brake(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_SBRK);
}

/**
  * @brief  Timer cpture/compare 4 overcapture interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch4ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH4OV);
}

/**
  * @brief  Timer cpture/compare 3 overcapture interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch3ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH3OV);
}

/**
  * @brief  Timer cpture/compare 2 overcapture interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch2ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH2OV);
}

/**
  * @brief  Timer cpture/compare 1 overcapture interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch1ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH1OV);
}

/**
  * @brief  TIMER BRK2 interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_it_brk2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_BRK2);
}

/**
  * @brief  Timer break interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_brk(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_BRK);
}

/**
  * @brief  Timer trigger interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_trgi(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_TRGI);
}

/**
  * @brief  Timer COM interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_com(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_COM);
}

/**
  * @brief  Timer capture/compare 4 interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch4(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH4);
}

/**
  * @brief  Timer capture/compare 3 interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch3(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH3);
}

/**
  * @brief  Timer capture/compare 2 interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH2);
}

/**
  * @brief  Timer capture/compare 1 interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_ch1(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_CH1);
}

/**
  * @brief  Timer update interrupt enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_it_upd(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IER, TIMER_IER_UPD);
}
/**
  * @}
  */

/** @defgroup IDR
  * @{
  */
/**
  * @brief  Timer IDR setup.
  * @param  TIMERx TIMER instance
  * @param  value (CH4OV | CH3OV | CH2OV | CH1OV | BRK | TRGI | COM | CH4 | CH3 | CH2 | CH1 | UPD)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_idr(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->IDR, value);
}

/**
  * @brief  TIMER TERR interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_it_conversion_error(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_TERR);
}

/**
  * @brief  TIMER IERR interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_it_index_error(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_IERR);
}

/**
  * @brief  TIMER DIR interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_it_direction_change(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_DIR);
}

/**
  * @brief  TIMER IDX interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_it_effective_index(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_IDX);
}

/**
  * @brief  TIMER CH6 interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_it_ch6(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH6);
}

/**
  * @brief  TIMER CH5 interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_it_ch5(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH5);
}

/**
  * @brief  TIMER SBRK interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_it_system_brake(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_SBRK);
}

/**
  * @brief  Timer cpture/compare 4 overcapture interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch4ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH4OV);
}

/**
  * @brief  Timer cpture/compare 3 overcapture interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch3ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH3OV);
}

/**
  * @brief  Timer cpture/compare 2 overcapture interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch2ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH2OV);
}

/**
  * @brief  Timer cpture/compare 1 overcapture interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch1ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH1OV);
}

/**
  * @brief  TIMER BRK2 interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_it_brk2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_BRK2);
}

/**
  * @brief  Timer break interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_brk(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_BRK);
}

/**
  * @brief  Timer trigger interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_trgi(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_TRGI);
}

/**
  * @brief  Timer COM interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_com(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_COM);
}

/**
  * @brief  Timer capture/compare 4 interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch4(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH4);
}

/**
  * @brief  Timer capture/compare 3 interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch3(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH3);
}

/**
  * @brief  Timer capture/compare 2 interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH2);
}

/**
  * @brief  Timer capture/compare 1 interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_ch1(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_CH1);
}

/**
  * @brief  Timer update interrupt disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_it_upd(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->IDR, TIMER_IDR_UPD);
}
/**
  * @}
  */

/** @defgroup IVS
  * @{
  */
/**
  * @brief  Get timer IVS setup.
  * @param  TIMERx TIMER instance
  * @retval Timer IVS setup.
  */

__STATIC_INLINE uint32_t md_timer_get_ivs(TIMER_TypeDef *TIMERx)
{
    return (READ_REG(TIMERx->IVS));
}

/**
  * @brief  TIMER TERR is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_it_conversion_error(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_TERR) == (TIMER_IVS_TERR));
}

/**
  * @brief  TIMER IERR is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_it_index_error(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_IERR) == (TIMER_IVS_IERR));
}

/**
  * @brief  TIMER DIR is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_it_direction_change(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_DIR) == (TIMER_IVS_DIR));
}

/**
  * @brief  TIMER IDX is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_it_effective_index(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_IDX) == (TIMER_IVS_IDX));
}

/**
  * @brief  TIMER CH6 is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch6(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH6) == (TIMER_IVS_CH6));
}

/**
  * @brief  TIMER CH5 is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch5(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH5) == (TIMER_IVS_CH5));
}

/**
  * @brief  TIMER SBRK is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_it_system_brake(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_SBRK) == (TIMER_IVS_SBRK));
}

/**
  * @brief  Indicates whether the timer capture/compare 4 overcapture interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch4ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH4OV) == (TIMER_IVS_CH4OV));
}

/**
  * @brief  Indicates whether the timer capture/compare 3 overcapture interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch3ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH3OV) == (TIMER_IVS_CH3OV));
}

/**
  * @brief  Indicates whether the timer cpture/compare 2 overcapture interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch2ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH2OV) == (TIMER_IVS_CH2OV));
}

/**
  * @brief  Indicates whether the timer capture/compare 1 overcapture interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch1ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH1OV) == (TIMER_IVS_CH1OV));
}

/**
  * @brief  TIMER BRK2 is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_it_brk2(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_BRK2) == (TIMER_IVS_BRK2));
}

/**
  * @brief  Indicates whether the timer break interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_brk(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_BRK) == (TIMER_IVS_BRK));
}

/**
  * @brief  Indicates whether the timer trigger interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_trgi(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_TRGI) == (TIMER_IVS_TRGI));
}

/**
  * @brief  Indicates whether the timer COM interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_com(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_COM) == (TIMER_IVS_COM));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 4 interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch4(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH4) == (TIMER_IVS_CH4));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 3 interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch3(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH3) == (TIMER_IVS_CH3));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch2(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH2) == (TIMER_IVS_CH2));
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_ch1(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_CH1) == (TIMER_IVS_CH1));
}

/**
  * @brief  Indicates whether the timer update interrupt is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_it_upd(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IVS, TIMER_IVS_UPD) == (TIMER_IVS_UPD));
}
/**
  * @}
  */

/** @defgroup RIF
  * @{
  */
/**
  * @brief  Get timer RIF flag.
  * @param  TIMERx TIMER instance
  * @retval Timer RIF flag.
  */

__STATIC_INLINE uint32_t md_timer_get_rif(TIMER_TypeDef *TIMERx)
{
    return (READ_REG(TIMERx->RIF));
}

/**
  * @brief  Get TIMER TERR
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_conversion_error(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_TERR) == (TIMER_RIF_TERR));
}

/**
  * @brief  Get TIMER IERR
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_index_error(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_IERR) == (TIMER_RIF_IERR));
}

/**
  * @brief  Get TIMER DIR
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_direction_change(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_DIR) == (TIMER_RIF_DIR));
}

/**
  * @brief  Get TIMER IDX
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_effective_index(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_IDX) == (TIMER_RIF_IDX));
}

/**
  * @brief  Get TIMER CH6
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch6(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH6) == (TIMER_RIF_CH6));
}

/**
  * @brief  Get TIMER CH5
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch5(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH5) == (TIMER_RIF_CH5));
}

/**
  * @brief  Get TIMER SBRK
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_system_brake(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_SBRK) == (TIMER_RIF_SBRK));
}

/**
  * @brief  Get timer capture/compare 4 overcapture interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch4ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH4OV) == (TIMER_RIF_CH4OV));
}

/**
  * @brief  Get timer capture/compare 3 overcapture interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch3ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH3OV) == (TIMER_RIF_CH3OV));
}

/**
  * @brief  Get timer capture/compare 2 overcapture interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch2ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH2OV) == (TIMER_RIF_CH2OV));
}

/**
  * @brief  Get timer capture/compare 1 overcapture interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch1ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH1OV) == (TIMER_RIF_CH1OV));
}

/**
  * @brief  Get TIMER BRK2
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_brk2(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_BRK2) == (TIMER_RIF_BRK2));
}

/**
  * @brief  Get timer break interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_brk(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_BRK) == (TIMER_RIF_BRK));
}

/**
  * @brief  Get timer trigger interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_trgi(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_TRGI) == (TIMER_RIF_TRGI));
}

/**
  * @brief  Get timer COM interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_com(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_COM) == (TIMER_RIF_COM));
}

/**
  * @brief  Get timer capture/compare 4 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch4(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH4) == (TIMER_RIF_CH4));
}

/**
  * @brief  Get timer capture/compare 3 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch3(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH3) == (TIMER_RIF_CH3));
}

/**
  * @brief  Get timer capture/compare 2 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch2(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH2) == (TIMER_RIF_CH2));
}

/**
  * @brief  Get timer capture/compare 1 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_ch1(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_CH1) == (TIMER_RIF_CH1));
}

/**
  * @brief  Get timer update interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_it_upd(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->RIF, TIMER_RIF_UPD) == (TIMER_RIF_UPD));
}
/**
  * @}
  */

/** @defgroup IFM
  * @{
  */
/**
  * @brief  Get timer IFM flag.
  * @param  TIMERx TIMER instance
  * @retval Timer IFM flag.
  */

__STATIC_INLINE uint32_t md_timer_get_ifm(TIMER_TypeDef *TIMERx)
{
    return (READ_REG(TIMERx->IFM));
}

/**
  * @brief  Get TIMER TERR
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_conversion_error(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_TERR) == (TIMER_IFM_TERR));
}

/**
  * @brief  Get TIMER IERR
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_index_error(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_IERR) == (TIMER_IFM_IERR));
}

/**
  * @brief  Get TIMER DIR
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_direction_change(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_DIR) == (TIMER_IFM_DIR));
}

/**
  * @brief  Get TIMER IDX
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_effective_index(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_IDX) == (TIMER_IFM_IDX));
}

/**
  * @brief  Get TIMER CH6
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch6(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH6) == (TIMER_IFM_CH6));
}

/**
  * @brief  Get TIMER CH5
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch5(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH5) == (TIMER_IFM_CH5));
}

/**
  * @brief  Get TIMER SBRK
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_system_brake(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_SBRK) == (TIMER_IFM_SBRK));
}

/**
  * @brief  Get timer capture/compare 4 overcapture interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch4ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH4OV) == (TIMER_IFM_CH4OV));
}

/**
  * @brief  Get timer capture/compare 3 overcapture interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch3ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH3OV) == (TIMER_IFM_CH3OV));
}

/**
  * @brief  Get timer capture/compare 2 overcapture interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch2ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH2OV) == (TIMER_IFM_CH2OV));
}

/**
  * @brief  Get timer capture/compare 1 overcapture interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch1ov(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH1OV) == (TIMER_IFM_CH1OV));
}

/**
  * @brief  Get TIMER BRK2
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_brk2(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_BRK2) == (TIMER_IFM_BRK2));
}

/**
  * @brief  Get timer break interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_brk(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_BRK) == (TIMER_IFM_BRK));
}

/**
  * @brief  Get timer trigger interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_trgi(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_TRGI) == (TIMER_IFM_TRGI));
}

/**
  * @brief  Get timer COM interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_com(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_COM) == (TIMER_IFM_COM));
}

/**
  * @brief  Get timer capture/compare 4 interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch4(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH4) == (TIMER_IFM_CH4));
}

/**
  * @brief  Get timer capture/compare 3 interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch3(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH3) == (TIMER_IFM_CH3));
}

/**
  * @brief  Get timer capture/compare 2 interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch2(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH2) == (TIMER_IFM_CH2));
}

/**
  * @brief  Get timer capture/compare 1 interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_ch1(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_CH1) == (TIMER_IFM_CH1));
}

/**
  * @brief  Get timer update interrupt flag masked.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_masked_it_upd(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->IFM, TIMER_IFM_UPD) == (TIMER_IFM_UPD));
}
/**
  * @}
  */

/** @defgroup ICR
  * @{
  */
/**
  * @brief  Timer ICR setup.
  * @param  TIMERx TIMER instance
  * @param  value (CH4OV | CH3OV | CH2OV | CH1OV | BRK | TRGI | COM | CH4 | CH3 | CH2 | CH1 | UPD)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_icr(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->ICR, value);
}

/**
  * @brief  Clear TIMER TERR interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_clear_it_conversion_error(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_TERR);
}

/**
  * @brief  Clear TIMER IERR interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_clear_it_index_error(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_IERR);
}

/**
  * @brief  Clear TIMER DIR interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_clear_it_direction_change(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_DIR);
}

/**
  * @brief  Clear TIMER IDX interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_clear_it_effective_index(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_IDX);
}

/**
  * @brief  Clear TIMER CH6 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_clear_it_ch6(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH6);
}

/**
  * @brief  Clear TIMER CH5 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_clear_it_ch5(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH5);
}

/**
  * @brief  Clear TIMER SBRK interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_clear_it_system_brake(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_SBRK);
}

/**
  * @brief  Clear timer cpture/compare 4 overcapture interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch4ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH4OV);
}

/**
  * @brief  Clear timer cpture/compare 3 overcapture interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch3ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH3OV);
}

/**
  * @brief  Clear timer cpture/compare 2 overcapture interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch2ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH2OV);
}

/**
  * @brief  Clear timer cpture/compare 1 overcapture interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch1ov(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH1OV);
}

/**
  * @brief  Clear TIMER BRK2 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_clear_it_brk2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_BRK2);
}

/**
  * @brief  Clear timer break interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_brk(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_BRK);
}

/**
  * @brief  Clear timer trigger interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_trgi(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_TRGI);
}

/**
  * @brief  Clear timer COM interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_com(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_COM);
}

/**
  * @brief  Clear timer capture/compare 4 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch4(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH4);
}

/**
  * @brief  Clear timer capture/compare 3 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch3(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH3);
}

/**
  * @brief  Clear timer capture/compare 2 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH2);
}

/**
  * @brief  Clear timer capture/compare 1 interrupt flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_ch1(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_CH1);
}

/**
  * @brief  Clear timer upadte flag.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_clear_it_upd(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ICR, TIMER_ICR_UPD);
}
/**
  * @}
  */

/** @defgroup SGE
  * @{
  */
/**
  * @brief  Timer SGE setup.
  * @param  TIMERx TIMER instance
  * @param  value (SGBRK | SGTRGI | SGCOM | SGCH4 | SGCH3 | SGCH2 | SGCH1 | SGUPD)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_sge(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->SGE, value);
}

/**
  * @brief  Timer break2 generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgbrk2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGBRK2);
}

/**
  * @brief  Timer break generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgbrk(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGBRK);
}

/**
  * @brief  Timer trigger generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgtrgi(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGTRGI);
}

/**
  * @brief  Timer COM generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgcom(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGCOM);
}

/**
  * @brief  Timer Capture/Compare 4 generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgch4(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGCH4);
}

/**
  * @brief  Timer Capture/Compare 3 generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgch3(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGCH3);
}

/**
  * @brief  Timer Capture/Compare 2 generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgch2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGCH2);
}

/**
  * @brief  Timer Capture/Compare 1 generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgch1(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGCH1);
}

/**
  * @brief  Timer update generation.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_trigger_sgupd(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->SGE, TIMER_SGE_SGUPD);
}
/**
  * @}
  */

/** @defgroup CHMR1
  * @{
  */
/**
  * @brief  Timer CHMR1 setup.
  * @param  TIMERx TIMER instance
  * @param  value output (CH2OCLREN | CH2MOD | CH2PEN | CH2FEN | CC2SSEL | CH1OCLREN | CH1MOD | CH1PEN | CH1FEN | CC1SSEL)
  *               input  (I2FLT | I2PRES | CC2SSEL | I1FLT | I1PRES | CC1SSEL)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr1(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CHMR1, value);
}

/**
  * @brief  GET Timer CHMR1 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CHMR1 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr1(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CHMR1));
}

/**
  * @brief  Timer output compare 2 clear enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare2_clear_ch2oclren(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2OCLREN);
}

/**
  * @brief  Timer output compare 2 clear disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare2_clear_ch2oclren(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2OCLREN);
}

/**
  * @brief  Indicates whether the timer output compare 2 clear is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare2_clear_ch2oclren(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2OCLREN) == (TIMER_CHMR1_OUTPUT_CH2OCLREN));
}

/**
  * @brief  Timer output compare 2 mode setup.
  * @param  TIMERx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare2_mode_ch2mod(TIMER_TypeDef *TIMERx, md_timer_outputmode_list_t OutputMode)
{
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2MOD, (OutputMode & 0x7) << TIMER_CHMR1_OUTPUT_CH2MOD_POSS);
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2MOD2, ((OutputMode & 0x18UL) >> 3) << TIMER_CHMR1_OUTPUT_CH2MOD2_POSS);
}

/**
  * @brief  Get timer output compare 2 mode.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  */

__STATIC_INLINE md_timer_outputmode_list_t md_timer_get_output_compare2_mode_ch2mod(TIMER_TypeDef *TIMERx)
{
    return (md_timer_outputmode_list_t)(((READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2MOD2) >> TIMER_CHMR1_OUTPUT_CH2MOD_POSS) << 3UL) | (READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2MOD) >> TIMER_CHMR1_OUTPUT_CH2MOD_POSS));
}

/**
  * @brief  Timer output compare 2 preload enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare2_preload_ch2pen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2PEN);
}

/**
  * @brief  Timer output compare 2 preload disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare2_preload_ch2pen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2PEN);
}

/**
  * @brief  Indicates whether the timer output compare 2 preload is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare2_preload_ch2pen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2PEN) == (TIMER_CHMR1_OUTPUT_CH2PEN));
}

/**
  * @brief  Timer output compare 2 fast enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare2_fast_ch2fen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2FEN);
}

/**
  * @brief  Timer output compare 2 fast disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare2_fast_ch2fen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2FEN);
}

/**
  * @brief  Indicates whether the timer output compare 2 fast is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare2_fast_ch2fen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH2FEN) == (TIMER_CHMR1_OUTPUT_CH2FEN));
}

/**
  * @brief  Timer cpture/compare 2 selection setup.
  * @param  TIMERx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc2_func_cc2ssel(TIMER_TypeDef *TIMERx, md_timer_chmode_t ChannelMode)
{
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CC2SSEL, (ChannelMode << TIMER_CHMR1_OUTPUT_CC2SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 2 selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  */

__STATIC_INLINE md_timer_chmode_t md_timer_get_cc2_func_cc2ssel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_chmode_t)(READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CC2SSEL) >> TIMER_CHMR1_OUTPUT_CC2SSEL_POSS);
}


/**
  * @brief  Timer output compare 1 clear enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare1_clear_ch1oclren(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1OCLREN);
}

/**
  * @brief  Timer output compare 1 clear disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare1_clear_ch1oclren(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1OCLREN);
}

/**
  * @brief  Indicates whether the timer output compare 1 clear is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare1_clear_ch1oclren(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1OCLREN) == (TIMER_CHMR1_OUTPUT_CH1OCLREN));
}

/**
  * @brief  Timer output compare 1 mode setup.
  * @param  TIMERx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare1_mode_ch1mod(TIMER_TypeDef *TIMERx, md_timer_outputmode_list_t OutputMode)
{
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1MOD, (OutputMode & 0x7UL) << TIMER_CHMR1_OUTPUT_CH1MOD_POSS);
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1MOD2, ((OutputMode & 0x18UL) >> 3) << TIMER_CHMR1_OUTPUT_CH1MOD2_POSS);
}

/**
  * @brief  Get timer output compare 1 mode.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  */

__STATIC_INLINE md_timer_outputmode_list_t md_timer_get_output_compare1_mode_ch1mod(TIMER_TypeDef *TIMERx)
{
    return (md_timer_outputmode_list_t)(((READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1MOD2) >> TIMER_CHMR1_OUTPUT_CH1MOD2_POSS) << 3UL) | (READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1MOD) >> TIMER_CHMR1_OUTPUT_CH1MOD_POSS));
}

/**
  * @brief  Timer output compare 1 preload enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare1_preload_ch1pen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1PEN);
}

/**
  * @brief  Timer output compare 1 preload disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare1_preload_ch1pen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1PEN);
}

/**
  * @brief  Indicates whether the timer output compare 1 preload is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare1_preload_ch1pen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1PEN) == (TIMER_CHMR1_OUTPUT_CH1PEN));
}

/**
  * @brief  Timer output compare 1 fast enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare1_fast_ch1fen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1FEN);
}

/**
  * @brief  Timer output compare 1 fast disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare1_fast_ch1fen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1FEN);
}

/**
  * @brief  Indicates whether the timer output compare 1 fast is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare1_fast_ch1fen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CH1FEN) == (TIMER_CHMR1_OUTPUT_CH1FEN));
}

/**
  * @brief  Timer cpture/compare 1 selection setup.
  * @param  TIMERx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc1_func_cc1ssel(TIMER_TypeDef *TIMERx, md_timer_chmode_t sel)
{
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CC1SSEL, sel << TIMER_CHMR1_OUTPUT_CC1SSEL_POSS);
}

/**
  * @brief  Get timer cpture/compare 1 selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  */

__STATIC_INLINE md_timer_chmode_t md_timer_get_cc1_func_cc1ssel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_chmode_t)(READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_OUTPUT_CC1SSEL) >> TIMER_CHMR1_OUTPUT_CC1SSEL_POSS);
}

/**
  * @brief  Timer input capture 2 filter setup.
  * @param  TIMERx TIMER instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture2_filter_i2flt(TIMER_TypeDef *TIMERx, md_timer_inputfilter_list_t InputFliter)
{
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_INPUT_I2FLT, InputFliter << TIMER_CHMR1_INPUT_I2FLT_POSS);
}

/**
  * @brief  Get timer input capture 2 filter.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  */

__STATIC_INLINE md_timer_inputfilter_list_t md_timer_get_input_capture2_filter_i2flt(TIMER_TypeDef *TIMERx)
{
    return (md_timer_inputfilter_list_t)(READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_INPUT_I2FLT) >> TIMER_CHMR1_INPUT_I2FLT_POSS);
}

/**
  * @brief  Timer input capture 2 prescaler setup.
  * @param  TIMERx TIMER instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture2_prescaler_ic2pres(TIMER_TypeDef *TIMERx, md_timer_inputprescale_t InputPrescale)
{
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_INPUT_I2PRES, InputPrescale << TIMER_CHMR1_INPUT_I2PRES_POSS);
}

/**
  * @brief  Get timer input capture 2 prescaler.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  */

__STATIC_INLINE md_timer_inputprescale_t md_timer_get_input_capture2_prescaler_ic2pres(TIMER_TypeDef *TIMERx)
{
    return (md_timer_inputprescale_t)(READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_INPUT_I2PRES) >> TIMER_CHMR1_INPUT_I2PRES_POSS);
}

/**
  * @brief  Timer input capture 1 filter setup.
  * @param  TIMERx TIMER instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture1_filter_i1flt(TIMER_TypeDef *TIMERx, md_timer_inputfilter_list_t InputFliter)
{
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_INPUT_I1FLT, InputFliter << TIMER_CHMR1_INPUT_I1FLT_POSS);
}

/**
  * @brief  Get timer input capture 1 filter.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  */

__STATIC_INLINE md_timer_inputfilter_list_t md_timer_get_input_capture1_filter_i1flt(TIMER_TypeDef *TIMERx)
{
    return (md_timer_inputfilter_list_t)(READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_INPUT_I1FLT) >> TIMER_CHMR1_INPUT_I1FLT_POSS);
}

/**
  * @brief  Timer input capture 1 prescaler setup.
  * @param  TIMERx TIMER instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture1_prescaler_ic1pres(TIMER_TypeDef *TIMERx, md_timer_inputprescale_t InputPrescale)
{
    MODIFY_REG(TIMERx->CHMR1, TIMER_CHMR1_INPUT_I1PRES, InputPrescale << TIMER_CHMR1_INPUT_I1PRES_POSS);
}

/**
  * @brief  Get timer input capture 1 prescaler.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  */

__STATIC_INLINE md_timer_inputprescale_t md_timer_get_input_capture1_prescaler_ic1pres(TIMER_TypeDef *TIMERx)
{
    return (md_timer_inputprescale_t)(READ_BIT(TIMERx->CHMR1, TIMER_CHMR1_INPUT_I1PRES) >> TIMER_CHMR1_INPUT_I1PRES_POSS);
}
/**
  * @}
  */

/** @defgroup CHMR2
  * @{
  */
/**
  * @brief  Timer CHMR2 setup.

  * @param  TIMERx TIMER instance
  * @param  value output (CH4OCLREN | CH4MOD | CH4PEN | CH4FEN | CC4SSEL | CH3OCLREN | CH3MOD | CH3PEN | CH3FEN | CC3SSEL)
  *               input  (I4FLT | I4PRES | CC4SSEL | I3FLT | I3PRES | CC3SSEL)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr2(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CHMR2, value);
}

/**
  * @brief  GET Timer CHMR2 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CHMR2 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr2(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CHMR2));
}

/**
  * @brief  Timer output compare 4 clear enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare4_clear_ch4oclren(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4OCLREN);
}

/**
  * @brief  Timer output compare 4 clear disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare4_clear_ch4oclren(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4OCLREN);
}

/**
  * @brief  Indicates whether the timer output compare 4 clear is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare4_clear_ch4oclren(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4OCLREN) == (TIMER_CHMR2_OUTPUT_CH4OCLREN));
}

/**
  * @brief  Timer output compare 4 mode setup.
  * @param  TIMERx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMPARE_PULSE_MODE
            @arg @ref MD_TIMER_OUTPUTMODE_DIRECTION_OUTPUT_MODE
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare4_mode_ch4mod(TIMER_TypeDef *TIMERx, md_timer_outputmode_list_t OutputMode)
{
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4MOD, (OutputMode & 0x7UL) << TIMER_CHMR2_OUTPUT_CH4MOD_POSS);
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4MOD2, ((OutputMode & 0x18UL) >> 3) << TIMER_CHMR2_OUTPUT_CH4MOD2_POSS);
}

/**
  * @brief  Get timer output compare 4 mode.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMPARE_PULSE_MODE
            @arg @ref MD_TIMER_OUTPUTMODE_DIRECTION_OUTPUT_MODE
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  */

__STATIC_INLINE md_timer_outputmode_list_t md_timer_get_output_compare4_mode_ch4mod(TIMER_TypeDef *TIMERx)
{
    return (md_timer_outputmode_list_t)(((READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4MOD2) >> TIMER_CHMR2_OUTPUT_CH4MOD2_POSS) << 3UL) | (READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4MOD) >> TIMER_CHMR2_OUTPUT_CH4MOD_POSS));
}

/**
  * @brief  Timer output compare 4 preload enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare4_preload_ch4pen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4PEN);
}

/**
  * @brief  Timer output compare 4 preload disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare4_preload_ch4pen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4PEN);
}

/**
  * @brief  Indicates whether the timer output compare 4 preload is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare4_preload_ch4pen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4PEN) == (TIMER_CHMR2_OUTPUT_CH4PEN));
}

/**
  * @brief  Timer output compare 4 fast enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare4_fast_ch4fen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4FEN);
}

/**
  * @brief  Timer output compare 4 fast disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare4_fast_ch4fen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4FEN);
}

/**
  * @brief  Indicates whether the timer output compare 4 fast is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare4_fast_ch4fen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH4FEN) == (TIMER_CHMR2_OUTPUT_CH4FEN));
}

/**
  * @brief  Timer cpture/compare 4 selection setup.
  * @param  TIMERx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc4_func_cc4ssel(TIMER_TypeDef *TIMERx, md_timer_chmode_t ChannelMode)
{
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CC4SSEL, (ChannelMode << TIMER_CHMR2_OUTPUT_CC4SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 4 selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  */

__STATIC_INLINE md_timer_chmode_t md_timer_get_cc4_func_cc4ssel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_chmode_t)(READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CC4SSEL) >> TIMER_CHMR2_OUTPUT_CC4SSEL_POSS);
}

/**
  * @brief  Timer output compare 3 clear enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare3_clear_ch3oclren(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3OCLREN);
}

/**
  * @brief  Timer output compare 3 clear disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare3_clear_ch3oclren(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3OCLREN);
}

/**
  * @brief  Indicates whether the timer output compare 3 clear is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare3_clear_ch3oclren(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3OCLREN) == (TIMER_CHMR2_OUTPUT_CH3OCLREN));
}

/**
  * @brief  Timer output compare 3 mode setup.
  * @param  TIMERx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMPARE_PULSE_MODE
            @arg @ref MD_TIMER_OUTPUTMODE_DIRECTION_OUTPUT_MODE
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare3_mode_ch3mod(TIMER_TypeDef *TIMERx, md_timer_outputmode_list_t OutputMode)
{
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3MOD, (OutputMode & 0x7UL) << TIMER_CHMR2_OUTPUT_CH3MOD_POSS);
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3MOD2, ((OutputMode & 0x18UL) >> 3) << TIMER_CHMR2_OUTPUT_CH3MOD2_POSS);
}

/**
  * @brief  Get timer output compare 3 mode.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMPARE_PULSE_MODE
            @arg @ref MD_TIMER_OUTPUTMODE_DIRECTION_OUTPUT_MODE
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  */

__STATIC_INLINE md_timer_outputmode_list_t md_timer_get_output_compare3_mode_ch3mod(TIMER_TypeDef *TIMERx)
{
    return (md_timer_outputmode_list_t)(((READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3MOD2) >> TIMER_CHMR2_OUTPUT_CH3MOD2_POSS) << 3UL) | (READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3MOD) >> TIMER_CHMR2_OUTPUT_CH3MOD_POSS));
}

/**
  * @brief  Timer output compare 3 preload enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare3_preload_ch3pen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3PEN);
}

/**
  * @brief  Timer output compare 3 preload disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare3_preload_ch3pen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3PEN);
}

/**
  * @brief  Indicates whether the timer output compare 3 preload is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare3_preload_ch3pen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3PEN) == (TIMER_CHMR2_OUTPUT_CH3PEN));
}

/**
  * @brief  Timer output compare 3 fast enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare3_fast_ch3fen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3FEN);
}

/**
  * @brief  Timer output compare 3 fast disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare3_fast_ch3fen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3FEN);
}

/**
  * @brief  Indicates whether the timer output compare 3 fast is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare3_fast_ch3fen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CH3FEN) == (TIMER_CHMR2_OUTPUT_CH3FEN));
}

/**
  * @brief  Timer cpture/compare 3 selection setup.
  * @param  TIMERx TIMER instance
  * @param  ChannelMode Channel mode selection
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc3_func_cc3ssel(TIMER_TypeDef *TIMERx, md_timer_chmode_t ChannelMode)
{
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CC3SSEL, (ChannelMode << TIMER_CHMR2_OUTPUT_CC3SSEL_POSS));
}

/**
  * @brief  Get timer cpture/compare 3 selection.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_CHMODE_OUTPUT
            @arg @ref MD_TIMER_CHMODE_INPUT_DIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_INDIRECT
            @arg @ref MD_TIMER_CHMODE_INPUT_TRC
  */

__STATIC_INLINE md_timer_chmode_t md_timer_get_cc3_func_cc3ssel(TIMER_TypeDef *TIMERx)
{
    return (md_timer_chmode_t)(READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_OUTPUT_CC3SSEL) >> TIMER_CHMR2_OUTPUT_CC3SSEL_POSS);
}

/**
  * @brief  Timer input capture 4 filter setup.
  * @param  TIMERx TIMER instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture4_filter_i4flt(TIMER_TypeDef *TIMERx, md_timer_inputfilter_list_t InputFliter)
{
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_INPUT_I4FLT, InputFliter << TIMER_CHMR2_INPUT_I4FLT_POSS);
}

/**
  * @brief  Get timer input capture 4 filter.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  */

__STATIC_INLINE md_timer_inputfilter_list_t md_timer_get_input_capture4_filter_i4flt(TIMER_TypeDef *TIMERx)
{
    return (md_timer_inputfilter_list_t)(READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_INPUT_I4FLT) >> TIMER_CHMR2_INPUT_I4FLT_POSS);
}

/**
  * @brief  Timer input capture 4 prescaler setup.
  * @param  TIMERx TIMER instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture4_prescaler_ic4pres(TIMER_TypeDef *TIMERx, md_timer_inputprescale_t InputPrescale)
{
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_INPUT_I4PRES, InputPrescale << TIMER_CHMR2_INPUT_I4PRES_POSS);
}

/**
  * @brief  Get timer input capture 4 prescaler.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  */

__STATIC_INLINE md_timer_inputprescale_t md_timer_get_input_capture4_prescaler_ic4pres(TIMER_TypeDef *TIMERx)
{
    return (md_timer_inputprescale_t)(READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_INPUT_I4PRES) >> TIMER_CHMR2_INPUT_I4PRES_POSS);
}


/**
  * @brief  Timer input capture 3 filter setup.
  * @param  TIMERx TIMER instance
  * @param  InputFliter Input capture filter
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture3_filter_i3flt(TIMER_TypeDef *TIMERx, md_timer_inputfilter_list_t InputFliter)
{
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_INPUT_I3FLT, InputFliter << TIMER_CHMR2_INPUT_I3FLT_POSS);
}

/**
  * @brief  Get timer input capture 3 filter.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N2
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N4
            @arg @ref MD_TIMER_INPUTFILTER_FDIV1N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV2N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV4N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV8N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV16N8
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N5
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N6
            @arg @ref MD_TIMER_INPUTFILTER_FDIV32N8
  */

__STATIC_INLINE md_timer_inputfilter_list_t md_timer_get_input_capture3_filter_i3flt(TIMER_TypeDef *TIMERx)
{
    return (md_timer_inputfilter_list_t)(READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_INPUT_I3FLT) >> TIMER_CHMR2_INPUT_I3FLT_POSS);
}

/**
  * @brief  Timer input capture 3 prescaler setup.
  * @param  TIMERx TIMER instance
  * @param  InputPrescale Input capture prescaler
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  * @retval None
  */

__STATIC_INLINE void md_timer_set_input_capture3_prescaler_ic3pres(TIMER_TypeDef *TIMERx, md_timer_inputprescale_t InputPrescale)
{
    MODIFY_REG(TIMERx->CHMR2, TIMER_CHMR2_INPUT_I3PRES, InputPrescale << TIMER_CHMR2_INPUT_I3PRES_POSS);
}

/**
  * @brief  Get timer input capture 3 prescaler.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV1
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV2
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV4
            @arg @ref MD_TIMER_INPUTPRESCALE_DIV8
  */

__STATIC_INLINE md_timer_inputprescale_t md_timer_get_input_capture3_prescaler_ic3pres(TIMER_TypeDef *TIMERx)
{
    return (md_timer_inputprescale_t)(READ_BIT(TIMERx->CHMR2, TIMER_CHMR2_INPUT_I3PRES) >> TIMER_CHMR2_INPUT_I3PRES_POSS);
}
/**
  * @}
  */

/** @defgroup CCEP
  * @{
  */
/**
  * @brief  Timer CCEP setup.
  * @param  TIMERx TIMER instance
  * @param  value (CC4NPOL | CC4POL | CC4EN | CC3NPOL | CC3NEN | CC3POL | CC3EN | CC2NPOL |
  *                CC2NEN | CC2POL | CC2EN | CC1NPOL | CC1NEN | CC1POL | CC1EN )
  * @retval None
  */

__STATIC_INLINE void md_timer_set_ccep(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CCEP, value);
}

/**
  * @brief  GET Timer CCEP register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CCEP register value.
  */

__STATIC_INLINE uint32_t md_timer_get_ccep(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CCEP));
}

/**
  * @brief  Timer capture/compare 6 output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc6_output_polarity_cc6pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC6POL, (OutputPolarity << TIMER_CCEP_CC6POL_POS));
}

__STATIC_INLINE void md_timer_set_cc6_input_edge_cc6pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC6POL, (OutputPolarity << TIMER_CCEP_CC6POL_POS));
}

/**
  * @brief  Get timer capture/compare 6 output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 6 output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc6_output_polarity_cc6pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC6POL));
}

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc6_input_edge_cc6pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC6POL));
}

/**
  * @brief  Timer Capture/Compare 6 output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc6_output_cc6en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC6EN);
}

__STATIC_INLINE void md_timer_enable_cc6_input_cc6en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC6EN);
}

/**
  * @brief  Timer Capture/Compare 6 output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc6_output_cc6en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC6EN);
}

__STATIC_INLINE void md_timer_disable_cc6_input_cc6en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC6EN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 6 output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc6_output_cc6en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC6EN) == (TIMER_CCEP_CC6EN));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc6_input_cc6en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC6EN) == (TIMER_CCEP_CC6EN));
}

/**
  * @brief  Timer capture/compare 5 output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc5_output_polarity_cc5pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC5POL, (OutputPolarity << TIMER_CCEP_CC5POL_POS));
}

__STATIC_INLINE void md_timer_set_cc5_input_edge_cc5pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC5POL, (OutputPolarity << TIMER_CCEP_CC5POL_POS));
}

/**
  * @brief  Get timer capture/compare 5 output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 5 output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc5_output_polarity_cc5pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC5POL));
}

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc5_input_edge_cc5pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC5POL));
}

/**
  * @brief  Timer Capture/Compare 5 output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc5_output_cc5en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC5EN);
}

__STATIC_INLINE void md_timer_enable_cc5_input_cc5en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC5EN);
}

/**
  * @brief  Timer Capture/Compare 5 output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc5_output_cc5en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC5EN);
}

__STATIC_INLINE void md_timer_disable_cc5_input_cc5en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC5EN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 5 output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc5_output_cc5en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC5EN) == (TIMER_CCEP_CC5EN));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc5_input_cc5en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC5EN) == (TIMER_CCEP_CC5EN));
}

/**
  * @brief  Timer capture/compare 4 complementary output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc4_complementary_polarity_cc4npol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC4NPOL, (OutputPolarity << TIMER_CCEP_CC4NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 4 complementary output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 4 complementary output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc4_complementary_polarity_cc4npol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC4NPOL));
}

/**
  * @brief  Timer Capture/Compare 4 output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc4_output_cc4nen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC4NEN);
}

__STATIC_INLINE void md_timer_enable_cc4_input_cc4nen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC4NEN);
}

/**
  * @brief  Timer Capture/Compare 4 output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc4_output_cc4nen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC4NEN);
}

__STATIC_INLINE void md_timer_disable_cc4_input_cc4nen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC4NEN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 4 output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc4_output_cc4nen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC4NEN) == (TIMER_CCEP_CC4NEN));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc4_input_cc4nen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC4NEN) == (TIMER_CCEP_CC4NEN));
}


/**
  * @brief  Timer capture/compare 4 output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc4_output_polarity_cc4pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC4POL, (OutputPolarity << TIMER_CCEP_CC4POL_POS));
}

__STATIC_INLINE void md_timer_set_cc4_input_edge_cc4pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC4POL, (OutputPolarity << TIMER_CCEP_CC4POL_POS));
}

/**
  * @brief  Get timer capture/compare 4 output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 4 output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc4_output_polarity_cc4pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC4POL));
}

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc4_input_edge_cc4pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC4POL));
}

/**
  * @brief  Timer Capture/Compare 4 output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc4_output_cc4en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC4EN);
}

__STATIC_INLINE void md_timer_enable_cc4_input_cc4en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC4EN);
}

/**
  * @brief  Timer Capture/Compare 4 output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc4_output_cc4en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC4EN);
}

__STATIC_INLINE void md_timer_disable_cc4_input_cc4en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC4EN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 4 output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc4_output_cc4en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC4EN) == (TIMER_CCEP_CC4EN));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc4_input_cc4en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC4EN) == (TIMER_CCEP_CC4EN));
}

/**
  * @brief  Timer capture/compare 3 complementary output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc3_complementary_polarity_cc3npol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC3NPOL, OutputPolarity << TIMER_CCEP_CC3NPOL_POS);
}

/**
  * @brief  Get timer capture/compare 3 complementary output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 3 complementary output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc3_complementary_polarity_cc3npol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC3NPOL) >> TIMER_CCEP_CC3NPOL_POS);
}

/**
  * @brief  Timer Capture/Compare 3 complementary output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc3_complementary_cc3nen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC3NEN);
}

/**
  * @brief  Timer Capture/Compare 3 complementary output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc3_complementary_cc3nen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC3NEN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 3 complementary output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc3_complementary_cc3nen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC3NEN) == (TIMER_CCEP_CC3NEN));
}

/**
  * @brief  Timer capture/compare 3 output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc3_output_polarity_cc3pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC3POL, (OutputPolarity << TIMER_CCEP_CC3POL_POS));
}

__STATIC_INLINE void md_timer_set_cc3_input_edge_cc3pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC3POL, (OutputPolarity << TIMER_CCEP_CC3POL_POS));
}

/**
  * @brief  Get timer capture/compare 3 output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 3 output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc3_output_polarity_cc3pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC3POL));
}

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc3_input_edge_cc3pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC3POL));
}

/**
  * @brief  Timer Capture/Compare 3 output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc3_output_cc3en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC3EN);
}

__STATIC_INLINE void md_timer_enable_cc3_input_cc3en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC3EN);
}

/**
  * @brief  Timer Capture/Compare 3 output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc3_output_cc3en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC3EN);
}

__STATIC_INLINE void md_timer_disable_cc3_input_cc3en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC3EN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 3 output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc3_output_cc3en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC3EN) == (TIMER_CCEP_CC3EN));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc3_input_cc3en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC3EN) == (TIMER_CCEP_CC3EN));
}

/**
  * @brief  Timer capture/compare 2 complementary output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc2_complementary_polarity_cc2npol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC2NPOL, (OutputPolarity << TIMER_CCEP_CC2NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 2 complementary output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 2 complementary output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc2_complementary_polarity_cc2npol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC2NPOL));
}

/**
  * @brief  Timer Capture/Compare 2 complementary output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc2_complementary_cc2nen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC2NEN);
}

/**
  * @brief  Timer Capture/Compare 2 complementary output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc2_complementary_cc2nen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC2NEN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 complementary output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc2_complementary_cc2nen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC2NEN) == (TIMER_CCEP_CC2NEN));
}

/**
  * @brief  Timer capture/compare 2 output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc2_output_polarity_cc2pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC2POL, (OutputPolarity << TIMER_CCEP_CC2POL_POS));
}

__STATIC_INLINE void md_timer_set_cc2_input_edge_cc2pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC2POL, (OutputPolarity << TIMER_CCEP_CC2POL_POS));
}

/**
  * @brief  Get timer capture/compare 2 output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 2 output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc2_output_polarity_cc2pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC2POL) >> TIMER_CCEP_CC2POL_POS);
}

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc2_input_edge_cc2pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC2POL) >> TIMER_CCEP_CC2POL_POS);
}

/**
  * @brief  Timer Capture/Compare 2 output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc2_output_cc2en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC2EN);
}

__STATIC_INLINE void md_timer_enable_cc2_input_cc2en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC2EN);
}

/**
  * @brief  Timer Capture/Compare 2 output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc2_output_cc2en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC2EN);
}

__STATIC_INLINE void md_timer_disable_cc2_input_cc2en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC2EN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc2_output_cc2en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC2EN) == (TIMER_CCEP_CC2EN));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc2_input_cc2en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC2EN) == (TIMER_CCEP_CC2EN));
}

/**
  * @brief  Timer capture/compare 1 complementary output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc1_complementary_polarity_cc1npol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC1NPOL, (OutputPolarity << TIMER_CCEP_CC1NPOL_POS));
}

/**
  * @brief  Get timer capture/compare 1 complementary output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 1 complementary output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc1_complementary_polarity_cc1npol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC1NPOL));
}

/**
  * @brief  Timer Capture/Compare 1 complementary output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc1_complementary_cc1nen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC1NEN);
}

/**
  * @brief  Timer Capture/Compare 1 complementary output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc1_complementary_cc1nen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC1NEN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 complementary output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc1_complementary_cc1nen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC1NEN) == (TIMER_CCEP_CC1NEN));
}

/**
  * @brief  Timer capture/compare 1 output polarity setup.
  * @param  TIMERx TIMER instance
  * @param  OutputPolarity Output polarity
            @arg @ref MD_TIMER_OUTPUTPOLARITY_HIGH
            @arg @ref MD_TIMER_OUTPUTPOLARITY_LOW
  * @retval None
  */

__STATIC_INLINE void md_timer_set_cc1_output_polarity_cc1pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC1POL, (OutputPolarity << TIMER_CCEP_CC1POL_POS));
}

__STATIC_INLINE void md_timer_set_cc1_input_edge_cc1pol(TIMER_TypeDef *TIMERx, md_timer_polarity_t OutputPolarity)
{
    MODIFY_REG(TIMERx->CCEP, TIMER_CCEP_CC1POL, (OutputPolarity << TIMER_CCEP_CC1POL_POS));
}

/**
  * @brief  Get timer capture/compare 1 output polarity.
  * @param  TIMERx TIMER instance
  * @retval Timer capture/compare 1 output polarity.
  */

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc1_output_polarity_cc1pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC1POL));
}

__STATIC_INLINE md_timer_polarity_t md_timer_get_cc1_input_edge_cc1pol(TIMER_TypeDef *TIMERx)
{
    return (md_timer_polarity_t)(READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC1POL));
}

/**
  * @brief  Timer Capture/Compare 1 output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_cc1_output_cc1en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC1EN);
}

__STATIC_INLINE void md_timer_enable_cc1_input_cc1en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CCEP, TIMER_CCEP_CC1EN);
}

/**
  * @brief  Timer Capture/Compare 1 output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_cc1_output_cc1en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC1EN);
}

__STATIC_INLINE void md_timer_disable_cc1_input_cc1en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CCEP, TIMER_CCEP_CC1EN);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_cc1_output_cc1en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC1EN) == (TIMER_CCEP_CC1EN));
}

__STATIC_INLINE uint32_t md_timer_is_enabled_cc1_input_cc1en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CCEP, TIMER_CCEP_CC1EN) == (TIMER_CCEP_CC1EN));
}
/**
  * @}
  */

/** @defgroup COUNT
  * @{
  */
/**
  * @brief  Set TIMER COUNT
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_count(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->COUNT, value);
}

/**
  * @brief  Get TIMER COUNT
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_count(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->COUNT));
}

/**
  * @brief  Get TIMER UPDRIF
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_active_flag_updrif(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->COUNT, TIMER_COUNT_UPDRIF) == (TIMER_COUNT_UPDRIF));
}

/**
  * @brief  Set TIMER CNTV
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_counter_value_cntv(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->COUNT, TIMER_COUNT_CNTV, value << TIMER_COUNT_CNTV_POSS);
}

/**
  * @brief  Get TIMER CNTV
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_counter_value_cntv(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->COUNT, TIMER_COUNT_CNTV) >> TIMER_COUNT_CNTV_POSS);
}

/**
  * @}
  */

/** @defgroup PRES
  * @{
  */
/**
  * @brief  Timer PRES setup.
  * @param  TIMERx TIMER instance
  * @param  value PRES
  * @retval None
  */

__STATIC_INLINE void md_timer_set_prescaler_value_pscv(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->PRES, value);
}

/**
  * @brief  GET Timer PRES register value.
  * @param  TIMERx TIMER instance
  * @retval Timer PRES register value.
  */

__STATIC_INLINE uint32_t md_timer_get_prescaler_value_pscv(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->PRES));
}
/**
  * @}
  */

/** @defgroup AR
  * @{
  */
/**
  * @brief  Timer AR setup.
  * @param  TIMERx TIMER instance
  * @param  value AR
  * @retval None
  */

__STATIC_INLINE void md_timer_set_auto_reload_value_arrv(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->AR, value);
}

/**
  * @brief  GET Timer AR register value.
  * @param  TIMERx TIMER instance
  * @retval Timer AR register value.
  */

__STATIC_INLINE uint32_t md_timer_get_auto_reload_value_arrv(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->AR));
}
/**
  * @}
  */

/** @defgroup REPAR
  * @{
  */
/**
  * @brief  Timer REPAR setup.
  * @param  TIMERx TIMER instance
  * @param  value REPAR
  * @retval None
  */

__STATIC_INLINE void md_timer_set_repetition_counter_value_repv(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->REPAR, value);
}

/**
  * @brief  GET Timer REPAR register value.
  * @param  TIMERx TIMER instance
  * @retval Timer REPAR register value.
  */

__STATIC_INLINE uint32_t md_timer_get_repetition_counter_value_repv(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->REPAR));
}
/**
  * @}
  */

/** @defgroup CCVAL1
  * @{
  */
/**
  * @brief  Timer CCVAL1 setup.
  * @param  TIMERx TIMER instance
  * @param  value CCVAL1
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare1_value_ccrv1(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CCVAL1, value);
}

/**
  * @brief  GET Timer CCVAL1 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CCVAL1 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare1_value_ccrv1(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CCVAL1));
}
/**
  * @}
  */

/** @defgroup CCVAL2
  * @{
  */
/**
  * @brief  Timer CCVAL2 setup.
  * @param  TIMERx TIMER instance
  * @param  value CCVAL2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare2_value_ccrv2(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CCVAL2, value);
}

/**
  * @brief  GET Timer CCVAL2 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CCVAL2 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare2_value_ccrv2(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CCVAL2));
}
/**
  * @}
  */

/** @defgroup CCVAL3
  * @{
  */
/**
  * @brief  Timer CCVAL3 setup.
  * @param  TIMERx TIMER instance
  * @param  value CCVAL3
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare3_value_ccrv3(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CCVAL3, value);
}

/**
  * @brief  GET Timer CCVAL3 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CCVAL3 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare3_value_ccrv3(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CCVAL3));
}
/**
  * @}
  */

/** @defgroup CCVAL4
  * @{
  */
/**
  * @brief  Timer CCVAL4 setup.
  * @param  TIMERx TIMER instance
  * @param  value CCVAL4
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare4_value_ccrv4(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CCVAL4, value);
}

/**
  * @brief  GET Timer CCVAL4 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CCVAL4 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare4_value_ccrv4(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CCVAL4));
}
/**
  * @}
  */

/** @defgroup BDCFG
  * @{
  */
/**
  * @brief  Timer BDCFG setup.
  * @param  TIMERx TIMER instance
  * @param  value (GOEN | AOEN | BRKP | BRKEN | OFFSSR | OFFSSI | LOCKLVL | DT)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_bdcfg(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->BDCFG, value);
}

/**
  * @brief  GET Timer BDCFG register value.
  * @param  TIMERx TIMER instance
  * @retval Timer BDCFG register value.
  */

__STATIC_INLINE uint32_t md_timer_get_bdcfg(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->BDCFG));
}

/**
  * @brief  TIMER enable BRK2BID
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brk2bid(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2BID);
}

/**
  * @brief TIMER disable BRK2BID
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brk2bid(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2BID);
}

/**
  * @brief TIMER is enabled BRK2BID
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brk2bid(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2BID) == (TIMER_BDCFG_BRK2BID));
}

/**
  * @brief  TIMER enable BRKBID
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brkbid(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKBID);
}

/**
  * @brief TIMER disable BRKBID
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brkbid(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKBID);
}

/**
  * @brief TIMER is enabled BRKBID
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brkbid(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKBID) == (TIMER_BDCFG_BRKBID));
}

/**
  * @brief  Set TIMER BRK2DSRM
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brk2dsrm(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_BRK2DSRM, value << TIMER_BDCFG_BRK2DSRM_POS);
}

/**
  * @brief  Get TIMER BRK2DSRM
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brk2dsrm(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2DSRM) >> TIMER_BDCFG_BRK2DSRM_POS);
}

/**
  * @brief  Set TIMER BRKDSRM
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brkdsrm(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_BRKDSRM, value << TIMER_BDCFG_BRKDSRM_POS);
}

/**
  * @brief  Get TIMER BRKDSRM
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brkdsrm(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKDSRM) >> TIMER_BDCFG_BRKDSRM_POS);
}

/**
  * @brief  Timer break polarity2 setup.
  * @param  TIMERx TIMER instance
  * @param  BreakPolarity Break polarity
  *         @arg @ref MD_TIMER_BREAKPOLARITY_LOW
  *         @arg @ref MD_TIMER_BREAKPOLARITY_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_break_polarity_brk2p(TIMER_TypeDef *TIMERx, md_timer_breakpolarity_t BreakPolarity)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_BRK2P, BreakPolarity << TIMER_BDCFG_BRK2P_POS);
}

/**
  * @brief  Get timer break polarity2.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_TIMER_BREAKPOLARITY_LOW
  *         @arg @ref MD_TIMER_BREAKPOLARITY_HIGH
  */

__STATIC_INLINE md_timer_breakpolarity_t md_timer_get_break_polarity_brk2p(TIMER_TypeDef *TIMERx)
{
    return (md_timer_breakpolarity_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2P) >> TIMER_BDCFG_BRK2P_POS);
}

/**
  * @brief  Timer break2 enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_break_brk2en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2EN);
}

/**
  * @brief  Timer break2 disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_break_brk2en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2EN);
}

/**
  * @brief  Indicates whether the timer break2 is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_break_brk2en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2EN) == (TIMER_BDCFG_BRK2EN));
}

/**
  * @brief  Set TIMER BRK2FLT
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N2
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N4
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV2N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV2N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV4N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV4N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV8N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV8N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N5
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N5
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N8
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brk2flt(TIMER_TypeDef *TIMERx, md_timer_brkflt_freq_t value)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_BRK2FLT, value << TIMER_BDCFG_BRK2FLT_POSS);
}

/**
  * @brief  Get TIMER BRK2FLT
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N2
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N4
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV2N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV2N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV4N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV4N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV8N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV8N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N5
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N5
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N8
  */
__STATIC_INLINE md_timer_brkflt_freq_t md_timer_get_brk2flt(TIMER_TypeDef *TIMERx)
{
    return (md_timer_brkflt_freq_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRK2FLT) >> TIMER_BDCFG_BRK2FLT_POSS);
}

/**
  * @brief  Set TIMER BRKFLT
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N2
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N4
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV2N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV2N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV4N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV4N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV8N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV8N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N5
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N5
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N8
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brkflt(TIMER_TypeDef *TIMERx, md_timer_brkflt_freq_t value)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_BRKFLT, value << TIMER_BDCFG_BRKFLT_POSS);
}

/**
  * @brief  Get TIMER BRKFLT
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N2
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N4
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV1N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV2N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV2N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV4N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV4N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV8N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV8N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N5
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV16N8
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N5
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N6
  *         @arg @ref MD_TIMER_BRKFLT_FREQ_FDIV32N8
  */
__STATIC_INLINE md_timer_brkflt_freq_t md_timer_get_brkflt(TIMER_TypeDef *TIMERx)
{
    return (md_timer_brkflt_freq_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKFLT) >> TIMER_BDCFG_BRKFLT_POSS);
}

/**
  * @brief  Timer main output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_main_output_goen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->BDCFG, TIMER_BDCFG_GOEN);
}

/**
  * @brief  Timer main output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_main_output_goen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->BDCFG, TIMER_BDCFG_GOEN);
}

/**
  * @brief  Indicates whether the timer main output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_main_output_goen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_GOEN) == (TIMER_BDCFG_GOEN));
}

/**
  * @brief  Timer automatic output enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_auto_output_aoen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->BDCFG, TIMER_BDCFG_AOEN);
}

/**
  * @brief  Timer automatic output disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_auto_output_aoen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->BDCFG, TIMER_BDCFG_AOEN);
}

/**
  * @brief  Indicates whether the timer automatic output is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_auto_output_aoen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_AOEN) == (TIMER_BDCFG_AOEN));
}

/**
  * @brief  Timer break polarity setup.
  * @param  TIMERx TIMER instance
  * @param  BreakPolarity Break polarity
  *         @arg @ref MD_TIMER_BREAKPOLARITY_LOW
  *         @arg @ref MD_TIMER_BREAKPOLARITY_HIGH
  * @retval None
  */

__STATIC_INLINE void md_timer_set_break_polarity_brkp(TIMER_TypeDef *TIMERx, md_timer_breakpolarity_t BreakPolarity)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_BRKP, BreakPolarity << TIMER_BDCFG_BRKP_POS);
}

/**
  * @brief  Get timer break polarity.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_TIMER_BREAKPOLARITY_LOW
  *         @arg @ref MD_TIMER_BREAKPOLARITY_HIGH
  */

__STATIC_INLINE md_timer_breakpolarity_t md_timer_get_break_polarity_brkp(TIMER_TypeDef *TIMERx)
{
    return (md_timer_breakpolarity_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKP) >> TIMER_BDCFG_BRKP_POS);
}

/**
  * @brief  Timer break enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_break_brken(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKEN);
}

/**
  * @brief  Timer break disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_break_brken(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKEN);
}

/**
  * @brief  Indicates whether the timer break is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_break_brken(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_BRKEN) == (TIMER_BDCFG_BRKEN));
}

/**
  * @brief  Timer off-state selection for run mode setup.
  * @param  TIMERx TIMER instance
  * @param  OffStateRun Off-state selection for run mode
  *         @arg @ref MD_TIMER_OFFSTATERUN_DISABLE
  *         @arg @ref MD_TIMER_OFFSTATERUN_ENABLE
  * @retval None
  */

__STATIC_INLINE void md_timer_set_offstate_run_selection_offssr(TIMER_TypeDef *TIMERx, md_timer_offstaterun_t OffStateRun)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_OFFSSR, OffStateRun << TIMER_BDCFG_OFFSSR_POS);
}

/**
  * @brief  Get timer off-state selection for run mode.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_TIMER_OFFSTATERUN_DISABLE
  *         @arg @ref MD_TIMER_OFFSTATERUN_ENABLE
  */

__STATIC_INLINE md_timer_offstaterun_t md_timer_get_offstate_run_selection_offssr(TIMER_TypeDef *TIMERx)
{
    return (md_timer_offstaterun_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_OFFSSR) >> TIMER_BDCFG_OFFSSR_POS);
}

/**
  * @brief  Timer off-state selection for idle mode setup.
  * @param  TIMERx TIMER instance
  * @param  OffStateIdle Off-state selection for idle mode
  *         @arg @ref MD_TIMER_OFFSTATEIDLE_DISABLE
  *         @arg @ref MD_TIMER_OFFSTATEIDLE_ENABLE
  * @retval None
  */

__STATIC_INLINE void md_timer_set_offstate_idle_selection_offssi(TIMER_TypeDef *TIMERx, md_timer_offstateidle_t OffStateIdle)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_OFFSSI, OffStateIdle << TIMER_BDCFG_OFFSSI_POS);
}

/**
  * @brief  Get timer off-state selection for idle mode.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_TIMER_OFFSTATEIDLE_DISABLE
  *         @arg @ref MD_TIMER_OFFSTATEIDLE_ENABLE
  */

__STATIC_INLINE md_timer_offstateidle_t md_timer_get_offstate_idle_selection_offssi(TIMER_TypeDef *TIMERx)
{
    return (md_timer_offstateidle_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_OFFSSI) >> TIMER_BDCFG_OFFSSI_POS);
}

/**
  * @brief  Timer lock configuration setup.
  * @param  TIMERx TIMER instance
  * @param  LockLevel Lock configuration
  *         @arg @ref MD_TIMER_LOCKLEVEL_0
  *         @arg @ref MD_TIMER_LOCKLEVEL_1
  *         @arg @ref MD_TIMER_LOCKLEVEL_2
  *         @arg @ref MD_TIMER_LOCKLEVEL_3
  * @retval None
  */

__STATIC_INLINE void md_timer_set_lock_config_locklvl(TIMER_TypeDef *TIMERx, md_timer_locklevel_t LockLevel)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_LOCKLVL, LockLevel << TIMER_BDCFG_LOCKLVL_POSS);
}

/**
  * @brief  Get timer lock configuration.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_TIMER_LOCKLEVEL_0
  *         @arg @ref MD_TIMER_LOCKLEVEL_1
  *         @arg @ref MD_TIMER_LOCKLEVEL_2
  *         @arg @ref MD_TIMER_LOCKLEVEL_3
  */

__STATIC_INLINE md_timer_locklevel_t md_timer_get_lock_config_locklvl(TIMER_TypeDef *TIMERx)
{
    return (md_timer_locklevel_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_LOCKLVL) >> TIMER_BDCFG_LOCKLVL_POSS);
}

/**
  * @brief  Timer dead-time generator setup.
  * @param  TIMERx TIMER instance
  * @param  DeadTime Dead-time generator (between Min_Data=0 and Max_Data=0xFF)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_dead_time_duration_dt(TIMER_TypeDef *TIMERx, uint32_t DeadTime)
{
    MODIFY_REG(TIMERx->BDCFG, TIMER_BDCFG_DT, DeadTime);
}

/**
  * @brief  Get timer dead-time generator.
  * @param  TIMERx TIMER instance
  * @retval Timer dead-time generator.
  */

__STATIC_INLINE uint32_t md_timer_get_dead_time_duration_dt(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->BDCFG, TIMER_BDCFG_DT));
}
/**
  * @}
  */

/** @defgroup DMAEN
  * @{
  */
/**
  * @brief  Timer DMAEN setup.
  * @param  TIMERx TIMER instance
  * @param  value (TRGI | COM | CH4 | CH3 | CH2 | CH1 | UPD)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_dmaen(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->DMAEN, value);
}

/**
  * @brief  GET Timer DMAEN register value.
  * @param  TIMERx TIMER instance
  * @retval Timer DMAEN register value.
  */

__STATIC_INLINE uint32_t md_timer_get_dmaen(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->DMAEN));
}

/**
  * @brief  Timer trigger DMA request enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_dma_trgi(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DMAEN, TIMER_DMAEN_TRGI);
}

/**
  * @brief  Timer trigger DMA request disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_dma_trgi(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DMAEN, TIMER_DMAEN_TRGI);
}

/**
  * @brief  Indicates whether the timer trigger DMA request is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_dma_trgi(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DMAEN, TIMER_DMAEN_TRGI) == (TIMER_DMAEN_TRGI));
}

/**
  * @brief  Timer COM DMA request enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_dma_com(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DMAEN, TIMER_DMAEN_COM);
}

/**
  * @brief  Timer COM DMA request disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_dma_com(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DMAEN, TIMER_DMAEN_COM);
}

/**
  * @brief  Indicates whether the timer trigger COM request is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_dma_com(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DMAEN, TIMER_DMAEN_COM) == (TIMER_DMAEN_COM));
}

/**
  * @brief  Timer Capture/Compare 4 DMA request enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_dma_ch4(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH4);
}

/**
  * @brief  Timer Capture/Compare 4 DMA request disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_dma_ch4(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH4);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 4 DMA request is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_dma_ch4(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH4) == (TIMER_DMAEN_CH4));
}

/**
  * @brief  Timer Capture/Compare 3 DMA request enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_dma_ch3(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH3);
}

/**
  * @brief  Timer Capture/Compare 3 DMA request disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_dma_ch3(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH3);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 3 DMA request is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_dma_ch3(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH3) == (TIMER_DMAEN_CH3));
}

/**
  * @brief  Timer Capture/Compare 2 DMA request enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_dma_ch2(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH2);
}

/**
  * @brief  Timer Capture/Compare 2 DMA request disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_dma_ch2(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH2);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 2 DMA request is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_dma_ch2(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH2) == (TIMER_DMAEN_CH2));
}

/**
  * @brief  Timer Capture/Compare 1 DMA request enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_dma_ch1(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH1);
}

/**
  * @brief  Timer Capture/Compare 1 DMA request disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_dma_ch1(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH1);
}

/**
  * @brief  Indicates whether the timer Capture/Compare 1 DMA request is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_dma_ch1(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DMAEN, TIMER_DMAEN_CH1) == (TIMER_DMAEN_CH1));
}

/**
  * @brief  Timer update DMA request enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_dma_upd(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DMAEN, TIMER_DMAEN_UPD);
}

/**
  * @brief  Timer update DMA request disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_dma_upd(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DMAEN, TIMER_DMAEN_UPD);
}

/**
  * @brief  Indicates whether the timer Capture/Compare update DMA request is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_dma_upd(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DMAEN, TIMER_DMAEN_UPD) == (TIMER_DMAEN_UPD));
}
/**
  * @}
  */
/** @defgroup CCVAL5
  * @{
  */
/**
  * @brief  Set TIMER CCVAL5
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ccval5(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CCVAL5, value);
}

/**
  * @brief  Get TIMER CCVAL5
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ccval5(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CCVAL5));
}

/**
  * @brief  Set TIMER GC5C3
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_gc5c3(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CCVAL5, TIMER_CCVAL5_GC5C3, value << TIMER_CCVAL5_GC5C3_POS);
}

/**
  * @brief  Get TIMER GC5C3
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_gc5c3(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CCVAL5, TIMER_CCVAL5_GC5C3) >> TIMER_CCVAL5_GC5C3_POS);
}

/**
  * @brief  Set TIMER GC5C2
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_gc5c2(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CCVAL5, TIMER_CCVAL5_GC5C2, value << TIMER_CCVAL5_GC5C2_POS);
}

/**
  * @brief  Get TIMER GC5C2
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_gc5c2(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CCVAL5, TIMER_CCVAL5_GC5C2) >> TIMER_CCVAL5_GC5C2_POS);
}

/**
  * @brief  Set TIMER GC5C1
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_gc5c1(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CCVAL5, TIMER_CCVAL5_GC5C1, value << TIMER_CCVAL5_GC5C1_POS);
}

/**
  * @brief  Get TIMER GC5C1
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_gc5c1(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CCVAL5, TIMER_CCVAL5_GC5C1) >> TIMER_CCVAL5_GC5C1_POS);
}

/**
  * @brief  Set TIMER CCRV5
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_capture_compare5_value_ccrv5(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CCVAL5, TIMER_CCVAL5_CCRV5, value << TIMER_CCVAL5_CCRV5_POSS);
}

/**
  * @brief  Get TIMER CCRV5
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_capture_compare5_value_ccrv5(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CCVAL5, TIMER_CCVAL5_CCRV5) >> TIMER_CCVAL5_CCRV5_POSS);
}
/**
  * @}
  */
/** @defgroup CCVAL6
  * @{
  */
/**
  * @brief  Timer CCVAL6 setup.
  * @param  TIMERx TIMER instance
  * @param  value CCVAL6
  * @retval None
  */

__STATIC_INLINE void md_timer_set_capture_compare6_value_ccrv6(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CCVAL6, value);
}

/**
  * @brief  GET Timer CCVAL6 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CCVAL6 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_capture_compare6_value_ccrv6(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CCVAL6));
}
/**
  * @}
  */
/** @defgroup CHMR3
  * @{
  */
/**
  * @brief  Timer CHMR3 setup.
  * @param  TIMERx TIMER instance
  * @param  value output (CH6MOD2|CH5MOD2|CH6OCLREN | CH6MOD | CH6PEN | CH2FEN | CH6FEN | CH5OCLREN | CH5MOD | CH5PEN | CH5FEN)
  * @retval None
  */

__STATIC_INLINE void md_timer_set_chmr3(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CHMR3, value);
}

/**
  * @brief  GET Timer CHMR3 register value.
  * @param  TIMERx TIMER instance
  * @retval Timer CHMR3 register value.
  */

__STATIC_INLINE uint32_t md_timer_get_chmr3(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CHMR3));
}

/**
  * @brief  Timer output compare 6 clear enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare6_clear_ch6oclren(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6OCLREN);
}

/**
  * @brief  Timer output compare 6 clear disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare6_clear_ch6oclren(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6OCLREN);
}

/**
  * @brief  Indicates whether the timer output compare 6 clear is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare6_clear_ch6oclren(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6OCLREN) == (TIMER_CHMR3_CH6OCLREN));
}

/**
  * @brief  Timer output compare 6 mode setup.
  * @param  TIMERx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare6_mode_ch6mod(TIMER_TypeDef *TIMERx, md_timer_outputmode_list_t OutputMode)
{
    MODIFY_REG(TIMERx->CHMR3, TIMER_CHMR3_CH6MOD, (OutputMode & 0x7) << TIMER_CHMR3_CH6MOD_POSS);
    MODIFY_REG(TIMERx->CHMR3, TIMER_CHMR3_CH6MOD2, ((OutputMode & 0x18UL) >> 3) << TIMER_CHMR3_CH6MOD2_POS);
}

/**
  * @brief  Get timer output compare 6 mode.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  */

__STATIC_INLINE md_timer_outputmode_list_t md_timer_get_output_compare6_mode_ch6mod(TIMER_TypeDef *TIMERx)
{
    return (md_timer_outputmode_list_t)(((READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6MOD2) >> TIMER_CHMR3_CH6MOD2_POS) << 3UL) | (READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6MOD) >> TIMER_CHMR3_CH6MOD_POSS));
}

/**
  * @brief  Timer output compare 6 preload enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare6_preload_ch6pen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6PEN);
}

/**
  * @brief  Timer output compare 6 preload disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare6_preload_ch6pen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6PEN);
}

/**
  * @brief  Indicates whether the timer output compare 6 preload is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare6_preload_ch6pen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6PEN) == (TIMER_CHMR3_CH6PEN));
}

/**
  * @brief  Timer output compare 6 fast enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare6_fast_ch6fen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6FEN);
}

/**
  * @brief  Timer output compare 6 fast disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare6_fast_ch6fen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6FEN);
}

/**
  * @brief  Indicates whether the timer output compare 6 fast is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare6_fast_ch6fen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH6FEN) == (TIMER_CHMR3_CH6FEN));
}

/**
  * @brief  Timer output compare 5 clear enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare5_clear_ch5oclren(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5OCLREN);
}

/**
  * @brief  Timer output compare 5 clear disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare5_clear_ch5oclren(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5OCLREN);
}

/**
  * @brief  Indicates whether the timer output compare 5 clear is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare5_clear_ch5oclren(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5OCLREN) == (TIMER_CHMR3_CH5OCLREN));
}

/**
  * @brief  Timer output compare 5 mode setup.
  * @param  TIMERx TIMER instance
  * @param  OutputMode Output compare mode
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  * @retval None
  */

__STATIC_INLINE void md_timer_set_output_compare5_mode_ch5mod(TIMER_TypeDef *TIMERx, md_timer_outputmode_list_t OutputMode)
{
    MODIFY_REG(TIMERx->CHMR3, TIMER_CHMR3_CH5MOD, (OutputMode & 0x7UL) << TIMER_CHMR3_CH5MOD_POSS);
    MODIFY_REG(TIMERx->CHMR3, TIMER_CHMR3_CH5MOD2, ((OutputMode & 0x18UL) >> 3) << TIMER_CHMR3_CH5MOD2_POS);
}

/**
  * @brief  Get timer output compare 5 mode.
  * @param  TIMERx TIMER instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_TIMER_OUTPUTMODE_DISABLE
            @arg @ref MD_TIMER_OUTPUTMODE_HIGHONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_LOWONMSTCH
            @arg @ref MD_TIMER_OUTPUTMODE_TOGGLE
            @arg @ref MD_TIMER_OUTPUTMODE_FORCELOW
            @arg @ref MD_TIMER_OUTPUTMODE_FORCEHIGH
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE1
            @arg @ref MD_TIMER_OUTPUTMODE_PWMMODE2
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_MULTI_TRIGGER_SINGLE_PULSE_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_COMBINED_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C2_PWM_MODE2
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE1
            @arg @ref MD_TIMER_OUTPUTMODE_ASYMMETRIC_C5_PWM_MODE2
  */

__STATIC_INLINE md_timer_outputmode_list_t md_timer_get_output_compare5_mode_ch5mod(TIMER_TypeDef *TIMERx)
{
    return (md_timer_outputmode_list_t)(((READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5MOD2) >> TIMER_CHMR3_CH5MOD2_POS) << 3UL) | (READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5MOD) >> TIMER_CHMR3_CH5MOD_POSS));
}

/**
  * @brief  Timer output compare 5 preload enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare5_preload_ch5pen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5PEN);
}

/**
  * @brief  Timer output compare 5 preload disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare5_preload_ch5pen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5PEN);
}

/**
  * @brief  Indicates whether the timer output compare 5 preload is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare5_preload_ch5pen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5PEN) == (TIMER_CHMR3_CH5PEN));
}

/**
  * @brief  Timer output compare 5 fast enable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_enable_output_compare5_fast_ch5fen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5FEN);
}

/**
  * @brief  Timer output compare 5 fast disable.
  * @param  TIMERx TIMER instance
  * @retval None
  */

__STATIC_INLINE void md_timer_disable_output_compare5_fast_ch5fen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5FEN);
}

/**
  * @brief  Indicates whether the timer output compare 5 fast is enabled.
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */

__STATIC_INLINE uint32_t md_timer_is_enabled_output_compare5_fast_ch5fen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->CHMR3, TIMER_CHMR3_CH5FEN) == (TIMER_CHMR3_CH5FEN));
}
/**
  * @}
  */
/** @defgroup DCFG2
  * @{
  */
/**
  * @brief  Set TIMER DCFG2
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_dcfg2(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->DCFG2, value);
}

/**
  * @brief  Get TIMER DCFG2
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_dcfg2(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->DCFG2));
}

/**
  * @brief  TIMER enable DTPEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_dtpen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DCFG2, TIMER_DCFG2_DTPEN);
}

/**
  * @brief TIMER disable DTPEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_dtpen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DCFG2, TIMER_DCFG2_DTPEN);
}

/**
  * @brief TIMER is enabled DTPEN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_dtpen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DCFG2, TIMER_DCFG2_DTPEN) == (TIMER_DCFG2_DTPEN));
}

/**
  * @brief  TIMER enable DTAEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_dtaen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->DCFG2, TIMER_DCFG2_DTAEN);
}

/**
  * @brief TIMER disable DTAEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_dtaen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->DCFG2, TIMER_DCFG2_DTAEN);
}

/**
  * @brief TIMER is enabled DTAEN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_dtaen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->DCFG2, TIMER_DCFG2_DTAEN) == (TIMER_DCFG2_DTAEN));
}

/**
  * @brief  Set TIMER DT2
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_dt2(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->DCFG2, TIMER_DCFG2_DT2, value << TIMER_DCFG2_DT2_POSS);
}

/**
  * @brief  Get TIMER DT2
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_dt2(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->DCFG2, TIMER_DCFG2_DT2) >> TIMER_DCFG2_DT2_POSS);
}
/**
  * @}
  */
/** @defgroup ENCR
  * @{
  */
/**
  * @brief  Set TIMER ENCR
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_encr(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->ENCR, value);
}

/**
  * @brief  Get TIMER ENCR
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_encr(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->ENCR));
}

/**
  * @brief  Set TIMER PWPRES
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_pwpres(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->ENCR, TIMER_ENCR_PWPRES, value << TIMER_ENCR_PWPRES_POSS);
}

/**
  * @brief  Get TIMER PWPRES
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_pwpres(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->ENCR, TIMER_ENCR_PWPRES) >> TIMER_ENCR_PWPRES_POSS);
}

/**
  * @brief  Set TIMER PW
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_pw(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->ENCR, TIMER_ENCR_PW, value << TIMER_ENCR_PW_POSS);
}

/**
  * @brief  Get TIMER PW
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_pw(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->ENCR, TIMER_ENCR_PW) >> TIMER_ENCR_PW_POSS);
}

/**
  * @brief  Set TIMER IDXPOS
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_idxpos(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->ENCR, TIMER_ENCR_IDXPOS, value << TIMER_ENCR_IDXPOS_POSS);
}

/**
  * @brief  Get TIMER IDXPOS
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_idxpos(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->ENCR, TIMER_ENCR_IDXPOS) >> TIMER_ENCR_IDXPOS_POSS);
}

/**
  * @brief  Set TIMER FIDX
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_fidx(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->ENCR, TIMER_ENCR_FIDX, value << TIMER_ENCR_FIDX_POS);
}

/**
  * @brief  Get TIMER FIDX
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_fidx(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->ENCR, TIMER_ENCR_FIDX) >> TIMER_ENCR_FIDX_POS);
}

/**
  * @brief  Set TIMER IDXDIR
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_idxdir(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->ENCR, TIMER_ENCR_IDXDIR, value << TIMER_ENCR_IDXDIR_POSS);
}

/**
  * @brief  Get TIMER IDXDIR
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_idxdir(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->ENCR, TIMER_ENCR_IDXDIR) >> TIMER_ENCR_IDXDIR_POSS);
}

/**
  * @brief  TIMER enable IDXEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_idxen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->ENCR, TIMER_ENCR_IDXEN);
}

/**
  * @brief TIMER disable IDXEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_idxen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->ENCR, TIMER_ENCR_IDXEN);
}

/**
  * @brief TIMER is enabled IDXEN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_idxen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->ENCR, TIMER_ENCR_IDXEN) == (TIMER_ENCR_IDXEN));
}
/**
  * @}
  */
/** @defgroup CHISEL
  * @{
  */
/**
  * @brief  Set TIMER CHISEL
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_chisel(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CHISEL, value);
}

/**
  * @brief  Get TIMER CHISEL
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_chisel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CHISEL));
}

/**
  * @brief  Set TIMER I4SEL
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_i4sel(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHISEL, TIMER_CHISEL_I4SEL, value << TIMER_CHISEL_I4SEL_POSS);
}

/**
  * @brief  Get TIMER I4SEL
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_i4sel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHISEL, TIMER_CHISEL_I4SEL) >> TIMER_CHISEL_I4SEL_POSS);
}

/**
  * @brief  Set TIMER I3SEL
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_i3sel(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHISEL, TIMER_CHISEL_I3SEL, value << TIMER_CHISEL_I3SEL_POSS);
}

/**
  * @brief  Get TIMER I3SEL
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_i3sel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHISEL, TIMER_CHISEL_I3SEL) >> TIMER_CHISEL_I3SEL_POSS);
}

/**
  * @brief  Set TIMER I2SEL
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_i2sel(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHISEL, TIMER_CHISEL_I2SEL, value << TIMER_CHISEL_I2SEL_POSS);
}

/**
  * @brief  Get TIMER I2SEL
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_i2sel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHISEL, TIMER_CHISEL_I2SEL) >> TIMER_CHISEL_I2SEL_POSS);
}

/**
  * @brief  Set TIMER I1SEL
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_i1sel(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHISEL, TIMER_CHISEL_I1SEL, value << TIMER_CHISEL_I1SEL_POSS);
}

/**
  * @brief  Get TIMER I1SEL
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_i1sel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHISEL, TIMER_CHISEL_I1SEL) >> TIMER_CHISEL_I1SEL_POSS);
}
/**
  * @}
  */
/** @defgroup AFR1
  * @{
  */
/**
  * @brief  Set TIMER AFR1
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_afr1(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->AFR1, value);
}

/**
  * @brief  Get TIMER AFR1
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_afr1(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->AFR1));
}

/**
  * @brief  Set TIMER ETRSEL
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_etrsel(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR1, TIMER_AFR1_ETRSEL, value << TIMER_AFR1_ETRSEL_POSS);
}

/**
  * @brief  Get TIMER ETRSEL
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_etrsel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR1, TIMER_AFR1_ETRSEL) >> TIMER_AFR1_ETRSEL_POSS);
}

/**
  * @brief  Set TIMER BRKEXTP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brkextp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR1, TIMER_AFR1_BRKEXTP, value << TIMER_AFR1_BRKEXTP_POS);
}

/**
  * @brief  Get TIMER BRKEXTP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brkextp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR1, TIMER_AFR1_BRKEXTP) >> TIMER_AFR1_BRKEXTP_POS);
}

/**
  * @brief  Set TIMER BRKCMP2P
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brkcmp2p(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR1, TIMER_AFR1_BRKCMP2P, value << TIMER_AFR1_BRKCMP2P_POS);
}

/**
  * @brief  Get TIMER BRKCMP2P
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brkcmp2p(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR1, TIMER_AFR1_BRKCMP2P) >> TIMER_AFR1_BRKCMP2P_POS);
}

/**
  * @brief  Set TIMER BRKCMP1P
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brkcmp1p(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR1, TIMER_AFR1_BRKCMP1P, value << TIMER_AFR1_BRKCMP1P_POS);
}

/**
  * @brief  Get TIMER BRKCMP1P
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brkcmp1p(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR1, TIMER_AFR1_BRKCMP1P) >> TIMER_AFR1_BRKCMP1P_POS);
}

/**
  * @brief  Set TIMER BRKINP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brkinp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR1, TIMER_AFR1_BRKINP, value << TIMER_AFR1_BRKINP_POS);
}

/**
  * @brief  Get TIMER BRKINP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brkinp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR1, TIMER_AFR1_BRKINP) >> TIMER_AFR1_BRKINP_POS);
}

/**
  * @brief  TIMER enable BRKEXTEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brkexten(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->AFR1, TIMER_AFR1_BRKEXTEN);
}

/**
  * @brief TIMER disable BRKEXTEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brkexten(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->AFR1, TIMER_AFR1_BRKEXTEN);
}

/**
  * @brief TIMER is enabled BRKEXTEN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brkexten(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->AFR1, TIMER_AFR1_BRKEXTEN) == (TIMER_AFR1_BRKEXTEN));
}

/**
  * @brief  TIMER enable BRKCMP2EN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brkcmp2en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->AFR1, TIMER_AFR1_BRKCMP2EN);
}

/**
  * @brief TIMER disable BRKCMP2EN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brkcmp2en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->AFR1, TIMER_AFR1_BRKCMP2EN);
}

/**
  * @brief TIMER is enabled BRKCMP2EN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brkcmp2en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->AFR1, TIMER_AFR1_BRKCMP2EN) == (TIMER_AFR1_BRKCMP2EN));
}

/**
  * @brief  TIMER enable BRKCMP1EN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brkcmp1en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->AFR1, TIMER_AFR1_BRKCMP1EN);
}

/**
  * @brief TIMER disable BRKCMP1EN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brkcmp1en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->AFR1, TIMER_AFR1_BRKCMP1EN);
}

/**
  * @brief TIMER is enabled BRKCMP1EN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brkcmp1en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->AFR1, TIMER_AFR1_BRKCMP1EN) == (TIMER_AFR1_BRKCMP1EN));
}

/**
  * @brief  TIMER enable BRKINEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brkinen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->AFR1, TIMER_AFR1_BRKINEN);
}

/**
  * @brief TIMER disable BRKINEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brkinen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->AFR1, TIMER_AFR1_BRKINEN);
}

/**
  * @brief TIMER is enabled BRKINEN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brkinen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->AFR1, TIMER_AFR1_BRKINEN) == (TIMER_AFR1_BRKINEN));
}
/**
  * @}
  */
/** @defgroup AFR2
  * @{
  */
/**
  * @brief  Set TIMER AFR2
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_afr2(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->AFR2, value);
}

/**
  * @brief  Get TIMER AFR2
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_afr2(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->AFR2));
}

/**
  * @brief  Set TIMER OCLRSEL
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_oclrsel(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR2, TIMER_AFR2_OCLRSEL, value << TIMER_AFR2_OCLRSEL_POSS);
}

/**
  * @brief  Get TIMER OCLRSEL
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_oclrsel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR2, TIMER_AFR2_OCLRSEL) >> TIMER_AFR2_OCLRSEL_POSS);
}

/**
  * @brief  Set TIMER BRK2EXTP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brk2extp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR2, TIMER_AFR2_BRK2EXTP, value << TIMER_AFR2_BRK2EXTP_POS);
}

/**
  * @brief  Get TIMER BRK2EXTP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brk2extp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2EXTP) >> TIMER_AFR2_BRK2EXTP_POS);
}

/**
  * @brief  Set TIMER BRK2CMP2P
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brk2cmp2p(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR2, TIMER_AFR2_BRK2CMP2P, value << TIMER_AFR2_BRK2CMP2P_POS);
}

/**
  * @brief  Get TIMER BRK2CMP2P
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brk2cmp2p(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2CMP2P) >> TIMER_AFR2_BRK2CMP2P_POS);
}

/**
  * @brief  Set TIMER BRK2CMP1P
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brk2cmp1p(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR2, TIMER_AFR2_BRK2CMP1P, value << TIMER_AFR2_BRK2CMP1P_POS);
}

/**
  * @brief  Get TIMER BRK2CMP1P
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brk2cmp1p(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2CMP1P) >> TIMER_AFR2_BRK2CMP1P_POS);
}

/**
  * @brief  Set TIMER BRK2INP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_brk2inp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->AFR2, TIMER_AFR2_BRK2INP, value << TIMER_AFR2_BRK2INP_POS);
}

/**
  * @brief  Get TIMER BRK2INP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_brk2inp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2INP) >> TIMER_AFR2_BRK2INP_POS);
}

/**
  * @brief  TIMER enable BRK2EXTEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brk2exten(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2EXTEN);
}

/**
  * @brief TIMER disable BRK2EXTEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brk2exten(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2EXTEN);
}

/**
  * @brief TIMER is enabled BRK2EXTEN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brk2exten(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2EXTEN) == (TIMER_AFR2_BRK2EXTEN));
}

/**
  * @brief  TIMER enable BRK2CMP2EN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brk2cmp2en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2CMP2EN);
}

/**
  * @brief TIMER disable BRK2CMP2EN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brk2cmp2en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2CMP2EN);
}

/**
  * @brief TIMER is enabled BRK2CMP2EN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brk2cmp2en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2CMP2EN) == (TIMER_AFR2_BRK2CMP2EN));
}

/**
  * @brief  TIMER enable BRK2CMP1EN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brk2cmp1en(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2CMP1EN);
}

/**
  * @brief TIMER disable BRK2CMP1EN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brk2cmp1en(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2CMP1EN);
}

/**
  * @brief TIMER is enabled BRK2CMP1EN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brk2cmp1en(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2CMP1EN) == (TIMER_AFR2_BRK2CMP1EN));
}

/**
  * @brief  TIMER enable BRK2INEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_enable_brk2inen(TIMER_TypeDef *TIMERx)
{
    SET_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2INEN);
}

/**
  * @brief TIMER disable BRK2INEN
  * @param  TIMERx TIMER instance
  * @retval None
  */
__STATIC_INLINE void md_timer_disable_brk2inen(TIMER_TypeDef *TIMERx)
{
    CLEAR_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2INEN);
}

/**
  * @brief TIMER is enabled BRK2INEN
  * @param  TIMERx TIMER instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_timer_is_enabled_brk2inen(TIMER_TypeDef *TIMERx)
{
    return (READ_BIT(TIMERx->AFR2, TIMER_AFR2_BRK2INEN) == (TIMER_AFR2_BRK2INEN));
}
/**
  * @}
  */
/** @defgroup TGOSEL
  * @{
  */
/**
  * @brief  Set TIMER TGOSEL
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_tgosel(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->TGOSEL, value);
}

/**
  * @brief  Get TIMER TGOSEL
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_tgosel(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->TGOSEL));
}

/**
  * @brief  Set TIMER CH6U
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch6u(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH6U, value << TIMER_TGOSEL_CH6U_POS);
}

/**
  * @brief  Get TIMER CH6U
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch6u(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH6U) >> TIMER_TGOSEL_CH6U_POS);
}

/**
  * @brief  Set TIMER CH5U
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch5u(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH5U, value << TIMER_TGOSEL_CH5U_POS);
}

/**
  * @brief  Get TIMER CH5U
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch5u(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH5U) >> TIMER_TGOSEL_CH5U_POS);
}

/**
  * @brief  Set TIMER CH4U
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch4u(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH4U, value << TIMER_TGOSEL_CH4U_POS);
}

/**
  * @brief  Get TIMER CH4U
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch4u(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH4U) >> TIMER_TGOSEL_CH4U_POS);
}

/**
  * @brief  Set TIMER CH3U
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch3u(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH3U, value << TIMER_TGOSEL_CH3U_POS);
}

/**
  * @brief  Get TIMER CH3U
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch3u(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH3U) >> TIMER_TGOSEL_CH3U_POS);
}

/**
  * @brief  Set TIMER CH2U
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch2u(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH2U, value << TIMER_TGOSEL_CH2U_POS);
}

/**
  * @brief  Get TIMER CH2U
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch2u(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH2U) >> TIMER_TGOSEL_CH2U_POS);
}

/**
  * @brief  Set TIMER CH1U
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch1u(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH1U, value << TIMER_TGOSEL_CH1U_POS);
}

/**
  * @brief  Get TIMER CH1U
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch1u(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH1U) >> TIMER_TGOSEL_CH1U_POS);
}

/**
  * @brief  Set TIMER CH6D
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch6d(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH6D, value << TIMER_TGOSEL_CH6D_POS);
}

/**
  * @brief  Get TIMER CH6D
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch6d(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH6D) >> TIMER_TGOSEL_CH6D_POS);
}

/**
  * @brief  Set TIMER CH5D
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch5d(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH5D, value << TIMER_TGOSEL_CH5D_POS);
}

/**
  * @brief  Get TIMER CH5D
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch5d(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH5D) >> TIMER_TGOSEL_CH5D_POS);
}

/**
  * @brief  Set TIMER CH4D
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch4d(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH4D, value << TIMER_TGOSEL_CH4D_POS);
}

/**
  * @brief  Get TIMER CH4D
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch4d(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH4D) >> TIMER_TGOSEL_CH4D_POS);
}

/**
  * @brief  Set TIMER CH3D
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch3d(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH3D, value << TIMER_TGOSEL_CH3D_POS);
}

/**
  * @brief  Get TIMER CH3D
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch3d(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH3D) >> TIMER_TGOSEL_CH3D_POS);
}

/**
  * @brief  Set TIMER CH2D
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch2d(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH2D, value << TIMER_TGOSEL_CH2D_POS);
}

/**
  * @brief  Get TIMER CH2D
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch2d(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH2D) >> TIMER_TGOSEL_CH2D_POS);
}

/**
  * @brief  Set TIMER CH1D
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch1d(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->TGOSEL, TIMER_TGOSEL_CH1D, value << TIMER_TGOSEL_CH1D_POS);
}

/**
  * @brief  Get TIMER CH1D
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch1d(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->TGOSEL, TIMER_TGOSEL_CH1D) >> TIMER_TGOSEL_CH1D_POS);
}
/**
  * @}
  */
/** @defgroup CHRMP
  * @{
  */
/**
  * @brief  Set TIMER CHRMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_chrmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    WRITE_REG(TIMERx->CHRMP, value);
}

/**
  * @brief  Get TIMER CHRMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_chrmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_REG(TIMERx->CHRMP));
}

/**
  * @brief  Set TIMER CH4NRMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch4nrmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHRMP, TIMER_CHRMP_CH4NRMP, value << TIMER_CHRMP_CH4NRMP_POSS);
}

/**
  * @brief  Get TIMER CH4NRMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch4nrmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHRMP, TIMER_CHRMP_CH4NRMP) >> TIMER_CHRMP_CH4NRMP_POSS);
}

/**
  * @brief  Set TIMER CH3NRMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch3nrmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHRMP, TIMER_CHRMP_CH3NRMP, value << TIMER_CHRMP_CH3NRMP_POSS);
}

/**
  * @brief  Get TIMER CH3NRMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch3nrmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHRMP, TIMER_CHRMP_CH3NRMP) >> TIMER_CHRMP_CH3NRMP_POSS);
}

/**
  * @brief  Set TIMER CH2NRMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch2nrmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHRMP, TIMER_CHRMP_CH2NRMP, value << TIMER_CHRMP_CH2NRMP_POSS);
}

/**
  * @brief  Get TIMER CH2NRMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch2nrmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHRMP, TIMER_CHRMP_CH2NRMP) >> TIMER_CHRMP_CH2NRMP_POSS);
}

/**
  * @brief  Set TIMER CH1NRMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch1nrmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHRMP, TIMER_CHRMP_CH1NRMP, value << TIMER_CHRMP_CH1NRMP_POSS);
}

/**
  * @brief  Get TIMER CH1NRMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch1nrmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHRMP, TIMER_CHRMP_CH1NRMP) >> TIMER_CHRMP_CH1NRMP_POSS);
}

/**
  * @brief  Set TIMER CH4RMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch4rmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHRMP, TIMER_CHRMP_CH4RMP, value << TIMER_CHRMP_CH4RMP_POSS);
}

/**
  * @brief  Get TIMER CH4RMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch4rmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHRMP, TIMER_CHRMP_CH4RMP) >> TIMER_CHRMP_CH4RMP_POSS);
}

/**
  * @brief  Set TIMER CH3RMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch3rmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHRMP, TIMER_CHRMP_CH3RMP, value << TIMER_CHRMP_CH3RMP_POSS);
}

/**
  * @brief  Get TIMER CH3RMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch3rmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHRMP, TIMER_CHRMP_CH3RMP) >> TIMER_CHRMP_CH3RMP_POSS);
}

/**
  * @brief  Set TIMER CH2RMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch2rmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHRMP, TIMER_CHRMP_CH2RMP, value << TIMER_CHRMP_CH2RMP_POSS);
}

/**
  * @brief  Get TIMER CH2RMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch2rmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHRMP, TIMER_CHRMP_CH2RMP) >> TIMER_CHRMP_CH2RMP_POSS);
}

/**
  * @brief  Set TIMER CH1RMP
  * @param  TIMERx TIMER instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_timer_set_ch1rmp(TIMER_TypeDef *TIMERx, uint32_t value)
{
    MODIFY_REG(TIMERx->CHRMP, TIMER_CHRMP_CH1RMP, value << TIMER_CHRMP_CH1RMP_POSS);
}

/**
  * @brief  Get TIMER CH1RMP
  * @param  TIMERx TIMER instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_timer_get_ch1rmp(TIMER_TypeDef *TIMERx)
{
    return (uint32_t)(READ_BIT(TIMERx->CHRMP, TIMER_CHRMP_CH1RMP) >> TIMER_CHRMP_CH1RMP_POSS);
}
/**
  * @}
  */

/** @defgroup MD_TIMER_Public_Functions_Group1 Initialization
  * @{
  */
extern void md_timer_base_set_config(TIMER_TypeDef *TIMERx, md_timer_base_init_t *init);
extern void md_timer_base_struct_init(md_timer_base_init_t *init);
extern void md_timer_oc_struct_init(md_timer_oc_init_t *init);
extern md_status_t md_timer_oc_init(TIMER_TypeDef *TIMERx, md_timer_channel_t channel, md_timer_oc_init_t *init);
extern void md_timer_ic_struct_init(md_timer_ic_init_t *init);
extern md_status_t md_timer_ic_init(TIMER_TypeDef *TIMERx, md_timer_channel_t Channel, md_timer_ic_init_t *init);
extern void md_timer_encoder_struct_init(md_timer_encoder_init_t *init);
extern md_status_t md_timer_encoder_init(TIMER_TypeDef *TIMERx, md_timer_encoder_init_t *init);
extern void md_timer_hallsensor_struct_init(md_timer_hall_sensor_init_t *init);
extern md_status_t md_timer_hallsensor_init(TIMER_TypeDef *TIMERx, md_timer_hall_sensor_init_t *init);
extern void md_timer_bdtr_struct_init(md_timer_break_dead_time_t *init);
extern md_status_t md_timer_bdtr_init(TIMER_TypeDef *TIMERx, md_timer_break_dead_time_t *init);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __MD_TIMER_H_ */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
