/**********************************************************************************
 *
 * @file    md_syscfg.h
 * @brief   header file of md_syscfg.c
 *
 * @date    13 Apr 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          13 Apr 2023     Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_SYSCFG_H__
#define __MD_SYSCFG_H__

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

/* Includes ------------------------------------------------------------------ */
#include <stdbool.h>
#include "md_utils.h"

/* Exported Macros ----------------------------------------------------------- */
/* Exported Variables -------------------------------------------------------- */
/* Exported Constants -------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_SYSCFG SYSCFG
  * @brief SYSCFG micro driver
  * @{
  */

/** @defgroup MD_SYSCFG_Public_Macros SYSCFG Public Macros
  * @{
  */

/* Exported Types ------------------------------------------------------------ */

/**
  * @brief Memory mapping Status
  */
typedef enum
{
    MD_SYSCFG_REALMOD_MAIN    = (0x00000000UL),  /** @brief Main Flash memory mapped at 0x00000000 */
    MD_SYSCFG_REALMOD_BOOTROM = (0x00000001UL),  /** @brief Bootrom mapped at 0x00000000 */
    MD_SYSCFG_REALMOD_SRAM    = (0x00000002UL),  /** @brief SRAM mapped at 0x00000000 */
}md_syscfg_real_memory_mode_t;

/**
  * @brief MD_SYSCFG_REMAP_MEMMOD_FLAG Memory mapping selection bits
  */
typedef enum
{
    MD_SYSCFG_MEMMOD_MAIN    = (0x00000000UL),  /** @brief Main Flash memory mapped at 0x00000000 */
    MD_SYSCFG_MEMMOD_BOOTROM = (0x00000001UL),  /** @brief Bootrom mapped at 0x00000000 */
    MD_SYSCFG_MEMMOD_SRAM    = (0x00000002UL),  /** @brief SRAM mapped at 0x00000000 */
}md_syscfg_memory_mode_t;

/**
  * @brief MD_SYSCFG_OSCTRIM_HOSCRCNT
  */
typedef enum
{
    MD_SYSCFG_HOSCRCNT_256   = (0x00000000UL), /** @brief 0x0 : 256   */
    MD_SYSCFG_HOSCRCNT_512   = (0x00000001UL), /** @brief 0x1 : 512   */
    MD_SYSCFG_HOSCRCNT_1024  = (0x00000002UL), /** @brief 0x2 : 1024  */
    MD_SYSCFG_HOSCRCNT_2048  = (0x00000003UL), /** @brief 0x3 : 2048  */
    MD_SYSCFG_HOSCRCNT_4096  = (0x00000004UL), /** @brief 0x4 : 4096  */
    MD_SYSCFG_HOSCRCNT_8192  = (0x00000005UL), /** @brief 0x5 : 8192  */
    MD_SYSCFG_HOSCRCNT_16384 = (0x00000006UL), /** @brief 0x6 : 16384 */
    MD_SYSCFG_HOSCRCNT_32768 = (0x00000007UL), /** @brief 0x7 : 32768 */
}md_syscfg_hosc_count_t;

/**
  * @brief MD_SYSCFG_MEMMOD_SMOD
  */
typedef enum
{
    MD_SYSCFG_SMOD_128KB = (0x0000007FUL), /** @brief 0x7F: 128KB */
    MD_SYSCFG_SMOD_64KB  = (0x0000003FUL), /** @brief 0x3F: 64KB */
    MD_SYSCFG_SMOD_32KB  = (0x0000001FUL), /** @brief 0x1F: 32KB */
    MD_SYSCFG_SMOD_16KB  = (0x0000000FUL), /** @brief 0x0F: 16KB */
    MD_SYSCFG_SMOD_8KB   = (0x00000007UL), /** @brief 0x07: 8KB  */
    MD_SYSCFG_SMOD_4KB   = (0x00000003UL), /** @brief 0x03: 4KB  */
    MD_SYSCFG_SMOD_2KB   = (0x00000001UL), /** @brief 0x01: 2KB  */
}md_syscfg_sram_mode_t;

/**
  * @brief MD_SYSCFG_MEMMOD_FMOD
  */
typedef enum
{
    MD_SYSCFG_FMOD_512KB = (0x000001FFUL), /** @brief 0x1FF: 512KB */
    MD_SYSCFG_FMOD_256KB = (0x000000FFUL), /** @brief 0xFF: 256KB */
    MD_SYSCFG_FMOD_128KB = (0x0000007FUL), /** @brief 0x7F: 128KB */
    MD_SYSCFG_FMOD_64KB  = (0x0000003FUL), /** @brief 0x3F: 64KB  */
    MD_SYSCFG_FMOD_32KB  = (0x0000001FUL), /** @brief 0x1F: 32KB  */
    MD_SYSCFG_FMOD_16KB  = (0x0000000FUL), /** @brief 0x0F: 16KB  */
}md_syscfg_flash_mode_t;

/**
  * @brief MD_SYSCFG_SYSSET_BORLS These bits are written by software to select the Low voltage reset voltage
  */
typedef enum
{
    MD_SYSCFG_SYS_BORLS_LV0 = (0x00000000UL),  /** @brief 0000: 2.0V */
    MD_SYSCFG_SYS_BORLS_LV1 = (0x00000001UL),  /** @brief 0001: 2.3V */
    MD_SYSCFG_SYS_BORLS_LV2 = (0x00000002UL),  /** @brief 0010: 2.6V */
    MD_SYSCFG_SYS_BORLS_LV3 = (0x00000003UL),  /** @brief 0011: 2.9V */
    MD_SYSCFG_SYS_BORLS_LV4 = (0x00000004UL),  /** @brief 0100: 3.2V */
    MD_SYSCFG_SYS_BORLS_LV5 = (0x00000005UL),  /** @brief 0101: 3.5V */
    MD_SYSCFG_SYS_BORLS_LV6 = (0x00000006UL),  /** @brief 0110: 3.8V */
    MD_SYSCFG_SYS_BORLS_LV7 = (0x00000007UL),  /** @brief 0111: 4.1V */
}md_syscfg_sys_bor_level_t;

/**
  * @brief MD_SYSCFG_CFG_DBGHEN DBG Halt enable bit
  */
typedef enum
{
    MD_SYSCFG_DBGHEN_CAN1     = (0x00004000UL),  /** @brief 1<<14: CAN1     */  
    MD_SYSCFG_DBGHEN_IWDT     = (0x00002000UL),  /** @brief 1<<13: IWDT     */
    MD_SYSCFG_DBGHEN_WWDT     = (0x00001000UL),  /** @brief 1<<12: WWDT     */
    MD_SYSCFG_DBGHEN_I2C1     = (0x00000400UL),  /** @brief 1<<10: I2C1     */
    MD_SYSCFG_DBGHEN_BS16T1   = (0x00000040UL),  /** @brief 1<<6 : BS16T1   */
    MD_SYSCFG_DBGHEN_GP32C4T2 = (0x00000020UL),  /** @brief 1<<5 : GP32C4T2 */
    MD_SYSCFG_DBGHEN_GP32C2T1 = (0x00000010UL),  /** @brief 1<<4 : GP32C2T1 */
    MD_SYSCFG_DBGHEN_GP16C2T2 = (0x00000008UL),  /** @brief 1<<3 : GP16C2T2 */
    MD_SYSCFG_DBGHEN_GP16C2T1 = (0x00000004UL),  /** @brief 1<<2 : GP16C2T1 */
    MD_SYSCFG_DBGHEN_AD16C6T2 = (0x00000002UL),  /** @brief 1<<1 : AD16C6T2 */
    MD_SYSCFG_DBGHEN_AD16C6T1 = (0x00000001UL),  /** @brief 1<<0 : AD16C6T1 */
}md_syscfg_dbghen_t;

/**
  * @brief MD_SYSCFG_PWRCON_BORLS These bits are written by software to select the Low voltage reset voltage
  */
typedef enum
{
    MD_SYSCFG_BORLS_LV0 = (0x00000000UL),  /** @brief 0000: 2.0V */
    MD_SYSCFG_BORLS_LV1 = (0x00000001UL),  /** @brief 0001: 2.3V */
    MD_SYSCFG_BORLS_LV2 = (0x00000002UL),  /** @brief 0010: 2.6V */
    MD_SYSCFG_BORLS_LV3 = (0x00000003UL),  /** @brief 0011: 2.9V */
    MD_SYSCFG_BORLS_LV4 = (0x00000004UL),  /** @brief 0100: 3.2V */
    MD_SYSCFG_BORLS_LV5 = (0x00000005UL),  /** @brief 0101: 3.5V */
    MD_SYSCFG_BORLS_LV6 = (0x00000006UL),  /** @brief 0110: 3.8V */
    MD_SYSCFG_BORLS_LV7 = (0x00000007UL),  /** @brief 0111: 4.1V */
}md_syscfg_bor_level_t;

/**
  * @brief MD_SYSCFG_PWRCON_LVDLS These bits are Low voltage detection voltage range selection
  */
typedef enum
{
    MD_SYSCFG_LVDLS_R1V9_F2V  = (0x00000000UL),  /** @brief 0000: R-1.9V, F-2.0V */
    MD_SYSCFG_LVDLS_R2V1_F2V2 = (0x00000001UL),  /** @brief 0001: R-2.1V, F-2.2V */
    MD_SYSCFG_LVDLS_R2V3_F2V4 = (0x00000002UL),  /** @brief 0010: R-2.3V, F-2.4V */
    MD_SYSCFG_LVDLS_R2V5_F2V6 = (0x00000003UL),  /** @brief 0011: R-2.5V, F-2.6V */
    MD_SYSCFG_LVDLS_R2V7_F2V8 = (0x00000004UL),  /** @brief 0100: R-2.7V, F-2.8V */
    MD_SYSCFG_LVDLS_R2V9_F3V  = (0x00000005UL),  /** @brief 0101: R-2.9V, F-3.0V */
    MD_SYSCFG_LVDLS_R3V1_F3V2 = (0x00000006UL),  /** @brief 0110: R-3.1V, F-3.2V */
    MD_SYSCFG_LVDLS_R3V3_F3V4 = (0x00000007UL),  /** @brief 0111: R-3.3V, F-3.4V */
    MD_SYSCFG_LVDLS_R3V5_F3V6 = (0x00000008UL),  /** @brief 1000: R-3.5V, F-3.6V */
    MD_SYSCFG_LVDLS_R3V7_F3V8 = (0x00000009UL),  /** @brief 1001: R-3.7V, F-3.8V */
    MD_SYSCFG_LVDLS_R3V9_F4V  = (0x0000000AUL),  /** @brief 1010: R-3.9V, F-4.0V */
    MD_SYSCFG_LVDLS_R4V1_F4V2 = (0x0000000BUL),  /** @brief 1011: R-4.1V, F-4.2V */
    MD_SYSCFG_LVDLS_R4V3_F4V4 = (0x0000000CUL),  /** @brief 1100: R-4.3V, F-4.4V */
    MD_SYSCFG_LVDLS_R4V5_F4V6 = (0x0000000DUL),  /** @brief 1101: R-4.5V, F-4.6V */
    MD_SYSCFG_LVDLS_R4V7_F4V8 = (0x0000000EUL),  /** @brief 1110: R-4.7V, F-4.8V */
    MD_SYSCFG_LVDLS_R4V9_F5V  = (0x0000000FUL),  /** @brief 1111: R-4.9V, F-5.0V */
}md_syscfg_lvd_level_t;

/**
  * @brief MD_SYSCFG_WKUP_WKEG WKEGx:WKUPx pin edge sensitive type select
  */
typedef enum
{
    MD_SYSCFG_WKEG_WKUP8 =(0x00000800UL),   /** @brief 1<<11    */
    MD_SYSCFG_WKEG_IWDT  =(0x00000400UL),   /** @brief 1<<10    */
    MD_SYSCFG_WKEG_LVD   =(0x00000200UL),   /** @brief 1<<9     */
    MD_SYSCFG_WKEG_NRST  =(0x00000100UL),   /** @brief 1<<8     */
    MD_SYSCFG_WKEG_WKUP7 =(0x00000080UL),   /** @brief 1<<7     */
    MD_SYSCFG_WKEG_WKUP6 =(0x00000040UL),   /** @brief 1<<6     */
    MD_SYSCFG_WKEG_WKUP5 =(0x00000020UL),   /** @brief 1<<5     */
    MD_SYSCFG_WKEG_WKUP4 =(0x00000010UL),   /** @brief 1<<4     */
    MD_SYSCFG_WKEG_WKUP3 =(0x00000008UL),   /** @brief 1<<3     */
    MD_SYSCFG_WKEG_WKUP2 =(0x00000004UL),   /** @brief 1<<2     */
    MD_SYSCFG_WKEG_WKUP1 =(0x00000002UL),   /** @brief 1<<1     */
    MD_SYSCFG_WKEG_WKUP0 =(0x00000001UL),   /** @brief 1<<0     */
}md_syscfg_wakeup_mode_t;

/**
  * @brief MD_SYSCFG_WKUP_WKEN WKENx: Enable WKUPx pin or wakeup event
  */
typedef enum
{
    MD_SYSCFG_WKEN_WKUP8 = (0x00000800UL),   /** @brief 1<<11    */
    MD_SYSCFG_WKEN_IWDT  = (0x00000400UL),   /** @brief 1<<10    */
    MD_SYSCFG_WKEN_LVD   = (0x00000200UL),   /** @brief 1<<9     */
    MD_SYSCFG_WKEN_NRST  = (0x00000100UL),   /** @brief 1<<8     */
    MD_SYSCFG_WKEN_WKUP7 = (0x00000080UL),   /** @brief 1<<7     */
    MD_SYSCFG_WKEN_WKUP6 = (0x00000040UL),   /** @brief 1<<6     */
    MD_SYSCFG_WKEN_WKUP5 = (0x00000020UL),   /** @brief 1<<5     */
    MD_SYSCFG_WKEN_WKUP4 = (0x00000010UL),   /** @brief 1<<4     */
    MD_SYSCFG_WKEN_WKUP3 = (0x00000008UL),   /** @brief 1<<3     */
    MD_SYSCFG_WKEN_WKUP2 = (0x00000004UL),   /** @brief 1<<2     */
    MD_SYSCFG_WKEN_WKUP1 = (0x00000002UL),   /** @brief 1<<1     */
    MD_SYSCFG_WKEN_WKUP0 = (0x00000001UL),   /** @brief 1<<0     */
}md_syscfg_wakeup_enable_t;

/**
  * @brief MD_SYSCFG_WKSR_LPLS Low Power Level Select
  */
typedef enum
{
    MD_SYSCFG_LPLS_STOP_MODE    = (0x00000001UL),  /** @brief 1 : Level 1, Stop Mode    */
}md_syscfg_lowpower_mode_t;

/**
  * @brief MD_SYSCFG_WKSR_LPVSEL Low Power LDO Voltage Selection
  */
typedef enum
{
    MD_SYSCFG_LPVSEL_1P4V = (0x00000001UL),  /** @brief 1 : 1.4V */
    MD_SYSCFG_LPVSEL_1P5V = (0x00000002UL),  /** @brief 2 : 1.5V */
}md_syscfg_ldo_lowpower_mode_voltage_t;

/**
  * @brief MD_SYSCFG_WKSR_FG FGx: WKUPx pin or wakeup event flag
  */
typedef enum
{
    MD_SYSCFG_FG_WKUP8 = (0x00000800UL),   /** @brief 1<<11    */
    MD_SYSCFG_FG_IWDT  = (0x00000400UL),   /** @brief 1<<10    */
    MD_SYSCFG_FG_LVD   = (0x00000200UL),   /** @brief 1<<9     */
    MD_SYSCFG_FG_NRST  = (0x00000100UL),   /** @brief 1<<8     */
    MD_SYSCFG_FG_WKUP7 = (0x00000080UL),   /** @brief 1<<7     */
    MD_SYSCFG_FG_WKUP6 = (0x00000040UL),   /** @brief 1<<6     */
    MD_SYSCFG_FG_WKUP5 = (0x00000020UL),   /** @brief 1<<5     */
    MD_SYSCFG_FG_WKUP4 = (0x00000010UL),   /** @brief 1<<4     */
    MD_SYSCFG_FG_WKUP3 = (0x00000008UL),   /** @brief 1<<3     */
    MD_SYSCFG_FG_WKUP2 = (0x00000004UL),   /** @brief 1<<2     */
    MD_SYSCFG_FG_WKUP1 = (0x00000002UL),   /** @brief 1<<1     */
    MD_SYSCFG_FG_WKUP0 = (0x00000001UL),   /** @brief 1<<0     */
}md_syscfg_wakeup_event_t;

/**
  * @}
  */

/* Exported Functions -------------------------------------------------------- */

/** @defgroup MD_SYSCFG_Public_Functions SYSCFG Public Functions
  * @{
  */

/** @defgroup REMAP
  * @{
  */
/**
  * @brief  Set SYSCFG_REMAP Register
  * @param  syscfg SYSCFG Instance
  * @param  remap
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_remap(SYSCFG_TypeDef *syscfg, uint32_t remap)
{
    WRITE_REG(syscfg->REMAP, remap);
}

/**
  * @brief  Get SYSCFG_REMAP Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_remap(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)READ_REG(syscfg->REMAP);
}

/**
  * @brief  Get Current flash start address status
  * @note   If set 0x1, it means second 4k Byte, If set 0x2, it means third 4k Byte, and so on.
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 31
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_flash_remap_base_real(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->REMAP, SYSCFG_REMAP_REALBASE) >> SYSCFG_REMAP_REALBASE_POSS);
}

/**
  * @brief  Get Current Memory mapping Status
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_SYSCFG_REALMOD_MAIN
  *         @arg @ref MD_SYSCFG_REALMOD_BOOTROM
  *         @arg @ref MD_SYSCFG_REALMOD_SRAM
  */
__STATIC_INLINE uint32_t md_syscfg_get_memory_mapping_real(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->REMAP, SYSCFG_REMAP_REALMOD) >> SYSCFG_REMAP_REALMOD_POSS);
}
/**
  * @brief  Set Remap Main Flash Base Address Selection
  * @note   If set 0x1, it means second 4k Byte, If set 0x2, it means third 4k Byte, and so on.
  * @param  syscfg SYSCFG Instance
  * @param  efbase This parameter can be one of the following values:
  *         @arg Max Value 31
  *         @arg Min Value 0
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_flash_remap_base(SYSCFG_TypeDef *syscfg, uint32_t efbase)
{
    MODIFY_REG(syscfg->REMAP, SYSCFG_REMAP_EFBASE, (efbase << SYSCFG_REMAP_EFBASE_POSS));
}
/**
  * @brief  Get Remap Main Flash Base Address Selection
  * @note   If read 0x1, it means second 4k Byte, If read 0x2, it means third 4k Byte, and so on.
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 31
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_flash_remap_base(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->REMAP, SYSCFG_REMAP_EFBASE) >> SYSCFG_REMAP_EFBASE_POSS);
}
/**
  * @brief  Set Memory mapping selection bits
  * @param  syscfg SYSCFG Instance
  * @param  memmod This parameter can be one of the following values:
  *         @arg @ref MD_SYSCFG_MEMMOD_MAIN
  *         @arg @ref MD_SYSCFG_MEMMOD_BOOTROM
  *         @arg @ref MD_SYSCFG_MEMMOD_SRAM
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_memory_mapping(SYSCFG_TypeDef *syscfg, md_syscfg_memory_mode_t memmod)
{
    MODIFY_REG(syscfg->REMAP, SYSCFG_REMAP_MEMMOD, memmod << SYSCFG_REMAP_MEMMOD_POSS);
}
/**
  * @brief  Get Memory mapping selection bits
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_SYSCFG_MEMMOD_MAIN
  *         @arg @ref MD_SYSCFG_MEMMOD_BOOTROM
  *         @arg @ref MD_SYSCFG_MEMMOD_SRAM
  */
__STATIC_INLINE uint32_t md_syscfg_get_memory_mapping(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->REMAP, SYSCFG_REMAP_MEMMOD) >> SYSCFG_REMAP_MEMMOD_POSS);
}
/**
  * @brief  Set Start Remap bit
  * @note   This bit set High to start remapping process, that will be auto clear when process was finish.
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_memory_remap(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->REMAP, SYSCFG_REMAP_REMAP);
}
/**
  * @}
  */

/** @defgroup SYSTRIM
  * @{
  */
/**
  * @brief  Get SYSCFG_SYSTRIM Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_systrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->SYSTRIM));
}

/**
  * @brief  Get LRC clock calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_lrctrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->SYSTRIM, SYSCFG_SYSTRIM_LRCTRIM) >> SYSCFG_SYSTRIM_LRCTRIM_POSS);
}

/**
  * @brief  Get Programmable low voltage detection calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_lvdtrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->SYSTRIM, SYSCFG_SYSTRIM_LVDTRIM) >> SYSCFG_SYSTRIM_LVDTRIM_POSS);
}

/**
  * @brief  Get BandGap 1V calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_bg1vtrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->SYSTRIM, SYSCFG_SYSTRIM_BG1VTRIM) >> SYSCFG_SYSTRIM_BG1VTRIM_POSS);
}

/**
  * @brief  Get BandGap 1.2V calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_bg1v2trim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->SYSTRIM, SYSCFG_SYSTRIM_BG1V2TRIM) >> SYSCFG_SYSTRIM_BG1V2TRIM_POSS);
}

/**
  * @}
  */

/** @defgroup CLKTRIM1
  * @{
  */
/**
  * @brief  Get SYSCFG_CLKTRIM1 Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_clktrim1(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->CLKTRIM1));
}

/**
  * @brief  Get HRC 16M clock calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_hrc16mtrimh(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->CLKTRIM1, SYSCFG_CLKTRIM1_HRC16MTRIMH) >> SYSCFG_CLKTRIM1_HRC16MTRIMH_POSS);
}

/**
  * @}
  */

/** @defgroup OSCTRIM
  * @{
  */
/**
  * @brief  Get SYSCFG_OSCTRIM Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_osctrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->OSCTRIM));
}

/**
  * @brief  Get HOSC clock stable count value
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_SYSCFG_HOSCRCNT_256  
  *         @arg @ref MD_SYSCFG_HOSCRCNT_512  
  *         @arg @ref MD_SYSCFG_HOSCRCNT_1024 
  *         @arg @ref MD_SYSCFG_HOSCRCNT_2048 
  *         @arg @ref MD_SYSCFG_HOSCRCNT_4096 
  *         @arg @ref MD_SYSCFG_HOSCRCNT_8192 
  *         @arg @ref MD_SYSCFG_HOSCRCNT_16384
  *         @arg @ref MD_SYSCFG_HOSCRCNT_32768
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_hosc_ready_count(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OSCTRIM, SYSCFG_OSCTRIM_HOSCRCNT) >> SYSCFG_OSCTRIM_HOSCRCNT_POSS);
}

/**
  * @brief  Get HOSC current selection
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_hosc_current_select(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OSCTRIM, SYSCFG_OSCTRIM_HOSCCURSEL) >> SYSCFG_OSCTRIM_HOSCCURSEL_POSS);
}
/**
  * @}
  */

/** @defgroup CMPTRIM
  * @{
  */
/**
  * @brief  Get SYSCFG_CMPTRIM Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_cmptrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->CMPTRIM));
}

/**
  * @brief  Get CMP1 PMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_cmp1ptrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->CMPTRIM, SYSCFG_CMPTRIM_CMP1PTRIM) >> SYSCFG_CMPTRIM_CMP1PTRIM_POSS);
}

/**
  * @brief  Get CMP1 NMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_cmp1ntrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->CMPTRIM, SYSCFG_CMPTRIM_CMP1NTRIM) >> SYSCFG_CMPTRIM_CMP1NTRIM_POSS);
}

/**
  * @brief  Get CMP2 PMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_cmp2ptrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->CMPTRIM, SYSCFG_CMPTRIM_CMP2PTRIM) >> SYSCFG_CMPTRIM_CMP2PTRIM_POSS);
}

/**
  * @brief  Get CMP2 NMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_cmp2ntrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->CMPTRIM, SYSCFG_CMPTRIM_CMP2NTRIM) >> SYSCFG_CMPTRIM_CMP2NTRIM_POSS);
}
/**
  * @}
  */
  
/** @defgroup OPATRIM1
  * @{
  */
/**
  * @brief  Get SYSCFG_OPATRIM1 Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_opatrim1(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->OPATRIM1));
}

/**
  * @brief  Get OPAMP1 PMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_opamp1ptrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OPATRIM1, SYSCFG_OPATRIM1_OPA1PTRIM) >> SYSCFG_OPATRIM1_OPA1PTRIM_POSS);
}

/**
  * @brief  Get OPAMP1 NMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_opamp1ntrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OPATRIM1, SYSCFG_OPATRIM1_OPA1NTRIM) >> SYSCFG_OPATRIM1_OPA1NTRIM_POSS);
}

/**
  * @brief  Get OPAMP2 PMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_opamp2ptrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OPATRIM1, SYSCFG_OPATRIM1_OPA2PTRIM) >> SYSCFG_OPATRIM1_OPA2PTRIM_POSS);
}

/**
  * @brief  Get OPAMP2 NMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_opamp2ntrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OPATRIM1, SYSCFG_OPATRIM1_OPA2NTRIM) >> SYSCFG_OPATRIM1_OPA2NTRIM_POSS);
}
/**
  * @}
  */  

/** @defgroup OPATRIM2
  * @{
  */
/**
  * @brief  Get SYSCFG_OPATRIM2 Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_opatrim2(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->OPATRIM2));
}

/**
  * @brief  Get OPAMP1 PMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_opamp3ptrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OPATRIM2, SYSCFG_OPATRIM2_OPA3PTRIM) >> SYSCFG_OPATRIM2_OPA3PTRIM_POSS);
}

/**
  * @brief  Get OPAMP1 NMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_opamp3ntrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OPATRIM2, SYSCFG_OPATRIM2_OPA3NTRIM) >> SYSCFG_OPATRIM2_OPA3NTRIM_POSS);
}

/**
  * @brief  Get OPAMP2 PMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_opamp4ptrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OPATRIM2, SYSCFG_OPATRIM2_OPA4PTRIM) >> SYSCFG_OPATRIM2_OPA4PTRIM_POSS);
}

/**
  * @brief  Get OPAMP2 NMOS calibration value
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_option_opamp4ntrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->OPATRIM2, SYSCFG_OPATRIM2_OPA4NTRIM) >> SYSCFG_OPATRIM2_OPA4NTRIM_POSS);
}
/**
  * @}
  */ 
  
/** @defgroup ADCVREF
  * @{
  */
/**
  * @brief  Get SYSCFG_ADCVREF Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_adc_vref(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->ADCVREF, SYSCFG_ADCVREF_ADCVREF) >> SYSCFG_ADCVREF_ADCVREF_POSS);
} 
/**
  * @}
  */ 
  
/** @defgroup ADCTEMP
  * @{
  */
/**
  * @brief  Get SYSCFG_ADCTEMP Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_adc_temp(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->ADCTEMP));
}    
  
/**
  * @brief  Get ADC temperature sensor high temperature measurement value
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_adc_temp_high(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->ADCTEMP, SYSCFG_ADCTEMP_ADCTEMPH) >> SYSCFG_ADCTEMP_ADCTEMPH_POSS);
} 

/**
  * @brief  Get ADC temperature sensor low temperature measurement value
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_adc_temp_low(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->ADCTEMP, SYSCFG_ADCTEMP_ADCTEMPL) >> SYSCFG_ADCTEMP_ADCTEMPL_POSS);
} 
/**
  * @}
  */   



/** @defgroup MEMMOD
  * @{
  */
/**
  * @brief  Get SYSCFG_MEMMOD Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_memmod(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->MEMMOD));
}

/**
  * @brief  Get memory size
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_SMOD_128KB
            @arg @ref MD_SYSCFG_SMOD_64KB
            @arg @ref MD_SYSCFG_SMOD_32KB
            @arg @ref MD_SYSCFG_SMOD_16KB
            @arg @ref MD_SYSCFG_SMOD_8KB
            @arg @ref MD_SYSCFG_SMOD_4KB
            @arg @ref MD_SYSCFG_SMOD_2KB
  */
__STATIC_INLINE uint32_t md_syscfg_get_sram_size(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->MEMMOD, SYSCFG_MEMMOD_SMOD) >> SYSCFG_MEMMOD_SMOD_POSS);
}

/**
  * @brief  Get flash size
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_FMOD_512KB
            @arg @ref MD_SYSCFG_FMOD_256KB
            @arg @ref MD_SYSCFG_FMOD_128KB
            @arg @ref MD_SYSCFG_FMOD_64KB
            @arg @ref MD_SYSCFG_FMOD_32KB
            @arg @ref MD_SYSCFG_FMOD_16KB
  */
__STATIC_INLINE uint32_t md_syscfg_get_flash_size(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->MEMMOD, SYSCFG_MEMMOD_FMOD) >> SYSCFG_MEMMOD_FMOD_POSS);
}
/**
  * @}
  */

/** @defgroup SYSSET
  * @{
  */
/**
  * @brief  Get SYSCFG_SYSSET Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_sysset(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->SYSSET));
}

/**
  * @brief  Get User configuration word IWDT switch
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_iwdt_hardware_enable(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->SYSSET, SYSCFG_SYSSET_SYS_IWDTEN) >> SYSCFG_SYSSET_SYS_IWDTEN_POSS);
}

/**
  * @brief  Get User configuration word IWDT switch
  * @param  syscfg SYSCFG Instance
  * @retval
  */
__STATIC_INLINE uint32_t md_syscfg_get_bor_hardware_enable(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->SYSSET, SYSCFG_SYSSET_SYS_BOREN) >> SYSCFG_SYSSET_SYS_BOREN_POSS);
}

/**
  * @brief  Get User configuration word IWDT switch
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_SYS_BORLS_LV0
            @arg @ref MD_SYSCFG_SYS_BORLS_LV1
            @arg @ref MD_SYSCFG_SYS_BORLS_LV2
            @arg @ref MD_SYSCFG_SYS_BORLS_LV3
            @arg @ref MD_SYSCFG_SYS_BORLS_LV4
            @arg @ref MD_SYSCFG_SYS_BORLS_LV5
            @arg @ref MD_SYSCFG_SYS_BORLS_LV6
            @arg @ref MD_SYSCFG_SYS_BORLS_LV7
  */
__STATIC_INLINE uint32_t md_syscfg_get_bor_level_hardware_select(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->SYSSET, SYSCFG_SYSSET_SYS_BORLS) >> SYSCFG_SYSSET_SYS_BORLS_POSS);
}
/**
  * @}
  */
  
/** @defgroup APWR
  * @{
  */
  
/**
  * @brief  Set SYSCFG_APWR Register
  * @param  syscfg SYSCFG Instance
  * @param  apwr  
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_apwr(SYSCFG_TypeDef *syscfg, uint32_t apwr)
{
    WRITE_REG(syscfg->APWR, apwr);
}
  
/**
  * @brief  Get SYSCFG_APWR Register
  * @param  syscfg SYSCFG Instance
  * @retval apwr
  */
__STATIC_INLINE uint32_t md_syscfg_get_apwr(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->APWR));
}

/**
  * @brief  ADC, CMP reference voltage power switch enable.
  * @param  syscfg SYSCFG Instance
  * @retval None.
  */
__STATIC_INLINE void md_syscfg_enable_adc_cmp_vref(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->APWR, SYSCFG_APWR_VREFINT_EN);
}

/**
  * @brief  ADC, CMP reference voltage power switch disable.
  * @param  syscfg SYSCFG Instance
  * @retval None.
  */
__STATIC_INLINE void md_syscfg_disable_adc_cmp_vref(SYSCFG_TypeDef *syscfg)
{
    CLEAR_BIT(syscfg->APWR, SYSCFG_APWR_VREFINT_EN);
}

/**
  * @brief  Check if ADC, CMP reference voltage power switch is enabled.
  * @param  syscfg SYSCFG Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_adc_cmp_vref(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->APWR, SYSCFG_APWR_VREFINT_EN) == (SYSCFG_APWR_VREFINT_EN));
}

/**
  * @brief  temperature sensor switch enable.
  * @param  syscfg SYSCFG Instance
  * @retval None.
  */
__STATIC_INLINE void md_syscfg_enable_temp_sensor(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->APWR, SYSCFG_APWR_VTSENSE_EN);
}

/**
  * @brief  temperature sensor switch disable.
  * @param  syscfg SYSCFG Instance
  * @retval None.
  */
__STATIC_INLINE void md_syscfg_disable_temp_sensor(SYSCFG_TypeDef *syscfg)
{
    CLEAR_BIT(syscfg->APWR, SYSCFG_APWR_VTSENSE_EN);
}

/**
  * @brief  Check if temperature sensor switch is enabled.
  * @param  syscfg SYSCFG Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_temp_sensor(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->APWR, SYSCFG_APWR_VTSENSE_EN) == (SYSCFG_APWR_VTSENSE_EN));
}

/**
  * @brief  VDDA power domain power switch enable.
  * @param  syscfg SYSCFG Instance
  * @retval None.
  */
__STATIC_INLINE void md_syscfg_enable_vdda(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->APWR, SYSCFG_APWR_VDDA_PWREN);
}

/**
  * @brief  VDDA power domain power switch disable.
  * @param  syscfg SYSCFG Instance
  * @retval None.
  */
__STATIC_INLINE void md_syscfg_disable_vdda(SYSCFG_TypeDef *syscfg)
{
    CLEAR_BIT(syscfg->APWR, SYSCFG_APWR_VDDA_PWREN);
}

/**
  * @brief  Check if VDDA power domain power switch is enabled.
  * @param  syscfg SYSCFG Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_vdda(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->APWR, SYSCFG_APWR_VDDA_PWREN) == (SYSCFG_APWR_VDDA_PWREN));
}
/**
  * @}
  */  

/** @defgroup CFG
  * @{
  */
/**
  * @brief  Set SYSCFG_CFG Register
  * @param  syscfg SYSCFG Instance
  * @param  cfg
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_cfg(SYSCFG_TypeDef *syscfg, uint32_t cfg)
{
    WRITE_REG(syscfg->CFG, cfg);
}

/**
  * @brief  Get SYSCFG_CFG Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_cfg(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->CFG));
}

/**
  * @brief  Set DBG Halt enable bit
  * @note   Set bit to stopped the counter clock when the core is halted
  * @param  syscfg SYSCFG Instance
  * @param  dbghen This parameter can be one of the following values:
            @arg @ref MD_SYSCFG_DBGHEN_CAN1    
            @arg @ref MD_SYSCFG_DBGHEN_IWDT    
            @arg @ref MD_SYSCFG_DBGHEN_WWDT    
            @arg @ref MD_SYSCFG_DBGHEN_I2C1    
            @arg @ref MD_SYSCFG_DBGHEN_BS16T1  
            @arg @ref MD_SYSCFG_DBGHEN_GP32C4T2
            @arg @ref MD_SYSCFG_DBGHEN_GP32C2T1
            @arg @ref MD_SYSCFG_DBGHEN_GP16C2T2
            @arg @ref MD_SYSCFG_DBGHEN_GP16C2T1
            @arg @ref MD_SYSCFG_DBGHEN_AD16C6T2
            @arg @ref MD_SYSCFG_DBGHEN_AD16C6T1
  * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_debug_hault(SYSCFG_TypeDef *syscfg, md_syscfg_dbghen_t dbghen)
{
    SET_BIT(syscfg->CFG, dbghen << SYSCFG_CFG_DBGHEN_POSS);
}

/**
  * @brief  Set DBG Halt disable bit
  * @note   Set bit to let the counter clock continuous output when the core is halted
  * @param  syscfg SYSCFG Instance
  * @param  dbghen This parameter can be one of the following values:
            @arg @ref MD_SYSCFG_DBGHEN_CAN1    
            @arg @ref MD_SYSCFG_DBGHEN_IWDT    
            @arg @ref MD_SYSCFG_DBGHEN_WWDT    
            @arg @ref MD_SYSCFG_DBGHEN_I2C1    
            @arg @ref MD_SYSCFG_DBGHEN_BS16T1  
            @arg @ref MD_SYSCFG_DBGHEN_GP32C4T2
            @arg @ref MD_SYSCFG_DBGHEN_GP32C2T1
            @arg @ref MD_SYSCFG_DBGHEN_GP16C2T2
            @arg @ref MD_SYSCFG_DBGHEN_GP16C2T1
            @arg @ref MD_SYSCFG_DBGHEN_AD16C6T2
            @arg @ref MD_SYSCFG_DBGHEN_AD16C6T1
  * @retval None
  */
__STATIC_INLINE void md_syscfg_disable_debug_hault(SYSCFG_TypeDef *syscfg, md_syscfg_dbghen_t dbghen)
{
    CLEAR_BIT(syscfg->CFG, dbghen << SYSCFG_CFG_DBGHEN_POSS);
}

/**
  * @brief  Get DBG Halt enable bit
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_DBGHEN_CAN1    
            @arg @ref MD_SYSCFG_DBGHEN_IWDT    
            @arg @ref MD_SYSCFG_DBGHEN_WWDT    
            @arg @ref MD_SYSCFG_DBGHEN_I2C1    
            @arg @ref MD_SYSCFG_DBGHEN_BS16T1  
            @arg @ref MD_SYSCFG_DBGHEN_GP32C4T2
            @arg @ref MD_SYSCFG_DBGHEN_GP32C2T1
            @arg @ref MD_SYSCFG_DBGHEN_GP16C2T2
            @arg @ref MD_SYSCFG_DBGHEN_GP16C2T1
            @arg @ref MD_SYSCFG_DBGHEN_AD16C6T2
            @arg @ref MD_SYSCFG_DBGHEN_AD16C6T1
  */
__STATIC_INLINE uint32_t md_syscfg_get_debug_hault(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->CFG, SYSCFG_CFG_DBGHEN) >> SYSCFG_CFG_DBGHEN_POSS);
}

/**
  * @brief  DBG Halt bit is enabled
  * @param  syscfg SYSCFG Instance
            @arg @ref MD_SYSCFG_DBGHEN_CAN1    
            @arg @ref MD_SYSCFG_DBGHEN_IWDT    
            @arg @ref MD_SYSCFG_DBGHEN_WWDT    
            @arg @ref MD_SYSCFG_DBGHEN_I2C1    
            @arg @ref MD_SYSCFG_DBGHEN_BS16T1  
            @arg @ref MD_SYSCFG_DBGHEN_GP32C4T2
            @arg @ref MD_SYSCFG_DBGHEN_GP32C2T1
            @arg @ref MD_SYSCFG_DBGHEN_GP16C2T2
            @arg @ref MD_SYSCFG_DBGHEN_GP16C2T1
            @arg @ref MD_SYSCFG_DBGHEN_AD16C6T2
            @arg @ref MD_SYSCFG_DBGHEN_AD16C6T1
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : IP debug hault enable
  *         @arg 0x0 : IP debug hault disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_debug_hault(SYSCFG_TypeDef *syscfg, md_syscfg_dbghen_t dbgh)
{
    return (READ_BIT(syscfg->CFG, dbgh << SYSCFG_CFG_DBGHEN_POSS) == (dbgh << SYSCFG_CFG_DBGHEN_POSS));
}

/**
  * @brief  Low voltage detection (LVD) event input switch enable
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_timer_break_lvd(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->CFG, SYSCFG_CFG_LVDLCK);
}
/**
  * @brief  Low voltage detection (LVD) event input switch disable
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_disable_timer_break_lvd(SYSCFG_TypeDef *syscfg)
{
    CLEAR_BIT(syscfg->CFG, SYSCFG_CFG_LVDLCK);
}
/**
  * @brief  Check if Low voltage detection (LVD) event input switch is enabled
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : LVD Timer break function enable
  *         @arg 0x0 : LVD Timer break function disable
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_timer_break_lvd(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->CFG, SYSCFG_CFG_LVDLCK) == (SYSCFG_CFG_LVDLCK));
}

/**
  * @brief  Clock Security Event (CSS) input switch enable
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_timer_break_css(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->CFG, SYSCFG_CFG_CSSLCK);
}
/**
  * @brief  Clock Security Event (CSS) input switch disable
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_disable_timer_break_css(SYSCFG_TypeDef *syscfg)
{
    CLEAR_BIT(syscfg->CFG, SYSCFG_CFG_CSSLCK);
}
/**
  * @brief  Check if Clock Security Event (CSS) input switch is enabled
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : CSS Timer break function enable
  *         @arg 0x0 : CSS Timer break function disable
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_timer_break_css(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->CFG, SYSCFG_CFG_CSSLCK) == (SYSCFG_CFG_CSSLCK));
}

/**
  * @brief  CPU Lockup event input switch enable
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_timer_break_cpu_lockup(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->CFG, SYSCFG_CFG_CPULCK);
}
/**
  * @brief  CPU Lockup event input switch disable
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_disable_timer_break_cpu_lockup(SYSCFG_TypeDef *syscfg)
{
    CLEAR_BIT(syscfg->CFG, SYSCFG_CFG_CPULCK);
}
/**
  * @brief  Check if CPU Lockup event input switch is enabled
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : CPU Lockup Timer break function enable
  *         @arg 0x0 : CPU Lockup Timer break function disable
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_timer_break_cpu_lockup(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->CFG, SYSCFG_CFG_CPULCK) == (SYSCFG_CFG_CPULCK));
}

/**
  * @brief  Get Backup register bus idle flag
  * @param  syscfg SYSCFG Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_syscfg_is_active_flag_backup_bus_idle(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->CFG, SYSCFG_CFG_BKREADY) == (SYSCFG_CFG_BKREADY));
}
/**
  * @}
  */

/** @defgroup PWRCON
  * @{
  */
/**
  * @brief  Set SYSCFG_PWRCON register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_pwrcon(SYSCFG_TypeDef *syscfg, uint32_t pwrcon)
{
    WRITE_REG(syscfg->PWRCON, pwrcon);
}

/**
  * @brief  Get SYSCFG_PWRCON Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_pwrcon(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->PWRCON));
}

/**
  * @brief  Get IWDT state
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : IWDT enable
  *         @arg 0x0 : IWDT disable
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_iwdt(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->PWRCON, SYSCFG_PWRCON_IWDTEN) == (SYSCFG_PWRCON_IWDTEN));
}

/**
  * @brief  Check if Low voltage reset switch is enabled
  * @param  syscfg SYSCFG Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_bor(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->PWRCON, SYSCFG_PWRCON_BOREN) == (SYSCFG_PWRCON_BOREN));
}

/**
  * @brief  Get Low voltage reset voltage interval selection
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_BORLS_LV0
            @arg @ref MD_SYSCFG_BORLS_LV1
            @arg @ref MD_SYSCFG_BORLS_LV2
            @arg @ref MD_SYSCFG_BORLS_LV3
            @arg @ref MD_SYSCFG_BORLS_LV4
            @arg @ref MD_SYSCFG_BORLS_LV5
            @arg @ref MD_SYSCFG_BORLS_LV6
            @arg @ref MD_SYSCFG_BORLS_LV7
  */
__STATIC_INLINE uint32_t md_syscfg_get_bor_level(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->PWRCON, SYSCFG_PWRCON_BORLS) >> SYSCFG_PWRCON_BORLS_POSS);
}

/**
  * @brief  Low voltage detection switch enable
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_lvd(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->PWRCON, SYSCFG_PWRCON_LVDEN);
}
/**
  * @brief  Low voltage detection switch disable
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_disable_lvd(SYSCFG_TypeDef *syscfg)
{
    CLEAR_BIT(syscfg->PWRCON, SYSCFG_PWRCON_LVDEN);
}
/**
  * @brief  Check if Low voltage detection switch is enabled
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : LVD enable
  *         @arg 0x0 : LVD disable
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_lvd(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->PWRCON, SYSCFG_PWRCON_LVDEN) == (SYSCFG_PWRCON_LVDEN));
}

/**
  * @brief  Set Low voltage detection voltage range selection
  * @param  syscfg SYSCFG Instance
  * @param  lvdls Low voltage detection voltage range selection
            @arg @ref MD_SYSCFG_LVDLS_R1V9_F2V
            @arg @ref MD_SYSCFG_LVDLS_R2V1_F2V2
            @arg @ref MD_SYSCFG_LVDLS_R2V3_F2V4
            @arg @ref MD_SYSCFG_LVDLS_R2V5_F2V6
            @arg @ref MD_SYSCFG_LVDLS_R2V7_F2V8
            @arg @ref MD_SYSCFG_LVDLS_R2V9_F3V
            @arg @ref MD_SYSCFG_LVDLS_R3V1_F3V2
            @arg @ref MD_SYSCFG_LVDLS_R3V3_F3V4
            @arg @ref MD_SYSCFG_LVDLS_R3V5_F3V6
            @arg @ref MD_SYSCFG_LVDLS_R3V7_F3V8
            @arg @ref MD_SYSCFG_LVDLS_R3V9_F4V
            @arg @ref MD_SYSCFG_LVDLS_R4V1_F4V2
            @arg @ref MD_SYSCFG_LVDLS_R4V3_F4V4
            @arg @ref MD_SYSCFG_LVDLS_R4V5_F4V6
            @arg @ref MD_SYSCFG_LVDLS_R4V7_F4V8
            @arg @ref MD_SYSCFG_LVDLS_R4V9_F5V
            @arg @ref MD_SYSCFG_LVDLS_R4V9_F5V
  * @retval None
*/
__STATIC_INLINE void md_syscfg_set_lvd_level(SYSCFG_TypeDef *syscfg, md_syscfg_lvd_level_t lvdls)
{
    MODIFY_REG(syscfg->PWRCON, SYSCFG_PWRCON_LVDLS, lvdls << SYSCFG_PWRCON_LVDLS_POSS);
}

/**
  * @brief  Get Low voltage detection voltage range selection
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_LVDLS_R1V9_F2V
            @arg @ref MD_SYSCFG_LVDLS_R2V1_F2V2
            @arg @ref MD_SYSCFG_LVDLS_R2V3_F2V4
            @arg @ref MD_SYSCFG_LVDLS_R2V5_F2V6
            @arg @ref MD_SYSCFG_LVDLS_R2V7_F2V8
            @arg @ref MD_SYSCFG_LVDLS_R2V9_F3V
            @arg @ref MD_SYSCFG_LVDLS_R3V1_F3V2
            @arg @ref MD_SYSCFG_LVDLS_R3V3_F3V4
            @arg @ref MD_SYSCFG_LVDLS_R3V5_F3V6
            @arg @ref MD_SYSCFG_LVDLS_R3V7_F3V8
            @arg @ref MD_SYSCFG_LVDLS_R3V9_F4V
            @arg @ref MD_SYSCFG_LVDLS_R4V1_F4V2
            @arg @ref MD_SYSCFG_LVDLS_R4V3_F4V4
            @arg @ref MD_SYSCFG_LVDLS_R4V5_F4V6
            @arg @ref MD_SYSCFG_LVDLS_R4V7_F4V8
            @arg @ref MD_SYSCFG_LVDLS_R4V9_F5V
  */
__STATIC_INLINE uint32_t md_syscfg_get_lvd_level(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->PWRCON, SYSCFG_PWRCON_LVDLS) >> SYSCFG_PWRCON_LVDLS_POSS);
}
/**
  * @}
  */

/** @defgroup WKTRIM
  * @{
  */

/**
  * @brief  Get SYSCFG_WKTRIM Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_wktrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->WKTRIM));
}

/**
  * @brief  Get Low voltage detection (LVD) calibration value
  * @param  syscfg SYSCFG Instance
  * @retval LVD calibration value
  */
__STATIC_INLINE uint32_t md_syscfg_get_lvdtrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->WKTRIM, SYSCFG_WKTRIM_LVDTRIM) >> SYSCFG_WKTRIM_LVDTRIM_POSS);
}

/**
  * @brief  Get LRC clock calibration value
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 127
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_lrctrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->WKTRIM, SYSCFG_WKTRIM_LRCTRIM) >> SYSCFG_WKTRIM_LRCTRIM_POSS);
}

/**
  * @brief  Get BandGap 1.2V calibration value
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value F
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_bg1v2trim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->WKTRIM, SYSCFG_WKTRIM_BG1V2TRIM) >> SYSCFG_WKTRIM_BG1V2TRIM_POSS);
}

/**
  * @brief  Get BandGap 1V calibration value
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value F
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_bg1vtrim(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->WKTRIM, SYSCFG_WKTRIM_BG1VTRIM) >> SYSCFG_WKTRIM_BG1VTRIM_POSS);
}
/**
  * @}
  */

/** @defgroup WKUP
  * @{
  */
/**
  * @brief  Set SYSCFG_WKUP register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_wkup(SYSCFG_TypeDef *syscfg, uint32_t wkup)
{
    WRITE_REG(syscfg->WKUP, wkup);
}

/**
  * @brief  Get SYSCFG_WKUP Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_wkup(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->WKUP));
}

/**
  * @brief  Enable Wake-up event rising edge mode
  * @param  syscfg SYSCFG Instance
  * @param  wkeg Wake-up event rising edge or falling edge mode
            @arg @ref MD_SYSCFG_WKEG_WKUP8
            @arg @ref MD_SYSCFG_WKEG_IWDT 
            @arg @ref MD_SYSCFG_WKEG_LVD  
            @arg @ref MD_SYSCFG_WKEG_NRST 
            @arg @ref MD_SYSCFG_WKEG_WKUP7
            @arg @ref MD_SYSCFG_WKEG_WKUP6
            @arg @ref MD_SYSCFG_WKEG_WKUP5
            @arg @ref MD_SYSCFG_WKEG_WKUP4
            @arg @ref MD_SYSCFG_WKEG_WKUP3
            @arg @ref MD_SYSCFG_WKEG_WKUP2
            @arg @ref MD_SYSCFG_WKEG_WKUP1
            @arg @ref MD_SYSCFG_WKEG_WKUP0
   * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_wakeup_edge_rising(SYSCFG_TypeDef *syscfg, md_syscfg_wakeup_mode_t wkeg)
{
    SET_BIT(syscfg->WKUP, wkeg << SYSCFG_WKUP_WKEG_POSS);
}

/**
  * @brief  Enable Wake-up event falling edge mode
  * @param  syscfg SYSCFG Instance
  * @param  wkeg Wake-up event rising edge or falling edge mode
            @arg @ref MD_SYSCFG_WKEG_WKUP8
            @arg @ref MD_SYSCFG_WKEG_IWDT 
            @arg @ref MD_SYSCFG_WKEG_LVD  
            @arg @ref MD_SYSCFG_WKEG_NRST 
            @arg @ref MD_SYSCFG_WKEG_WKUP7
            @arg @ref MD_SYSCFG_WKEG_WKUP6
            @arg @ref MD_SYSCFG_WKEG_WKUP5
            @arg @ref MD_SYSCFG_WKEG_WKUP4
            @arg @ref MD_SYSCFG_WKEG_WKUP3
            @arg @ref MD_SYSCFG_WKEG_WKUP2
            @arg @ref MD_SYSCFG_WKEG_WKUP1
            @arg @ref MD_SYSCFG_WKEG_WKUP0
   * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_wakeup_edge_falling(SYSCFG_TypeDef *syscfg, md_syscfg_wakeup_mode_t wkeg)
{
    CLEAR_BIT(syscfg->WKUP, wkeg << SYSCFG_WKUP_WKEG_POSS);
}

/**
  * @brief  Get Wake-up event rising edge or falling edge mode
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_WKEG_WKUP8
            @arg @ref MD_SYSCFG_WKEG_IWDT 
            @arg @ref MD_SYSCFG_WKEG_LVD  
            @arg @ref MD_SYSCFG_WKEG_NRST 
            @arg @ref MD_SYSCFG_WKEG_WKUP7
            @arg @ref MD_SYSCFG_WKEG_WKUP6
            @arg @ref MD_SYSCFG_WKEG_WKUP5
            @arg @ref MD_SYSCFG_WKEG_WKUP4
            @arg @ref MD_SYSCFG_WKEG_WKUP3
            @arg @ref MD_SYSCFG_WKEG_WKUP2
            @arg @ref MD_SYSCFG_WKEG_WKUP1
            @arg @ref MD_SYSCFG_WKEG_WKUP0
  *         @arg 0x1 : rising edge enable
  *         @arg 0x0 : falling edge enable
  */
__STATIC_INLINE uint32_t md_syscfg_get_wakeup_edge(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->WKUP, SYSCFG_WKUP_WKEG) >> SYSCFG_WKUP_WKEG_POSS);
}

/**
  * @brief  Wake-up event rising edge is enabled
  * @param  syscfg SYSCFG Instance
            @arg @ref MD_SYSCFG_WKEG_WKUP8
            @arg @ref MD_SYSCFG_WKEG_IWDT 
            @arg @ref MD_SYSCFG_WKEG_LVD  
            @arg @ref MD_SYSCFG_WKEG_NRST 
            @arg @ref MD_SYSCFG_WKEG_WKUP7
            @arg @ref MD_SYSCFG_WKEG_WKUP6
            @arg @ref MD_SYSCFG_WKEG_WKUP5
            @arg @ref MD_SYSCFG_WKEG_WKUP4
            @arg @ref MD_SYSCFG_WKEG_WKUP3
            @arg @ref MD_SYSCFG_WKEG_WKUP2
            @arg @ref MD_SYSCFG_WKEG_WKUP1
            @arg @ref MD_SYSCFG_WKEG_WKUP0
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : rising edge enable
  *         @arg 0x0 : falling edge enable
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_wakeup_edge_rising(SYSCFG_TypeDef *syscfg, md_syscfg_wakeup_mode_t wkeg)
{
    return (READ_BIT(syscfg->WKUP, wkeg << SYSCFG_WKUP_WKEG_POSS) == (wkeg << SYSCFG_WKUP_WKEG_POSS));
}

/**
  * @brief  Wake-up event falling edge is enabled
  * @param  syscfg SYSCFG Instance
            The retval can be one of the following values:
            @arg @ref MD_SYSCFG_WKEG_WKUP8
            @arg @ref MD_SYSCFG_WKEG_IWDT 
            @arg @ref MD_SYSCFG_WKEG_LVD  
            @arg @ref MD_SYSCFG_WKEG_NRST 
            @arg @ref MD_SYSCFG_WKEG_WKUP7
            @arg @ref MD_SYSCFG_WKEG_WKUP6
            @arg @ref MD_SYSCFG_WKEG_WKUP5
            @arg @ref MD_SYSCFG_WKEG_WKUP4
            @arg @ref MD_SYSCFG_WKEG_WKUP3
            @arg @ref MD_SYSCFG_WKEG_WKUP2
            @arg @ref MD_SYSCFG_WKEG_WKUP1
            @arg @ref MD_SYSCFG_WKEG_WKUP0
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : falling edge enable
  *         @arg 0x0 : rising edge enable
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_wakeup_edge_falling(SYSCFG_TypeDef *syscfg, uint32_t wkeg)
{
    return (READ_BIT(syscfg->WKUP, wkeg << SYSCFG_WKUP_WKEG_POSS) == 0);
}

/**
  * @brief  Enable Wake-up pin or wake-up event switch
  * @param  syscfg SYSCFG Instance
  * @param  wken Wake-up pin or wake-up event switch
            @arg @ref MD_SYSCFG_WKEN_WKUP8
            @arg @ref MD_SYSCFG_WKEN_IWDT 
            @arg @ref MD_SYSCFG_WKEN_LVD  
            @arg @ref MD_SYSCFG_WKEN_NRST 
            @arg @ref MD_SYSCFG_WKEN_WKUP7
            @arg @ref MD_SYSCFG_WKEN_WKUP6
            @arg @ref MD_SYSCFG_WKEN_WKUP5
            @arg @ref MD_SYSCFG_WKEN_WKUP4
            @arg @ref MD_SYSCFG_WKEN_WKUP3
            @arg @ref MD_SYSCFG_WKEN_WKUP2
            @arg @ref MD_SYSCFG_WKEN_WKUP1
            @arg @ref MD_SYSCFG_WKEN_WKUP0
  * @retval None
  */
__STATIC_INLINE void md_syscfg_enable_wakeup(SYSCFG_TypeDef *syscfg, md_syscfg_wakeup_enable_t wken)
{
    SET_BIT(syscfg->WKUP, wken);
}

/**
  * @brief  disable Wake-up pin or wake-up event switch
  * @param  syscfg SYSCFG Instance
  * @param  wken Wake-up pin or wake-up event switch
            @arg @ref MD_SYSCFG_WKEN_WKUP8
            @arg @ref MD_SYSCFG_WKEN_IWDT 
            @arg @ref MD_SYSCFG_WKEN_LVD  
            @arg @ref MD_SYSCFG_WKEN_NRST 
            @arg @ref MD_SYSCFG_WKEN_WKUP7
            @arg @ref MD_SYSCFG_WKEN_WKUP6
            @arg @ref MD_SYSCFG_WKEN_WKUP5
            @arg @ref MD_SYSCFG_WKEN_WKUP4
            @arg @ref MD_SYSCFG_WKEN_WKUP3
            @arg @ref MD_SYSCFG_WKEN_WKUP2
            @arg @ref MD_SYSCFG_WKEN_WKUP1
            @arg @ref MD_SYSCFG_WKEN_WKUP0
  * @retval None
  */
__STATIC_INLINE void md_syscfg_disable_wakeup(SYSCFG_TypeDef *syscfg, md_syscfg_wakeup_enable_t wken)
{
    CLEAR_BIT(syscfg->WKUP, wken);
}

/**
  * @brief  Wake-up pin or wake-up event switch is enabled
  * @param  syscfg SYSCFG Instance
  * @param  wken Wake-up pin or wake-up event switch
            @arg @ref MD_SYSCFG_WKEN_WKUP8
            @arg @ref MD_SYSCFG_WKEN_IWDT 
            @arg @ref MD_SYSCFG_WKEN_LVD  
            @arg @ref MD_SYSCFG_WKEN_NRST 
            @arg @ref MD_SYSCFG_WKEN_WKUP7
            @arg @ref MD_SYSCFG_WKEN_WKUP6
            @arg @ref MD_SYSCFG_WKEN_WKUP5
            @arg @ref MD_SYSCFG_WKEN_WKUP4
            @arg @ref MD_SYSCFG_WKEN_WKUP3
            @arg @ref MD_SYSCFG_WKEN_WKUP2
            @arg @ref MD_SYSCFG_WKEN_WKUP1
            @arg @ref MD_SYSCFG_WKEN_WKUP0
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : wakeup event enable
  *         @arg 0x0 : wakeup event disable
  */
__STATIC_INLINE uint32_t md_syscfg_is_enabled_wakeup(SYSCFG_TypeDef *syscfg, md_syscfg_wakeup_enable_t wken)
{
    return (READ_BIT(syscfg->WKUP, wken) == wken);
}
/**
  * @}
  */

/** @defgroup WKSR
  * @{
  */
/**
  * @brief  Set SYSCFG_WKSR register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_wksr(SYSCFG_TypeDef *syscfg, uint32_t wksr)
{
    WRITE_REG(syscfg->WKSR, wksr);
}

/**
  * @brief  Get SYSCFG_WKSR Register
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_syscfg_get_wksr(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->WKSR));
}


/**
  * @brief  Clear the wake-up flag
  * @param  syscfg SYSCFG Instance
  * @retval None
  */
__STATIC_INLINE void md_syscfg_clear_flag_wakeup(SYSCFG_TypeDef *syscfg)
{
    SET_BIT(syscfg->WKSR, SYSCFG_WKSR_WKCLR);
}

/**
  * @brief  Set Low power mode selection
  * @param  syscfg SYSCFG Instance
  * @param  lpls Low power mode selection
            @arg @ref MD_SYSCFG_LPLS_STOP0_MODE
            @arg @ref MD_SYSCFG_LPLS_STOP1_MODE
            @arg @ref MD_SYSCFG_LPLS_STANDBY_MODE
            @arg @ref MD_SYSCFG_LPLS_SHUTDOWN_MODE
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_lowpower_mode(SYSCFG_TypeDef *syscfg, md_syscfg_lowpower_mode_t lpls)
{
    MODIFY_REG(syscfg->WKSR, SYSCFG_WKSR_LPLS, lpls << SYSCFG_WKSR_LPLS_POSS);
}

/**
  * @brief  Get Low power mode selection
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_LPLS_STOP0_MODE
            @arg @ref MD_SYSCFG_LPLS_STOP1_MODE
            @arg @ref MD_SYSCFG_LPLS_STANDBY_MODE
            @arg @ref MD_SYSCFG_LPLS_SHUTDOWN_MODE
  */
__STATIC_INLINE uint32_t md_syscfg_get_lowpower_mode(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->WKSR, SYSCFG_WKSR_LPLS) >> SYSCFG_WKSR_LPLS_POSS);
}

/**
  * @brief  Set Low Power LDO Voltage Selection
  * @param  syscfg SYSCFG Instance
  * @param  lpvsel Low Power LDO Voltage Selection
            @arg @ref MD_SYSCFG_LPVSEL_1P3V 
            @arg @ref MD_SYSCFG_LPVSEL_1P4V 
            @arg @ref MD_SYSCFG_LPVSEL_1P5V 
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_lowpower_voltage(SYSCFG_TypeDef *syscfg, md_syscfg_ldo_lowpower_mode_voltage_t lpvsel)
{
    MODIFY_REG(syscfg->WKSR, SYSCFG_WKSR_LPLS, lpvsel << SYSCFG_WKSR_LPLS_POSS);
}

/**
  * @brief  Get Low Power LDO Voltage Selection
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_LPVSEL_1P3V 
            @arg @ref MD_SYSCFG_LPVSEL_1P4V 
            @arg @ref MD_SYSCFG_LPVSEL_1P5V 
  */
__STATIC_INLINE uint32_t md_syscfg_get_lowpower_voltage(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->WKSR, SYSCFG_WKSR_LPLS) >> SYSCFG_WKSR_LPLS_POSS);
}

/**
  * @brief  Get Wake flag
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : wakeup flag exist
  *         @arg 0x0 : wakeup flag non-existent
  */
__STATIC_INLINE uint32_t md_syscfg_is_active_flag_wakeup(SYSCFG_TypeDef *syscfg)
{
    return (READ_BIT(syscfg->WKSR, SYSCFG_WKSR_FLAG) == (SYSCFG_WKSR_FLAG));
}

/**
  * @brief  Get Low-power mode wake-up flag
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_FG_WKUP8
            @arg @ref MD_SYSCFG_FG_IWDT 
            @arg @ref MD_SYSCFG_FG_LVD  
            @arg @ref MD_SYSCFG_FG_NRST 
            @arg @ref MD_SYSCFG_FG_WKUP7
            @arg @ref MD_SYSCFG_FG_WKUP6
            @arg @ref MD_SYSCFG_FG_WKUP5
            @arg @ref MD_SYSCFG_FG_WKUP4
            @arg @ref MD_SYSCFG_FG_WKUP3
            @arg @ref MD_SYSCFG_FG_WKUP2
            @arg @ref MD_SYSCFG_FG_WKUP1
            @arg @ref MD_SYSCFG_FG_WKUP0            
  */
__STATIC_INLINE uint32_t md_syscfg_get_wakeup_event(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_BIT(syscfg->WKSR, SYSCFG_WKSR_FG) >> SYSCFG_WKSR_FG_POSS);
}

/**
  * @brief  Get Low-power mode wake-up flag
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_SYSCFG_FG_WKUP8
            @arg @ref MD_SYSCFG_FG_IWDT 
            @arg @ref MD_SYSCFG_FG_LVD  
            @arg @ref MD_SYSCFG_FG_NRST 
            @arg @ref MD_SYSCFG_FG_WKUP7
            @arg @ref MD_SYSCFG_FG_WKUP6
            @arg @ref MD_SYSCFG_FG_WKUP5
            @arg @ref MD_SYSCFG_FG_WKUP4
            @arg @ref MD_SYSCFG_FG_WKUP3
            @arg @ref MD_SYSCFG_FG_WKUP2
            @arg @ref MD_SYSCFG_FG_WKUP1
            @arg @ref MD_SYSCFG_FG_WKUP0 
  */
__STATIC_INLINE uint32_t md_syscfg_is_active_flag_wakeup_event(SYSCFG_TypeDef *syscfg, md_syscfg_wakeup_event_t wkfg)
{
    return (READ_BIT(syscfg->WKSR, wkfg) == wkfg);
}
/**
  * @}
  */

/** @defgroup BKREG0
  * @{
  */
/**
  * @brief  Set Backup Register 0.
  * @note   This field was saved in the always on block, reset by power reset.
  * @param  syscfg SYSCFG Instance
  * @param  bkreg0 This parameter can be one of the following values:
  *         @arg Max Value 0xFFFFFFFF
  *         @arg Min Value 0
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_bkreg0(SYSCFG_TypeDef *syscfg, uint32_t bkreg0)
{
    WRITE_REG(syscfg->BKREG0, bkreg0);
}
/**
  * @brief  Get Backup Register 0.
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xFFFFFFFF
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_bkreg0(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->BKREG0));
}
/**
  * @}
  */

/** @defgroup BKREG1
  * @{
  */
/**
  * @brief  Set Backup Register 1.
  * @note   This field was saved in the always on block, reset by power reset.
  * @param  syscfg SYSCFG Instance
  * @param  bkreg1 This parameter can be one of the following values:
  *         @arg Max Value 0xFFFFFFFF
  *         @arg Min Value 0
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_bkreg1(SYSCFG_TypeDef *syscfg, uint32_t bkreg1)
{
    WRITE_REG(syscfg->BKREG1, bkreg1);
}
/**
  * @brief  Get Backup Register 1.
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xFFFFFFFF
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_bkreg1(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->BKREG1));
}
/**
  * @}
  */

/** @defgroup BKREG2
  * @{
  */
/**
  * @brief  Set Backup Register 2.
  * @note   This field was saved in the always on block, reset by power reset.
  * @param  syscfg SYSCFG Instance
  * @param  bkreg2 This parameter can be one of the following values:
  *         @arg Max Value 0xFFFFFFFF
  *         @arg Min Value 0
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_bkreg2(SYSCFG_TypeDef *syscfg, uint32_t bkreg2)
{
    WRITE_REG(syscfg->BKREG2, bkreg2);
}
/**
  * @brief  Get Backup Register 2.
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xFFFFFFFF
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_bkreg2(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->BKREG2));
}
/**
  * @}
  */

/** @defgroup BKREG3
  * @{
  */
/**
  * @brief  Set Backup Register 3.
  * @note   This field was saved in the always on block, reset by power reset.
  * @param  syscfg SYSCFG Instance
  * @param  bkreg3 This parameter can be one of the following values:
  *         @arg Max Value 0xFFFFFFFF
  *         @arg Min Value 0
  * @retval None
  */
__STATIC_INLINE void md_syscfg_set_bkreg3(SYSCFG_TypeDef *syscfg, uint32_t bkreg3)
{
    WRITE_REG(syscfg->BKREG3, bkreg3);
}
/**
  * @brief  Get Backup Register 3.
  * @param  syscfg SYSCFG Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xFFFFFFFF
  *         @arg Min Value 0
  */
__STATIC_INLINE uint32_t md_syscfg_get_bkreg3(SYSCFG_TypeDef *syscfg)
{
    return (uint32_t)(READ_REG(syscfg->BKREG3));
}
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif  /* __cplusplus */
#endif  /* __MD_SYSCFG_H__ */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
