/**********************************************************************************
 *
 * @file    md_sva.h
 * @brief   header file of md_sva.c
 *
 * @date    19 Apr 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Apr 2023     Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_SVA_H__
#define __MD_SVA_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes -------------------------------------------------------------------*/
#include "md_utils.h"

/* Exported Constants -------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */

/* Exported Macros ----------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_SVA SVA
  * @brief SVA micro driver
  * @{
  */

/** @defgroup MD_SVA_Public_Macros SVA Public Macros
  * @{
  */

/* Exported Types ------------------------------------------------------------ */

/**
  * @brief SVA_CTRL SVA_CTRL Register
  */
typedef enum
{ 
    MD_SVA_DMA_DISABLE = (0U),
    MD_SVA_DMA_ENABLE  = (1U),
}md_sva_dma_enable_t;  

typedef enum
{ 
    MD_SVA_OUTPUTMUL_DISABLE = (0U),
    MD_SVA_OUTPUTMUL_ENABLE  = (1U),
}md_sva_output_mul_enable_t;

typedef enum
{ 
    MD_SVA_INPUT_NO_EXCHANGE = (0),
    MD_SVA_INPUT_EXCHANGE    = (1),
}md_sva_input_exchange_t;    

typedef enum
{   
    MD_SVA_MODE_CLARKE                          = (0U),
    MD_SVA_MODE_PARK                            = (1U),
    MD_SVA_MODE_CLARKD_AND_PARK                 = (2U),
    MD_SVA_MODE_INVERSE_PARK                    = (3U),
    MD_SVA_MODE_INVERSE_CLARKE                  = (4U),
    MD_SVA_MODE_INVERSE_PARK_AND_INVERSE_CLARKE = (5U),
    MD_SVA_MODE_SVPWM                           = (6U),
    MD_SVA_MODE_INVERSE_PARK_AND_SVPWM          = (7U),
    MD_SVA_MODE_RESET                           = (0xFU),    
}md_sva_op_mode_t;

/**
  * @brief SVA_STATUS SVA_STATUS Register
  */
typedef enum
{ 
    MD_SVA_PWMA_B_LESS_THRESHOLD = (1U),
    MD_SVA_PWMB_C_LESS_THRESHOLD = (2U),
    MD_SVA_PWMA_C_LESS_THRESHOLD = (4U),
}md_sva_sector_flag_t;
/**
  * @}
  */

/** @defgroup MD_SVA_Pubulic_Types SVA Pubulic Types
  * @{
  */
typedef struct
{
    md_sva_op_mode_t op_mode;                   /*!< Specifies the op mode of the SVA.
                                                    This parameter can be a value of @ref md_sva_op_mode_t */
    uint32_t input_data0;                       /*!< Specifies the input data0 of the SVA. */
    
    uint32_t input_data1;                       /*!< Specifies the input data1 of the SVA. */
    
    uint32_t input_angle;                       /*!< Specifies the angle of the SVA. */
    
    md_sva_dma_enable_t dma_en;                 /*!< Specifies the dma enable of the SVA.
                                                    This parameter can be a value of @ref md_sva_dma_enable_t */
    md_sva_output_mul_enable_t outputmul_en;    /*!< Specifies the output mul enable of the SVA.
                                                    This parameter can be a value of @ref md_sva_output_mul_enable_t */
    uint32_t outputmul;                         /*!< Specifies the output mul of the SVA. */
    
    md_sva_input_exchange_t input_exchange;     /*!< Specifies the input exchange of the SVA.
                                                    This parameter can be a value of @ref md_sva_input_exchange_t */
} md_sva_config_typedef;

/**
  * @}
  */

/* Exported Functions -------------------------------------------------------- */

/** @defgroup MD_SVA_Public_Functions SVA Public Functions
  * @{
  */

/** @defgroup IER
  * @{
  */

/**
  * @brief  SVA operation completed interrupt enable.
  * @param  sva SVA Instance.
  * @retval None.
  */
__STATIC_INLINE void md_sva_enable_it_done(SVA_TypeDef *sva)
{
    SET_BIT(sva->IER, SVA_IER_DONE);
}

/**
  * @}
  */

/** @defgroup IDR
  * @{
  */

/**
  * @brief  SVA operation completed interrupt disable.
  * @param  sva SVA Instance.
  * @retval None.
  */
__STATIC_INLINE void md_sva_disable_it_done(SVA_TypeDef *sva)
{
    SET_BIT(sva->IDR, SVA_IDR_DONE);
}

/**
  * @}
  */

/** @defgroup IVS
  * @{
  */

/**
  * @brief  Check if SVA operation completed interrupt is enabled.
  * @param  sva SVA Instance.
  * @retval None.
  */
__STATIC_INLINE uint32_t md_sva_is_enabled_it_done(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->IVS, SVA_IVS_DONE) == SVA_IVS_DONE);
}

/**
  * @}
  */

/** @defgroup RIF
  * @{
  */

/**
  * @brief  Get SVA operation completed interrupt flag.
  * @param  sva SVA Instance.
  * @retval None.
  */
__STATIC_INLINE uint32_t md_sva_is_active_it_done(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->RIF, SVA_RIF_DONE == SVA_RIF_DONE));
}

/**
  * @}
  */

/** @defgroup IFM
  * @{
  */

/**
  * @brief  Get SVA operation completed interrupt flag masked.
  * @param  sva SVA Instance.
  * @retval None.
  */
__STATIC_INLINE uint32_t md_sva_is_masked_it_done(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->IFM, SVA_IFM_DONE) == SVA_IFM_DONE);
}

/**
  * @}
  */

/** @defgroup ICR
  * @{
  */

/**
  * @brief  SVA operation completed interrupt flag clear.
  * @param  sva SVA Instance.
  * @retval None.
  */
__STATIC_INLINE void md_sva_clear_it_done(SVA_TypeDef *sva)
{
    SET_BIT(sva->ICR, SVA_ICR_DONE);
}

/**
  * @}
  */

/** @defgroup INDATA0
  * @{
  */

/**
  * @brief  Set SVA input data 0
  * @param  sva SVA Instance.
  * @param  data
  */
__STATIC_INLINE void md_sva_set_input_data_0(SVA_TypeDef *sva, uint32_t data)
{
    WRITE_REG(sva->INDATA0,  data);
}

/**
  * @brief  Get SVA input data 0
  * @param  sva SVA Instance.
  * @retval data Values
  */
__STATIC_INLINE uint32_t md_sva_get_input_data_0(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->INDATA0);
}

/**
  * @}
  */

/** @defgroup INDATA1
  * @{
  */

/**
  * @brief  Set SVA input data 1
  * @param  sva SVA Instance.
  * @param  data
  */
__STATIC_INLINE void md_sva_set_input_data_1(SVA_TypeDef *sva, uint32_t data)
{
    WRITE_REG(sva->INDATA1,  data);
}

/**
  * @brief  Get SVA input data 1
  * @param  sva SVA Instance.
  * @retval data Values
  */
__STATIC_INLINE uint32_t md_sva_get_input_data_1(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->INDATA1);
}

/**
  * @}
  */

/** @defgroup INANGLE
  * @{
  */

/**
  * @brief  Set SVA input angle
  * @param  sva SVA Instance.
  * @param  angle
  */
__STATIC_INLINE void md_sva_set_input_angle(SVA_TypeDef *sva, uint32_t angle)
{
    WRITE_REG(sva->INANGLE,  angle);
}

/**
  * @brief  Get SVA input angle
  * @param  sva SVA Instance.
  * @retval angle Values
  */
__STATIC_INLINE uint32_t md_sva_get_input_angle(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->INANGLE);
}
/**
  * @}
  */

/** @defgroup CTRL
  * @{
  */

/**
  * @brief  Set SVA_CTRL register
  * @param  sva SVA Instance.
  * @param  ctrl
  */
__STATIC_INLINE void md_sva_set_ctrl(SVA_TypeDef *sva, uint32_t ctrl)
{
    WRITE_REG(sva->CTRL,  ctrl);
}

/**
  * @brief  Get SVA_CTRL register
  * @param  sva SVA Instance.
  * @retval ctrl Values
  */
__STATIC_INLINE uint32_t md_sva_get_ctrl(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->CTRL);
}

/**
  * @brief  DMA request switch enable
  * @param  sva SVA Instance.
  * @retval None
  */
__STATIC_INLINE void md_sva_enable_dma_request(SVA_TypeDef *sva)
{
    SET_BIT(sva->CTRL,  SVA_CTRL_DMA_EN);
}

/**
  * @brief  DMA request switch disable
  * @param  sva SVA Instance.
  * @retval None
  */
__STATIC_INLINE void md_sva_disable_dma_request(SVA_TypeDef *sva)
{
    CLEAR_BIT(sva->CTRL,  SVA_CTRL_DMA_EN);
}

/**
  * @brief  Check if DMA request switch is enabled
  * @param  sva SVA Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_sva_is_enabled_dma_request(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->CTRL, SVA_CTRL_DMA_EN) == (SVA_CTRL_DMA_EN));
}

/**
  * @brief  SVA output multiple function switch enable
  * @param  sva SVA Instance.
  * @retval None
  */
__STATIC_INLINE void md_sva_enable_output_multiple(SVA_TypeDef *sva)
{
    SET_BIT(sva->CTRL,  SVA_CTRL_OUTMUL_EN);
}

/**
  * @brief  SVA output multiple function switch disable
  * @param  sva SVA Instance.
  * @retval None
  */
__STATIC_INLINE void md_sva_disable_output_multiple(SVA_TypeDef *sva)
{
    CLEAR_BIT(sva->CTRL,  SVA_CTRL_OUTMUL_EN);
}

/**
  * @brief  Check if SVA output multiple function switch is enabled
  * @param  sva SVA Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_sva_is_enabled_output_multiple(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->CTRL, SVA_CTRL_OUTMUL_EN) == (SVA_CTRL_OUTMUL_EN));
}

/**
  * @brief  Set Input parameter exchange
  * @param  sva SVA Instance.
  * @retval None
  */
__STATIC_INLINE void md_sva_set_input_exchange(SVA_TypeDef *sva)
{
    SET_BIT(sva->CTRL,  SVA_CTRL_EXCHAHGE);
}

/**
  * @brief  Set Input parameter exchange
  * @param  sva SVA Instance.
  * @retval None
  */
__STATIC_INLINE void md_sva_set_input_no_exchange(SVA_TypeDef *sva)
{
    CLEAR_BIT(sva->CTRL,  SVA_CTRL_EXCHAHGE);
}

/**
  * @brief  Check if Input parameter exchange
  * @param  sva SVA Instance.
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_SVA_INPUT_NO_EXCHANGE
  *         @arg @ref MD_SVA_INPUT_EXCHANGE
  */
__STATIC_INLINE uint32_t md_sva_get_input_exchange(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->CTRL, SVA_CTRL_EXCHAHGE) >> SVA_CTRL_EXCHAHGE_POS);
}

/**
  * @brief  Set SVA operation mode
  * @param  sva SVA Instance.
  * @param  mode This parameter can be one of the following values:
  *         @arg @ref MD_SVA_MODE_CLARKE
  *         @arg @ref MD_SVA_MODE_PARK
  *         @arg @ref MD_SVA_MODE_CLARKD_AND_PARK
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK
  *         @arg @ref MD_SVA_MODE_INVERSE_CLARKE
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK_AND_INVERSE_CLARKE
  *         @arg @ref MD_SVA_MODE_SVPWM
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK_AND_SVPWM
  * @retval None
  */
__STATIC_INLINE void md_sva_set_operation_mode(SVA_TypeDef *sva, md_sva_op_mode_t mode)
{
    MODIFY_REG(sva->CTRL, SVA_CTRL_OP_MODE, mode << SVA_CTRL_OP_MODE_POSS);
}

/**
  * @brief  Get SVA operation mode
  * @param  sva SVA Instance.
  * @retval mode This parameter can be one of the following values:
  *         @arg @ref MD_SVA_MODE_CLARKE
  *         @arg @ref MD_SVA_MODE_PARK
  *         @arg @ref MD_SVA_MODE_CLARKD_AND_PARK
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK
  *         @arg @ref MD_SVA_MODE_INVERSE_CLARKE
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK_AND_INVERSE_CLARKE
  *         @arg @ref MD_SVA_MODE_SVPWM
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK_AND_SVPWM
  */
__STATIC_INLINE uint32_t md_sva_get_operation_mode(SVA_TypeDef *sva)
{
    return (uint32_t)(READ_BIT(sva->CTRL, SVA_CTRL_OP_MODE) >> SVA_CTRL_OP_MODE);
}

/**
  * @brief  SVA trigger enable
  * @param  sva SVA Instance.
  * @retval None
  */
__STATIC_INLINE void md_sva_enable_trigger(SVA_TypeDef *sva)
{
    SET_BIT(sva->CTRL,  SVA_CTRL_STR_TRIG);
}

/**
  * @}
  */

/** @defgroup OTDATA0
  * @{
  */

/**
  * @brief  Get SVA output data 0
  * @param  sva SVA Instance.
  * @retval data Values
  */
__STATIC_INLINE uint32_t md_sva_get_output_data_0(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->OTDATA0);
}

/**
  * @}
  */

/** @defgroup OTDATA1
  * @{
  */

/**
  * @brief  Get SVA output data 1
  * @param  sva SVA Instance.
  * @retval data Values
  */
__STATIC_INLINE uint32_t md_sva_get_output_data_1(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->OTDATA1);
}

/**
  * @}
  */

/** @defgroup OTDATA2
  * @{
  */

/**
  * @brief  Get SVA output data 2
  * @param  sva SVA Instance.
  * @retval data Values
  */
__STATIC_INLINE uint32_t md_sva_get_output_data_2(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->OTDATA2);
}

/**
  * @}
  */


/** @defgroup STATUS
  * @{
  */
/**
  * @brief  Get SVA_STATUS Register
  * @param  sva SVA Instance.
  */
__STATIC_INLINE uint32_t md_sva_get_status(SVA_TypeDef *sva)
{
    return READ_REG(sva->STATUS);
}

/**
  * @brief  Check sector swap flag
  * @param  sva SVA Instance.
  * @retval status This parameter can be one of the following values:
  *         @arg @ref MD_SVA_PWMA_B_LESS_THRESHOLD
  *         @arg @ref MD_SVA_PWMB_C_LESS_THRESHOLD
  *         @arg @ref MD_SVA_PWMA_C_LESS_THRESHOLD
  */
__STATIC_INLINE uint32_t md_sva_is_active_flag_sector_swap(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->STATUS, SVA_STATUS_SECTOR_FLAG) >> (SVA_STATUS_SECTOR_FLAG_POSS));
}

/**
  * @brief  Get SVA operation mode
  * @param  sva SVA Instance.
  * @retval mode This parameter can be one of the following values:
  *         @arg @ref MD_SVA_MODE_CLARKE
  *         @arg @ref MD_SVA_MODE_PARK
  *         @arg @ref MD_SVA_MODE_CLARKD_AND_PARK
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK
  *         @arg @ref MD_SVA_MODE_INVERSE_CLARKE
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK_AND_INVERSE_CLARKE
  *         @arg @ref MD_SVA_MODE_SVPWM
  *         @arg @ref MD_SVA_MODE_INVERSE_PARK_AND_SVPWM
  *         @arg @ref MD_SVA_MODE_RESET
  */
__STATIC_INLINE uint32_t md_sva_get_operation_mode_status(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->STATUS, SVA_STATUS_SECTOR_FLAG) >> (SVA_STATUS_SECTOR_FLAG_POSS));
}

/**
  * @brief  Check SVA busy flag
  * @param  sva SVA Instance.
  * @retval status This parameter can be one of the following values:
  *         @arg @ref MD_SVA_STANDBY
  *         @arg @ref MD_SVA_BUSY
  */
__STATIC_INLINE uint32_t md_sva_is_active_flag_busy(SVA_TypeDef *sva)
{
    return (READ_BIT(sva->STATUS, SVA_STATUS_BUSY) >> (SVA_STATUS_BUSY_POS));
}

/**
  * @}
  */

/** @defgroup OUTMUL
  * @{
  */

/**
  * @brief  Set SVA output multiplication factor
  * @param  sva SVA Instance.
  * @param  factor
  */
__STATIC_INLINE void md_sva_set_output_multiple(SVA_TypeDef *sva, uint32_t factor)
{
    WRITE_REG(sva->OUTMUL, factor);
}

/**
  * @brief  Get SVA output multiplication factor
  * @param  sva SVA Instance.
  * @retval factor
  */
__STATIC_INLINE uint32_t sva_sva_get_output_multiple(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->OUTMUL);
}

/**
  * @}
  */

/** @defgroup PWMTH
  * @{
  */

/**
  * @brief  Set SVA PWM width detection threshold
  * @param  sva SVA Instance.
  * @param  group Channel select
  * @param  width
  */
__STATIC_INLINE void md_sva_set_pwm_width_threshold(SVA_TypeDef *sva, uint32_t width)
{
    WRITE_REG(sva->PWMTH, width);
}

/**
  * @brief  Get SVA PWM width detection threshold
  * @param  sva SVA Instance.
  * @retval width
  */
__STATIC_INLINE uint32_t md_sva_get_pwm_width_threshold(SVA_TypeDef *sva)
{
    return (uint32_t)READ_REG(sva->PWMTH);
}

/**
  * @}
  */

/** @defgroup SVA_Function
  * @{
  */
void md_sva_config(SVA_TypeDef *SVAx, md_sva_config_typedef *config);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
  
#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __MD_SVA_H__ */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
