/**********************************************************************************
 *
 * @file    md_rcu.h
 * @brief   header file of md_rcu.c
 *
 * @date    09 Apr. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          09 Apr. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_RCU_H__
#define __MD_RCU_H__

#ifdef __cplusplus
extern "C" {
#endif  /* __cplusplus */

/* Includes -------------------------------------------------------------------*/
#include <stdbool.h>
#include "md_utils.h"

/* Exported Constants -------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */


/** @defgroup MD_RCU RCU
  * @brief RCU micro driver
  * @{
  */

/* Exported Macros ----------------------------------------------------------- */

/** @defgroup MD_RCU_Public_Macros RCU Public Macros
  * @{
  */

#ifndef __HOSC
#define __HOSC          (8000000UL)
#endif
#define __HRC           (16000000UL)
#define __LRC           (   32000UL)
#define __PLL_CLKREF    ( 4000000UL)

/* Exported Types ------------------------------------------------------------ */

/**
  * @brief MD_RCU_LRCRDY_FLAG LRC clock ready flag
  */
typedef enum
{  
    MD_RCU_LRCRDY_NOT_READY = (0x00000000UL),  /** @brief LRC oscillator not ready */
    MD_RCU_LRCRDY_READY     = (0x00000001UL),  /** @brief LRC oscillator ready */
} md_rcu_lrc_ready_t;

/**
  * @brief MD_RCU_PLL_FLAG PLL clock ready flag
  */
typedef enum 
{ 
    MD_RCU_PLLRDY_NOT_READY = (0x00000000UL),  /** @brief PLL not ready */
    MD_RCU_PLLRDY_READY     = (0x00000001UL),  /** @brief PLL ready */
} md_rcu_pll_ready_t;

/**
  * @brief MD_RCU_HOSCRDY_FLAG HOSC clock ready flag
  */
typedef enum
{ 
    MD_RCU_HOSCRDY_NOT_READY = (0x00000000UL),  /** @brief HOSC oscillator not ready */
    MD_RCU_HOSCRDY_READY     = (0x00000001UL),  /** @brief HOSC oscillator ready */
} md_rcu_hosc_ready_t;

/**
  * @brief MD_RCU_HRCRDY_FLAG HRC clock ready flag
  */
typedef enum
{ 
    MD_RCU_HRCRDY_NOT_READY = (0x00000000UL),  /** @brief HRC oscillator not ready */
    MD_RCU_HRCRDY_READY     = (0x00000001UL),  /** @brief HRC oscillator ready */
} md_rcu_hrc_ready_t;

/**
  * @brief MD_RCU_MCO_DIV Microcontroller clock output select
  */
typedef enum
{ 
    MD_RCU_MPRE_MCO_DIV1   = (0x00000000UL),  /** @brief MCO DIV = 1  */
    MD_RCU_MPRE_MCO_DIV2   = (0x00000001UL),  /** @brief MCO DIV = 2  */
    MD_RCU_MPRE_MCO_DIV4   = (0x00000002UL),  /** @brief MCO DIV = 4  */
    MD_RCU_MPRE_MCO_DIV8   = (0x00000003UL),  /** @brief MCO DIV = 8  */
    MD_RCU_MPRE_MCO_DIV16  = (0x00000004UL),  /** @brief MCO DIV = 16 */
    MD_RCU_MPRE_MCO_DIV32  = (0x00000005UL),  /** @brief MCO DIV = 32 */
    MD_RCU_MPRE_MCO_DIV64  = (0x00000006UL),  /** @brief MCO DIV = 64 */
    MD_RCU_MPRE_MCO_DIV128 = (0x00000007UL),  /** @brief MCO DIV = 128*/
} md_rcu_mco_div_t;

/**
  * @brief MD_RCU_MCO_OUTPUT Microcontroller clock output select
  */
typedef enum
{ 
    MD_RCU_MSW_MCO_DISABLE      = (0x00000000UL),  /** @brief MCO output disabled, no clock on MCO */
    MD_RCU_MSW_MCO_LRC          = (0x00000001UL),  /** @brief Internal low speed (LRC) oscillator clock selected */
    MD_RCU_MSW_MCO_HRC          = (0x00000003UL),  /** @brief Internal high speed (HRC) oscillator clock selected */
    MD_RCU_MSW_MCO_HOSC         = (0x00000004UL),  /** @brief External high speed (HOSC) oscillator clock selected */
    MD_RCU_MSW_MCO_PLL          = (0x00000006UL),  /** @brief PLL */
    MD_RCU_MSW_MCO_SYSCLK       = (0x00000008UL),  /** @brief System clock selected(SYSCLK) */
    MD_RCU_MSW_MCO_HCLK         = (0x00000009UL),  /** @brief AHB clock selected(HCLK) */
    MD_RCU_MSW_MCO_PCLK         = (0x0000000AUL),  /** @brief APB clock selected(PCLK) */
    MD_RCU_MSW_MCO_ADC          = (0x0000000BUL),  /** @brief ADC clock selected(ADCCLK) */
    MD_RCU_MSW_MCO_I2C_KERNEL   = (0x0000000CUL),  /** @brief I2C kernel clock selected */
    MD_RCU_MSW_MCO_ADC_KERNEL   = (0x0000000DUL),  /** @brief ADC kernel clock selected */
    MD_RCU_MSW_MCO_TIMER_KERNEL = (0x0000000EUL),  /** @brief TIMER kernel clock selected */
} md_rcu_mco_source_t;

/**
  * @brief MD_RCU_PLLSRC PLL reference clock source
  */
typedef enum
{ 
    MD_RCU_PLLSRC_HRC  = (0x00000000UL),  /** @brief HRC/PREDIV selected as PLL reference clock */
    MD_RCU_PLLSRC_HOSC = (0x00000001UL),  /** @brief HOSC/PREDIV selected as PLL reference clock */
} md_rcu_pll_source_t;

/**
  * @brief MD_RCU_PLL_CLK PLL clock output
  */
typedef enum
{ 
    MD_RCU_PLLCLK_PASS = (0U),
    MD_RCU_PLLCLK_4M   = (4000000U),   /** @brief PLL 4MHz */
    MD_RCU_PLLCLK_8M   = (8000000U),   /** @brief PLL 8MHz */
    MD_RCU_PLLCLK_12M  = (12000000U),  /** @brief PLL 12MHz */
    MD_RCU_PLLCLK_16M  = (16000000U),  /** @brief PLL 16MHz */
    MD_RCU_PLLCLK_24M  = (24000000U),  /** @brief PLL 24MHz */
    MD_RCU_PLLCLK_32M  = (32000000U),  /** @brief PLL 32MHz */
    MD_RCU_PLLCLK_36M  = (36000000U),  /** @brief PLL 36MHz */
    MD_RCU_PLLCLK_40M  = (40000000U),  /** @brief PLL 40MHz */
    MD_RCU_PLLCLK_48M  = (48000000U),  /** @brief PLL 48MHz */
    MD_RCU_PLLCLK_64M  = (64000000U),  /** @brief PLL 64MHz */
    MD_RCU_PLLCLK_72M  = (72000000U),  /** @brief PLL 72MHz */
} md_rcu_pll_output_t;

/**
  * @brief MD_RCU_PLL_DIV Microcontroller clock output select
  */
typedef enum
{ 
    MD_RCU_PREDIV_PLL_DIV1  = (0x00000000UL),  /** @brief PLL DIV = 1 */
    MD_RCU_PREDIV_PLL_DIV2  = (0x00000001UL),  /** @brief PLL DIV = 2 */
    MD_RCU_PREDIV_PLL_DIV3  = (0x00000002UL),  /** @brief PLL DIV = 3 */
    MD_RCU_PREDIV_PLL_DIV4  = (0x00000003UL),  /** @brief PLL DIV = 4 */
    MD_RCU_PREDIV_PLL_DIV5  = (0x00000004UL),  /** @brief PLL DIV = 5 */
    MD_RCU_PREDIV_PLL_DIV6  = (0x00000005UL),  /** @brief PLL DIV = 6 */
    MD_RCU_PREDIV_PLL_DIV7  = (0x00000006UL),  /** @brief PLL DIV = 7 */
    MD_RCU_PREDIV_PLL_DIV8  = (0x00000007UL),  /** @brief PLL DIV = 8 */
    MD_RCU_PREDIV_PLL_DIV9  = (0x00000008UL),  /** @brief PLL DIV = 9 */
    MD_RCU_PREDIV_PLL_DIV10 = (0x00000009UL),  /** @brief PLL DIV = 10*/
    MD_RCU_PREDIV_PLL_DIV11 = (0x0000000AUL),  /** @brief PLL DIV = 11*/
    MD_RCU_PREDIV_PLL_DIV12 = (0x0000000BUL),  /** @brief PLL DIV = 12*/
    MD_RCU_PREDIV_PLL_DIV13 = (0x0000000CUL),  /** @brief PLL DIV = 13*/
    MD_RCU_PREDIV_PLL_DIV14 = (0x0000000DUL),  /** @brief PLL DIV = 14*/
    MD_RCU_PREDIV_PLL_DIV15 = (0x0000000EUL),  /** @brief PLL DIV = 15*/
    MD_RCU_PREDIV_PLL_DIV16 = (0x0000000FUL),  /** @brief PLL DIV = 16*/
} md_rcu_pll_input_div_t;


/**
  * @brief MD_RCU_APB_CLK_DIV APB clock prescaler(PCLK)
  */
typedef enum
{ 
    MD_RCU_PPRE_HCLK_DIV_1  = (0x00000000UL),  /** @brief HCLK not divided */
    MD_RCU_PPRE_HCLK_DIV_2  = (0x00000004UL),  /** @brief HCLK divided by 2 */
    MD_RCU_PPRE_HCLK_DIV_4  = (0x00000005UL),  /** @brief HCLK divided by 4 */
    MD_RCU_PPRE_HCLK_DIV_8  = (0x00000006UL),  /** @brief HCLK divided by 8 */
    MD_RCU_PPRE_HCLK_DIV_16 = (0x00000007UL),  /** @brief HCLK divided by 16 */
} md_rcu_apb_clock_div_t;

/**
  * @brief MD_RCU_AHB_CLK_DIV AHB clock prescaler(HCLK)
  */
typedef enum
{
    MD_RCU_HPRE_SYSCLK_DIV_1   = (0x00000000UL),  /** @brief SYSCLK not divided    */
    MD_RCU_HPRE_SYSCLK_DIV_2   = (0x00000008UL),  /** @brief SYSCLK divided by 2   */
    MD_RCU_HPRE_SYSCLK_DIV_4   = (0x00000009UL),  /** @brief SYSCLK divided by 4   */
    MD_RCU_HPRE_SYSCLK_DIV_8   = (0x0000000AUL),  /** @brief SYSCLK divided by 8   */
    MD_RCU_HPRE_SYSCLK_DIV_16  = (0x0000000BUL),  /** @brief SYSCLK divided by 16  */
    MD_RCU_HPRE_SYSCLK_DIV_64  = (0x0000000CUL),  /** @brief SYSCLK divided by 64  */
    MD_RCU_HPRE_SYSCLK_DIV_128 = (0x0000000DUL),  /** @brief SYSCLK divided by 128 */
    MD_RCU_HPRE_SYSCLK_DIV_256 = (0x0000000EUL),  /** @brief SYSCLK divided by 256 */
    MD_RCU_HPRE_SYSCLK_DIV_512 = (0x0000000FUL),  /** @brief SYSCLK divided by 512 */
} md_rcu_ahb_clock_div_t;

/**
  * @brief MD_RCU_SYS_CLK_SWS System clock switch(SYSCLK)
  */
typedef enum
{
    MD_RCU_SWS_SYSCLK_HRC  = (0x00000000UL),  /** @brief HRC selected as system clock   */
    MD_RCU_SWS_SYSCLK_HOSC = (0x00000001UL),  /** @brief HOSC selected as system clock  */
    MD_RCU_SWS_SYSCLK_PLL  = (0x00000002UL),  /** @brief PLL selected as system clock  */
    MD_RCU_SWS_SYSCLK_LRC  = (0x00000003UL),  /** @brief LRC selected as system clock */
} md_rcu_clock_status_t;

/**
  * @brief MD_RCU_SYS_CLK_SW System clock switch(SYSCLK)
  */
typedef enum
{
    MD_RCU_SW_SYSCLK_HRC  = (0x00000000UL),  /** @brief HRC selected as system clock   */
    MD_RCU_SW_SYSCLK_HOSC = (0x00000001UL),  /** @brief HOSC selected as system clock  */
    MD_RCU_SW_SYSCLK_PLL  = (0x00000002UL),  /** @brief PLL selected as system clock  */
    MD_RCU_SW_SYSCLK_LRC  = (0x00000003UL),  /** @brief LRC selected as system clock */
} md_rcu_clock_t;

/**
  * @brief MD_RCU_CFG1_FM
  */
typedef enum
{
    MD_RCU_FM_PLL_DIV_2  = (0x00000000UL), /** @brief PLL Output divider 2  */
    MD_RCU_FM_PLL_DIV_4  = (0x00000001UL), /** @brief PLL Output divider 4  */
    MD_RCU_FM_PLL_DIV_6  = (0x00000002UL), /** @brief PLL Output divider 6  */
    MD_RCU_FM_PLL_DIV_8  = (0x00000003UL), /** @brief PLL Output divider 8  */
    MD_RCU_FM_PLL_DIV_12 = (0x00000004UL), /** @brief PLL Output divider 12 */
    MD_RCU_FM_PLL_DIV_16 = (0x00000005UL), /** @brief PLL Output divider 16 */
    MD_RCU_FM_PLL_DIV_24 = (0x00000006UL), /** @brief PLL Output divider 24 */
    MD_RCU_FM_PLL_DIV_32 = (0x00000007UL), /** @brief PLL Output divider 32 */
    MD_RCU_FM_PLL_DIV_64 = (0x00000008UL), /** @brief PLL Output divider 64 */
} md_rcu_pll_fvco_div_t;

/**
  * @brief MD_RCU_CFG2_I2C_KSRC
  */
typedef enum
{
    MD_RCU_CFG2_I2C_KERNEL_DISABLE = (0x00000000UL),  /** @brief I2C Kernal source DISABLE  */
    MD_RCU_CFG2_I2C_KERNEL_APBCLK  = (0x00000001UL),  /** @brief I2C Kernal source APBCLK   */
    MD_RCU_CFG2_I2C_KERNEL_SYSCLK  = (0x00000002UL),  /** @brief I2C Kernal source SYSCLK   */
    MD_RCU_CFG2_I2C_KERNEL_HRCCLK  = (0x00000003UL),  /** @brief I2C Kernal source HRCCLK   */
} md_rcu_i2c_clock_source_t;

/**
  * @brief MD_RCU_CFG2_ADC_KSRC
  */
typedef enum
{
    MD_RCU_CFG2_ADC_KERNEL_DISABLE = (0x00000000UL),  /** @brief ADC Kernal source DISABLE  */
    MD_RCU_CFG2_ADC_KERNEL_PLLCLK  = (0x00000001UL),  /** @brief ADC Kernal source PLLCLK   */
    MD_RCU_CFG2_ADC_KERNEL_SYSCLK  = (0x00000002UL),  /** @brief ADC Kernal source SYSCLK   */
    MD_RCU_CFG2_ADC_KERNEL_AHBCLK  = (0x00000003UL),  /** @brief ADC Kernal source AHBCLK   */
} md_rcu_adc_clock_source_t;


#define MD_RCU_HRCON    RCU_CON_HRCON       /** @brief HRC Enable */
#define MD_RCU_HOSCON   RCU_CON_HOSCON      /** @brief HOSC Enable */
#define MD_RCU_PLLON    RCU_CON_PLLON       /** @brief PLL Enable */

#define MD_RCU_LRCON    RCU_LCON_LRCON      /** @brief LRC Enable */

/**
  * @brief MD_RCU_SRCSEL Calibration Clock Source Select
  */
#define MD_RCU_SRCSEL_HRC_4MHZ      (0x00000000UL)

/**
  * @brief MD_RCU_HRCSEL HRCSEL calibration value selection
  */
typedef enum
{
    MD_RCU_HRCSEL_OPTIONBYTE = (0x00000000UL),
    MD_RCU_HRCSEL_HRCTRIM    = (0x00000001UL),
} md_rcu_hrc_calibration_value_selection_t;

/**
  * @brief MD_RCU_LRCSEL LRCSEL calibration value selection
  */
typedef enum
{
    MD_RCU_LRCSEL_OPTIONBYTE = (0x00000000UL),
    MD_RCU_LRCSEL_LRCTRIM    = (0x00000001UL),
} md_rcu_lrc_calibration_value_selection_t;
/**
  * @}
  */
  
/** @defgroup MD_RCU_Pubulic_Types RCU Pubulic Types
  * @{
  */
/**
  * @brief MD_RCU_Public_Types RCU Init Type
  */

typedef struct
{
    md_rcu_mco_div_t  Mpre;         /*! MCO clock prescaler */
    md_rcu_mco_source_t  Msw;       /*! MCO clock output */
    md_rcu_pll_source_t  PllSrc;    /*! PLL clock source */
    uint32_t  Pllclk;               /*! PLL clock */
    md_rcu_apb_clock_div_t  Ppre;   /*! APB clock prescaler */
    md_rcu_ahb_clock_div_t  Hpre;   /*! AHB clock prescaler */
    md_rcu_clock_t  Sw;             /*! system clock */
    uint32_t  HS_Clock;             /*! high speed clock enable */
    uint32_t  LS_Clock;             /*! low speed clock enable */
} md_rcu_init_typedef;

/**
  * @}
  */

/* Exported Functions -------------------------------------------------------- */

/** @defgroup MD_RCU_Public_Functions RCU Public Functions
  * @{
  */

/** @defgroup CON
  * @{
  */
/**
  * @brief  Set RCU_CON Register
  * @param  rcu RCU Instance
  * @param  con
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_con(RCU_TypeDef *rcu, uint32_t con)
{
    WRITE_REG(rcu->CON, con);
}

/**
  * @brief  Get RCU_CON Register
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_con(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->CON));
}

/**
  * @brief  Clock security system enable(CSS)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc_css(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_CSSON);
}

/**
  * @brief  Clock security system disable(CSS)
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc_css(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_CSSON);
}

/**
  * @brief  Get Clock security system (CSS)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Hosc css enable
  *         @arg 0x0 : Hosc css disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc_css(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_CSSON) == (RCU_CON_CSSON));
}

/**
  * @brief  Get PLL clock ready flag
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PLLRDY_NOT_READY
  *         @arg @ref MD_RCU_PLLRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_pll_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_PLLRDY) == (RCU_CON_PLLRDY));
}

/**
  * @brief  PLL enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_pll(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_PLLON);
}

/**
  * @brief  PLL disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_pll(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_PLLON);
}

/**
  * @brief  Get PLL bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : PLL enable
  *         @arg 0x0 : PLL disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_pll(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_PLLON) == (RCU_CON_PLLON));
}

/**
  * @brief  HOSC bypass mode enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc_bypass(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HOSCBYP);
}

/**
  * @brief  HOSC bypass mode disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc_bypass(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HOSCBYP);
}

/**
  * @brief  Get HOSC bypass mode bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Hosc bypass enable
  *         @arg 0x0 : Hosc bypass disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc_bypass(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCBYP) == (RCU_CON_HOSCBYP));
}

/**
  * @brief  Get HOSC clock ready flag
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HOSCRDY_NOT_READY
  *         @arg @ref MD_RCU_HOSCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCRDY) == (RCU_CON_HOSCRDY));
}

/**
  * @brief  HOSC enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HOSCON);
}

/**
  * @brief  HOSC disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hosc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HOSCON);
}

/**
  * @brief  Get HOSC bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Hosc enable
  *         @arg 0x0 : Hosc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HOSCON) == (RCU_CON_HOSCON));
}

/**
  * @brief  Get HRCRDY clock ready flag
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HRCRDY_NOT_READY
  *         @arg @ref MD_RCU_HRCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HRCRDY) == (RCU_CON_HRCRDY));
}

/**
  * @brief  HRC enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_hrc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CON, RCU_CON_HRCON);
}

/**
  * @brief  HRC disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_hrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CON, RCU_CON_HRCON);
}

/**
  * @brief  Get HRC bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Hrc enable
  *         @arg 0x0 : Hrc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_hrc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CON, RCU_CON_HRCON) == (RCU_CON_HRCON));
}
/**
  * @}
  */

/** @defgroup CFG
  * @{
  */
/**
  * @brief  Set RCU_CFG Register
  * @param  rcu RCU Instance
  * @param  cfg
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_cfg(RCU_TypeDef *rcu, uint32_t cfg)
{
    WRITE_REG(rcu->CFG, cfg);
}

/**
  * @brief  Get RCU_CFG Register
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_cfg(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->CFG));
}

/**
  * @brief  Set MCO clock output prescaler
  * @note   Output Frequency = MCO/(MPRE+1)
  * @param  rcu RCU Instance
  * @param  mpre This parameter can be one of the following values:
  *         @arg @ref MD_RCU_MPRE_MCO_DIV1
  *         @arg @ref MD_RCU_MPRE_MCO_DIV2
  *         @arg @ref MD_RCU_MPRE_MCO_DIV4
  *         @arg @ref MD_RCU_MPRE_MCO_DIV8
  *         @arg @ref MD_RCU_MPRE_MCO_DIV16
  *         @arg @ref MD_RCU_MPRE_MCO_DIV32
  *         @arg @ref MD_RCU_MPRE_MCO_DIV64
  *         @arg @ref MD_RCU_MPRE_MCO_DIV128
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_mco_div(RCU_TypeDef *rcu, md_rcu_mco_div_t mpre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_MPRE, mpre << RCU_CFG_MPRE_POSS);
}

/**
  * @brief  Get MCO clock output prescaler
  * @note   Output Frequency = MCO/(MPRE+1)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_MPRE_MCO_DIV1
  *         @arg @ref MD_RCU_MPRE_MCO_DIV2
  *         @arg @ref MD_RCU_MPRE_MCO_DIV4
  *         @arg @ref MD_RCU_MPRE_MCO_DIV8
  *         @arg @ref MD_RCU_MPRE_MCO_DIV16
  *         @arg @ref MD_RCU_MPRE_MCO_DIV32
  *         @arg @ref MD_RCU_MPRE_MCO_DIV64
  *         @arg @ref MD_RCU_MPRE_MCO_DIV128
  */
__STATIC_INLINE uint32_t md_rcu_get_mco_div(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_MPRE) >> RCU_CFG_MPRE_POSS);
}

/**
  * @brief  Set MCO clock output (MCO)
  * @param  rcu RCU Instance
  * @param  msw This parameter can be one of the following values:
  *         @arg @ref MD_RCU_MSW_MCO_DISABLE
  *         @arg @ref MD_RCU_MSW_MCO_LRC
  *         @arg @ref MD_RCU_MSW_MCO_HRC
  *         @arg @ref MD_RCU_MSW_MCO_HOSC
  *         @arg @ref MD_RCU_MSW_MCO_PLL
  *         @arg @ref MD_RCU_MSW_MCO_SYSCLK
  *         @arg @ref MD_RCU_MSW_MCO_HCLK
  *         @arg @ref MD_RCU_MSW_MCO_PCLK
  *         @arg @ref MD_RCU_MSW_MCO_ADC
  *         @arg @ref MD_RCU_MSW_MCO_I2C_KERNEL
  *         @arg @ref MD_RCU_MSW_MCO_ADC_KERNEL
  *         @arg @ref MD_RCU_MSW_MCO_TIMER_KERNEL
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_mco_source(RCU_TypeDef *rcu, md_rcu_mco_source_t msw)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_MSW, msw << RCU_CFG_MSW_POSS);
}

/**
  * @brief  Get Microcontroller clock output (MCO)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_MSW_MCO_DISABLE
  *         @arg @ref MD_RCU_MSW_MCO_LRC
  *         @arg @ref MD_RCU_MSW_MCO_HRC
  *         @arg @ref MD_RCU_MSW_MCO_HOSC
  *         @arg @ref MD_RCU_MSW_MCO_PLL
  *         @arg @ref MD_RCU_MSW_MCO_SYSCLK
  *         @arg @ref MD_RCU_MSW_MCO_HCLK
  *         @arg @ref MD_RCU_MSW_MCO_PCLK
  *         @arg @ref MD_RCU_MSW_MCO_ADC
  *         @arg @ref MD_RCU_MSW_MCO_I2C_KERNEL
  *         @arg @ref MD_RCU_MSW_MCO_ADC_KERNEL
  *         @arg @ref MD_RCU_MSW_MCO_TIMER_KERNEL
  */
__STATIC_INLINE uint32_t md_rcu_get_mco_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_MSW) >> RCU_CFG_MSW_POSS);
}

/**
  * @brief  Set PLL reference clock source
  * @param  rcu RCU Instance
  * @param  pllsrc This parameter can be one of the following values:
  *         @arg @ref MD_RCU_PLLSRC_HRC
  *         @arg @ref MD_RCU_PLLSRC_HOSC
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_pll_source(RCU_TypeDef *rcu, md_rcu_pll_source_t pllsrc)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_PLLSRC, pllsrc << RCU_CFG_PLLSRC_POSS);
}

/**
  * @brief  Get PLL reference clock source
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PLLSRC_HRC
  *         @arg @ref MD_RCU_PLLSRC_HOSC
  */
__STATIC_INLINE uint32_t md_rcu_get_pll_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_PLLSRC) >> RCU_CFG_PLLSRC_POSS);
}

/**
  * @brief  Set PLL reference clock pre-divider
  * @param  rcu RCU Instance
  * @param  prediv This parameter can be one of the following values:
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV1
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV2
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV3
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV4
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV5
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV6
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV7
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV8
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV9
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV10
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV11
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV12
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV13
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV14
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV15
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV16
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_pll_prediv(RCU_TypeDef *rcu, uint32_t prediv)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_PREDIV, prediv << RCU_CFG_PREDIV_POSS);
}
/**
  * @brief  Get PLL reference clock pre-divider
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV1
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV2
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV3
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV4
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV5
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV6
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV7
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV8
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV9
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV10
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV11
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV12
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV13
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV14
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV15
  *         @arg @ref MD_RCU_PREDIV_PLL_DIV16
  */
__STATIC_INLINE uint32_t md_rcu_get_pll_prediv(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_PREDIV) >> RCU_CFG_PREDIV_POSS);
}

/**
  * @brief  Set APB clock prescaler(PCLK)
  * @param  rcu RCU Instance
  * @param  ppre This parameter can be one of the following values:
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_1
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_2
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_4
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_8
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_16
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_pclk_div(RCU_TypeDef *rcu, md_rcu_apb_clock_div_t ppre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_PPRE, ppre << RCU_CFG_PPRE_POSS);
}

/**
  * @brief  Get APB clock prescaler(PCLK)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_1
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_2
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_4
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_8
  *         @arg @ref MD_RCU_PPRE_HCLK_DIV_16
  */
__STATIC_INLINE uint32_t md_rcu_get_pclk_div(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_PPRE) >> RCU_CFG_PPRE_POSS);
}

/**
  * @brief  Set AHB clock prescaler(HCLK)
  * @param  rcu RCU Instance
  * @param  hpre This parameter can be one of the following values:
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_1
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_2
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_4
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_8
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_16
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_64
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_128
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_256
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_512
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hclk_div(RCU_TypeDef *rcu, md_rcu_ahb_clock_div_t hpre)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_HPRE, hpre << RCU_CFG_HPRE_POSS);
}

/**
  * @brief  Get AHB clock prescaler(HCLK)
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_1
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_2
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_4
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_8
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_16
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_64
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_128
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_256
  *         @arg @ref MD_RCU_HPRE_SYSCLK_DIV_512
  */
__STATIC_INLINE uint32_t md_rcu_get_hclk_div(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_HPRE) >> RCU_CFG_HPRE_POSS);
}

/**
  * @brief  System clock selection status
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_SWS_SYSCLK_HRC
  *         @arg @ref MD_RCU_SWS_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SWS_SYSCLK_PLL
  *         @arg @ref MD_RCU_SWS_SYSCLK_LRC
  */
__STATIC_INLINE uint32_t md_rcu_get_current_system_clock(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_SWS) >> RCU_CFG_SWS_POSS);
}

/**
  * @brief  Select system clock (SYSCLK) source
  * @param  rcu RCU Instance
  * @param  sw This parameter can be one of the following values:
  *         @arg @ref MD_RCU_SW_SYSCLK_HRC
  *         @arg @ref MD_RCU_SW_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SW_SYSCLK_PLL
  *         @arg @ref MD_RCU_SW_SYSCLK_LRC

  * @retval None
  */
__STATIC_INLINE void md_rcu_set_system_clock_source(RCU_TypeDef *rcu, md_rcu_clock_t sw)
{
    MODIFY_REG(rcu->CFG, RCU_CFG_SW, sw << RCU_CFG_SW_POSS);
}

/**
  * @brief  Get Select system clock (SYSCLK) source
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_SW_SYSCLK_HRC
  *         @arg @ref MD_RCU_SW_SYSCLK_HOSC
  *         @arg @ref MD_RCU_SW_SYSCLK_PLL
  *         @arg @ref MD_RCU_SW_SYSCLK_LRC
  */
__STATIC_INLINE uint32_t md_rcu_get_system_clock_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG, RCU_CFG_SW) >> RCU_CFG_SW_POSS);
}
/**
  * @}
  */

/** @defgroup CFG1
  * @{
  */
/**
  * @brief  Set RCU_CFG1 Register
  * @param  rcu RCU Instance
  * @param  pll
  */
__STATIC_INLINE void md_rcu_set_cfg1(RCU_TypeDef *rcu, uint32_t cfg1)
{
    WRITE_REG(rcu->CFG1, cfg1);
}

/**
  * @brief  Get RCU_CFG1 Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_cfg1(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->CFG1));
}

/**
  * @brief  Set PLL VCO frequency fractional multiplication value
  * @param  rcu RCU Instance
  * @param  fk
  * @retval
  */
__STATIC_INLINE void md_rcu_set_pll_fk(RCU_TypeDef *rcu, uint32_t fk)
{
    MODIFY_REG(rcu->CFG1, RCU_CFG1_FK, (fk << RCU_CFG1_FK_POSS));
}

/**
  * @brief  Get PLL VCO frequency fractional multiplication value
  * @param  rcu RCU Instance
  * @retval fk
  */
__STATIC_INLINE uint32_t md_rcu_get_pll_fk(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG1, RCU_CFG1_FK) >> RCU_CFG1_FK_POSS);
}

/**
  * @brief  Set PLL VCO frequency integer multiplication value
  * @param  rcu RCU Instance
  * @param  fn
  * @retval
  */
__STATIC_INLINE void md_rcu_set_pll_fn(RCU_TypeDef *rcu, uint32_t fn)
{
    MODIFY_REG(rcu->CFG1, RCU_CFG1_FN, (fn << RCU_CFG1_FN_POSS));
}

/**
  * @brief  Get PLL VCO frequency integer multiplication value
  * @param  rcu RCU Instance
  * @retval fn
  */
__STATIC_INLINE uint32_t md_rcu_get_pll_fn(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG1, RCU_CFG1_FN) >> RCU_CFG1_FN_POSS);
}

/**
  * @brief  Set PLL clock output frequency division ratio
  * @param  rcu RCU Instance
  *         @arg @ref MD_RCU_FM_PLL_DIV_2
  *         @arg @ref MD_RCU_FM_PLL_DIV_4
  *         @arg @ref MD_RCU_FM_PLL_DIV_6
  *         @arg @ref MD_RCU_FM_PLL_DIV_8
  *         @arg @ref MD_RCU_FM_PLL_DIV_12
  *         @arg @ref MD_RCU_FM_PLL_DIV_16
  *         @arg @ref MD_RCU_FM_PLL_DIV_24
  *         @arg @ref MD_RCU_FM_PLL_DIV_32
  *         @arg @ref MD_RCU_FM_PLL_DIV_64
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_pll_fm(RCU_TypeDef *rcu, md_rcu_pll_fvco_div_t fm)
{
    MODIFY_REG(rcu->CFG1, RCU_CFG1_FM, fm << RCU_CFG1_FM_POSS);
}

/**
  * @brief  Get PLL clock output frequency division ratio
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_FM_PLL_DIV_2
  *         @arg @ref MD_RCU_FM_PLL_DIV_4
  *         @arg @ref MD_RCU_FM_PLL_DIV_6
  *         @arg @ref MD_RCU_FM_PLL_DIV_8
  *         @arg @ref MD_RCU_FM_PLL_DIV_12
  *         @arg @ref MD_RCU_FM_PLL_DIV_16
  *         @arg @ref MD_RCU_FM_PLL_DIV_24
  *         @arg @ref MD_RCU_FM_PLL_DIV_32
  *         @arg @ref MD_RCU_FM_PLL_DIV_64
  */
__STATIC_INLINE uint32_t md_rcu_get_pll_fm(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG1, RCU_CFG1_FM) >> RCU_CFG1_FM_POSS);
}
/**
  * @}
  */

/** @defgroup CFG2
  * @{
  */
/**
  * @brief  Set RCU_CFG2 Register
  * @param  rcu RCU Instance
  * @param  cfg2
  */
__STATIC_INLINE void md_rcu_set_cfg2(RCU_TypeDef *rcu, uint32_t cfg2)
{
    WRITE_REG(rcu->CFG2, cfg2);
}

/**
  * @brief  Get RCU_CFG2 Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_cfg2(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->CFG2));
}

/**
  * @brief  Set Current system clock frequency
  * @param  rcu RCU Instance
  * @param  sysfreq
  *         @arg Max Value 0xFF
  *         @arg Min Value 0x4
  * @retval
  */
__STATIC_INLINE void md_rcu_set_current_system_frequency(RCU_TypeDef *rcu, uint32_t sysfreq)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_SYSFREQ, (sysfreq << RCU_CFG2_SYSFREQ_POSS));
}

/**
  * @brief  Get Current system clock frequency
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xFF
  *         @arg Min Value 0x4
  */
__STATIC_INLINE uint32_t md_rcu_get_current_system_frequency(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_SYSFREQ) >> RCU_CFG2_SYSFREQ_POSS);
}

/**
  * @brief  Set PLL stabilization time setting
  * @param  rcu RCU Instance
  * @param  time
  *         @arg Max Value 0xFF
  *         @arg Min Value 0x4
  * @retval
  */
__STATIC_INLINE void md_rcu_set_pll_stable_time(RCU_TypeDef *rcu, uint32_t time)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_PLLRDYCNT, (time << RCU_CFG2_PLLRDYCNT_POSS));
}

/**
  * @brief  Get PLL stabilization time setting
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg Max Value 0xFF
  *         @arg Min Value 0x4
  */
__STATIC_INLINE uint32_t md_rcu_get_pll_stable_time(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_PLLRDYCNT) >> RCU_CFG2_PLLRDYCNT_POSS);
}

/**
  * @brief  HOSC 32 divider switch enalbe
  * @param  rcu RCU Instance
  * @retval
  */
__STATIC_INLINE void md_rcu_enable_hosc_32_divider(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->CFG2, RCU_CFG2_HOSC32EN);
}

/**
  * @brief  HOSC 32 divider switch disalbe
  * @param  rcu RCU Instance
  * @retval
  */
__STATIC_INLINE void md_rcu_disable_hosc_32_divider(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->CFG2, RCU_CFG2_HOSC32EN);
}

/**
  * @brief  Check if HOSC 32 divider switch enabled.
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_uart_is_enabled_hosc_32_divider(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->CFG2, RCU_CFG2_HOSC32EN) == (RCU_CFG2_HOSC32EN));
}

/**
  * @brief  Set I2C kernel clock source
  * @param  rcu RCU Instance
  * @param  ksrc
  * @retval
  */
__STATIC_INLINE void md_rcu_set_i2c_kernel_source(RCU_TypeDef *rcu, uint32_t ksrc)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_I2C_KSRC, (ksrc << RCU_CFG2_I2C_KSRC_POSS));
}

/**
  * @brief  Get I2C kernel clock source
  * @param  rcu RCU Instance
  * @retval ksrc
  */
__STATIC_INLINE uint32_t md_rcu_get_i2c_kernel_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_I2C_KSRC) >> RCU_CFG2_I2C_KSRC_POSS);
}

/**
  * @brief  Set ADC kernel clock source
  * @param  rcu RCU Instance
  * @param  ksrc
            @arg MD_RCU_CFG2_ADC_KERNEL_DISABLE
            @arg MD_RCU_CFG2_ADC_KERNEL_APBCLK
            @arg MD_RCU_CFG2_ADC_KERNEL_SYSCLK
            @arg MD_RCU_CFG2_ADC_KERNEL_HRCCLK
  * @retval
  */
__STATIC_INLINE void md_rcu_set_adc_kernel_source(RCU_TypeDef *rcu, uint32_t ksrc)
{
    MODIFY_REG(rcu->CFG2, RCU_CFG2_ADC_KSRC, (ksrc << RCU_CFG2_ADC_KSRC_POSS));
}

/**
  * @brief  Get ADC kernel clock source
  * @param  rcu RCU Instance
  * @retval ksrc
            @arg MD_RCU_CFG2_ADC_KERNEL_DISABLE
            @arg MD_RCU_CFG2_ADC_KERNEL_APBCLK
            @arg MD_RCU_CFG2_ADC_KERNEL_SYSCLK
            @arg MD_RCU_CFG2_ADC_KERNEL_HRCCLK
  */
__STATIC_INLINE uint32_t md_rcu_get_adc_kernel_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->CFG2, RCU_CFG2_ADC_KSRC) >> RCU_CFG2_ADC_KSRC_POSS);
}

/**
  * @}
  */

/** @defgroup IER
  * @{
  */
/**
  * @brief  Set RCU_IER Register
  * @param  rcu RCU Instance
  * @param  ier
  */
__STATIC_INLINE void md_rcu_set_ier(RCU_TypeDef *rcu, uint32_t ier)
{
    WRITE_REG(rcu->IER, ier);
}

/**
  * @brief  Clock security system interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_CSSHOSC);
}

/**
  * @brief  PLL clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_pll_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_PLLRDY);
}

/**
  * @brief  HOSC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_HOSCRDY);
}

/**
  * @brief  HRC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_HRCRDY);
}

/**
  * @brief  LRC clock source stable interrupt enable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_enable_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IER, RCU_IER_LRCRDY);
}
/**
  * @}
  */

/** @defgroup IDR
  * @{
  */
/**
  * @brief  Set RCU_IDR Register
  * @param  rcu RCU Instance
  * @param  idr
  */
__STATIC_INLINE void md_rcu_set_idr(RCU_TypeDef *rcu, uint32_t idr)
{
    WRITE_REG(rcu->IDR, idr);
}

/**
  * @brief  Clock security system interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_CSSHOSC);
}

/**
  * @brief  PLL clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_pll_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_PLLRDY);
}

/**
  * @brief  HOSC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_HOSCRDY);
}

/**
  * @brief  HRC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_HRCRDY);
}

/**
  * @brief  LRC clock source stable interrupt disable
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE void md_rcu_disable_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->IDR, RCU_IDR_LRCRDY);
}
/**
  * @}
  */

/** @defgroup IVS
  * @{
  */
/**
  * @brief  Set RCU_IVS Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_ivs(RCU_TypeDef *rcu)
{
    return (uint32_t) READ_REG(rcu->IVS);
}

/**
  * @brief  Clock security system interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_CSSHOSC) == (RCU_IVS_CSSHOSC));
}

/**
  * @brief  PLL clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_pll_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_PLLRDY) == (RCU_IVS_PLLRDY));
}

/**
  * @brief  HOSC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_HOSCRDY) == (RCU_IVS_HOSCRDY));
}

/**
  * @brief  HRC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_HRCRDY) == (RCU_IVS_HRCRDY));
}

/**
  * @brief  LRC clock source stable interrupt active state
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IVS, RCU_IVS_LRCRDY) == (RCU_IVS_LRCRDY));
}
/**
  * @}
  */

/** @defgroup RIF
  * @{
  */
/**
  * @brief  Set RCU_RIF Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_rif(RCU_TypeDef *rcu)
{
    return (uint32_t)READ_REG(rcu->RIF);
}

/**
  * @brief  Clock security system unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_CSSHOSC) == (RCU_RIF_CSSHOSC));
}

/**
  * @brief  PLL clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_pll_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_PLLRDY) == (RCU_RIF_PLLRDY));
}

/**
  * @brief  HOSC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_HOSCRDY) == (RCU_RIF_HOSCRDY));
}

/**
  * @brief  HRC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_HRCRDY) == (RCU_RIF_HRCRDY));
}

/**
  * @brief  LRC clock source stable unmasked interrupt flag status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_active_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RIF, RCU_RIF_LRCRDY) == (RCU_RIF_LRCRDY));
}
/**
  * @}
  */

/** @defgroup IFM
  * @{
  */
/**
  * @brief  Set RCU_IFM Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_ifm(RCU_TypeDef *rcu)
{
    return (uint32_t)READ_REG(rcu->IFM);
}

/**
  * @brief  Clock security system interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_css_hosc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_CSSHOSC) == (RCU_IFM_CSSHOSC));
}

/**
  * @brief  PLL clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_pll_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_PLLRDY) == (RCU_IFM_PLLRDY));
}

/**
  * @brief  HOSC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_hosc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_HOSCRDY) == (RCU_IFM_HOSCRDY));
}

/**
  * @brief  HRC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_hrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_HRCRDY) == (RCU_IFM_HRCRDY));
}

/**
  * @brief  LRC clock source stable interrupt flag mask status
  * @param  rcu RCU Instance
  * @retval None.
  */
__STATIC_INLINE uint32_t md_rcu_is_masked_it_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->IFM, RCU_IFM_LRCRDY) == (RCU_IFM_LRCRDY));
}
/**
  * @}
  */

/** @defgroup ICR
  * @{
  */
/**
  * @brief  Set RCU_ICR Register
  * @param  rcu RCU Instance
  * @param  icr
  */
__STATIC_INLINE void md_rcu_set_icr(RCU_TypeDef *rcu, uint32_t icr)
{
    WRITE_REG(rcu->ICR, icr);
}

/**
  * @brief  Clear Clock security system interrupt
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_css_hosc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_CSSHOSC);
}

/**
  * @brief  Clear PLL clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_pll_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_PLLRDY);
}

/**
  * @brief  Clear HOSC clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_hosc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_HOSCRDY);
}

/**
  * @brief  Clear HRC clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_hrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_HRCRDY);
}

/**
  * @brief  Clear LRC clock source stable interrupt flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_it_lrc_ready(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->ICR, RCU_ICR_LRCRDY);
}
/**
  * @}
  */

/** @defgroup AHBRST
  * @{
  */
/**
  * @brief  Set RCU_AHBRST Register
  * @param  rcu RCU Instance
  * @param  ahbrst
  */
__STATIC_INLINE void md_rcu_set_ahbrst(RCU_TypeDef *rcu, uint32_t ahbrst)
{
    WRITE_REG(rcu->AHBRST, ahbrst);
}

/**
  * @brief  Get RCU_AHBRST Register
  * @param  rcu RCU Instance
  * @param  ahbrst
  */
__STATIC_INLINE uint32_t md_rcu_get_ahbrst(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->AHBRST));
}

/**
  * @brief  ADC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_adc_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_ADCEN);
}

/**
  * @brief  ADC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_adc_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_ADCEN);
}

/**
  * @brief  Get ADC reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_adc_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_ADCEN) == (RCU_AHBRST_ADCEN));
}

/**
  * @brief  GPIOC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN);
}

/**
  * @brief  GPIOC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN);
}

/**
  * @brief  Get GPIOC reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPCEN) == (RCU_AHBRST_GPCEN));
}

/**
  * @brief  GPIOB reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiob_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPBEN);
}

/**
  * @brief  GPIOB reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiob_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPBEN);
}

/**
  * @brief  Get GPIOB reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiob_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPBEN) == (RCU_AHBRST_GPBEN));
}

/**
  * @brief  GPIOA reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioa_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_GPAEN);
}

/**
  * @brief  GPIOA reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioa_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_GPAEN);
}

/**
  * @brief  Get GPIOA reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioa_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_GPAEN) == (RCU_AHBRST_GPAEN));
}

/**
  * @brief  CALC reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_calc_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_CALCEN);
}

/**
  * @brief  CALC reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_calc_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_CALCEN);
}

/**
  * @brief  Get CALC reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_calc_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_CALCEN) == (RCU_AHBRST_CALCEN));
}

/**
  * @brief  SVA reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_sva_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_SVAEN);
}

/**
  * @brief  SVA reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_sva_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_SVAEN);
}

/**
  * @brief  Get SVA reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_sva_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_SVAEN) == (RCU_AHBRST_SVAEN));
}

/**
  * @brief  IIR reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_iir_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_IIREN);
}

/**
  * @brief  IIR reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_iir_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_IIREN);
}

/**
  * @brief  Get IIR reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_iir_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_IIREN) == (RCU_AHBRST_IIREN));
}

/**
  * @brief  PI reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_pi_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_PIEN);
}

/**
  * @brief  PI reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_pi_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_PIEN);
}

/**
  * @brief  Get PI reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_pi_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_PIEN) == (RCU_AHBRST_PIEN));
}

/**
  * @brief  DMAMUX reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_dmamux_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_DMAMUXEN);
}

/**
  * @brief  DMAMUX reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_dmamux_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_DMAMUXEN);
}

/**
  * @brief  Get DMAMUX reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_dmamux_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_DMAMUXEN) == (RCU_AHBRST_DMAMUXEN));
}

/**
  * @brief  DMA1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_dma1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBRST, RCU_AHBRST_DMA1EN);
}

/**
  * @brief  DMA1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_dma1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBRST, RCU_AHBRST_DMA1EN);
}

/**
  * @brief  Get DMA1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_dma1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBRST, RCU_AHBRST_DMA1EN) == (RCU_AHBRST_DMA1EN));
}

/**
  * @}
  */

/** @defgroup APB1RST
  * @{
  */
/**
  * @brief  Set RCU_APB1RST Register
  * @param  rcu RCU Instance
  * @param  apb1rst
  */
__STATIC_INLINE void md_rcu_set_apb1rst(RCU_TypeDef *rcu, uint32_t apb1rst)
{
    WRITE_REG(rcu->APB1RST, apb1rst);
}

/**
  * @brief  Get RCU_APB1RST Register
  * @param  rcu RCU Instance
  * @param  apb1rst
  */
__STATIC_INLINE uint32_t md_rcu_get_apb1rst(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB1RST));
}

/**
  * @brief  CAN1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_can1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_CAN1EN);
}

/**
  * @brief  CAN1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_can1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_CAN1EN);
}

/**
  * @brief  Get CAN1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_can1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_CAN1EN) == (RCU_APB1RST_CAN1EN));
}

/**
  * @brief  I2C1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN);
}

/**
  * @brief  I2C1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN);
}

/**
  * @brief  Get I2C1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_I2C1EN) == (RCU_APB1RST_I2C1EN));
}

/**
  * @brief  UART2 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart2_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_UART2EN);
}

/**
  * @brief  UART2 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart2_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_UART2EN);
}

/**
  * @brief  Get UART2 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart2_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_UART2EN) == (RCU_APB1RST_UART2EN));
}

/**
  * @brief  WWDT reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN);
}

/**
  * @brief  WWDT reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN);
}

/**
  * @brief  Get WWDT reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_WWDTEN) == (RCU_APB1RST_WWDTEN));
}

/**
  * @brief  BS16T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN);
}

/**
  * @brief  BS16T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN);
}

/**
  * @brief  Get BS16T1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_BS16T1EN) == (RCU_APB1RST_BS16T1EN));
}

/**
  * @brief  GP32C4T2 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t2_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T2EN);
}

/**
  * @brief  GP32C4T2 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t2_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T2EN);
}

/**
  * @brief  Get GP32C4T2 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t2_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T2EN) == (RCU_APB1RST_GP32C4T2EN));
}

/**
  * @brief  GP32C4T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN);
}

/**
  * @brief  GP32C4T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN);
}

/**
  * @brief  Get GP32C4T1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1RST, RCU_APB1RST_GP32C4T1EN) == (RCU_APB1RST_GP32C4T1EN));
}
/**
  * @}
  */

/** @defgroup APB2RST
  * @{
  */
/**
  * @brief  Set RCU_APB2RST Register
  * @param  rcu RCU Instance
  * @param  apb2rst
  */
__STATIC_INLINE void md_rcu_set_apb2rst(RCU_TypeDef *rcu, uint32_t apb2rst)
{
    WRITE_REG(rcu->APB2RST, apb2rst);
}

/**
  * @brief  Get RCU_APB2RST Register
  * @param  rcu RCU Instance
  * @param  apb2rst
  */
__STATIC_INLINE uint32_t md_rcu_get_apb2rst(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB2RST));
}

/**
  * @brief  VRES reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_vres_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_VRESEN);
}

/**
  * @brief  VRES reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_vres_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_VRESEN);
}

/**
  * @brief  Get VRES reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_vres_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_VRESEN) == (RCU_APB2RST_VRESEN));
}

/**
  * @brief  CMP reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_cmp_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_CMPEN);
}

/**
  * @brief  CMP reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_cmp_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_CMPEN);
}

/**
  * @brief  Get CMP reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_cmp_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_CMPEN) == (RCU_APB2RST_CMPEN));
}

/**
  * @brief  GP16C2T2 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN);
}

/**
  * @brief  GP16C2T2 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN);
}

/**
  * @brief  Get GP16C2T2 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T2EN) == (RCU_APB2RST_GP16C2T2EN));
}

/**
  * @brief  GP16C2T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN);
}

/**
  * @brief  GP16C2T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN);
}

/**
  * @brief  Get GP16C2T1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_GP16C2T1EN) == (RCU_APB2RST_GP16C2T1EN));
}

/**
  * @brief  UART1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN);
}

/**
  * @brief  UART1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN);
}

/**
  * @brief  Get UART1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_UART1EN) == (RCU_APB2RST_UART1EN));
}

/**
  * @brief  AD16C6T2 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ad16c6t2_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_AD16C6T2EN);
}

/**
  * @brief  AD16C6T2 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ad16c6t2_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_AD16C6T2EN);
}

/**
  * @brief  Get AD16C6T2 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ad16c6t2_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_AD16C6T2EN) == (RCU_APB2RST_AD16C6T2EN));
}

/**
  * @brief  SPI1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN);
}

/**
  * @brief  SPI1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN);
}

/**
  * @brief  Get SPI1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_SPI1EN) == (RCU_APB2RST_SPI1EN));
}

/**
  * @brief  AD16C6T1 reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ad16c6t1_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_AD16C6T1EN);
}

/**
  * @brief  AD16C6T1 reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ad16c6t1_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_AD16C6T1EN);
}

/**
  * @brief  Get AD16C6T1 reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ad16c6t1_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_AD16C6T1EN) == (RCU_APB2RST_AD16C6T1EN));
}

/**
  * @brief  OPAMP reset request enable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_opamp_reset(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2RST, RCU_APB2RST_OPAMPEN);
}

/**
  * @brief  OPAMP reset request disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_opamp_reset(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2RST, RCU_APB2RST_OPAMPEN);
}

/**
  * @brief  Get OPAMP reset request bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : Reset enable
  *         @arg 0x0 : Reset disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_opamp_reset(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2RST, RCU_APB2RST_OPAMPEN) == (RCU_APB2RST_OPAMPEN));
}

/**
  * @}
  */

/** @defgroup AHBEN
  * @{
  */
/**
  * @brief  Set RCU_AHBEN Register
  * @param  rcu RCU Instance
  * @param  ahben
  */
__STATIC_INLINE void md_rcu_set_ahben(RCU_TypeDef *rcu, uint32_t ahben)
{
    WRITE_REG(rcu->AHBEN, ahben);
}

/**
  * @brief  Get RCU_AHBEN Register
  * @param  rcu RCU Instance
  * @param  ahben
  */
__STATIC_INLINE uint32_t md_rcu_get_ahben(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->AHBEN));
}

/**
  * @brief  ADC clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_adc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_ADCEN);
}

/**
  * @brief  ADC Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_adc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_ADCEN);
}

/**
  * @brief  Get ADC Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : adc enable
  *         @arg 0x0 : adc disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_adc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_ADCEN) == (RCU_AHBEN_ADCEN));
}

/**
  * @brief  GPIOC clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN);
}

/**
  * @brief  GPIOC Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN);
}

/**
  * @brief  Get GPIOC Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpioc enable
  *         @arg 0x0 : gpioc disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPCEN) == (RCU_AHBEN_GPCEN));
}

/**
  * @brief  GPIOB clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiob(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPBEN);
}

/**
  * @brief  GPIOB Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiob(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPBEN);
}

/**
  * @brief  Get GPIOB Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpiob enable
  *         @arg 0x0 : gpiob disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiob(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPBEN) == (RCU_AHBEN_GPBEN));
}

/**
  * @brief  GPIOA clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioa(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_GPAEN);
}

/**
  * @brief  GPIOA Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioa(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_GPAEN);
}

/**
  * @brief  Get GPIOA Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpioa enable
  *         @arg 0x0 : gpioa disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioa(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_GPAEN) == (RCU_AHBEN_GPAEN));
}

/**
  * @brief  CALC clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_calc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_CALCEN);
}

/**
  * @brief  CALC Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_calc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_CALCEN);
}

/**
  * @brief  Get CALC Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : calc enable
  *         @arg 0x0 : calc disable.
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_calc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_CALCEN) == (RCU_AHBEN_CALCEN));
}

/**
  * @brief  SVA clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_sva(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_SVAEN);
}

/**
  * @brief  SVA Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_sva(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_SVAEN);
}

/**
  * @brief  Get SVA Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : sva enable
  *         @arg 0x0 : sva disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_sva(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_SVAEN) == (RCU_AHBEN_SVAEN));
}

/**
  * @brief  IIR clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_iir(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_IIREN);
}

/**
  * @brief  IIR Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_iir(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_IIREN);
}

/**
  * @brief  Get IIR Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : iir enable
  *         @arg 0x0 : iir disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_iir(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_IIREN) == (RCU_AHBEN_IIREN));
}

/**
  * @brief  PI clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_pi(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_PIEN);
}

/**
  * @brief  PI Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_pi(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_PIEN);
}

/**
  * @brief  Get PI Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : pi enable
  *         @arg 0x0 : pi disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_pi(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_PIEN) == (RCU_AHBEN_PIEN));
}

/**
  * @brief  DMAMUX clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_dmamux(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_DMAMUXEN);
}

/**
  * @brief  DMAMUX Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_dmamux(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_DMAMUXEN);
}

/**
  * @brief  Get DMAMUX Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : dmamux enable
  *         @arg 0x0 : dmamux disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_dmamux(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_DMAMUXEN) == (RCU_AHBEN_DMAMUXEN));
}

/**
  * @brief  DMA1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_dma1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBEN, RCU_AHBEN_DMA1EN);
}

/**
  * @brief  DMA1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_dma1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBEN, RCU_AHBEN_DMA1EN);
}

/**
  * @brief  Get DMA1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : dma1 enable
  *         @arg 0x0 : dma1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_dma1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBEN, RCU_AHBEN_DMA1EN) == (RCU_AHBEN_DMA1EN));
}

/**
  * @}
  */

/** @defgroup APB1EN
  * @{
  */
/**
  * @brief  Set RCU_APB1EN Register
  * @param  rcu RCU Instance
  * @param  apb1en
  */
__STATIC_INLINE void md_rcu_set_apb1en(RCU_TypeDef *rcu, uint32_t apb1en)
{
    WRITE_REG(rcu->APB1EN, apb1en);
}

/**
  * @brief  Get RCU_APB1EN Register
  * @param  rcu RCU Instance
  * @param  apb1en
  */
__STATIC_INLINE uint32_t md_rcu_get_apb1en(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB1EN));
}

/**
  * @brief  CAN1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_can1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_CAN1EN);
}

/**
  * @brief  CAN1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_can1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_CAN1EN);
}

/**
  * @brief  Get CAN1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : can1 enable
  *         @arg 0x0 : can1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_can1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_CAN1EN) == (RCU_APB1EN_CAN1EN));
}

/**
  * @brief  I2C1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN);
}

/**
  * @brief  I2C1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN);
}

/**
  * @brief  Get I2C1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : i2c1 enable
  *         @arg 0x0 : i2c1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_I2C1EN) == (RCU_APB1EN_I2C1EN));
}

/**
  * @brief  UART2 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart2(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_UART2EN);
}

/**
  * @brief  UART2 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart2(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_UART2EN);
}

/**
  * @brief  Get UART2 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart2 enable
  *         @arg 0x0 : uart2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart2(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_UART2EN) == (RCU_APB1EN_UART2EN));
}

/**
  * @brief  WWDT clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN);
}

/**
  * @brief  WWDT Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN);
}

/**
  * @brief  Get WWDT Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : wwdt enable
  *         @arg 0x0 : wwdt disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_WWDTEN) == (RCU_APB1EN_WWDTEN));
}

/**
  * @brief  BS16T1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN);
}

/**
  * @brief  BS16T1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN);
}

/**
  * @brief  Get BS16T1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : bs16t1 enable
  *         @arg 0x0 : bs16t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_BS16T1EN) == (RCU_APB1EN_BS16T1EN));
}

/**
  * @brief  GP32C4T2 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t2(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T2EN);
}

/**
  * @brief  GP32C4T2 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t2(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T2EN);
}

/**
  * @brief  Get GP32C4T2 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp32c4t2 enable
  *         @arg 0x0 : gp32c4t2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t2(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T2EN) == (RCU_APB1EN_GP32C4T2EN));
}

/**
  * @brief  GP32C4T1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN);
}

/**
  * @brief  GP32C4T1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN);
}

/**
  * @brief  Get GP32C4T1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp32c4t1 enable
  *         @arg 0x0 : gp32c4t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1EN, RCU_APB1EN_GP32C4T1EN) == (RCU_APB1EN_GP32C4T1EN));
}
/**
  * @}
  */

/** @defgroup APB2EN
  * @{
  */
/**
  * @brief  Set RCU_APB2EN Register
  * @param  rcu RCU Instance
  * @param  apb2en
  */
__STATIC_INLINE void md_rcu_set_apb2en(RCU_TypeDef *rcu, uint32_t apb2en)
{
    WRITE_REG(rcu->APB2EN, apb2en);
}

/**
  * @brief  Get RCU_APB2EN Register
  * @param  rcu RCU Instance
  * @param  apb2en
  */
__STATIC_INLINE uint32_t md_rcu_get_apb2en(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB2EN));
}

/**
  * @brief  VRES clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_vres(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_VRESEN);
}

/**
  * @brief  VRES Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_vres(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_VRESEN);
}

/**
  * @brief  Get VRES Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : cmp enable
  *         @arg 0x0 : cmp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_vres(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_VRESEN) == (RCU_APB2EN_VRESEN));
}

/**
  * @brief  CMP clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_cmp(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_CMPEN);
}

/**
  * @brief  CMP Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_cmp(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_CMPEN);
}

/**
  * @brief  Get CMP Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : cmp enable
  *         @arg 0x0 : cmp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_cmp(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_CMPEN) == (RCU_APB2EN_CMPEN));
}

/**
  * @brief  GP16C2T2 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN);
}

/**
  * @brief  GP16C2T2 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN);
}

/**
  * @brief  Get GP16C2T2 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t2 enable
  *         @arg 0x0 : gp16c2t2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T2EN) == (RCU_APB2EN_GP16C2T2EN));
}

/**
  * @brief  GP16C2T1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN);
}

/**
  * @brief  GP16C2T1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN);
}

/**
  * @brief  Get GP16C2T1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t1 enable
  *         @arg 0x0 : gp16c2t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_GP16C2T1EN) == (RCU_APB2EN_GP16C2T1EN));
}

/**
  * @brief  UART1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN);
}

/**
  * @brief  UART1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN);
}

/**
  * @brief  Get UART1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart1 enable
  *         @arg 0x0 : uart1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_UART1EN) == (RCU_APB2EN_UART1EN));
}

/**
  * @brief  AD16C6T2 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ad16c6t2(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_AD16C6T2EN);
}

/**
  * @brief  AD16C6T2 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ad16c6t2(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_AD16C6T2EN);
}

/**
  * @brief  Get AD16C6T2 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : ad16c6t1 enable
  *         @arg 0x0 : ad16c6t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ad16c6t2(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_AD16C6T2EN) == (RCU_APB2EN_AD16C6T2EN));
}

/**
  * @brief  SPI1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN);
}

/**
  * @brief  SPI1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN);
}

/**
  * @brief  Get SPI1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : spi1 enable
  *         @arg 0x0 : spi1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_SPI1EN) == (RCU_APB2EN_SPI1EN));
}

/**
  * @brief  AD16C6T1 clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ad16c6t1(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_AD16C6T1EN);
}

/**
  * @brief  AD16C6T1 Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ad16c6t1(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_AD16C6T1EN);
}

/**
  * @brief  Get AD16C6T1 Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : ad16c6t1 enable
  *         @arg 0x0 : ad16c6t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ad16c6t1(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_AD16C6T1EN) == (RCU_APB2EN_AD16C6T1EN));
}

/**
  * @brief  OPAMP clock enable bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_opamp(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2EN, RCU_APB2EN_OPAMPEN);
}

/**
  * @brief  OPAMP Clock disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_opamp(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2EN, RCU_APB2EN_OPAMPEN);
}

/**
  * @brief  Get OPAMP Clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : opamp enable
  *         @arg 0x0 : opamp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_opamp(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2EN, RCU_APB2EN_OPAMPEN) == (RCU_APB2EN_OPAMPEN));
}

/**
  * @}
  */

/** @defgroup AHBSL
  * @{
  */
/**
  * @brief  Set RCU_AHBSL Register
  * @param  rcu RCU Instance
  * @param  ahbsl
  */
__STATIC_INLINE void md_rcu_set_ahbsl(RCU_TypeDef *rcu, uint32_t ahbsl)
{
    WRITE_REG(rcu->AHBSL, ahbsl);
}

/**
  * @brief  Get RCU_AHBSL Register
  * @param  rcu RCU Instance
  * @param  ahbsl
  */
__STATIC_INLINE uint32_t md_rcu_get_ahbsl(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->AHBSL));
}

/**
  * @brief  ADC clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_adc_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_ADCEN);
}

/**
  * @brief  ADC clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_adc_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_ADCEN);
}

/**
  * @brief  Get ADC clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : adc enable
  *         @arg 0x0 : adc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_adc_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_ADCEN) == (RCU_AHBSL_ADCEN));
}

/**
  * @brief  GPC clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN);
}

/**
  * @brief  GPC clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN);
}

/**
  * @brief  Get GPC clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpioc enable
  *         @arg 0x0 : gpioc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioc_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPCEN) == (RCU_AHBSL_GPCEN));
}

/**
  * @brief  GPB clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpiob_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPBEN);
}

/**
  * @brief  GPB clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpiob_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPBEN);
}

/**
  * @brief  Get GPB clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpiob enable
  *         @arg 0x0 : gpiob disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpiob_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPBEN) == (RCU_AHBSL_GPBEN));
}

/**
  * @brief  GPA clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gpioa_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_GPAEN);
}

/**
  * @brief  GPA clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gpioa_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_GPAEN);
}

/**
  * @brief  Get GPA clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gpioa enable
  *         @arg 0x0 : gpioa disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gpioa_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_GPAEN) == (RCU_AHBSL_GPAEN));
}

/**
  * @brief  CALC clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_calc_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_CALCEN);
}

/**
  * @brief  CALC clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_calc_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_CALCEN);
}

/**
  * @brief  Get CALC clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : calc enable
  *         @arg 0x0 : calc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_calc_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_CALCEN) == (RCU_AHBSL_CALCEN));
}

/**
  * @brief  SVA clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_sva_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_SVAEN);
}

/**
  * @brief  SVA clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_sva_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_SVAEN);
}

/**
  * @brief  Get SVA clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : sva enable
  *         @arg 0x0 : sva disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_sva_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_SVAEN) == (RCU_AHBSL_SVAEN));
}

/**
  * @brief  IIR clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_iir_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_IIREN);
}

/**
  * @brief  IIR clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_iir_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_IIREN);
}

/**
  * @brief  Get IIR clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : iir enable
  *         @arg 0x0 : iir disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_iir_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_IIREN) == (RCU_AHBSL_IIREN));
}

/**
  * @brief  PI clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_pi_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_PIEN);
}

/**
  * @brief  PI clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_pi_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_PIEN);
}

/**
  * @brief  Get PI clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : pi enable
  *         @arg 0x0 : pi disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_pi_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_PIEN) == (RCU_AHBSL_PIEN));
}

/**
  * @brief  DMAMUX clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_dmamux_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_DMAMUXEN);
}

/**
  * @brief  DMAMUX clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_dmamux_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_DMAMUXEN);
}

/**
  * @brief  Get DMAMUX clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : dmamux enable
  *         @arg 0x0 : dmamux disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_dmamux_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_DMAMUXEN) == (RCU_AHBSL_DMAMUXEN));
}

/**
  * @brief  DMA1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_dma1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->AHBSL, RCU_AHBSL_DMA1EN);
}

/**
  * @brief  DMA1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_dma1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->AHBSL, RCU_AHBSL_DMA1EN);
}

/**
  * @brief  Get DMA1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : dma1 enable
  *         @arg 0x0 : dma1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_dma1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->AHBSL, RCU_AHBSL_DMA1EN) == (RCU_AHBSL_DMA1EN));
}

/**
  * @}
  */

/** @defgroup APB1SL
  * @{
  */
/**
  * @brief  Set RCU_APB1SL Register
  * @param  rcu RCU Instance
  * @param  apb1sl
  */
__STATIC_INLINE void md_rcu_set_apb1sl(RCU_TypeDef *rcu, uint32_t apb1sl)
{
    WRITE_REG(rcu->APB1SL, apb1sl);
}

/**
  * @brief  Get RCU_APB1SL Register
  * @param  rcu RCU Instance
  * @param  apb1sl
  */
__STATIC_INLINE uint32_t md_rcu_get_apb1sl(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB1SL));
}

/**
  * @brief  CAN1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_can1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_CAN1EN);
}

/**
  * @brief  CAN1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_can1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_CAN1EN);
}

/**
  * @brief  Get CAN1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : can1 enable
  *         @arg 0x0 : can1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_can1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_CAN1EN) == (RCU_APB1SL_CAN1EN));
}

/**
  * @brief  I2C1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN);
}

/**
  * @brief  I2C1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN);
}

/**
  * @brief  Get I2C1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : i2c1 enable
  *         @arg 0x0 : i2c1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_i2c1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_I2C1EN) == (RCU_APB1SL_I2C1EN));
}

/**
  * @brief  UART2 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart2_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_UART2EN);
}

/**
  * @brief  UART2 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart2_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_UART2EN);
}

/**
  * @brief  Get UART2 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart2 enable
  *         @arg 0x0 : uart2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart2_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_UART2EN) == (RCU_APB1SL_UART2EN));
}

/**
  * @brief  WWDT clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN);
}

/**
  * @brief WWDT clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN);
}

/**
  * @brief  Get WWDT clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : wwdt enable
  *         @arg 0x0 : wwdt disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_wwdt_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_WWDTEN) == (RCU_APB1SL_WWDTEN));
}

/**
  * @brief  BS16T1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN);
}

/**
  * @brief  BS16T1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN);
}

/**
  * @brief  Get BS16T1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : bs16t1 enable
  *         @arg 0x0 : bs16t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_bs16t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_BS16T1EN) == (RCU_APB1SL_BS16T1EN));
}

/**
  * @brief  GP32C4T2 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T2EN);
}

/**
  * @brief  GP32C4T2 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T2EN);
}

/**
  * @brief  Get GP32C4T2 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp32c4t2 enable
  *         @arg 0x0 : gp32c4t2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T2EN) == (RCU_APB1SL_GP32C4T2EN));
}

/**
  * @brief  GP32C4T1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN);
}

/**
  * @brief  GP32C4T1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN);
}

/**
  * @brief  Get GP32C4T1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp32c4t1 enable
  *         @arg 0x0 : gp32c4t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp32c4t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB1SL, RCU_APB1SL_GP32C4T1EN) == (RCU_APB1SL_GP32C4T1EN));
}
/**
  * @}
  */

/** @defgroup APB2SL
  * @{
  */
/**
  * @brief  Set RCU_APB2SL Register
  * @param  rcu RCU Instance
  * @param  apb2sl
  */
__STATIC_INLINE void md_rcu_set_apb2sl(RCU_TypeDef *rcu, uint32_t apb2sl)
{
    WRITE_REG(rcu->APB2SL, apb2sl);
}

/**
  * @brief  Get RCU_APB2SL Register
  * @param  rcu RCU Instance
  * @param  apb2sl
  */
__STATIC_INLINE uint32_t md_rcu_get_apb2sl(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->APB2SL));
}

/**
  * @brief  VRES clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_vres_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_VRESEN);
}

/**
  * @brief  VRES clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_vres_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_VRESEN);
}

/**
  * @brief  Get VRES clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : cmp enable
  *         @arg 0x0 : cmp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_vres_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_VRESEN) == (RCU_APB2SL_VRESEN));
}

/**
  * @brief  CMP clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_cmp_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_CMPEN);
}

/**
  * @brief  CMP clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_cmp_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_CMPEN);
}

/**
  * @brief  Get CMP clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : cmp enable
  *         @arg 0x0 : cmp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_cmp_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_CMPEN) == (RCU_APB2SL_CMPEN));
}

/**
  * @brief  GP16C2T2 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN);
}

/**
  * @brief  GP16C2T2 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN);
}

/**
  * @brief  Get GP16C2T2 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t2 enable
  *         @arg 0x0 : gp16c2t2 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T2EN) == (RCU_APB2SL_GP16C2T2EN));
}

/**
  * @brief  GP16C2T1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN);
}

/**
  * @brief  GP16C2T1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN);
}

/**
  * @brief  Get GP16C2T1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : gp16c2t1 enable
  *         @arg 0x0 : gp16c2t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_gp16c2t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_GP16C2T1EN) == (RCU_APB2SL_GP16C2T1EN));
}

/**
  * @brief  UART1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN);
}

/**
  * @brief  UART1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN);
}

/**
  * @brief  Get UART1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : uart1 enable
  *         @arg 0x0 : uart1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_uart1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_UART1EN) == (RCU_APB2SL_UART1EN));
}

/**
  * @brief  AD16C6T2 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ad16c6t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_AD16C6T2EN);
}

/**
  * @brief  AD16C6T2 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ad16c6t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_AD16C6T2EN);
}

/**
  * @brief  Get AD16C6T2 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : ad16c6t1 enable
  *         @arg 0x0 : ad16c6t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ad16c6t2_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_AD16C6T2EN) == (RCU_APB2SL_AD16C6T2EN));
}

/**
  * @brief  SPI1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN);
}

/**
  * @brief  SPI1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN);
}

/**
  * @brief  Get SPI1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : spi1 enable
  *         @arg 0x0 : spi1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_spi1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_SPI1EN) == (RCU_APB2SL_SPI1EN));
}

/**
  * @brief  AD16C6T1 clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_ad16c6t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_AD16C6T1EN);
}

/**
  * @brief  AD16C6T1 clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_ad16c6t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_AD16C6T1EN);
}

/**
  * @brief  Get AD16C6T1 clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : ad16c6t1 enable
  *         @arg 0x0 : ad16c6t1 disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_ad16c6t1_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_AD16C6T1EN) == (RCU_APB2SL_AD16C6T1EN));
}

/**
  * @brief  OPAMP clock enable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_opamp_in_sleep_mode(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->APB2SL, RCU_APB2SL_OPAMPEN);
}

/**
  * @brief  OPAMP clock disable during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_opamp_in_sleep_mode(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->APB2SL, RCU_APB2SL_OPAMPEN);
}

/**
  * @brief  Get OPAMP clock during Sleep mode bit.
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : opamp enable
  *         @arg 0x0 : opamp disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_opamp_in_sleep_mode(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->APB2SL, RCU_APB2SL_OPAMPEN) == (RCU_APB2SL_OPAMPEN));
}

/**
  * @}
  */

/** @defgroup LCON
  * @{
  */
/**
  * @brief  Set RCU_LCON Register
  * @param  rcu RCU Instance
  * @param  lcon
  */
__STATIC_INLINE void md_rcu_set_lcon(RCU_TypeDef *rcu, uint32_t lcon)
{
    WRITE_REG(rcu->LCON, lcon);
}

/**
  * @brief  Get RCU_LCON Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_lcon(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_REG(rcu->LCON));
}

/**
  * @brief  Get LRCRDY clock ready flag
  * @note
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_LRCRDY_NOT_READY
  *         @arg @ref MD_RCU_LRCRDY_READY
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_lrc_ready(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_LRCRDY) == (RCU_LCON_LRCRDY));
}

/**
  * @brief  Set LRC enbale bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_enable_lrc(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->LCON, RCU_LCON_LRCON);
}

/**
  * @brief  Set LRC disable bit
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_disable_lrc(RCU_TypeDef *rcu)
{
    CLEAR_BIT(rcu->LCON, RCU_LCON_LRCON);
}

/**
  * @brief  Get LRC clock bit
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg 0x1 : lrc enable
  *         @arg 0x0 : lrc disable
  */
__STATIC_INLINE uint32_t md_rcu_is_enabled_lrc(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->LCON, RCU_LCON_LRCON) == (RCU_LCON_LRCON));
}
/**
  * @}
  */

/** @defgroup RSTF
  * @{
  */
/**
  * @brief  Set RCU_RSTF Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE void md_rcu_set_rstf(RCU_TypeDef *rcu, uint32_t rstf)
{
    WRITE_REG(rcu->RSTF, rstf);
}

/**
  * @brief  Get RCU_RSTF Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_rstf(RCU_TypeDef *rcu)
{
    return (uint32_t)READ_REG(rcu->RSTF);
}

/**
  * @brief  Get Low power reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_wakeup(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_LPRSTF) == (RCU_RSTF_LPRSTF));
}

/**
  * @brief  Get WWDT reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_wwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_WWDTRSTF) == (RCU_RSTF_WWDTRSTF));
}

/**
  * @brief  Get IWDT reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_iwdt(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_IWDTRSTF) == (RCU_RSTF_IWDTRSTF));
}

/**
  * @brief  Get SW reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_software(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_SWRSTF) == (RCU_RSTF_SWRSTF));
}

/**
  * @brief  Get Option Byte reload reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_option_byte_reload(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_OBLRSTF) == (RCU_RSTF_OBLRSTF));
}

/**
  * @brief  Get NRST external pin reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_nrst(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_NRSTF) == (RCU_RSTF_NRSTF));
}

/**
  * @brief  Get BOR reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_bor(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_BORRSTF) == (RCU_RSTF_BORRSTF));
}

/**
  * @brief  Get POR/PDR reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_rcu_is_active_flag_por(RCU_TypeDef *rcu)
{
    return (READ_BIT(rcu->RSTF, RCU_RSTF_PORRSTF) == (RCU_RSTF_PORRSTF));
}

/**
  * @brief  Clear reset flag
  * @param  rcu RCU Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE void md_rcu_clear_flag_reset_flag(RCU_TypeDef *rcu)
{
    SET_BIT(rcu->RSTF, RCU_RSTF_CLRFLG);
}
/**
  * @}
  */

/** @defgroup HTRIM
  * @{
  */
/**
  * @brief  Set RCU_HTRIM Register
  * @param  rcu RCU Instance
  * @param  hrctrim
  */
__STATIC_INLINE void md_rcu_set_htrim(RCU_TypeDef *rcu, uint32_t hrctrim)
{
    WRITE_REG(rcu->HTRIM, hrctrim);
}

/**
  * @brief  Get RCU_HTRIM Register
  * @param  rcu RCU Instance
  */
__STATIC_INLINE uint32_t md_rcu_get_htrim(RCU_TypeDef *rcu)
{
    return (uint32_t)READ_REG(rcu->HTRIM);
}

/**
  * @brief  Get HRC calibration value
  * @param  rcu RCU Instance
  * @retval hrc trim value
  */
__STATIC_INLINE uint32_t md_rcu_get_hrctrim_value_real(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->HTRIM, RCU_HTRIM_HRCTRIM_ST) >> RCU_HTRIM_HRCTRIM_ST_POSS);
}

/**
  * @brief  Set HRC calibration value selection
  * @param  rcu RCU Instance
  * @param  hrcsel
  *         @arg @ref MD_RCU_HRCSEL_OPTIONBYTE
  *         @arg @ref MD_RCU_HRCSEL_HRCTRIM
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hrctrim_source(RCU_TypeDef *rcu, md_rcu_hrc_calibration_value_selection_t hrcsel)
{
    MODIFY_REG(rcu->HTRIM, RCU_HTRIM_HRCSEL, hrcsel << RCU_HTRIM_HRCSEL_POS);
}

/**
  * @brief  Get HRC calibration value selection
  * @param  rcu RCU Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_RCU_HRCSEL_OPTIONBYTE
  *         @arg @ref MD_RCU_HRCSEL_HRCTRIM
  */
__STATIC_INLINE uint32_t md_rcu_get_hrctrim_source(RCU_TypeDef *rcu)
{
    return (uint32_t)(READ_BIT(rcu->HTRIM, RCU_HTRIM_HRCSEL) >> RCU_HTRIM_HRCSEL_POS);
}

/**
  * @brief  Set HRC calibration value
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE void md_rcu_set_hrctrim(RCU_TypeDef *rcu, uint32_t hrctrim)
{
  MODIFY_REG(rcu->HTRIM, RCU_HTRIM_HRCTRIM, (hrctrim << RCU_HTRIM_HRCTRIM_POSS));
}

/**
  * @brief  Get HRC calibration value
  * @param  rcu RCU Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_rcu_get_hrctrim(RCU_TypeDef *rcu)
{
  return (uint32_t)(READ_BIT(rcu->HTRIM, RCU_HTRIM_HRCTRIM) >> RCU_HTRIM_HRCTRIM_POSS);
}

/**
  * @}
  */

/** @defgroup RCU_Function
  * @{
  */
void md_rcu_pll_init(RCU_TypeDef *, md_rcu_init_typedef *);
void md_rcu_sys_init(RCU_TypeDef *, md_rcu_init_typedef *);

extern md_status_t md_rcu_clock_config(md_rcu_clock_t clk, uint32_t clock);
extern void md_rcu_pll_config(md_rcu_pll_source_t input, md_rcu_pll_output_t output);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */
  
#ifdef __cplusplus
}
#endif  /* __cplusplus */
#endif  /* __MD_RCU_H__ */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
