/**********************************************************************************
 *
 * @file    md_opamp.h
 * @brief   header file of md_opamp.c
 *
 * @date    18 Apr 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          18 Apr 2023     Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_OPAMP_H__
#define __MD_OPAMP_H__

#ifdef __cplusplus
extern "C" {
#endif

/* Includes -------------------------------------------------------------------*/
#include "es32m0502.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_OPAMP OPAMP
  * @brief OPAMP micro driver
  * @{
  */

/** @defgroup MD_OPAMP_Public_Macros OPAMP Public Macros
  * @{
  */

/**
  * @brief Operational Amplifiers x lock
  */  
typedef enum
{
    MD_OPA_CSR_LOCK_READ_WRITE = (0x0U),
    MD_OPA_CSR_LOCK_ONLY_READ  = (0x1U),
}md_opa_csr_lock_t;

/**
  * @brief Operational output level
  */  
typedef enum
{
    MD_OPA_OUTPUT_LOW_LEVEL  = (0x0U),
    MD_OPA_OUTPUT_HIGH_LEVEL = (0x1U),
}md_opa_output_level_t;

/**
  * @brief Operational Amplifiers x internal output switch
  */ 
typedef enum
{  
    MD_OPA_INTO_IO  = (0x0U),
    MD_OPA_INTO_ADC = (0x1U),
}md_opa_internal_output_t;

/**
  * @brief Operational Amplifiers x speed mode
  */ 
typedef enum
{
    MD_OPA_NORMAL_MODE    = (0x0U),
    MD_OPA_HIGHSPEED_MODE = (0x1U),
}md_opa_speed_mode_t;

/**
  * @brief Operational Amplifiers x VN selection
  */ 
typedef enum
{
    MD_OPA_VN_VINN0      =(0x0U),
    MD_OPA_VN_VINN1      =(0x1U),
    MD_OPA_VN_RESISTANCE =(0x2U),
    MD_OPA_VN_OUTPUT     =(0x3U),
}md_opa_vn_selection_t;

/**
  * @brief Operational Amplifiers x trim selection
  */ 
typedef enum
{
    MD_OPA_TRIM_DEFAULT =(0x0U),
    MD_OPA_TRIM_USER    =(0x1U),
}md_opa_trim_selection_t;

/**
  * @brief Operational Amplifiers x VP selection
  */ 
typedef enum
{
    MD_OPA_VP_VINP0 = (0x0U),
    MD_OPA_VP_VINP1 = (0x1U),
    MD_OPA_VP_VINP2 = (0x2U),
    MD_OPA_VP_VINP3 = (0x3U),
}md_opa_vp_selection_t;

/**
  * @brief Operational Amplifiers x tcmr lock
  */  
typedef enum
{
    MD_OPA_TCMR_LOCK_READ_WRITE =(0x0U),
    MD_OPA_TCMR_LOCK_ONLY_READ  =(0x1U),
}md_opa_tcmr_lock_t;

/**
  * @brief Operational Amplifiers x VPS selection
  */  
typedef enum
{
    MD_OPA_VPS_VINP0 = (0x0U),
    MD_OPA_VPS_VINP1 = (0x1U),
    MD_OPA_VPS_VINP2 = (0x2U),
    MD_OPA_VPS_VINP3 = (0x3U),
}md_opa_vps_selection_t;

/**
  * @brief Operational Amplifiers x VNS selection
  */
typedef enum
{  
    MD_OPA_VNS_VINN0      = (0x0U),
    MD_OPA_VNS_VINN1      = (0x1U),
    MD_OPA_VNS_RESISTANCE = (0x0U),
    MD_OPA_VNS_OUTPUT     = (0x1U),
}md_opa_vns_selection_t;

/**
  * @}
  */

/** @defgroup MD_OPAMP_Public_Functions OPAMP Public Functions
  * @{
  */

/** @defgroup OPA1_CSR
  * @{
  */

/**
  * @brief  Set OPA1_CSR Register
  * @param  opamp OPAMP Instance
  * @param  opa1_csr
  */
__STATIC_INLINE void md_opa1_set_csr(OPAMP_TypeDef *opamp, uint32_t opa1_csr)
{
    WRITE_REG(opamp->OPA1_CSR, opa1_csr);
}

/**
  * @brief  Get OPA1_CSR Register
  * @param  opamp OPAMP Instance
  * @retval opa1_csr Values
  */
__STATIC_INLINE uint32_t md_opa1_get_csr(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA1_CSR);
}

/**
  * @brief  OPA1 CSR locked W/R locked W/R
  * @param  opamp OPAMP Instance
  * @retval None
  */
__STATIC_INLINE void md_opa1_enable_csr_lock(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_LOCK);
}

/**
  * @brief  Get OPA1 CSR locked W/R
  * @param  opamp OPAMP Instance
  * @retval The retval can be a value of the value:
  *         @arg @ref MD_OPA_CSR_LOCK_READ_WRITE
  *         @arg @ref MD_OPA_CSR_LOCK_ONLY_READ 
  */
__STATIC_INLINE uint32_t md_opa1_is_enabled_csr_lock(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_LOCK) == (OPAMP_OPA1_CSR_LOCK));
}

/**
  * @brief  Get Operational Amplifiers 1 output
  * @param  opamp OPAMP Instance
  * @retval output
  */
__STATIC_INLINE uint32_t md_opa1_get_output(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_CAL_OUT) >> OPAMP_OPA1_CSR_CAL_OUT_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 NMOS trim value
  * @param  opamp OPAMP Instance
  * @param  value
  */
__STATIC_INLINE void md_opa1_set_nmos_trim(OPAMP_TypeDef *opamp, uint32_t value)
{
    MODIFY_REG(opamp->OPA1_CSR, OPAMP_OPA1_CSR_TRIMOFFSETN, value << OPAMP_OPA1_CSR_TRIMOFFSETN_POSS);
}

/**
  * @brief  Get Set Operational Amplifiers 1 NMOS trim value
  * @param  opamp OPAMP Instance
  * @retval trim value
  */
__STATIC_INLINE uint32_t md_opa1_get_nmos_trim(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_TRIMOFFSETN) >> OPAMP_OPA1_CSR_TRIMOFFSETN_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 PMOS trim value
  * @param  opamp OPAMP Instance
  * @param  value
  */
__STATIC_INLINE void md_opa1_set_pmos_trim(OPAMP_TypeDef *opamp, uint32_t value)
{
    MODIFY_REG(opamp->OPA1_CSR, OPAMP_OPA1_CSR_TRIMOFFSETP, value << OPAMP_OPA1_CSR_TRIMOFFSETP_POSS);
}

/**
  * @brief  Get Set Operational Amplifiers 1 PMOS trim value
  * @param  opamp OPAMP Instance
  * @retval trim value
  */
__STATIC_INLINE uint32_t md_opa1_get_pmos_trim(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_TRIMOFFSETP) >> OPAMP_OPA1_CSR_TRIMOFFSETP_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 gain
  * @param  opamp OPAMP Instance
  * @param  gain 
  */
__STATIC_INLINE void md_opa1_set_pga_gain(OPAMP_TypeDef *opamp, uint32_t gain)
{
    MODIFY_REG(opamp->OPA1_CSR, OPAMP_OPA1_CSR_PGA_GAIN, gain << OPAMP_OPA1_CSR_PGA_GAIN_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 gain
  * @param  opamp OPAMP Instance
  * @retval gain 
  */
__STATIC_INLINE uint32_t md_opa1_get_pga_gain(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_PGA_GAIN) >> OPAMP_OPA1_CSR_PGA_GAIN_POSS);
}

/**
  * @brief  Operational Amplifiers 1 Calibration Switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_enable_calibration_switch(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_CALSWON);
}

/**
  * @brief  Operational Amplifiers 1 Calibration Switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_disable_calibration_switch(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_CALSWON);
}

/**
  * @brief  Check if Operational Amplifiers 1 Calibration Switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa1_is_enabled_calibration_switch(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_CALSWON) == (OPAMP_OPA1_CSR_CALSWON_POS));
}

/**
  * @brief  Operational Amplifiers 1 Calibration Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_enable_calibration(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_CALON);
}

/**
  * @brief  Operational Amplifiers 1 CalibrationDisable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_disable_calibration(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_CALON);
}

/**
  * @brief  Check if Operational Amplifiers 1 Calibrationis Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa1_is_enabled_calibration(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_CALON) == (OPAMP_OPA1_CSR_CALON));
}

/**
  * @brief  Set Operational Amplifiers 1 internal output switch
  * @param  opamp OPAMP Instance
  * @param  opainto This parameter can be one of the following values:
  *         @arg @ref MD_OPA_INTO_IO 
  *         @arg @ref MD_OPA_INTO_ADC
  */
__STATIC_INLINE void md_opa1_set_into(OPAMP_TypeDef *opamp, md_opa_internal_output_t opainto)
{
    MODIFY_REG(opamp->OPA1_CSR, OPAMP_OPA1_CSR_OPAINTOEN, opainto << OPAMP_OPA1_CSR_OPAINTOEN_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 internal output switch
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_INTO_IO 
  *         @arg @ref MD_OPA_INTO_ADC
  */
__STATIC_INLINE uint32_t md_opa1_get_into(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_OPAINTOEN) >> OPAMP_OPA1_CSR_OPAINTOEN_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 speed mode
  * @param  opamp OPAMP Instance
  * @param  opainto This parameter can be one of the following values:
  *         @arg @ref MD_OPA_NORMAL_MODE   
  *         @arg @ref MD_OPA_HIGHSPEED_MODE
  */
__STATIC_INLINE void md_opa1_set_speed_mode(OPAMP_TypeDef *opamp, md_opa_speed_mode_t opahsm)
{
    MODIFY_REG(opamp->OPA1_CSR, OPAMP_OPA1_CSR_OPAHSM, opahsm << OPAMP_OPA1_CSR_OPAHSM_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 speed mode
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_NORMAL_MODE   
  *         @arg @ref MD_OPA_HIGHSPEED_MODE
  */
__STATIC_INLINE uint32_t md_opa1_get_speed_mode(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_OPAHSM) >> OPAMP_OPA1_CSR_OPAHSM_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 Negative Input Selection
  * @param  opamp OPAMP Instance
  * @param  vn_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VN_VINN0     
  *         @arg @ref MD_OPA_VN_VINN1     
  *         @arg @ref MD_OPA_VN_RESISTANCE
  *         @arg @ref MD_OPA_VN_OUTPUT    
  */
__STATIC_INLINE void md_opa1_set_vn_sel(OPAMP_TypeDef *opamp, md_opa_vn_selection_t vn_sel)
{
    MODIFY_REG(opamp->OPA1_CSR, OPAMP_OPA1_CSR_VN_SEL, vn_sel << OPAMP_OPA1_CSR_VN_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 Negative Input Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VN_VINN0     
  *         @arg @ref MD_OPA_VN_VINN1     
  *         @arg @ref MD_OPA_VN_RESISTANCE
  *         @arg @ref MD_OPA_VN_OUTPUT  
  */
__STATIC_INLINE uint32_t md_opa1_get_vn_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_VN_SEL) >> OPAMP_OPA1_CSR_VN_SEL_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 calibration valule Selection
  * @param  opamp OPAMP Instance
  * @param  value_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_TRIM_DEFAULT
  *         @arg @ref MD_OPA_TRIM_USER   
  */
__STATIC_INLINE void md_opa1_set_trim_value_source(OPAMP_TypeDef *opamp, md_opa_trim_selection_t value_sel)
{
    MODIFY_REG(opamp->OPA1_CSR, OPAMP_OPA1_CSR_USERTRIM, value_sel << OPAMP_OPA1_CSR_USERTRIM_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 calibration valule Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_TRIM_DEFAULT
  *         @arg @ref MD_OPA_TRIM_USER 
  */
__STATIC_INLINE uint32_t md_opa1_get_trim_value_source(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_USERTRIM) >> OPAMP_OPA1_CSR_USERTRIM_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 Positive Input Selection
  * @param  opamp OPAMP Instance
  * @param  vp_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VP_VINP0
  *         @arg @ref MD_OPA_VP_VINP1
  *         @arg @ref MD_OPA_VP_VINP2
  *         @arg @ref MD_OPA_VP_VINP3
  */
__STATIC_INLINE void md_opa1_set_vp_sel(OPAMP_TypeDef *opamp, md_opa_vp_selection_t vp_sel)
{
    MODIFY_REG(opamp->OPA1_CSR, OPAMP_OPA1_CSR_VP_SEL, vp_sel << OPAMP_OPA1_CSR_VP_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 Positive Input Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VP_VINP0
  *         @arg @ref MD_OPA_VP_VINP1
  *         @arg @ref MD_OPA_VP_VINP2
  *         @arg @ref MD_OPA_VP_VINP3
  */
__STATIC_INLINE uint32_t md_opa1_get_vp_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_VP_SEL) >> OPAMP_OPA1_CSR_VP_SEL_POSS);
}

/**
  * @brief  Operational Amplifiers 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_enable(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_OPAEN);
}

/**
  * @brief  Operational Amplifiers 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_disable(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_OPAEN);
}

/**
  * @brief  Check if Operational Amplifiers 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa1_is_enabled(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA1_CSR, OPAMP_OPA1_CSR_OPAEN) == (OPAMP_OPA1_CSR_OPAEN));
}

/**
  * @}
  */

/** @defgroup OPA2_CSR
  * @{
  */

/**
  * @brief  Set OPA2_CSR Register
  * @param  opamp OPAMP Instance
  * @param  opa2_csr
  */
__STATIC_INLINE void md_opa2_set_csr(OPAMP_TypeDef *opamp, uint32_t opa2_csr)
{
    WRITE_REG(opamp->OPA2_CSR, opa2_csr);
}

/**
  * @brief  Get OPA2_CSR Register
  * @param  opamp OPAMP Instance
  * @retval opa2_csr Values
  */
__STATIC_INLINE uint32_t md_opa2_get_csr(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA2_CSR);
}

/**
  * @brief  OPA2_CSR locked W/R locked W/R
  * @param  opamp OPAMP Instance
  * @retval None
  */
__STATIC_INLINE void md_opa2_enable_csr_lock(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_LOCK);
}

/**
  * @brief  Get OPA2_CSR locked W/R
  * @param  opamp OPAMP Instance
  * @retval The retval can be a value of the value:
  *         @arg @ref MD_OPA_CSR_LOCK_READ_WRITE
  *         @arg @ref MD_OPA_CSR_LOCK_ONLY_READ 
  */
__STATIC_INLINE uint32_t md_opa2_is_enabled_csr_lock(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_LOCK) == (OPAMP_OPA2_CSR_LOCK));
}

/**
  * @brief  Get Operational Amplifiers 1 output
  * @param  opamp OPAMP Instance
  * @retval output
  */
__STATIC_INLINE uint32_t md_opa2_get_output(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_CAL_OUT) >> OPAMP_OPA2_CSR_CAL_OUT_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 NMOS trim value
  * @param  opamp OPAMP Instance
  * @param  value
  */
__STATIC_INLINE void md_opa2_set_nmos_trim(OPAMP_TypeDef *opamp, uint32_t value)
{
    MODIFY_REG(opamp->OPA2_CSR, OPAMP_OPA2_CSR_TRIMOFFSETN, value << OPAMP_OPA2_CSR_TRIMOFFSETN_POSS);
}

/**
  * @brief  Get Set Operational Amplifiers 1 NMOS trim value
  * @param  opamp OPAMP Instance
  * @retval trim value
  */
__STATIC_INLINE uint32_t md_opa2_get_nmos_trim(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_TRIMOFFSETN) >> OPAMP_OPA2_CSR_TRIMOFFSETN_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 PMOS trim value
  * @param  opamp OPAMP Instance
  * @param  value
  */
__STATIC_INLINE void md_opa2_set_pmos_trim(OPAMP_TypeDef *opamp, uint32_t value)
{
    MODIFY_REG(opamp->OPA2_CSR, OPAMP_OPA2_CSR_TRIMOFFSETP, value << OPAMP_OPA2_CSR_TRIMOFFSETP_POSS);
}

/**
  * @brief  Get Set Operational Amplifiers 1 PMOS trim value
  * @param  opamp OPAMP Instance
  * @retval trim value
  */
__STATIC_INLINE uint32_t md_opa2_get_pmos_trim(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_TRIMOFFSETP) >> OPAMP_OPA2_CSR_TRIMOFFSETP_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 gain
  * @param  opamp OPAMP Instance
  * @param  gain 
  */
__STATIC_INLINE void md_opa2_set_pga_gain(OPAMP_TypeDef *opamp, uint32_t gain)
{
    MODIFY_REG(opamp->OPA2_CSR, OPAMP_OPA2_CSR_PGA_GAIN, gain << OPAMP_OPA2_CSR_PGA_GAIN_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 gain
  * @param  opamp OPAMP Instance
  * @retval gain 
  */
__STATIC_INLINE uint32_t md_opa2_get_pga_gain(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_PGA_GAIN) >> OPAMP_OPA2_CSR_PGA_GAIN_POSS);
}

/**
  * @brief  Operational Amplifiers 1 Calibration Switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_enable_calibration_switch(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_CALSWON);
}

/**
  * @brief  Operational Amplifiers 1 Calibration Switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_disable_calibration_switch(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_CALSWON);
}

/**
  * @brief  Check if Operational Amplifiers 1 Calibration Switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa2_is_enabled_calibration_switch(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_CALSWON) == (OPAMP_OPA2_CSR_CALSWON_POS));
}

/**
  * @brief  Operational Amplifiers 1 Calibration Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_enable_calibration(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_CALON);
}

/**
  * @brief  Operational Amplifiers 1 CalibrationDisable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_disable_calibration(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_CALON);
}

/**
  * @brief  Check if Operational Amplifiers 1 Calibrationis Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa2_is_enabled_calibration(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_CALON) == (OPAMP_OPA2_CSR_CALON));
}

/**
  * @brief  Set Operational Amplifiers 1 internal output switch
  * @param  opamp OPAMP Instance
  * @param  opainto This parameter can be one of the following values:
  *         @arg @ref MD_OPA_INTO_IO 
  *         @arg @ref MD_OPA_INTO_ADC
  */
__STATIC_INLINE void md_opa2_set_into(OPAMP_TypeDef *opamp, md_opa_internal_output_t opainto)
{
    MODIFY_REG(opamp->OPA2_CSR, OPAMP_OPA2_CSR_OPAINTOEN, opainto << OPAMP_OPA2_CSR_OPAINTOEN_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 internal output switch
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_INTO_IO 
  *         @arg @ref MD_OPA_INTO_ADC
  */
__STATIC_INLINE uint32_t md_opa2_get_into(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_OPAINTOEN) >> OPAMP_OPA2_CSR_OPAINTOEN_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 speed mode
  * @param  opamp OPAMP Instance
  * @param  opainto This parameter can be one of the following values:
  *         @arg @ref MD_OPA_NORMAL_MODE   
  *         @arg @ref MD_OPA_HIGHSPEED_MODE
  */
__STATIC_INLINE void md_opa2_set_speed_mode(OPAMP_TypeDef *opamp, md_opa_speed_mode_t opahsm)
{
    MODIFY_REG(opamp->OPA2_CSR, OPAMP_OPA2_CSR_OPAHSM, opahsm << OPAMP_OPA2_CSR_OPAHSM_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 speed mode
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_NORMAL_MODE   
  *         @arg @ref MD_OPA_HIGHSPEED_MODE
  */
__STATIC_INLINE uint32_t md_opa2_get_speed_mode(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_OPAHSM) >> OPAMP_OPA2_CSR_OPAHSM_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 Negative Input Selection
  * @param  opamp OPAMP Instance
  * @param  vn_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VN_VINN0     
  *         @arg @ref MD_OPA_VN_VINN1     
  *         @arg @ref MD_OPA_VN_RESISTANCE
  *         @arg @ref MD_OPA_VN_OUTPUT    
  */
__STATIC_INLINE void md_opa2_set_vn_sel(OPAMP_TypeDef *opamp, md_opa_vn_selection_t vn_sel)
{
    MODIFY_REG(opamp->OPA2_CSR, OPAMP_OPA2_CSR_VN_SEL, vn_sel << OPAMP_OPA2_CSR_VN_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 Negative Input Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VN_VINN0     
  *         @arg @ref MD_OPA_VN_VINN1     
  *         @arg @ref MD_OPA_VN_RESISTANCE
  *         @arg @ref MD_OPA_VN_OUTPUT  
  */
__STATIC_INLINE uint32_t md_opa2_get_vn_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_VN_SEL) >> OPAMP_OPA2_CSR_VN_SEL_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 calibration valule Selection
  * @param  opamp OPAMP Instance
  * @param  value_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_TRIM_DEFAULT
  *         @arg @ref MD_OPA_TRIM_USER   
  */
__STATIC_INLINE void md_opa2_set_trim_value_source(OPAMP_TypeDef *opamp, md_opa_trim_selection_t value_sel)
{
    MODIFY_REG(opamp->OPA2_CSR, OPAMP_OPA2_CSR_USERTRIM, value_sel << OPAMP_OPA2_CSR_USERTRIM_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 calibration valule Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_TRIM_DEFAULT
  *         @arg @ref MD_OPA_TRIM_USER 
  */
__STATIC_INLINE uint32_t md_opa2_get_trim_value_source(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_USERTRIM) >> OPAMP_OPA2_CSR_USERTRIM_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 Positive Input Selection
  * @param  opamp OPAMP Instance
  * @param  vp_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VP_VINP0
  *         @arg @ref MD_OPA_VP_VINP1
  *         @arg @ref MD_OPA_VP_VINP2
  *         @arg @ref MD_OPA_VP_VINP3
  */
__STATIC_INLINE void md_opa2_set_vp_sel(OPAMP_TypeDef *opamp, md_opa_vp_selection_t vp_sel)
{
    MODIFY_REG(opamp->OPA2_CSR, OPAMP_OPA2_CSR_VP_SEL, vp_sel << OPAMP_OPA2_CSR_VP_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 Positive Input Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VP_VINP0
  *         @arg @ref MD_OPA_VP_VINP1
  *         @arg @ref MD_OPA_VP_VINP2
  *         @arg @ref MD_OPA_VP_VINP3
  */
__STATIC_INLINE uint32_t md_opa2_get_vp_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_VP_SEL) >> OPAMP_OPA2_CSR_VP_SEL_POSS);
}

/**
  * @brief  Operational Amplifiers 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_enable(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_OPAEN);
}

/**
  * @brief  Operational Amplifiers 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_disable(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_OPAEN);
}

/**
  * @brief  Check if Operational Amplifiers 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa2_is_enabled(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA2_CSR, OPAMP_OPA2_CSR_OPAEN) == (OPAMP_OPA2_CSR_OPAEN));
}

/**
  * @}
  */

/** @defgroup OPA3_CSR
  * @{
  */

/**
  * @brief  Set OPA3_CSR Register
  * @param  opamp OPAMP Instance
  * @param  opa3_csr
  */
__STATIC_INLINE void md_opa3_set_csr(OPAMP_TypeDef *opamp, uint32_t opa3_csr)
{
    WRITE_REG(opamp->OPA3_CSR, opa3_csr);
}

/**
  * @brief  Get OPA3_CSR Register
  * @param  opamp OPAMP Instance
  * @retval opa3_csr Values
  */
__STATIC_INLINE uint32_t md_opa3_get_csr(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA3_CSR);
}

/**
  * @brief  OPA3_CSR locked W/R locked W/R
  * @param  opamp OPAMP Instance
  * @retval None
  */
__STATIC_INLINE void md_opa3_enable_csr_lock(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_LOCK);
}

/**
  * @brief  Get OPA3_CSR locked W/R
  * @param  opamp OPAMP Instance
  * @retval The retval can be a value of the value:
  *         @arg @ref MD_OPA_CSR_LOCK_READ_WRITE
  *         @arg @ref MD_OPA_CSR_LOCK_ONLY_READ 
  */
__STATIC_INLINE uint32_t md_opa3_is_enabled_csr_lock(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_LOCK) == (OPAMP_OPA3_CSR_LOCK));
}

/**
  * @brief  Get Operational Amplifiers 1 output
  * @param  opamp OPAMP Instance
  * @retval output
  */
__STATIC_INLINE uint32_t md_opa3_get_output(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_CAL_OUT) >> OPAMP_OPA3_CSR_CAL_OUT_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 NMOS trim value
  * @param  opamp OPAMP Instance
  * @param  value
  */
__STATIC_INLINE void md_opa3_set_nmos_trim(OPAMP_TypeDef *opamp, uint32_t value)
{
    MODIFY_REG(opamp->OPA3_CSR, OPAMP_OPA3_CSR_TRIMOFFSETN, value << OPAMP_OPA3_CSR_TRIMOFFSETN_POSS);
}

/**
  * @brief  Get Set Operational Amplifiers 1 NMOS trim value
  * @param  opamp OPAMP Instance
  * @retval trim value
  */
__STATIC_INLINE uint32_t md_opa3_get_nmos_trim(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_TRIMOFFSETN) >> OPAMP_OPA3_CSR_TRIMOFFSETN_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 PMOS trim value
  * @param  opamp OPAMP Instance
  * @param  value
  */
__STATIC_INLINE void md_opa3_set_pmos_trim(OPAMP_TypeDef *opamp, uint32_t value)
{
    MODIFY_REG(opamp->OPA3_CSR, OPAMP_OPA3_CSR_TRIMOFFSETP, value << OPAMP_OPA3_CSR_TRIMOFFSETP_POSS);
}

/**
  * @brief  Get Set Operational Amplifiers 1 PMOS trim value
  * @param  opamp OPAMP Instance
  * @retval trim value
  */
__STATIC_INLINE uint32_t md_opa3_get_pmos_trim(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_TRIMOFFSETP) >> OPAMP_OPA3_CSR_TRIMOFFSETP_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 gain
  * @param  opamp OPAMP Instance
  * @param  gain 
  */
__STATIC_INLINE void md_opa3_set_pga_gain(OPAMP_TypeDef *opamp, uint32_t gain)
{
    MODIFY_REG(opamp->OPA3_CSR, OPAMP_OPA3_CSR_PGA_GAIN, gain << OPAMP_OPA3_CSR_PGA_GAIN_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 gain
  * @param  opamp OPAMP Instance
  * @retval gain 
  */
__STATIC_INLINE uint32_t md_opa3_get_pga_gain(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_PGA_GAIN) >> OPAMP_OPA3_CSR_PGA_GAIN_POSS);
}

/**
  * @brief  Operational Amplifiers 1 Calibration Switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_enable_calibration_switch(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_CALSWON);
}

/**
  * @brief  Operational Amplifiers 1 Calibration Switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_disable_calibration_switch(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_CALSWON);
}

/**
  * @brief  Check if Operational Amplifiers 1 Calibration Switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa3_is_enabled_calibration_switch(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_CALSWON) == (OPAMP_OPA3_CSR_CALSWON_POS));
}

/**
  * @brief  Operational Amplifiers 1 Calibration Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_enable_calibration(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_CALON);
}

/**
  * @brief  Operational Amplifiers 1 CalibrationDisable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_disable_calibration(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_CALON);
}

/**
  * @brief  Check if Operational Amplifiers 1 Calibrationis Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa3_is_enabled_calibration(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_CALON) == (OPAMP_OPA3_CSR_CALON));
}

/**
  * @brief  Set Operational Amplifiers 1 internal output switch
  * @param  opamp OPAMP Instance
  * @param  opainto This parameter can be one of the following values:
  *         @arg @ref MD_OPA_INTO_IO 
  *         @arg @ref MD_OPA_INTO_ADC
  */
__STATIC_INLINE void md_opa3_set_into(OPAMP_TypeDef *opamp, md_opa_internal_output_t opainto)
{
    MODIFY_REG(opamp->OPA3_CSR, OPAMP_OPA3_CSR_OPAINTOEN, opainto << OPAMP_OPA3_CSR_OPAINTOEN_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 internal output switch
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_INTO_IO 
  *         @arg @ref MD_OPA_INTO_ADC
  */
__STATIC_INLINE uint32_t md_opa3_get_into(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_OPAINTOEN) >> OPAMP_OPA3_CSR_OPAINTOEN_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 speed mode
  * @param  opamp OPAMP Instance
  * @param  opainto This parameter can be one of the following values:
  *         @arg @ref MD_OPA_NORMAL_MODE   
  *         @arg @ref MD_OPA_HIGHSPEED_MODE
  */
__STATIC_INLINE void md_opa3_set_speed_mode(OPAMP_TypeDef *opamp, md_opa_speed_mode_t opahsm)
{
    MODIFY_REG(opamp->OPA3_CSR, OPAMP_OPA3_CSR_OPAHSM, opahsm << OPAMP_OPA3_CSR_OPAHSM_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 speed mode
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_NORMAL_MODE   
  *         @arg @ref MD_OPA_HIGHSPEED_MODE
  */
__STATIC_INLINE uint32_t md_opa3_get_speed_mode(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_OPAHSM) >> OPAMP_OPA3_CSR_OPAHSM_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 Negative Input Selection
  * @param  opamp OPAMP Instance
  * @param  vn_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VN_VINN0     
  *         @arg @ref MD_OPA_VN_VINN1     
  *         @arg @ref MD_OPA_VN_RESISTANCE
  *         @arg @ref MD_OPA_VN_OUTPUT    
  */
__STATIC_INLINE void md_opa3_set_vn_sel(OPAMP_TypeDef *opamp, md_opa_vn_selection_t vn_sel)
{
    MODIFY_REG(opamp->OPA3_CSR, OPAMP_OPA3_CSR_VN_SEL, vn_sel << OPAMP_OPA3_CSR_VN_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 Negative Input Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VN_VINN0     
  *         @arg @ref MD_OPA_VN_VINN1     
  *         @arg @ref MD_OPA_VN_RESISTANCE
  *         @arg @ref MD_OPA_VN_OUTPUT  
  */
__STATIC_INLINE uint32_t md_opa3_get_vn_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_VN_SEL) >> OPAMP_OPA3_CSR_VN_SEL_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 calibration valule Selection
  * @param  opamp OPAMP Instance
  * @param  value_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_TRIM_DEFAULT
  *         @arg @ref MD_OPA_TRIM_USER   
  */
__STATIC_INLINE void md_opa3_set_trim_value_source(OPAMP_TypeDef *opamp, md_opa_trim_selection_t value_sel)
{
    MODIFY_REG(opamp->OPA3_CSR, OPAMP_OPA3_CSR_USERTRIM, value_sel << OPAMP_OPA3_CSR_USERTRIM_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 calibration valule Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_TRIM_DEFAULT
  *         @arg @ref MD_OPA_TRIM_USER 
  */
__STATIC_INLINE uint32_t md_opa3_get_trim_value_source(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_USERTRIM) >> OPAMP_OPA3_CSR_USERTRIM_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 Positive Input Selection
  * @param  opamp OPAMP Instance
  * @param  vp_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VP_VINP0
  *         @arg @ref MD_OPA_VP_VINP1
  *         @arg @ref MD_OPA_VP_VINP2
  *         @arg @ref MD_OPA_VP_VINP3
  */
__STATIC_INLINE void md_opa3_set_vp_sel(OPAMP_TypeDef *opamp, md_opa_vp_selection_t vp_sel)
{
    MODIFY_REG(opamp->OPA3_CSR, OPAMP_OPA3_CSR_VP_SEL, vp_sel << OPAMP_OPA3_CSR_VP_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 Positive Input Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VP_VINP0
  *         @arg @ref MD_OPA_VP_VINP1
  *         @arg @ref MD_OPA_VP_VINP2
  *         @arg @ref MD_OPA_VP_VINP3
  */
__STATIC_INLINE uint32_t md_opa3_get_vp_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_VP_SEL) >> OPAMP_OPA3_CSR_VP_SEL_POSS);
}

/**
  * @brief  Operational Amplifiers 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_enable(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_OPAEN);
}

/**
  * @brief  Operational Amplifiers 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_disable(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_OPAEN);
}

/**
  * @brief  Check if Operational Amplifiers 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa3_is_enabled(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA3_CSR, OPAMP_OPA3_CSR_OPAEN) == (OPAMP_OPA3_CSR_OPAEN));
}

/**
  * @}
  */  
  
/** @defgroup OPA4_CSR
  * @{
  */

/**
  * @brief  Set OPA4_CSR Register
  * @param  opamp OPAMP Instance
  * @param  opa4_csr
  */
__STATIC_INLINE void md_opa4_set_csr(OPAMP_TypeDef *opamp, uint32_t opa4_csr)
{
    WRITE_REG(opamp->OPA4_CSR, opa4_csr);
}

/**
  * @brief  Get OPA4_CSR Register
  * @param  opamp OPAMP Instance
  * @retval opa4_csr Values
  */
__STATIC_INLINE uint32_t md_opa4_get_csr(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA4_CSR);
}

/**
  * @brief  OPA4_CSR locked W/R locked W/R
  * @param  opamp OPAMP Instance
  * @retval None
  */
__STATIC_INLINE void md_opa4_enable_csr_lock(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_LOCK);
}

/**
  * @brief  Get OPA4_CSR locked W/R
  * @param  opamp OPAMP Instance
  * @retval The retval can be a value of the value:
  *         @arg @ref MD_OPA_CSR_LOCK_READ_WRITE
  *         @arg @ref MD_OPA_CSR_LOCK_ONLY_READ 
  */
__STATIC_INLINE uint32_t md_opa4_is_enabled_csr_lock(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_LOCK) == (OPAMP_OPA4_CSR_LOCK));
}

/**
  * @brief  Get Operational Amplifiers 1 output
  * @param  opamp OPAMP Instance
  * @retval output
  */
__STATIC_INLINE uint32_t md_opa4_get_output(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_CAL_OUT) >> OPAMP_OPA4_CSR_CAL_OUT_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 NMOS trim value
  * @param  opamp OPAMP Instance
  * @param  value
  */
__STATIC_INLINE void md_opa4_set_nmos_trim(OPAMP_TypeDef *opamp, uint32_t value)
{
    MODIFY_REG(opamp->OPA4_CSR, OPAMP_OPA4_CSR_TRIMOFFSETN, value << OPAMP_OPA4_CSR_TRIMOFFSETN_POSS);
}

/**
  * @brief  Get Set Operational Amplifiers 1 NMOS trim value
  * @param  opamp OPAMP Instance
  * @retval trim value
  */
__STATIC_INLINE uint32_t md_opa4_get_nmos_trim(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_TRIMOFFSETN) >> OPAMP_OPA4_CSR_TRIMOFFSETN_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 PMOS trim value
  * @param  opamp OPAMP Instance
  * @param  value
  */
__STATIC_INLINE void md_opa4_set_pmos_trim(OPAMP_TypeDef *opamp, uint32_t value)
{
    MODIFY_REG(opamp->OPA4_CSR, OPAMP_OPA4_CSR_TRIMOFFSETP, value << OPAMP_OPA4_CSR_TRIMOFFSETP_POSS);
}

/**
  * @brief  Get Set Operational Amplifiers 1 PMOS trim value
  * @param  opamp OPAMP Instance
  * @retval trim value
  */
__STATIC_INLINE uint32_t md_opa4_get_pmos_trim(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_TRIMOFFSETP) >> OPAMP_OPA4_CSR_TRIMOFFSETP_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 gain
  * @param  opamp OPAMP Instance
  * @param  gain 
  */
__STATIC_INLINE void md_opa4_set_pga_gain(OPAMP_TypeDef *opamp, uint32_t gain)
{
    MODIFY_REG(opamp->OPA4_CSR, OPAMP_OPA4_CSR_PGA_GAIN, gain << OPAMP_OPA4_CSR_PGA_GAIN_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 gain
  * @param  opamp OPAMP Instance
  * @retval gain 
  */
__STATIC_INLINE uint32_t md_opa4_get_pga_gain(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_PGA_GAIN) >> OPAMP_OPA4_CSR_PGA_GAIN_POSS);
}

/**
  * @brief  Operational Amplifiers 1 Calibration Switch Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_enable_calibration_switch(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_CALSWON);
}

/**
  * @brief  Operational Amplifiers 1 Calibration Switch Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_disable_calibration_switch(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_CALSWON);
}

/**
  * @brief  Check if Operational Amplifiers 1 Calibration Switch is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa4_is_enabled_calibration_switch(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_CALSWON) == (OPAMP_OPA4_CSR_CALSWON_POS));
}

/**
  * @brief  Operational Amplifiers 1 Calibration Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_enable_calibration(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_CALON);
}

/**
  * @brief  Operational Amplifiers 1 CalibrationDisable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_disable_calibration(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_CALON);
}

/**
  * @brief  Check if Operational Amplifiers 1 Calibrationis Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa4_is_enabled_calibration(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_CALON) == (OPAMP_OPA4_CSR_CALON));
}

/**
  * @brief  Set Operational Amplifiers 1 internal output switch
  * @param  opamp OPAMP Instance
  * @param  opainto This parameter can be one of the following values:
  *         @arg @ref MD_OPA_INTO_IO 
  *         @arg @ref MD_OPA_INTO_ADC
  */
__STATIC_INLINE void md_opa4_set_into(OPAMP_TypeDef *opamp, md_opa_internal_output_t opainto)
{
    MODIFY_REG(opamp->OPA4_CSR, OPAMP_OPA4_CSR_OPAINTOEN, opainto << OPAMP_OPA4_CSR_OPAINTOEN_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 internal output switch
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_INTO_IO 
  *         @arg @ref MD_OPA_INTO_ADC
  */
__STATIC_INLINE uint32_t md_opa4_get_into(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_OPAINTOEN) >> OPAMP_OPA4_CSR_OPAINTOEN_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 speed mode
  * @param  opamp OPAMP Instance
  * @param  opainto This parameter can be one of the following values:
  *         @arg @ref MD_OPA_NORMAL_MODE   
  *         @arg @ref MD_OPA_HIGHSPEED_MODE
  */
__STATIC_INLINE void md_opa4_set_speed_mode(OPAMP_TypeDef *opamp, md_opa_speed_mode_t opahsm)
{
    MODIFY_REG(opamp->OPA4_CSR, OPAMP_OPA4_CSR_OPAHSM, opahsm << OPAMP_OPA4_CSR_OPAHSM_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 speed mode
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_NORMAL_MODE   
  *         @arg @ref MD_OPA_HIGHSPEED_MODE
  */
__STATIC_INLINE uint32_t md_opa4_get_speed_mode(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_OPAHSM) >> OPAMP_OPA4_CSR_OPAHSM_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 Negative Input Selection
  * @param  opamp OPAMP Instance
  * @param  vn_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VN_VINN0     
  *         @arg @ref MD_OPA_VN_VINN1     
  *         @arg @ref MD_OPA_VN_RESISTANCE
  *         @arg @ref MD_OPA_VN_OUTPUT    
  */
__STATIC_INLINE void md_opa4_set_vn_sel(OPAMP_TypeDef *opamp, md_opa_vn_selection_t vn_sel)
{
    MODIFY_REG(opamp->OPA4_CSR, OPAMP_OPA4_CSR_VN_SEL, vn_sel << OPAMP_OPA4_CSR_VN_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 Negative Input Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VN_VINN0     
  *         @arg @ref MD_OPA_VN_VINN1     
  *         @arg @ref MD_OPA_VN_RESISTANCE
  *         @arg @ref MD_OPA_VN_OUTPUT  
  */
__STATIC_INLINE uint32_t md_opa4_get_vn_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_VN_SEL) >> OPAMP_OPA4_CSR_VN_SEL_POSS);
}

/**
  * @brief  Set Operational Amplifiers 1 calibration valule Selection
  * @param  opamp OPAMP Instance
  * @param  value_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_TRIM_DEFAULT
  *         @arg @ref MD_OPA_TRIM_USER   
  */
__STATIC_INLINE void md_opa4_set_trim_value_source(OPAMP_TypeDef *opamp, md_opa_trim_selection_t value_sel)
{
    MODIFY_REG(opamp->OPA4_CSR, OPAMP_OPA4_CSR_USERTRIM, value_sel << OPAMP_OPA4_CSR_USERTRIM_POS);
}

/**
  * @brief  Get Operational Amplifiers 1 calibration valule Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_TRIM_DEFAULT
  *         @arg @ref MD_OPA_TRIM_USER 
  */
__STATIC_INLINE uint32_t md_opa4_get_trim_value_source(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_USERTRIM) >> OPAMP_OPA4_CSR_USERTRIM_POS);
}

/**
  * @brief  Set Operational Amplifiers 1 Positive Input Selection
  * @param  opamp OPAMP Instance
  * @param  vp_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VP_VINP0
  *         @arg @ref MD_OPA_VP_VINP1
  *         @arg @ref MD_OPA_VP_VINP2
  *         @arg @ref MD_OPA_VP_VINP3
  */
__STATIC_INLINE void md_opa4_set_vp_sel(OPAMP_TypeDef *opamp, md_opa_vp_selection_t vp_sel)
{
    MODIFY_REG(opamp->OPA4_CSR, OPAMP_OPA4_CSR_VP_SEL, vp_sel << OPAMP_OPA4_CSR_VP_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers 1 Positive Input Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VP_VINP0
  *         @arg @ref MD_OPA_VP_VINP1
  *         @arg @ref MD_OPA_VP_VINP2
  *         @arg @ref MD_OPA_VP_VINP3
  */
__STATIC_INLINE uint32_t md_opa4_get_vp_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_VP_SEL) >> OPAMP_OPA4_CSR_VP_SEL_POSS);
}

/**
  * @brief  Operational Amplifiers 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_enable(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_OPAEN);
}

/**
  * @brief  Operational Amplifiers 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_disable(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_OPAEN);
}

/**
  * @brief  Check if Operational Amplifiers 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa4_is_enabled(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA4_CSR, OPAMP_OPA4_CSR_OPAEN) == (OPAMP_OPA4_CSR_OPAEN));
}

/**
  * @}
  */   
  
/** @defgroup OPA1_TCMR
  * @{
  */

/**
  * @brief  Set OPA1_TCMR Register
  * @param  opamp OPAMP Instance
  * @param  opa1_tcmr
  */
__STATIC_INLINE void md_opa1_set_tcmr(OPAMP_TypeDef *opamp, uint32_t opa1_tcmr)
{
    WRITE_REG(opamp->OPA1_TCMR, opa1_tcmr);
}

/**
  * @brief  Get OPA1_TCMR Register
  * @param  opamp OPAMP Instance
  * @retval opa1_tcmr Values
  */
__STATIC_INLINE uint32_t md_opa1_get_tcmr(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA1_TCMR);
}

/**
  * @brief  OPA1_TCMR locked W/R locked W/R
  * @param  opamp OPAMP Instance
  * @retval None
  */
__STATIC_INLINE void md_opa1_enable_tcrm_lock(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_LOCK);
}

/**
  * @brief  Get OPA1_TCMR locked W/R
  * @param  opamp OPAMP Instance
  * @retval The retval can be a value of the value:
  *         @arg @ref MD_OPA_CSR_LOCK_READ_WRITE
  *         @arg @ref MD_OPA_CSR_LOCK_ONLY_READ 
  */
__STATIC_INLINE uint32_t md_opa1_is_enabled_tcrm_lock(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_LOCK) == (OPAMP_OPA1_TCMR_LOCK));
}

/**
  * @brief  Advanced Timer 2 Multiplexer Control Switch 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_enable_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_AD2CM_EN);
}

/**
  * @brief  Advanced Timer 2 Multiplexer Control Switch 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_disable_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_AD2CM_EN);
}

/**
  * @brief  Check if Advanced Timer 2 Multiplexer Control Switch 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa1_is_enabled_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_AD2CM_EN) == (OPAMP_OPA1_TCMR_AD2CM_EN));
}

/**
  * @brief  Advanced Timer 1 Multiplexer Control Switch 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_enable_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_AD1CM_EN);
}

/**
  * @brief  Advanced Timer 1 Multiplexer Control Switch 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa1_disable_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_AD1CM_EN);
}

/**
  * @brief  Check if Advanced Timer 1 Multiplexer Control Switch 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa1_is_enabled_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_AD1CM_EN) == (OPAMP_OPA1_TCMR_AD1CM_EN));
}

/**
  * @brief  Set Operational Amplifiers x Positive Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @param  vps_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VPS_VINP0
  *         @arg @ref MD_OPA_VPS_VINP1
  *         @arg @ref MD_OPA_VPS_VINP2
  *         @arg @ref MD_OPA_VPS_VINP3
  */
__STATIC_INLINE void md_opa1_set_vps_sel(OPAMP_TypeDef *opamp, md_opa_vps_selection_t vps_sel)
{
    MODIFY_REG(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_VPS_SEL, vps_sel << OPAMP_OPA1_TCMR_VPS_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers x Positive Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VPS_VINP0
  *         @arg @ref MD_OPA_VPS_VINP1
  *         @arg @ref MD_OPA_VPS_VINP2
  *         @arg @ref MD_OPA_VPS_VINP3
  */
__STATIC_INLINE uint32_t md_opa1_get_vps_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_VPS_SEL) >> OPAMP_OPA1_TCMR_VPS_SEL_POSS);
}

/**
  * @brief  Set Operational Amplifiers x Negative Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @param  vns_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VNS_VINN0     
  *         @arg @ref MD_OPA_VNS_VINN1     
  *         @arg @ref MD_OPA_VNS_RESISTANCE
  *         @arg @ref MD_OPA_VNS_OUTPUT    
  */
__STATIC_INLINE void md_opa1_set_vns_sel(OPAMP_TypeDef *opamp, md_opa_vns_selection_t vns_sel)
{
    MODIFY_REG(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_VNS_SEL, vns_sel << OPAMP_OPA1_TCMR_VNS_SEL_POS);
}

/**
  * @brief  Get Operational Amplifiers x Negative Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VNS_VINN0     
  *         @arg @ref MD_OPA_VNS_VINN1     
  *         @arg @ref MD_OPA_VNS_RESISTANCE
  *         @arg @ref MD_OPA_VNS_OUTPUT    
  */
__STATIC_INLINE uint32_t md_opa1_get_vns_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA1_TCMR, OPAMP_OPA1_TCMR_VNS_SEL) >> OPAMP_OPA1_TCMR_VNS_SEL_POS);
}

/**
  * @}
  */
  
/** @defgroup OPA2_TCMR
  * @{
  */

/**
  * @brief  Set OPA2_TCMR Register
  * @param  opamp OPAMP Instance
  * @param  opa2_tcmr
  */
__STATIC_INLINE void md_opa2_set_tcmr(OPAMP_TypeDef *opamp, uint32_t opa2_tcmr)
{
    WRITE_REG(opamp->OPA2_TCMR, opa2_tcmr);
}

/**
  * @brief  Get OPA2_TCMR Register
  * @param  opamp OPAMP Instance
  * @retval opa2_tcmr Values
  */
__STATIC_INLINE uint32_t md_opa2_get_tcmr(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA2_TCMR);
}

/**
  * @brief  OPA2_TCMR locked W/R locked W/R
  * @param  opamp OPAMP Instance
  * @retval None
  */
__STATIC_INLINE void md_opa2_enable_tcrm_lock(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_LOCK);
}

/**
  * @brief  Get OPA2_TCMR locked W/R
  * @param  opamp OPAMP Instance
  * @retval The retval can be a value of the value:
  *         @arg @ref MD_OPA_CSR_LOCK_READ_WRITE
  *         @arg @ref MD_OPA_CSR_LOCK_ONLY_READ 
  */
__STATIC_INLINE uint32_t md_opa2_is_enabled_tcrm_lock(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_LOCK) == (OPAMP_OPA2_TCMR_LOCK));
}

/**
  * @brief  Advanced Timer 2 Multiplexer Control Switch 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_enable_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_AD2CM_EN);
}

/**
  * @brief  Advanced Timer 2 Multiplexer Control Switch 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_disable_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_AD2CM_EN);
}

/**
  * @brief  Check if Advanced Timer 2 Multiplexer Control Switch 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa2_is_enabled_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_AD2CM_EN) == (OPAMP_OPA2_TCMR_AD2CM_EN));
}

/**
  * @brief  Advanced Timer 1 Multiplexer Control Switch 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_enable_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_AD1CM_EN);
}

/**
  * @brief  Advanced Timer 1 Multiplexer Control Switch 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa2_disable_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_AD1CM_EN);
}

/**
  * @brief  Check if Advanced Timer 1 Multiplexer Control Switch 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa2_is_enabled_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_AD1CM_EN) == (OPAMP_OPA2_TCMR_AD1CM_EN));
}

/**
  * @brief  Set Operational Amplifiers x Positive Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @param  vps_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VPS_VINP0
  *         @arg @ref MD_OPA_VPS_VINP1
  *         @arg @ref MD_OPA_VPS_VINP2
  *         @arg @ref MD_OPA_VPS_VINP3
  */
__STATIC_INLINE void md_opa2_set_vps_sel(OPAMP_TypeDef *opamp, md_opa_vps_selection_t vps_sel)
{
    MODIFY_REG(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_VPS_SEL, vps_sel << OPAMP_OPA2_TCMR_VPS_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers x Positive Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VPS_VINP0
  *         @arg @ref MD_OPA_VPS_VINP1
  *         @arg @ref MD_OPA_VPS_VINP2
  *         @arg @ref MD_OPA_VPS_VINP3
  */
__STATIC_INLINE uint32_t md_opa2_get_vps_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_VPS_SEL) >> OPAMP_OPA2_TCMR_VPS_SEL_POSS);
}

/**
  * @brief  Set Operational Amplifiers x Negative Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @param  vns_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VNS_VINN0     
  *         @arg @ref MD_OPA_VNS_VINN1     
  *         @arg @ref MD_OPA_VNS_RESISTANCE
  *         @arg @ref MD_OPA_VNS_OUTPUT    
  */
__STATIC_INLINE void md_opa2_set_vns_sel(OPAMP_TypeDef *opamp, md_opa_vns_selection_t vns_sel)
{
    MODIFY_REG(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_VNS_SEL, vns_sel << OPAMP_OPA2_TCMR_VNS_SEL_POS);
}

/**
  * @brief  Get Operational Amplifiers x Negative Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VNS_VINN0     
  *         @arg @ref MD_OPA_VNS_VINN1     
  *         @arg @ref MD_OPA_VNS_RESISTANCE
  *         @arg @ref MD_OPA_VNS_OUTPUT    
  */
__STATIC_INLINE uint32_t md_opa2_get_vns_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA2_TCMR, OPAMP_OPA2_TCMR_VNS_SEL) >> OPAMP_OPA2_TCMR_VNS_SEL_POS);
}

/**
  * @}
  */  
  
/** @defgroup OPA3_TCMR
  * @{
  */

/**
  * @brief  Set OPA3_TCMR Register
  * @param  opamp OPAMP Instance
  * @param  opa3_tcmr
  */
__STATIC_INLINE void md_opa3_set_tcmr(OPAMP_TypeDef *opamp, uint32_t opa3_tcmr)
{
    WRITE_REG(opamp->OPA3_TCMR, opa3_tcmr);
}

/**
  * @brief  Get OPA3_TCMR Register
  * @param  opamp OPAMP Instance
  * @retval opa3_tcmr Values
  */
__STATIC_INLINE uint32_t md_opa3_get_tcmr(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA3_TCMR);
}

/**
  * @brief  OPA3_TCMR locked W/R locked W/R
  * @param  opamp OPAMP Instance
  * @retval None
  */
__STATIC_INLINE void md_opa3_enable_tcrm_lock(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_LOCK);
}

/**
  * @brief  Get OPA3_TCMR locked W/R
  * @param  opamp OPAMP Instance
  * @retval The retval can be a value of the value:
  *         @arg @ref MD_OPA_CSR_LOCK_READ_WRITE
  *         @arg @ref MD_OPA_CSR_LOCK_ONLY_READ 
  */
__STATIC_INLINE uint32_t md_opa3_is_enabled_tcrm_lock(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_LOCK) == (OPAMP_OPA3_TCMR_LOCK));
}

/**
  * @brief  Advanced Timer 2 Multiplexer Control Switch 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_enable_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_AD2CM_EN);
}

/**
  * @brief  Advanced Timer 2 Multiplexer Control Switch 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_disable_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_AD2CM_EN);
}

/**
  * @brief  Check if Advanced Timer 2 Multiplexer Control Switch 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa3_is_enabled_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_AD2CM_EN) == (OPAMP_OPA3_TCMR_AD2CM_EN));
}

/**
  * @brief  Advanced Timer 1 Multiplexer Control Switch 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_enable_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_AD1CM_EN);
}

/**
  * @brief  Advanced Timer 1 Multiplexer Control Switch 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa3_disable_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_AD1CM_EN);
}

/**
  * @brief  Check if Advanced Timer 1 Multiplexer Control Switch 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa3_is_enabled_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_AD1CM_EN) == (OPAMP_OPA3_TCMR_AD1CM_EN));
}

/**
  * @brief  Set Operational Amplifiers x Positive Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @param  vps_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VPS_VINP0
  *         @arg @ref MD_OPA_VPS_VINP1
  *         @arg @ref MD_OPA_VPS_VINP2
  *         @arg @ref MD_OPA_VPS_VINP3
  */
__STATIC_INLINE void md_opa3_set_vps_sel(OPAMP_TypeDef *opamp, md_opa_vps_selection_t vps_sel)
{
    MODIFY_REG(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_VPS_SEL, vps_sel << OPAMP_OPA3_TCMR_VPS_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers x Positive Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VPS_VINP0
  *         @arg @ref MD_OPA_VPS_VINP1
  *         @arg @ref MD_OPA_VPS_VINP2
  *         @arg @ref MD_OPA_VPS_VINP3
  */
__STATIC_INLINE uint32_t md_opa3_get_vps_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_VPS_SEL) >> OPAMP_OPA3_TCMR_VPS_SEL_POSS);
}

/**
  * @brief  Set Operational Amplifiers x Negative Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @param  vns_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VNS_VINN0     
  *         @arg @ref MD_OPA_VNS_VINN1     
  *         @arg @ref MD_OPA_VNS_RESISTANCE
  *         @arg @ref MD_OPA_VNS_OUTPUT    
  */
__STATIC_INLINE void md_opa3_set_vns_sel(OPAMP_TypeDef *opamp, md_opa_vns_selection_t vns_sel)
{
    MODIFY_REG(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_VNS_SEL, vns_sel << OPAMP_OPA3_TCMR_VNS_SEL_POS);
}

/**
  * @brief  Get Operational Amplifiers x Negative Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VNS_VINN0     
  *         @arg @ref MD_OPA_VNS_VINN1     
  *         @arg @ref MD_OPA_VNS_RESISTANCE
  *         @arg @ref MD_OPA_VNS_OUTPUT    
  */
__STATIC_INLINE uint32_t md_opa3_get_vns_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA3_TCMR, OPAMP_OPA3_TCMR_VNS_SEL) >> OPAMP_OPA3_TCMR_VNS_SEL_POS);
}

/**
  * @}
  */   

/** @defgroup OPA4_TCMR
  * @{
  */

/**
  * @brief  Set OPA4_TCMR Register
  * @param  opamp OPAMP Instance
  * @param  opa4_tcmr
  */
__STATIC_INLINE void md_opa4_set_tcmr(OPAMP_TypeDef *opamp, uint32_t opa4_tcmr)
{
    WRITE_REG(opamp->OPA4_TCMR, opa4_tcmr);
}

/**
  * @brief  Get OPA4_TCMR Register
  * @param  opamp OPAMP Instance
  * @retval opa4_tcmr Values
  */
__STATIC_INLINE uint32_t md_opa4_get_tcmr(OPAMP_TypeDef *opamp)
{
    return (uint32_t)READ_REG(opamp->OPA4_TCMR);
}

/**
  * @brief  OPA4_TCMR locked W/R locked W/R
  * @param  opamp OPAMP Instance
  * @retval None
  */
__STATIC_INLINE void md_opa4_enable_tcrm_lock(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_LOCK);
}

/**
  * @brief  Get OPA4_TCMR locked W/R
  * @param  opamp OPAMP Instance
  * @retval The retval can be a value of the value:
  *         @arg @ref MD_OPA_CSR_LOCK_READ_WRITE
  *         @arg @ref MD_OPA_CSR_LOCK_ONLY_READ 
  */
__STATIC_INLINE uint32_t md_opa4_is_enabled_tcrm_lock(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_LOCK) == (OPAMP_OPA4_TCMR_LOCK));
}

/**
  * @brief  Advanced Timer 2 Multiplexer Control Switch 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_enable_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_AD2CM_EN);
}

/**
  * @brief  Advanced Timer 2 Multiplexer Control Switch 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_disable_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_AD2CM_EN);
}

/**
  * @brief  Check if Advanced Timer 2 Multiplexer Control Switch 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa4_is_enabled_adtimer2_mux(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_AD2CM_EN) == (OPAMP_OPA4_TCMR_AD2CM_EN));
}

/**
  * @brief  Advanced Timer 1 Multiplexer Control Switch 1 Enable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_enable_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    SET_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_AD1CM_EN);
}

/**
  * @brief  Advanced Timer 1 Multiplexer Control Switch 1 Disable.
  * @param  opamp OPAMP Instance
  * @retval None.
  */
__STATIC_INLINE void md_opa4_disable_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    CLEAR_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_AD1CM_EN);
}

/**
  * @brief  Check if Advanced Timer 1 Multiplexer Control Switch 1 is Enabled.
  * @param  opamp OPAMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_opa4_is_enabled_adtimer1_mux(OPAMP_TypeDef *opamp)
{
    return (READ_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_AD1CM_EN) == (OPAMP_OPA4_TCMR_AD1CM_EN));
}

/**
  * @brief  Set Operational Amplifiers x Positive Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @param  vps_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VPS_VINP0
  *         @arg @ref MD_OPA_VPS_VINP1
  *         @arg @ref MD_OPA_VPS_VINP2
  *         @arg @ref MD_OPA_VPS_VINP3
  */
__STATIC_INLINE void md_opa4_set_vps_sel(OPAMP_TypeDef *opamp, md_opa_vps_selection_t vps_sel)
{
    MODIFY_REG(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_VPS_SEL, vps_sel << OPAMP_OPA4_TCMR_VPS_SEL_POSS);
}

/**
  * @brief  Get Operational Amplifiers x Positive Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VPS_VINP0
  *         @arg @ref MD_OPA_VPS_VINP1
  *         @arg @ref MD_OPA_VPS_VINP2
  *         @arg @ref MD_OPA_VPS_VINP3
  */
__STATIC_INLINE uint32_t md_opa4_get_vps_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_VPS_SEL) >> OPAMP_OPA4_TCMR_VPS_SEL_POSS);
}

/**
  * @brief  Set Operational Amplifiers x Negative Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @param  vns_sel This parameter can be one of the following values:
  *         @arg @ref MD_OPA_VNS_VINN0     
  *         @arg @ref MD_OPA_VNS_VINN1     
  *         @arg @ref MD_OPA_VNS_RESISTANCE
  *         @arg @ref MD_OPA_VNS_OUTPUT    
  */
__STATIC_INLINE void md_opa4_set_vns_sel(OPAMP_TypeDef *opamp, md_opa_vns_selection_t vns_sel)
{
    MODIFY_REG(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_VNS_SEL, vns_sel << OPAMP_OPA4_TCMR_VNS_SEL_POS);
}

/**
  * @brief  Get Operational Amplifiers x Negative Input Group 2 Selection
  * @param  opamp OPAMP Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_OPA_VNS_VINN0     
  *         @arg @ref MD_OPA_VNS_VINN1     
  *         @arg @ref MD_OPA_VNS_RESISTANCE
  *         @arg @ref MD_OPA_VNS_OUTPUT    
  */
__STATIC_INLINE uint32_t md_opa4_get_vns_sel(OPAMP_TypeDef *opamp)
{
    return (uint32_t)(READ_BIT(opamp->OPA4_TCMR, OPAMP_OPA4_TCMR_VNS_SEL) >> OPAMP_OPA4_TCMR_VNS_SEL_POS);
}

/**
  * @}
  */   

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif
#endif

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
