/**********************************************************************************
 *
 * @file    md_iir.h
 * @brief   header file of md_iir.c
 *
 * @date    17 Apr 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Apr 2023     Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_IIR_H__
#define __MD_IIR_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes -------------------------------------------------------------------*/
#include "md_utils.h"

/* Exported Constants -------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */

/* Exported Macros ----------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_IIR IIR
  * @brief IIR micro driver
  * @{
  */

/* Exported Types ------------------------------------------------------------ */

/** @defgroup MD_IIR_Public_Macros IIR Public Types
  * @{
  */
typedef enum
{
    MD_IIR_MODE_ONE_STAGE_BIQUAD = (0x00000000U),
    MD_IIR_MODE_SECONDARY_BIQUAD = (0x00000001U),
}md_iir_mode_t;

typedef enum
{
    MD_IIR_1ST_ORDER_LOW_PASS_FILTER,
    MD_IIR_2ST_ORDER_LOW_PASS_FILTER,
    MD_IIR_1ST_ORDER_HIGH_PASS_FILTER,
    MD_IIR_2ST_ORDER_HIGH_PASS_FILTER
} md_iir_order_pass_filter_t;

/**
  * @}
  */
  
/** @defgroup MD_IIR_Pubulic_Types IIR Pubulic Types
  * @{
  */  
typedef struct
{
    uint32_t intd1_0;
    uint32_t intd1_1;
    uint32_t intd2_0;
    uint32_t intd2_1;
} md_iir_delay_register_typedef;

typedef struct
{
    md_iir_mode_t mode;     /**
                              @ref MD_IIR_MODE_ONE_STAGE_BIQUAD
                              @ref MD_IIR_MODE_SECONDARY_BIQUAD
                            */
    uint32_t scale_factor_0;
    uint32_t coeb_b1_0;
    uint32_t coeb_b2_0;
    uint32_t coeb_b3_0;
    uint32_t coeb_a2_0;
    uint32_t coeb_a3_0;
    uint32_t scale_factor_1;
    uint32_t coeb_b1_1;
    uint32_t coeb_b2_1;
    uint32_t coeb_b3_1;
    uint32_t coeb_a2_1;
    uint32_t coeb_a3_1;
    md_iir_delay_register_typedef delay_register;
} md_iir_init_typedef;
/**
  * @}
  */

/** @defgroup MD_IIR_Public_Functions IIR Public Functions
  * @{
  */

/** @defgroup IIR_CTRL
  * @{
  */

/* Exported Functions -------------------------------------------------------- */

/**
  * @brief  Set IIR_CTRL Register
  * @param  iir IIR Instance
  * @param  iir_ctrl
  */
__STATIC_INLINE void md_iir_set_ctrl(IIR_TypeDef *iir, uint32_t iir_ctrl)
{
    WRITE_REG(iir->CTRL, iir_ctrl);
}

/**
  * @brief  Get IIR_CTRL Register
  * @param  iir IIR Instance
  * @retval iir_ctrl Values
  */
__STATIC_INLINE uint32_t md_iir_get_ctrl(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->CTRL);
}

/**
  * @brief  Set IIR Mode Selection
  * @param  iir IIR Instance
  * @param  mode This parameter can be one of the following values:
  *         @arg @ref MD_IIR_MODE_ONE_STAGE_BIQUAD
  *         @arg @ref MD_IIR_MODE_SECONDARY_BIQUAD
  */
__STATIC_INLINE void md_iir_set_mode(IIR_TypeDef *iir, md_iir_mode_t mode)
{
    MODIFY_REG(iir->CTRL, IIR_CTRL_MODE, mode << IIR_CTRL_MODE_POS);
}

/**
  * @brief  Get IIR Mode Selection
  * @param  iir IIR Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_IIR_MODE_ONE_STAGE_BIQUAD
  *         @arg @ref MD_IIR_MODE_SECONDARY_BIQUAD
  */
__STATIC_INLINE uint32_t md_iir_get_mode(IIR_TypeDef *iir)
{
    return (uint32_t)(READ_BIT(iir->CTRL, IIR_CTRL_MODE) >> IIR_CTRL_MODE_POS);
}

/**
  * @brief  Set IIR initail
  * @param  iir IIR Instance
  * @param  none
  */
__STATIC_INLINE void md_iir_set_reset(IIR_TypeDef *iir)
{
    SET_BIT(iir->CTRL, IIR_CTRL_INIT);
}

/**
  * @brief  Set IIR start
  * @param  iir IIR Instance
  * @param  none
  */
__STATIC_INLINE void md_iir_set_start(IIR_TypeDef *iir)
{
    SET_BIT(iir->CTRL, IIR_CTRL_STR_TRIG);
}

/**
  * @}
  */

/** @defgroup IIR_STATUS
  * @{
  */

/**
  * @brief  Get IIR_STATUS Register
  * @param  iir IIR Instance
  * @retval 0:ready/1:busy
  */
__STATIC_INLINE uint32_t md_iir_is_active_flag_busy(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->STATUS);
}

/**
  * @}
  */

/** @defgroup IIR_INDATA
  * @{
  */

/**
  * @brief  Set IIR_INDATA Register
  * @param  iir IIR Instance
  * @param  data
  * @retval none
  */
__STATIC_INLINE void md_iir_set_input_data(IIR_TypeDef *iir, uint32_t data)
{
    WRITE_REG(iir->INDATA, data);
}

/**
  * @brief  Get IIR_INDATA Register
  * @param  iir IIR Instance
  * @retval iir data
  */
__STATIC_INLINE uint32_t md_iir_get_input_data(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->INDATA);
}

/**
  * @}
  */

/** @defgroup IIR_OUTDATA
  * @{
  */

/**
  * @brief  Get IIR_OUTDATA Register
  * @param  iir IIR Instance
  * @retval iir data
  */
__STATIC_INLINE uint32_t md_iir_get_output_data(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->OUTDATA);
}

/**
  * @}
  */

/** @defgroup IIR_SCALE_0
  * @{
  */

/**
  * @brief  Set IIR_SCALE_0 Register
  * @param  iir IIR Instance
  * @param  factor
  * @retval none
  */
__STATIC_INLINE void md_iir_set_first_input_factor(IIR_TypeDef *iir, uint32_t factor)
{
    WRITE_REG(iir->SCALE_0, factor);
}

/**
  * @brief  Get IIR_SCALE_0 Register
  * @param  iir IIR Instance
  * @retval iir factor
  */
__STATIC_INLINE uint32_t md_iir_get_first_input_factor(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->SCALE_0);
}

/**
  * @}
  */

/** @defgroup IIR_COEB1_0
  * @{
  */

/**
  * @brief  Set IIR_COEB1_0 Register
  * @param  iir IIR Instance
  * @param  coeb1
  * @retval none
  */
__STATIC_INLINE void md_iir_set_first_coefficient_b1(IIR_TypeDef *iir, uint32_t coeb1)
{
    WRITE_REG(iir->COEB1_0, coeb1);
}

/**
  * @brief  Get IIR_COEB1_0 Register
  * @param  iir IIR Instance
  * @retval iir coeb1
  */
__STATIC_INLINE uint32_t md_iir_get_first_coefficient_b1(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEB1_0);
}

/**
  * @}
  */

/** @defgroup IIR_COEB2_0
  * @{
  */

/**
  * @brief  Set IIR_COEB2_0 Register
  * @param  iir IIR Instance
  * @param  coeb2
  * @retval none
  */
__STATIC_INLINE void md_iir_set_first_coefficient_b2(IIR_TypeDef *iir, uint32_t coeb2)
{
    WRITE_REG(iir->COEB2_0, coeb2);
}

/**
  * @brief  Get IIR_COEB2_0 Register
  * @param  iir IIR Instance
  * @retval iir coeb2
  */
__STATIC_INLINE uint32_t md_iir_get_first_coefficient_b2(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEB2_0);
}

/**
  * @}
  */

/** @defgroup IIR_COEB3_0
  * @{
  */

/**
  * @brief  Set IIR_COEB3_0 Register
  * @param  iir IIR Instance
  * @param  coeb3
  * @retval none
  */
__STATIC_INLINE void md_iir_set_first_coefficient_b3(IIR_TypeDef *iir, uint32_t coeb3)
{
    WRITE_REG(iir->COEB3_0, coeb3);
}

/**
  * @brief  Get IIR_COEB3_0 Register
  * @param  iir IIR Instance
  * @retval iir coeb3
  */
__STATIC_INLINE uint32_t md_iir_get_first_coefficient_b3(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEB3_0);
}

/**
  * @}
  */

/** @defgroup IIR_COEA2_0
  * @{
  */

/**
  * @brief  Set IIR_COEA2_0 Register
  * @param  iir IIR Instance
  * @param  coea2
  * @retval none
  */
__STATIC_INLINE void md_iir_set_first_coefficient_a2(IIR_TypeDef *iir, uint32_t coea2)
{
    WRITE_REG(iir->COEA2_0, coea2);
}

/**
  * @brief  Get IIR_COEA2_0 Register
  * @param  iir IIR Instance
  * @retval iir coea2
  */
__STATIC_INLINE uint32_t md_iir_get_first_coefficient_a2(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEA2_0);
}

/**
  * @}
  */

/** @defgroup IIR_COEA3_0
  * @{
  */

/**
  * @brief  Set IIR_COEA3_0 Register
  * @param  iir IIR Instance
  * @param  coea3
  * @retval none
  */
__STATIC_INLINE void md_iir_set_first_coefficient_a3(IIR_TypeDef *iir, uint32_t coea3)
{
    WRITE_REG(iir->COEA3_0, coea3);
}

/**
  * @brief  Get IIR_COEA3_0 Register
  * @param  iir IIR Instance
  * @retval iir coea3
  */
__STATIC_INLINE uint32_t md_iir_get_first_coefficient_a3(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEA3_0);
}

/**
  * @}
  */

/** @defgroup IIR_INTD1_0
  * @{
  */

/**
  * @brief  Set IIR_INTD1_0 Register
  * @param  iir IIR Instance
  * @param  register1
  * @retval none
  */
__STATIC_INLINE void md_iir_set_first_delay_register1(IIR_TypeDef *iir, uint32_t register1)
{
    WRITE_REG(iir->INTD1_0, register1);
}

/**
  * @brief  Get IIR_INTD1_0 Register
  * @param  iir IIR Instance
  * @retval iir register1
  */
__STATIC_INLINE uint32_t md_iir_get_first_delay_register1(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->INTD1_0);
}

/**
  * @}
  */

/** @defgroup IIR_INTD2_0
  * @{
  */

/**
  * @brief  Set IIR_INTD2_0 Register
  * @param  iir IIR Instance
  * @param  register2
  * @retval none
  */
__STATIC_INLINE void md_iir_set_first_delay_register2(IIR_TypeDef *iir, uint32_t register2)
{
    WRITE_REG(iir->INTD2_0, register2);
}

/**
  * @brief  Get IIR_INTD2_0 Register
  * @param  iir IIR Instance
  * @retval register2
  */
__STATIC_INLINE uint32_t md_iir_get_first_delay_register2(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->INTD2_0);
}

/**
  * @}
  */

/** @defgroup IIR_SCALE_1
  * @{
  */

/**
  * @brief  Set IIR_SCALE_1 Register
  * @param  iir IIR Instance
  * @param  factor
  * @retval none
  */
__STATIC_INLINE void md_iir_set_second_input_factor(IIR_TypeDef *iir, uint32_t factor)
{
    WRITE_REG(iir->SCALE_1, factor);
}

/**
  * @brief  Get IIR_SCALE_1 Register
  * @param  iir IIR Instance
  * @retval iir factor
  */
__STATIC_INLINE uint32_t md_iir_get_second_input_factor(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->SCALE_1);
}

/**
  * @}
  */

/** @defgroup IIR_COEB1_1
  * @{
  */

/**
  * @brief  Set IIR_COEB1_1 Register
  * @param  iir IIR Instance
  * @param  coeb1
  * @retval none
  */
__STATIC_INLINE void md_iir_set_second_coefficient_b1(IIR_TypeDef *iir, uint32_t coeb1)
{
    WRITE_REG(iir->COEB1_1, coeb1);
}

/**
  * @brief  Get IIR_COEB1_1 Register
  * @param  iir IIR Instance
  * @retval iir coeb1
  */
__STATIC_INLINE uint32_t md_iir_get_second_coefficient_b1(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEB1_1);
}

/**
  * @}
  */

/** @defgroup IIR_COEB2_1
  * @{
  */

/**
  * @brief  Set IIR_COEB2_1 Register
  * @param  iir IIR Instance
  * @param  coeb2
  * @retval none
  */
__STATIC_INLINE void md_iir_set_second_coefficient_b2(IIR_TypeDef *iir, uint32_t coeb2)
{
    WRITE_REG(iir->COEB2_1, coeb2);
}

/**
  * @brief  Get IIR_COEB2_1 Register
  * @param  iir IIR Instance
  * @retval iir coeb2
  */
__STATIC_INLINE uint32_t md_iir_get_second_coefficient_b2(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEB2_1);
}

/**
  * @}
  */

/** @defgroup IIR_COEB3_1
  * @{
  */

/**
  * @brief  Set IIR_COEB3_1 Register
  * @param  iir IIR Instance
  * @param  coeb3
  * @retval none
  */
__STATIC_INLINE void md_iir_set_second_coefficient_b3(IIR_TypeDef *iir, uint32_t coeb3)
{
    WRITE_REG(iir->COEB3_1, coeb3);
}

/**
  * @brief  Get IIR_COEB3_1 Register
  * @param  iir IIR Instance
  * @retval iir coeb3
  */
__STATIC_INLINE uint32_t md_iir_get_second_coefficient_b3(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEB3_1);
}

/**
  * @}
  */

/** @defgroup IIR_COEA2_1
  * @{
  */

/**
  * @brief  Set IIR_COEA2_1 Register
  * @param  iir IIR Instance
  * @param  coea2
  * @retval none
  */
__STATIC_INLINE void md_iir_set_second_coefficient_a2(IIR_TypeDef *iir, uint32_t coea2)
{
    WRITE_REG(iir->COEA2_1, coea2);
}

/**
  * @brief  Get IIR_COEA2_1 Register
  * @param  iir IIR Instance
  * @retval iir coea2
  */
__STATIC_INLINE uint32_t md_iir_get_second_coefficient_a2(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEA2_1);
}

/**
  * @}
  */

/** @defgroup IIR_COEA3_1
  * @{
  */

/**
  * @brief  Set IIR_COEA3_1 Register
  * @param  iir IIR Instance
  * @param  coea3
  * @retval none
  */
__STATIC_INLINE void md_iir_set_second_coefficient_a3(IIR_TypeDef *iir, uint32_t coea3)
{
    WRITE_REG(iir->COEA3_1, coea3);
}

/**
  * @brief  Get IIR_COEA3_1 Register
  * @param  iir IIR Instance
  * @retval iir coea3
  */
__STATIC_INLINE uint32_t md_iir_get_second_coefficient_a3(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->COEA3_1);
}

/**
  * @}
  */

/** @defgroup IIR_INTD1_1
  * @{
  */

/**
  * @brief  Set IIR_INTD1_1 Register
  * @param  iir IIR Instance
  * @param  register1
  * @retval none
  */
__STATIC_INLINE void md_iir_set_second_delay_register1(IIR_TypeDef *iir, uint32_t register1)
{
    WRITE_REG(iir->INTD1_1, register1);
}

/**
  * @brief  Get IIR_INTD1_1 Register
  * @param  iir IIR Instance
  * @retval iir register1
  */
__STATIC_INLINE uint32_t md_iir_get_second_delay_register1(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->INTD1_1);
}

/**
  * @}
  */

/** @defgroup IIR_INTD2_1
  * @{
  */

/**
  * @brief  Set IIR_INTD2_1 Register
  * @param  iir IIR Instance
  * @param  register2
  * @retval none
  */
__STATIC_INLINE void md_iir_set_second_delay_register2(IIR_TypeDef *iir, uint32_t register2)
{
    WRITE_REG(iir->INTD2_1, register2);
}

/**
  * @brief  Get IIR_INTD2_1 Register
  * @param  iir IIR Instance
  * @retval register2
  */
__STATIC_INLINE uint32_t md_iir_get_second_delay_register2(IIR_TypeDef *iir)
{
    return (uint32_t)READ_REG(iir->INTD2_1);
}

/**
  * @}
  */

/** @defgroup IIR_Function
  * @{
  */
void md_iir_init(IIR_TypeDef *iir, md_iir_init_typedef *iir_init);
void md_iir_coefficient_calculate(uint32_t *coef, double num);
void md_iir_coefficient_determines(IIR_TypeDef *iir, uint32_t sampling_frequency, uint32_t cut_off_frequency, md_iir_order_pass_filter_t order);
double md_iir_coefficient_reverse_calculate(uint32_t coef);
void md_iir_data_calculate(uint32_t *coef, double num);
double md_iir_data_reverse_calculate(uint32_t coef);
void md_iir_get_delay_register(IIR_TypeDef *iir, md_iir_delay_register_typedef *iir_intd);
void md_iir_set_delay_register(IIR_TypeDef *iir, md_iir_delay_register_typedef *iir_intd);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __MD_IIR_H__ */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
