/**********************************************************************************
 *
 * @file    md_dma.h
 * @brief   Header file of DMA module driver.
 *
 * @date    23 Apr. 2024
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          23 Apr. 2024    Lisq            the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */


#ifndef __MD_DMA_H__
#define __MD_DMA_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes ------------------------------------------------------------------ */

#include "md_utils.h"
/** @addtogroup Micro_Driver
  * @{
  */
/** @defgroup MD_DMA DMA
  * @brief DMA micro driver
  * @{
  */

/* Exported Constants -------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */

/* Exported Types ------------------------------------------------------------ */

/** @defgroup MD_DMA_Public_Types DMA Public Types
  * @{
  */

/**
  * @brief Input signal to DMA channel
  */
typedef enum
{
    MD_DMA_MSIGSEL_UART1_TX       = 0U,
    MD_DMA_MSIGSEL_UART2_TX       = 1U,
    MD_DMA_MSIGSEL_NONE           = 2U,
    MD_DMA_MSIGSEL_SPI1_TX        = 5U,
    MD_DMA_MSIGSEL_I2C1_TX        = 7U,
    MD_DMA_MSIGSEL_UART1_RX       = 15U,
    MD_DMA_MSIGSEL_UART2_RX       = 16U,
    MD_DMA_MSIGSEL_SPI1_RX        = 20U,
    MD_DMA_MSIGSEL_I2C1_RX        = 22U,
    MD_DMA_MSIGSEL_ADC1           = 25U,
    MD_DMA_MSIGSEL_ADC2           = 26U,
    MD_DMA_MSIGSEL_SVA            = 28U,
    MD_DMA_MSIGSEL_BS16T1_UP      = 32U,
    MD_DMA_MSIGSEL_AD16C6T1_CH1   = 33U,
    MD_DMA_MSIGSEL_AD16C6T1_CH2   = 34U,
    MD_DMA_MSIGSEL_AD16C6T1_CH3   = 35U,
    MD_DMA_MSIGSEL_AD16C6T1_CH4   = 36U,
    MD_DMA_MSIGSEL_AD16C6T1_UP    = 37U,
    MD_DMA_MSIGSEL_AD16C6T1_TRIG  = 38U,
    MD_DMA_MSIGSEL_AD16C6T1_COM   = 39U,
    MD_DMA_MSIGSEL_AD16C6T2_CH1   = 40U,
    MD_DMA_MSIGSEL_AD16C6T2_CH2   = 41U,
    MD_DMA_MSIGSEL_AD16C6T2_CH3   = 42U,
    MD_DMA_MSIGSEL_AD16C6T2_CH4   = 43U,
    MD_DMA_MSIGSEL_AD16C6T2_UP    = 44U,
    MD_DMA_MSIGSEL_AD16C6T2_TRIG  = 45U,
    MD_DMA_MSIGSEL_AD16C6T2_COM   = 46U,
    MD_DMA_MSIGSEL_GP32C4T1_CH1   = 47U,
    MD_DMA_MSIGSEL_GP32C4T1_CH2   = 48U,
    MD_DMA_MSIGSEL_GP32C4T1_CH3   = 49U,
    MD_DMA_MSIGSEL_GP32C4T1_CH4   = 50U,
    MD_DMA_MSIGSEL_GP32C4T1_UP    = 51U,
    MD_DMA_MSIGSEL_GP32C4T1_TRIG  = 52U,
    MD_DMA_MSIGSEL_GP32C4T2_CH1   = 53U,
    MD_DMA_MSIGSEL_GP32C4T2_CH2   = 54U,
    MD_DMA_MSIGSEL_GP32C4T2_CH3   = 55U,
    MD_DMA_MSIGSEL_GP32C4T2_CH4   = 56U,
    MD_DMA_MSIGSEL_GP32C4T2_UP    = 57U,
    MD_DMA_MSIGSEL_GP32C4T2_TRIG  = 58U,
    MD_DMA_MSIGSEL_GP16C2T1_CH1   = 64U,
    MD_DMA_MSIGSEL_GP16C2T1_CH2   = 65U,
    MD_DMA_MSIGSEL_GP16C2T1_UP    = 66U,
    MD_DMA_MSIGSEL_GP16C2T1_TRIG  = 67U,
    MD_DMA_MSIGSEL_GP16C2T1_COM   = 68U,
    MD_DMA_MSIGSEL_GP16C2T2_CH1   = 70U,
    MD_DMA_MSIGSEL_GP16C2T2_CH2   = 71U,
    MD_DMA_MSIGSEL_GP16C2T2_UP    = 72U,
    MD_DMA_MSIGSEL_GP16C2T2_TRIG  = 73U,
    MD_DMA_MSIGSEL_GP16C2T2_COM   = 74U,
} md_dma_msigsel_t;
/**
  * @brief Priority of DMA channel
  */
typedef enum
{
    MD_DMA_LOW_PRIORITY     = 0,
    MD_DMA_MEDIUM_PRIORITY  = 1,
    MD_DMA_HIGH_PRIORITY    = 2,
    MD_DMA_HIGHEST_PRIORITY = 3
} md_dma_priority_t;

/**
  * @brief data increment
  */
typedef enum
{
    MD_DMA_DATA_INC_DISABLE  = 0x0U,
    MD_DMA_DATA_INC_ENABLE   = 0x1U,
} md_dma_data_inc_t;

/**
  * @brief Data size
  */
typedef enum
{
    MD_DMA_DATA_SIZE_BYTE     = 0x0U,   /**< Byte */
    MD_DMA_DATA_SIZE_HALFWORD = 0x1U,   /**< Halfword */
    MD_DMA_DATA_SIZE_WORD     = 0x2U,   /**< Word */
} md_dma_data_size_t;

/**
  * @brief Control how many DMA transfers can occur
  *        before the controller re-arbitrates
  */
typedef enum
{
    MD_DMA_R_POWER_1    = 0x0U,  /**< Arbitrates after each DMA transfer */
    MD_DMA_R_POWER_2    = 0x1U,  /**< Arbitrates after 2 DMA transfer */
    MD_DMA_R_POWER_4    = 0x2U,  /**< Arbitrates after 4 DMA transfer */
    MD_DMA_R_POWER_8    = 0x3U,  /**< Arbitrates after 8 DMA transfer */
    MD_DMA_R_POWER_16   = 0x4U,  /**< Arbitrates after 16 DMA transfer */
    MD_DMA_R_POWER_32   = 0x5U,  /**< Arbitrates after 32 DMA transfer */
    MD_DMA_R_POWER_64   = 0x6U,  /**< Arbitrates after 64 DMA transfer */
    MD_DMA_R_POWER_128  = 0x7U,  /**< Arbitrates after 128 DMA transfer */
    MD_DMA_R_POWER_256  = 0x8U,  /**< Arbitrates after 256 DMA transfer */
    MD_DMA_R_POWER_512  = 0x9U,  /**< Arbitrates after 512 DMA transfer */
    MD_DMA_R_POWER_1024 = 0xAU,  /**< Arbitrates after 1024 DMA transfer */
} md_dma_arbiter_config_t;

/**
  * @brief Direction
  */
typedef enum
{
    MD_DMA_DIR_TO_SRAM = 0x0U,  /**< From peripheral to sram */
    MD_DMA_DIR_TO_PERH = 0x1U,  /**< From sram to peripheral */
} md_dma_dir_t;

/**
  * @brief DMA channal configure structure
  */
typedef struct
{
    void *src;                              /**< Source data begin pointer */
    void *dst;                              /**< Destination data begin pointer */
    uint16_t size;                          /**< The total number of DMA transfers that DMA cycle contains */
    md_dma_data_size_t src_data_width;     /**< Data width, @ref dma_data_width_t */
    md_dma_data_inc_t src_inc;             /**< Source increment type */
    md_dma_data_size_t dst_data_width;     /**< Data width, @ref dma_data_width_t */
    md_dma_data_inc_t dst_inc;             /**< Destination increment type */
    md_dma_arbiter_config_t R_power;       /**< Control how many DMA transfers can occur before re-arbitrates */
    md_dma_priority_t priority;            /**< Channel priority */
    md_dma_dir_t dir;                      /**< Direction */
    type_func_t mem_to_mem;                 /**< Use primary descriptor or alternate descriptor */
    type_func_t circle_mode;                /**< Enable/Disable the useburst setting for this channel */
    type_func_t irq_tc;                     /**< Enable/disable interrupt complete */
    type_func_t irq_ht;                     /**< Enable/disable interrupt half */
    md_dma_msigsel_t msigsel;              /**< Input signal to DMA channel @ref dma_msigsel_t */
    uint8_t channel;                        /**< Channel index */
} md_dma_config_t;
/**
  * @}
  */

/* Exported Macros ----------------------------------------------------------- */

/** @defgroup MD_DMA_Public_Macros DMA Public Macros
  * @{
  */
/**
  * brief MD_DMA_CHANNEL DMA channel
  */
#define MD_DMA_CH_0 0x0U   /**< Channel 0 */
#define MD_DMA_CH_1 0x1U   /**< Channel 1 */
#define MD_DMA_CH_2 0x2U   /**< Channel 2 */
#define MD_DMA_CH_3 0x3U   /**< Channel 3 */
#define MD_DMA_CH_4 0x4U   /**< Channel 4 */
#define MD_DMA_CH_5 0x5U   /**< Channel 5 */

/**
  * @brief DMA interrupt flag
  */
#define MD_DMA_CH0_TC DMA_IER_CH0TC_MSK   /**< interrupt mask 1 */
#define MD_DMA_CH0_HT DMA_IER_CH0HT_MSK   /**< interrupt mask 2 */
#define MD_DMA_CH1_TC DMA_IER_CH0TC_MSK   /**< interrupt mask 3 */
#define MD_DMA_CH1_HT DMA_IER_CH0HT_MSK   /**< interrupt mask 4 */
#define MD_DMA_CH2_TC DMA_IER_CH0TC_MSK   /**< interrupt mask 5 */
#define MD_DMA_CH2_HT DMA_IER_CH0HT_MSK   /**< interrupt mask 6 */
#define MD_DMA_CH3_TC DMA_IER_CH0TC_MSK   /**< interrupt mask 7 */
#define MD_DMA_CH3_HT DMA_IER_CH0HT_MSK   /**< interrupt mask 8 */
#define MD_DMA_CH4_TC DMA_IER_CH0TC_MSK   /**< interrupt mask 9 */
#define MD_DMA_CH4_HT DMA_IER_CH0HT_MSK   /**< interrupt mask 10 */
#define MD_DMA_CH5_TC DMA_IER_CH0TC_MSK   /**< interrupt mask 11 */
#define MD_DMA_CH5_HT DMA_IER_CH0HT_MSK   /**< interrupt mask 12 */
/**
  * @}
  */

/* Exported Functions -------------------------------------------------------- */

/** @defgroup MD_DMA_Public_Functions DMA Public Functions
  * @{
  */
/** @defgroup MD_DMA_Public_Functions_Group2 STATUS
  * @{
  */

/**
  * @brief  Check if DMA peripheral is enabled
  * @param ch: DMA channel (0-5)
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */

/**
  * @brief  Check if DMA peripheral is enabled
  * @param ch: DMA channel (0-5)
  * @retval Status:
  *         - 0: DISABLE
  *         - 1: ENABLE
  */
__STATIC_INLINE uint32_t md_dma_is_enabled(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BIT(DMAx->CH[ch].CON, DMA_CON_CHEN_POS);
}
/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group3 Enable/Disable
  * @{
  */
/**
  * @brief  Enable DMA peripheral
  * @param ch: DMA channel(0-5)
  * @retval None
  */
__STATIC_INLINE void md_dma_enable_channel(DMA_TypeDef *DMAx, uint32_t ch)
{
    SET_BIT(DMAx->CH[ch].CON, DMA_CON_CHEN_MSK);
}

/**
  * @brief  Disable DMA peripheral
  * @param ch: DMA channel(0-5)
  * @retval None
  */
__STATIC_INLINE void md_dma_disable_channel(DMA_TypeDef *DMAx, uint32_t ch)
{
    CLEAR_BIT(DMAx->CH[ch].CON, DMA_CON_CHEN_MSK);
}

/**
  * @}
  */

/** @defgroup MD_DMA_Public_Functions_Group4 SAR
  * @{
  */
/**
  * @brief  Set DMA source data address
  * @param  ch: DMA channel
  * @param  addr: Source data address
  * @retval None
  */
__STATIC_INLINE void md_dma_set_source_addr(DMA_TypeDef *DMAx, uint32_t ch, void *addr)
{
    WRITE_REG(DMAx->CH[ch].SAR, (uint32_t)addr);
}

/**
  * @brief  Get DMA source data address
  * @param  ch: DMA channel
  * @retval DMA source data address
  */
__STATIC_INLINE void *md_dma_get_source_addr(DMA_TypeDef *DMAx, uint32_t ch)
{
    return (void *)DMAx->CH[ch].SAR;
}
/**
  * @}
  */

/** @defgroup MD_DMA_Public_Functions_Group5 DAR
  * @{
  */

/**
  * @brief  Set DMA destination data address
  * @param  ch: DMA channel
  * @param  addr: DMA destination data address
  * @retval None
  */
__STATIC_INLINE void md_dma_set_dest_addr(DMA_TypeDef *DMAx, uint32_t ch, void *addr)
{
    WRITE_REG(DMAx->CH[ch].DAR, (uint32_t)addr);
}

/**
  * @brief  Get DMA destination data address
  * @param  ch: DMA channel
  * @retval The destination data address
  */
__STATIC_INLINE void *md_dma_get_dest_addr(DMA_TypeDef *DMAx, uint32_t ch)
{
    return (void *)DMAx->CH[ch].DAR;
}

/**
  * @}
  */

/** @defgroup MD_DMA_Public_Functions_Group6 CON
  * @{
  */

/**
  * @brief  Sets DMA source data width
  * @param  ch: DMA channel
  * @param  data_width: This parameter can be one of the @ref md_dma_data_size_t
  * @retval None
  */
__STATIC_INLINE void md_dma_set_source_width(DMA_TypeDef *DMAx, uint32_t ch, uint32_t data_width)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_SDWSEL_MSK, data_width << DMA_CON_SDWSEL_POSS);
}

/**
  * @brief  Gets DMA source data width
  * @param  ch: DMA channel
  * @retval The return data width, see @ref md_dma_data_size_t
  */
__STATIC_INLINE uint32_t md_dma_get_source_width(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BITS(DMAx->CH[ch].CON, DMA_CON_SDWSEL_MSK, DMA_CON_SDWSEL_POSS);
}

/**
  * @brief  Set DMA source data increment
  * @param  ch: DMA channel
  * @param  src_inc: ENABLE(1) or DISABLE(0)
  * @retval None
  */
__STATIC_INLINE void md_dma_set_source_inc(DMA_TypeDef *DMAx, uint32_t ch, uint32_t src_inc)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_SINC_MSK, src_inc << DMA_CON_SINC_POS);
}

/**
  * @brief  Get DMA source data increment
  * @param  ch: DMA channel
  * @retval ENABLE(1) or DISABLE(0)
  */
__STATIC_INLINE uint32_t md_dma_get_source_inc(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BIT(DMAx->CH[ch].CON, DMA_CON_SINC_POS);
}

/**
  * @brief  Sets DMA dest data width
  * @param  ch: DMA channel
  * @param  data_width: This parameter can be one of the @ref md_dma_data_size_t
  * @retval None
  */
__STATIC_INLINE void md_dma_set_dest_width(DMA_TypeDef *DMAx, uint32_t ch, uint32_t data_width)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_DDWSEL_MSK, data_width << DMA_CON_DDWSEL_POSS);
}

/**
  * @brief  Gets DMA dest data width
  * @param  ch: DMA channel
  * @retval The return data width, see @ref md_dma_data_size_t
  */
__STATIC_INLINE uint32_t md_dma_get_dest_width(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BITS(DMAx->CH[ch].CON, DMA_CON_DDWSEL_MSK, DMA_CON_DDWSEL_POSS);
}
/**
  * @brief  Set DMA destination data increment
  * @param  ch: DMA channel
  * @param  dst_inc This parameter can be one of the @ref md_dma_data_inc_t
  * @retval None
  */
__STATIC_INLINE void md_dma_set_dest_inc(DMA_TypeDef *DMAx, uint32_t ch, uint32_t dst_inc)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_DINC_MSK, dst_inc << DMA_CON_DINC_POS);
}

/**
  * @brief  Get DMA destination data increment
  * @param  ch: DMA channel
  * @retval The return destination data increment, see @ref md_dma_data_inc_t
  */
__STATIC_INLINE uint32_t md_dma_get_dest_inc(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BIT(DMAx->CH[ch].CON, DMA_CON_SINC_POS);
}

/**
  * @brief  Set how many DMA transfers can occur before the controller rearbitrates
  * @param  ch: DMA channel
  * @param  r_power: The r_power value, see @ref md_dma_arbiter_config_t
  * @retval None
  */
__STATIC_INLINE void md_dma_set_r_power(DMA_TypeDef *DMAx, uint32_t ch, uint32_t r_power)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_MAX_BURST_MSK, r_power << DMA_CON_MAX_BURST_POSS);
}

/**
  * @brief  Get how many DMA transfers can occur before the controller rearbitrates
  * @param  ch: DMA channel
  * @retval The r_power value, see @ref md_dma_arbiter_config_t
  */
__STATIC_INLINE uint32_t md_dma_get_r_power(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BITS(DMAx->CH[ch].CON, DMA_CON_MAX_BURST_MSK, DMA_CON_MAX_BURST_POSS);
}

/**
  * @brief  Enable or disable DMA memory to memory transfer mode
  * @param  ch: DMA channel
  * @param  enable: 1-Enable,0-Disable
  * @retval None
  */
__STATIC_INLINE void md_dma_set_mem_to_mem(DMA_TypeDef *DMAx, uint32_t ch, uint32_t enable)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_M2M_MSK, enable << DMA_CON_M2M_POS);
}
/**
  * @brief  Get whether DMA memory to memory transfer is enabled
  * @param  ch: DMA channel
  * @retval 1-Enable,0-Disable
  */
__STATIC_INLINE uint32_t md_dma_get_mem_to_mem(DMA_TypeDef *DMAx, uint32_t ch)
{
    return (READ_BIT(DMAx->CH[ch].CON, DMA_CON_M2M_MSK) == DMA_CON_M2M_MSK);
}

/**
  * @brief  Set DMA data transfer direction (However,NO actual effect)
  * @param  ch: DMA channel
  * @param  dir: 0-peri->mem,1-mem->peri
  * @retval None
  */
__STATIC_INLINE void md_dma_set_data_dir(DMA_TypeDef *DMAx, uint32_t ch, uint32_t dir)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_DIR_MSK, dir << DMA_CON_DIR_POS);
}
/**
  * @brief  Get DMA data transfer direction (However,NO actual effect)
  * @param  ch: DMA channel
  * @retval 0-peri->mem,1-mem->peri
  */
__STATIC_INLINE uint32_t md_dma_get_data_dir(DMA_TypeDef *DMAx, uint32_t ch)
{
    return (READ_BIT(DMAx->CH[ch].CON, DMA_CON_DIR_MSK) == DMA_CON_DIR_MSK);
}

/**
  * @brief  Set DMA data transfer direction
  * @param  ch: DMA channel
  * @param  enable: 1-Enable,0-Disable
  * @retval None
  */
__STATIC_INLINE void md_dma_set_circle_mode(DMA_TypeDef *DMAx, uint32_t ch, uint32_t enable)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_CIRC_MSK, enable << DMA_CON_CIRC_POS);
}
/**
  * @brief  Get whether DMA mem to mem transfer is enabled
  * @param  ch: DMA channel
  * @retval 1-Enable,0-Disable
  */
__STATIC_INLINE uint32_t md_dma_get_circle_mode(DMA_TypeDef *DMAx, uint32_t ch)
{
    return (READ_BIT(DMAx->CH[ch].CON, DMA_CON_CIRC_MSK) == DMA_CON_CIRC_MSK);
}
/**
  * @brief  Set channel priority
  * @param  ch: DMA channel
  * @param  priority: Priority of DMA channel see @ref md_dma_priority_t
  * @retval None
  */
__STATIC_INLINE void md_dma_set_priority(DMA_TypeDef *DMAx, uint32_t ch, uint32_t priority)
{
    MODIFY_REG(DMAx->CH[ch].CON, DMA_CON_CHPRI_MSK, priority << DMA_CON_CHPRI_POSS);
}
/**
  * @brief  Get channel priority
  * @param  ch: DMA channel
  * @retval Priority of DMA channel see @ref md_dma_priority_t
  */
__STATIC_INLINE uint32_t md_dma_get_priority(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BITS(DMAx->CH[ch].CON, DMA_CON_CHPRI_MSK, DMA_CON_CHPRI_POSS);
}
/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group7 NDT
  * @{
  */
/**
  * @brief  Set the total numbers of DMA transfers
  * @param  ch: DMA channel
  * @param  n: The size of DMA transfers
  * @retval None
  */
__STATIC_INLINE void md_dma_set_transfer_size(DMA_TypeDef *DMAx, uint32_t ch, uint16_t n)
{
    MODIFY_REG(DMAx->CH[ch].NDT, DMA_NDT_TNDT_MSK, n << DMA_NDT_TNDT_POSS);
}

/**
  * @brief  Get the total numbers of DMA transfers
  * @param  ch: DMA channel
  * @retval The size
  */
__STATIC_INLINE uint32_t md_dma_get_transfer_size(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BITS(DMAx->CH[ch].NDT, DMA_NDT_TNDT_MSK, DMA_NDT_TNDT_POSS);
}
/**
  * @brief  Get the total numbers of DMA remaining data transfers
  * @param  ch: DMA channel
  * @retval The size
  */
__STATIC_INLINE uint32_t md_dma_get_remaining_size(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BITS(DMAx->CH[ch].NDT, DMA_NDT_NRDT_MSK, DMA_NDT_NRDT_POSS);
}
/**
  * @}
  */

/** @defgroup MD_DMA_Public_Functions_Group8 IFM
  * @{
  */
/**
  * @brief  Get DMA flag
  * @param  ch_flag: This parameter can be one or combined of the following values:
  *         @arg @ref MD_DMA_CH0_TC
  *         @arg @ref MD_DMA_CH0_HT
  *         @arg @ref MD_DMA_CH1_TC
  *         @arg @ref MD_DMA_CH1_HT
  *         @arg @ref MD_DMA_CH2_TC
  *         @arg @ref MD_DMA_CH2_HT
  *         @arg @ref MD_DMA_CH3_TC
  *         @arg @ref MD_DMA_CH3_HT
  *         @arg @ref MD_DMA_CH4_TC
  *         @arg @ref MD_DMA_CH4_HT
  *         @arg @ref MD_DMA_CH5_TC
  *         @arg @ref MD_DMA_CH5_HT
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_active_flag(DMA_TypeDef *DMAx, uint32_t ch_flag)
{
    return (READ_BIT(DMAx->IFM, ch_flag) == ch_flag);
}
/**
  * @brief  Get DMA transfer complete flag
  * @param  ch: DMA channel
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_active_flag_tc(DMA_TypeDef *DMAx, uint32_t ch)
{
    return md_dma_is_active_flag(DMAx, DMA_IFM_CH0TC_MSK << (ch * 2));
}
/**
  * @brief  Get DMA half transfer flag
  * @param  ch: DMA channel
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_active_flag_ht(DMA_TypeDef *DMAx, uint32_t ch)
{
    return md_dma_is_active_flag(DMAx, DMA_IFM_CH0HT_MSK << (ch * 2));
}
/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group9 ICR
  * @{
  */
/**
  * @brief  Clear DMA flag
  * @param  ch_flag: This parameter can be one or combined of the following values:
  *         @arg @ref MD_DMA_CH0_TC
  *         @arg @ref MD_DMA_CH0_HT
  *         @arg @ref MD_DMA_CH1_TC
  *         @arg @ref MD_DMA_CH1_HT
  *         @arg @ref MD_DMA_CH2_TC
  *         @arg @ref MD_DMA_CH2_HT
  *         @arg @ref MD_DMA_CH3_TC
  *         @arg @ref MD_DMA_CH3_HT
  *         @arg @ref MD_DMA_CH4_TC
  *         @arg @ref MD_DMA_CH4_HT
  *         @arg @ref MD_DMA_CH5_TC
  *         @arg @ref MD_DMA_CH5_HT
  * @retval None
  */
__STATIC_INLINE void md_dma_clear_flag(DMA_TypeDef *DMAx, uint32_t ch_flag)
{
    SET_BIT(DMAx->ICR, ch_flag);
}
/**
  * @brief  Clear DMA transfer complete flag
  * @param  ch: DMA channel
  * @retval None
  */
__STATIC_INLINE void md_dma_clear_flag_tc(DMA_TypeDef *DMAx, uint32_t ch)
{
    md_dma_clear_flag(DMAx, DMA_ICR_CH0TC_MSK << (ch * 2));
}
/**
  * @brief  Clear DMA half transfer flag
  * @param  ch: DMA channel
  * @retval None
  */
__STATIC_INLINE void md_dma_clear_flag_ht(DMA_TypeDef *DMAx, uint32_t ch)
{
    md_dma_clear_flag(DMAx, DMA_ICR_CH0HT_MSK << (ch * 2));
}

/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group10 IER
  * @{
  */
/**
  * @brief  Enable DMA CHx interrupt
  * @param  ch_flag: This parameter can be one or combined of the following values:
  *         @arg @ref MD_DMA_CH0_TC
  *         @arg @ref MD_DMA_CH0_HT
  *         @arg @ref MD_DMA_CH1_TC
  *         @arg @ref MD_DMA_CH1_HT
  *         @arg @ref MD_DMA_CH2_TC
  *         @arg @ref MD_DMA_CH2_HT
  *         @arg @ref MD_DMA_CH3_TC
  *         @arg @ref MD_DMA_CH3_HT
  *         @arg @ref MD_DMA_CH4_TC
  *         @arg @ref MD_DMA_CH4_HT
  *         @arg @ref MD_DMA_CH5_TC
  *         @arg @ref MD_DMA_CH5_HT
  * @retval None
  */
__STATIC_INLINE void md_dma_enable_it(DMA_TypeDef *DMAx, uint32_t ch_flag)
{
    SET_BIT(DMAx->IER, ch_flag);
}
/**
  * @brief  Enable DMA transfer complete interrupt
  * @param  ch: DMA channel
  * @retval None
  */
__STATIC_INLINE void md_dma_enable_it_tc(DMA_TypeDef *DMAx, uint32_t ch)
{
    md_dma_enable_it(DMAx, DMA_IER_CH0TC_MSK << (ch * 2));
}
/**
  * @brief  Enable DMA half transfer interrupt
  * @param  ch: DMA channel
  * @retval None
  */
__STATIC_INLINE void md_dma_enable_it_ht(DMA_TypeDef *DMAx, uint32_t ch)
{
    md_dma_enable_it(DMAx, DMA_IER_CH0HT_MSK << (ch * 2));
}
/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group11 IDR
  * @{
  */
/**
  * @brief  Disable DMA CHx interrupt
  * @param  ch_flag: This parameter can be one or combined of the following values:
  *         @arg @ref MD_DMA_CH0_TC
  *         @arg @ref MD_DMA_CH0_HT
  *         @arg @ref MD_DMA_CH1_TC
  *         @arg @ref MD_DMA_CH1_HT
  *         @arg @ref MD_DMA_CH2_TC
  *         @arg @ref MD_DMA_CH2_HT
  *         @arg @ref MD_DMA_CH3_TC
  *         @arg @ref MD_DMA_CH3_HT
  *         @arg @ref MD_DMA_CH4_TC
  *         @arg @ref MD_DMA_CH4_HT
  *         @arg @ref MD_DMA_CH5_TC
  *         @arg @ref MD_DMA_CH5_HT
  * @retval None
  */
__STATIC_INLINE void md_dma_disable_it(DMA_TypeDef *DMAx, uint32_t ch_flag)
{
    SET_BIT(DMAx->IDR, ch_flag);
}
/**
  * @brief  Disable DMA transfer complete interrupt
  * @param  ch: DMA channel
  * @retval None
  */
__STATIC_INLINE void md_dma_disable_it_tc(DMA_TypeDef *DMAx, uint32_t ch)
{
    md_dma_disable_it(DMAx, DMA_IDR_CH0TC_MSK << (ch * 2));
}
/**
  * @brief  Disable DMA half transfer interrupt
  * @param  ch: DMA channel
  * @retval None
  */
__STATIC_INLINE void md_dma_disable_it_ht(DMA_TypeDef *DMAx, uint32_t ch)
{
    md_dma_disable_it(DMAx, DMA_IDR_CH0HT_MSK << (ch * 2));
}
/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group12 IVS
  * @{
  */
/**
  * @brief  Check if DMA CHx interrupt is enabled
  * @param  ch_flag: This parameter can be one or combined of the following values:
  *         @arg @ref MD_DMA_CH0_TC
  *         @arg @ref MD_DMA_CH0_HT
  *         @arg @ref MD_DMA_CH1_TC
  *         @arg @ref MD_DMA_CH1_HT
  *         @arg @ref MD_DMA_CH2_TC
  *         @arg @ref MD_DMA_CH2_HT
  *         @arg @ref MD_DMA_CH3_TC
  *         @arg @ref MD_DMA_CH3_HT
  *         @arg @ref MD_DMA_CH4_TC
  *         @arg @ref MD_DMA_CH4_HT
  *         @arg @ref MD_DMA_CH5_TC
  *         @arg @ref MD_DMA_CH5_HT
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_enabled_it(DMA_TypeDef *DMAx, uint32_t ch_flag)
{
    return (READ_BIT(DMAx->IVS, ch_flag) == ch_flag);
}
/**
  * @brief  Check if DMA transfer complete interrupt is enabled
  * @param  ch: DMA channel
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_enabled_it_tc(DMA_TypeDef *DMAx, uint32_t ch)
{
    return md_dma_is_enabled_it(DMAx, DMA_IVS_CH0TC_MSK << (ch * 2));
}
/**
  * @brief  Check if DMA half transfer interrupt is enabled
  * @param  ch: DMA channel
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_enabled_it_ht(DMA_TypeDef *DMAx, uint32_t ch)
{
    return md_dma_is_enabled_it(DMAx, DMA_IVS_CH0HT_MSK << (ch * 2));
}
/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group13 RIF
  * @{
  */
/**
  * @brief  Check if DMA CHx interrupt is enabled
  * @param  ch_flag: This parameter can be one or combined of the following values:
  *         @arg @ref MD_DMA_CH0_TC
  *         @arg @ref MD_DMA_CH0_HT
  *         @arg @ref MD_DMA_CH1_TC
  *         @arg @ref MD_DMA_CH1_HT
  *         @arg @ref MD_DMA_CH2_TC
  *         @arg @ref MD_DMA_CH2_HT
  *         @arg @ref MD_DMA_CH3_TC
  *         @arg @ref MD_DMA_CH3_HT
  *         @arg @ref MD_DMA_CH4_TC
  *         @arg @ref MD_DMA_CH4_HT
  *         @arg @ref MD_DMA_CH5_TC
  *         @arg @ref MD_DMA_CH5_HT
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_active_rif(DMA_TypeDef *DMAx, uint32_t ch_flag)
{
    return (READ_BIT(DMAx->RIF, ch_flag) == ch_flag);
}
/**
  * @brief  Check if DMA transfer complete interrupt is enabled
  * @param  ch: DMA channel
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_active_rif_tc(DMA_TypeDef *DMAx, uint32_t ch)
{
    return md_dma_is_active_rif(DMAx, DMA_RIF_CH0TC_MSK << (ch * 2));
}
/**
  * @brief  Check if DMA half transfer interrupt is enabled
  * @param  ch: DMA channel
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_dma_is_active_rif_ht(DMA_TypeDef *DMAx, uint32_t ch)
{
    return md_dma_is_active_rif(DMAx, DMA_RIF_CH0HT_MSK << (ch * 2));
}
/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group14 DMA_CH_SELCON
  * @{
  */

/**
  * @brief  Set channel msigsel
  * @param  ch: DMA channel
  * @param  msigsel: This parameter can be one of the @ref md_dma_msigsel_t
  * @retval None
  */
__STATIC_INLINE void md_dma_set_channel_msigsel(DMA_TypeDef *DMAx, uint32_t ch, uint32_t msigsel)
{
    MODIFY_REG(DMA_MUX->CH_SELCON[ch], DMA_SELCON_MSIGSEL_MSK, msigsel << DMA_SELCON_MSIGSEL_POSS);
}

/**
  * @brief  get channel msigsel
  * @param  ch: DMA channel
  * @retval The msigsel, see @ref md_dma_msigsel_t
  */
__STATIC_INLINE uint32_t md_dma_get_channel_msigsel(DMA_TypeDef *DMAx, uint32_t ch)
{
    return READ_BITS(DMA_MUX->CH_SELCON[ch], DMA_SELCON_MSIGSEL_MSK, DMA_SELCON_MSIGSEL_POSS);
}
/**
  * @}
  */
/** @defgroup MD_DMA_Public_Functions_Group1 Initialization
  * @{
  */
extern void md_dma_reset(DMA_TypeDef *DMAx);
extern void md_dma_init(DMA_TypeDef *DMAx, uint32_t ch, md_dma_config_t *config);
extern void md_dma_config_struct(md_dma_config_t *config);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __MD_DMA_H__ */

/************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
