/**********************************************************************************
 *
 * @file    md_cmp.h
 * @brief   header file of md_cmp.c
 *
 * @date    17 Apr 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          17 Apr 2023     Ginger          the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_CMP_H__
#define __MD_CMP_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes -------------------------------------------------------------------*/

#include "md_utils.h"

/* Exported Constants -------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */

/* Exported Macros ----------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_CMP CMP
  * @brief CMP micro driver
  * @{
  */

/* Exported Types ------------------------------------------------------------ */

/** @defgroup MD_CMP_Public_Macros CMP Public Macros
  * @{
  */

typedef enum
{
    MD_CMP_CFG_LOCK_READ_WRITE = (0x00000000UL),       /*!< @brief CMP_CFG[31:0] bits can be read and written normally        */
    MD_CMP_CFG_LOCK_ONLY_READ  = (0x00000001UL),       /*!< @brief The CMP_CFG[31:0] bits are for read operations only.        */
}md_cmp_lock_t;

typedef enum
{
    MD_CMP_CFG_VALUE_OUTPUT_LOW_LEVEL  = (0x00000000UL),       /*!< @brief CMP output is low level.        */
    MD_CMP_CFG_VALUE_OUTPUT_HIGH_LEVEL = (0x00000001UL),       /*!< @brief CMP output is high level.        */
}md_cmp_output_level_t;

typedef enum
{
    MD_CMP_CFG_BLANKSEL_CLOSE        = (0x00000000UL),       /*!< @brief 00000:Close        */
    MD_CMP_CFG_BLANKSEL_AD16C6T1_OC6 = (0x00000001UL),       /*!< @brief xxxx1:AD16C6T1_OC6 */
    MD_CMP_CFG_BLANKSEL_AD16C6T2_OC6 = (0x00000002UL),       /*!< @brief xxx1x:AD16C6T2_OC6 */
    MD_CMP_CFG_BLANKSEL_GP32C4T1_OC4 = (0x00000004UL),       /*!< @brief xx1xx:GP32C4T1_OC4 */
    MD_CMP_CFG_BLANKSEL_GP32C4T2_OC4 = (0x00000008UL),       /*!< @brief x1xxx:GP32C4T2_OC4 */
    MD_CMP_CFG_BLANKSEL_GP16C2T1_OC2 = (0x00000010UL),       /*!< @brief 1xxxx:GP16C2T1_OC2 */
    MD_CMP_CFG_BLANKSEL_GP16C2T2_OC2 = (0x00000020UL),       /*!< @brief 1xxxx:GP16C2T2_OC2 */
}md_cmp_blank_sel_t;

typedef enum
{
    MD_CMP_CFG1_INPSEL_PA04 = (0x00000000UL),       /*!< @brief 000:PA04 */
    MD_CMP_CFG1_INPSEL_PA05 = (0x00000001UL),       /*!< @brief 001:PA05 */
    MD_CMP_CFG1_INPSEL_PA06 = (0x00000002UL),       /*!< @brief 002:PA06 */
    MD_CMP_CFG1_INPSEL_PA00 = (0x00000003UL),       /*!< @brief 003:PA00 */
}md_cmp1_postive_input_t;

typedef enum
{
    MD_CMP_CFG2_INPSEL_PA01 = (0x00000000UL),       /*!< @brief 000:PA01 */
    MD_CMP_CFG2_INPSEL_PA02 = (0x00000001UL),       /*!< @brief 001:PA02 */
    MD_CMP_CFG2_INPSEL_PA03 = (0x00000002UL),       /*!< @brief 002:PA03 */
    MD_CMP_CFG2_INPSEL_PB11 = (0x00000003UL),       /*!< @brief 003:PB11 */
}md_cmp2_postive_input_t;

typedef enum
{
    MD_CMP_CFG1_INNSEL_PB07    = (0x00000000UL),       /*!< @brief 000:PB07    */
    MD_CMP_CFG1_INNSEL_PA10    = (0x00000001UL),       /*!< @brief 001:PA10    */
    MD_CMP_CFG1_INNSEL_PA11    = (0x00000002UL),       /*!< @brief 002:PA11    */
    MD_CMP_CFG1_INNSEL_PA07    = (0x00000003UL),       /*!< @brief 003:PA07    */
    MD_CMP_CFG1_INNSEL_VRES1   = (0x00000004UL),       /*!< @brief 004:VRES1   */
    MD_CMP_CFG1_INNSEL_VRES2   = (0x00000005UL),       /*!< @brief 005:VRES2   */
    MD_CMP_CFG1_INNSEL_VRESINT = (0x00000006UL),       /*!< @brief 006:VRESINT */ 
}md_cmp1_negative_input_t;

typedef enum
{
    MD_CMP_CFG2_INNSEL_PB06    = (0x00000000UL),       /*!< @brief 000:PB06    */
    MD_CMP_CFG2_INNSEL_PA09    = (0x00000001UL),       /*!< @brief 001:PA09    */
    MD_CMP_CFG2_INNSEL_PA08    = (0x00000002UL),       /*!< @brief 002:PA08    */
    MD_CMP_CFG2_INNSEL_PB10    = (0x00000003UL),       /*!< @brief 003:PB10    */
    MD_CMP_CFG2_INNSEL_VRES1   = (0x00000004UL),       /*!< @brief 004:VRES1   */
    MD_CMP_CFG2_INNSEL_VRES2   = (0x00000005UL),       /*!< @brief 005:VRES2   */
    MD_CMP_CFG2_INNSEL_VRESINT = (0x00000006UL),       /*!< @brief 006:VRESINT */ 
}md_cmp2_negative_input_t;

/** @defgroup ALD_CMP_Pubulic_Types CMP Pubulic Types
  * @{
  */

typedef enum
{
    MD_CMP_CFG_LOCK_RW  = 0x0,
    MD_CMP_CFG_LOCK_R   = 0x1,
} md_cmp_cfg_lock_t;

typedef enum
{
    MD_CMP_CFG_VALUE_L  = 0x0,
    MD_CMP_CFG_VALUE_H  = 0x1,
} md_cmp_cfg_value_t;

typedef enum
{
    MD_CMP_CFG_HYST_CLOSE  = 0x0,
    MD_CMP_CFG_HYST_OPEN   = 0x1,
} md_cmp_cfg_hyst_t;

typedef enum
{
    MD_CMP_CFG_POLARITY_NOT   = 0x0,
    MD_CMP_CFG_POLARITY_YES   = 0x1,
} md_cmp_cfg_polarity_t;

typedef enum
{
    MD_CMP_CFG_EN_CLOSE  = 0x0,
    MD_CMP_CFG_EN_OPEN   = 0x1,
} md_cmp_cfg_en_t;

typedef enum
{
    MD_CMP_CFG2_WINOUT_CLOSE                = 0x0,
    MD_CMP_CFG2_WINOUT_OPEN                 = 0x1,
} md_cmp_cfg2_winout_t;

typedef enum
{
    MD_CMP_CFG2_WINMODE_CLOSE                = 0x0,
    MD_CMP_CFG2_WINMODE_OPEN                 = 0x1,
} md_cmp_cfg2_winmode_t;

/**
  * @brief CMP init structure definition
  */

typedef struct
{
    md_cmp_cfg_en_t         en;
    md_cmp_cfg_lock_t       lock;
    md_cmp_blank_sel_t   blanksel;
    md_cmp_cfg_hyst_t       hyst;
    md_cmp_cfg_polarity_t   polarity;
    md_cmp1_postive_input_t    inpsel;    
    md_cmp1_negative_input_t    innsel;
} md_cmp1_init_t;

typedef struct
{
    md_cmp_cfg_en_t         en;
    md_cmp_cfg_lock_t       lock;
    md_cmp_blank_sel_t   blanksel;
    md_cmp_cfg_hyst_t       hyst;
    md_cmp_cfg_polarity_t   polarity;
    md_cmp2_postive_input_t    inpsel;    
    md_cmp2_negative_input_t    innsel;
    md_cmp_cfg2_winout_t    winout;
    md_cmp_cfg2_winmode_t   winmode;    
} md_cmp2_init_t;

/**
  * @}
  */

/* Exported Functions -------------------------------------------------------- */

/** @defgroup MD_CMP_Public_Functions CMP Public Functions
  * @{
  */

/** @defgroup CFG1
  * @{
  */
/**
  * @brief  Set CMP control register (CMP_CFG1)
  * @param  cmp CMP Instance
  * @param  value The value write in CMP_CFG1
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_cfg1(CMP_TypeDef *cmp, uint32_t value)
{
    WRITE_REG(cmp->CFG1, value);
}

/**
  * @brief  Get CMP control register (CMP_CFG1)
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_cmp_get_cfg1(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_REG(cmp->CFG1));
}

/**
  * @brief  Comparator 1 locked W/R
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp1_enable_lock(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_LOCK);
}

/**
  * @brief  Get Comparator 1 locked
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG_LOCK_READ_WRITE
            @arg @ref MD_CMP_CFG_LOCK_ONLY_READ
  */
__STATIC_INLINE uint32_t md_cmp1_is_enabled_lock(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_LOCK) == (CMP_CFG1_LOCK));
}

/**
  * @brief  Get Comparator 1 output
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG_VALUE_OUTPUT_LOW_LEVEL
            @arg @ref MD_CMP_CFG_VALUE_OUTPUT_HIGH_LEVEL
  */
__STATIC_INLINE uint32_t md_cmp1_get_output_level(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_VALUE) >> CMP_CFG1_VALUE_POS);
}

/**
  * @brief  Set Comparator 1 masking function
  * @param  cmp CMP Instance
  * @param  blanksel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG_BLANKSEL_CLOSE       
  *         @arg MD_CMP_CFG_BLANKSEL_AD16C6T1_OC6
  *         @arg MD_CMP_CFG_BLANKSEL_AD16C6T2_OC6
  *         @arg MD_CMP_CFG_BLANKSEL_GP32C4T1_OC4
  *         @arg MD_CMP_CFG_BLANKSEL_GP32C4T2_OC4
  *         @arg MD_CMP_CFG_BLANKSEL_GP16C2T1_OC2
  *         @arg MD_CMP_CFG_BLANKSEL_GP16C2T2_OC2
  * @retval None
  */
__STATIC_INLINE void md_cmp1_set_blanking_source(CMP_TypeDef *cmp, md_cmp_blank_sel_t blanksel)
{
    MODIFY_REG(cmp->CFG1, CMP_CFG1_BLANKSEL, blanksel << CMP_CFG1_BLANKSEL_POSS);
}

/**
  * @brief  Return Comparator 1 masking function
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG_BLANKSEL_CLOSE       
  *         @arg MD_CMP_CFG_BLANKSEL_AD16C6T1_OC6
  *         @arg MD_CMP_CFG_BLANKSEL_AD16C6T2_OC6
  *         @arg MD_CMP_CFG_BLANKSEL_GP32C4T1_OC4
  *         @arg MD_CMP_CFG_BLANKSEL_GP32C4T2_OC4
  *         @arg MD_CMP_CFG_BLANKSEL_GP16C2T1_OC2
  *         @arg MD_CMP_CFG_BLANKSEL_GP16C2T2_OC2
  */
__STATIC_INLINE uint32_t md_cmp1_get_blanking_source(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_BLANKSEL) >> CMP_CFG1_BLANKSEL_POSS);
} 

/**
  * @brief  Enable Comparator 1 hysteresis function
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp1_enable_hysteresis(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_HYST);
}
/**
  * @brief  Disable Comparator 1 hysteresis function
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp1_disable_hysteresis(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG1, CMP_CFG1_HYST);
}

/**
  * @brief  Check if Comparator 1 hysteresis function
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp1_is_enabled_hysteresis(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_HYST) == (CMP_CFG1_HYST));
}

/**
  * @brief  Enable Comparator 1 output polarity
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp1_enable_polarity(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_POLARITY);
}

/**
  * @brief  Disable Comparator 1 output polarity
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp1_disable_polarity(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG1, CMP_CFG1_POLARITY);
}

/**
  * @brief  Check if Comparator 1 output polarity
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp1_is_enabled_polarity(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_POLARITY) == (CMP_CFG1_POLARITY));
}

/**
  * @brief  Set Comparator 1 positive input selection
  * @param  cmp CMP Instance
  * @param  inpsel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG1_INPSEL_PA04
  *         @arg MD_CMP_CFG1_INPSEL_PA05
  *         @arg MD_CMP_CFG1_INPSEL_PA06
  *         @arg MD_CMP_CFG1_INPSEL_PA00
  * @retval None
  */
__STATIC_INLINE void md_cmp1_set_input_positive_select(CMP_TypeDef *cmp, md_cmp1_postive_input_t inpsel)
{
    MODIFY_REG(cmp->CFG1, CMP_CFG1_INPSEL, inpsel << CMP_CFG1_INPSEL_POSS);
}

/**
  * @brief  Return Comparator 1 positive input selection
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG1_INPSEL_PA04
  *         @arg MD_CMP_CFG1_INPSEL_PA05
  *         @arg MD_CMP_CFG1_INPSEL_PA06
  *         @arg MD_CMP_CFG1_INPSEL_PA00
  */
__STATIC_INLINE uint32_t md_cmp1_get_input_positive_select(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_INPSEL) >> CMP_CFG1_INPSEL_POSS);
}

/**
  * @brief  Set Comparator 1 negative input selection
  * @param  cmp CMP Instance
  * @param  innsel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG1_INNSEL_PB07     
  *         @arg MD_CMP_CFG1_INNSEL_PA10     
  *         @arg MD_CMP_CFG1_INNSEL_PA11     
  *         @arg MD_CMP_CFG1_INNSEL_PA07     
  *         @arg MD_CMP_CFG1_INNSEL_VRES1    
  *         @arg MD_CMP_CFG1_INNSEL_VRES2    
  *         @arg MD_CMP_CFG1_INNSEL_VRESINT  
  * @retval None
  */
__STATIC_INLINE void md_cmp1_set_input_negative_select(CMP_TypeDef *cmp, md_cmp1_negative_input_t innsel)
{
    MODIFY_REG(cmp->CFG1, CMP_CFG1_INNSEL, innsel << CMP_CFG1_INNSEL_POSS);
}

/**
  * @brief  Return Comparator 1 negative input selection
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG1_INNSEL_PB07     
  *         @arg MD_CMP_CFG1_INNSEL_PA10     
  *         @arg MD_CMP_CFG1_INNSEL_PA11     
  *         @arg MD_CMP_CFG1_INNSEL_PA07     
  *         @arg MD_CMP_CFG1_INNSEL_VRES1    
  *         @arg MD_CMP_CFG1_INNSEL_VRES2    
  *         @arg MD_CMP_CFG1_INNSEL_VRESINT  
  */
__STATIC_INLINE uint32_t md_cmp1_get_input_negative_select(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG1, CMP_CFG1_INNSEL) >> CMP_CFG1_INNSEL_POSS);
}

/**
  * @brief  Enable Comparator 1 switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp1_enable(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG1_EN);
}
/**
  * @brief  Disable Comparator 1 switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp1_disable(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG1, CMP_CFG1_EN);
}

/**
  * @brief  Get Comparator 1 switch
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp1_is_enabled(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG1, CMP_CFG1_EN) == (CMP_CFG1_EN));
}
/**
  * @}
  */

/** @defgroup CFG2
  * @{
  */
/**
  * @brief  Set CMP control register (CMP_CFG2)
  * @param  cmp CMP Instance
  * @param  value The value write in CMP_CFG2
  * @retval None
  */
__STATIC_INLINE void md_cmp_set_cfg2(CMP_TypeDef *cmp, uint32_t value)
{
    WRITE_REG(cmp->CFG2, value);
}

/**
  * @brief  Get CMP control register (CMP_CFG1)
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_cmp_get_cfg2(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_REG(cmp->CFG2));
}

/**
  * @brief  Comparator 2 locked W/R
  * @param  cmp CMP Instance
  * @param  lock
            @arg @ref MD_CMP_CFG2_LOCK_READ_WRITE
            @arg @ref MD_CMP_CFG2_LOCK_ONLY_READ
  * @retval None
  */
__STATIC_INLINE void md_cmp2_enable_lock(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG1, CMP_CFG2_LOCK);
}

/**
  * @brief  Get Comparator 2 locked
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG_LOCK_READ_WRITE
            @arg @ref MD_CMP_CFG_LOCK_ONLY_READ
  */
__STATIC_INLINE uint32_t md_cmp2_is_enabled_lock(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG2, CMP_CFG2_LOCK) == (CMP_CFG2_LOCK));
}

/**
  * @brief  Get Comparator 2 output
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
            @arg @ref MD_CMP_CFG2_VALUE_OUTPUT_LOW_LEVEL
            @arg @ref MD_CMP_CFG2_VALUE_OUTPUT_HIGH_LEVEL
  */
__STATIC_INLINE uint32_t md_cmp2_get_output_level(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_VALUE) >> CMP_CFG2_VALUE_POS);
}

/**
  * @brief  Set Comparator 2 masking function
  * @param  cmp CMP Instance
  * @param  blanksel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG_BLANKSEL_CLOSE       
  *         @arg MD_CMP_CFG_BLANKSEL_AD16C6T1_OC6
  *         @arg MD_CMP_CFG_BLANKSEL_AD16C6T2_OC6
  *         @arg MD_CMP_CFG_BLANKSEL_GP32C4T1_OC4
  *         @arg MD_CMP_CFG_BLANKSEL_GP32C4T2_OC4
  *         @arg MD_CMP_CFG_BLANKSEL_GP16C2T1_OC2
  *         @arg MD_CMP_CFG_BLANKSEL_GP16C2T2_OC2 
  * @retval None
  */
__STATIC_INLINE void md_cmp2_set_blanking_source(CMP_TypeDef *cmp, md_cmp_blank_sel_t blanksel)
{
    MODIFY_REG(cmp->CFG2, CMP_CFG2_BLANKSEL, blanksel << CMP_CFG2_BLANKSEL_POSS);
}

/**
  * @brief  Return Comparator 2 masking function
  * @param  cmp CMP Instance
  * @retval None
  *         @arg MD_CMP_CFG_BLANKSEL_CLOSE       
  *         @arg MD_CMP_CFG_BLANKSEL_AD16C6T1_OC6
  *         @arg MD_CMP_CFG_BLANKSEL_AD16C6T2_OC6
  *         @arg MD_CMP_CFG_BLANKSEL_GP32C4T1_OC4
  *         @arg MD_CMP_CFG_BLANKSEL_GP32C4T2_OC4
  *         @arg MD_CMP_CFG_BLANKSEL_GP16C2T1_OC2
  *         @arg MD_CMP_CFG_BLANKSEL_GP16C2T2_OC2 
  */
__STATIC_INLINE uint32_t md_cmp2_get_blanking_source(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_BLANKSEL) >> CMP_CFG2_BLANKSEL_POSS);
}

/**
  * @brief  Enable Comparator 2 hysteresis function
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_enable_hysteresis(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG2, CMP_CFG2_HYST);
}
/**
  * @brief  Disable Comparator 2 hysteresis function
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_disable_hysteresis(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG2, CMP_CFG2_HYST);
}

/**
  * @brief  Check if Comparator 2 hysteresis function
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp2_is_enabled_hysteresis(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG2, CMP_CFG2_HYST) == (CMP_CFG2_HYST));
}

/**
  * @brief  Enable Comparator 2 output polarity
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_enable_polarity(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG2, CMP_CFG2_POLARITY);
}

/**
  * @brief  Disable Comparator 2 output polarity
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_disable_polarity(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG2, CMP_CFG2_POLARITY);
}

/**
  * @brief  Check if Comparator 2 output polarity
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp2_is_enabled_polarity(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG2, CMP_CFG2_POLARITY) == (CMP_CFG2_POLARITY));
}

/**
  * @brief  Set Comparator 2 positive input selection
  * @param  cmp CMP Instance
  * @param  inpsel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG2_INPSEL_PA01 
  *         @arg MD_CMP_CFG2_INPSEL_PA02 
  *         @arg MD_CMP_CFG2_INPSEL_PA03 
  *         @arg MD_CMP_CFG2_INPSEL_PB11 
  * @retval None
  */
__STATIC_INLINE void md_cmp2_set_input_positive_select(CMP_TypeDef *cmp, md_cmp2_postive_input_t inpsel)
{
    MODIFY_REG(cmp->CFG2, CMP_CFG2_INPSEL, inpsel << CMP_CFG2_INPSEL_POSS);
}

/**
  * @brief  Return Comparator 2 positive input selection
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG2_INPSEL_PA01 
  *         @arg MD_CMP_CFG2_INPSEL_PA02 
  *         @arg MD_CMP_CFG2_INPSEL_PA03 
  *         @arg MD_CMP_CFG2_INPSEL_PB11  
  */
__STATIC_INLINE uint32_t md_cmp2_get_input_positive_select(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_INPSEL) >> CMP_CFG2_INPSEL_POSS);
}

/**
  * @brief  Set Comparator 2 negative input selection
  * @param  cmp CMP Instance
  * @param  innsel This parameter can be one of the following values:
  *         @arg MD_CMP_CFG2_INNSEL_PB06     
  *         @arg MD_CMP_CFG2_INNSEL_PA09     
  *         @arg MD_CMP_CFG2_INNSEL_PA08     
  *         @arg MD_CMP_CFG2_INNSEL_PB10     
  *         @arg MD_CMP_CFG2_INNSEL_VRES1    
  *         @arg MD_CMP_CFG2_INNSEL_VRES2    
  *         @arg MD_CMP_CFG2_INNSEL_VRESINT  
  * @retval None
  */
__STATIC_INLINE void md_cmp2_set_input_negative_select(CMP_TypeDef *cmp, md_cmp2_negative_input_t innsel)
{
    MODIFY_REG(cmp->CFG2, CMP_CFG2_INNSEL, innsel << CMP_CFG2_INNSEL_POSS);
}

/**
  * @brief  Return Comparator 2 negative input selection
  * @param  cmp CMP Instance
  * @retval The retval can be a value of the value:
  *         @arg MD_CMP_CFG2_INNSEL_PB06     
  *         @arg MD_CMP_CFG2_INNSEL_PA09     
  *         @arg MD_CMP_CFG2_INNSEL_PA08     
  *         @arg MD_CMP_CFG2_INNSEL_PB10     
  *         @arg MD_CMP_CFG2_INNSEL_VRES1    
  *         @arg MD_CMP_CFG2_INNSEL_VRES2    
  *         @arg MD_CMP_CFG2_INNSEL_VRESINT
  */
__STATIC_INLINE uint32_t md_cmp2_get_input_negative_select(CMP_TypeDef *cmp)
{
    return (uint32_t)(READ_BIT(cmp->CFG2, CMP_CFG2_INNSEL) >> CMP_CFG2_INNSEL_POSS);
}

/**
  * @brief  Enable Comparator 2 window output mode
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_enable_window_out(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG2, CMP_CFG2_WINOUT);
}
/**
  * @brief  Disable Comparator 2 window output mode
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_disable_window_out(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG2, CMP_CFG2_WINOUT);
}

/**
  * @brief  Get Comparator 2 window output mode
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp_is_enabled_window_out(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG2, CMP_CFG2_WINOUT) == (CMP_CFG2_WINOUT));
}

/**
  * @brief  Enable Comparator 2 window monitor mode
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_enable_window_mode(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG2, CMP_CFG2_WINMODE);
}
/**
  * @brief  Disable Comparator 2 window monitor mode
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp_disable_window_mode(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG2, CMP_CFG2_WINMODE);
}

/**
  * @brief  Get Comparator 2 window monitor mode
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp_is_enabled_window_mode(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG2, CMP_CFG2_WINMODE) == (CMP_CFG2_WINMODE));
}

/**
  * @brief  Enable Comparator 2 switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_enable(CMP_TypeDef *cmp)
{
    SET_BIT(cmp->CFG2, CMP_CFG2_EN);
}
/**
  * @brief  Disable Comparator 2 switch
  * @param  cmp CMP Instance
  * @retval None
  */
__STATIC_INLINE void md_cmp2_disable(CMP_TypeDef *cmp)
{
    CLEAR_BIT(cmp->CFG2, CMP_CFG2_EN);
}

/**
  * @brief  Get Comparator 2 switch
  * @param  cmp CMP Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_cmp2_is_enabled(CMP_TypeDef *cmp)
{
    return (READ_BIT(cmp->CFG2, CMP_CFG2_EN) == (CMP_CFG2_EN));
}

/**
  * @}
  */

/** @addtogroup CMP_Public_Functions
  * @{
  */
void md_cmp1_init(md_cmp1_init_t *hperh);
void md_cmp2_init(md_cmp2_init_t *hperh);
uint32_t md_cmp_read_cmp1(CMP_TypeDef *cmp);
uint32_t md_cmp_read_cmp2(CMP_TypeDef *cmp);
void md_cmp_deinit(void);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __MD_CMP_H__ */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
