/**********************************************************************************
 *
 * @file    md_can.h
 * @brief   header file of md_can.c
 *
 * @date    19 Apr 2023
 * @author  Kevin Zeng
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          19 Apr 2023     Kevin Zeng      the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */
/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_CAN_H__
#define __MD_CAN_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes -------------------------------------------------------------------*/
#include "md_utils.h"

/* Exported Constants -------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */

/** @addtogroup Micro_Driver
  * @{
  */
  
/** @defgroup MD_CAN CAN
  * @brief CAN micro driver
  * @{
  */ 

/* Exported Macros ----------------------------------------------------------- */

/** @defgroup MD_CAN_Public_Macros CAN Public Macros
  * @{
  */
/**
  * @brief last error code
  */
#define MD_CAN_RESET_ERROR_RECORD             (7UL)

/* Exported Types ------------------------------------------------------------ */

/**
  * @brief Receive FIFO overflow processing mode
  */
typedef enum
{
    MD_CAN_RXFIFO_OVERFLOW_OVERRIDE=(0UL),
    MD_CAN_RXFIFO_OVERFLOW_FLUSH=(1UL)
}md_can_rxfifo_overflow_list_t;
/**
  * @brief Send email priority
  */
typedef enum
{
    MD_CAN_SENDMAIL_ORDER_MESSAGE_ID=(0UL),
    MD_CAN_SENDMAIL_ORDER_REQUEST=(1UL)
}md_can_sendmail_otder_list_t;

/**
  * @brief sleep request
  */
typedef enum
{
    MD_CAN_SLEEP_REQUEST_LEAVE=(0UL),
    MD_CAN_SLEEP_REQUEST_ENTER=(1UL)
}md_can_sleep_request_list_t;

/**
  * @brief initialization request
  */
typedef enum
{
    MD_CAN_INITIAL_REQUEST_LEAVE=(0UL),
    MD_CAN_INITIAL_REQUEST_ENTER=(1UL)
}md_can_initial_request_list_t;

/**
  * @brief Email stop request
  */
typedef enum
{
    MD_CAN_MAILBOX_STOP_REQUEST_CANCEL=(0UL),
    MD_CAN_MAILBOX_STOP_REQUEST=(1UL)
}md_can_mailbox_stop_request_list_t;

/**
  * @brief Release FIFO receive mailbox
  */
typedef enum
{
    MD_CAN_RXFIFO_KEEP_MESSAGE=(0UL),
    MD_CAN_RXFIFO_FREE_MESSAGE=(1UL)
}md_can_rxfifo_release_list_t;

/**
  * @brief Bus Closed Flag
  */
typedef enum
{
    MD_CAN_CLEAR_BUS_OFF_FLAG=(0UL),
    MD_CAN_SET_BUS_OFF_FLAG=(1UL)
}md_can_bus_closed_flag_t;

/**
  * @brief silent mode (debug)
  */
typedef enum
{
    MD_CAN_SILENT_NORMAL_MODE=(0UL),
    MD_CAN_SILENT_SILENCE_MODE=(1UL)
}md_can_filter_silent_mode_list_t;

/**
  * @brief Filter initialization mode
  */
typedef enum
{
    MD_CAN_FILTER_WORK_MODE=(0UL),
    MD_CAN_FILTER_INITIAL_MODE=(1UL)
}md_can_filter_initial_mode_list_t;

/**
  * @brief MD_CAN_FILTER Num
  */
typedef enum
{
    MD_CAN_FILTER_0=(0UL),   /*!< Select filter 0 */
    MD_CAN_FILTER_1=(1UL),   /*!< Select filter 1 */
    MD_CAN_FILTER_2=(2UL),   /*!< Select filter 2 */
    MD_CAN_FILTER_3=(3UL),   /*!< Select filter 3 */
    MD_CAN_FILTER_4=(4UL),   /*!< Select filter 4 */
    MD_CAN_FILTER_5=(5UL),   /*!< Select filter 5 */
    MD_CAN_FILTER_6=(6UL),   /*!< Select filter 6 */
    MD_CAN_FILTER_7=(7UL),   /*!< Select filter 7 */
    MD_CAN_FILTER_8=(8UL),   /*!< Select filter 8 */
    MD_CAN_FILTER_9=(9UL),   /*!< Select filter 9 */
    MD_CAN_FILTER_10=(10UL), /*!< Select filter 10 */
    MD_CAN_FILTER_11=(11UL), /*!< Select filter 11 */
    MD_CAN_FILTER_12=(12UL), /*!< Select filter 12 */
    MD_CAN_FILTER_13=(13UL)  /*!< Select filter 13 */
}md_can_filter_list_t;

/**
  * @brief filter pattern
  */
typedef enum
{
    MD_CAN_FILTER_MOD_MASK=(0UL),
    MD_CAN_FILTER_MOD_LIST_OF_IDENTIFIERS=(1UL)
}md_can_filter_mode_list_t;
/**
  * @brief width selection
  */
typedef enum
{
    MD_CAN_FILTER_SEL_16BIT=(0UL),
    MD_CAN_FILTER_SEL_32BIT=(1UL)
}md_can_filter_width_list_t;
/**
  * @brief FIFO Allocation of Filters
  */
typedef enum
{
    MD_CAN_FILTER_ASSIGN_RXFIFO0=(0UL),
    MD_CAN_FILTER_ASSIGN_RXFIFO1=(1UL)
}md_can_filter_fifo_allocation_t;
/**
  * @brief filter start
  */
typedef enum
{
    MD_CAN_FILTER_DEACTIVATE=(0UL),
    MD_CAN_FILTER_ACTIVE=(1UL)
}md_can_filter_active_action_t;

typedef enum
{
    INITIAL_MODE,
    NORMAL_MODE,
    SLEEP_MODE
} md_can_mode_t;

typedef enum
{
    NORMAL_DEBUGMODE,
    LOOPBACK_DEBUGMODE,
    SILENT_DEBUGMODE,
    SILENT_LOOPBACK_DEBUGMODE
} md_can_debugmode_t;

typedef enum
{
    FIFO0,
    FIFO1
} md_can_rx_fifo_t;

typedef enum
{
    STANDARD_IDENTIFIER = 0,
    EXTENDED_IDENTIFIER = 1
} md_can_id_type_t;

typedef enum
{
    DATA_FRAME = 0,
    REMOTE_FRAME = 1
} md_frame_type_t;
/**
  * @}
  */

/** @defgroup MD_CAN_Pubulic_Types CAN Pubulic Types
  * @{
  */
typedef struct
{
    uint32_t bpsc;
    uint32_t seg1;
    uint32_t seg2;
    uint32_t resjw;
} md_can_bittime_config_typedef;

typedef struct
{
    md_can_bittime_config_typedef bit_time;
    md_can_sendmail_otder_list_t transfer_mailbox_priority;
    md_can_rxfifo_overflow_list_t receiver_fifo_overflow_mode;
    TypeFunc auto_resend_disable;
    TypeFunc auto_wakeup_enable;
    TypeFunc auto_quit_busSwitch_enable;
    TypeFunc timer_trigger_communication_enable;
} md_can_init_typedef;

typedef struct
{
    md_can_filter_mode_list_t filter_mode; 
    md_can_filter_width_list_t filter_width; 
    md_can_filter_fifo_allocation_t filter_fifo_assign;

    struct
    {
        struct
        {
            uint32_t id;
            uint32_t mask;
            md_can_id_type_t id_type;
            md_frame_type_t frame_type;
        } identifier_32bit_mask;

        struct
        {
            uint32_t id1;
            uint32_t id2;
            md_can_id_type_t id1_type;
            md_frame_type_t frame1_type;
            md_can_id_type_t id2_type;
            md_frame_type_t frame2_type;
        } identifier_32bit_list;

        struct
        {
            uint32_t id1;
            uint32_t mask1;
            uint32_t id2;
            uint32_t mask2;
            md_can_id_type_t id1_type;
            md_frame_type_t frame1_type;
            md_can_id_type_t id2_type;
            md_frame_type_t frame2_type;
        } identifier_16bit_mask;

        struct
        {
            uint32_t id1;
            uint32_t id2;
            uint32_t id3;
            uint32_t id4;
            md_can_id_type_t id1_type;
            md_frame_type_t frame1_type;
            md_can_id_type_t id2_type;
            md_frame_type_t frame2_type;
            md_can_id_type_t id3_type;
            md_frame_type_t frame3_type;
            md_can_id_type_t id4_type;
            md_frame_type_t frame4_type;
        } identifier_16bit_list;
    } filter_mode_select;

} md_can_filter_config_typedef;

typedef struct
{
    md_can_id_type_t id_type;
    uint32_t id;
    md_frame_type_t frame_type;
    uint32_t length;
    uint32_t data[8];
    TypeFunc send_global_time_enable;
    uint32_t message_timestamp;
} md_can_txmailbox_typedef;

typedef struct
{
    md_can_id_type_t id_type;
    uint32_t id;
    md_frame_type_t frame_type;
    uint32_t length;
    uint32_t data[8];
    uint32_t filter_match_index;
    uint32_t message_timestamp;
} md_can_rxmailbox_typedef;
/**
  * @}
  */

/* Exported Functions -------------------------------------------------------- */

/** @defgroup MD_CAN_Public_Functions CAN Public Functions
  * @{
  */

/** @defgroup CON
  * @{
  */
/**
  * @brief  CAN Set CON
  * @param  CANx CAN Instance
  * @param  Reg_Value is a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_con(CAN_TypeDef *CANx, uint32_t Reg_Value)
{
    WRITE_REG(CANx->CON, Reg_Value);
}

/**
  * @brief  CAN Get CON
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_con(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->CON));
}

/**
  * @brief  CAN enable debug stop
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_debug_stop(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->CON, CAN_CON_DBGSTP);
}

/**
  * @brief  CAN disable debug stop
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_debug_stop(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->CON, CAN_CON_DBGSTP);
}

/**
  * @brief  CAN is enabled debug stop
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_debug_stop(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->CON, CAN_CON_DBGSTP) == (CAN_CON_DBGSTP));
}

/**
  * @brief  CAN enable software reset
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_software_reset(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->CON, CAN_CON_RST);
}

/**
  * @brief  CAN disable software reset
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_software_reset(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->CON, CAN_CON_RST);
}

/**
  * @brief  CAN is enabled software reset
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_software_reset(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->CON, CAN_CON_RST) == (CAN_CON_RST));
}

/**
  * @brief  CAN enable timer trigger connect
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_timer_trigger_connect(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->CON, CAN_CON_TTCEN);
}

/**
  * @brief  CAN disable timer trigger connect
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_timer_trigger_connect(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->CON, CAN_CON_TTCEN);
}

/**
  * @brief  CAN is enabled timer trigger connect
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_timer_trigger_connect(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->CON, CAN_CON_TTCEN) == (CAN_CON_TTCEN));
}

/**
  * @brief  CAN enable aboffen
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_aboffen(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->CON, CAN_CON_ABOFFEN);
}

/**
  * @brief  CAN disable aboffen
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_aboffen(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->CON, CAN_CON_ABOFFEN);
}

/**
  * @brief  CAN is enabled aboffen
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_aboffen(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->CON, CAN_CON_ABOFFEN) == (CAN_CON_ABOFFEN));
}

/**
  * @brief  CAN enable auto wake up
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_auto_wake_up(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->CON, CAN_CON_AWKEN);
}

/**
  * @brief  CAN disable auto wake up
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_auto_wake_up(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->CON, CAN_CON_AWKEN);
}

/**
  * @brief  CAN is enabled auto wake up
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_auto_wake_up(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->CON, CAN_CON_AWKEN) == (CAN_CON_AWKEN));
}

/**
  * @brief  CAN enable artxdis
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_artxdis(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->CON, CAN_CON_ARTXDIS);
}

/**
  * @brief  CAN disable artxdis
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_artxdis(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->CON, CAN_CON_ARTXDIS);
}

/**
  * @brief  CAN is enabled artxdis
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_artxdis(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->CON, CAN_CON_ARTXDIS) == (CAN_CON_ARTXDIS));
}

/**
  * @brief  Set CAN RXFOPM
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_RXFIFO_OVERFLOW_OVERRIDE
  *         @arg @ref MD_CAN_RXFIFO_OVERFLOW_FLUSH
  * @retval None
  */
__STATIC_INLINE void md_can_set_RXFOPM(CAN_TypeDef *CANx, md_can_rxfifo_overflow_list_t value)
{
    MODIFY_REG(CANx->CON, CAN_CON_RXFOPM, value << CAN_CON_RXFOPM_POS);
}

/**
  * @brief  Get CAN RXFOPM
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_rxfifo_overflow_list_t md_can_get_RXFOPM(CAN_TypeDef *CANx)
{
    return (md_can_rxfifo_overflow_list_t)(READ_BIT(CANx->CON, CAN_CON_RXFOPM) >> CAN_CON_RXFOPM_POS);
}

/**
  * @brief  Set CAN TXMP
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_SENDMAIL_ORDER_MESSAGE_ID
  *         @arg @ref MD_CAN_SENDMAIL_ORDER_REQUEST
  * @retval None
  */
__STATIC_INLINE void md_can_set_TXMP(CAN_TypeDef *CANx, md_can_sendmail_otder_list_t value)
{
    MODIFY_REG(CANx->CON, CAN_CON_TXMP, value << CAN_CON_TXMP_POS);
}

/**
  * @brief  Get CAN TXMP
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_sendmail_otder_list_t md_can_get_TXMP(CAN_TypeDef *CANx)
{
    return (md_can_sendmail_otder_list_t)(READ_BIT(CANx->CON, CAN_CON_TXMP) >> CAN_CON_TXMP_POS);
}

/**
  * @brief  Set CAN sleep request
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_SLEEP_REQUEST_LEAVE
  *         @arg @ref MD_CAN_SLEEP_REQUEST_ENTER
  * @retval None
  */
__STATIC_INLINE void md_can_set_sleep_request(CAN_TypeDef *CANx, md_can_sleep_request_list_t value)
{
    MODIFY_REG(CANx->CON, CAN_CON_SLPREQ, value << CAN_CON_SLPREQ_POS);
}

/**
  * @brief  Get CAN sleep request
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_sleep_request_list_t md_can_get_sleep_request(CAN_TypeDef *CANx)
{
    return (md_can_sleep_request_list_t)(READ_BIT(CANx->CON, CAN_CON_SLPREQ) >> CAN_CON_SLPREQ_POS);
}

/**
  * @brief  Set CAN initial request
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_INITIAL_REQUEST_LEAVE
  *         @arg @ref MD_CAN_INITIAL_REQUEST_ENTER
  * @retval None
  */
__STATIC_INLINE void md_can_set_init_request(CAN_TypeDef *CANx, md_can_initial_request_list_t value)
{
    MODIFY_REG(CANx->CON, CAN_CON_INIREQ, value << CAN_CON_INIREQ_POS);
}

/**
  * @brief  Get CAN initial request
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_initial_request_list_t md_can_get_init_request(CAN_TypeDef *CANx)
{
    return (md_can_initial_request_list_t)(READ_BIT(CANx->CON, CAN_CON_INIREQ) >> CAN_CON_INIREQ_POS);
}
/**
  * @}
  */


/** @defgroup STAT
  * @{
  */
/**
  * @brief  Get CAN STAT
  * @note   All bits are read-only
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_stat(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->STAT));
}

/**
  * @brief  Get CAN STAT RX
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_rx(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_RX) == (CAN_STAT_RX));
}

/**
  * @brief  Get CAN STAT PRESMP
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_presmp(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_PRESMP) == (CAN_STAT_PRESMP));
}

/**
  * @brief  Get CAN STAT RXSTAT
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_rxstat(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_RXSTAT) == (CAN_STAT_RXSTAT));
}

/**
  * @brief  Get CAN STAT TXSTAT
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_txstat(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_TXSTAT) == (CAN_STAT_TXSTAT));
}

/**
  * @brief  Get CAN STAT SLPIF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_slpif(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_SLPIF) == (CAN_STAT_SLPIF));
}

/**
  * @brief  Get CAN STAT WKIF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_wkif(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_WKIF) == (CAN_STAT_WKIF));
}

/**
  * @brief  Get CAN STAT ERRIF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_errif(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_ERRIF) == (CAN_STAT_ERRIF));
}

/**
  * @brief  Get CAN STAT SLPSTAT
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_slpstat(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_SLPSTAT) == (CAN_STAT_SLPSTAT));
}

/**
  * @brief  Get CAN STAT INISTAT
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_inistat(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->STAT, CAN_STAT_INISTAT) == (CAN_STAT_INISTAT));
}
/**
  * @}
  */

/** @defgroup IFC
  * @{
  */
/**
  * @brief  Set CAN IFC
  * @note   None
  * @param  CANx CAN Instance
  * @param  ifc
  * @retval None
  */
__STATIC_INLINE void md_can_set_ifc(CAN_TypeDef *CANx, uint32_t ifc)
{
    WRITE_REG(CANx->IFC, ifc);
}

/**
  * @brief  Clear SLPIFC interrupt flag
  * @note   None
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_clear_it_slpifc(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IFC, CAN_IFC_SLPIFC);
}

/**
  * @brief  Clear WKIFC interrupt flag
  * @note   None
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_clear_it_wkifc(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IFC, CAN_IFC_WKIFC);
}

/**
  * @brief  Clear ERRIFC interrupt flag
  * @note   None
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_clear_it_errifc(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IFC, CAN_IFC_ERRIFC);
}
/**
  * @}
  */

/** @defgroup TXSTAT
  * @{
  */

/**
  * @brief  Set CAN TXSTAT
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txstat(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXSTAT, value);
}

/**
  * @brief  Get CAN TXSTAT
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txstat(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXSTAT));
}

/**
  * @brief  Get CAN TXSTAT TXM2LPF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_txm2lpf(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_TXM2LPF) == (CAN_TXSTAT_TXM2LPF));
}

/**
  * @brief  Get CAN TXSTAT TXM1LPF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_txm1lpf(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_TXM1LPF) == (CAN_TXSTAT_TXM1LPF));
}

/**
  * @brief  Get CAN TXSTAT TXM0LPF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_txm0lpf(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_TXM0LPF) == (CAN_TXSTAT_TXM0LPF));
}

/**
  * @brief  Get CAN TXSTAT TXM2EF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_txm2ef(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_TXM2EF) == (CAN_TXSTAT_TXM2EF));
}

/**
  * @brief  Get CAN TXSTAT TXM1EF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_txm1ef(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_TXM1EF) == (CAN_TXSTAT_TXM1EF));
}

/**
  * @brief  Get CAN TXSTAT TXM0EF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_txm0ef(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_TXM0EF) == (CAN_TXSTAT_TXM0EF));
}

/**
  * @brief  Get CAN TXSTAT CODE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_next_free_mailbox(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXSTAT, CAN_TXSTAT_CODE) >> CAN_TXSTAT_CODE_POSS);
}

/**
  * @brief  Set CAN TXSTAT M2STPREQ
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_MAILBOX_STOP_REQUEST_CANCEL
  *         @arg @ref MD_CAN_MAILBOX_STOP_REQUEST
  * @retval None
  */
__STATIC_INLINE void md_can_set_mailbox2_stop_request(CAN_TypeDef *CANx, md_can_mailbox_stop_request_list_t value)
{
    MODIFY_REG(CANx->TXSTAT, CAN_TXSTAT_M2STPREQ, value << CAN_TXSTAT_M2STPREQ_POS);
}

/**
  * @brief  Get CAN TXSTAT M2STPREQ
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_mailbox_stop_request_list_t md_can_get_mailbox2_stop_request(CAN_TypeDef *CANx)
{
    return (md_can_mailbox_stop_request_list_t)(READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M2STPREQ) >> CAN_TXSTAT_M2STPREQ_POS);
}

/**
  * @brief  Get CAN TXSTAT M2TXERR
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox2_transmit_error(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M2TXERR) == (CAN_TXSTAT_M2TXERR));
}

/**
  * @brief  Get CAN TXSTAT M2ARBLST
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox2_arbitration_lost(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M2ARBLST) == (CAN_TXSTAT_M2ARBLST));
}

/**
  * @brief  Get CAN TXSTAT M2TXC
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox2_transmit_complete(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M2TXC) == (CAN_TXSTAT_M2TXC));
}

/**
  * @brief  Get CAN TXSTAT M2REQC
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox2_request_complete(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M2REQC) == (CAN_TXSTAT_M2REQC));
}

/**
  * @brief  Set CAN TXSTAT M1STPREQ
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_MAILBOX_STOP_REQUEST_CANCEL
  *         @arg @ref MD_CAN_MAILBOX_STOP_REQUEST
  * @retval None
  */
__STATIC_INLINE void md_can_set_mailbox1_stop_request(CAN_TypeDef *CANx, md_can_mailbox_stop_request_list_t value)
{
    MODIFY_REG(CANx->TXSTAT, CAN_TXSTAT_M1STPREQ, value << CAN_TXSTAT_M1STPREQ_POS);
}

/**
  * @brief  Get CAN TXSTAT M1STPREQ
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_mailbox_stop_request_list_t md_can_get_mailbox1_stop_request(CAN_TypeDef *CANx)
{
    return (md_can_mailbox_stop_request_list_t)(READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M1STPREQ) >> CAN_TXSTAT_M1STPREQ_POS);
}

/**
  * @brief  Get CAN TXSTAT M1TXERR
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox1_transmit_error(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M1TXERR) == (CAN_TXSTAT_M1TXERR));
}

/**
  * @brief  Get CAN TXSTAT M1ARBLST
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox1_arbitration_lost(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M1ARBLST) == (CAN_TXSTAT_M1ARBLST));
}

/**
  * @brief  Get CAN TXSTAT M1TXC
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox1_transmit_complete(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M1TXC) == (CAN_TXSTAT_M1TXC));
}

/**
  * @brief  Get CAN TXSTAT M1REQC
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox1_request_complete(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M1REQC) == (CAN_TXSTAT_M1REQC));
}

/**
  * @brief  Set CAN TXSTAT M0STPREQ
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_MAILBOX_STOP_REQUEST_CANCEL
  *         @arg @ref MD_CAN_MAILBOX_STOP_REQUEST
  * @retval None
  */
__STATIC_INLINE void md_can_set_mailbox0_stop_request(CAN_TypeDef *CANx, md_can_mailbox_stop_request_list_t value)
{
    MODIFY_REG(CANx->TXSTAT, CAN_TXSTAT_M0STPREQ, value << CAN_TXSTAT_M0STPREQ_POS);
}

/**
  * @brief  Get CAN TXSTAT M0STPREQ
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_mailbox_stop_request_list_t md_can_get_mailbox0_stop_request(CAN_TypeDef *CANx)
{
    return (md_can_mailbox_stop_request_list_t)(READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M0STPREQ) >> CAN_TXSTAT_M0STPREQ_POS);
}

/**
  * @brief  Get CAN TXSTAT M0TXERR
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox0_transmit_error(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M0TXERR) == (CAN_TXSTAT_M0TXERR));
}

/**
  * @brief  Get CAN TXSTAT M0ARBLST
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox0_arbitration_lost(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M0ARBLST) == (CAN_TXSTAT_M0ARBLST));
}

/**
  * @brief  Get CAN TXSTAT M0TXC
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox0_transmit_complete(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M0TXC) == (CAN_TXSTAT_M0TXC));
}

/**
  * @brief  Get CAN TXSTAT M0REQC
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_mailbox0_request_complete(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXSTAT, CAN_TXSTAT_M0REQC) == (CAN_TXSTAT_M0REQC));
}
/**
  * @}
  */

/** @defgroup TXSTATC
  * @{
  */

/**
  * @brief  Set CAN TXSTATC
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txstatc(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXSTATC, value);
}

/**
* @brief  Clear CAN TXSTATC M2TXERR
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox2_transmit_error(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M2TXERR);
}

/**
* @brief  Clear CAN TXSTATC M2ARBLST
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox2_arbitration_lost(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M2ARBLST);
}

/**
* @brief  Clear CAN TXSTATC M2TXC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox2_transmit_complete(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M2TXC);
}

/**
* @brief  Clear CAN TXSTATC M2REQC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox2_request_complete(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M2REQC);
}

/**
* @brief  Clear CAN TXSTATC M1TXERR
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox1_transmit_error(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M1TXERR);
}

/**
* @brief  Clear CAN TXSTATC M1ARBLST
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox1_arbitration_lost(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M1ARBLST);
}

/**
* @brief  Clear CAN TXSTATC M1TXC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox1_transmit_complete(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M1TXC);
}

/**
* @brief  Clear CAN TXSTATC M1REQC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox1_request_complete(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M1REQC);
}

/**
* @brief  Clear CAN TXSTATC M0TXERR
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox0_transmit_error(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M0TXERR);
}

/**
* @brief  Clear CAN TXSTATC M0ARBLST
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox0_arbitration_lost(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M0ARBLST);
}

/**
* @brief  Clear CAN TXSTATC M0TXC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox0_transmit_complete(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M0TXC);
}

/**
* @brief  Clear CAN TXSTATC M0REQC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_mailbox0_request_complete(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXSTATC, CAN_TXSTATC_M0REQC);
}
/**
  * @}
  */

/** @defgroup RXF0
  * @{
  */
/**
  * @brief  Set CAN RXF0
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_rxfifo0_rxf0(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->RXF0, value);
}

/**
  * @brief  Get CAN RXF0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_rxf0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF0));
}

/**
  * @brief  Set CAN RXF0 FREE
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_RXFIFO_KEEP_MESSAGE
  *         @arg @ref MD_CAN_RXFIFO_FREE_MESSAGE
  * @retval None
  */
__STATIC_INLINE void md_can_set_rxfifo0_free(CAN_TypeDef *CANx, md_can_rxfifo_release_list_t value)
{
    MODIFY_REG(CANx->RXF0, CAN_RXF0_FREE, value << CAN_RXF0_FREE_POS);
}

/**
  * @brief  Get CAN RXF0 FREE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_rxfifo_release_list_t md_can_get_rxfifo0_free(CAN_TypeDef *CANx)
{
    return (md_can_rxfifo_release_list_t)(READ_BIT(CANx->RXF0, CAN_RXF0_FREE) >> CAN_RXF0_FREE_POS);
}

/**
  * @brief  Get CAN RXF0 OVR
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_rxfifo0_overflow(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->RXF0, CAN_RXF0_OVR) == (CAN_RXF0_OVR));
}

/**
  * @brief  Get CAN RXF0 FULL
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_rxfifo0_full(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->RXF0, CAN_RXF0_FULL) == (CAN_RXF0_FULL));
}

/**
  * @brief  Get CAN RXF0 PEND
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_pending_count(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0, CAN_RXF0_PEND) >> CAN_RXF0_PEND_POSS);
}
/**
  * @}
  */

/** @defgroup RXF0C
  * @{
  */
/**
  * @brief  Set CAN RXF0C
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_rxf0c(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->RXF0C, value);
}
/**
* @brief  Clear CAN RXF0C OVRC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_rxfifo0_overflow(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->RXF0C, CAN_RXF0C_OVRC);
}
/**
* @brief  Clear CAN RXF0C FULLC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_rxfifo0_full(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->RXF0C, CAN_RXF0C_FULLC);
}
/**
  * @}
  */
/** @defgroup RXF1
  * @{
  */
/**
  * @brief  Set CAN RXF1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_rxfifo1_rxf1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->RXF1, value);
}

/**
  * @brief  Get CAN RXF1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_rxf1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF1));
}

/**
  * @brief  Set CAN RXF1 FREE
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_RXFIFO_KEEP_MESSAGE
  *         @arg @ref MD_CAN_RXFIFO_FREE_MESSAGE
  * @retval None
  */
__STATIC_INLINE void md_can_set_rxfifo1_free(CAN_TypeDef *CANx, md_can_rxfifo_release_list_t value)
{
    MODIFY_REG(CANx->RXF1, CAN_RXF1_FREE, value << CAN_RXF1_FREE_POS);
}

/**
  * @brief  Get CAN RXF1 FREE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_rxfifo_release_list_t md_can_get_rxfifo1_free(CAN_TypeDef *CANx)
{
    return (md_can_rxfifo_release_list_t)(READ_BIT(CANx->RXF1, CAN_RXF1_FREE) >> CAN_RXF1_FREE_POS);
}

/**
  * @brief  Get CAN RXF1 OVR
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_rxfifo1_overflow(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->RXF1, CAN_RXF1_OVR) == (CAN_RXF1_OVR));
}

/**
  * @brief  Get CAN RXF1 FULL
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_rxfifo1_full(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->RXF1, CAN_RXF1_FULL) == (CAN_RXF1_FULL));
}

/**
  * @brief  Get CAN RXF1 PEND
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_pending_count(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1, CAN_RXF1_PEND) >> CAN_RXF1_PEND_POSS);
}
/**
  * @}
  */
/** @defgroup RXF1C
  * @{
  */

/**
  * @brief  Set CAN RXF1C
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_rxf1c(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->RXF1C, value);
}
/**
* @brief  Clear CAN RXF1C OVRC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_rxfifo1_overflow(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->RXF1C, CAN_RXF1C_OVRC);
}
/**
* @brief  Clear CAN RXF0C FULLC
* @param  CANx CAN Instance
* @retval None.
*/
__STATIC_INLINE void md_can_clear_flag_rxfifo1_full(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->RXF1C, CAN_RXF1C_FULLC);
}

/**
  * @}
  */
/** @defgroup IE
  * @{
  */
/**
  * @brief  Set CAN IE
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_ie(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->IE, value);
}

/**
  * @brief  Get CAN IE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_ie(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->IE));
}

/**
  * @brief  CAN enable SLPIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_sleep(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_SLPIE);
}

/**
  * @brief  CAN disable SLPIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_sleep(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_SLPIE);
}

/**
  * @brief  CAN is active SLPIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_sleep(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_SLPIE) == (CAN_IE_SLPIE));
}

/**
  * @brief  CAN enable WKIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_wakeup(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_WKIE);
}

/**
  * @brief  CAN disable WKIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_wakeup(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_WKIE);
}

/**
  * @brief  CAN is active WKIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_wakeup(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_WKIE) == (CAN_IE_WKIE));
}

/**
  * @brief  CAN enable ERRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_error(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_ERRIE);
}

/**
  * @brief  CAN disable ERRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_error(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_ERRIE);
}

/**
  * @brief  CAN is active ERRIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_error(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_ERRIE) == (CAN_IE_ERRIE));
}

/**
  * @brief  CAN enable PRERRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_previous_error(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_PRERRIE);
}

/**
  * @brief  CAN disable PRERRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_previous_error(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_PRERRIE);
}

/**
  * @brief  CAN is active PRERRIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_previous_error(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_PRERRIE) == (CAN_IE_PRERRIE));
}

/**
  * @brief  CAN enable BOFFIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_bus_off(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_BOFFIE);
}

/**
  * @brief  CAN disable BOFFIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_bus_off(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_BOFFIE);
}

/**
  * @brief  CAN is active BOFFIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_bus_off(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_BOFFIE) == (CAN_IE_BOFFIE));
}

/**
  * @brief  CAN enable PERRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_passive_error(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_PERRIE);
}

/**
  * @brief  CAN disable PERRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_passive_error(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_PERRIE);
}

/**
  * @brief  CAN is active PERRIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_passive_error(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_PERRIE) == (CAN_IE_PERRIE));
}

/**
  * @brief  CAN enable WARNIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_warning(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_WARNIE);
}

/**
  * @brief  CAN disable WARNIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_warning(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_WARNIE);
}

/**
  * @brief  CAN is active WARNIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_warning(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_WARNIE) == (CAN_IE_WARNIE));
}

/**
  * @brief  CAN enable F1OVRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_fifo1_overflow(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_F1OVRIE);
}

/**
  * @brief  CAN disable F1OVRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_fifo1_overflow(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_F1OVRIE);
}

/**
  * @brief  CAN is active F1OVRIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_fifo1_overflow(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_F1OVRIE) == (CAN_IE_F1OVRIE));
}

/**
  * @brief  CAN enable F1FULIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_fifo1_full(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_F1FULIE);
}

/**
  * @brief  CAN disable F1FULIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_fifo1_full(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_F1FULIE);
}

/**
  * @brief  CAN is active F1FULIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_fifo1_full(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_F1FULIE) == (CAN_IE_F1FULIE));
}

/**
  * @brief  CAN enable F1PIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_fifo1_pending(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_F1PIE);
}

/**
  * @brief  CAN disable F1PIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_fifo1_pending(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_F1PIE);
}

/**
  * @brief  CAN is active F1PIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_fifo1_pending(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_F1PIE) == (CAN_IE_F1PIE));
}

/**
  * @brief  CAN enable F0OVRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_fifo0_overflow(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_F0OVRIE);
}

/**
  * @brief  CAN disable F0OVRIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_fifo0_overflow(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_F0OVRIE);
}

/**
  * @brief  CAN is active F0OVRIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_fifo0_overflow(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_F0OVRIE) == (CAN_IE_F0OVRIE));
}

/**
  * @brief  CAN enable F0FULIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_fifo0_full(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_F0FULIE);
}

/**
  * @brief  CAN disable F0FULIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_fifo0_full(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_F0FULIE);
}

/**
  * @brief  CAN is active F0FULIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_fifo0_full(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_F0FULIE) == (CAN_IE_F0FULIE));
}

/**
  * @brief  CAN enable F0PIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_fifo0_pending(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_F0PIE);
}

/**
  * @brief  CAN disable F0PIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_fifo0_pending(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_F0PIE);
}

/**
  * @brief  CAN is active F0PIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_fifo0_pending(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_F0PIE) == (CAN_IE_F0PIE));
}

/**
  * @brief  CAN enable TXMEIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_it_trasmit_mailbox_empty(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->IE, CAN_IE_TXMEIE);
}

/**
  * @brief  CAN disable TXMEIE
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_it_trasmit_mailbox_empty(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->IE, CAN_IE_TXMEIE);
}

/**
  * @brief  CAN is active TXMEIE
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_it_trasmit_mailbox_empty(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->IE, CAN_IE_TXMEIE) == (CAN_IE_TXMEIE));
}
/**
  * @}
  */

/** @defgroup ERRSTAT
  * @{
  */

/**
  * @brief  Set CAN ERRSTAT
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_errstat(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->ERRSTAT, value);
}

/**
  * @brief  Get CAN ERRSTAT
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_errstat(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->ERRSTAT));
}

/**
  * @brief  Get CAN RXERRC
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_receive_errorcode(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->ERRSTAT, CAN_ERRSTAT_RXERRC) >> CAN_ERRSTAT_RXERRC_POSS);
}

/**
  * @brief  Get CAN TXERRC
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_transmit_errorcode(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->ERRSTAT, CAN_ERRSTAT_TXERRC) >> CAN_ERRSTAT_TXERRC_POSS);
}

/**
  * @brief  Software clear CAN PRERRF
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_set_previous_error_software_clear(CAN_TypeDef *CANx)
{
    MODIFY_REG(CANx->ERRSTAT, CAN_ERRSTAT_PRERRF, MD_CAN_RESET_ERROR_RECORD << CAN_ERRSTAT_PRERRF_POSS);
}

/**
  * @brief  Get CAN PRERRF
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_previous_error(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->ERRSTAT, CAN_ERRSTAT_PRERRF) >> CAN_ERRSTAT_PRERRF_POSS);
}

/**
  * @brief  Set CAN BOFF
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_CLEAR_BUS_OFF_FLAG
  *         @arg @ref MD_CAN_SET_BUS_OFF_FLAG
  * @retval None
  */
__STATIC_INLINE void md_can_set_bus_off_in_debug_mode(CAN_TypeDef *CANx, md_can_bus_closed_flag_t value)
{
    MODIFY_REG(CANx->ERRSTAT, CAN_ERRSTAT_BOFF, value << CAN_ERRSTAT_BOFF_POS);
}

/**
  * @brief  Get CAN PERRF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_passive_error(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->ERRSTAT, CAN_ERRSTAT_PERRF) == (CAN_ERRSTAT_PERRF));
}

/**
  * @brief  Get CAN WARNF
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_active_flag_warning(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->ERRSTAT, CAN_ERRSTAT_WARNF) == (CAN_ERRSTAT_WARNF));
}
/**
  * @}
  */
/** @defgroup BTIME
  * @{
  */
/**
  * @brief  Set CAN BTIME
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_btime(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->BTIME, value);
}

/**
  * @brief  Get CAN BTIME
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_btime(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->BTIME));
}

/**
  * @brief  Set CAN SILENT
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_SILENT_NORMAL_MODE
  *         @arg @ref MD_CAN_SILENT_SILENCE_MODE
  * @retval None
  */
__STATIC_INLINE void md_can_set_silent(CAN_TypeDef *CANx, md_can_filter_silent_mode_list_t value)
{
    MODIFY_REG(CANx->BTIME, CAN_BTIME_SILENT, value << CAN_BTIME_SILENT_POS);
}

/**
  * @brief  Get CAN SILENT
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_filter_silent_mode_list_t md_can_get_silent(CAN_TypeDef *CANx)
{
    return (md_can_filter_silent_mode_list_t)(READ_BIT(CANx->BTIME, CAN_BTIME_SILENT) >> CAN_BTIME_SILENT_POS);
}

/**
  * @brief  CAN enable LOOP
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_loopmode(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->BTIME, CAN_BTIME_LOOP);
}

/**
  * @brief  CAN disable LOOP
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_loopmode(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->BTIME, CAN_BTIME_LOOP);
}

/**
  * @brief  CAN is enabled LOOP
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_loopmode(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->BTIME, CAN_BTIME_LOOP) == (CAN_BTIME_LOOP));
}

/**
  * @brief  Set CAN RESJW
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_resjw(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->BTIME, CAN_BTIME_RESJW, value << CAN_BTIME_RESJW_POSS);
}

/**
  * @brief  Get CAN RESJW
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_resjw(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->BTIME, CAN_BTIME_RESJW) >> CAN_BTIME_RESJW_POSS);
}

/**
  * @brief  Set CAN SEG2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_seg2(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->BTIME, CAN_BTIME_SEG2, value << CAN_BTIME_SEG2_POSS);
}

/**
  * @brief  Get CAN SEG2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_seg2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->BTIME, CAN_BTIME_SEG2) >> CAN_BTIME_SEG2_POSS);
}

/**
  * @brief  Set CAN SEG1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_seg1(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->BTIME, CAN_BTIME_SEG1, value << CAN_BTIME_SEG1_POSS);
}

/**
  * @brief  Get CAN SEG1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_seg1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->BTIME, CAN_BTIME_SEG1) >> CAN_BTIME_SEG1_POSS);
}

/**
  * @brief  Set CAN BPSC
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_bpsc(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->BTIME, CAN_BTIME_BPSC, value << CAN_BTIME_BPSC_POSS);
}

/**
  * @brief  Get CAN BPSC
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_bpsc(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->BTIME, CAN_BTIME_BPSC) >> CAN_BTIME_BPSC_POSS);
}
/**
  * @}
  */

/** @defgroup TXID0
  * @{
  */
/**
  * @brief  Set CAN TXID0
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txid0(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXID0, value);
}

/**
  * @brief  Get CAN TXID0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txid0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXID0));
}

/**
  * @brief  Set CAN standard ID
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_standardid(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID0, CAN_TXID0_STDID10_0_EXID28_18, value << CAN_TXID0_STDID10_0_EXID28_18_POSS);
    CLEAR_BIT(CANx->TXID0, CAN_TXID0_EXID17_0);
}

/**
  * @brief  Get CAN standard ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_standardid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID0, CAN_TXID0_STDID10_0_EXID28_18) >> CAN_TXID0_STDID10_0_EXID28_18_POSS);
}

/**
  * @brief  Set CAN expand ID
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_expandid(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID0, (CAN_TXID0_STDID10_0_EXID28_18 | CAN_TXID0_EXID17_0), value << CAN_TXID0_EXID17_0_POSS);
}

/**
  * @brief  Get CAN expand ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_expandid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID0, CAN_TXID0_STDID10_0_EXID28_18 | CAN_TXID0_EXID17_0) >> CAN_TXID0_EXID17_0_POSS);
}

/**
  * @brief  Set CAN IDE
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_ide(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID0, CAN_TXID0_IDE, value << CAN_TXID0_IDE_POS);
}

/**
  * @brief  Get CAN IDE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_ide(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID0, CAN_TXID0_IDE) >> CAN_TXID0_IDE_POS);
}

/**
  * @brief  Set CAN RTR
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_rtr(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID0, CAN_TXID0_RTR, value << CAN_TXID0_RTR_POS);
}

/**
  * @brief  Get CAN RTR
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_rtr(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID0, CAN_TXID0_RTR) >> CAN_TXID0_RTR_POS);
}

/**
  * @brief  CAN enable TXMREQ
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_txmailbox0_request(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXID0, CAN_TXID0_TXMREQ);
}

/**
  * @brief  CAN disable TXMREQ
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_txmailbox0_request(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->TXID0, CAN_TXID0_TXMREQ);
}

/**
  * @brief  CAN is enabled TXMREQ
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_txmailbox0_request(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXID0, CAN_TXID0_TXMREQ) == (CAN_TXID0_TXMREQ));
}
/**
  * @}
  */
/** @defgroup TXFCON0
  * @{
  */
/**
  * @brief  Set CAN TXFCON0
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_txfcon0(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXFCON0, value);
}

/**
  * @brief  Get CAN TXFCON0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_txfcon0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXFCON0));
}

/**
  * @brief  Set CAN STAMP
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_stamp(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON0, CAN_TXFCON0_STAMP, value << CAN_TXFCON0_STAMP_POSS);
}

/**
  * @brief  Get CAN STAMP
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_stamp(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON0, CAN_TXFCON0_STAMP) >> CAN_TXFCON0_STAMP_POSS);
}

/**
  * @brief  Set CAN TXGT
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_txgt(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON0, CAN_TXFCON0_TXGT, value << CAN_TXFCON0_TXGT_POS);
}

/**
  * @brief  Get CAN TXGT
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_txgt(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON0, CAN_TXFCON0_TXGT) >> CAN_TXFCON0_TXGT_POS);
}

/**
  * @brief  Set CAN DLEN
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_datalength(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON0, CAN_TXFCON0_DLEN, value << CAN_TXFCON0_DLEN_POSS);
}

/**
  * @brief  Get CAN DLEN
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_datalength(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON0, CAN_TXFCON0_DLEN) >> CAN_TXFCON0_DLEN_POSS);
}
/**
  * @}
  */
/** @defgroup TXDL0
  * @{
  */
/**
  * @brief  Set CAN TXDL0
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txdl0(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXDL0, value);
}

/**
  * @brief  Get CAN TXDL0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txdl0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXDL0));
}

/**
  * @brief  Set CAN BYTE3
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_byte3(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL0, CAN_TXDL0_BYTE3, value << CAN_TXDL0_BYTE3_POSS);
}

/**
  * @brief  Get CAN BYTE3
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_byte3(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL0, CAN_TXDL0_BYTE3) >> CAN_TXDL0_BYTE3_POSS);
}

/**
  * @brief  Set CAN BYTE2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_byte2(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL0, CAN_TXDL0_BYTE2, value << CAN_TXDL0_BYTE2_POSS);
}

/**
  * @brief  Get CAN BYTE2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_byte2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL0, CAN_TXDL0_BYTE2) >> CAN_TXDL0_BYTE2_POSS);
}

/**
  * @brief  Set CAN BYTE1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_byte1(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL0, CAN_TXDL0_BYTE1, value << CAN_TXDL0_BYTE1_POSS);
}

/**
  * @brief  Get CAN BYTE1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_byte1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL0, CAN_TXDL0_BYTE1) >> CAN_TXDL0_BYTE1_POSS);
}

/**
  * @brief  Set CAN BYTE0
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_byte0(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL0, CAN_TXDL0_BYTE0, value << CAN_TXDL0_BYTE0_POSS);
}

/**
  * @brief  Get CAN BYTE0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_byte0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL0, CAN_TXDL0_BYTE0) >> CAN_TXDL0_BYTE0_POSS);
}
/**
  * @}
  */
/** @defgroup TXDH0
  * @{
  */
/**
  * @brief  Set CAN TXDH0
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txdh0(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXDH0, value);
}

/**
  * @brief  Get CAN TXDH0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txdh0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXDH0));
}

/**
  * @brief  Set CAN BYTE7
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_byte7(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH0, CAN_TXDH0_BYTE7, value << CAN_TXDH0_BYTE7_POSS);
}

/**
  * @brief  Get CAN BYTE7
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_byte7(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH0, CAN_TXDH0_BYTE7) >> CAN_TXDH0_BYTE7_POSS);
}

/**
  * @brief  Set CAN BYTE6
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_byte6(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH0, CAN_TXDH0_BYTE6, value << CAN_TXDH0_BYTE6_POSS);
}

/**
  * @brief  Get CAN BYTE6
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_byte6(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH0, CAN_TXDH0_BYTE6) >> CAN_TXDH0_BYTE6_POSS);
}

/**
  * @brief  Set CAN BYTE5
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_byte5(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH0, CAN_TXDH0_BYTE5, value << CAN_TXDH0_BYTE5_POSS);
}

/**
  * @brief  Get CAN BYTE5
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_byte5(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH0, CAN_TXDH0_BYTE5) >> CAN_TXDH0_BYTE5_POSS);
}

/**
  * @brief  Set CAN BYTE4
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox0_byte4(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH0, CAN_TXDH0_BYTE4, value << CAN_TXDH0_BYTE4_POSS);
}

/**
  * @brief  Get CAN BYTE4
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox0_byte4(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH0, CAN_TXDH0_BYTE4) >> CAN_TXDH0_BYTE4_POSS);
}
/**
  * @}
  */
/** @defgroup TXID1
  * @{
  */
/**
  * @brief  Set CAN TXID1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txid1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXID1, value);
}

/**
  * @brief  Get CAN TXID1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txid1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXID1));
}

/**
  * @brief  Set CAN standard ID
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_standardid(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID1, CAN_TXID1_STDID10_0_EXID28_18, value << CAN_TXID1_STDID10_0_EXID28_18_POSS);
    CLEAR_BIT(CANx->TXID1, CAN_TXID1_EXID17_0);
}

/**
  * @brief  Get CAN standard ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_standardid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID1, CAN_TXID1_STDID10_0_EXID28_18) >> CAN_TXID1_STDID10_0_EXID28_18_POSS);
}

/**
  * @brief  Set CAN expand ID
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_expandid(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID1, (CAN_TXID1_STDID10_0_EXID28_18 | CAN_TXID1_EXID17_0), value << CAN_TXID1_EXID17_0_POSS);
}

/**
  * @brief  Get CAN expand ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_expandid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID1, CAN_TXID1_STDID10_0_EXID28_18 | CAN_TXID1_EXID17_0) >> CAN_TXID1_EXID17_0_POSS);
}

/**
  * @brief  Set CAN IDE
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_ide(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID1, CAN_TXID1_IDE, value << CAN_TXID1_IDE_POS);
}

/**
  * @brief  Get CAN IDE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_ide(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID1, CAN_TXID1_IDE) >> CAN_TXID1_IDE_POS);
}

/**
  * @brief  Set CAN RTR
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_rtr(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID1, CAN_TXID1_RTR, value << CAN_TXID1_RTR_POS);
}

/**
  * @brief  Get CAN RTR
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_rtr(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID1, CAN_TXID1_RTR) >> CAN_TXID1_RTR_POS);
}

/**
  * @brief  CAN enable TXMREQ
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_txmailbox1_request(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXID1, CAN_TXID1_TXMREQ);
}

/**
  * @brief  CAN disable TXMREQ
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_txmailbox1_request(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->TXID1, CAN_TXID1_TXMREQ);
}

/**
  * @brief  CAN is enabled TXMREQ
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_txmailbox1_request(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXID1, CAN_TXID1_TXMREQ) == (CAN_TXID1_TXMREQ));
}
/**
  * @}
  */
/** @defgroup TXFCON1
  * @{
  */
/**
  * @brief  Set CAN TXFCON1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txfcon1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXFCON1, value);
}

/**
  * @brief  Get CAN TXFCON1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txfcon1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXFCON1));
}

/**
  * @brief  Set CAN STAMP
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_stamp(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON1, CAN_TXFCON1_STAMP, value << CAN_TXFCON1_STAMP_POSS);
}

/**
  * @brief  Get CAN STAMP
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_stamp(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON1, CAN_TXFCON1_STAMP) >> CAN_TXFCON1_STAMP_POSS);
}

/**
  * @brief  Set CAN TXGT
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_txgt(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON1, CAN_TXFCON1_TXGT, value << CAN_TXFCON1_TXGT_POS);
}

/**
  * @brief  Get CAN TXGT
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_txgt(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON1, CAN_TXFCON1_TXGT) >> CAN_TXFCON1_TXGT_POS);
}

/**
  * @brief  Set CAN DLEN
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_datalength(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON1, CAN_TXFCON1_DLEN, value << CAN_TXFCON1_DLEN_POSS);
}

/**
  * @brief  Get CAN DLEN
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_datalength(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON1, CAN_TXFCON1_DLEN) >> CAN_TXFCON1_DLEN_POSS);
}
/**
  * @}
  */
/** @defgroup TXDL1
  * @{
  */
/**
  * @brief  Set CAN TXDL1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txdl1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXDL1, value);
}

/**
  * @brief  Get CAN TXDL1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txdl1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXDL1));
}

/**
  * @brief  Set CAN BYTE3
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_byte3(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL1, CAN_TXDL1_BYTE3, value << CAN_TXDL1_BYTE3_POSS);
}

/**
  * @brief  Get CAN BYTE3
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_byte3(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL1, CAN_TXDL1_BYTE3) >> CAN_TXDL1_BYTE3_POSS);
}

/**
  * @brief  Set CAN BYTE2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_byte2(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL1, CAN_TXDL1_BYTE2, value << CAN_TXDL1_BYTE2_POSS);
}

/**
  * @brief  Get CAN BYTE2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_byte2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL1, CAN_TXDL1_BYTE2) >> CAN_TXDL1_BYTE2_POSS);
}

/**
  * @brief  Set CAN BYTE1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_byte1(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL1, CAN_TXDL1_BYTE1, value << CAN_TXDL1_BYTE1_POSS);
}

/**
  * @brief  Get CAN BYTE1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_byte1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL1, CAN_TXDL1_BYTE1) >> CAN_TXDL1_BYTE1_POSS);
}

/**
  * @brief  Set CAN BYTE0
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_byte0(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL1, CAN_TXDL1_BYTE0, value << CAN_TXDL1_BYTE0_POSS);
}

/**
  * @brief  Get CAN BYTE0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_byte0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL1, CAN_TXDL1_BYTE0) >> CAN_TXDL1_BYTE0_POSS);
}
/**
  * @}
  */
/** @defgroup TXDH1
  * @{
  */
/**
  * @brief  Set CAN TXDH1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txdh1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXDH1, value);
}

/**
  * @brief  Get CAN TXDH1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txdh1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXDH1));
}

/**
  * @brief  Set CAN BYTE7
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_byte7(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH1, CAN_TXDH1_BYTE7, value << CAN_TXDH1_BYTE7_POSS);
}

/**
  * @brief  Get CAN BYTE7
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_byte7(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH1, CAN_TXDH1_BYTE7) >> CAN_TXDH1_BYTE7_POSS);
}

/**
  * @brief  Set CAN BYTE6
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_byte6(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH1, CAN_TXDH1_BYTE6, value << CAN_TXDH1_BYTE6_POSS);
}

/**
  * @brief  Get CAN BYTE6
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_byte6(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH1, CAN_TXDH1_BYTE6) >> CAN_TXDH1_BYTE6_POSS);
}

/**
  * @brief  Set CAN BYTE5
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_byte5(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH1, CAN_TXDH1_BYTE5, value << CAN_TXDH1_BYTE5_POSS);
}

/**
  * @brief  Get CAN BYTE5
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_byte5(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH1, CAN_TXDH1_BYTE5) >> CAN_TXDH1_BYTE5_POSS);
}

/**
  * @brief  Set CAN BYTE4
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox1_byte4(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH1, CAN_TXDH1_BYTE4, value << CAN_TXDH1_BYTE4_POSS);
}

/**
  * @brief  Get CAN BYTE4
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox1_byte4(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH1, CAN_TXDH1_BYTE4) >> CAN_TXDH1_BYTE4_POSS);
}
/**
  * @}
  */
/** @defgroup TXID2
  * @{
  */
/**
  * @brief  Set CAN TXID2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txid2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXID2, value);
}

/**
  * @brief  Get CAN TXID2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txid2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXID2));
}

/**
  * @brief  Set CAN standard ID
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_standardid(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID2, CAN_TXID2_STDID10_0_EXID28_18, value << CAN_TXID2_STDID10_0_EXID28_18_POSS);
    CLEAR_BIT(CANx->TXID2, CAN_TXID2_EXID17_0);
}

/**
  * @brief  Get CAN standard ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_standardid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID2, CAN_TXID2_STDID10_0_EXID28_18) >> CAN_TXID2_STDID10_0_EXID28_18_POSS);
}

/**
  * @brief  Set CAN expand ID
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_expandid(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID2, (CAN_TXID2_STDID10_0_EXID28_18 | CAN_TXID2_EXID17_0), value << CAN_TXID2_EXID17_0_POSS);
}

/**
  * @brief  Get CAN expand ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_expandid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID2, CAN_TXID2_STDID10_0_EXID28_18 | CAN_TXID2_EXID17_0) >> CAN_TXID2_EXID17_0_POSS);
}

/**
  * @brief  Set CAN IDE
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_ide(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID2, CAN_TXID2_IDE, value << CAN_TXID2_IDE_POS);
}

/**
  * @brief  Get CAN IDE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_ide(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID2, CAN_TXID2_IDE) >> CAN_TXID2_IDE_POS);
}

/**
  * @brief  Set CAN RTR
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_rtr(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXID2, CAN_TXID2_RTR, value << CAN_TXID2_RTR_POS);
}

/**
  * @brief  Get CAN RTR
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_rtr(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXID2, CAN_TXID2_RTR) >> CAN_TXID2_RTR_POS);
}

/**
  * @brief  CAN enable TXMREQ
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_enable_txmailbox2_request(CAN_TypeDef *CANx)
{
    SET_BIT(CANx->TXID2, CAN_TXID2_TXMREQ);
}

/**
  * @brief  CAN disable TXMREQ
  * @param  CANx CAN Instance
  * @retval None
  */
__STATIC_INLINE void md_can_disable_txmailbox2_request(CAN_TypeDef *CANx)
{
    CLEAR_BIT(CANx->TXID2, CAN_TXID2_TXMREQ);
}

/**
  * @brief  CAN is enabled TXMREQ
  * @param  CANx CAN Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_can_is_enabled_txmailbox2_request(CAN_TypeDef *CANx)
{
    return (READ_BIT(CANx->TXID2, CAN_TXID2_TXMREQ) == (CAN_TXID2_TXMREQ));
}
/**
  * @}
  */
/** @defgroup TXFCON2
  * @{
  */
/**
  * @brief  Set CAN TXFCON2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txfcon2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXFCON2, value);
}

/**
  * @brief  Get CAN TXFCON2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txfcon2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXFCON2));
}

/**
  * @brief  Set CAN STAMP
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_stamp(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON2, CAN_TXFCON2_STAMP, value << CAN_TXFCON2_STAMP_POSS);
}

/**
  * @brief  Get CAN STAMP
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_stamp(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON2, CAN_TXFCON2_STAMP) >> CAN_TXFCON2_STAMP_POSS);
}

/**
  * @brief  Set CAN TXGT
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_txgt(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON2, CAN_TXFCON2_TXGT, value << CAN_TXFCON2_TXGT_POS);
}

/**
  * @brief  Get CAN TXGT
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_txgt(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON2, CAN_TXFCON2_TXGT) >> CAN_TXFCON2_TXGT_POS);
}

/**
  * @brief  Set CAN DLEN
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_datalength(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXFCON2, CAN_TXFCON2_DLEN, value << CAN_TXFCON2_DLEN_POSS);
}

/**
  * @brief  Get CAN DLEN
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_datalength(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXFCON2, CAN_TXFCON2_DLEN) >> CAN_TXFCON2_DLEN_POSS);
}
/**
  * @}
  */
/** @defgroup TXDL2
  * @{
  */
/**
  * @brief  Set CAN TXDL2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txdl2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXDL2, value);
}

/**
  * @brief  Get CAN TXDL2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txdl2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXDL2));
}

/**
  * @brief  Set CAN BYTE3
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_byte3(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL2, CAN_TXDL2_BYTE3, value << CAN_TXDL2_BYTE3_POSS);
}

/**
  * @brief  Get CAN BYTE3
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_byte3(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL2, CAN_TXDL2_BYTE3) >> CAN_TXDL2_BYTE3_POSS);
}

/**
  * @brief  Set CAN BYTE2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_byte2(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL2, CAN_TXDL2_BYTE2, value << CAN_TXDL2_BYTE2_POSS);
}

/**
  * @brief  Get CAN BYTE2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_byte2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL2, CAN_TXDL2_BYTE2) >> CAN_TXDL2_BYTE2_POSS);
}

/**
  * @brief  Set CAN BYTE1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_byte1(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL2, CAN_TXDL2_BYTE1, value << CAN_TXDL2_BYTE1_POSS);
}

/**
  * @brief  Get CAN BYTE1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_byte1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL2, CAN_TXDL2_BYTE1) >> CAN_TXDL2_BYTE1_POSS);
}

/**
  * @brief  Set CAN BYTE0
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_byte0(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDL2, CAN_TXDL2_BYTE0, value << CAN_TXDL2_BYTE0_POSS);
}

/**
  * @brief  Get CAN BYTE0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_byte0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDL2, CAN_TXDL2_BYTE0) >> CAN_TXDL2_BYTE0_POSS);
}
/**
  * @}
  */
/** @defgroup TXDH2
  * @{
  */
/**
  * @brief  Set CAN TXDH2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txdh2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->TXDH2, value);
}

/**
  * @brief  Get CAN TXDH2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txdh2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->TXDH2));
}

/**
  * @brief  Set CAN BYTE7
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_byte7(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH2, CAN_TXDH2_BYTE7, value << CAN_TXDH2_BYTE7_POSS);
}

/**
  * @brief  Get CAN BYTE7
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_byte7(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH2, CAN_TXDH2_BYTE7) >> CAN_TXDH2_BYTE7_POSS);
}

/**
  * @brief  Set CAN BYTE6
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_byte6(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH2, CAN_TXDH2_BYTE6, value << CAN_TXDH2_BYTE6_POSS);
}

/**
  * @brief  Get CAN BYTE6
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_byte6(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH2, CAN_TXDH2_BYTE6) >> CAN_TXDH2_BYTE6_POSS);
}

/**
  * @brief  Set CAN BYTE5
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_byte5(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH2, CAN_TXDH2_BYTE5, value << CAN_TXDH2_BYTE5_POSS);
}

/**
  * @brief  Get CAN BYTE5
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_byte5(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH2, CAN_TXDH2_BYTE5) >> CAN_TXDH2_BYTE5_POSS);
}

/**
  * @brief  Set CAN BYTE4
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_txmailbox2_byte4(CAN_TypeDef *CANx, uint32_t value)
{
    MODIFY_REG(CANx->TXDH2, CAN_TXDH2_BYTE4, value << CAN_TXDH2_BYTE4_POSS);
}

/**
  * @brief  Get CAN BYTE4
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_txmailbox2_byte4(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->TXDH2, CAN_TXDH2_BYTE4) >> CAN_TXDH2_BYTE4_POSS);
}
/**
  * @}
  */
/** @defgroup RXF0ID
  * @{
  */
/**
  * @brief  Get CAN RXF0ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxf0id(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF0ID));
}

/**
  * @brief  Get CAN rxfifo0 standard ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_standardid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0ID, CAN_RXF0ID_STDID10_0_EXID28_18) >> CAN_RXF0ID_STDID10_0_EXID28_18_POSS);
}

/**
  * @brief  Get CAN rxfifo0 expand ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_expandid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0ID, CAN_RXF0ID_STDID10_0_EXID28_18 | CAN_RXF0ID_EXID17_0) >> CAN_RXF0ID_EXID17_0_POSS);
}

/**
  * @brief  Get CAN rxfifo0 IDE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_ide(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0ID, CAN_RXF0ID_IDE) >> CAN_RXF0ID_IDE_POS);
}

/**
  * @brief  Get CAN rxfifo0 RTR
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_rtr(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0ID, CAN_RXF0ID_RTR) >> CAN_RXF0ID_RTR_POS);
}
/**
  * @}
  */
/** @defgroup RXF0INF
  * @{
  */
/**
  * @brief  Get CAN RXF0INF
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxf0inf(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF0INF));
}

/**
  * @brief  Get CAN rxfifo0 STAMP
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_stamp(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0INF, CAN_RXF0INF_STAMP) >> CAN_RXF0INF_STAMP_POSS);
}

/**
  * @brief  Get CAN rxfifo0 FLTIDX
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_fltidx(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0INF, CAN_RXF0INF_FLTIDX) >> CAN_RXF0INF_FLTIDX_POSS);
}

/**
  * @brief  Get CAN rxfifo0 DLEN
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_datalength(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0INF, CAN_RXF0INF_DLEN) >> CAN_RXF0INF_DLEN_POSS);
}
/**
  * @}
  */
/** @defgroup RXF0DL
  * @{
  */
/**
  * @brief  Get CAN RXF0DL
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxf0dl(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF0DL));
}

/**
  * @brief  Get CAN rxfifo0 BYTE3
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_byte3(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0DL, CAN_RXF0DL_BYTE3) >> CAN_RXF0DL_BYTE3_POSS);
}

/**
  * @brief  Get CAN rxfifo0 BYTE2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_byte2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0DL, CAN_RXF0DL_BYTE2) >> CAN_RXF0DL_BYTE2_POSS);
}

/**
  * @brief  Get CAN rxfifo0 BYTE1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_byte1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0DL, CAN_RXF0DL_BYTE1) >> CAN_RXF0DL_BYTE1_POSS);
}

/**
  * @brief  Get CAN rxfifo0 BYTE0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_byte0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0DL, CAN_RXF0DL_BYTE0) >> CAN_RXF0DL_BYTE0_POSS);
}
/**
  * @}
  */
/** @defgroup RXF0DH
  * @{
  */
/**
  * @brief  Get CAN RXF0DH
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxf0dh(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF0DH));
}

/**
  * @brief  Get CAN rxfifo0 BYTE7
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_byte7(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0DH, CAN_RXF0DH_BYTE7) >> CAN_RXF0DH_BYTE7_POSS);
}

/**
  * @brief  Get CAN rxfifo0 BYTE6
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_byte6(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0DH, CAN_RXF0DH_BYTE6) >> CAN_RXF0DH_BYTE6_POSS);
}

/**
  * @brief  Get CAN rxfifo0 BYTE5
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_byte5(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0DH, CAN_RXF0DH_BYTE5) >> CAN_RXF0DH_BYTE5_POSS);
}

/**
  * @brief  Get CAN rxfifo0 BYTE4
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo0_byte4(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF0DH, CAN_RXF0DH_BYTE4) >> CAN_RXF0DH_BYTE4_POSS);
}
/**
  * @}
  */
/** @defgroup RXF1ID
  * @{
  */
/**
  * @brief  Get CAN RXF1ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxf1id(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF1ID));
}

/**
  * @brief  Get CAN rxfifo1 standard ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_standardid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1ID, CAN_RXF1ID_STDID10_0_EXID28_18) >> CAN_RXF1ID_STDID10_0_EXID28_18_POSS);
}

/**
  * @brief  Get CAN rxfifo1 expand ID
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_expandid(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1ID, CAN_RXF1ID_STDID10_0_EXID28_18 | CAN_RXF1ID_EXID17_0) >> CAN_RXF1ID_EXID17_0_POSS);
}

/**
  * @brief  Get CAN rxfifo1 IDE
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_ide(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1ID, CAN_RXF1ID_IDE) >> CAN_RXF1ID_IDE_POS);
}

/**
  * @brief  Get CAN rxfifo1 RTR
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_rtr(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1ID, CAN_RXF1ID_RTR) >> CAN_RXF1ID_RTR_POS);
}
/**
  * @}
  */
/** @defgroup RXF1INF
  * @{
  */
/**
  * @brief  Get CAN RXF1INF
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxf1inf(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF1INF));
}

/**
  * @brief  Get CAN rxfifo1 STAMP
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_stamp(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1INF, CAN_RXF1INF_STAMP) >> CAN_RXF1INF_STAMP_POSS);
}

/**
  * @brief  Get CAN rxfifo1 FLTIDX
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_fltidx(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1INF, CAN_RXF1INF_FLTIDX) >> CAN_RXF1INF_FLTIDX_POSS);
}

/**
  * @brief  Get CAN rxfifo1 DLEN
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_datalength(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1INF, CAN_RXF1INF_DLEN) >> CAN_RXF1INF_DLEN_POSS);
}
/**
  * @}
  */
/** @defgroup RXF1DL
  * @{
  */
/**
  * @brief  Get CAN RXF1DL
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxf1dl(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF1DL));
}

/**
  * @brief  Get CAN rxfifo1 BYTE3
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_byte3(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1DL, CAN_RXF1DL_BYTE3) >> CAN_RXF1DL_BYTE3_POSS);
}

/**
  * @brief  Get CAN rxfifo1 BYTE2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_byte2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1DL, CAN_RXF1DL_BYTE2) >> CAN_RXF1DL_BYTE2_POSS);
}

/**
  * @brief  Get CAN rxfifo1 BYTE1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_byte1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1DL, CAN_RXF1DL_BYTE1) >> CAN_RXF1DL_BYTE1_POSS);
}

/**
  * @brief  Get CAN rxfifo1 BYTE0
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_byte0(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1DL, CAN_RXF1DL_BYTE0) >> CAN_RXF1DL_BYTE0_POSS);
}
/**
  * @}
  */
/** @defgroup RXF1DH
  * @{
  */
/**
  * @brief  Get CAN RXF1DH
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxf1dh(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->RXF1DH));
}

/**
  * @brief  Get CAN rxfifo1 BYTE7
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_byte7(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1DH, CAN_RXF1DH_BYTE7) >> CAN_RXF1DH_BYTE7_POSS);
}

/**
  * @brief  Get CAN rxfifo1 BYTE6
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_byte6(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1DH, CAN_RXF1DH_BYTE6) >> CAN_RXF1DH_BYTE6_POSS);
}

/**
  * @brief  Get CAN rxfifo1 BYTE5
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_byte5(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1DH, CAN_RXF1DH_BYTE5) >> CAN_RXF1DH_BYTE5_POSS);
}

/**
  * @brief  Get CAN rxfifo1 BYTE4
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_rxfifo1_byte4(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_BIT(CANx->RXF1DH, CAN_RXF1DH_BYTE4) >> CAN_RXF1DH_BYTE4_POSS);
}
/**
  * @}
  */
/** @defgroup FLTCON
  * @{
  */
/**
  * @brief  Set CAN FLTCON
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_fltcon(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLTCON, value);
}

/**
  * @brief  Get CAN FLTCON
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_fltcon(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLTCON));
}

/**
  * @brief  Set CAN FLTINI
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_FILTER_WORK_MODE
  *         @arg @ref MD_CAN_FILTER_INITIAL_MODE
  * @retval None
  */
__STATIC_INLINE void md_can_set_fltini(CAN_TypeDef *CANx, md_can_filter_initial_mode_list_t value)
{
    MODIFY_REG(CANx->FLTCON, CAN_FLTCON_FLTINI, value << CAN_FLTCON_FLTINI_POS);
}

/**
  * @brief  Get CAN FLTINI
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_filter_initial_mode_list_t md_can_get_fltini(CAN_TypeDef *CANx)
{
    return (md_can_filter_initial_mode_list_t)(READ_BIT(CANx->FLTCON, CAN_FLTCON_FLTINI) >> CAN_FLTCON_FLTINI_POS);
}
/**
  * @}
  */
/** @defgroup FLTM
  * @{
  */
/**
  * @brief  Set CAN FLTM
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_fltm(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLTM, value);
}

/**
  * @brief  Get CAN FLTM
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_fltm(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLTM));
}

/**
  * @brief  Set CAN MOD
  * @param  CANx CAN Instance
  * @param  a 32-bit filter_number
  *         @arg @ref MD_CAN_FILTER_0
  *         @arg @ref MD_CAN_FILTER_1
  *         @arg @ref MD_CAN_FILTER_2
  *         @arg @ref MD_CAN_FILTER_3
  *         @arg @ref MD_CAN_FILTER_4
  *         @arg @ref MD_CAN_FILTER_5
  *         @arg @ref MD_CAN_FILTER_6
  *         @arg @ref MD_CAN_FILTER_7
  *         @arg @ref MD_CAN_FILTER_8
  *         @arg @ref MD_CAN_FILTER_9
  *         @arg @ref MD_CAN_FILTER_10
  *         @arg @ref MD_CAN_FILTER_11
  *         @arg @ref MD_CAN_FILTER_12
  *         @arg @ref MD_CAN_FILTER_13
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_FILTER_MOD_MASK
  *         @arg @ref MD_CAN_FILTER_MOD_LIST_OF_IDENTIFIERS
  * @retval None
  */
__STATIC_INLINE void md_can_set_mod(CAN_TypeDef *CANx, md_can_filter_list_t filter_number, md_can_filter_mode_list_t value)
{
    MODIFY_REG(CANx->FLTM, ((1UL << filter_number) << CAN_FLTM_MODx_POSS), (value << filter_number) << CAN_FLTM_MODx_POSS);
}

/**
  * @brief  Get CAN MOD
  * @param  CANx CAN Instance
  * @param  a 32-bit filter_number
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_filter_mode_list_t md_can_get_mod(CAN_TypeDef *CANx, md_can_filter_list_t filter_number)
{
    return (md_can_filter_mode_list_t)((READ_BIT(CANx->FLTM, (1UL << filter_number) << CAN_FLTM_MODx_POSS) >> CAN_FLTM_MODx_POSS) >> filter_number);
}
/**
  * @}
  */

/** @defgroup FLTWS
  * @{
  */
/**
  * @brief  Set CAN FLTWS
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_fltws(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLTWS, value);
}

/**
  * @brief  Get CAN FLTWS
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_fltws(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLTWS));
}

/**
  * @brief  Set CAN SEL
  * @param  CANx CAN Instance
  * @param  a 32-bit filter_number
  *         @arg @ref MD_CAN_FILTER_0
  *         @arg @ref MD_CAN_FILTER_1
  *         @arg @ref MD_CAN_FILTER_2
  *         @arg @ref MD_CAN_FILTER_3
  *         @arg @ref MD_CAN_FILTER_4
  *         @arg @ref MD_CAN_FILTER_5
  *         @arg @ref MD_CAN_FILTER_6
  *         @arg @ref MD_CAN_FILTER_7
  *         @arg @ref MD_CAN_FILTER_8
  *         @arg @ref MD_CAN_FILTER_9
  *         @arg @ref MD_CAN_FILTER_10
  *         @arg @ref MD_CAN_FILTER_11
  *         @arg @ref MD_CAN_FILTER_12
  *         @arg @ref MD_CAN_FILTER_13
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_FILTER_SEL_16BIT
  *         @arg @ref MD_CAN_FILTER_SEL_32BIT
  * @retval None
  */
__STATIC_INLINE void md_can_set_sel(CAN_TypeDef *CANx, md_can_filter_list_t filter_number, md_can_filter_width_list_t value)
{
    MODIFY_REG(CANx->FLTWS, ((1UL << filter_number) << CAN_FLTWS_SELx_POSS), (value << filter_number) << CAN_FLTWS_SELx_POSS);
}

/**
  * @brief  Get CAN SEL
  * @param  CANx CAN Instance
  * @param  a 32-bit filter_number
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_filter_width_list_t md_can_get_sel(CAN_TypeDef *CANx, md_can_filter_list_t filter_number)
{
    return (md_can_filter_width_list_t)((READ_BIT(CANx->FLTWS, (1UL << filter_number) << CAN_FLTWS_SELx_POSS) >> CAN_FLTWS_SELx_POSS) >> filter_number);
}
/**
  * @}
  */
/** @defgroup FLTAS
  * @{
  */
/**
  * @brief  Set CAN FLTAS
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_fltas(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLTAS, value);
}

/**
  * @brief  Get CAN FLTAS
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_fltas(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLTAS));
}

/**
  * @brief  Set CAN ASSIGN
  * @param  CANx CAN Instance
  * @param  a 32-bit filter_number
  *         @arg @ref MD_CAN_FILTER_0
  *         @arg @ref MD_CAN_FILTER_1
  *         @arg @ref MD_CAN_FILTER_2
  *         @arg @ref MD_CAN_FILTER_3
  *         @arg @ref MD_CAN_FILTER_4
  *         @arg @ref MD_CAN_FILTER_5
  *         @arg @ref MD_CAN_FILTER_6
  *         @arg @ref MD_CAN_FILTER_7
  *         @arg @ref MD_CAN_FILTER_8
  *         @arg @ref MD_CAN_FILTER_9
  *         @arg @ref MD_CAN_FILTER_10
  *         @arg @ref MD_CAN_FILTER_11
  *         @arg @ref MD_CAN_FILTER_12
  *         @arg @ref MD_CAN_FILTER_13
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_FILTER_ASSIGN_RXFIFO0
  *         @arg @ref MD_CAN_FILTER_ASSIGN_RXFIFO1
  * @retval None
  */
__STATIC_INLINE void md_can_set_assign(CAN_TypeDef *CANx, md_can_filter_list_t filter_number, md_can_filter_fifo_allocation_t value)
{
    MODIFY_REG(CANx->FLTAS, ((1UL << filter_number) << CAN_FLTAS_ASSIGNx_POSS), (value << filter_number) << CAN_FLTAS_ASSIGNx_POSS);
}

/**
  * @brief  Get CAN ASSIGN
  * @param  CANx CAN Instance
  * @param  a 32-bit filter_number
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_filter_fifo_allocation_t md_can_get_assign(CAN_TypeDef *CANx, md_can_filter_list_t filter_number)
{
    return (md_can_filter_fifo_allocation_t)((READ_BIT(CANx->FLTAS, (1UL << filter_number) << CAN_FLTAS_ASSIGNx_POSS) >> CAN_FLTAS_ASSIGNx_POSS) >> filter_number);
}
/**
  * @}
  */
/** @defgroup FLTGO
  * @{
  */
/**
  * @brief  Set CAN FLTGO
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_fltgo(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLTGO, value);
}

/**
  * @brief  Get CAN FLTGO
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_fltgo(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLTGO));
}

/**
  * @brief  Set CAN GO
  * @param  CANx CAN Instance
  * @param  a 32-bit filter_number
  *         @arg @ref MD_CAN_FILTER_0
  *         @arg @ref MD_CAN_FILTER_1
  *         @arg @ref MD_CAN_FILTER_2
  *         @arg @ref MD_CAN_FILTER_3
  *         @arg @ref MD_CAN_FILTER_4
  *         @arg @ref MD_CAN_FILTER_5
  *         @arg @ref MD_CAN_FILTER_6
  *         @arg @ref MD_CAN_FILTER_7
  *         @arg @ref MD_CAN_FILTER_8
  *         @arg @ref MD_CAN_FILTER_9
  *         @arg @ref MD_CAN_FILTER_10
  *         @arg @ref MD_CAN_FILTER_11
  *         @arg @ref MD_CAN_FILTER_12
  *         @arg @ref MD_CAN_FILTER_13
  * @param  a 32-bit value
  *         @arg @ref MD_CAN_FILTER_DEACTIVATE
  *         @arg @ref MD_CAN_FILTER_ACTIVE
  * @retval None
  */
__STATIC_INLINE void md_can_set_go(CAN_TypeDef *CANx, md_can_filter_list_t filter_number, md_can_filter_active_action_t value)
{
    MODIFY_REG(CANx->FLTGO, ((1UL << filter_number) << CAN_FLTGO_GOx_POSS), (value << filter_number) << CAN_FLTGO_GOx_POSS);
}

/**
  * @brief  Get CAN GO
  * @param  CANx CAN Instance
  * @param  a 32-bit filter_number
  * @retval a 32-bit value
  */
__STATIC_INLINE md_can_filter_active_action_t md_can_get_go(CAN_TypeDef *CANx, md_can_filter_list_t filter_number)
{
    return (md_can_filter_active_action_t)((READ_BIT(CANx->FLTGO, (1UL << filter_number) << CAN_FLTGO_GOx_POSS) >> CAN_FLTGO_GOx_POSS) >> filter_number);
}
/**
  * @}
  */
/** @defgroup FLT0R1
  * @{
  */
/**
  * @brief  Set CAN FLT0R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter0_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT0R1, value);
}

/**
  * @brief  Set CAN FLT0R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter0_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT0R1));
}
/**
  * @}
  */
/** @defgroup FLT0R2
  * @{
  */
/**
  * @brief  Set CAN FLT0R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter0_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT0R2, value);
}

/**
  * @brief  Set CAN FLT0R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter0_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT0R2));
}
/**
  * @}
  */
/** @defgroup FLT1R1
  * @{
  */
/**
  * @brief  Set CAN FLT1R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter1_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT1R1, value);
}

/**
  * @brief  Set CAN FLT1R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter1_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT1R1));
}
/**
  * @}
  */
/** @defgroup FLT1R2
  * @{
  */
/**
  * @brief  Set CAN FLT1R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter1_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT1R2, value);
}

/**
  * @brief  Set CAN FLT1R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter1_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT1R2));
}
/**
  * @}
  */
/** @defgroup FLT2R1
  * @{
  */
/**
  * @brief  Set CAN FLT2R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter2_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT2R1, value);
}

/**
  * @brief  Set CAN FLT2R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter2_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT2R1));
}
/**
  * @}
  */
/** @defgroup FLT2R2
  * @{
  */
/**
  * @brief  Set CAN FLT2R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter2_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT2R2, value);
}

/**
  * @brief  Set CAN FLT2R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter2_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT2R2));
}
/**
  * @}
  */
/** @defgroup FLT3R1
  * @{
  */
/**
  * @brief  Set CAN FLT3R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter3_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT3R1, value);
}

/**
  * @brief  Set CAN FLT3R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter3_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT3R1));
}
/**
  * @}
  */
/** @defgroup FLT3R2
  * @{
  */
/**
  * @brief  Set CAN FLT3R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter3_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT3R2, value);
}

/**
  * @brief  Set CAN FLT3R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter3_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT3R2));
}
/**
  * @}
  */
/** @defgroup FLT4R1
  * @{
  */
/**
  * @brief  Set CAN FLT4R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter4_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT4R1, value);
}

/**
  * @brief  Set CAN FLT4R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter4_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT4R1));
}
/**
  * @}
  */
/** @defgroup FLT4R2
  * @{
  */
/**
  * @brief  Set CAN FLT4R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter4_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT4R2, value);
}

/**
  * @brief  Set CAN FLT4R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter4_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT4R2));
}
/**
  * @}
  */
/** @defgroup FLT5R1
  * @{
  */
/**
  * @brief  Set CAN FLT5R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter5_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT5R1, value);
}

/**
  * @brief  Set CAN FLT5R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter5_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT5R1));
}
/**
  * @}
  */
/** @defgroup FLT5R2
  * @{
  */
/**
  * @brief  Set CAN FLT5R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter5_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT5R2, value);
}

/**
  * @brief  Set CAN FLT5R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter5_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT5R2));
}
/**
  * @}
  */
/** @defgroup FLT6R1
  * @{
  */
/**
  * @brief  Set CAN FLT6R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter6_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT6R1, value);
}

/**
  * @brief  Set CAN FLT6R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter6_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT6R1));
}
/**
  * @}
  */
/** @defgroup FLT6R2
  * @{
  */
/**
  * @brief  Set CAN FLT6R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter6_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT6R2, value);
}

/**
  * @brief  Set CAN FLT6R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter6_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT6R2));
}
/**
  * @}
  */
/** @defgroup FLT7R1
  * @{
  */
/**
  * @brief  Set CAN FLT7R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter7_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT7R1, value);
}

/**
  * @brief  Set CAN FLT7R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter7_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT7R1));
}
/**
  * @}
  */
/** @defgroup FLT7R2
  * @{
  */
/**
  * @brief  Set CAN FLT7R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter7_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT7R2, value);
}

/**
  * @brief  Set CAN FLT7R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter7_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT7R2));
}
/**
  * @}
  */
/** @defgroup FLT8R1
  * @{
  */
/**
  * @brief  Set CAN FLT8R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter8_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT8R1, value);
}

/**
  * @brief  Set CAN FLT8R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter8_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT8R1));
}
/**
  * @}
  */
/** @defgroup FLT8R2
  * @{
  */
/**
  * @brief  Set CAN FLT8R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter8_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT8R2, value);
}

/**
  * @brief  Set CAN FLT8R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter8_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT8R2));
}
/**
  * @}
  */
/** @defgroup FLT9R1
  * @{
  */
/**
  * @brief  Set CAN FLT9R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter9_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT9R1, value);
}

/**
  * @brief  Set CAN FLT9R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter9_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT9R1));
}
/**
  * @}
  */
/** @defgroup FLT9R2
  * @{
  */
/**
  * @brief  Set CAN FLT9R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter9_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT9R2, value);
}

/**
  * @brief  Set CAN FLT9R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter9_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT9R2));
}
/**
  * @}
  */
/** @defgroup FLT10R1
  * @{
  */
/**
  * @brief  Set CAN FLT10R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter10_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT10R1, value);
}

/**
  * @brief  Set CAN FLT10R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter10_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT10R1));
}
/**
  * @}
  */
/** @defgroup FLT10R2
  * @{
  */
/**
  * @brief  Set CAN FLT10R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter10_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT10R2, value);
}

/**
  * @brief  Set CAN FLT10R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter10_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT10R2));
}
/**
  * @}
  */
/** @defgroup FLT11R1
  * @{
  */
/**
  * @brief  Set CAN FLT11R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter11_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT11R1, value);
}

/**
  * @brief  Set CAN FLT11R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter11_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT11R1));
}
/**
  * @}
  */
/** @defgroup FLT11R2
  * @{
  */
/**
  * @brief  Set CAN FLT11R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter11_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT11R2, value);
}

/**
  * @brief  Set CAN FLT11R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter11_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT11R2));
}
/**
  * @}
  */
/** @defgroup FLT12R1
  * @{
  */
/**
  * @brief  Set CAN FLT12R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter12_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT12R1, value);
}

/**
  * @brief  Set CAN FLT12R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter12_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT12R1));
}
/**
  * @}
  */
/** @defgroup FLT12R2
  * @{
  */
/**
  * @brief  Set CAN FLT12R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter12_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT12R2, value);
}

/**
  * @brief  Set CAN FLT12R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter12_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT12R2));
}
/**
  * @}
  */
/** @defgroup FLT13R1
  * @{
  */
/**
  * @brief  Set CAN FLT13R1
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter13_register1(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT13R1, value);
}

/**
  * @brief  Set CAN FLT13R1
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter13_register1(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT13R1));
}
/**
  * @}
  */
/** @defgroup FLT13R2
  * @{
  */
/**
  * @brief  Set CAN FLT13R2
  * @param  CANx CAN Instance
  * @param  a 32-bit value
  * @retval None
  */
__STATIC_INLINE void md_can_set_filter13_register2(CAN_TypeDef *CANx, uint32_t value)
{
    WRITE_REG(CANx->FLT13R2, value);
}

/**
  * @brief  Set CAN FLT13R2
  * @param  CANx CAN Instance
  * @retval a 32-bit value
  */
__STATIC_INLINE uint32_t md_can_get_filter13_register2(CAN_TypeDef *CANx)
{
    return (uint32_t)(READ_REG(CANx->FLT13R2));
}
/**
  * @}
  */

/** @defgroup CAN_Function
  * @{
  */
void md_can_init(CAN_TypeDef *CANx, md_can_init_typedef *init);
void md_can_filter_config(CAN_TypeDef *CANx, md_can_filter_list_t filter_number, md_can_filter_config_typedef *filter_config);
void md_can_bittime_config(CAN_TypeDef *CANx, md_can_bittime_config_typedef *config);
void md_can_debugmode_config(CAN_TypeDef *CANx, md_can_debugmode_t mode);
void md_can_mode_config(CAN_TypeDef *CANx, md_can_mode_t mode);
void md_can_send(CAN_TypeDef *CANx, md_can_txmailbox_typedef *message);
void md_can_receive(CAN_TypeDef *CANx, md_can_rx_fifo_t fifo_num, md_can_rxmailbox_typedef *message);

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __MD_CAN_H__ */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
