/**********************************************************************************
 *
 * @file    md_adc.h
 * @brief   header file of md_adc.c
 *
 * @date    11 Apr 2023
 * @author  AE Team
 * @note
 *          Change Logs:
 *          Date            Author          Notes
 *          11 Apr 2023     AE Team         the first version
 *
 * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
 *
 * SPDX-License-Identifier: Apache-2.0
 *
 * Licensed under the Apache License, Version 2.0 (the License); you may
 * not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 * www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an AS IS BASIS, WITHOUT
 * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 *
 **********************************************************************************
 */

/* Define to prevent recursive inclusion -------------------------------------*/
#ifndef __MD_ADC_H__
#define __MD_ADC_H__

#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */

/* Includes ------------------------------------------------------------------ */
#include "md_utils.h"

/* Exported Macros ----------------------------------------------------------- */

/* Exported Variables -------------------------------------------------------- */

/* Exported Constants -------------------------------------------------------- */

/* Exported Types ------------------------------------------------------------ */

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_ADC ADC
  * @brief ADC micro driver
  * @{
  */

/** @defgroup MD_ADC_Public_Macros ADC Public Macros
  * @{
  */

typedef enum
{
    MD_ADC_CH0                 = (0x00000000U),
    MD_ADC_CH1                 = (0x00000001U),
    MD_ADC_CH2                 = (0x00000002U),
    MD_ADC_CH3                 = (0x00000003U),
    MD_ADC_CH4                 = (0x00000004U),
    MD_ADC_CH5                 = (0x00000005U),
    MD_ADC_CH6                 = (0x00000006U),
    MD_ADC_CH7                 = (0x00000007U),
    MD_ADC_CH8                 = (0x00000008U),
    MD_ADC_CH9                 = (0x00000009U),
    MD_ADC_CH10                = (0x0000000AU),
    MD_ADC_CH11                = (0x0000000BU),
    MD_ADC_CH12                = (0x0000000CU),
    MD_ADC_CH13                = (0x0000000DU),
    MD_ADC_CH14                = (0x0000000EU),
    MD_ADC_CH15                = (0x0000000FU),
    MD_ADC_CH16                = (0x00000010U),
    MD_ADC_CH17                = (0x00000011U),
    MD_ADC_CH18                = (0x00000012U)
} md_adc_channel_list_t;

typedef enum
{
    MD_ADC_CFG1_AWD1_ALL_CHANNEL    = (0x00000000U),
    MD_ADC_CFG1_AWD1_SINGLE_CHANNEL = (0x00000001U)
} md_adc_analog_watchdog1_channel_select_list_t;

typedef enum
{
    MD_ADC_CFG1_JSQR_NOT_EMPTY      = (0x00000000U),
    MD_ADC_CFG1_JSQR_EMPTY          = (0x00000001U)
} md_adc_inject_discontinuous_channel_mode_list_t;

typedef enum
{
    MD_ADC_CFG1_DISCNUM_1            = (0x00000000U),
    MD_ADC_CFG1_DISCNUM_2            = (0x00000001U),
    MD_ADC_CFG1_DISCNUM_3            = (0x00000002U),
    MD_ADC_CFG1_DISCNUM_4            = (0x00000003U),
    MD_ADC_CFG1_DISCNUM_5            = (0x00000004U),
    MD_ADC_CFG1_DISCNUM_6            = (0x00000005U),
    MD_ADC_CFG1_DISCNUM_7            = (0x00000006U),
    MD_ADC_CFG1_DISCNUM_8            = (0x00000007U)
} md_adc_regular_discontinous_num_list_t;

typedef enum
{
    MD_ADC_CFG1_ALIGN_RIGHT          = (0x00000000U),
    MD_ADC_CFG1_ALIGN_LEFT           = (0x00000001U)
} md_adc_data_align_list_t;

typedef enum
{
    MD_ADC_CFG1_REGULAR_SINGLE       = (0x00000000U),
    MD_ADC_CFG1_REGULAR_CONTINUOUS   = (0x00000001U)
} md_adc_regular_continute_mode_list_t;

typedef enum
{
    MD_ADC_CFG1_OVRRMOD_PRESERVED    = (0x00000000U),
    MD_ADC_CFG1_OVRRMOD_OVERWRITTEN  = (0x00000001U)
} md_adc_regular_overflow_mode_list_t;

typedef enum
{
    MD_ADC_CFG1_TRIG_SOFTWARE            = (0x00000000U),
    MD_ADC_CFG1_TRIG_EXT_RISING          = (0x00000001U),
    MD_ADC_CFG1_TRIG_EXT_FALLING         = (0x00000002U),
    MD_ADC_CFG1_TRIG_EXT_BOTH            = (0x00000003U)
} md_adc_regular_trigger_mode_list_t;

typedef enum
{
    MD_ADC_CFG1_REXT_AD16C6T1_CH1       = (0x00000000U),
    MD_ADC_CFG1_REXT_AD16C6T1_CH2       = (0x00000001U),
    MD_ADC_CFG1_REXT_AD16C6T1_CH3       = (0x00000002U),
    MD_ADC_CFG1_REXT_GP32C4T1_CH2       = (0x00000003U),
    MD_ADC_CFG1_REXT_GP32C4T2_TRGOUT    = (0x00000004U),
    MD_ADC_CFG1_REXT_AD16C6T1_CH4       = (0x00000005U),
    MD_ADC_CFG1_REXT_EXTI_ADTE1         = (0x00000006U),
    MD_ADC_CFG1_REXT_AD16C6T2_TRGOUT    = (0x00000007U),
    MD_ADC_CFG1_REXT_AD16C6T2_TRGOUT2   = (0x00000008U),
    MD_ADC_CFG1_REXT_AD16C6T1_TRGOUT    = (0x00000009U),
    MD_ADC_CFG1_REXT_AD16C6T1_TRGOUT2   = (0x0000000AU),
    MD_ADC_CFG1_REXT_GP32C4T1_TRGOUT    = (0x0000000BU),
    MD_ADC_CFG1_REXT_AD16C6T2_CH4       = (0x0000000CU),
    MD_ADC_CFG1_REXT_BS16T_TRGOUT       = (0x0000000DU),
    MD_ADC_CFG1_REXT_GP16C2T1_TRGOUT    = (0x0000000EU),
    MD_ADC_CFG1_REXT_GP32C4T2_CH4       = (0x0000000FU),
    MD_ADC_CFG1_REXT_GP16C2T1_CH1       = (0x00000010U),
    MD_ADC_CFG1_REXT_GP16C2T1_CH2       = (0x00000011U),
    MD_ADC_CFG1_REXT_AD16C6T2_CH1       = (0x00000012U),
    MD_ADC_CFG1_REXT_AD16C6T2_CH2       = (0x00000013U),
    MD_ADC_CFG1_REXT_AD16C6T2_CH3       = (0x00000014U),
    MD_ADC_CFG1_REXT_GP16C2T2_CH1       = (0x00000015U),
    MD_ADC_CFG1_REXT_GP16C2T2_CH2       = (0x00000016U),
    MD_ADC_CFG1_REXT_GP16C2T2_TRGOUT    = (0x00000017U),
    MD_ADC_CFG1_REXT_GP32C4T1_CH1       = (0x00000018U),
    MD_ADC_CFG1_REXT_GP32C4T1_CH3       = (0x00000019U),
    MD_ADC_CFG1_REXT_GP32C4T1_CH4       = (0x0000001AU),
    MD_ADC_CFG1_REXT_GP32C4T2_CH1       = (0x0000001CU),
    MD_ADC_CFG1_REXT_GP32C4T2_CH2       = (0x0000001DU),
    MD_ADC_CFG1_REXT_GP32C4T2_CH3       = (0x0000001EU)
} md_adc_regular_external_trigger_source_list_t;

typedef enum
{
    MD_ADC_CFG1_RESOL_12BIT       = (0x00000000U),
    MD_ADC_CFG1_RESOL_10BIT       = (0x00000001U),
    MD_ADC_CFG1_RESOL_8BIT        = (0x00000002U),
    MD_ADC_CFG1_RESOL_6BIT        = (0x00000003U)
} md_adc_data_resolution_t;

typedef enum
{
    MD_ADC_CFG2_SWTRIG_CONVERT    = (0x00000000U),
    MD_ADC_CFG2_SWTRIG_SAMPLE     = (0x00000001U)
} md_adc_software_trigger_action_t;

typedef enum
{
    MD_ADC_CFG2_GCOMP_DISABLE           = (0x00000000U),
    MD_ADC_CFG2_GCOMP_ENABLE            = (0x00000001U)
} md_adc_gain_t;

typedef enum
{
    MD_ADC_CFG2_ROVSM_CONTINUOUS  = (0x00000000U),
    MD_ADC_CFG2_ROVSM_RESUME      = (0x00000001U)
} md_adc_regular_oversample_mode_list_t;

typedef enum
{
    MD_ADC_CFG2_TROVS_CONTINUOUS       = (0x00000000U),
    MD_ADC_CFG2_TROVS_DISCONTINUOUS    = (0x00000001U)
} md_adc_oversample_trigger_mode_list_t;

typedef enum
{
    MD_ADC_CFG2_OVSSHIFT_0_BITS    = (0x00000000U),
    MD_ADC_CFG2_OVSSHIFT_1_BITS    = (0x00000001U),
    MD_ADC_CFG2_OVSSHIFT_2_BITS    = (0x00000002U),
    MD_ADC_CFG2_OVSSHIFT_3_BITS    = (0x00000003U),
    MD_ADC_CFG2_OVSSHIFT_4_BITS    = (0x00000004U),
    MD_ADC_CFG2_OVSSHIFT_5_BITS    = (0x00000005U),
    MD_ADC_CFG2_OVSSHIFT_6_BITS    = (0x00000006U),
    MD_ADC_CFG2_OVSSHIFT_7_BITS    = (0x00000007U),
    MD_ADC_CFG2_OVSSHIFT_8_BITS    = (0x00000008U)
} md_adc_oversample_shift_list_t;

typedef enum
{
    MD_ADC_CFG2_OVSRATIO_2X         = (0x00000000U),
    MD_ADC_CFG2_OVSRATIO_4X         = (0x00000001U),
    MD_ADC_CFG2_OVSRATIO_8X         = (0x00000002U),
    MD_ADC_CFG2_OVSRATIO_16X        = (0x00000003U),
    MD_ADC_CFG2_OVSRATIO_32X        = (0x00000004U),
    MD_ADC_CFG2_OVSRATIO_64X        = (0x00000005U),
    MD_ADC_CFG2_OVSRATIO_128X       = (0x00000006U),
    MD_ADC_CFG2_OVSRATIO_256X       = (0x00000007U)
} md_adc_oversample_radio_list_t;

typedef enum
{
    MD_ADC_AWD1_FILTER_NONE             = (0x00000000U),
    MD_ADC_AWD1_FILTER_2SAMPLES         = (0x00000001U),
    MD_ADC_AWD1_FILTER_3SAMPLES         = (0x00000002U),
    MD_ADC_AWD1_FILTER_4SAMPLES         = (0x00000003U),
    MD_ADC_AWD1_FILTER_5SAMPLES         = (0x00000004U),
    MD_ADC_AWD1_FILTER_6SAMPLES         = (0x00000005U),
    MD_ADC_AWD1_FILTER_7SAMPLES         = (0x00000006U),
    MD_ADC_AWD1_FILTER_8SAMPLES         = (0x00000007U)
} md_adc_watchdog_filter_t;

typedef enum
{
    MD_ADC_RSQLEN_1              = (0x00000000U),
    MD_ADC_RSQLEN_2              = (0x00000001U),
    MD_ADC_RSQLEN_3              = (0x00000002U),
    MD_ADC_RSQLEN_4              = (0x00000003U),
    MD_ADC_RSQLEN_5              = (0x00000004U),
    MD_ADC_RSQLEN_6              = (0x00000005U),
    MD_ADC_RSQLEN_7              = (0x00000006U),
    MD_ADC_RSQLEN_8              = (0x00000007U),
    MD_ADC_RSQLEN_9              = (0x00000008U),
    MD_ADC_RSQLEN_10             = (0x00000009U),
    MD_ADC_RSQLEN_11             = (0x0000000AU),
    MD_ADC_RSQLEN_12             = (0x0000000BU),
    MD_ADC_RSQLEN_13             = (0x0000000CU),
    MD_ADC_RSQLEN_14             = (0x0000000DU),
    MD_ADC_RSQLEN_15             = (0x0000000EU),
    MD_ADC_RSQLEN_16             = (0x0000000FU)
} md_adc_regular_length_t;

typedef enum
{
    MD_ADC_JSQR_TRIG_SOFTWARE         = (0x00000000U),
    MD_ADC_JSQR_TRIG_EXT_RISING       = (0x00000001U),
    MD_ADC_JSQR_TRIG_EXT_FALLING      = (0x00000002U),
    MD_ADC_JSQR_TRIG_EXT_BOTH         = (0x00000003U)
} md_adc_inject_trigger_mode_list_t;

typedef enum
{
    MD_ADC_JSQR_JEXT_AD16C6T1_TRGOUT    = (0x00000000U),
    MD_ADC_JSQR_JEXT_AD16C6T1_CH4       = (0x00000001U),
    MD_ADC_JSQR_JEXT_GP32C4T1_TRGOUT    = (0x00000002U),
    MD_ADC_JSQR_JEXT_GP32C4T1_CH1       = (0x00000003U),
    MD_ADC_JSQR_JEXT_GP32C4T2_CH4       = (0x00000004U),
    MD_ADC_JSQR_JEXT_GP32C4T2_CH2       = (0x00000005U),
    MD_ADC_JSQR_JEXT_EXTI_ADTE2         = (0x00000006U),
    MD_ADC_JSQR_JEXT_AD16C6T2_CH4       = (0x00000007U),
    MD_ADC_JSQR_JEXT_AD16C6T1_TRGOUT2   = (0x00000008U),
    MD_ADC_JSQR_JEXT_AD16C6T2_TRGOUT    = (0x00000009U),
    MD_ADC_JSQR_JEXT_AD16C6T2_TRGOUT2   = (0x0000000AU),
    MD_ADC_JSQR_JEXT_GP32C4T2_CH3       = (0x0000000BU),
    MD_ADC_JSQR_JEXT_GP32C4T2_TRGOUT    = (0x0000000CU),
    MD_ADC_JSQR_JEXT_GP32C4T2_CH1       = (0x0000000DU),
    MD_ADC_JSQR_JEXT_BS16T_TRGOUT       = (0x0000000EU),
    MD_ADC_JSQR_JEXT_GP16C2T1_TRGOUT    = (0x0000000FU),
    MD_ADC_JSQR_JEXT_GP16C2T1_CH1       = (0x00000010U),
    MD_ADC_JSQR_JEXT_GP16C2T1_CH2       = (0x00000011U),
    MD_ADC_JSQR_JEXT_AD16C6T2_CH1       = (0x00000012U),
    MD_ADC_JSQR_JEXT_AD16C6T2_CH2       = (0x00000013U),
    MD_ADC_JSQR_JEXT_AD16C6T2_CH3       = (0x00000014U),
    MD_ADC_JSQR_JEXT_GP16C2T2_CH2       = (0x00000016U),
    MD_ADC_JSQR_JEXT_GP16C2T2_TRGOUT    = (0x00000017U),
    MD_ADC_JSQR_JEXT_GP32C4T1_CH2       = (0x00000018U),
    MD_ADC_JSQR_JEXT_GP32C4T1_CH3       = (0x00000019U),
    MD_ADC_JSQR_JEXT_GP32C4T1_CH4       = (0x0000001AU),
    MD_ADC_JSQR_JEXT_GP16C2T2_CH1       = (0x0000001BU),
    MD_ADC_JSQR_JEXT_AD16C6T1_CH1       = (0x0000001CU),
    MD_ADC_JSQR_JEXT_AD16C6T1_CH2       = (0x0000001DU),
    MD_ADC_JSQR_JEXT_AD16C6T1_CH3       = (0x0000001EU)
} md_adc_inject_trigger_source_t;

typedef enum
{
    MD_ADC_JSQLEN_1                     = (0x00000000U),
    MD_ADC_JSQLEN_2                     = (0x00000001U),
    MD_ADC_JSQLEN_3                     = (0x00000002U),
    MD_ADC_JSQLEN_4                     = (0x00000003U)
} md_adc_inject_length_t;

typedef enum
{
    MD_ADC_NEGATIVE_OFFSET              = (0x00000000U),
    MD_ADC_POSITIVE_OFFSET              = (0x00000001U)
} md_adc_offset_direction_t;

typedef enum
{
    MD_ADC_CCR_PRESCALE_1               = (0x00000000U),
    MD_ADC_CCR_PRESCALE_2               = (0x00000001U),
    MD_ADC_CCR_PRESCALE_4               = (0x00000002U),
    MD_ADC_CCR_PRESCALE_6               = (0x00000003U),
    MD_ADC_CCR_PRESCALE_8               = (0x00000004U),
    MD_ADC_CCR_PRESCALE_10              = (0x00000005U),
    MD_ADC_CCR_PRESCALE_12              = (0x00000006U),
    MD_ADC_CCR_PRESCALE_16              = (0x00000007U),
    MD_ADC_CCR_PRESCALE_32              = (0x00000008U),
    MD_ADC_CCR_PRESCALE_64              = (0x00000009U),
    MD_ADC_CCR_PRESCALE_128             = (0x0000000AU),
    MD_ADC_CCR_PRESCALE_256             = (0x0000000BU)
} md_adc_kernal_clock_prescale_t;

typedef enum
{
    MD_ADC_CCR_DMA_DISABLE              = (0x00000000U),
    MD_ADC_CCR_DMA_12_AND_10_RESOLUTION = (0x00000002U),
    MD_ADC_CCR_DMA_8_AND_6_RESOLUTION   = (0x00000003U)
} md_adc_dualmode_dma_mode_t;

typedef enum
{
    MD_ADC_WATCHDOG_CHANNEL_0             = (0),
    MD_ADC_WATCHDOG_CHANNEL_1             = (1),
    MD_ADC_WATCHDOG_CHANNEL_2             = (2),
    MD_ADC_WATCHDOG_CHANNEL_3             = (3),
    MD_ADC_WATCHDOG_CHANNEL_4             = (4),
    MD_ADC_WATCHDOG_CHANNEL_5             = (5),
    MD_ADC_WATCHDOG_CHANNEL_6             = (6),
    MD_ADC_WATCHDOG_CHANNEL_7             = (7),
    MD_ADC_WATCHDOG_CHANNEL_8             = (8),
    MD_ADC_WATCHDOG_CHANNEL_9             = (9),
    MD_ADC_WATCHDOG_CHANNEL_10            = (10),
    MD_ADC_WATCHDOG_CHANNEL_11            = (11),
    MD_ADC_WATCHDOG_CHANNEL_12            = (12),
    MD_ADC_WATCHDOG_CHANNEL_13            = (13),
    MD_ADC_WATCHDOG_CHANNEL_14            = (14),
    MD_ADC_WATCHDOG_CHANNEL_15            = (15),
    MD_ADC_WATCHDOG_CHANNEL_16            = (16),
    MD_ADC_WATCHDOG_CHANNEL_17            = (17),
    MD_ADC_WATCHDOG_CHANNEL_18            = (18)
} md_adc_watchdog_channel_t;

typedef enum
{
    MD_ADC_STANDALONE_MODE                                      = 0,
    MD_ADC_REGULAR_SIMULTANEOUS_AND_INJECTED_SIMULTANEOUS_MODE  = 1,
    MD_ADC_REGULAR_SIMULTANEOUS_AND_INTERLEAVED_MODE            = 2,
    MD_ADC_INTERLEAVED_AND_INJECTED_SIMULTANEOUS_MODE           = 3,
    MD_ADC_INJECTED_SIMULTANEOUS_MODE                           = 5,
    MD_ADC_REGULAR_SIMULTANEOUS_MODE                            = 6,
    MD_ADC_INTERLEAVED_MODE                                     = 7,
    MD_ADC_ALTERNATE_TRIGGER_MODE                               = 8,
} md_adc_dual_mode_t;

typedef enum
{
    MD_ADC_KERNEL_DISABLE   = 0,
    MD_ADC_KERNEL_PLLCLK    = 1,
    MD_ADC_KERNEL_SYSCLK    = 2,
    MD_ADC_KERNEL_AHBCLK    = 3
} md_adc_kernel_clock_source_t;

typedef enum
{
    MD_ADC_LOWPOWER_DISABLE,
    MD_ADC_LOWPOWER_ENABLE
} md_adc_lowerpower_mode_t;

typedef enum
{
    MD_ADC_DMA_DISABLE,
    MD_ADC_DMA_ENABLE
} md_adc_dma_enable_t;

typedef enum
{
    MD_ADC_SAMPLE_TIME_TRIGGER_MODE_DISABLE,
    MD_ADC_SAMPLE_TIME_TRIGGER_MODE_ENABLE
} md_adc_sample_time_trigger_mode_enable_t;

typedef enum
{
    MD_ADC_BULB_MODE_DISABLE,
    MD_ADC_BULB_MODE_ENABLE
} md_adc_bulb_mode_enable_t;

typedef enum
{
    MD_ADC_EXTRA_INCREASE_SAMPLE_TIME_DISABLE,
    MD_ADC_EXTRA_INCREASE_SAMPLE_TIME_ENABLE
} md_adc_extra_increase_sample_time_t;

typedef enum
{
    MD_ADC_INJECT_AUTO_DISABLE,
    MD_ADC_INJECT_AUTO_ENABLE
} md_adc_inject_auto_t;

typedef enum
{
    MD_ADC_REGULAR_DISCONTINUOUS_MODE_DISABLE,
    MD_ADC_REGULAR_DISCONTINUOUS_MODE_ENABLE
} md_adc_regular_discontinuous_mode_t;

typedef enum
{
    MD_ADC_INJECT_DISCONTINUOUS_MODE_DISABLE,
    MD_ADC_INJECT_DISCONTINUOUS_MODE_ENABLE
} md_adc_inject_discontinuous_mode_t;

typedef enum
{
    MD_ADC_INJECT_QUEUE_DISABLE,
    MD_ADC_INJECT_QUEUE_ENABLE
} md_adc_inject_queue_enable_t;

typedef enum
{
    MD_ADC_REGULAR_OVERSAMPLE_DISABLE,
    MD_ADC_REGULAR_OVERSAMPLE_ENABLE
} md_adc_regular_oversample_enable_t;

typedef enum
{
    MD_ADC_INJECT_OVERSAMPLE_DISABLE,
    MD_ADC_INJECT_OVERSAMPLE_ENABLE
} md_adc_inject_oversample_enable_t;

typedef enum
{
    MD_ADC_DATA_SATURATION_DISABLE,
    MD_ADC_DATA_SATURATION_ENABLE
} md_adc_data_saturation_reg_t;

typedef enum
{
    MD_ADC_OFFSET_DISABLE,
    MD_ADC_OFFSET_ENABLE
} md_adc_offset_reg_t;

typedef enum
{
    MD_ADC_REGULAR_ANALOG_WATCHDOG1_DISABLE,
    MD_ADC_REGULAR_ANALOG_WATCHDOG1_ENABLE
} md_adc_regular_analog_watchdog1_enable_t;

typedef enum
{
    MD_ADC_INJECT_ANALOG_WATCHDOG1_DISABLE,
    MD_ADC_INJECT_ANALOG_WATCHDOG1_ENABLE
} md_adc_inject_analog_watchdog1_enable_t;

extern unsigned int ADC_KernelClock_Frequency;

/**
  * @}
  */

/** @defgroup MD_ADC_Pubulic_Types ADC Pubulic Types
  * @{
  */
  
typedef struct
{
    double conversion_time; //unint:ns
    uint32_t smp;           //The register settings that should be returned after the calculation is completed.
} md_adc_sample_time_typedef;

typedef struct
{
    md_adc_kernel_clock_source_t                          clock_source;
    md_adc_kernal_clock_prescale_t                        clock_source_prescale;
    md_adc_lowerpower_mode_t                              adc_lowpower_mode;
    md_adc_data_resolution_t                              data_resolution;
    md_adc_data_align_list_t                              data_align;
    md_adc_regular_overflow_mode_list_t                   regular_overflow_mode;
    md_adc_dma_enable_t                                   dma_en;                  
    struct
    {
        md_adc_dualmode_dma_mode_t                        dma_mode;
        uint32_t                                          delay_timr_between_samples;
        md_adc_dual_mode_t                                mode_select;
    } dual_mode;
} md_adc_init_typedef;


typedef struct
{
    md_adc_sample_time_trigger_mode_enable_t              sample_time_trigger_mode_en;                
    md_adc_bulb_mode_enable_t                             bulb_mod_en;
    md_adc_extra_increase_sample_time_t                   extra_increase_sample_time_en;
    md_adc_inject_auto_t                                  inject_auto_en;
    md_adc_regular_continute_mode_list_t                  regular_continute_mode;
    struct
    {
        md_adc_channel_list_t  array_1;
        md_adc_channel_list_t  array_2;
        md_adc_channel_list_t  array_3;
        md_adc_channel_list_t  array_4;
    } inject_channel_array;
    md_adc_inject_length_t    inject_channel_length;

    struct
    {
        md_adc_channel_list_t  array_1;
        md_adc_channel_list_t  array_2;
        md_adc_channel_list_t  array_3;
        md_adc_channel_list_t  array_4;
        md_adc_channel_list_t  array_5;
        md_adc_channel_list_t  array_6;
        md_adc_channel_list_t  array_7;
        md_adc_channel_list_t  array_8;
        md_adc_channel_list_t  array_9;
        md_adc_channel_list_t  array_10;
        md_adc_channel_list_t  array_11;
        md_adc_channel_list_t  array_12;
        md_adc_channel_list_t  array_13;
        md_adc_channel_list_t  array_14;
        md_adc_channel_list_t  array_15;
        md_adc_channel_list_t  array_16;
    } regular_channel_array;
    md_adc_regular_length_t    regular_channel_length;

    struct
    {
        uint32_t  ch0;
        uint32_t  ch1;
        uint32_t  ch2;
        uint32_t  ch3;
        uint32_t  ch4;
        uint32_t  ch5;
        uint32_t  ch6;
        uint32_t  ch7;
        uint32_t  ch8;
        uint32_t  ch9;
        uint32_t  ch10;
        uint32_t  ch11;
        uint32_t  ch12;
        uint32_t  ch13;
        uint32_t  ch14;
        uint32_t  ch15;
        uint32_t  ch16;
        uint32_t  ch17;
        uint32_t  ch18;
    } sample_time_config;

} md_adc_input_channel_config_typedef;

typedef struct
{
    md_adc_regular_external_trigger_source_list_t       regular_external_trigger_source;
    md_adc_regular_trigger_mode_list_t                  regular_trigger_mode;
    md_adc_regular_discontinuous_mode_t                 regular_discontinuous_mode_en;
    md_adc_regular_discontinous_num_list_t              regular_discontinuous_channel_length;
} md_adc_regular_trigger_config_typedef;

typedef struct
{
    md_adc_inject_trigger_source_t                      inject_external_trigger_source;
    md_adc_inject_trigger_mode_list_t                   inject_trigger_mode;
    md_adc_inject_discontinuous_mode_t                  inject_discontinuous_mode_en;
    md_adc_inject_discontinuous_channel_mode_list_t     inject_discontinuous_channel_mode;
    md_adc_inject_queue_enable_t inject_queue_en;
} md_adc_inject_trigger_config_typedef;

typedef struct
{
    md_adc_regular_oversample_mode_list_t               regular_oversample_mode;
    md_adc_oversample_trigger_mode_list_t               oversample_trigger_mode;
    md_adc_oversample_shift_list_t                      oversample_shift;
    md_adc_oversample_radio_list_t                      oversample_ratio;
    md_adc_regular_oversample_enable_t                  regular_oversample_en;
    md_adc_inject_oversample_enable_t                   inject_oversample_en;
    struct
    {
        uint32_t                                      offset_reg1;
        md_adc_offset_direction_t                       positive_offset_reg1;
        md_adc_data_saturation_reg_t                    data_saturation_reg1_en;
        md_adc_channel_list_t                           offset_reg1_channel;
        md_adc_offset_reg_t                             offset_reg1_en;

        uint32_t                                      offset_reg2;
        md_adc_offset_direction_t                       positive_offset_reg2;
        md_adc_data_saturation_reg_t                    data_saturation_reg2_en;
        md_adc_channel_list_t                           offset_reg2_channel;
        md_adc_offset_reg_t                             offset_reg2_en;

        uint32_t                                      offset_reg3;
        md_adc_offset_direction_t                       positive_offset_reg3;
        md_adc_data_saturation_reg_t                    data_saturation_reg3_en;
        md_adc_channel_list_t                           offset_reg3_channel;
        md_adc_offset_reg_t                             offset_reg3_en;

        uint32_t                                      offset_reg4;
        md_adc_offset_direction_t                       positive_offset_reg4;
        md_adc_data_saturation_reg_t                    data_saturation_reg4_en;
        md_adc_channel_list_t                           offset_reg4_channel;
        md_adc_offset_reg_t offset_reg4_en;
    } offset_cofig;

} md_adc_oversample_offset_config_typedef;

typedef struct
{
    md_adc_regular_analog_watchdog1_enable_t            regular_analog_watchdog1_en;
    md_adc_inject_analog_watchdog1_enable_t             inject_analog_watchdog1_en;
    md_adc_analog_watchdog1_channel_select_list_t       analog_watchdog1_channel_select;
    md_adc_watchdog_filter_t                            analog_watchdog1_filter;
    md_adc_watchdog_channel_t                           analog_watchdog1_channel;
    uint32_t                                          analog_watchdog1_channel_lower_limit;
    uint32_t                                          analog_watchdog1_channel_upper_limit;
    uint32_t                                          analog_watchdog2_channel;
    uint32_t                                          analog_watchdog2_channel_lower_limit;
    uint32_t                                          analog_watchdog2_channel_upper_limit;
    uint32_t                                          analog_watchdog3_channel;
    uint32_t                                          analog_watchdog3_channel_lower_limit;
    uint32_t                                          analog_watchdog3_channel_upper_limit;
} md_adc_watchdog_config_typedef;

/* Exported Functions -------------------------------------------------------- */

/**
  * @}
  */
  
/** @defgroup MD_ADC_Public_Functions ADC Public Functions
  * @{
  */

/** @defgroup IER
  * @{
  */

/**
  * @brief  Set ADC_IER Register
  * @param  ADCx ADC Instance
  * @param  ier
  */
__STATIC_INLINE void md_adc_set_ier(ADC_TypeDef *ADCx, uint32_t ier)
{
    WRITE_REG(ADCx->IER, ier);
}

/**
  * @brief  TO interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_time_out(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_TO);
}

/**
  * @brief  JQOVF interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_inject_queue_overflow(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_JQOVF);
}

/**
  * @brief  AWD3 interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_analog_watch_dog3(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_AWD3);
}

/**
  * @brief  AWD2 interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_analog_watch_dog2(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_AWD2);
}

/**
  * @brief  AWD1 interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_analog_watch_dog1(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_AWD1);
}

/**
  * @brief  JEOS interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_inject_channel_sequence_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_JEOS);
}

/**
  * @brief  JEOC interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_inject_channel_conversion_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_JEOC);
}

/**
  * @brief  ROVR interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_regular_data_overflow(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_ROVR);
}

/**
  * @brief  REOS interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_regular_channel_sequence_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_REOS);
}

/**
  * @brief  REOC interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_regular_channel_conversion_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_REOC);
}

/**
  * @brief  EOSMP interrupt Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_it_sampling_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IER, ADC_IER_EOSMP);
}

/**
  * @}
  */

/** @defgroup IDR
  * @{
  */

/**
  * @brief  Set ADC_IDR Register
  * @param  ADCx ADC Instance
  * @param  idr
  */
__STATIC_INLINE void md_adc_set_idr(ADC_TypeDef *ADCx, uint32_t idr)
{
    WRITE_REG(ADCx->IDR, idr);
}

/**
  * @brief  TO interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_time_out(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_TO);
}

/**
  * @brief  JQOVF interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_inject_queue_overflow(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_JQOVF);
}

/**
  * @brief  AWD3 interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_analog_watch_dog3(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_AWD3);
}

/**
  * @brief  AWD2 interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_analog_watch_dog2(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_AWD2);
}

/**
  * @brief  AWD1 interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_analog_watch_dog1(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_AWD1);
}

/**
  * @brief  JEOS interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_inject_channel_sequence_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_JEOS);
}

/**
  * @brief  JEOC interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_inject_channel_conversion_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_JEOC);
}

/**
  * @brief  ROVR interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_regular_data_overflow(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_ROVR);
}

/**
  * @brief  REOS interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_regular_channel_sequence_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_REOS);
}

/**
  * @brief  REOC interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_regular_channel_conversion_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_REOC);
}

/**
  * @brief  EOSMP interrupt Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_it_sampling_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->IDR, ADC_IDR_EOSMP);
}

/**
  * @}
  */

/** @defgroup IVS
  * @{
  */

/**
  * @brief  Get ADC_IVS Register
  * @param  ADCx ADC Instance
  */
__STATIC_INLINE uint32_t md_adc_get_ivs(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->IVS);
}

/**
  * @brief  Check if TO interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_time_out(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_TO) == (ADC_IVS_TO));
}

/**
  * @brief  Check if JQOVF interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_inject_queue_overflow(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_JQOVF) == (ADC_IVS_JQOVF));
}

/**
  * @brief  Check if AWD3 interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_analog_watch_dog3(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_AWD3) == (ADC_IVS_AWD3));
}

/**
  * @brief  Check if AWD2 interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_analog_watch_dog2(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_AWD2) == (ADC_IVS_AWD2));
}

/**
  * @brief  Check if AWD1 interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_analog_watch_dog1(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_AWD1) == (ADC_IVS_AWD1));
}

/**
  * @brief  Check if JEOS interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_inject_channel_sequence_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_JEOS) == (ADC_IVS_JEOS));
}

/**
  * @brief  Check if JEOC interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_inject_channel_conversion_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_JEOC) == (ADC_IVS_JEOC));
}

/**
  * @brief  Check if ROVR interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_regular_data_overflow(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_ROVR) == (ADC_IVS_ROVR));
}

/**
  * @brief  Check if REOS interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_regular_channel_sequence_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_REOS) == (ADC_IVS_REOS));
}

/**
  * @brief  Check if REOC interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_regular_channel_conversion_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_REOC) == (ADC_IVS_REOC));
}

/**
  * @brief  Check if EOSMP interrupt is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_it_sampling_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IVS, ADC_IVS_EOSMP) == (ADC_IVS_EOSMP));
}

/**
  * @}
  */

/** @defgroup RIF
  * @{
  */


/**
  * @brief  Get ADC_RIF Register
  * @param  ADCx ADC Instance
  */
__STATIC_INLINE uint32_t md_adc_get_rif(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->RIF);
}

/**
  * @brief  Check TO interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_time_out(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_TO) == (ADC_RIF_TO));
}

/**
  * @brief  Check JQOVF interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_inject_queue_overflow(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_JQOVF) == (ADC_RIF_JQOVF));
}

/**
  * @brief  Check AWD3 interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_analog_watch_dog3(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_AWD3) == (ADC_RIF_AWD3));
}

/**
  * @brief  Check AWD2 interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_analog_watch_dog2(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_AWD2) == (ADC_RIF_AWD2));
}

/**
  * @brief  Check AWD1 interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_analog_watch_dog1(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_AWD1) == (ADC_RIF_AWD1));
}

/**
  * @brief  Check JEOS interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_inject_channel_sequence_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_JEOS) == (ADC_RIF_JEOS));
}

/**
  * @brief  Check JEOC interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_inject_channel_conversion_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_JEOC) == (ADC_RIF_JEOC));
}

/**
  * @brief  Check ROVR interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_regular_data_overflow(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_ROVR) == (ADC_RIF_ROVR));
}

/**
  * @brief  Check REOS interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_regular_channel_sequence_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_REOS) == (ADC_RIF_REOS));
}

/**
  * @brief  Check REOC interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_regular_channel_conversion_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_REOC) == (ADC_RIF_REOC));
}

/**
  * @brief  Check EOSMP interrupt Flag Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_active_flag_sampling_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->RIF, ADC_RIF_EOSMP) == (ADC_RIF_EOSMP));
}

/**
  * @}
  */

/** @defgroup IFM
  * @{
  */

/**
  * @brief  Get ADC_IFM Register
  * @param  ADCx ADC Instance
  */
__STATIC_INLINE uint32_t md_adc_get_ifm(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->IFM);
}

/**
  * @brief  Check TO interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_time_out(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_TO) == (ADC_IFM_TO));
}

/**
  * @brief  Check JQOVF interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_inject_queue_overflow(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_JQOVF) == (ADC_IFM_JQOVF));
}

/**
  * @brief  Check AWD3 interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_analog_watch_dog3(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_AWD3) == (ADC_IFM_AWD3));
}

/**
  * @brief  Check AWD2 interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_analog_watch_dog2(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_AWD2) == (ADC_IFM_AWD2));
}

/**
  * @brief  Check AWD1 interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_analog_watch_dog1(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_AWD1) == (ADC_IFM_AWD1));
}

/**
  * @brief  Check JEOS interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_inject_channel_sequence_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_JEOS) == (ADC_IFM_JEOS));
}

/**
  * @brief  Check JEOC interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_inject_channel_conversion_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_JEOC) == (ADC_IFM_JEOC));
}

/**
  * @brief  Check ROVR interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_regular_data_overflow(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_ROVR) == (ADC_IFM_ROVR));
}

/**
  * @brief  Check REOS interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_regular_channel_sequence_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_REOS) == (ADC_IFM_REOS));
}

/**
  * @brief  Check REOC interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_regular_channel_conversion_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_REOC) == (ADC_IFM_REOC));
}

/**
  * @brief  Check EOSMP interrupt Flag Masked Status.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_masked_it_sampling_end(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->IFM, ADC_IFM_EOSMP) == (ADC_IFM_EOSMP));
}

/**
  * @}
  */

/** @defgroup ICR
  * @{
  */

/**
  * @brief  Set ADC_ICR Register
  * @param  ADCx ADC Instance
  * @param  icr
  */
__STATIC_INLINE void md_adc_set_icr(ADC_TypeDef *ADCx, uint32_t icr)
{
    WRITE_REG(ADCx->ICR, icr);
}

/**
  * @brief  TO interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_time_out(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_TO);
}

/**
  * @brief  JQOVF interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_inject_queue_overflow(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_JQOVF);
}

/**
  * @brief  AWD3 interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_analog_watch_dog3(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_AWD3);
}

/**
  * @brief  AWD2 interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_analog_watch_dog2(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_AWD2);
}

/**
  * @brief  AWD1 interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_analog_watch_dog1(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_AWD1);
}

/**
  * @brief  JEOS interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_inject_channel_sequence_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_JEOS);
}

/**
  * @brief  JEOC interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_inject_channel_conversion_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_JEOC);
}

/**
  * @brief  ROVR interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_regular_data_overflow(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_ROVR);
}

/**
  * @brief  REOS interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_regular_channel_sequence_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_REOS);
}

/**
  * @brief  REOC interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_regular_channel_conversion_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_REOC);
}

/**
  * @brief  EOSMP interrupt Clear.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_clear_it_sampling_end(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->ICR, ADC_ICR_EOSMP);
}

/**
  * @}
  */

/** @defgroup CON
  * @{
  */

/**
  * @brief  Set ADC_CON Register
  * @param  ADCx ADC Instance
  * @param  con
  */
__STATIC_INLINE void md_adc_set_con(ADC_TypeDef *ADCx, uint32_t con)
{
    WRITE_REG(ADCx->CON, con);
}

/**
  * @brief  Get ADC_CON Register
  * @param  ADCx ADC Instance
  * @retval ADC_CON Values
  */
__STATIC_INLINE uint32_t md_adc_get_con(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->CON);
}

/**
  * @brief  ADC Low Power Mode Switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_low_power_mode(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CON, ADC_CON_LP_EN);
}

/**
  * @brief  ADC Low Power Mode Switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_low_power_mode(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CON, ADC_CON_LP_EN);
}

/**
  * @brief  Check if ADC Low Power Mode Switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_low_power_mode(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CON, ADC_CON_LP_EN) == (ADC_CON_LP_EN));
}

/**
  * @brief  Set ADC stops inject conversion
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_set_stop_injected(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CON, ADC_CON_JSTP);
}

/**
  * @brief  Check if ADC stops injecting conversion
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_get_stop_injected(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CON, ADC_CON_JSTP) == (ADC_CON_JSTP));
}

/**
  * @brief  Set ADC stop regular conversion
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_set_stop_regular(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CON, ADC_CON_RSTP);
}

/**
  * @brief  Check if ADC stop regular conversion
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_get_stop_regular(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CON, ADC_CON_RSTP) == (ADC_CON_RSTP));
}

/**
  * @brief  Enable ADC injected conversion
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_set_start_injected(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CON, ADC_CON_JSTART);
}

/**
  * @brief  Check if ADC injected conversion is enabled
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_get_start_injected(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CON, ADC_CON_JSTART) == (ADC_CON_JSTART));
}

/**
  * @brief  Enable ADC regular conversion
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_set_start_regular(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CON, ADC_CON_RSTART);
}

/**
  * @brief  Check if ADC regular conversion is enabled
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_get_start_regular(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CON, ADC_CON_RSTART) == (ADC_CON_RSTART));
}

/**
  * @brief  Set ADC disable command.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_adcpower(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CON, ADC_CON_ADCDIS);
}

/**
  * @brief  Get ADC disable control.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disabled_adcpower(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CON, ADC_CON_ADCDIS) == (ADC_CON_ADCDIS));
}

/**
  * @brief  Set ADC enable control.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_adcpower(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CON, ADC_CON_ADCEN);
}

/**
  * @brief  Get ADC enable control.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_adcpower(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CON, ADC_CON_ADCEN) == (ADC_CON_ADCEN));
}
/**
  * @}
  */

/** @defgroup CFG1
  * @{
  */

/**
  * @brief  Set ADC_CFG1 Register
  * @param  ADCx ADC Instance
  * @param  cfg1
  */
__STATIC_INLINE void md_adc_set_cfg1(ADC_TypeDef *ADCx, uint32_t cfg1)
{
    WRITE_REG(ADCx->CFG1, cfg1);
}

/**
  * @brief  Get ADC_CFG1 Register
  * @param  ADCx ADC Instance
  * @retval cfg1 Values
  */
__STATIC_INLINE uint32_t md_adc_get_cfg1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->CFG1);
}

/**
  * @brief  Inject queue switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_injected_queue(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG1, ADC_CFG1_JQEN);
}

/**
  * @brief  Inject queue switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_injected_queue(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG1, ADC_CFG1_JQEN);
}

/**
  * @brief  Check if Inject queue switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_injected_queue(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG1, ADC_CFG1_JQEN) == (ADC_CFG1_JQEN));
}

/**
  * @brief  Set Analog watchdog 1 monitoring channel selection
  * @param  ADCx ADC Instance
  * @param  channel This parameter can be one of the following values:
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_0
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_1
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_2
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_3
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_4
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_5
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_6
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_7
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_8
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_9
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_10
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_11
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_12
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_13
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_14
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_15
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_16
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_17
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd1_monitor_channel(ADC_TypeDef *ADCx, md_adc_watchdog_channel_t channel)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_AWD1CH, channel << ADC_CFG1_AWD1CH_POSS);
}

/**
  * @brief  Get Analog watchdog 1 monitoring channel selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_0
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_1
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_2
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_3
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_4
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_5
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_6
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_7
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_8
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_9
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_10
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_11
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_12
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_13
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_14
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_15
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_16
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_17
  *         @arg @ref MD_ADC_WATCHDOG_CHANNEL_18
  */
__STATIC_INLINE md_adc_watchdog_channel_t md_adc_get_awd1_monitor_channel(ADC_TypeDef *ADCx)
{
    return (md_adc_watchdog_channel_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_AWD1CH) >> ADC_CFG1_AWD1CH_POSS);
}

/**
  * @brief  Automatic injection conversion mode switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_injected_auto_convert(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG1, ADC_CFG1_JAUTO);
}

/**
  * @brief  Automatic injection conversion mode switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_injected_auto_convert(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG1, ADC_CFG1_JAUTO);
}

/**
  * @brief  Check if Automatic injection conversion mode switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_injected_auto_convert(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG1, ADC_CFG1_JAUTO) == (ADC_CFG1_JAUTO));
}

/**
  * @brief  Analog watchdog 1 switch injected into channel Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_injected_awd(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG1, ADC_CFG1_AWD1JEN);
}

/**
  * @brief  Analog watchdog 1 switch injected into channel Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_injected_awd(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG1, ADC_CFG1_AWD1JEN);
}

/**
  * @brief  Check if Analog watchdog 1 switch injected into channel is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_injected_awd(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG1, ADC_CFG1_AWD1JEN) == (ADC_CFG1_AWD1JEN));
}

/**
  * @brief  Analog watchdog 1 switch regular into channel Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_regular_awd(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG1, ADC_CFG1_AWD1REN);
}

/**
  * @brief  Analog watchdog 1 switch regular into channel Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_regular_awd(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG1, ADC_CFG1_AWD1REN);
}

/**
  * @brief  Check if Analog watchdog 1 switch regular into channel is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_regular_awd(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG1, ADC_CFG1_AWD1REN) == (ADC_CFG1_AWD1REN));
}

/**
  * @brief  Set Analog watchdog 1 monitoring channel selection
  * @param  ADCx ADC Instance
  * @param  channel This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_AWD1_ALL_CHANNEL
  *         @arg @ref MD_ADC_CFG1_AWD1_SINGLE_CHANNEL
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd1_channel_sel(ADC_TypeDef *ADCx, md_adc_analog_watchdog1_channel_select_list_t sel)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_AWD1SEL, sel << ADC_CFG1_AWD1SEL_POS);
}

/**
  * @brief  Get Analog watchdog 1 monitoring channel selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_AWD1_ALL_CHANNEL
  *         @arg @ref MD_ADC_CFG1_AWD1_SINGLE_CHANNEL
  */
__STATIC_INLINE md_adc_analog_watchdog1_channel_select_list_t md_adc_get_awd1_channel_sel(ADC_TypeDef *ADCx)
{
    return (md_adc_analog_watchdog1_channel_select_list_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_AWD1SEL) >> ADC_CFG1_AWD1SEL_POS);
}

/**
  * @brief  Set JSQR queue mode
  * @param  ADCx ADC Instance
  * @param  channel This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_JSQR_NOT_EMPTY
  *         @arg @ref MD_ADC_CFG1_JSQR_EMPTY
  * @retval None
  */
__STATIC_INLINE void md_adc_set_jsqr_mode(ADC_TypeDef *ADCx, md_adc_inject_discontinuous_channel_mode_list_t mode)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_JQM, mode << ADC_CFG1_JQM_POS);
}

/**
  * @brief  Get JSQR queue mode
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_JSQR_NOT_EMPTY
  *         @arg @ref MD_ADC_CFG1_JSQR_EMPTY
  */
__STATIC_INLINE md_adc_inject_discontinuous_channel_mode_list_t md_adc_get_jsqr_mode(ADC_TypeDef *ADCx)
{
    return (md_adc_inject_discontinuous_channel_mode_list_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_JQM) >> ADC_CFG1_JQM_POS);
}

/**
  * @brief  Discontinuous mode switch for injection channels Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_injected_discontinous(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG1, ADC_CFG1_JDISCEN);
}

/**
  * @brief  Discontinuous mode switch for injection channels Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_injected_discontinous(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG1, ADC_CFG1_JDISCEN);
}

/**
  * @brief  Check if Discontinuous mode switch for injection channels is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_injected_discontinous(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG1, ADC_CFG1_JDISCEN) == (ADC_CFG1_JDISCEN));
}

/**
  * @brief  Set Number of regular conversion channels for non-sequential mode
  * @param  ADCx ADC Instance
  * @param  num This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_DISCNUM_0
  *         @arg @ref MD_ADC_CFG1_DISCNUM_1
  *         @arg @ref MD_ADC_CFG1_DISCNUM_2
  *         @arg @ref MD_ADC_CFG1_DISCNUM_3
  *         @arg @ref MD_ADC_CFG1_DISCNUM_4
  *         @arg @ref MD_ADC_CFG1_DISCNUM_5
  *         @arg @ref MD_ADC_CFG1_DISCNUM_6
  *         @arg @ref MD_ADC_CFG1_DISCNUM_7
  *         @arg @ref MD_ADC_CFG1_DISCNUM_8
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_discontinous_num(ADC_TypeDef *ADCx, md_adc_regular_discontinous_num_list_t num)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_DISCNUM, num << ADC_CFG1_DISCNUM_POSS);
}

/**
  * @brief  Get Number of regular conversion channels for non-sequential mode
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_DISCNUM_0
  *         @arg @ref MD_ADC_CFG1_DISCNUM_1
  *         @arg @ref MD_ADC_CFG1_DISCNUM_2
  *         @arg @ref MD_ADC_CFG1_DISCNUM_3
  *         @arg @ref MD_ADC_CFG1_DISCNUM_4
  *         @arg @ref MD_ADC_CFG1_DISCNUM_5
  *         @arg @ref MD_ADC_CFG1_DISCNUM_6
  *         @arg @ref MD_ADC_CFG1_DISCNUM_7
  *         @arg @ref MD_ADC_CFG1_DISCNUM_8
  */
__STATIC_INLINE md_adc_regular_discontinous_num_list_t md_adc_get_regular_discontinous_num(ADC_TypeDef *ADCx)
{
    return (md_adc_regular_discontinous_num_list_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_DISCNUM) >> ADC_CFG1_DISCNUM_POSS);
}

/**
  * @brief  Discontinuous mode switch for regular channelss Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_regular_discontinous(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG1, ADC_CFG1_DISCEN);
}

/**
  * @brief  Discontinuous mode switch for regular channels Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_regular_discontinous(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG1, ADC_CFG1_DISCEN);
}

/**
  * @brief  Check if Discontinuous mode switch for regular channels is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_regular_discontinous(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG1, ADC_CFG1_DISCEN) == (ADC_CFG1_DISCEN));
}

/**
  * @brief  Set data alignment mode
  * @param  ADCx ADC Instance
  * @param  align This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_ALIGN_RIGHT
  *         @arg @ref MD_ADC_CFG1_ALIGN_LEFT
  * @retval None
  */
__STATIC_INLINE void md_adc_set_data_alignment(ADC_TypeDef *ADCx, md_adc_data_align_list_t align)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_ALIGN, align << ADC_CFG1_ALIGN_POS);
}

/**
  * @brief  Get data alignment mode
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_ALIGN_RIGHT
  *         @arg @ref MD_ADC_CFG1_ALIGN_LEFT
  */
__STATIC_INLINE md_adc_data_align_list_t md_adc_get_data_alignment(ADC_TypeDef *ADCx)
{
    return (md_adc_data_align_list_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_ALIGN) >> ADC_CFG1_ALIGN_POS);
}

/**
  * @brief  Delay Conversion Mode Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_auto_delay_mode(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG1, ADC_CFG1_AUTODLY);
}

/**
  * @brief  Delay Conversion Mode Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_auto_delay_mode(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG1, ADC_CFG1_AUTODLY);
}

/**
  * @brief  Check if Delay Conversion Mode is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_auto_delay_mode(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG1, ADC_CFG1_AUTODLY) == (ADC_CFG1_AUTODLY));
}

/**
  * @brief  Set Regular Continuous Conversion Mode Selection
  * @param  ADCx ADC Instance
  * @param  mode This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_REGULAR_SINGLE
  *         @arg @ref MD_ADC_CFG1_REGULAR_CONTINUOUS
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_convsersion_mode(ADC_TypeDef *ADCx, md_adc_regular_continute_mode_list_t mode)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_RCONT, mode << ADC_CFG1_RCONT_POS);
}

/**
  * @brief  Get Regular Continuous Conversion Mode Selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_REGULAR_SINGLE
  *         @arg @ref MD_ADC_CFG1_REGULAR_CONTINUOUS
  */
__STATIC_INLINE md_adc_regular_continute_mode_list_t md_adc_get_regular_convsersion_mode(ADC_TypeDef *ADCx)
{
    return (md_adc_regular_continute_mode_list_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_RCONT) >> ADC_CFG1_RCONT_POS);
}

/**
  * @brief  Set Regular data overflow mode
  * @param  ADCx ADC Instance
  * @param  mode This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_OVRRMOD_PRESERVED
  *         @arg @ref MD_ADC_CFG1_OVRRMOD_OVERWRITTEN
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_overflow_mode(ADC_TypeDef *ADCx, md_adc_regular_overflow_mode_list_t mode)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_OVRRMOD, mode << ADC_CFG1_OVRRMOD_POS);
}

/**
  * @brief  Get Regular data overflow mode
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_OVRRMOD_PRESERVED
  *         @arg @ref MD_ADC_CFG1_OVRRMOD_OVERWRITTEN
  */
__STATIC_INLINE md_adc_regular_overflow_mode_list_t md_adc_get_regular_overflow_mode(ADC_TypeDef *ADCx)
{
    return (md_adc_regular_overflow_mode_list_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_OVRRMOD) >> ADC_CFG1_OVRRMOD_POS);
}

/**
  * @brief  Set Regular external trigger on with polarity selection
  * @param  ADCx ADC Instance
  * @param  polarity This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_TRIG_SOFTWARE
  *         @arg @ref MD_ADC_CFG1_TRIG_EXT_RISING
  *         @arg @ref MD_ADC_CFG1_TRIG_EXT_FALLING
  *         @arg @ref MD_ADC_CFG1_TRIG_EXT_BOTH
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_trigger_mode(ADC_TypeDef *ADCx, md_adc_regular_trigger_mode_list_t polarity)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_REXTEN, polarity << ADC_CFG1_REXTEN_POSS);
}

/**
  * @brief  Get Regular external trigger on with polarity selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_TRIG_SOFTWARE
  *         @arg @ref MD_ADC_CFG1_TRIG_EXT_RISING
  *         @arg @ref MD_ADC_CFG1_TRIG_EXT_FALLING
  *         @arg @ref MD_ADC_CFG1_TRIG_EXT_BOTH
  */
__STATIC_INLINE md_adc_regular_trigger_mode_list_t md_adc_get_regular_trigger_mode(ADC_TypeDef *ADCx)
{
    return (md_adc_regular_trigger_mode_list_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_REXTEN) >> ADC_CFG1_REXTEN_POSS);
}

/**
  * @brief  Set General external trigger event selection
  * @param  ADCx ADC Instance
  * @param  sel This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_CH3
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_CH4
  *         @arg @ref MD_ADC_CFG1_REXT_EXTI_ADTE1
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_TRGOUT2
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_TRGOUT2
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_CH4
  *         @arg @ref MD_ADC_CFG1_REXT_BS16T_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T1_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_CH4
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T1_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T1_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_CH3
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T2_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T2_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T2_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_CH3
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_CH4
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_CH3
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_external_trigger_source(ADC_TypeDef *ADCx, md_adc_regular_external_trigger_source_list_t sel)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_REXTSEL, sel << ADC_CFG1_REXTSEL_POSS);
}

/**
  * @brief  Get General external trigger event selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_CH3
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_CH4
  *         @arg @ref MD_ADC_CFG1_REXT_EXTI_ADTE1
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_TRGOUT2
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T1_TRGOUT2
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_CH4
  *         @arg @ref MD_ADC_CFG1_REXT_BS16T_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T1_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_CH4
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T1_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T1_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_AD16C6T2_CH3
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T2_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T2_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_GP16C2T2_TRGOUT
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_CH3
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T1_CH4
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_CH1
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_CH2
  *         @arg @ref MD_ADC_CFG1_REXT_GP32C4T2_CH3
  */
__STATIC_INLINE md_adc_regular_external_trigger_source_list_t md_adc_get_regular_external_trigger_source(ADC_TypeDef *ADCx)
{
    return (md_adc_regular_external_trigger_source_list_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_REXTSEL) >> ADC_CFG1_REXTSEL_POSS);
}

/**
  * @brief  Set ADC data resolution
  * @param  ADCx ADC Instance
  * @param  resolution This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_RESOL_12BIT
  *         @arg @ref MD_ADC_CFG1_RESOL_10BIT
  *         @arg @ref MD_ADC_CFG1_RESOL_8BIT
  *         @arg @ref MD_ADC_CFG1_RESOL_6BIT
  * @retval None
  */
__STATIC_INLINE void md_adc_set_resolution_selection(ADC_TypeDef *ADCx, md_adc_data_resolution_t resolution)
{
    MODIFY_REG(ADCx->CFG1, ADC_CFG1_RESOL, resolution << ADC_CFG1_RESOL_POSS);
}

/**
  * @brief  Get ADC data resolution
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG1_RESOL_12BIT
  *         @arg @ref MD_ADC_CFG1_RESOL_10BIT
  *         @arg @ref MD_ADC_CFG1_RESOL_8BIT
  *         @arg @ref MD_ADC_CFG1_RESOL_6BIT
  */
__STATIC_INLINE md_adc_data_resolution_t md_adc_get_resolution_selection(ADC_TypeDef *ADCx)
{
    return (md_adc_data_resolution_t)(READ_BIT(ADCx->CFG1, ADC_CFG1_RESOL) >> ADC_CFG1_RESOL_POSS);
}

/**
  * @brief  DMA Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_dma_access(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG1, ADC_CFG1_DMAEN);
}

/**
  * @brief  DMA Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_dma_access(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG1, ADC_CFG1_DMAEN);
}

/**
  * @brief  Check if DMA is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_dma_access(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG1, ADC_CFG1_DMAEN) == (ADC_CFG1_DMAEN));
}

/**
  * @}
  */

/** @defgroup CFG2
  * @{
  */

/**
  * @brief  Set ADC_CFG2 Register
  * @param  ADCx ADC Instance
  * @param  cfg2
  */
__STATIC_INLINE void md_adc_set_cfg2(ADC_TypeDef *ADCx, uint32_t cfg2)
{
    WRITE_REG(ADCx->CFG2, cfg2);
}

/**
  * @brief  Get ADC_CFG2 Register
  * @param  ADCx ADC Instance
  * @retval cfg2 Values
  */
__STATIC_INLINE uint32_t md_adc_get_cfg2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->CFG2);
}

/**
  * @brief  Sample Time Controlled Trigger Mode Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_sample_time_trigger(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG2, ADC_CFG2_SMPTRIG);
}

/**
  * @brief  Sample Time Controlled Trigger Mode Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_sample_time_trigger(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG2, ADC_CFG2_SMPTRIG);
}

/**
  * @brief  Check if Sample Time Controlled Trigger Mode is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_sample_time_trigger(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG2, ADC_CFG2_SMPTRIG) == (ADC_CFG2_SMPTRIG));
}

/**
  * @brief  Shutter (Bulb) sampling mode Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_bulb_mode(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG2, ADC_CFG2_BULB);
}

/**
  * @brief  Shutter (Bulb) sampling mode Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_bulb_mode(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG2, ADC_CFG2_BULB);
}

/**
  * @brief  Check if Shutter (Bulb) sampling mode is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_bulb_mode(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG2, ADC_CFG2_BULB) == (ADC_CFG2_BULB));
}

/**
  * @brief  Set Software Control of Sample Time Controlled Trigger Mode
  * @param  ADCx ADC Instance
  * @param  trigger This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_SWTRIG_CONVERT
  *         @arg @ref MD_ADC_CFG2_SWTRIG_SAMPLE
  * @retval None
  */
__STATIC_INLINE void md_adc_set_software_trigger(ADC_TypeDef *ADCx, md_adc_software_trigger_action_t trigger)
{
    MODIFY_REG(ADCx->CFG2, ADC_CFG2_SWTRIG, trigger << ADC_CFG2_SWTRIG_POS);
}

/**
  * @brief  Get Software Control of Sample Time Controlled Trigger Mode
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_SWTRIG_CONVERT
  *         @arg @ref MD_ADC_CFG2_SWTRIG_SAMPLE
  */
__STATIC_INLINE md_adc_software_trigger_action_t md_adc_get_software_trigger(ADC_TypeDef *ADCx)
{
    return (md_adc_software_trigger_action_t)(READ_BIT(ADCx->CFG2, ADC_CFG2_SWTRIG) >> ADC_CFG2_SWTRIG_POS);
}

/**
  * @brief  Set Gain compensation mode
  * @param  ADCx ADC Instance
  * @param  mode This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_GCOMP_DISABLE
  *         @arg @ref MD_ADC_CFG2_GCOMP_ENABLE
  * @retval None
  */
__STATIC_INLINE void md_adc_set_gain_compensation_mode(ADC_TypeDef *ADCx, md_adc_gain_t mode)
{
    MODIFY_REG(ADCx->CFG2, ADC_CFG2_GCOMP, mode << ADC_CFG2_GCOMP_POS);
}

/**
  * @brief  Get Gain compensation mode
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_GCOMP_DISABLE
  *         @arg @ref MD_ADC_CFG2_GCOMP_ENABLE
  */
__STATIC_INLINE md_adc_gain_t md_adc_get_gain_compensation_mode(ADC_TypeDef *ADCx)
{
    return (md_adc_gain_t)(READ_BIT(ADCx->CFG2, ADC_CFG2_GCOMP) >> ADC_CFG2_GCOMP_POS);
}

/**
  * @brief  Set Regular Oversampling Mode
  * @param  ADCx ADC Instance
  * @param  mode This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_ROVSM_CONTINUOUS
  *         @arg @ref MD_ADC_CFG2_ROVSM_RESUME
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_oversampling_mode(ADC_TypeDef *ADCx, md_adc_regular_oversample_mode_list_t mode)
{
    MODIFY_REG(ADCx->CFG2, ADC_CFG2_ROVSM, mode << ADC_CFG2_ROVSM_POS);
}

/**
  * @brief  Get Regular Oversampling Mode
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_ROVSM_CONTINUOUS
  *         @arg @ref MD_ADC_CFG2_ROVSM_RESUME
  */
__STATIC_INLINE md_adc_regular_oversample_mode_list_t md_adc_get_regular_oversampling_mode(ADC_TypeDef *ADCx)
{
    return (md_adc_regular_oversample_mode_list_t)(READ_BIT(ADCx->CFG2, ADC_CFG2_ROVSM) >> ADC_CFG2_ROVSM_POS);
}

/**
  * @brief  Set Oversample trigger mode
  * @param  ADCx ADC Instance
  * @param  mode This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_TROVS_CONTINUOUS
  *         @arg @ref MD_ADC_CFG2_TROVS_DISCONTINUOUS
  * @retval None
  */
__STATIC_INLINE void md_adc_set_oversampling_trigger_mode(ADC_TypeDef *ADCx, md_adc_oversample_trigger_mode_list_t mode)
{
    MODIFY_REG(ADCx->CFG2, ADC_CFG2_TROVS, mode << ADC_CFG2_TROVS_POS);
}

/**
  * @brief  Get Oversample trigger mode
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_TROVS_CONTINUOUS
  *         @arg @ref MD_ADC_CFG2_TROVS_DISCONTINUOUS
  */
__STATIC_INLINE md_adc_oversample_trigger_mode_list_t md_adc_get_oversampling_trigger_mode(ADC_TypeDef *ADCx)
{
    return (md_adc_oversample_trigger_mode_list_t)(READ_BIT(ADCx->CFG2, ADC_CFG2_TROVS) >> ADC_CFG2_TROVS_POS);
}

/**
  * @brief  Set Oversampled Data Shift
  * @param  ADCx ADC Instance
  * @param  shift This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_0_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_1_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_2_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_3_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_4_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_5_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_6_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_7_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_8_BITS
  * @retval None
  */
__STATIC_INLINE void md_adc_set_oversampling_data_shift(ADC_TypeDef *ADCx, md_adc_oversample_shift_list_t shift)
{
    MODIFY_REG(ADCx->CFG2, ADC_CFG2_OVSSHIFT, shift << ADC_CFG2_OVSSHIFT_POSS);
}

/**
  * @brief  Get Oversampled Data Shift
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_0_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_1_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_2_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_3_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_4_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_5_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_6_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_7_BITS
  *         @arg @ref MD_ADC_CFG2_OVSSHIFT_8_BITS
  */
__STATIC_INLINE md_adc_oversample_shift_list_t md_adc_get_oversampling_data_shift(ADC_TypeDef *ADCx)
{
    return (md_adc_oversample_shift_list_t)(READ_BIT(ADCx->CFG2, ADC_CFG2_OVSSHIFT) >> ADC_CFG2_OVSSHIFT_POSS);
}

/**
  * @brief  Set oversampling rate
  * @param  ADCx ADC Instance
  * @param  rate This parameter can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_2X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_4X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_8X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_16X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_32X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_64X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_128X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_256X
  * @retval None
  */
__STATIC_INLINE void md_adc_set_oversampling_rate(ADC_TypeDef *ADCx, md_adc_oversample_radio_list_t rate)
{
    MODIFY_REG(ADCx->CFG2, ADC_CFG2_OVSRATIO, rate << ADC_CFG2_OVSRATIO_POSS);
}

/**
  * @brief  Get oversampling rate
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_2X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_4X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_8X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_16X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_32X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_64X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_128X
  *         @arg @ref MD_ADC_CFG2_OVSRATIO_256X
  */
__STATIC_INLINE md_adc_oversample_radio_list_t md_adc_get_oversampling_rate(ADC_TypeDef *ADCx)
{
    return (md_adc_oversample_radio_list_t)(READ_BIT(ADCx->CFG2, ADC_CFG2_OVSRATIO) >> ADC_CFG2_OVSRATIO_POSS);
}

/**
  * @brief  Inject oversampling switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_injected_oversampling(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG2, ADC_CFG2_JOVSEN);
}

/**
  * @brief  Inject oversampling switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_injected_oversampling(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG2, ADC_CFG2_JOVSEN);
}

/**
  * @brief  Check if Inject oversampling switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_injected_oversampling(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG2, ADC_CFG2_JOVSEN) == (ADC_CFG2_JOVSEN));
}

/**
  * @brief  Regular oversampling switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_regular_oversampling(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->CFG2, ADC_CFG2_ROVSEN);
}

/**
  * @brief  Regular oversampling switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_regular_oversampling(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->CFG2, ADC_CFG2_ROVSEN);
}

/**
  * @brief  Check if Regular oversampling switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_regular_oversampling(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->CFG2, ADC_CFG2_ROVSEN) == (ADC_CFG2_ROVSEN));
}

/**
  * @}
  */

/** @defgroup SMPT1
  * @{
  */

/**
  * @brief  Set ADC_SMPT1 Register
  * @param  ADCx ADC Instance
  * @param  smpt1
  */
__STATIC_INLINE void md_adc_set_smpt1(ADC_TypeDef *ADCx, uint32_t smpt1)
{
    WRITE_REG(ADCx->SMPT1, smpt1);
}

/**
  * @brief  Get ADC_SMPT1 Register
  * @param  ADCx ADC Instance
  * @retval smpt1 Values
  */
__STATIC_INLINE uint32_t md_adc_get_smpt1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->SMPT1);
}

/**
  * @brief  Set Sampling time for channel 3
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_3(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT1, ADC_SMPT1_SMP3, time << ADC_SMPT1_SMP3_POSS);
}

/**
  * @brief  Get Sampling time for channel 3
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_3(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT1, ADC_SMPT1_SMP3) >> ADC_SMPT1_SMP3_POSS);
}

/**
  * @brief  Set Sampling time for channel 2
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_2(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT1, ADC_SMPT1_SMP2, time << ADC_SMPT1_SMP2_POSS);
}

/**
  * @brief  Get Sampling time for channel 2
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT1, ADC_SMPT1_SMP2) >> ADC_SMPT1_SMP2_POSS);
}

/**
  * @brief  Set Sampling time for channel 1
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_1(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT1, ADC_SMPT1_SMP1, time << ADC_SMPT1_SMP1_POSS);
}

/**
  * @brief  Get Sampling time for channel 1
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT1, ADC_SMPT1_SMP1) >> ADC_SMPT1_SMP1_POSS);
}

/**
  * @brief  Set Sampling time for channel 0
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_0(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT1, ADC_SMPT1_SMP0, time << ADC_SMPT1_SMP0_POSS);
}

/**
  * @brief  Get Sampling time for channel 0
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_0(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT1, ADC_SMPT1_SMP0) >> ADC_SMPT1_SMP0_POSS);
}

/**
  * @}
  */

/** @defgroup SMPT2
  * @{
  */

/**
  * @brief  Set ADC_SMPT2 Register
  * @param  ADCx ADC Instance
  * @param  smpt2
  */
__STATIC_INLINE void md_adc_set_smpt2(ADC_TypeDef *ADCx, uint32_t smpt2)
{
    WRITE_REG(ADCx->SMPT2, smpt2);
}

/**
  * @brief  Get ADC_SMPT2 Register
  * @param  ADCx ADC Instance
  * @retval smpt2 Values
  */
__STATIC_INLINE uint32_t md_adc_get_smpt2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->SMPT2);
}

/**
  * @brief  Set Sampling time for channel 7
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_7(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT2, ADC_SMPT2_SMP7, time << ADC_SMPT2_SMP7_POSS);
}

/**
  * @brief  Get Sampling time for channel 7
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_7(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT2, ADC_SMPT2_SMP7) >> ADC_SMPT2_SMP7_POSS);
}

/**
  * @brief  Set Sampling time for channel 6
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_6(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT2, ADC_SMPT2_SMP6, time << ADC_SMPT2_SMP6_POSS);
}

/**
  * @brief  Get Sampling time for channel 6
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_6(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT2, ADC_SMPT2_SMP6) >> ADC_SMPT2_SMP6_POSS);
}

/**
  * @brief  Set Sampling time for channel 5
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_5(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT2, ADC_SMPT2_SMP5, time << ADC_SMPT2_SMP5_POSS);
}

/**
  * @brief  Get Sampling time for channel 5
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_5(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT2, ADC_SMPT2_SMP5) >> ADC_SMPT2_SMP5_POSS);
}

/**
  * @brief  Set Sampling time for channel 4
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_4(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT2, ADC_SMPT2_SMP4, time << ADC_SMPT2_SMP4_POSS);
}

/**
  * @brief  Get Sampling time for channel 4
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_4(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT2, ADC_SMPT2_SMP4) >> ADC_SMPT2_SMP4_POSS);
}

/**
  * @}
  */

/** @defgroup SMPT3
  * @{
  */

/**
  * @brief  Set ADC_SMPT3 Register
  * @param  ADCx ADC Instance
  * @param  smpt3
  */
__STATIC_INLINE void md_adc_set_smpt3(ADC_TypeDef *ADCx, uint32_t smpt3)
{
    WRITE_REG(ADCx->SMPT3, smpt3);
}

/**
  * @brief  Get ADC_SMPT3 Register
  * @param  ADCx ADC Instance
  * @retval smpt3 Values
  */
__STATIC_INLINE uint32_t md_adc_get_smpt3(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->SMPT3);
}

/**
  * @brief  Set Sampling time for channel 11
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_11(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT3, ADC_SMPT3_SMP11, time << ADC_SMPT3_SMP11_POSS);
}

/**
  * @brief  Get Sampling time for channel 11
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_11(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT3, ADC_SMPT3_SMP11) >> ADC_SMPT3_SMP11_POSS);
}

/**
  * @brief  Set Sampling time for channel 10
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_10(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT3, ADC_SMPT3_SMP10, time << ADC_SMPT3_SMP10_POSS);
}

/**
  * @brief  Get Sampling time for channel 10
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_10(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT3, ADC_SMPT3_SMP10) >> ADC_SMPT3_SMP10_POSS);
}

/**
  * @brief  Set Sampling time for channel 9
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_9(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT3, ADC_SMPT3_SMP9, time << ADC_SMPT3_SMP9_POSS);
}

/**
  * @brief  Get Sampling time for channel 9
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_9(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT3, ADC_SMPT3_SMP9) >> ADC_SMPT3_SMP9_POSS);
}

/**
  * @brief  Set Sampling time for channel 8
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_8(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT3, ADC_SMPT3_SMP8, time << ADC_SMPT3_SMP8_POSS);
}

/**
  * @brief  Get Sampling time for channel 8
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_8(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT3, ADC_SMPT3_SMP8) >> ADC_SMPT3_SMP8_POSS);
}

/**
  * @}
  */

/** @defgroup SMPT4
  * @{
  */

/**
  * @brief  Set ADC_SMPT4 Register
  * @param  ADCx ADC Instance
  * @param  smpt4
  */
__STATIC_INLINE void md_adc_set_smpt4(ADC_TypeDef *ADCx, uint32_t smpt4)
{
    WRITE_REG(ADCx->SMPT4, smpt4);
}

/**
  * @brief  Get ADC_SMPT4 Register
  * @param  ADCx ADC Instance
  * @retval smpt4 Values
  */
__STATIC_INLINE uint32_t md_adc_get_smpt4(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->SMPT4);
}

/**
  * @brief  Set Sampling time for channel 15
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_15(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT4, ADC_SMPT4_SMP15, time << ADC_SMPT4_SMP15_POSS);
}

/**
  * @brief  Get Sampling time for channel 15
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_15(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT4, ADC_SMPT4_SMP15) >> ADC_SMPT4_SMP15_POSS);
}

/**
  * @brief  Set Sampling time for channel 14
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_14(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT4, ADC_SMPT4_SMP14, time << ADC_SMPT4_SMP14_POSS);
}

/**
  * @brief  Get Sampling time for channel 14
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_14(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT4, ADC_SMPT4_SMP14) >> ADC_SMPT4_SMP14_POSS);
}

/**
  * @brief  Set Sampling time for channel 13
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_13(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT4, ADC_SMPT4_SMP13, time << ADC_SMPT4_SMP13_POSS);
}

/**
  * @brief  Get Sampling time for channel 13
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_13(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT4, ADC_SMPT4_SMP13) >> ADC_SMPT4_SMP13_POSS);
}

/**
  * @brief  Set Sampling time for channel 12
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_12(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT4, ADC_SMPT4_SMP12, time << ADC_SMPT4_SMP12_POSS);
}

/**
  * @brief  Get Sampling time for channel 12
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_12(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT4, ADC_SMPT4_SMP12) >> ADC_SMPT4_SMP12_POSS);
}

/**
  * @}
  */

/** @defgroup SMPT5
  * @{
  */

/**
  * @brief  Set ADC_SMPT5 Register
  * @param  ADCx ADC Instance
  * @param  smpt5
  */
__STATIC_INLINE void md_adc_set_smpt5(ADC_TypeDef *ADCx, uint32_t smpt5)
{
    WRITE_REG(ADCx->SMPT5, smpt5);
}

/**
  * @brief  Get ADC_SMPT5 Register
  * @param  ADCx ADC Instance
  * @retval smpt5 Values
  */
__STATIC_INLINE uint32_t md_adc_get_smpt5(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->SMPT5);
}

/**
  * @brief  Additional sample time switch Enable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_enable_add_sample_time(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->SMPT5, ADC_SMPT5_SMPPLUS);
}

/**
  * @brief  Additional sample time switch Disable.
  * @param  ADCx ADC Instance
  * @retval None.
  */
__STATIC_INLINE void md_adc_disable_add_sample_time(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->SMPT5, ADC_SMPT5_SMPPLUS);
}

/**
  * @brief  Check if Additional sample time switch is Enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_enabled_add_sample_time(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->SMPT5, ADC_SMPT5_SMPPLUS) == (ADC_SMPT5_SMPPLUS));
}
/**
  * @brief  Set Sampling time for channel 18
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_18(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT5, ADC_SMPT5_SMP18, time << ADC_SMPT5_SMP18_POSS);
}

/**
  * @brief  Get Sampling time for channel 18
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_18(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT5, ADC_SMPT5_SMP18) >> ADC_SMPT5_SMP18_POSS);
}

/**
  * @brief  Set Sampling time for channel 17
  * @param  ADCx ADC Instance
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_17(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT5, ADC_SMPT5_SMP17, time << ADC_SMPT5_SMP17_POSS);
}

/**
  * @brief  Get Sampling time for channel 17
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_17(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT5, ADC_SMPT5_SMP17) >> ADC_SMPT5_SMP17_POSS);
}

/**
  * @brief  Set Sampling time for channel 16
  * @param  time This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sampletime_channel_16(ADC_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->SMPT5, ADC_SMPT5_SMP16, time << ADC_SMPT5_SMP16_POSS);
}

/**
  * @brief  Get Sampling time for channel 16
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_sampletime_channel_16(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->SMPT5, ADC_SMPT5_SMP16) >> ADC_SMPT5_SMP16_POSS);
}

/**
  * @}
  */


/** @defgroup TR1
  * @{
  */

/**
  * @brief  Set ADC_TR1 Register
  * @param  ADCx ADC Instance
  * @param  tr1
  */
__STATIC_INLINE void md_adc_set_tr1(ADC_TypeDef *ADCx, uint32_t tr1)
{
    WRITE_REG(ADCx->TR1, tr1);
}

/**
  * @brief  Get ADC_TR1 Register
  * @param  ADCx ADC Instance
  * @retval tr1 Values
  */
__STATIC_INLINE uint32_t md_adc_get_tr1(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->TR1);
}

/**
  * @brief  Set Analog Watchdog 1 Upper Threshold
  * @param  ht This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd1_high_threshold(ADC_TypeDef *ADCx, uint32_t ht)
{
    MODIFY_REG(ADCx->TR1, ADC_TR1_HT1, ht << ADC_TR1_HT1_POSS);
}

/**
  * @brief  Get Analog Watchdog 1 Upper Threshold
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_awd1_high_threshold(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->TR1, ADC_TR1_HT1) >> ADC_TR1_HT1_POSS);
}

/**
  * @brief  Set Analog watchdog filter parameters
  * @param  ht This parameter can be one of the following values:
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_AWD1_FILTER_NONE
  *         @arg @ref MD_ADC_AWD1_FILTER_2SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_3SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_4SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_5SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_6SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_7SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_8SAMPLES
  */
__STATIC_INLINE void md_adc_set_awd1_filter(ADC_TypeDef *ADCx, md_adc_watchdog_filter_t ht)
{
    MODIFY_REG(ADCx->TR1, ADC_TR1_AWDFILT, ht << ADC_TR1_AWDFILT_POSS);
}

/**
  * @brief  Get Analog watchdog filter parameters
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  *         @arg @ref MD_ADC_AWD1_FILTER_NONE
  *         @arg @ref MD_ADC_AWD1_FILTER_2SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_3SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_4SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_5SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_6SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_7SAMPLES
  *         @arg @ref MD_ADC_AWD1_FILTER_8SAMPLES
  */
__STATIC_INLINE md_adc_watchdog_filter_t md_adc_get_awd1_filter(ADC_TypeDef *ADCx)
{
    return (md_adc_watchdog_filter_t)(READ_BIT(ADCx->TR1, ADC_TR1_AWDFILT) >> ADC_TR1_AWDFILT_POSS);
}

/**
  * @brief  Set Analog Watchdog 1 Lower Threshold
  * @param  lt This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd1_low_threshold(ADC_TypeDef *ADCx, uint32_t lt)
{
    MODIFY_REG(ADCx->TR1, ADC_TR1_LT1, lt << ADC_TR1_LT1_POSS);
}

/**
  * @brief  Get Analog Watchdog 1 Lower Threshold
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_awd1_low_threshold(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->TR1, ADC_TR1_LT1) >> ADC_TR1_LT1_POSS);
}

/**
  * @}
  */

/** @defgroup TR2
  * @{
  */

/**
  * @brief  Set ADC_TR2 Register
  * @param  ADCx ADC Instance
  * @param  tr2
  */
__STATIC_INLINE void md_adc_set_tr2(ADC_TypeDef *ADCx, uint32_t tr2)
{
    WRITE_REG(ADCx->TR2, tr2);
}

/**
  * @brief  Get ADC_TR2 Register
  * @param  ADCx ADC Instance
  * @retval tr2 Values
  */
__STATIC_INLINE uint32_t md_adc_get_tr2(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->TR2);
}

/**
  * @brief  Set Analog Watchdog 2 Upper Threshold
  * @param  ht This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd2_high_threshold(ADC_TypeDef *ADCx, uint32_t ht)
{
    MODIFY_REG(ADCx->TR2, ADC_TR2_HT2, ht << ADC_TR2_HT2_POSS);
}

/**
  * @brief  Get Analog Watchdog 2 Upper Threshold
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_awd2_high_threshold(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->TR2, ADC_TR2_HT2) >> ADC_TR2_HT2_POSS);
}

/**
  * @brief  Set Analog Watchdog 2 Lower Threshold
  * @param  lt This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd2_low_threshold(ADC_TypeDef *ADCx, uint32_t lt)
{
    MODIFY_REG(ADCx->TR2, ADC_TR2_LT2, lt << ADC_TR2_LT2_POSS);
}

/**
  * @brief  Get Analog Watchdog 2 Lower Threshold
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_awd2_low_threshold(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->TR2, ADC_TR2_LT2) >> ADC_TR2_LT2_POSS);
}

/**
  * @}
  */

/** @defgroup TR3
  * @{
  */

/**
  * @brief  Set ADC_TR3 Register
  * @param  ADCx ADC Instance
  * @param  tr3
  */
__STATIC_INLINE void md_adc_set_tr3(ADC_TypeDef *ADCx, uint32_t tr3)
{
    WRITE_REG(ADCx->TR3, tr3);
}

/**
  * @brief  Get ADC_TR3 Register
  * @param  ADCx ADC Instance
  * @retval tr3 Values
  */
__STATIC_INLINE uint32_t md_adc_get_tr3(ADC_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->TR3);
}

/**
  * @brief  Set Analog Watchdog 3 Upper Threshold
  * @param  ht This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd3_high_threshold(ADC_TypeDef *ADCx, uint32_t ht)
{
    MODIFY_REG(ADCx->TR3, ADC_TR3_HT3, ht << ADC_TR3_HT3_POSS);
}

/**
  * @brief  Get Analog Watchdog 3 Upper Threshold
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_awd3_high_threshold(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->TR3, ADC_TR3_HT3) >> ADC_TR3_HT3_POSS);
}

/**
  * @brief  Set Analog Watchdog 3 Lower Threshold
  * @param  lt This parameter can be one of the following values:
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd3_low_threshold(ADC_TypeDef *ADCx, uint32_t lt)
{
    MODIFY_REG(ADCx->TR3, ADC_TR3_LT3, lt << ADC_TR3_LT3_POSS);
}

/**
  * @brief  Get Analog Watchdog 3 Lower Threshold
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_awd3_low_threshold(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->TR3, ADC_TR3_LT3) >> ADC_TR3_LT3_POSS);
}

/**
  * @}
  */

/** @defgroup SQR1
  * @{
  */
/**
  * @brief  Set ADC_SQR1 Register
  * @param  ADCx ADC Instance
  * @param  sqr1
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sqr1(ADC_TypeDef *ADCx, uint32_t sqr1)
{
    WRITE_REG(ADCx->SQR1, sqr1);
}

/**
  * @brief  Get ADC_SQR1 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_sqr1(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->SQR1));
}

/**
  * @brief  Set 4th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq4
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_4th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq4)
{
    MODIFY_REG(ADCx->SQR1, ADC_SQR1_RSQ4, rsq4 << ADC_SQR1_RSQ4_POSS);
}

/**
  * @brief  Get 4th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_4th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR1, ADC_SQR1_RSQ4)) >> ADC_SQR1_RSQ4_POSS);
}

/**
  * @brief  Set 3rd conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq3
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_3rd(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq3)
{
    MODIFY_REG(ADCx->SQR1, ADC_SQR1_RSQ3, rsq3 << ADC_SQR1_RSQ3_POSS);
}

/**
  * @brief  Get 3rd conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_3rd(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR1, ADC_SQR1_RSQ3)) >> ADC_SQR1_RSQ3_POSS);
}

/**
  * @brief  Set 2nd conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq2
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_2nd(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq2)
{
    MODIFY_REG(ADCx->SQR1, ADC_SQR1_RSQ2, rsq2 << ADC_SQR1_RSQ2_POSS);
}

/**
  * @brief  Get 2nd conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_2nd(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR1, ADC_SQR1_RSQ2)) >> ADC_SQR1_RSQ2_POSS);
}

/**
  * @brief  Set 1st conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq4
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_1st(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq1)
{
    MODIFY_REG(ADCx->SQR1, ADC_SQR1_RSQ1, rsq1 << ADC_SQR1_RSQ1_POSS);
}

/**
  * @brief  Get 1st conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_1st(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR1, ADC_SQR1_RSQ1)) >> ADC_SQR1_RSQ1_POSS);
}

/**
  * @brief  Set Regular channel sequence length.
  * @param  ADCx ADC Instance
  * @param  nsl
            @arg @ref MD_ADC_RSQLEN_1
            @arg @ref MD_ADC_RSQLEN_2
            @arg @ref MD_ADC_RSQLEN_3
            @arg @ref MD_ADC_RSQLEN_4
            @arg @ref MD_ADC_RSQLEN_5
            @arg @ref MD_ADC_RSQLEN_6
            @arg @ref MD_ADC_RSQLEN_7
            @arg @ref MD_ADC_RSQLEN_8
            @arg @ref MD_ADC_RSQLEN_9
            @arg @ref MD_ADC_RSQLEN_10
            @arg @ref MD_ADC_RSQLEN_11
            @arg @ref MD_ADC_RSQLEN_12
            @arg @ref MD_ADC_RSQLEN_13
            @arg @ref MD_ADC_RSQLEN_14
            @arg @ref MD_ADC_RSQLEN_15
            @arg @ref MD_ADC_RSQLEN_16
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_length(ADC_TypeDef *ADCx, md_adc_regular_length_t nsl)
{
    MODIFY_REG(ADCx->SQR1, ADC_SQR1_RSQLEN, nsl << ADC_SQR1_RSQLEN_POSS);
}

/**
  * @brief  Get Regular channel sequence length.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_RSQLEN_1
            @arg @ref MD_ADC_RSQLEN_2
            @arg @ref MD_ADC_RSQLEN_3
            @arg @ref MD_ADC_RSQLEN_4
            @arg @ref MD_ADC_RSQLEN_5
            @arg @ref MD_ADC_RSQLEN_6
            @arg @ref MD_ADC_RSQLEN_7
            @arg @ref MD_ADC_RSQLEN_8
            @arg @ref MD_ADC_RSQLEN_9
            @arg @ref MD_ADC_RSQLEN_10
            @arg @ref MD_ADC_RSQLEN_11
            @arg @ref MD_ADC_RSQLEN_12
            @arg @ref MD_ADC_RSQLEN_13
            @arg @ref MD_ADC_RSQLEN_14
            @arg @ref MD_ADC_RSQLEN_15
            @arg @ref MD_ADC_RSQLEN_16
  */
__STATIC_INLINE md_adc_regular_length_t md_adc_get_regular_sequence_length(ADC_TypeDef *ADCx)
{
    return (md_adc_regular_length_t)((READ_BIT(ADCx->SQR1, ADC_SQR1_RSQLEN)) >> ADC_SQR1_RSQLEN_POSS);
}
/**
  * @}
  */

/** @defgroup SQR2
  * @{
  */

/**
  * @brief  Set ADC_SQR2 Register
  * @param  ADCx ADC Instance
  * @param  sqr2
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sqr2(ADC_TypeDef *ADCx, uint32_t sqr2)
{
    WRITE_REG(ADCx->SQR2, sqr2);
}

/**
  * @brief  Get ADC_SQR2 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_sqr2(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->SQR2));
}

/**
  * @brief  Set 9th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq9
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_9th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq9)
{
    MODIFY_REG(ADCx->SQR2, ADC_SQR2_RSQ9, rsq9 << ADC_SQR2_RSQ9_POSS);
}

/**
  * @brief  Get 9th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_9th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR2, ADC_SQR2_RSQ9)) >> ADC_SQR2_RSQ9_POSS);
}

/**
  * @brief  Set 8th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq8
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_8th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq8)
{
    MODIFY_REG(ADCx->SQR2, ADC_SQR2_RSQ8, rsq8 << ADC_SQR2_RSQ8_POSS);
}

/**
  * @brief  Get 8th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_8th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR2, ADC_SQR2_RSQ8)) >> ADC_SQR2_RSQ8_POSS);
}

/**
  * @brief  Set 7th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq7
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_7th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq7)
{
    MODIFY_REG(ADCx->SQR2, ADC_SQR2_RSQ7, rsq7 << ADC_SQR2_RSQ7_POSS);
}

/**
  * @brief  Get 7th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_7th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR2, ADC_SQR2_RSQ7)) >> ADC_SQR2_RSQ7_POSS);
}

/**
  * @brief  Set 6th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq6
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_6th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq6)
{
    MODIFY_REG(ADCx->SQR2, ADC_SQR2_RSQ6, rsq6 << ADC_SQR2_RSQ6_POSS);
}

/**
  * @brief  Get 6th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_6th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR2, ADC_SQR2_RSQ6)) >> ADC_SQR2_RSQ6_POSS);
}

/**
  * @brief  Set 5th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq5
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_5th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq5)
{
    MODIFY_REG(ADCx->SQR2, ADC_SQR2_RSQ5, rsq5 << ADC_SQR2_RSQ5_POSS);
}

/**
  * @brief  Get 5th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_5th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR2, ADC_SQR2_RSQ5)) >> ADC_SQR2_RSQ5_POSS);
}

/**
  * @}
  */

/** @defgroup SQR3
  * @{
  */

/**
  * @brief  Set ADC_SQR3 Register
  * @param  ADCx ADC Instance
  * @param  sqr3
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sqr3(ADC_TypeDef *ADCx, uint32_t sqr3)
{
    WRITE_REG(ADCx->SQR3, sqr3);
}

/**
  * @brief  Get ADC_SQR3 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_sqr3(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->SQR3));
}

/**
  * @brief  Set 14th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq14
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_14th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq14)
{
    MODIFY_REG(ADCx->SQR3, ADC_SQR3_RSQ14, rsq14 << ADC_SQR3_RSQ14_POSS);
}

/**
  * @brief  Get 14th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_14th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR3, ADC_SQR3_RSQ14)) >> ADC_SQR3_RSQ14_POSS);
}

/**
  * @brief  Set 13th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq13
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_13th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq13)
{
    MODIFY_REG(ADCx->SQR3, ADC_SQR3_RSQ13, rsq13 << ADC_SQR3_RSQ13_POSS);
}

/**
  * @brief  Get 13th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_13th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR3, ADC_SQR3_RSQ13)) >> ADC_SQR3_RSQ13_POSS);
}

/**
  * @brief  Set 12th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq12
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_12th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq12)
{
    MODIFY_REG(ADCx->SQR3, ADC_SQR3_RSQ12, rsq12 << ADC_SQR3_RSQ12_POSS);
}

/**
  * @brief  Get 12th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_12th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR3, ADC_SQR3_RSQ12)) >> ADC_SQR3_RSQ12_POSS);
}

/**
  * @brief  Set 11th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq11
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_11th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq11)
{
    MODIFY_REG(ADCx->SQR3, ADC_SQR3_RSQ11, rsq11 << ADC_SQR3_RSQ11_POSS);
}

/**
  * @brief  Get 11th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_11th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR3, ADC_SQR3_RSQ11)) >> ADC_SQR3_RSQ11_POSS);
}

/**
  * @brief  Set 10th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq10
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_10th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq10)
{
    MODIFY_REG(ADCx->SQR3, ADC_SQR3_RSQ10, rsq10 << ADC_SQR3_RSQ10_POSS);
}

/**
  * @brief  Get 10th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_10th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR3, ADC_SQR3_RSQ10)) >> ADC_SQR3_RSQ10_POSS);
}

/**
  * @}
  */

/** @defgroup SQR4
  * @{
  */

/**
  * @brief  Set ADC_SQR4 Register
  * @param  ADCx ADC Instance
  * @param  sqr4
  * @retval None
  */
__STATIC_INLINE void md_adc_set_sqr4(ADC_TypeDef *ADCx, uint32_t sqr4)
{
    WRITE_REG(ADCx->SQR4, sqr4);
}

/**
  * @brief  Get ADC_SQR4 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_sqr4(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->SQR4));
}

/**
  * @brief  Set 16th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq16
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_16th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq16)
{
    MODIFY_REG(ADCx->SQR4, ADC_SQR4_RSQ16, rsq16 << ADC_SQR4_RSQ16_POSS);
}

/**
  * @brief  Get 16th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_16th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR4, ADC_SQR4_RSQ16)) >> ADC_SQR4_RSQ16_POSS);
}

/**
  * @brief  Set 15th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @param  rsq15
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_regular_sequence_selection_15th(ADC_TypeDef *ADCx, md_adc_channel_list_t rsq15)
{
    MODIFY_REG(ADCx->SQR4, ADC_SQR4_RSQ15, rsq15 << ADC_SQR4_RSQ15_POSS);
}

/**
  * @brief  Get 15th conversion in regular sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_regular_sequence_selection_15th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->SQR4, ADC_SQR4_RSQ15)) >> ADC_SQR4_RSQ15_POSS);
}

/**
  * @}
  */

/** @defgroup DR
  * @{
  */

/**
  * @brief  Get ADC_DR Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_regular_data(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->DR, ADC_DR_RDATA));
}

/**
  * @}
  */

/** @defgroup JSQR
  * @{
  */

/**
  * @brief  Set ADC_JSQR Register
  * @param  ADCx ADC Instance
  * @param  jsqr
  * @retval None
  */
__STATIC_INLINE void md_adc_set_jsqr(ADC_TypeDef *ADCx, uint32_t jsqr)
{
    WRITE_REG(ADCx->JSQR, jsqr);
}

/**
  * @brief  Get ADC_JSQR Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_jsqr(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->JSQR));
}

/**
  * @brief  Set 4th conversion in the injected sequence.
  * @param  ADCx ADC Instance
  * @param  jsq4
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_injected_sequence_selection_4th(ADC_TypeDef *ADCx, md_adc_channel_list_t jsq4)
{
    MODIFY_REG(ADCx->JSQR, ADC_JSQR_JSQ4, jsq4 << ADC_JSQR_JSQ4_POSS);
}

/**
  * @brief  Get 4th conversion in the injected sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_injected_sequence_selection_4th(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->JSQR, ADC_JSQR_JSQ4)) >> ADC_JSQR_JSQ4_POSS);
}

/**
  * @brief  Set 3rd conversion in the injected sequence.
  * @param  ADCx ADC Instance
  * @param  jsq3
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_injected_sequence_selection_3rd(ADC_TypeDef *ADCx, md_adc_channel_list_t jsq3)
{
    MODIFY_REG(ADCx->JSQR, ADC_JSQR_JSQ3, jsq3 << ADC_JSQR_JSQ3_POSS);
}

/**
  * @brief  Get 3rd conversion in the injected sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_injected_sequence_selection_3rd(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->JSQR, ADC_JSQR_JSQ3)) >> ADC_JSQR_JSQ3_POSS);
}

/**
  * @brief  Set 2nd conversion in the injected sequence.
  * @param  ADCx ADC Instance
  * @param  jsq2
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_injected_sequence_selection_2nd(ADC_TypeDef *ADCx, md_adc_channel_list_t jsq2)
{
    MODIFY_REG(ADCx->JSQR, ADC_JSQR_JSQ2, jsq2 << ADC_JSQR_JSQ2_POSS);
}

/**
  * @brief  Get 2nd conversion in the injected sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_injected_sequence_selection_2nd(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->JSQR, ADC_JSQR_JSQ2)) >> ADC_JSQR_JSQ2_POSS);
}

/**
  * @brief  Set 1st conversion in the injected sequence.
  * @param  ADCx ADC Instance
  * @param  jsq1
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_injected_sequence_selection_1st(ADC_TypeDef *ADCx, md_adc_channel_list_t jsq1)
{
    MODIFY_REG(ADCx->JSQR, ADC_JSQR_JSQ1, jsq1 << ADC_JSQR_JSQ1_POSS);
}

/**
  * @brief  Get 1st conversion in the injected sequence.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_injected_sequence_selection_1st(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)((READ_BIT(ADCx->JSQR, ADC_JSQR_JSQ1)) >> ADC_JSQR_JSQ1_POSS);
}

/**
  * @brief  Set External Trigger Enable and Polarity Selection for injected channels.
  * @param  ADCx ADC Instance
  * @param  iets
            @arg @ref MD_ADC_JSQR_TRIG_SOFTWARE
            @arg @ref MD_ADC_JSQR_TRIG_EXT_RISING
            @arg @ref MD_ADC_JSQR_TRIG_EXT_FALLING
            @arg @ref MD_ADC_JSQR_TRIG_EXT_BOTH
  * @retval None
  */
__STATIC_INLINE void md_adc_set_injected_trigger_mode(ADC_TypeDef *ADCx, md_adc_inject_trigger_mode_list_t iets)
{
    MODIFY_REG(ADCx->JSQR, ADC_JSQR_JEXTEN, iets << ADC_JSQR_JEXTEN_POSS);
}

/**
  * @brief  Get External Trigger Enable and Polarity Selection for injected channels.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_JSQR_TRIG_SOFTWARE
            @arg @ref MD_ADC_JSQR_TRIG_EXT_RISING
            @arg @ref MD_ADC_JSQR_TRIG_EXT_FALLING
            @arg @ref MD_ADC_JSQR_TRIG_EXT_BOTH
  */
__STATIC_INLINE md_adc_inject_trigger_mode_list_t md_adc_get_injected_trigger_mode(ADC_TypeDef *ADCx)
{
    return (md_adc_inject_trigger_mode_list_t)((READ_BIT(ADCx->JSQR, ADC_JSQR_JEXTEN)) >> ADC_JSQR_JEXTEN_POSS);
}

/**
  * @brief  Set External Trigger Selection for injected group.
  * @param  ADCx ADC Instance
  * @param  iextsel
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_CH4
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_CH1
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_CH4
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_CH2
            @arg @ref MD_ADC_JSQR_JEXT_EXTI_ADTE2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_CH4
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_TRGOUT2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_TRGOUT2
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_CH3
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_CH1
            @arg @ref MD_ADC_JSQR_JEXT_BS16T_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T1_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T1_CH1
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T1_CH2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_CH1
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_CH2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_CH3
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T2_CH2
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T2_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_CH2
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_CH3
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_CH4
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T2_CH1
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_CH1
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_CH2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_CH3
  * @retval None
  */
__STATIC_INLINE void md_adc_set_injected_external_trigger_source(ADC_TypeDef *ADCx, md_adc_inject_trigger_source_t iextsel)
{
    MODIFY_REG(ADCx->JSQR, ADC_JSQR_JEXTSEL, iextsel << ADC_JSQR_JEXTSEL_POSS);
}

/**
  * @brief  Get External Trigger Selection for injected group.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_CH4
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_CH1
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_CH4
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_CH2
            @arg @ref MD_ADC_JSQR_JEXT_EXTI_ADTE2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_CH4
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_TRGOUT2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_TRGOUT2
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_CH3
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T2_CH1
            @arg @ref MD_ADC_JSQR_JEXT_BS16T_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T1_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T1_CH1
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T1_CH2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_CH1
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_CH2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T2_CH3
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T2_CH2
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T2_TRGOUT
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_CH2
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_CH3
            @arg @ref MD_ADC_JSQR_JEXT_GP32C4T1_CH4
            @arg @ref MD_ADC_JSQR_JEXT_GP16C2T2_CH1
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_CH1
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_CH2
            @arg @ref MD_ADC_JSQR_JEXT_AD16C6T1_CH3
  */
__STATIC_INLINE md_adc_inject_trigger_source_t md_adc_get_injected_external_trigger_source(ADC_TypeDef *ADCx)
{
    return (md_adc_inject_trigger_source_t)((READ_BIT(ADCx->JSQR, ADC_JSQR_JEXTSEL)) >> ADC_JSQR_JEXTSEL_POSS);
}

/**
  * @brief  Set Injected channel sequence length.
  * @param  ADCx ADC Instance
  * @param  isl
            @arg @ref MD_ADC_JSQLEN_1
            @arg @ref MD_ADC_JSQLEN_2
            @arg @ref MD_ADC_JSQLEN_3
            @arg @ref MD_ADC_JSQLEN_4
  * @retval None
  */
__STATIC_INLINE void md_adc_set_injected_sequence_length(ADC_TypeDef *ADCx, md_adc_inject_length_t isl)
{
    MODIFY_REG(ADCx->JSQR, ADC_JSQR_JSQLEN, isl << ADC_JSQR_JSQLEN_POSS);
}

/**
  * @brief  Get Injected channel sequence length.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_JSQLEN_1
            @arg @ref MD_ADC_JSQLEN_2
            @arg @ref MD_ADC_JSQLEN_3
            @arg @ref MD_ADC_JSQLEN_4
  */
__STATIC_INLINE md_adc_inject_length_t md_adc_get_injected_sequence_length(ADC_TypeDef *ADCx)
{
    return (md_adc_inject_length_t)((READ_BIT(ADCx->JSQR, ADC_JSQR_JSQLEN)) >> ADC_JSQR_JSQLEN_POSS);
}

/**
  * @}
  */

/** @defgroup OFST1
  * @{
  */

/**
  * @brief  Set ADC_OFST1 Register
  * @param  ADCx ADC Instance
  * @param  ofst1
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ofst1(ADC_TypeDef *ADCx, uint32_t ofst1)
{
    WRITE_REG(ADCx->OFST1, ofst1);
}

/**
  * @brief  Get ADC_OFST1 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_ofst1(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->OFST1));
}

/**
  * @brief  Offset enable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_offset_compensation_1(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->OFST1, ADC_OFST1_OFFSETEN);
}

/**
  * @brief  Offset Disable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_offset_compensation_1(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->OFST1, ADC_OFST1_OFFSETEN);
}

/**
  * @brief  Check if Offset is enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disableb_offset_compensation_1(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->OFST1, ADC_OFST1_OFFSETEN) == (ADC_OFST1_OFFSETEN));
}

/**
  * @brief  Set Channel selection for the data offset.
  * @param  ADCx ADC Instance
  * @param  offch
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_channel_selection_1(ADC_TypeDef *ADCx, md_adc_channel_list_t offch)
{
    MODIFY_REG(ADCx->OFST1, ADC_OFST1_OFFSETCH, offch << ADC_OFST1_OFFSETCH_POSS);
}

/**
  * @brief  Get Channel selection for the data offset.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_compensation_channel_selection_1(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)(READ_BIT(ADCx->OFST1, ADC_OFST1_OFFSETCH) >> ADC_OFST1_OFFSETCH_POSS);
}

/**
  * @brief  Saturation enable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_compensation_saturation_1(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->OFST1, ADC_OFST1_SATEN);
}

/**
  * @brief  Saturation Disable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_compensation_saturation_1(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->OFST1, ADC_OFST1_SATEN);
}

/**
  * @brief  Check if Saturation is enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disabled_compensation_saturation_1(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->OFST1, ADC_OFST1_SATEN) == (ADC_OFST1_SATEN));
}

/**
  * @brief  Set Positive offset.
  * @param  ADCx ADC Instance
  * @param  offpen
            @arg @ref MD_ADC_NEGATIVE_OFFSET
            @arg @ref MD_ADC_POSITIVE_OFFSET
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_polarity_1(ADC_TypeDef *ADCx, md_adc_offset_direction_t offpen)
{
    MODIFY_REG(ADCx->OFST1, ADC_OFST1_POSEN, offpen << ADC_OFST1_POSEN_POS);
}

/**
  * @brief  Get Positive offset.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_NEGATIVE_OFFSET
            @arg @ref MD_ADC_POSITIVE_OFFSET
  */
__STATIC_INLINE md_adc_offset_direction_t md_adc_get_compensation_polarity_1(ADC_TypeDef *ADCx)
{
    return (md_adc_offset_direction_t)((READ_BIT(ADCx->OFST1, ADC_OFST1_POSEN)) >> ADC_OFST1_POSEN_POS);
}

/**
  * @brief  Set Data offset y for the channel programmed into bits OFFCH[4:0].
  * @param  ADCx ADC Instance
  * @param  off
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_value_1(ADC_TypeDef *ADCx, uint32_t off)
{
    MODIFY_REG(ADCx->OFST1, ADC_OFST1_OFFSET, off << ADC_OFST1_OFFSET_POSS);
}

/**
  * @brief  Get Data offset y for the channel programmed into bits OFFCH[4:0].
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_compensation_value_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->OFST1, ADC_OFST1_OFFSET)) >> ADC_OFST1_OFFSET_POSS);
}

/**
  * @}
  */

/** @defgroup OFST2
  * @{
  */

/**
  * @brief  Set ADC_OFST2 Register
  * @param  ADCx ADC Instance
  * @param  ofst2
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ofst2(ADC_TypeDef *ADCx, uint32_t ofst2)
{
    WRITE_REG(ADCx->OFST2, ofst2);
}

/**
  * @brief  Get ADC_OFST2 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_ofst2(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->OFST2));
}

/**
  * @brief  Offset enable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_offset_compensation_2(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->OFST2, ADC_OFST2_OFFSETEN);
}

/**
  * @brief  Offset Disable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_offset_compensation_2(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->OFST2, ADC_OFST2_OFFSETEN);
}

/**
  * @brief  Check if Offset is enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disableb_offset_compensation_2(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->OFST2, ADC_OFST2_OFFSETEN) == (ADC_OFST2_OFFSETEN));
}

/**
  * @brief  Set Channel selection for the data offset.
  * @param  ADCx ADC Instance
  * @param  offch
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_channel_selection_2(ADC_TypeDef *ADCx, md_adc_channel_list_t offch)
{
    MODIFY_REG(ADCx->OFST2, ADC_OFST2_OFFSETCH, offch << ADC_OFST2_OFFSETCH_POSS);
}

/**
  * @brief  Get Channel selection for the data offset.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_compensation_channel_selection_2(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)(READ_BIT(ADCx->OFST2, ADC_OFST2_OFFSETCH) >> ADC_OFST2_OFFSETCH_POSS);
}

/**
  * @brief  Saturation enable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_compensation_saturation_2(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->OFST2, ADC_OFST2_SATEN);
}

/**
  * @brief  Saturation Disable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_compensation_saturation_2(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->OFST2, ADC_OFST2_SATEN);
}

/**
  * @brief  Check if Saturation is enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disabled_compensation_saturation_2(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->OFST2, ADC_OFST2_SATEN) == (ADC_OFST2_SATEN));
}

/**
  * @brief  Set Positive offset.
  * @param  ADCx ADC Instance
  * @param  offpen
            @arg @ref MD_ADC_NEGATIVE_OFFSET
            @arg @ref MD_ADC_POSITIVE_OFFSET
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_polarity_2(ADC_TypeDef *ADCx, md_adc_offset_direction_t offpen)
{
    MODIFY_REG(ADCx->OFST2, ADC_OFST2_POSEN, offpen << ADC_OFST2_POSEN_POS);
}

/**
  * @brief  Get Positive offset.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_NEGATIVE_OFFSET
            @arg @ref MD_ADC_POSITIVE_OFFSET
  */
__STATIC_INLINE md_adc_offset_direction_t md_adc_get_compensation_polarity_2(ADC_TypeDef *ADCx)
{
    return (md_adc_offset_direction_t)((READ_BIT(ADCx->OFST2, ADC_OFST2_POSEN)) >> ADC_OFST2_POSEN_POS);
}

/**
  * @brief  Set Data offset y for the channel programmed into bits OFFCH[4:0].
  * @param  ADCx ADC Instance
  * @param  off
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_value_2(ADC_TypeDef *ADCx, uint32_t off)
{
    MODIFY_REG(ADCx->OFST2, ADC_OFST2_OFFSET, off << ADC_OFST2_OFFSET_POSS);
}

/**
  * @brief  Get Data offset y for the channel programmed into bits OFFCH[4:0].
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_compensation_value_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->OFST2, ADC_OFST2_OFFSET)) >> ADC_OFST2_OFFSET_POSS);
}

/**
  * @}
  */

/** @defgroup OFST3
  * @{
  */

/**
  * @brief  Set ADC_OFST3 Register
  * @param  ADCx ADC Instance
  * @param  ofst3
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ofst3(ADC_TypeDef *ADCx, uint32_t ofst3)
{
    WRITE_REG(ADCx->OFST3, ofst3);
}

/**
  * @brief  Get ADC_OFST3 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_ofst3(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->OFST3));
}

/**
  * @brief  Offset enable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_offset_compensation_3(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->OFST3, ADC_OFST3_OFFSETEN);
}

/**
  * @brief  Offset Disable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_offset_compensation_3(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->OFST3, ADC_OFST3_OFFSETEN);
}

/**
  * @brief  Check if Offset is enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disableb_offset_compensation_3(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->OFST3, ADC_OFST3_OFFSETEN) == (ADC_OFST3_OFFSETEN));
}

/**
  * @brief  Set Channel selection for the data offset.
  * @param  ADCx ADC Instance
  * @param  offch
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_channel_selection_3(ADC_TypeDef *ADCx, md_adc_channel_list_t offch)
{
    MODIFY_REG(ADCx->OFST3, ADC_OFST3_OFFSETCH, offch << ADC_OFST3_OFFSETCH_POSS);
}

/**
  * @brief  Get Channel selection for the data offset.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_compensation_channel_selection_3(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)(READ_BIT(ADCx->OFST3, ADC_OFST3_OFFSETCH) >> ADC_OFST3_OFFSETCH_POSS);
}

/**
  * @brief  Saturation enable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_compensation_saturation_3(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->OFST3, ADC_OFST3_SATEN);
}

/**
  * @brief  Saturation Disable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_compensation_saturation_3(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->OFST3, ADC_OFST3_SATEN);
}

/**
  * @brief  Check if Saturation is enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disabled_compensation_saturation_3(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->OFST3, ADC_OFST3_SATEN) == (ADC_OFST3_SATEN));
}

/**
  * @brief  Set Positive offset.
  * @param  ADCx ADC Instance
  * @param  offpen
            @arg @ref MD_ADC_NEGATIVE_OFFSET
            @arg @ref MD_ADC_POSITIVE_OFFSET
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_polarity_3(ADC_TypeDef *ADCx, md_adc_offset_direction_t offpen)
{
    MODIFY_REG(ADCx->OFST3, ADC_OFST3_POSEN, offpen << ADC_OFST3_POSEN_POS);
}

/**
  * @brief  Get Positive offset.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_NEGATIVE_OFFSET
            @arg @ref MD_ADC_POSITIVE_OFFSET
  */
__STATIC_INLINE md_adc_offset_direction_t md_adc_get_compensation_polarity_3(ADC_TypeDef *ADCx)
{
    return (md_adc_offset_direction_t)((READ_BIT(ADCx->OFST3, ADC_OFST3_POSEN)) >> ADC_OFST3_POSEN_POS);
}

/**
  * @brief  Set Data offset y for the channel programmed into bits OFFCH[4:0].
  * @param  ADCx ADC Instance
  * @param  off
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_value_3(ADC_TypeDef *ADCx, uint32_t off)
{
    MODIFY_REG(ADCx->OFST3, ADC_OFST3_OFFSET, off << ADC_OFST3_OFFSET_POSS);
}

/**
  * @brief  Get Data offset y for the channel programmed into bits OFFCH[4:0].
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_compensation_value_3(ADC_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->OFST3, ADC_OFST3_OFFSET)) >> ADC_OFST3_OFFSET_POSS);
}

/**
  * @}
  */

/** @defgroup OFST4
  * @{
  */

/**
  * @brief  Set ADC_OFST4 Register
  * @param  ADCx ADC Instance
  * @param  ofst4
  * @retval None
  */
__STATIC_INLINE void md_adc_set_ofst4(ADC_TypeDef *ADCx, uint32_t ofst4)
{
    WRITE_REG(ADCx->OFST4, ofst4);
}

/**
  * @brief  Get ADC_OFST4 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_ofst4(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_REG(ADCx->OFST4));
}

/**
  * @brief  Offset enable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_offset_compensation_4(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->OFST4, ADC_OFST4_OFFSETEN);
}

/**
  * @brief  Offset Disable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_offset_compensation_4(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->OFST4, ADC_OFST4_OFFSETEN);
}

/**
  * @brief  Check if Offset is enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disableb_offset_compensation_4(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->OFST4, ADC_OFST4_OFFSETEN) == (ADC_OFST4_OFFSETEN));
}

/**
  * @brief  Set Channel selection for the data offset.
  * @param  ADCx ADC Instance
  * @param  offch
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_channel_selection_4(ADC_TypeDef *ADCx, md_adc_channel_list_t offch)
{
    MODIFY_REG(ADCx->OFST4, ADC_OFST4_OFFSETCH, offch << ADC_OFST4_OFFSETCH_POSS);
}

/**
  * @brief  Get Channel selection for the data offset.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CH0
            @arg @ref MD_ADC_CH1
            @arg @ref MD_ADC_CH2
            @arg @ref MD_ADC_CH3
            @arg @ref MD_ADC_CH4
            @arg @ref MD_ADC_CH5
            @arg @ref MD_ADC_CH6
            @arg @ref MD_ADC_CH7
            @arg @ref MD_ADC_CH8
            @arg @ref MD_ADC_CH9
            @arg @ref MD_ADC_CH10
            @arg @ref MD_ADC_CH11
            @arg @ref MD_ADC_CH12
            @arg @ref MD_ADC_CH13
            @arg @ref MD_ADC_CH14
            @arg @ref MD_ADC_CH15
            @arg @ref MD_ADC_CH16
            @arg @ref MD_ADC_CH17
            @arg @ref MD_ADC_CH18
  */
__STATIC_INLINE md_adc_channel_list_t md_adc_get_compensation_channel_selection_4(ADC_TypeDef *ADCx)
{
    return (md_adc_channel_list_t)(READ_BIT(ADCx->OFST4, ADC_OFST4_OFFSETCH) >> ADC_OFST4_OFFSETCH_POSS);
}

/**
  * @brief  Saturation enable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_enable_compensation_saturation_4(ADC_TypeDef *ADCx)
{
    SET_BIT(ADCx->OFST4, ADC_OFST4_SATEN);
}

/**
  * @brief  Saturation Disable.
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE void md_adc_disable_compensation_saturation_4(ADC_TypeDef *ADCx)
{
    CLEAR_BIT(ADCx->OFST4, ADC_OFST4_SATEN);
}

/**
  * @brief  Check if Saturation is enabled.
  * @param  ADCx ADC Instance
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_adc_is_disabled_compensation_saturation_4(ADC_TypeDef *ADCx)
{
    return (READ_BIT(ADCx->OFST4, ADC_OFST4_SATEN) == (ADC_OFST4_SATEN));
}

/**
  * @brief  Set Positive offset.
  * @param  ADCx ADC Instance
  * @param  offpen
            @arg @ref MD_ADC_NEGATIVE_OFFSET
            @arg @ref MD_ADC_POSITIVE_OFFSET
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_polarity_4(ADC_TypeDef *ADCx, md_adc_offset_direction_t offpen)
{
    MODIFY_REG(ADCx->OFST4, ADC_OFST4_POSEN, offpen << ADC_OFST4_POSEN_POS);
}

/**
  * @brief  Get Positive offset.
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_NEGATIVE_OFFSET
            @arg @ref MD_ADC_POSITIVE_OFFSET
  */
__STATIC_INLINE md_adc_offset_direction_t md_adc_get_compensation_polarity_4(ADC_TypeDef *ADCx)
{
    return (md_adc_offset_direction_t)((READ_BIT(ADCx->OFST4, ADC_OFST4_POSEN)) >> ADC_OFST4_POSEN_POS);
}

/**
  * @brief  Set Data offset y for the channel programmed into bits OFFCH[4:0].
  * @param  ADCx ADC Instance
  * @param  off
  * @retval None
  */
__STATIC_INLINE void md_adc_set_compensation_value_4(ADC_TypeDef *ADCx, uint32_t off)
{
    MODIFY_REG(ADCx->OFST4, ADC_OFST4_OFFSET, off << ADC_OFST4_OFFSET_POSS);
}

/**
  * @brief  Get Data offset y for the channel programmed into bits OFFCH[4:0].
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_compensation_value_4(ADC_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->OFST4, ADC_OFST4_OFFSET)) >> ADC_OFST4_OFFSET_POSS);
}

/**
  * @}
  */

/** @defgroup JDR1
  * @{
  */

/**
  * @brief  Get ADC_JDR1 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_injected_data_1(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->JDR1, ADC_JDR1_JDATA1));
}

/**
  * @}
  */

/** @defgroup JDR2
  * @{
  */

/**
  * @brief  Get ADC_JDR2 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_injected_data_2(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->JDR2, ADC_JDR2_JDATA2));
}

/**
  * @}
  */

/** @defgroup JDR3
  * @{
  */

/**
  * @brief  Get ADC_JDR3 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_injected_data_3(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->JDR3, ADC_JDR3_JDATA3));
}

/**
  * @}
  */

/** @defgroup JDR4
  * @{
  */

/**
  * @brief  Get ADC_JDR4 Register
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_injected_data_4(ADC_TypeDef *ADCx)
{
    return (uint32_t)(READ_BIT(ADCx->JDR4, ADC_JDR4_JDATA4));
}

/**
  * @}
  */

/** @defgroup AWD2CR
  * @{
  */

/**
  * @brief  Set Analog Watchdog 2 Channel Selector Switch
  * @param  ADCx ADC Instance
  * @param  ch
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd2_channel(ADC_TypeDef *ADCx, uint32_t ch)
{
    MODIFY_REG(ADCx->AWD2CR, ADC_AWD2CR_AWD2CH, ch << ADC_AWD2CR_AWD2CH_POSS);
}

/**
  * @brief  Get Analog Watchdog 2 Channel Selector Switch
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_awd2_channel(ADC_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->AWD2CR, ADC_AWD2CR_AWD2CH)) >> ADC_AWD2CR_AWD2CH_POSS);
}

/**
  * @}
  */

/** @defgroup AWD3CR
  * @{
  */

/**
  * @brief  Set Analog Watchdog 3 Channel Selector Switch
  * @param  ADCx ADC Instance
  * @param  ch
  * @retval None
  */
__STATIC_INLINE void md_adc_set_awd3_channel(ADC_TypeDef *ADCx, uint32_t ch)
{
    MODIFY_REG(ADCx->AWD3CR, ADC_AWD3CR_AWD3CH, ch << ADC_AWD3CR_AWD3CH_POSS);
}

/**
  * @brief  Get Analog Watchdog 3 Channel Selector Switch
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_awd3_channel(ADC_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->AWD3CR, ADC_AWD3CR_AWD3CH)) >> ADC_AWD3CR_AWD3CH_POSS);
}

/**
  * @}
  */

/** @defgroup GCOMP
  * @{
  */

/**
  * @brief  Set Gain Compensation Coefficient
  * @param  ADCx ADC Instance
  * @param  coefficient
  * @retval None
  */
__STATIC_INLINE void md_adc_set_gain_coefficient(ADC_TypeDef *ADCx, uint32_t coefficient)
{
    MODIFY_REG(ADCx->GCOMP, ADC_GCOMP_COEF, coefficient << ADC_GCOMP_COEF_POSS);
}

/**
  * @brief  Get Gain Compensation Coefficient
  * @param  ADCx ADC Instance
  * @retval None
  */
__STATIC_INLINE uint32_t md_adc_get_gain_coefficient(ADC_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->GCOMP, ADC_GCOMP_COEF)) >> ADC_GCOMP_COEF_POSS);
}

/**
  * @}
  */

/** @defgroup CCR
  * @{
  */

/**
  * @brief  Set ADC_CCR Register
  * @param  ADCx ADC Instance
  * @param  ccr
  */
__STATIC_INLINE void md_adc_set_ccr(ADC_COMM_TypeDef *ADCx, uint32_t ccr)
{
    WRITE_REG(ADCx->CCR, ccr);
}

/**
  * @brief  Get ADC_CCR Register
  * @param  ADCx ADC Instance
  * @retval ccr Values
  */
__STATIC_INLINE uint32_t md_adc_get_ccr(ADC_COMM_TypeDef *ADCx)
{
    return (uint32_t)READ_REG(ADCx->CCR);
}

/**
  * @brief  Set ADC Clock Divider Configuration
  * @param  ADCx ADC Instance
  * @param  divider
            @arg @ref MD_ADC_CCR_PRESCALE_1
            @arg @ref MD_ADC_CCR_PRESCALE_2
            @arg @ref MD_ADC_CCR_PRESCALE_4
            @arg @ref MD_ADC_CCR_PRESCALE_8
            @arg @ref MD_ADC_CCR_PRESCALE_10
            @arg @ref MD_ADC_CCR_PRESCALE_12
            @arg @ref MD_ADC_CCR_PRESCALE_16
            @arg @ref MD_ADC_CCR_PRESCALE_32
            @arg @ref MD_ADC_CCR_PRESCALE_64
            @arg @ref MD_ADC_CCR_PRESCALE_128
            @arg @ref MD_ADC_CCR_PRESCALE_256
  * @retval None
  */
__STATIC_INLINE void md_adc_set_clock_divider(ADC_COMM_TypeDef *ADCx, md_adc_kernal_clock_prescale_t divider)
{
    MODIFY_REG(ADCx->CCR, ADC_CCR_PRESCALE, divider << ADC_CCR_PRESCALE_POSS);
}

/**
  * @brief  Get ADC Clock Divider Configuration
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CCR_PRESCALE_1
            @arg @ref MD_ADC_CCR_PRESCALE_2
            @arg @ref MD_ADC_CCR_PRESCALE_4
            @arg @ref MD_ADC_CCR_PRESCALE_8
            @arg @ref MD_ADC_CCR_PRESCALE_10
            @arg @ref MD_ADC_CCR_PRESCALE_12
            @arg @ref MD_ADC_CCR_PRESCALE_16
            @arg @ref MD_ADC_CCR_PRESCALE_32
            @arg @ref MD_ADC_CCR_PRESCALE_64
            @arg @ref MD_ADC_CCR_PRESCALE_128
            @arg @ref MD_ADC_CCR_PRESCALE_256
  */
__STATIC_INLINE md_adc_kernal_clock_prescale_t md_adc_get_clock_divider(ADC_COMM_TypeDef *ADCx)
{
    return (md_adc_kernal_clock_prescale_t)((READ_BIT(ADCx->CCR, ADC_CCR_PRESCALE)) >> ADC_CCR_PRESCALE_POSS);
}

/**
  * @brief  Set ADC Clock Divider Configuration
  * @param  ADCx ADC Instance
  * @param  divider
            @arg @ref MD_ADC_CCR_DMA_DISABLE
            @arg @ref MD_ADC_CCR_DMA_12_AND_10_RESOLUTION
            @arg @ref MD_ADC_CCR_DMA_8_AND_6_RESOLUTION
  * @retval None
  */
__STATIC_INLINE void md_adc_set_dma_mode(ADC_COMM_TypeDef *ADCx, md_adc_dualmode_dma_mode_t divider)
{
    MODIFY_REG(ADCx->CCR, ADC_CCR_DMAMODE, divider << ADC_CCR_DMAMODE_POSS);
}

/**
  * @brief  Get ADC Clock Divider Configuration
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
            @arg @ref MD_ADC_CCR_DMA_DISABLE
            @arg @ref MD_ADC_CCR_DMA_12_AND_10_RESOLUTION
            @arg @ref MD_ADC_CCR_DMA_8_AND_6_RESOLUTION
  */
__STATIC_INLINE md_adc_dualmode_dma_mode_t md_adc_get_dma_mode(ADC_COMM_TypeDef *ADCx)
{
    return (md_adc_dualmode_dma_mode_t)((READ_BIT(ADCx->CCR, ADC_CCR_DMAMODE)) >> ADC_CCR_DMAMODE_POSS);
}

/**
  * @brief  Set Delay register between 2 samples
  * @param  ADCx ADC Instance
  * @param  time
  * @retval None
  */
__STATIC_INLINE void md_adc_set_delay_time(ADC_COMM_TypeDef *ADCx, uint32_t time)
{
    MODIFY_REG(ADCx->CCR, ADC_CCR_DELAYSEL, time << ADC_CCR_DELAYSEL_POSS);
}

/**
  * @brief  Get Delay register between 2 samples
  * @param  ADCx ADC Instance
  * @retval DELAYSEL
  */
__STATIC_INLINE uint32_t md_adc_get_delay_time(ADC_COMM_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->CCR, ADC_CCR_DELAYSEL)) >> ADC_CCR_DELAYSEL_POSS);
}

/**
  * @brief  Set Dual ADC Mode Selection
  * @param  ADCx ADC Instance
  * @param  modes
  * @retval None
  */
__STATIC_INLINE void md_adc_set_dual_mode(ADC_COMM_TypeDef *ADCx, md_adc_dual_mode_t mode)
{
    MODIFY_REG(ADCx->CCR, ADC_CCR_DUALMODE, mode << ADC_CCR_DUALMODE_POSS);
}

/**
  * @brief  Get Dual ADC Mode Selection
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE md_adc_dual_mode_t md_adc_get_dual_mode(ADC_COMM_TypeDef *ADCx)
{
    return (md_adc_dual_mode_t)((READ_BIT(ADCx->CCR, ADC_CCR_DUALMODE)) >> ADC_CCR_DUALMODE_POSS);
}

/**
  * @}
  */

/** @defgroup CDR
  * @{
  */

/**
  * @brief  Get ADC2 regular conversion data
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_adc2_regular_data(ADC_COMM_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->CDR, ADC_CDR_ADC2_RDATA)) >> ADC_CDR_ADC2_RDATA_POSS);
}

/**
  * @brief  Get ADC1 regular conversion data
  * @param  ADCx ADC Instance
  * @retval The retval can be one of the following values:
  */
__STATIC_INLINE uint32_t md_adc_get_adc1_regular_data(ADC_COMM_TypeDef *ADCx)
{
    return (uint32_t)((READ_BIT(ADCx->CDR, ADC_CDR_ADC1_RDATA)) >> ADC_CDR_ADC1_RDATA_POSS);
}
/**
  * @}
  */

/** @defgroup ADC_Function
  * @{
  */
static void md_adc_open(ADC_TypeDef *adc);
static void md_adc_close(ADC_TypeDef *adc);
void md_adc_init(ADC_TypeDef *adc, ADC_COMM_TypeDef *adc12_common, md_adc_init_typedef *adc_init);
void md_adc_input_channel_config(ADC_TypeDef *adc, md_adc_input_channel_config_typedef *config);
void md_adc_regular_trigger_config(ADC_TypeDef *adc, md_adc_regular_trigger_config_typedef *config);
void md_adc_inject_trigger_config(ADC_TypeDef *adc, md_adc_inject_trigger_config_typedef *config);
void md_adc_oversample_offset_config(ADC_TypeDef *adc, md_adc_oversample_offset_config_typedef *config);
void md_adc_watchdog_config(ADC_TypeDef *adc, md_adc_watchdog_config_typedef *config);
void md_adc_inject_sequence_reset(ADC_TypeDef *adc);
md_adc_sample_time_typedef md_adc_sampletime_calculate(ADC_TypeDef *adc, ADC_COMM_TypeDef *adc12_common, double sampletime);
/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */

/**
  * @}
  */


#ifdef __cplusplus
}
#endif /* __cplusplus */
#endif /* __MD_ADC_H__ */

/******************* (C) COPYRIGHT Eastsoft Microelectronics *****END OF FILE****/
