/**
  *********************************************************************************
  *
  * @file    md_pmu.c
  * @brief   Power management unit driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "md_conf.h"


/** @addtogroup Micro_Driver
  * @{
  */
#ifdef MD_PMU

/** @addtogroup MD_PMU
  * @{
  */

/** @addtogroup MD_PMU_Public_Functions
  * @{
  */
/** @addtogroup MD_PMU_Public_Functions_Group1
  * @{
  */
/**
  * @brief  Reset the PMU register
  * @retval None
  */
void md_pmu_reset(void)
{
    SYSCFG_UNLOCK();
    WRITE_REG(PMU->CR, 0x0);
    WRITE_REG(PMU->LVDCR, 0x0);
    WRITE_REG(PMU->PWRCR, 0xFF);
    SYSCFG_LOCK();

    return;
}

/**
  * @brief  Enter stop1 mode
  * @retval None
  */
void md_pmu_stop1_enter(void)
{
    SYSCFG_UNLOCK();
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, MD_PMU_LP_STOP1 << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Enter stop2 mode
  * @retval None
  */
void md_pmu_stop2_enter(void)
{
    SYSCFG_UNLOCK();
    SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, MD_PMU_LP_STOP2 << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Enter standby mode
  * @param  port: The port whick wake up the standby mode.
  * @param  level: Wakeup level.
  * @retval None
  */
void md_pmu_standby_enter(md_pmu_wakeup_port_t port, md_pmu_wakeup_level_t level)
{
    if (port == MD_PMU_STANDBY_PORT_SEL_NONE)
    {
        SYSCFG_UNLOCK();
        CLEAR_BIT(PMU->BKPCR0, PMU_BKPCR0_WKPEN_MSK);
        SYSCFG_LOCK();
        return;
    }

    SYSCFG_UNLOCK();
    SET_BIT(PMU->BKPCR0, PMU_BKPCR0_WKPEN_MSK);
    MODIFY_REG(PMU->BKPCR0, PMU_BKPCR0_WKPS_MSK, port << PMU_BKPCR0_WKPS_POSS);
    MODIFY_REG(PMU->BKPCR0, PMU_BKPCR0_WKPL_MSK, level << PMU_BKPCR0_WKPL_POS);
    SET_BIT(PMU->BKPCR0, PMU_BKPCR0_STBWKEN_MSK);

    SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, MD_PMU_LP_STANDBY << PMU_CR_LPM_POSS);
    SYSCFG_LOCK();

    SysTick->CTRL &= ~SysTick_CTRL_ENABLE_Msk;
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
    SysTick->CTRL |= SysTick_CTRL_ENABLE_Msk;

    return;
}

/**
  * @brief  Configure lvd using specified parameters.
  * @param  sel: LVD threshold voltage.
  * @param  mode: LVD trigger mode.
  * @param  state: New state, ENABLE/DISABLE;
  * @retval None
  */
void md_pmu_lvd_config(md_pmu_lvd_voltage_sel_t sel, md_pmu_lvd_trigger_mode_t mode, type_func_t state)
{
    SYSCFG_UNLOCK();

    if (state)
    {
        MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDS_MSK, sel << PMU_LVDCR_LVDS_POSS);
        MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDIFS_MSK, mode << PMU_LVDCR_LVDIFS_POSS);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
    }
    else
    {
        SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
        CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
        CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
    }

    SYSCFG_LOCK();
    return;
}

/**
  * @brief  Write data into backup ram.
  * @param  idx: Index of backup word.
  * @param  data: Data which will be written to backup ram.
  * @retval None
  */
void md_bkpc_write_ram(uint8_t idx, uint32_t data)
{
    SYSCFG_UNLOCK();
    PMU->BKPREG[idx] = data;
    SYSCFG_LOCK();

    return;
}

/**
  * @brief  Read data from backup ram.
  * @param  idx: Index of backup word.
  * @retval The data.
  */
uint32_t ald_bkpc_read_ram(uint8_t idx)
{
    return PMU->BKPREG[idx];
}
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
#endif
/**
 * @}
 */
