/**
  *********************************************************************************
  *
  * @file    md_syscfg.h
  * @brief   SYSCFG module driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __MD_SYSCFG_H__
#define __MD_SYSCFG_H__

#ifdef __cplusplus
 extern "C" {
#endif

#include "md_utils.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_SYSCFG SYSCFG
  * @brief SYSCFG module driver
  * @{
  */

/** @defgroup MD_SYSCFG_Public_Macros SYSCFG Public Macros
  * @{
  */
#define SYSCFG_LOCK()  		WRITE_REG(SYSCFG->PROT, 0x0U)
#define SYSCFG_UNLOCK()		WRITE_REG(SYSCFG->PROT, 0x55AA6996U)
#define GET_SYSCFG_LOCK()	READ_BIT(SYSCFG->PROT, SYSCFG_PROT_PROT_MSK)

#define SYSCFG_BOOTFLASH_MAPPING_ENABLE()	SET_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK)
#define SYSCFG_BOOTFLASH_MAPPING_DISABLE()	CLEAR_BIT(SYSCFG->MEMRMP, SYSCFG_MEMRMP_BFRMPEN_MSK)
/**
  * @}
  */
/** @defgroup MD_SYSCFG_Public_Functions SYSCFG Public Functions
  * @{
  */
/** @defgroup MD_SYSCFG_Public_Functions_Group1 VTOR
  * @{
  */
/**
  * @brief  Set the offset of the interrup vector map
  * @param  offset: Offset of the interrup vector map
  * @param  status: ENABLE/DISABLE
  * @retval None
  */
__STATIC_INLINE__ void md_vtor_config(uint32_t offset, type_func_t status)
{
	SCB->VTOR = status ? (offset & ~0x3FU) : 0;
	return;
}
/**
  * @}
  */
/** @defgroup MD_SYSCFG_Public_Functions_Group2 USBCFG
  * @{
  */
/**
  * @brief  USB LSB padding enable
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_enable_txlbse(void)
{
	SET_BIT(SYSCFG->USBCFG, SYSCFG_USBCFG_TXLBSE_MSK);
}

/**
  * @brief  USB LSB padding disable
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_disable_txlbse(void)
{
	CLEAR_BIT(SYSCFG->USBCFG, SYSCFG_USBCFG_TXLBSE_MSK);
}

/**
  * @brief  Whether USB LSB padding is enabled
  * @retval 1-enable, 0-disable
  */
__STATIC_INLINE uint32_t md_syscfg_usb_is_enable_txlbse(void)
{
	return READ_BITS(SYSCFG->USBCFG, SYSCFG_USBCFG_TXLBSE_MSK, SYSCFG_USBCFG_TXLBSE_POS);
}

/**
  * @brief  Set USB high-speed mode drive capability
  * @param  cap: Drive capability, [0-3]
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_set_hsdrv(uint8_t cap)
{
	MODIFY_REG(SYSCFG->USBCFG, SYSCFG_USBCFG_HSDRV_MSK, cap << SYSCFG_USBCFG_HSDRV_POSS);
}

/**
  * @brief  Get USB high-speed mode drive capability
  * @retval Drive capability, [0-3]
  */
__STATIC_INLINE uint32_t md_syscfg_usb_get_hsdrv(void)
{
	return READ_BITS(SYSCFG->USBCFG, SYSCFG_USBCFG_HSDRV_MSK, SYSCFG_USBCFG_HSDRV_POSS);
}

/**
  * @brief  Set USB M clock divider
  * @param  div: USB M clock divider, [0-15]
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_set_ckdivm(uint8_t div)
{
	MODIFY_REG(SYSCFG->USBCFG, SYSCFG_USBCFG_CKDIVM_MSK, div << SYSCFG_USBCFG_CKDIVM_POSS);
}

/**
  * @brief  Get USB M clock divider
  * @retval USB M clock divider, [0-15]
  */
__STATIC_INLINE uint32_t md_syscfg_usb_get_ckdivm(void)
{
	return READ_BITS(SYSCFG->USBCFG, SYSCFG_USBCFG_CKDIVM_MSK, SYSCFG_USBCFG_CKDIVM_POSS);
}

/**
  * @brief  Set USB N clock divider
  * @param  div: USB N clock divider, [0-63]
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_set_ckdivn(uint8_t div)
{
	MODIFY_REG(SYSCFG->USBCFG, SYSCFG_USBCFG_CKDIVN_MSK, div << SYSCFG_USBCFG_CKDIVN_POSS);
}

/**
  * @brief  Get USB N clock divider
  * @retval USB N clock divider, [0-63]
  */
__STATIC_INLINE uint32_t md_syscfg_usb_get_ckdivn(void)
{
	return READ_BITS(SYSCFG->USBCFG, SYSCFG_USBCFG_CKDIVN_MSK, SYSCFG_USBCFG_CKDIVN_POSS);
}

/**
  * @brief  Set USB current
  * @param  cur: Current
  *                0x0-30uA
  *                0x1-40uA
  *                0x2-50uA
  *                0x3-60uA
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_set_itrm(uint8_t cur)
{
	MODIFY_REG(SYSCFG->USBCFG, SYSCFG_USBCFG_ITRM_MSK, cur << SYSCFG_USBCFG_ITRM_POSS);
}

/**
  * @brief  Get USB current
  * @retval Current
  *          0x0-30uA
  *          0x1-40uA
  *          0x2-50uA
  *          0x3-60uA
  */
__STATIC_INLINE uint32_t md_syscfg_usb_get_itrm(void)
{
	return READ_BITS(SYSCFG->USBCFG, SYSCFG_USBCFG_ITRM_MSK, SYSCFG_USBCFG_ITRM_POSS);
}

/**
  * @brief  USB PHY clock enable
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_enable_cken(void)
{
	SET_BIT(SYSCFG->USBCFG, SYSCFG_USBCFG_CKSEL_MSK);
}

/**
  * @brief  USB PHY clock disable
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_disable_cken(void)
{
	CLEAR_BIT(SYSCFG->USBCFG, SYSCFG_USBCFG_CKSEL_MSK);
}

/**
  * @brief  Whether PHY clock is enabled
  * @retval 1-enable, 0-disable
  */
__STATIC_INLINE uint32_t md_syscfg_usb_is_enable_cken(void)
{
	return READ_BITS(SYSCFG->USBCFG, SYSCFG_USBCFG_CKSEL_MSK, SYSCFG_USBCFG_CKSEL_POS);
}

/**
  * @brief  USB PLL bypass flag enable
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_enable_clkrdybp(void)
{
	SET_BIT(SYSCFG->USBCFG, SYSCFG_USBCFG_CLKRDYBP_MSK);
}

/**
  * @brief  USB PLL bypass flag disable
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_disable_clkrdybp(void)
{
	CLEAR_BIT(SYSCFG->USBCFG, SYSCFG_USBCFG_CLKRDYBP_MSK);
}

/**
  * @brief  Whether PLL bypass flag is enabled
  * @retval 1-enable, 0-disable
  */
__STATIC_INLINE uint32_t md_syscfg_usb_is_enable_clkrdybp(void)
{
	return READ_BITS(SYSCFG->USBCFG, SYSCFG_USBCFG_CLKRDYBP_MSK, SYSCFG_USBCFG_CLKRDYBP_POS);
}

/**
  * @brief  Whether PLL is stable
  * @retval 1-enable, 0-disable
  */
__STATIC_INLINE uint32_t md_syscfg_usb_is_stable_clkrdy(void)
{
	return READ_BITS(SYSCFG->USBCFG, SYSCFG_USBCFG_CLKRDY_MSK, SYSCFG_USBCFG_CLKRDY_POS);
}
/**
  * @}
  */
#ifdef ES32F36xx
/** @defgroup MD_SYSCFG_Public_Functions_Group3 TBKCFG
  * @{
  */
/**
  * @brief  Enable clock safety event as the timer brake source
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_enable_cssbke(void)
{
	SET_BIT(SYSCFG->TBKCFG, SYSCFG_TBKCFG_CSSBKE_MSK);
}

/**
  * @brief  Disable clock safety event as the timer brake source
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_disable_cssbke(void)
{
	CLEAR_BIT(SYSCFG->TBKCFG, SYSCFG_TBKCFG_CSSBKE_MSK);
}

/**
  * @brief  Whether clock safety event is the timer brake source
  * @retval 1-enable, 0-disable
  */
__STATIC_INLINE uint32_t md_syscfg_usb_is_enable_cssbke(void)
{
	return READ_BITS(SYSCFG->TBKCFG, SYSCFG_TBKCFG_CSSBKE_MSK, SYSCFG_TBKCFG_CSSBKE_POS);
}

/**
  * @brief  Enable LVD event as the timer brake source
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_enable_lvdbke(void)
{
	SET_BIT(SYSCFG->TBKCFG, SYSCFG_TBKCFG_LVDBKE_MSK);
}

/**
  * @brief  Disable LVD event as the timer brake source
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_disable_lvdbke(void)
{
	CLEAR_BIT(SYSCFG->TBKCFG, SYSCFG_TBKCFG_LVDBKE_MSK);
}

/**
  * @brief  Whether LVD event is the timer brake source
  * @retval 1-enable, 0-disable
  */
__STATIC_INLINE uint32_t md_syscfg_usb_is_enable_lvdbke(void)
{
	return READ_BITS(SYSCFG->TBKCFG, SYSCFG_TBKCFG_LVDBKE_MSK, SYSCFG_TBKCFG_LVDBKE_POS);
}

/**
  * @brief  Enable CPU lock event as the timer brake source
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_enable_clubke(void)
{
	SET_BIT(SYSCFG->TBKCFG, SYSCFG_TBKCFG_CLUBKE_MSK);
}

/**
  * @brief  Disable CPU lock as the timer brake source
  * @retval None
  */
__STATIC_INLINE void md_syscfg_usb_disable_clubke(void)
{
	CLEAR_BIT(SYSCFG->TBKCFG, SYSCFG_TBKCFG_CLUBKE_MSK);
}

/**
  * @brief  Whether CPU lock is the timer brake source
  * @retval 1-enable, 0-disable
  */
__STATIC_INLINE uint32_t md_syscfg_usb_is_enable_clubke(void)
{
	return READ_BITS(SYSCFG->TBKCFG, SYSCFG_TBKCFG_CLUBKE_MSK, SYSCFG_TBKCFG_CLUBKE_POS);
}
/**
  * @}
  */
#endif
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */

#ifdef __cplusplus
}
#endif

#endif
