/**
  *********************************************************************************
  *
  * @file    md_bkpc.h
  * @brief   Header file of BKPC module driver.
  *
  * @version V1.0
  * @date    01 Jul 2020
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          01 Jul 2020     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __MD_BKPC_H__
#define __MD_BKPC_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_BKPC BKPC
  * @brief BKPC micro driver
  * @{
  */

/** @defgroup MD_BKPC_Public_Types BKPC Public Types
  * @{
  */
/**
  * @brief BKPC preipheral clock select.
  */
typedef enum {
	MD_BKPC_PREH_CLK_LOSM    = 0x0U,	/**< LOSM */
	MD_BKPC_PREH_CLK_LRC     = 0x1U,	/**< LRC */
	MD_BKPC_PREH_CLK_HRC_1M  = 0x2U,	/**< HRC down to 1MHz */
	MD_BKPC_PREH_CLK_HOSC_1M = 0x3U,	/**< HOSC down to 1MHz */
} md_bkpc_preh_clk_t;

/**
  * @brief Standby wakeup port select
  */
typedef enum {
	MD_PMU_STANDBY_PORT_SEL_PA0  = 0x0U,	/**< Wakeup by PA0 */
	MD_PMU_STANDBY_PORT_SEL_PA1  = 0x1U,	/**< Wakeup by PA1 */
	MD_PMU_STANDBY_PORT_SEL_PA2  = 0x2U,	/**< Wakeup by PA2 */
	MD_PMU_STANDBY_PORT_SEL_PA3  = 0x3U,	/**< Wakeup by PA3 */
	MD_PMU_STANDBY_PORT_SEL_PA4  = 0x4U,	/**< Wakeup by PA4 */
	MD_PMU_STANDBY_PORT_SEL_PA5  = 0x5U,	/**< Wakeup by PA5 */
	MD_PMU_STANDBY_PORT_SEL_PA6  = 0x6U,	/**< Wakeup by PA6 */
	MD_PMU_STANDBY_PORT_SEL_PA7  = 0x7U,	/**< Wakeup by PA7 */
	MD_PMU_STANDBY_PORT_SEL_NONE = 0xFU,	/**< Wakeup by other source */
} md_bkpc_wakeup_port_t;

/**
  * @brief Standby wakeup level
  */
typedef enum {
	MD_PMU_STANDBY_LEVEL_HIGH = 0x0U,	/**< High level */
	MD_PMU_STANDBY_LEVEL_LOW  = 0x1U,	/**< Low level */
} md_bkpc_wakeup_level_t;
/**
  * @}
  */
/** @defgroup MD_BKPC_Public_Macros BKPC Public Macros
  * @{
  */
#define MD_BKPC_LOCK()		(WRITE_REG(BKPC->PROT, 0U))
#define MD_BKPC_UNLOCK()	(WRITE_REG(BKPC->PROT, 0x9669AA55U))
/**
  * @}
  */
/** @defgroup MD_BKPC_Public_Functions BKPC Public Functions
  * @{
  */
/** @defgroup MD_BKPC_Public_Functions_Group2 PROT
  * @{
  */
/**
  * @brief  Lock bkpc register.
  * @retval None
  */
__STATIC_INLINE void md_bkpc_lock_enable(void)
{
	WRITE_REG(BKPC->PROT, 0);
}

/**
  * @brief  Unlock bkpc register.
  * @retval None
  */
__STATIC_INLINE void md_bkpc_lock_disable(void)
{
	WRITE_REG(BKPC->PROT, 0x9669AA55U);
}

/**
  * @brief  Check if lock is enabled or disabled.
  * @retval Status:
  *           - 0: Unlock
  *           - 1: Lock
  */
__STATIC_INLINE uint32_t md_bkpc_lock_is_enable(void)
{
	return READ_BITS(BKPC->PROT, BKPC_PROT_PROT_MSK, BKPC_PROT_PROT_POS);
}
/**
  * @}
  */
/** @defgroup MD_BKPC_Public_Functions_Group3 CR
  * @{
  */
/**
  * @brief  Enable LOSC
  * @retval None
  */
__STATIC_INLINE void md_bkpc_losc_enable(void)
{
	SET_BIT(BKPC->CR, BKPC_CR_LOSCEN_MSK);
}

/**
  * @brief  Disable LOSC
  * @retval None
  */
__STATIC_INLINE void md_bkpc_losc_disable(void)
{
	CLEAR_BIT(BKPC->CR, BKPC_CR_LOSCEN_MSK);
}

/**
  * @brief  Check if LOSC is enabled or disabled.
  * @retval Status:
  *           - 0: Disable
  *           - 1: Enable
  */
__STATIC_INLINE uint32_t md_bkpc_losc_is_enable(void)
{
	return READ_BITS(BKPC->CR, BKPC_CR_LOSCEN_MSK, BKPC_CR_LOSCEN_POS);
}

/**
  * @brief  Enable LOSM
  * @retval None
  */
__STATIC_INLINE void md_bkpc_losm_enable(void)
{
	SET_BIT(BKPC->CR, BKPC_CR_LOSMEN_MSK);
}

/**
  * @brief  Disable LOSM
  * @retval None
  */
__STATIC_INLINE void md_bkpc_losm_disable(void)
{
	CLEAR_BIT(BKPC->CR, BKPC_CR_LOSMEN_MSK);
}

/**
  * @brief  Check if LOSM is enabled or disabled.
  * @retval Status:
  *           - 0: Disable
  *           - 1: Enable
  */
__STATIC_INLINE uint32_t md_bkpc_losm_is_enable(void)
{
	return READ_BITS(BKPC->CR, BKPC_CR_LOSMEN_MSK, BKPC_CR_LOSMEN_POS);
}

/**
  * @brief  Enable LRC
  * @retval None
  */
__STATIC_INLINE void md_bkpc_lrc_enable(void)
{
	SET_BIT(BKPC->CR, BKPC_CR_LRCEN_MSK);
}

/**
  * @brief  Disable LRC
  * @retval None
  */
__STATIC_INLINE void md_bkpc_lrc_disable(void)
{
	CLEAR_BIT(BKPC->CR, BKPC_CR_LRCEN_MSK);
}

/**
  * @brief  Check if LRC is enabled or disabled.
  * @retval Status:
  *           - 0: Disable
  *           - 1: Enable
  */
__STATIC_INLINE uint32_t md_bkpc_lrc_is_enable(void)
{
	return READ_BITS(BKPC->CR, BKPC_CR_LRCEN_MSK, BKPC_CR_LRCEN_POS);
}

/**
  * @brief  Enable MRST wakeup standby mode
  * @retval None
  */
__STATIC_INLINE void md_bkpc_mrst_wakeup_enable(void)
{
	SET_BIT(BKPC->CR, BKPC_CR_MRST_WKPEN_MSK);
}

/**
  * @brief  Disable MRST wakeup standby mode
  * @retval None
  */
__STATIC_INLINE void md_bkpc_mrst_wakeup_disable(void)
{
	CLEAR_BIT(BKPC->CR, BKPC_CR_MRST_WKPEN_MSK);
}

/**
  * @brief  Check if MRST wakeup is enabled or disabled.
  * @retval Status:
  *           - 0: Disable
  *           - 1: Enable
  */
__STATIC_INLINE uint32_t md_bkpc_mrst_wakeup_is_enable(void)
{
	return READ_BITS(BKPC->CR, BKPC_CR_MRST_WKPEN_MSK, BKPC_CR_MRST_WKPEN_POS);
}

/**
  * @brief  Enable I/O wakeup standby mode
  * @retval None
  */
__STATIC_INLINE void md_bkpc_gpio_wakeup_enable(void)
{
	SET_BIT(BKPC->CR, BKPC_CR_WKPEN_MSK);
}

/**
  * @brief  Disable I/O wakeup standby mode
  * @retval None
  */
__STATIC_INLINE void md_bkpc_gpio_wakeup_disable(void)
{
	CLEAR_BIT(BKPC->CR, BKPC_CR_WKPEN_MSK);
}

/**
  * @brief  Check if I/O wakeup is enabled or disabled.
  * @retval Status:
  *           - 0: Disable
  *           - 1: Enable
  */
__STATIC_INLINE uint32_t md_bkpc_gpio_wakeup_is_enable(void)
{
	return READ_BITS(BKPC->CR, BKPC_CR_WKPEN_MSK, BKPC_CR_WKPEN_POS);
}

/**
  * @brief  Set wakeup port
  * @param  port: Wakeup port:
  *                 @arg @ref MD_PMU_STANDBY_PORT_SEL_PA0
  *                 @arg @ref MD_PMU_STANDBY_PORT_SEL_PA1
  *                 @arg @ref MD_PMU_STANDBY_PORT_SEL_PA2
  *                 @arg @ref MD_PMU_STANDBY_PORT_SEL_PA3
  *                 @arg @ref MD_PMU_STANDBY_PORT_SEL_PA4
  *                 @arg @ref MD_PMU_STANDBY_PORT_SEL_PA5
  *                 @arg @ref MD_PMU_STANDBY_PORT_SEL_PA6
  *                 @arg @ref MD_PMU_STANDBY_PORT_SEL_PA7
  * @retval None
  */
__STATIC_INLINE void md_bkpc_gpio_wakeup_port_set(md_bkpc_wakeup_port_t port)
{
	MODIFY_REG(BKPC->CR, BKPC_CR_WKPS_MSK, port << BKPC_CR_WKPS_POSS);
}

/**
  * @brief  Get wakeup port
  * @retval Wakeup port, @ref md_bkpc_wakeup_port_t
  */
__STATIC_INLINE uint32_t md_bkpc_gpio_wakeup_port_get(void)
{
	return READ_BITS(BKPC->CR, BKPC_CR_WKPS_MSK, BKPC_CR_WKPS_POSS);
}

/**
  * @brief  Set wakeup level
  * @param  level: Wakeup level:
  *                  @arg @ref MD_PMU_STANDBY_LEVEL_HIGH
  *                  @arg @ref MD_PMU_STANDBY_LEVEL_LOW
  * @retval None
  */
__STATIC_INLINE void md_bkpc_gpio_wakeup_level_set(md_bkpc_wakeup_level_t level)
{
	MODIFY_REG(BKPC->CR, BKPC_CR_WKPOL_MSK, level << BKPC_CR_WKPOL_POS);
}

/**
  * @brief  Get wakeup level
  * @retval Wakeup level, @ref md_bkpc_wakeup_level_t
  */
__STATIC_INLINE uint32_t md_bkpc_gpio_wakeup_level_get(void)
{
	return READ_BITS(BKPC->CR, BKPC_CR_WKPOL_MSK, BKPC_CR_WKPOL_POS);
}
/**
  * @}
  */
/** @defgroup MD_BKPC_Public_Functions_Group4 PCCR
  * @{
  */
/**
  * @brief  Set RTC clock
  * @param  clock: RTC clock:
  *                  @arg @ref MD_BKPC_PREH_CLK_LOSM
  *                  @arg @ref MD_BKPC_PREH_CLK_LRC
  *                  @arg @ref MD_BKPC_PREH_CLK_HRC_1M
  *                  @arg @ref MD_BKPC_PREH_CLK_HOSC_1M
  * @retval None
  */
__STATIC_INLINE void md_bkpc_rtc_clock_set(md_bkpc_preh_clk_t clock)
{
	MODIFY_REG(BKPC->PCCR, BKPC_PCCR_RTCCS_MSK, clock << BKPC_PCCR_RTCCS_POSS);
}

/**
  * @brief  Get RTC clock
  * @retval RTC clock, @ref md_bkpc_preh_clk_t
  */
__STATIC_INLINE uint32_t md_bkpc_rtc_clock_get(void)
{
	return READ_BITS(BKPC->PCCR, BKPC_PCCR_RTCCS_MSK, BKPC_PCCR_RTCCS_POSS);
}

/**
  * @brief  Set Tsense clock
  * @param  clock: Tsense clock:
  *                  @arg @ref MD_BKPC_PREH_CLK_LOSM
  *                  @arg @ref MD_BKPC_PREH_CLK_LRC
  *                  @arg @ref MD_BKPC_PREH_CLK_HRC_1M
  *                  @arg @ref MD_BKPC_PREH_CLK_HOSC_1M
  * @retval None
  */
__STATIC_INLINE void md_bkpc_tsense_clock_set(md_bkpc_preh_clk_t clock)
{
	MODIFY_REG(BKPC->PCCR, BKPC_PCCR_TSENSECS_MSK, clock << BKPC_PCCR_TSENSECS_POSS);
}

/**
  * @brief  Get Tsense clock
  * @retval Tsense clock, @ref md_bkpc_preh_clk_t
  */
__STATIC_INLINE uint32_t md_bkpc_tsense_clock_get(void)
{
	return READ_BITS(BKPC->PCCR, BKPC_PCCR_TSENSECS_MSK, BKPC_PCCR_TSENSECS_POSS);
}
/**
  * @}
  */
/** @defgroup MD_BKPC_Public_Functions_Group5 PDCR
  * @{
  */
/**
  * @brief  Get backup ISO status
  * @retval status:
  *           - 0: Invalid
  *           - 1: Valid
  */
__STATIC_INLINE uint32_t md_bkpc_iso_status_get(void)
{
	return READ_BITS(BKPC->PCR, BKPC_PDCR_ISOLATE_MSK, BKPC_PDCR_ISOLATE_POS);
}
/**
  * @}
  */
/** @defgroup MD_BKPC_Public_Functions_Group1 Configure
  * @{
  */
extern void md_bkpc_standby_wakeup_config(md_bkpc_wakeup_port_t port, md_bkpc_wakeup_level_t level);
extern void md_bkpc_rtc_clock_config(md_bkpc_preh_clk_t clock);
extern void md_bkpc_tsense_clock_config(md_bkpc_preh_clk_t clock);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
