/**
  *********************************************************************************
  *
  * @file    md_dac.c
  * @brief   DAC module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "md_conf.h"

/** @addtogroup Micro_Driver
  * @{
  */
#ifdef MD_DAC

/** @addtogroup MD_DAC
  * @{
  */

/** @addtogroup MD_DAC_Public_Functions
  * @{
  */
/** @addtogroup MD_DAC_Public_Functions_Group1
  * @{
  */

/**
  * @brief  Set each @ref md_dac_init_t field to default value.
  * @param  init: Pointer to a @ref md_dac_init_t structure
  *              whose fields will be set to default values.
  * @retval None
  */
void md_dac_struct_init(md_dac_init_t *init)
{
    init->soft_trig     = MD_DAC_WR_DBUF_TRG;
    init->out_mode      = MD_DAC_OUTPUT_ENHANCE;
    init->output_gpio   = ENABLE;
    init->output_op     = ENABLE;
    init->noise_mode    = MD_DAC_NWM_None;
    init->noise_width   = MD_DAC_NWBW_12;
    init->align_mode    = MD_DAC_ALIGN_R12;
}

/**
  * @brief  Initializes the DAC peripheral.
  * @param  hperh: Pointer to a dac_handle_t structure that contains
  *         the configuration information for the specified DAC module.
  * @retval Status, see @ref ald_status_t.
  */
md_status_t md_dac_init(md_dac_init_t *init)
{
    uint32_t tmp = 0;

    SET_BIT(DAC0->SWTRG, DAC_SWTRG_RST_MSK);

    tmp = ((init->soft_trig << DAC_CON_SWTEN_POSS) | (init->pis_ch << DAC_CON_EXTS_POSS) |
           (init->out_mode << DAC_CON_MSEL_POSS) | (init->noise_mode << DAC_CON_NWM_POSS) |
           (init->noise_width << DAC_CON_NWBW_POSS) | (init->output_gpio << DAC_CON_DACOE_POSS));

    if (init->output_op == DISABLE)
    {
        tmp |= (1U << DAC_CON_OPOFF_POSS);
    }

    WRITE_REG(DAC0->CON, tmp);
    md_dac_enable(DAC0);

    return MD_OK;
}

/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
#endif
/**
 * @}
 */
