/**
  *********************************************************************************
  *
  * @file    md_pmu.h
  * @brief   Header file of PMU module driver.
  *
  * @version V1.0
  * @date    16 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          16 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __MD_PMU_H__
#define __MD_PMU_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_PMU PMU
  * @brief PMU micro driver
  * @{
  */

/** @defgroup MD_PMU_Public_Types PMU Public Types
  * @{
  */

/**
  * @brief Low power mode
  */
typedef enum
{
    MD_PMU_LP_STOP1   = 0x0U,        /**< Stop1 */
    MD_PMU_LP_STOP2   = 0x1U,        /**< Stop2 */
    MD_PMU_LP_STANDBY = 0x2U,        /**< Standby */
    MD_PMU_LP_SHUTOFF = 0x2U,        /**< Shutoff */
} md_pmu_lp_mode_t;

/**
  * @brief Standby wakeup port select
  */
typedef enum
{
    MD_PMU_STANDBY_PORT_SEL_PA0  = 0x0U,    /**< Wakeup by PA0 */
    MD_PMU_STANDBY_PORT_SEL_PA1  = 0x1U,    /**< Wakeup by PA1 */
    MD_PMU_STANDBY_PORT_SEL_PA2  = 0x2U,    /**< Wakeup by PA2 */
    MD_PMU_STANDBY_PORT_SEL_PA3  = 0x3U,    /**< Wakeup by PA3 */
    MD_PMU_STANDBY_PORT_SEL_PA4  = 0x4U,    /**< Wakeup by PA4 */
    MD_PMU_STANDBY_PORT_SEL_PA5  = 0x5U,    /**< Wakeup by PA5 */
    MD_PMU_STANDBY_PORT_SEL_PA6  = 0x6U,    /**< Wakeup by PA6 */
    MD_PMU_STANDBY_PORT_SEL_PA7  = 0x7U,    /**< Wakeup by PA7 */
    MD_PMU_STANDBY_PORT_SEL_NONE = 0xFU,    /**< Wakeup by other source */
} md_pmu_wakeup_port_t;

/**
  * @brief Standby wakeup level
  */
typedef enum
{
    MD_PMU_STANDBY_LEVEL_HIGH = 0x0U,   /**< High level */
    MD_PMU_STANDBY_LEVEL_LOW  = 0x1U,   /**< Low level */
} md_pmu_wakeup_level_t;


/**
  * @brief status types
  */
typedef enum
{
    MD_PMU_SR_WUF      = (1U << 0), /**< WakeUp status */
    MD_PMU_SR_STANDBYF = (1U << 1), /**< Standby status */
    MD_PMU_SR_RSTAT    = (1U << 3), /**< RUN/LPRUAN status */
} md_pmu_status_t;

/**
  * @brief LVD voltage select
  */
typedef enum
{
    MD_PMU_LVD_VOL_SEL_2_1 = 0x0U,  /**< 2.1V */
    MD_PMU_LVD_VOL_SEL_2_3 = 0x1U,  /**< 2.3V */
    MD_PMU_LVD_VOL_SEL_2_5 = 0x2U,  /**< 2.5V */
    MD_PMU_LVD_VOL_SEL_2_7 = 0x3U,  /**< 2.7V */
    MD_PMU_LVD_VOL_SEL_2_9 = 0x4U,  /**< 2.9V */
    MD_PMU_LVD_VOL_SEL_3_5 = 0x5U,  /**< 3.5V */
    MD_PMU_LVD_VOL_SEL_3_9 = 0x6U,  /**< 3.9V */
    MD_PMU_LVD_VOL_SEL_4_5 = 0x7U,  /**< 4.5V */
} md_pmu_lvd_voltage_sel_t;

/**
  * @brief LVD trigger mode
  */
typedef enum
{
    MD_PMU_LVD_TRIGGER_RISING_EDGE    = 0x0U,   /**< Rising edge */
    MD_PMU_LVD_TRIGGER_FALLING_EDGE   = 0x1U,   /**< Falling edge */
    MD_PMU_LVD_TRIGGER_HIGH_LEVEL     = 0x2U,   /**< High level */
    MD_PMU_LVD_TRIGGER_LOW_LEVEL      = 0x3U,   /**< Low level */
    MD_PMU_LVD_TRIGGER_RISING_FALLING = 0x4U,   /**< Rising and falling edge */
} md_pmu_lvd_trigger_mode_t;

/**
  * @brief SRAM/CAN/ROM power selection
  */
typedef enum
{
    MD_PMU_POWER_SRAM0    = 0x1U,      /**< SRAM0 */
    MD_PMU_POWER_SRAM1    = 0x2U,      /**< SRAM1 */
    MD_PMU_POWER_SRAM2    = 0x4U,      /**< SRAM2 */
    MD_PMU_POWER_SRAM3    = 0x8U,      /**< SRAM3 */
    MD_PMU_PWRER_SRAM_ALL = 0xF,       /**< SRAM  */
    MD_PMU_POWER_CAN      = 0x100U,    /**< CAN */
    MD_PMU_POWER_ROM      = 0x1000U,   /**< ROM */
} md_pmu_perh_power_t;

/**
  * @brief LDO output voltage selest in low power mode
  */
typedef enum
{
    MD_PMU_LDO_LPMODE_OUTPUT_1_3 = 0x0,    /**< 1.3V */
    MD_PMU_LDO_LPMODE_OUTPUT_1_4 = 0x1,    /**< 1.4V */
    MD_PMU_LDO_LPMODE_OUTPUT_1_5 = 0x2,    /**< 1.5V */
    MD_PMU_LDO_LPMODE_OUTPUT_1_6 = 0x4,    /**< 1.6V */
} md_pmu_ldo_lpmode_output_t;

/**
  * @brief Standby wakeup pin
  */
typedef enum
{
    MD_PMU_PIN_PA0 = 0x0,  /**< PA0 */
    MD_PMU_PIN_PA1 = 0x1,  /**< PA1 */
    MD_PMU_PIN_PA2 = 0x2,  /**< PA2 */
    MD_PMU_PIN_PA3 = 0x3,  /**< PA3 */
    MD_PMU_PIN_PA4 = 0x4,  /**< PA4 */
    MD_PMU_PIN_PA5 = 0x5,  /**< PA5 */
    MD_PMU_PIN_PA6 = 0x6,  /**< PA6 */
    MD_PMU_PIN_PA7 = 0x7,  /**< PA7 */
} md_pmu_wakeup_pin_t;

/**
  * @}
  */

/** @defgroup MD_PMU_Public_Functions PMU Public Functions
  * @{
  */
/** @defgroup MD_PMU_Public_Functions_Group2 CR
  * @{
  */
/**
  * @brief  Set pmu low power mode.
  * @param  mode: low power mode @see md_pmu_lp_mode_t
  * @retval None
  */
__STATIC_INLINE void md_pmu_set_low_power_mode(md_pmu_lp_mode_t mode)
{
    MODIFY_REG(PMU->CR, PMU_CR_LPM_MSK, mode << PMU_CR_LPM_POSS);
}

/**
  * @brief  Get pmu low power mode.
  * @retval low power mode @see md_pmu_lp_mode_t
  */
__STATIC_INLINE uint32_t md_pmu_get_low_power_mode(void)
{
    return READ_BITS(PMU->CR, PMU_CR_LPM_MSK, PMU_CR_LPM_POSS);
}

/**
  * @brief  Clear wuf flag.
  * @retval None
  */
__STATIC_INLINE void md_pmu_clear_flag_cwuf(void)
{
    SET_BIT(PMU->CR, PMU_CR_CWUF_MSK);
}

/**
  * @brief  Clear standby flag.
  * @retval None
  */
__STATIC_INLINE void md_pmu_clear_flag_standby(void)
{
    SET_BIT(PMU->CR, PMU_CR_CSTANDBYF_MSK);
}

/**
  * @brief  Clear SHUTOFF flag.
  * @retval None
  */
__STATIC_INLINE void md_pmu_clear_flag_shutoff(void)
{
    SET_BIT(PMU->CR, PMU_CR_CSHUTOFFF_MSK);
}

/**
  * @brief  Enable LDO low power mode.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_ldo_lp(void)
{
    SET_BIT(PMU->CR, PMU_CR_LPRUN_MSK);
}

/**
  * @brief  Disable LDO low power mode.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_ldo_lp(void)
{
    CLEAR_BIT(PMU->CR, PMU_CR_LPRUN_MSK);
}

/**
  * @brief  Check if LDO low power is enabled or disabled.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_stop_lp(void)
{
    return (READ_BIT(PMU->CR, PMU_CR_LPRUN_MSK) == (PMU_CR_LPRUN_MSK));
}

/**
  * @brief  Disable low speed clock in stop2 mode.
  * @retval None
  */
__STATIC_INLINE void md_pmu_stop2_disable_low_clk(void)
{
    SET_BIT(PMU->CR, PMU_CR_LPCLKDIS_MSK);
}

/**
  * @brief  Enable low speed clock in stop2 mode.
  * @retval None
  */
__STATIC_INLINE void md_pmu_stop2_enable_low_clk(void)
{
    CLEAR_BIT(PMU->CR, PMU_CR_LPCLKDIS_MSK);
}

/**
  * @brief  Enable the LDO low power in stop mode.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_stop_ldo(void)
{
    SET_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
}

/**
  * @brief  Disable the LDO low power in stop mode.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_stop_ldo(void)
{
    CLEAR_BIT(PMU->CR, PMU_CR_LPSTOP_MSK);
}

/**
  * @brief  Check if LDO low power in stop mode is enabled or disabled.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_stop_ldo(void)
{
    return (READ_BIT(PMU->CR, PMU_CR_LPSTOP_MSK) == (PMU_CR_LPSTOP_MSK));
}

/**
  * @brief  Set the LDO output voltage.
  * @param  vol: LDO output voltage
  *              - 0: 1.3V
  *              - 1: 1.4V
  *              - 2: 1.4V
  *              - 3: 1.6V
  * @retval None
  */
__STATIC_INLINE void md_pmu_set_ldo_lp_output(uint32_t vol)
{
    MODIFY_REG(PMU->CR, PMU_CR_LPOPSEL_MSK, vol << PMU_CR_LPOPSEL_POSS);
}

/**
  * @brief  Set the flash voltage mode in stop.
  * @param  
  *              - 0: normal work
  *              - 1: power-off
  *              - 2: low-power 
  *              - 3: power-off
  * @retval None
  */
__STATIC_INLINE void md_pmu_stop_flash_voltage_mode(uint32_t mode)
{
    MODIFY_REG(PMU->CR, PMU_CR_STOPFM_MSK, mode << PMU_CR_LPOPSEL_POSS);
}

/**
  * @}
  */
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group4 SR
  * @{
  */

/**
  * @brief  Get Shutoff flag .
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_shutoff(void)
{
    return (READ_BIT(PMU->SR, PMU_SR_SHUTOFFF_MSK) == PMU_SR_SHUTOFFF_MSK);
}

/**
  * @brief  Get Standby flag .
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_standby(void)
{
    return (READ_BIT(PMU->SR, PMU_SR_STANDBYF_MSK) == PMU_SR_STANDBYF_MSK);
}

/**
  * @brief  Get Wakeup flag .
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_wakeup(void)
{
    return (READ_BIT(PMU->SR, PMU_SR_WUF_MSK) == PMU_SR_WUF_MSK);
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group5 LVDCR
  * @{
  */
/**
  * @brief  Get LVD Status flag .
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_lvdo(void)
{
    return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDO_MSK) == PMU_LVDCR_LVDO_MSK);
}

/**
  * @brief  Enable the LVD filter .
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_lvd_filter(void)
{
    SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK);
}

/**
  * @brief  Disable the LVD filter .
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_lvd_filter(void)
{
    CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK);
}

/**
  * @brief  Check if the LVD filter is enable or disable.
  * @retval State for bit (1 or 0)
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_lvd_filter(void)
{
    return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDFLT_MSK) == PMU_LVDCR_LVDFLT_MSK);
}

/**
  * @brief  Set LVD Interrupt mode.
  * @param  mode: lvd interrupt mode @see md_pmu_lvd_trigger_mode_t.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_set_lvd_it_mode(md_pmu_lvd_trigger_mode_t mode)
{
    MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDIFS_MSK, mode << PMU_LVDCR_LVDIFS_POSS);
}

/**
  * @brief  Get LVD Interrupt mode.
  * @retval value for lvd it mode.
  */
__STATIC_INLINE uint32_t md_pmu_get_lvd_it_mode(void)
{
    return READ_BITS(PMU->LVDCR, PMU_LVDCR_LVDIFS_MSK, PMU_LVDCR_LVDIFS_POSS);
}

/**
  * @brief  Set LVD voltage threshold value.
  * @param  value: threshold value @see md_pmu_lvd_voltage_sel_t.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_set_lvd_vol_threshold(md_pmu_lvd_voltage_sel_t value)
{
    MODIFY_REG(PMU->LVDCR, PMU_LVDCR_LVDS_MSK, value << PMU_LVDCR_LVDS_POSS);
}

/**
  * @brief  Get LVD voltage threshold value.
  * @retval value for lvd voltage threshold @see pmu_lvd_voltage_sel_t.
  */
__STATIC_INLINE uint32_t md_pmu_get_lvd_vol_threshold(void)
{
    return (READ_BITS(PMU->LVDCR, PMU_LVDCR_LVDS_MSK, PMU_LVDCR_LVDS_POSS));
}

/**
  * @brief  Set LVD interrupt flag clear bit.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_set_lvd_it_flag(void)
{
    SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDCIF_MSK);
}

/**
  * @brief  Get LVD interrupt flag.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_active_flag_lvd_it(void)
{
    return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDIF_MSK) == PMU_LVDCR_LVDIF_MSK);
}

/**
  * @brief  Enable LVD interrupt.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_enable_lvd_it(void)
{
    SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
}

/**
  * @brief  Disable LVD interrupt.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_disable_lvd_it(void)
{
    CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK);
}

/**
  * @brief  Check if LVD interrupt is enable or disable.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_lvd_it(void)
{
    return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDIE_MSK) == PMU_LVDCR_LVDIE_MSK);
}

/**
  * @brief  Enable LVD.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_enable_lvd(void)
{
    SET_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
}

/**
  * @brief  Disable LVD interrupt.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_disable_lvd(void)
{
    CLEAR_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK);
}

/**
  * @brief  Check if LVD is enable or disable.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_lvd_is_enable(void)
{
    return (READ_BIT(PMU->LVDCR, PMU_LVDCR_LVDEN_MSK) == PMU_LVDCR_LVDEN_MSK);
}
/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group6 PWRCR
  * @{
  */
/**
  * @brief  Enable BXCAN power.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_enable_bxcan(void)
{
    SET_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN0_MSK);
}

/**
  * @brief  Disable BXCAN power.
  * @retval None.
  */
__STATIC_INLINE void md_pmu_disable_bxcan(void)
{
    CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN0_MSK);
}

/**
  * @brief  Check if BXCAN power is enable or disable.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pmu_is_enable_bxcan(void)
{
    return (READ_BIT(PMU->PWRCR, PMU_PWRCR_BXCAN0_MSK) == PMU_PWRCR_BXCAN0_MSK);
}

/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group7 PWRCR
  * @{
  */
/**
  * @brief  Enable rom power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_rom_power(void)
{
    SET_BIT(PMU->PWRCR, PMU_PWRCR_ROM_MSK);
}

/**
  * @brief  Disable rom power.
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_rom_power(void)
{
    CLEAR_BIT(PMU->PWRCR, PMU_PWRCR_ROM_MSK);
}

/**
  * @brief  Check if rom power is enable or disable.
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint8_t md_pmu_is_enable_rom_power(void)
{
    return (READ_BIT(PMU->PWRCR, PMU_PWRCR_ROM_MSK) == PMU_PWRCR_ROM_MSK);
}

/**
  * @brief  Enable sram power.
  * @param  idx: Index of sram, [0-3]
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_sram_power(uint8_t idx)
{
    SET_BIT(PMU->PWRCR, (1 << idx));
}

/**
  * @brief  Disable sram power.
  * @param  idx: Index of sram, [0-5]
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_sram_power(uint8_t idx)
{
    CLEAR_BIT(PMU->PWRCR, (1 << idx));
}

/**
  * @brief  Check if sram power is enable or disable.
  * @param  idx: Index of sram, [0-3]
  * @retval State for 1 or 0.
  */
__STATIC_INLINE uint8_t md_pmu_is_enable_sram_power(uint8_t idx)
{
    return (READ_BIT(PMU->PWRCR, (1 << idx)) == (1 << idx));
}

/**
  * @brief  Standby RTC wakeup enable.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_enable_standby_rtc_wakeup_stbwken(void)
{
    uint32_t tmp = PMU->BKPCR0;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    SET_BIT(tmp, PMU_BKPCR0_STBWKEN_MSK);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR0 = tmp;
    WRITE_REG(SYSCFG->PROT, 0); 
}



/**
  * @brief  Standby RTC wakeup disable.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_disable_standby_rtc_wakeup_stbwken(void)
{
    uint32_t tmp = PMU->BKPCR0;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    CLEAR_BIT(tmp, PMU_BKPCR0_STBWKEN_MSK);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR0 = tmp;
    WRITE_REG(SYSCFG->PROT, 0); 
}

/**
  * @brief  Wakeup polarity high.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_wakeup_pin_high_wkpl(void)
{
    uint32_t tmp = PMU->BKPCR0;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    CLEAR_BIT(tmp, PMU_BKPCR0_WKPL_MSK);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR0 = tmp;
    WRITE_REG(SYSCFG->PROT, 0); 
}

/**
  * @brief  Wakeup polarity low.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_wakeup_pin_low_wkpl(void)
{
    uint32_t tmp = PMU->BKPCR0;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    SET_BIT(tmp, PMU_BKPCR0_WKPL_MSK);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR0 = tmp;
    WRITE_REG(SYSCFG->PROT, 0); 
}

/**
  * @brief  Enable wakeup pin.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_wakeup_pin_enable_wkpen(void)
{
    uint32_t tmp = PMU->BKPCR0;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    SET_BIT(tmp, PMU_BKPCR0_WKPEN_MSK);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR0 = tmp;
    WRITE_REG(SYSCFG->PROT, 0);   
}

/**
  * @brief  Disable wakeup pin.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_wakeup_pin_disable_wkpen(void)
{
    uint32_t tmp = PMU->BKPCR0;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    CLEAR_BIT(tmp, PMU_BKPCR0_WKPEN_MSK);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR0 = tmp;
    WRITE_REG(SYSCFG->PROT, 0);
}

/**
  * @brief  Wakeup pin selection.
  * @param  pin: PA0 ~ PA7 @md_pmu_wakeup_port_t
  * @retval None
  */
__STATIC_INLINE void md_pmu_wakeup_pin_sel_wkps(md_pmu_wakeup_port_t pin)
{
    uint32_t tmp = PMU->BKPCR0;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    MODIFY_REG(tmp, PMU_BKPCR0_WKPS_MSK, pin << PMU_BKPCR0_WKPS_POSS);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR0 = tmp;
    WRITE_REG(SYSCFG->PROT, 0);    
}

/**
  * @brief  Backup RAM retention enable.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_bkram_retention_enable_bkramret(void)
{
    uint32_t tmp = PMU->BKPCR1;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    SET_BIT(tmp, PMU_BKPCR1_BKRAMRET_MSK); 
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR1 = tmp;
    WRITE_REG(SYSCFG->PROT, 0);    
}

/**
  * @brief  Backup RAM retention disable.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_bkram_retention_disable_bkramret(void)
{
    uint32_t tmp = PMU->BKPCR1;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    CLEAR_BIT(tmp, PMU_BKPCR1_BKRAMRET_MSK); 
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR1 = tmp;
    WRITE_REG(SYSCFG->PROT, 0);   
}

/**
  * @brief  Backup RAM power enable.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_bkram_power_enable_bkrampwr(void)
{
    uint32_t tmp = PMU->BKPCR1;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    SET_BIT(tmp, PMU_BKPCR1_BKRAMPWR_MSK); 
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR1 = tmp;
    WRITE_REG(SYSCFG->PROT, 0); 
}

/**
  * @brief  Backup RAM power disable.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_bkram_power_disable_bkrampwr(void)
{
    uint32_t tmp = PMU->BKPCR1;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    CLEAR_BIT(tmp, PMU_BKPCR1_BKRAMPWR_MSK); 
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR1 = tmp;
    WRITE_REG(SYSCFG->PROT, 0);   
}

/**
  * @brief  LDO output mode selection in standby.
  * @param  pin: SRAM/CAN/ROM power selection @md_pmu_ldo_lpmode_output_t
  * @retval None
  */
__STATIC_INLINE void md_pmu_standby_ldo_sel_stblpvs(md_pmu_ldo_lpmode_output_t pwr)
{
    uint32_t tmp = PMU->BKPCR1;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    MODIFY_REG(tmp, PMU_BKPCR1_STBLPVS_MSK, pwr << PMU_BKPCR1_STBLPVS_POSS); 
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPCR1 = tmp;
    WRITE_REG(SYSCFG->PROT, 0);    
}

/**
  * @brief  Get rtc wakeup flag.
  * @param  None
  * @retval RTC flag 
  */
__STATIC_INLINE uint32_t md_pmu_get_rtc_wakeup_flag_rtc(void)
{
    return READ_BIT(PMU->BKPSR, PMU_BKPSR_RTC_MSK);
}

/**
  * @brief  Clean rtc wakeup flag.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_clean_rtc_wakeup_flag_rtc(void)
{
    uint32_t tmp = PMU->BKPSR;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    SET_BIT(tmp, PMU_BKPSR_RTC_MSK);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPSR = tmp;
    WRITE_REG(SYSCFG->PROT, 0);    
}

/**
  * @brief  Get external pin wakeup flag.
  * @param  None
  * @retval wakeup flag 
  */
__STATIC_INLINE uint32_t md_pmu_get_pin_wakeup_flag_wkp(void)
{
    return READ_BIT(PMU->BKPSR, PMU_BKPSR_WKP_MSK);
}

/**
  * @brief  Clean external wakeup flag.
  * @param  None
  * @retval None
  */
__STATIC_INLINE void md_pmu_clean_pin_wakeup_flag_wkp(void)
{
    uint32_t tmp = PMU->BKPSR;
    
    tmp &= 0xFFFF00FF;
    tmp |= 0x5A00;

    SET_BIT(tmp, PMU_BKPSR_WKP_MSK);
    
    WRITE_REG(SYSCFG->PROT, 0x55AA6996U);
    PMU->BKPSR = tmp;
    WRITE_REG(SYSCFG->PROT, 0);  
}

/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group8 VREFCR
  * @{
  */

/**
  * @}
  */
/** @defgroup MD_PMU_Public_Functions_Group1 Initialization
  * @{
  */
__STATIC_INLINE__ void md_pmu_sleep()
{
    __WFI();
}

__STATIC_INLINE__ void md_pmu_sleep_deep()
{
    SCB->SCR |= SCB_SCR_SLEEPDEEP_Msk;
    __WFI();
}

extern void md_pmu_reset(void);
extern void md_pmu_stop1_enter(void);
extern void md_pmu_stop2_enter(void);
extern void md_pmu_standby_enter(md_pmu_wakeup_pin_t pin, md_pmu_wakeup_level_t levle);
extern void md_pmu_perh_power_config(md_pmu_perh_power_t perh, type_func_t state);
extern void md_pmu_lvd_config(md_pmu_lvd_voltage_sel_t sel, md_pmu_lvd_trigger_mode_t mode, type_func_t state);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
