/**
  *********************************************************************************
  *
  * @file    md_pis.h
  * @brief   Header file of PIS module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __MD_PIS_H__
#define __MD_PIS_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"
#include "type.h"

/**
  * @addtogroup Micro_Driver
  * @{
  */

/**
  * @defgroup MD_PIS PIS
  * @brief PIS micro driver
  * @{
  */

/** @defgroup MD_PIS_Public_Types PIS Public Types
  * @verbatim
      In PIS module, for the convenience of code maintenance,
      TIMERx is used to indicate the sequence of the timer peripheral.
      Different product series TIMERx represent different meanings:
      1. For ES32F361x_362x series:
      TIMER0 ----> AD16C4T0
      TIMER2 ----> GP32C4T0
      TIMER3 ----> GP32C4T1
      TIMER4 ----> BS16T0
      TIMER5 ----> BS16T1
      TIMER6 ----> GP16C4T0
      TIMER7 ----> GP16C4T1
    @endverbatim
  * @{
  */

/**
  * @brief Producer entry
  */
typedef enum
{
    MD_PIS_SOFTWARE_LEVEL       = 0x00U,    /**< Software level trigger */
    MD_PIS_SOFTWARE_PULSE       = 0x01U,    /**< Software pulse trigger */
    MD_PIS_GPIOA_PIN0           = 0x10U,    /**< Pin0, level,support async */
    MD_PIS_GPIOA_PIN1           = 0x11U,    /**< Pin1, level,support async */
    MD_PIS_GPIOA_PIN2           = 0x12U,    /**< Pin2, level,support async */
    MD_PIS_GPIOA_PIN3           = 0x13U,    /**< Pin3, level,support async */
    MD_PIS_GPIOA_PIN4           = 0x14U,    /**< Pin4, level,support async */
    MD_PIS_GPIOA_PIN5           = 0x15U,    /**< Pin5, level,support async */
    MD_PIS_GPIOA_PIN6           = 0x16U,    /**< Pin6, level,support async */
    MD_PIS_GPIOA_PIN7           = 0x17U,    /**< Pin7, level,support async */
    MD_PIS_GPIOA_PIN8           = 0x18U,    /**< Pin8, level,support async */
    MD_PIS_GPIOA_PIN9           = 0x19U,    /**< Pin9, level,support async */
    MD_PIS_GPIOA_PIN10          = 0x1aU,    /**< Pin10, level,support async */
    MD_PIS_GPIOA_PIN11          = 0x1bU,    /**< Pin11, level,support async */
    MD_PIS_GPIOA_PIN12          = 0x1cU,    /**< Pin12, level,support async */
    MD_PIS_GPIOA_PIN13          = 0x1dU,    /**< Pin13, level,support async */
    MD_PIS_GPIOA_PIN14          = 0x1eU,    /**< Pin14, level,support async */
    MD_PIS_GPIOA_PIN15          = 0x1fU,    /**< Pin15, level,support async */
    MD_PIS_GPIOB_PIN0           = 0x20U,    /**< Pin0, level,support async */
    MD_PIS_GPIOB_PIN1           = 0x21U,    /**< Pin1, level,support async */
    MD_PIS_GPIOB_PIN2           = 0x22U,    /**< Pin2, level,support async */
    MD_PIS_GPIOB_PIN3           = 0x23U,    /**< Pin3, level,support async */
    MD_PIS_GPIOB_PIN4           = 0x24U,    /**< Pin4, level,support async */
    MD_PIS_GPIOB_PIN5           = 0x25U,    /**< Pin5, level,support async */
    MD_PIS_GPIOB_PIN6           = 0x26U,    /**< Pin6, level,support async */
    MD_PIS_GPIOB_PIN7           = 0x27U,    /**< Pin7, level,support async */
    MD_PIS_GPIOB_PIN8           = 0x28U,    /**< Pin8, level,support async */
    MD_PIS_GPIOB_PIN9           = 0x29U,    /**< Pin9, level,support async */
    MD_PIS_GPIOB_PIN10          = 0x2aU,    /**< Pin10, level,support async */
    MD_PIS_GPIOB_PIN11          = 0x2bU,    /**< Pin11, level,support async */
    MD_PIS_GPIOB_PIN12          = 0x2cU,    /**< Pin12, level,support async */
    MD_PIS_GPIOB_PIN13          = 0x2dU,    /**< Pin13, level,support async */
    MD_PIS_GPIOB_PIN14          = 0x2eU,    /**< Pin14, level,support async */
    MD_PIS_GPIOB_PIN15          = 0x2fU,    /**< Pin15, level,support async */
    MD_PIS_GPIOC_PIN0           = 0x30U,    /**< Pin0, level,support async */
    MD_PIS_GPIOC_PIN1           = 0x31U,    /**< Pin1, level,support async */
    MD_PIS_GPIOC_PIN2           = 0x32U,    /**< Pin2, level,support async */
    MD_PIS_GPIOC_PIN3           = 0x33U,    /**< Pin3, level,support async */
    MD_PIS_GPIOC_PIN4           = 0x34U,    /**< Pin4, level,support async */
    MD_PIS_GPIOC_PIN5           = 0x35U,    /**< Pin5, level,support async */
    MD_PIS_GPIOC_PIN6           = 0x36U,    /**< Pin6, level,support async */
    MD_PIS_GPIOC_PIN7           = 0x37U,    /**< Pin7, level,support async */
    MD_PIS_GPIOC_PIN8           = 0x38U,    /**< Pin8, level,support async */
    MD_PIS_GPIOC_PIN9           = 0x39U,    /**< Pin9, level,support async */
    MD_PIS_GPIOC_PIN10          = 0x3aU,    /**< Pin10, level,support async */
    MD_PIS_GPIOC_PIN11          = 0x3bU,    /**< Pin11, level,support async */
    MD_PIS_GPIOC_PIN12          = 0x3cU,    /**< Pin12, level,support async */
    MD_PIS_GPIOC_PIN13          = 0x3dU,    /**< Pin13, level,support async */
    MD_PIS_GPIOC_PIN14          = 0x3eU,    /**< Pin14, level,support async */
    MD_PIS_GPIOC_PIN15          = 0x3fU,    /**< Pin15, level,support async */
    MD_PIS_GPIOD_PIN0           = 0x40U,    /**< Pin0, level,support async */
    MD_PIS_GPIOD_PIN1           = 0x41U,    /**< Pin1, level,support async */
    MD_PIS_GPIOD_PIN2           = 0x42U,    /**< Pin2, level,support async */
    MD_PIS_GPIOD_PIN3           = 0x43U,    /**< Pin3, level,support async */
    MD_PIS_GPIOD_PIN4           = 0x44U,    /**< Pin4, level,support async */
    MD_PIS_GPIOD_PIN5           = 0x45U,    /**< Pin5, level,support async */
    MD_PIS_GPIOD_PIN6           = 0x46U,    /**< Pin6, level,support async */
    MD_PIS_GPIOD_PIN7           = 0x47U,    /**< Pin7, level,support async */
    MD_PIS_GPIOD_PIN8           = 0x48U,    /**< Pin8, level,support async */
    MD_PIS_GPIOD_PIN9           = 0x49U,    /**< Pin9, level,support async */
    MD_PIS_GPIOD_PIN10          = 0x4aU,    /**< Pin10, level,support async */
    MD_PIS_GPIOD_PIN11          = 0x4bU,    /**< Pin11, level,support async */
    MD_PIS_GPIOD_PIN12          = 0x4cU,    /**< Pin12, level,support async */
    MD_PIS_GPIOD_PIN13          = 0x4dU,    /**< Pin13, level,support async */
    MD_PIS_GPIOD_PIN14          = 0x4eU,    /**< Pin14, level,support async */
    MD_PIS_GPIOD_PIN15          = 0x4fU,    /**< Pin15, level,support async */
    MD_PIS_GPIOF_PIN0           = 0x60U,    /**< Pin0, level,support async */
    MD_PIS_GPIOF_PIN1           = 0x61U,    /**< Pin1, level,support async */
    MD_PIS_GPIOF_PIN2           = 0x62U,    /**< Pin2, level,support async */
    MD_PIS_GPIOF_PIN3           = 0x63U,    /**< Pin3, level,support async */
    MD_PIS_GPIOF_PIN4           = 0x64U,    /**< Pin4, level,support async */
    MD_PIS_GPIOF_PIN5           = 0x65U,    /**< Pin5, level,support async */
    MD_PIS_GPIOF_PIN6           = 0x66U,    /**< Pin6, level,support async */
    MD_PIS_GPIOF_PIN7           = 0x67U,    /**< Pin7, level,support async */
    MD_PIS_GPIOF_PIN8           = 0x68U,    /**< Pin8, level,support async */
    MD_PIS_GPIOF_PIN9           = 0x69U,    /**< Pin9, level,support async */
    MD_PIS_GPIOF_PIN10          = 0x6aU,    /**< Pin10, level,support async */
    MD_PIS_GPIOF_PIN11          = 0x6bU,    /**< Pin11, level,support async */
    MD_PIS_GPIOF_PIN12          = 0x6cU,    /**< Pin12, level,support async */
    MD_PIS_GPIOF_PIN13          = 0x6dU,    /**< Pin13, level,support async */
    MD_PIS_GPIOF_PIN14          = 0x6eU,    /**< Pin14, level,support async */
    MD_PIS_GPIOF_PIN15          = 0x6fU,    /**< Pin15, level,support async */
    MD_PIS_GPIOH_PIN0           = 0x70U,    /**< Pin0, level,support async */
    MD_PIS_GPIOH_PIN1           = 0x71U,    /**< Pin1, level,support async */
    MD_PIS_GPIOH_PIN2           = 0x72U,    /**< Pin2, level,support async */
    MD_PIS_GPIOH_PIN3           = 0x73U,    /**< Pin3, level,support async */
    MD_PIS_GPIOH_PIN4           = 0x74U,    /**< Pin4, level,support async */
    MD_PIS_GPIOH_PIN5           = 0x75U,    /**< Pin5, level,support async */
    MD_PIS_GPIOH_PIN6           = 0x76U,    /**< Pin6, level,support async */
    MD_PIS_GPIOH_PIN7           = 0x77U,    /**< Pin7, level,support async */
    MD_PIS_GPIOH_PIN8           = 0x78U,    /**< Pin8, level,support async */
    MD_PIS_GPIOH_PIN9           = 0x79U,    /**< Pin9, level,support async */
    MD_PIS_GPIOH_PIN10          = 0x7aU,    /**< Pin10, level,support async */
    MD_PIS_GPIOH_PIN11          = 0x7bU,    /**< Pin11, level,support async */
    MD_PIS_GPIOH_PIN12          = 0x7cU,    /**< Pin12, level,support async */
    MD_PIS_GPIOH_PIN13          = 0x7dU,    /**< Pin13, level,support async */
    MD_PIS_GPIOH_PIN14          = 0x7eU,    /**< Pin14, level,support async */
    MD_PIS_GPIOH_PIN15          = 0x7fU,    /**< Pin15, level,support async */
    MD_PIS_UART0_RESERVE0       = 0x80U,    /**< Uart0 reserve bit 0 */
    MD_PIS_UART0_RESERVE1       = 0x81U,    /**< Uart0 reserve bit 1*/
    MD_PIS_UART0_IRDAOUT        = 0x82U,    /**< Uart0 irdaout, level,support async */
    MD_PIS_UART0_RTSOUT         = 0x83U,    /**< Uart0 rtsout, level,support async */
    MD_PIS_UART0_TXOUT          = 0x84U,    /**< Uart0 txout, level,support async */
    MD_PIS_UART0_SEND_INT_PULSE = 0x85U,    /**< Uart0 syn send, pulse,support async */
    MD_PIS_UART0_RECV_INT_PULSE = 0x86U,    /**< Uart0 syn recv, pulse,support async */
    MD_PIS_UART1_RESERVE0       = 0x90U,    /**< Uart1 reserve bit 0 */
    MD_PIS_UART1_RESERVE1       = 0x91U,    /**< Uart1 reserve bit 1*/
    MD_PIS_UART1_IRDA           = 0x92U,    /**< Uart1 irdaout, level,support async */
    MD_PIS_UART1_RTS            = 0x93U,    /**< Uart1 rtsout, level,support async */
    MD_PIS_UART1_TXOUT          = 0x94U,    /**< Uart1 txout, level,support async */
    MD_PIS_UART1_SEND_INT_PULSE = 0x95U,    /**< Uart1 syn send, pulse,support async */
    MD_PIS_UART1_RECV_INT_PULSE = 0x96U,    /**< Uart1 syn recv, pulse,support async */
    MD_PIS_UART2_RESERVE0       = 0xa0U,    /**< Uart2 reserve bit 0 */
    MD_PIS_UART2_RESERVE1       = 0xa1U,    /**< Uart2 reserve bit 1*/
    MD_PIS_UART2_RTS            = 0xa3U,    /**< Uart2 rtsout, level,support async */
    MD_PIS_UART2_TXOUT          = 0xa4U,    /**< Uart2 txout, level,support async */
    MD_PIS_UART2_SEND_INT_PULSE = 0xa5U,    /**< Uart2 syn send, pulse,support async */
    MD_PIS_UART2_RECV_INT_PULSE = 0xa6U,    /**< Uart2 syn recv, pulse,support async */
    MD_PIS_UART3_RESERVE0       = 0xb0U,    /**< Uart3 reserve bit 0 */
    MD_PIS_UART3_RESERVE1       = 0xb1U,    /**< Uart3 reserve bit 1*/
    MD_PIS_UART3_RTS            = 0xb3U,    /**< Uart3 rtsout, level,support async */
    MD_PIS_UART3_TXOUT          = 0xb4U,    /**< Uart3 txout, level,support async */
    MD_PIS_UART3_SEND_INT_PULSE = 0xb5U,    /**< Uart3 syn send, pulse,support async */
    MD_PIS_UART3_RECV_INT_PULSE = 0xb6U,    /**< Uart3 syn recv, pulse,support async */
    MD_PIS_UART4_RESERVE0       = 0xc0U,    /**< Uart4 reserve bit 0 */
    MD_PIS_UART4_RESERVE1       = 0xc1U,    /**< Uart4 reserve bit 1*/
    MD_PIS_UART4_RTS            = 0xc3U,    /**< Uart4 rtsout, level,support async */
    MD_PIS_UART4_TXOUT          = 0xc4U,    /**< Uart4 txout, level,support async */
    MD_PIS_UART4_SEND_INT_PULSE = 0xc5U,    /**< Uart4 syn send, pulse,support async */
    MD_PIS_UART4_RECV_INT_PULSE = 0xc6U,    /**< Uart4 syn recv, pulse,support async */
    MD_PIS_UART5_RESERVE0       = 0xd0U,    /**< Uart5 reserve bit 0 */
    MD_PIS_UART5_RESERVE1       = 0xd1U,    /**< Uart5 reserve bit 1*/
    MD_PIS_UART5_RTS            = 0xd3U,    /**< Uart5 rtsout, level,support async */
    MD_PIS_UART5_TXOUT          = 0xd4U,    /**< Uart5 txout, level,support async */
    MD_PIS_UART5_SEND_INT_PULSE = 0xd5U,    /**< Uart5 syn send, pulse,support async */
    MD_PIS_UART5_RECV_INT_PULSE = 0xd6U,    /**< Uart5 syn recv, pulse,support async */
    MD_PIS_SPI0_RECV            = 0xe0U,    /**< Spi0 recv, plck1 pulse */
    MD_PIS_SPI0_SEND            = 0xe1U,    /**< Spi0 send, plck1 pulse */
    MD_PIS_SPI0_NE              = 0xe2U,    /**< Spi0 ne, plck1 level */
    MD_PIS_SPI1_RECV            = 0xf0U,    /**< Spi1 recv, plck1 pulse */
    MD_PIS_SPI1_SEND            = 0xf1U,    /**< Spi1 send, plck1 pulse */
    MD_PIS_SPI1_NE              = 0xf2U,    /**< Spi1 ne, plck1 level */
    MD_PIS_I2C0_RECV            = 0x100U,   /**< I2c0 recv, plck1 level */
    MD_PIS_I2C0_SEND            = 0x101U,   /**< I2c0 send, plck1 level */
    MD_PIS_I2C1_RECV            = 0x110U,   /**< I2c1 recv, plck1 level */
    MD_PIS_I2C1_SEND            = 0x111U,   /**< I2c1 send, plck1 level */
    MD_PIS_TIMER0_UPDATA        = 0x120U,   /**< AD16C4T0 updata, plck1 pulse */
    MD_PIS_TIMER0_TRGOUT        = 0x121U,   /**< AD16C4T0 trgout, plck1 pulse */
    MD_PIS_TIMER0_INPUT_1       = 0x122U,   /**< AD16C4T0 chan1 input catch, plck1 pulse */
    MD_PIS_TIMER0_OUTPUT_1      = 0x123U,   /**< AD16C4T0 chan1 output compare, plck1 pulse */
    MD_PIS_TIMER0_INPUT_2       = 0x124U,   /**< AD16C4T0 chan2 input catch, plck1 pulse */
    MD_PIS_TIMER0_OUTPUT_2      = 0x125U,   /**< AD16C4T0 chan2 output compare, plck1 pulse */
    MD_PIS_TIMER0_INPUT_3       = 0x126U,   /**< AD16C4T0 chan3 input catch, plck1 pulse */
    MD_PIS_TIMER0_OUTPUT_3      = 0x127U,   /**< AD16C4T0 chan3 output compare, plck1 pulse */
    MD_PIS_TIMER0_INPUT_4       = 0x128U,   /**< AD16C4T0 chan4 input catch, plck1 pulse */
    MD_PIS_TIMER0_OUTPUT_4      = 0x129U,   /**< AD16C4T0 chan4 output compare, plck1 pulse */
    MD_PIS_TIMER2_TRGOUT        = 0x141U,   /**< GP32C4T0 trgout, plck1 pulse */
    MD_PIS_TIMER2_INPUT_1       = 0x142U,   /**< GP32C4T0 chan1 input catch, plck1 pulse */
    MD_PIS_TIMER2_OUTPUT_1      = 0x143U,   /**< GP32C4T0 chan1 output compare, plck1 pulse */
    MD_PIS_TIMER2_INPUT_2       = 0x144U,   /**< GP32C4T0 chan2 input catch, plck1 pulse */
    MD_PIS_TIMER2_OUTPUT_2      = 0x145U,   /**< GP32C4T0 chan2 output compare, plck1 pulse */
    MD_PIS_TIMER2_INPUT_3       = 0x146U,   /**< GP32C4T0 chan3 input catch, plck1 pulse */
    MD_PIS_TIMER2_OUTPUT_3      = 0x147U,   /**< GP32C4T0 chan3 output compare, plck1 pulse */
    MD_PIS_TIMER2_INPUT_4       = 0x148U,   /**< GP32C4T0 chan4 input catch, plck1 pulse */
    MD_PIS_TIMER2_OUTPUT_4      = 0x149U,   /**< GP32C4T0 chan4 output compare, plck1 pulse */
    MD_PIS_TIMER3_TRGOUT        = 0x151U,   /**< GP32C4T1 trgout, plck1 pulse */
    MD_PIS_TIMER3_INPUT_1       = 0x152U,   /**< GP32C4T1 chan1 input catch, plck1 pulse */
    MD_PIS_TIMER3_OUTPUT_1      = 0x153U,   /**< GP32C4T1 chan1 output compare, plck1 pulse */
    MD_PIS_TIMER3_INPUT_2       = 0x154U,   /**< GP32C4T1 chan2 input catch, plck1 pulse */
    MD_PIS_TIMER3_OUTPUT_2      = 0x155U,   /**< GP32C4T1 chan2 output compare, plck1 pulse */
    MD_PIS_TIMER3_INPUT_3       = 0x156U,   /**< GP32C4T1 chan3 input catch, plck1 pulse */
    MD_PIS_TIMER3_OUTPUT_3      = 0x157U,   /**< GP32C4T1 chan3 output compare, plck1 pulse */
    MD_PIS_TIMER3_INPUT_4       = 0x158U,   /**< GP32C4T1 chan4 input catch, plck1 pulse */
    MD_PIS_TIMER3_OUTPUT_4      = 0x159U,   /**< GP32C4T1 chan4 output compare, plck1 pulse */
    MD_PIS_TIMER6_TRGOUT        = 0x1D1U,   /**< GP16C4T0 trgout, plck1 pulse */
    MD_PIS_TIMER6_INPUT_1       = 0x1D2U,   /**< GP16C4T0 chan1 input catch, plck1 pulse */
    MD_PIS_TIMER6_OUTPUT_1      = 0x1D3U,   /**< GP16C4T0 chan1 output compare, plck1 pulse */
    MD_PIS_TIMER6_INPUT_2       = 0x1D4U,   /**< GP16C4T0 chan2 input catch, plck1 pulse */
    MD_PIS_TIMER6_OUTPUT_2      = 0x1D5U,   /**< GP16C4T0 chan2 output compare, plck1 pulse */
    MD_PIS_TIMER6_INPUT_3       = 0x1D6U,   /**< GP16C4T0 chan3 input catch, plck1 pulse */
    MD_PIS_TIMER6_OUTPUT_3      = 0x1D7U,   /**< GP16C4T0 chan3 output compare, plck1 pulse */
    MD_PIS_TIMER6_INPUT_4       = 0x1D8U,   /**< GP16C4T0 chan4 input catch, plck1 pulse */
    MD_PIS_TIMER6_OUTPUT_4      = 0x1D9U,   /**< GP16C4T0 chan4 output compare, plck1 pulse */
    MD_PIS_TIMER7_TRGOUT        = 0x1E1U,   /**< GP16C4T1 trgout, plck1 pulse */
    MD_PIS_TIMER7_INPUT_1       = 0x1E2U,   /**< GP16C4T1 chan1 input catch, plck1 pulse */
    MD_PIS_TIMER7_OUTPUT_1      = 0x1E3U,   /**< GP16C4T1 chan1 output compare, plck1 pulse */
    MD_PIS_TIMER7_INPUT_2       = 0x1E4U,   /**< GP16C4T1 chan2 input catch, plck1 pulse */
    MD_PIS_TIMER7_OUTPUT_2      = 0x1E5U,   /**< GP16C4T1 chan2 output compare, plck1 pulse */
    MD_PIS_TIMER7_INPUT_3       = 0x1E6U,   /**< GP16C4T1 chan3 input catch, plck1 pulse */
    MD_PIS_TIMER7_OUTPUT_3      = 0x1E7U,   /**< GP16C4T1 chan3 output compare, plck1 pulse */
    MD_PIS_TIMER7_INPUT_4       = 0x1E8U,   /**< GP16C4T1 chan4 input catch, plck1 pulse */
    MD_PIS_TIMER7_OUTPUT_4      = 0x1E9U,   /**< GP16C4T1 chan4 output compare, plck1 pulse */
    MD_PIS_RTC_CLOCK            = 0x160U,   /**< Rtc clock, pulse,support async */
    MD_PIS_RTC_ALARM            = 0x161U,   /**< Rtc alarm, pulse,support async */
    MD_PIS_LPTIMER0_SYN_UPDATA  = 0x170U,   /**< Lptimer0 syn updata, pulse,support async */
    MD_PIS_LPTIMER0_ASY_UPDATA  = 0x171U,   /**< Lptimer0 asy updata, pulse,support async */
    MD_PIS_LPUART0_ASY_RECV     = 0x180U,   /**< Lpuart0 asy recv, pulse,support async */
    MD_PIS_LPUART0_ASY_SEND     = 0x181U,   /**< Lpuart0 asy send, pulse,support async */
    MD_PIS_LPUART0_SYN_RECV     = 0x182U,   /**< Lpuart0 syn recv, pulse,support async */
    MD_PIS_LPUART0_SYN_SEND     = 0x183U,   /**< Lpuart0 syn recv, pulse,support async */
    MD_PIS_DMA_CH0              = 0x190U,   /**< Dma ch0 transmit completely, support async */
    MD_PIS_DMA_CH1              = 0x191U,   /**< Dma ch1 transmit completely, support async */
    MD_PIS_DMA_CH2              = 0x192U,   /**< Dma ch2 transmit completely, support async */
    MD_PIS_DMA_CH3              = 0x193U,   /**< Dma ch3 transmit completely, support async */
    MD_PIS_DMA_CH4              = 0x194U,   /**< Dma ch4 transmit completely, support async */
    MD_PIS_DMA_CH5              = 0x195U,   /**< Dma ch5 transmit completely, support async */
    MD_PIS_DMA_CH6              = 0x196U,   /**< Dma ch6 transmit completely, support async */
    MD_PIS_DMA_CH7              = 0x197U,   /**< Dma ch7 transmit completely, support async */
    MD_PIS_DMA_CH8              = 0x198U,   /**< Dma ch8 transmit completely, support async */
    MD_PIS_DMA_CH9              = 0x199U,   /**< Dma ch9 transmit completely, support async */
    MD_PIS_DMA_CH10             = 0x19AU,   /**< Dma chA transmit completely, support async */
    MD_PIS_DMA_CH11             = 0x19BU,   /**< Dma chB transmit completely, support async */
    MD_PIS_DMA_CH               = 0x19FU,   /**< Dma channel transmit completely, support async */
    MD_PIS_LVD                  = 0x1A0U,   /**< Lvd compare output */
    MD_PIS_LVDEX                = 0x1A1U,   /**< Lvdex compare output */
    MD_PIS_TIMER4_TRGOUT        = 0X1B1U,   /**< BS16T0, trgout */
    MD_PIS_TIMER5_TRGOUT        = 0X1C1U,   /**< BS16T1, trgout */
    MD_PIS_ADC0_INSERT          = 0x1F0U,   /**< Adc0 insert, pclk2 pulse,support async */
    MD_PIS_ADC0_NORMAL          = 0x1F1U,   /**< Adc0 normal, pclk2 pulse,support async */
    MD_PIS_ADC0_WDT             = 0x1F2U,   /**< Adc0 watch dog */   
    MD_PIS_SYS_HSCO             = 0x210U,   /**< System high clock */
    MD_PIS_SYS_LSCO             = 0x211U,   /**< System low clock */
    MD_PIS_SYS_BUZ              = 0x212U,   /**< System low clock */ 
} md_pis_input_src_t;

/**
  * @brief Consumer entry
  */
typedef enum
{
    MD_PIS_CH0_TIMER0_ITR0    = 0x0090U,    /**< AD16C4T0 internal input chan0 */
    MD_PIS_CH1_TIMER0_ITR1    = 0x0091U,    /**< AD16C4T0 internal input chan1 */
    MD_PIS_CH2_TIMER0_ITR2    = 0x0092U,    /**< AD16C4T0 internal input chan2 */
    MD_PIS_CH3_TIMER0_ITR3    = 0x0093U,    /**< AD16C4T0 internal input chan3 */
    MD_PIS_CH4_TIMER0_ITR4    = 0x0094U,    /**< AD16C4T0 internal input chan4 */
    MD_PIS_CH5_TIMER0_ITR5    = 0x0095U,    /**< AD16C4T0 internal input chan5 */
    MD_PIS_CH6_TIMER0_ITR6    = 0x0096U,    /**< AD16C4T0 internal input chan6 */
    MD_PIS_CH7_TIMER0_ITR7    = 0x0097U,    /**< AD16C4T0 internal input chan7 */
    MD_PIS_CH8_TIMER0_ITR8    = 0x0098U,    /**< AD16C4T0 internal input chan8 */
    MD_PIS_CH9_TIMER0_ITR9    = 0x0099U,    /**< AD16C4T0 internal input chan9 */
    MD_PIS_CH10_TIMER0_ITR10  = 0x009AU,    /**< AD16C4T0 internal input chan10 */
    MD_PIS_CH11_TIMER0_ITR11  = 0x009BU,    /**< AD16C4T0 internal input chan11 */
    MD_PIS_CH12_TIMER0_ITR12  = 0x009CU,    /**< AD16C4T0 internal input chan0 */
    MD_PIS_CH13_TIMER0_ITR13  = 0x009DU,    /**< AD16C4T0 internal input chan1 */
    MD_PIS_CH14_TIMER0_ITR14  = 0x009EU,    /**< AD16C4T0 internal input chan2 */
    MD_PIS_CH15_TIMER0_ITR15  = 0x009FU,    /**< AD16C4T0 internal input chan3 */
    MD_PIS_CH0_TIMER2_ITR0    = 0x0090U,    /**< GP32C4T0 internal input chan0 */
    MD_PIS_CH1_TIMER2_ITR1    = 0x0091U,    /**< GP32C4T0 internal input chan1 */
    MD_PIS_CH2_TIMER2_ITR2    = 0x0092U,    /**< GP32C4T0 internal input chan2 */
    MD_PIS_CH3_TIMER2_ITR3    = 0x0093U,    /**< GP32C4T0 internal input chan3 */
    MD_PIS_CH4_TIMER2_ITR4    = 0x0094U,    /**< GP32C4T0 internal input chan4 */
    MD_PIS_CH5_TIMER2_ITR5    = 0x0095U,    /**< GP32C4T0 internal input chan5 */
    MD_PIS_CH6_TIMER2_ITR6    = 0x0096U,    /**< GP32C4T0 internal input chan6 */
    MD_PIS_CH7_TIMER2_ITR7    = 0x0097U,    /**< GP32C4T0 internal input chan7 */
    MD_PIS_CH8_TIMER2_ITR8    = 0x0098U,    /**< GP32C4T0 internal input chan8 */
    MD_PIS_CH9_TIMER2_ITR8    = 0x0099U,    /**< GP32C4T0 internal input chan9 */
    MD_PIS_CH10_TIMER2_ITR10  = 0x009AU,    /**< GP32C4T0 internal input chan10 */
    MD_PIS_CH11_TIMER2_ITR11  = 0x009BU,    /**< GP32C4T0 internal input chan11 */
    MD_PIS_CH12_TIMER2_ITR12  = 0x009CU,    /**< GP32C4T0 internal input chan12 */
    MD_PIS_CH13_TIMER2_ITR13  = 0x009DU,    /**< GP32C4T0 internal input chan13 */
    MD_PIS_CH14_TIMER2_ITR14  = 0x009EU,    /**< GP32C4T0 internal input chan14 */
    MD_PIS_CH15_TIMER2_ITR15  = 0x009FU,    /**< GP32C4T0 internal input chan15 */
    MD_PIS_CH5_TIMER2_CH1IN   = 0x1005U,    /**< GP32C4T0 input chan1 */
    MD_PIS_CH6_TIMER2_CH2IN   = 0x1106U,    /**< GP32C4T0 input chan2 */
    MD_PIS_CH7_TIMER2_CH3IN   = 0x1207U,    /**< GP32C4T0 input chan3 */
    MD_PIS_CH8_TIMER2_CH4IN   = 0x1308U,    /**< GP32C4T0 input chan4 */
    MD_PIS_CH0_TIMER3_ITR0    = 0x0090U,    /**< GP32C4T1 internal input chan0 */
    MD_PIS_CH1_TIMER3_ITR1    = 0x0091U,    /**< GP32C4T1 internal input chan1 */
    MD_PIS_CH2_TIMER3_ITR2    = 0x0092U,    /**< GP32C4T1 internal input chan2 */
    MD_PIS_CH3_TIMER3_ITR3    = 0x0093U,    /**< GP32C4T1 internal input chan3 */
    MD_PIS_CH4_TIMER3_ITR4    = 0x0094U,    /**< GP32C4T1 internal input chan4 */
    MD_PIS_CH5_TIMER3_ITR5    = 0x0095U,    /**< GP32C4T1 internal input chan5 */
    MD_PIS_CH6_TIMER3_ITR6    = 0x0096U,    /**< GP32C4T1 internal input chan6 */
    MD_PIS_CH7_TIMER3_ITR7    = 0x0097U,    /**< GP32C4T1 internal input chan7 */
    MD_PIS_CH8_TIMER3_ITR8    = 0x0098U,    /**< GP32C4T1 internal input chan8 */
    MD_PIS_CH9_TIMER3_ITR9    = 0x0099U,    /**< GP32C4T1 internal input chan9 */
    MD_PIS_CH10_TIMER3_ITR10  = 0x009AU,    /**< GP32C4T1 internal input chan10 */
    MD_PIS_CH11_TIMER3_ITR11  = 0x009BU,    /**< GP32C4T1 internal input chan11*/
    MD_PIS_CH12_TIMER3_ITR12  = 0x009CU,    /**< GP32C4T1 internal input chan12 */
    MD_PIS_CH13_TIMER3_ITR13  = 0x009DU,    /**< GP32C4T1 internal input chan13 */
    MD_PIS_CH14_TIMER3_ITR14  = 0x009EU,    /**< GP32C4T1 internal input chan14 */
    MD_PIS_CH15_TIMER3_ITR15  = 0x009FU,    /**< GP32C4T1 internal input chan15 */
    MD_PIS_CH5_TIMER3_CH1IN   = 0x1805U,    /**< GP32C4T1 input chan1 */
    MD_PIS_CH6_TIMER3_CH2IN   = 0x1906U,    /**< GP32C4T1 input chan2 */
    MD_PIS_CH7_TIMER3_CH3IN   = 0x1A07U,    /**< GP32C4T1 input chan3 */
    MD_PIS_CH8_TIMER3_CH4IN   = 0x1B08U,    /**< GP32C4T1 input chan4 */
    MD_PIS_CH0_ADC0_NORMAL    = 0x0090U,    /**< ADC0 normal transform */
    MD_PIS_CH1_ADC0_NORMAL    = 0x0091U,    /**< ADC0 normal transform */
    MD_PIS_CH2_ADC0_NORMAL    = 0x0092U,    /**< ADC0 normal transform */
    MD_PIS_CH3_ADC0_NORMAL    = 0x0093U,    /**< ADC0 normal transform */
    MD_PIS_CH4_ADC0_NORMAL    = 0x0094U,    /**< ADC0 normal transform */
    MD_PIS_CH5_ADC0_NORMAL    = 0x0095U,    /**< ADC0 normal transform */
    MD_PIS_CH6_ADC0_NORMAL    = 0x0096U,    /**< ADC0 normal transform */
    MD_PIS_CH7_ADC0_NORMAL    = 0x0097U,    /**< ADC0 normal transform */
    MD_PIS_CH8_ADC0_NORMAL    = 0x0098U,    /**< ADC0 normal transform */
    MD_PIS_CH9_ADC0_NORMAL    = 0x0099U,    /**< ADC0 normal transform */
    MD_PIS_CH10_ADC0_NORMAL   = 0x009AU,    /**< ADC0 normal transform */
    MD_PIS_CH11_ADC0_NORMAL   = 0x009BU,    /**< ADC0 normal transform */
    MD_PIS_CH12_ADC0_NORMAL   = 0x009CU,    /**< ADC0 normal transform */
    MD_PIS_CH13_ADC0_NORMAL   = 0x009DU,    /**< ADC0 normal transform */
    MD_PIS_CH14_ADC0_NORMAL   = 0x009EU,    /**< ADC0 normal transform */
    MD_PIS_CH15_ADC0_NORMAL   = 0x009FU,    /**< ADC0 normal transform */
    MD_PIS_CH0_ADC0_INSERT    = 0x0090U,    /**< ADC0 insert transform */
    MD_PIS_CH1_ADC0_INSERT    = 0x0091U,    /**< ADC0 insert transform */
    MD_PIS_CH2_ADC0_INSERT    = 0x0092U,    /**< ADC0 insert transform */
    MD_PIS_CH3_ADC0_INSERT    = 0x0093U,    /**< ADC0 insert transform */
    MD_PIS_CH4_ADC0_INSERT    = 0x0094U,    /**< ADC0 insert transform */
    MD_PIS_CH5_ADC0_INSERT    = 0x0095U,    /**< ADC0 insert transform */
    MD_PIS_CH6_ADC0_INSERT    = 0x0096U,    /**< ADC0 insert transform */
    MD_PIS_CH7_ADC0_INSERT    = 0x0097U,    /**< ADC0 insert transform */
    MD_PIS_CH8_ADC0_INSERT    = 0x0098U,    /**< ADC0 insert transform */
    MD_PIS_CH9_ADC0_INSERT    = 0x0099U,    /**< ADC0 insert transform */
    MD_PIS_CH10_ADC0_INSERT   = 0x009AU,    /**< ADC0 insert transform */
    MD_PIS_CH11_ADC0_INSERT   = 0x009BU,    /**< ADC0 insert transform */
    MD_PIS_CH12_ADC0_INSERT   = 0x009CU,    /**< ADC0 insert transform */
    MD_PIS_CH13_ADC0_INSERT   = 0x009DU,    /**< ADC0 insert transform */
    MD_PIS_CH14_ADC0_INSERT   = 0x009EU,    /**< ADC0 insert transform */
    MD_PIS_CH15_ADC0_INSERT   = 0x009FU,    /**< ADC0 insert transform */
    MD_PIS_CH0_DAC_CONVERT    = 0x0090U,    /**< DAC convert */
    MD_PIS_CH1_DAC_CONVERT    = 0x0091U,    /**< DAC convert */
    MD_PIS_CH2_DAC_CONVERT    = 0x0092U,    /**< DAC convert */
    MD_PIS_CH3_DAC_CONVERT    = 0x0093U,    /**< DAC convert */
    MD_PIS_CH4_DAC_CONVERT    = 0x0094U,    /**< DAC convert */
    MD_PIS_CH5_DAC_CONVERT    = 0x0095U,    /**< DAC convert */
    MD_PIS_CH6_DAC_CONVERT    = 0x0096U,    /**< DAC convert */
    MD_PIS_CH7_DAC_CONVERT    = 0x0097U,    /**< DAC convert */
    MD_PIS_CH8_DAC_CONVERT    = 0x0098U,    /**< DAC convert */
    MD_PIS_CH9_DAC_CONVERT    = 0x0099U,    /**< DAC convert */
    MD_PIS_CH10_DAC_CONVERT   = 0x009AU,    /**< DAC convert */
    MD_PIS_CH11_DAC_CONVERT   = 0x009BU,    /**< DAC convert */
    MD_PIS_CH12_DAC_CONVERT   = 0x009CU,    /**< DAC convert */
    MD_PIS_CH13_DAC_CONVERT   = 0x009DU,    /**< DAC convert */
    MD_PIS_CH14_DAC_CONVERT   = 0x009EU,    /**< DAC convert */
    MD_PIS_CH15_DAC_CONVERT   = 0x009FU,    /**< DAC convert */
    MD_PIS_CH0_LPTIM0_EXT0    = 0x0090U,    /**< Lptim0 external trigger 0 */
    MD_PIS_CH1_LPTIM0_EXT1    = 0x0091U,    /**< Lptim0 external trigger 1 */
    MD_PIS_CH2_LPTIM0_EXT2    = 0x0092U,    /**< Lptim0 external trigger 2 */
    MD_PIS_CH3_LPTIM0_EXT3    = 0x0093U,    /**< Lptim0 external trigger 3 */
    MD_PIS_CH4_LPTIM0_EXT4    = 0x0094U,    /**< Lptim0 external trigger 4 */
    MD_PIS_CH5_LPTIM0_EXT5    = 0x0095U,    /**< Lptim0 external trigger 5 */
    MD_PIS_CH6_LPTIM0_EXT6    = 0x0096U,    /**< Lptim0 external trigger 6 */
    MD_PIS_CH7_LPTIM0_EXT7    = 0x0097U,    /**< Lptim0 external trigger 7 */
    MD_PIS_CH0_DMA_REQUEST    = 0x0090U,    /**< DMA request */
    MD_PIS_CH1_DMA_REQUEST    = 0x0091U,    /**< DMA request */
    MD_PIS_CH2_DMA_REQUEST    = 0x0092U,    /**< DMA request */
    MD_PIS_CH3_DMA_REQUEST    = 0x0093U,    /**< DMA request */
    MD_PIS_CH4_DMA_REQUEST    = 0x0094U,    /**< DMA request */
    MD_PIS_CH5_DMA_REQUEST    = 0x0095U,    /**< DMA request */
    MD_PIS_CH6_DMA_REQUEST    = 0x0096U,    /**< DMA request */
    MD_PIS_CH7_DMA_REQUEST    = 0x0097U,    /**< DMA request */
    MD_PIS_CH8_DMA_REQUEST    = 0x0098U,    /**< DMA request */
    MD_PIS_CH9_DMA_REQUEST    = 0x0099U,    /**< DMA request */
    MD_PIS_CH10_DMA_REQUEST   = 0x009AU,    /**< DMA request */
    MD_PIS_CH11_DMA_REQUEST   = 0x009BU,    /**< DMA request */
    MD_PIS_CH12_DMA_REQUEST   = 0x009CU,    /**< DMA request */
    MD_PIS_CH13_DMA_REQUEST   = 0x009DU,    /**< DMA request */
    MD_PIS_CH14_DMA_REQUEST   = 0x009EU,    /**< DMA request */
    MD_PIS_CH15_DMA_REQUEST   = 0x009FU,    /**< DMA request */
    MD_PIS_CH15_LPUART0_RXD   = 0x081FU,    /**< Lpuart Rx data */
    MD_PIS_CH14_UART5_RXD     = 0x051EU,    /**< Uart5 Rx data */
    MD_PIS_CH13_UART4_RXD     = 0x041DU,    /**< Uart4 Rx data */
    MD_PIS_CH12_UART3_RXD     = 0x031CU,    /**< Uart3 Rx data */
    MD_PIS_CH11_UART2_RXD     = 0x021BU,    /**< Uart2 Rx data */
    MD_PIS_CH10_UART1_RXD     = 0x011AU,    /**< Uart1 Rx data */
    MD_PIS_CH9_UART0_RXD      = 0x0019U,    /**< Uart0 Rx data  */
    MD_PIS_CH8_SPI1_CLK       = 0x0F18U,    /**< Spi1 clk */
    MD_PIS_CH7_SPI1_RX        = 0x0E17U,    /**< Spi1 rx */
    MD_PIS_CH6_SPI0_CLK       = 0x0D16U,    /**< SPI0 CLK */
    MD_PIS_CH5_SPI0_RX        = 0x0C15U,    /**< SPI0 RX */
    MD_PIS_CH4_TIMER0_CH4IN   = 0x0304U,    /**< AD16C4T0 input chan4 */
    MD_PIS_CH3_TIMER0_CH3IN   = 0x0203U,    /**< AD16C4T0 input chan3 */
    MD_PIS_CH2_TIMER0_CH2IN   = 0x0102U,    /**< AD16C4T0 input chan2 */
    MD_PIS_CH1_TIMER0_CH1IN   = 0x0001U,    /**< AD16C4T0 input chan1 */
    MD_PIS_CH0_TIMER0_BRKIN   = 0x0400U,    /**< AD16C4T0 break in */
    MD_PIS_GPIO_CH0_OUTPUT    = 0x0040U,    /**< PIS channel0 OER output */
    MD_PIS_GPIO_CH1_OUTPUT    = 0x0141U,    /**< PIS channel1 OER output */
    MD_PIS_GPIO_CH2_OUTPUT    = 0x0242U,    /**< PIS channel2 OER output */
    MD_PIS_GPIO_CH3_OUTPUT    = 0x0343U,    /**< PIS channel3 OER output */
    MD_PIS_TRIG_RESERVE       = 0xFFFFU,    /**< Other Consumer */
} md_pis_trig_t;

/**
  * @brief Clock select
  */
typedef enum
{
    MD_PIS_CLK_PCLK1    = 0U,   /**< APB1 peripherals clock */
    MD_PIS_CLK_PCLK2    = 1U,   /**< APB2 peripherals clock */
    MD_PIS_CLK_SYS      = 2U,   /**< AHB peripherals clock */
    MD_PIS_CLK_RESERVE  = 3U,   /**< reserve clock */
} md_pis_clock_t;

/**
  * @brief Indirect value,no care of it.
  */
typedef enum
{
    MD_PIS_CON_0    = 0U,  /**< Con 0 */
    MD_PIS_CON_1    = 1U,  /**< Con 1 */
    MD_PIS_OER      = 4U,  /**< OER */
    MD_PIS_CON_NONE = 9U,  /**< None */
} md_pis_con_t;

/**
  * @brief Level select
  */
typedef enum
{
    MD_PIS_EDGE_NONE    = 0U,   /**< None edge */
    MD_PIS_EDGE_UP      = 1U,   /**< Up edge */
    MD_PIS_EDGE_DOWN    = 2U,   /**< Down edge */
    MD_PIS_EDGE_UP_DOWN = 3U,   /**< Up and down edge */
} md_pis_edge_t;

/**
  * @brief Output style
  */
typedef enum
{
    MD_PIS_OUT_LEVEL = 0U,  /**< Level */
    MD_PIS_OUT_PULSE = 1U,  /**< Pulse */
} md_pis_output_t;
/**
  * @brief Sync select
  */
typedef enum
{
    MD_PIS_SYN_DIRECT         = 0U, /**< Direct */
    MD_PIS_SYN_LEVEL_ASY_APB1 = 1U, /**< Producer level signal and Consumer in APB1 */
    MD_PIS_SYN_LEVEL_ASY_APB2 = 2U, /**< Producer level signal and Consumer in APB2 */
    MD_PIS_SYN_LEVEL_ASY_AHB  = 3U, /**< Producer level signal and Consumer in AHB */
    MD_PIS_SYN_PULSE_ASY_APB1 = 4U, /**< Producer Pulse signal and Consumer in APB1 */
    MD_PIS_SYN_PULSE_ASY_APB2 = 5U, /**< Producer Pulse signal and Consumer in APB2 */
    MD_PIS_SYN_PULSE_ASY_AHB  = 6U, /**< Producer Pulse signal and Consumer in AHB */
} md_pis_syncsel_t;

/**
  * @brief Pis channel
  */
typedef enum
{
    MD_PIS_CH_0 = 0U,   /**< Channel 0 */
    MD_PIS_CH_1 = 1U,   /**< Channel 1 */
    MD_PIS_CH_2 = 2U,   /**< Channel 2 */
    MD_PIS_CH_3 = 3U,   /**< Channel 3 */
    MD_PIS_CH_4 = 4U,   /**< Channel 4 */
    MD_PIS_CH_5 = 5U,   /**< Channel 5 */
    MD_PIS_CH_6 = 6U,   /**< Channel 6 */
    MD_PIS_CH_7 = 7U,   /**< Channel 7 */
    MD_PIS_CH_8 = 8U,   /**< Channel 8 */
    MD_PIS_CH_9 = 9U,   /**< Channel 9 */
    MD_PIS_CH_10 = 10U, /**< Channel 10 */
    MD_PIS_CH_11 = 11U, /**< Channel 11 */
    MD_PIS_CH_12 = 12U, /**< Channel 12 */
    MD_PIS_CH_13 = 13U, /**< Channel 13 */
    MD_PIS_CH_14 = 14U, /**< Channel 14 */
    MD_PIS_CH_15 = 15U, /**< Channel 15 */
} md_pis_ch_t;

/**
  * @brief PIS init structure definition
  */
typedef struct
{
    md_pis_input_src_t p_src;   /**< Producer entry */
    md_pis_output_t p_output;   /**< Producer signal mode */
    md_pis_clock_t p_clk;       /**< Producer module clock */
    md_pis_edge_t p_edge;       /**< Producer module pin output edge */
    md_pis_trig_t c_trig;       /**< Consumer entry */
    md_pis_clock_t c_clk;       /**< Consumer clock */
} md_pis_init_t;

/**
  * @brief PIS state structures definition
  */
typedef enum
{
    MD_PIS_STATE_RESET   = 0x00U,   /**< Peripheral is not initialized */
    MD_PIS_STATE_READY   = 0x01U,   /**< Peripheral Initialized and ready for use */
    MD_PIS_STATE_BUSY    = 0x02U,   /**< An internal process is ongoing */
    MD_PIS_STATE_TIMEOUT = 0x03U,   /**< Timeout state */
    MD_PIS_STATE_ERROR   = 0x04U,   /**< Error */
} md_pis_state_t;

/**
  * @brief PIS modulate target
  */
typedef enum
{
    MD_PIS_UART0_TX   = 0U, /**< Modulate uart0 tx */
    MD_PIS_UART1_TX   = 1U, /**< Modulate uart1 tx */
    MD_PIS_UART2_TX   = 2U, /**< Modulate uart2 tx */
    MD_PIS_UART3_TX   = 3U, /**< Modulate uart3 tx */
    MD_PIS_LPUART0_TX = 4U, /**< Modulate lpuart0 tx */
} md_pis_modu_targ_t;

/**
  * @brief PIS modulate level
  */
typedef enum
{
    MD_PIS_LOW_LEVEL  = 0U, /**< Modulate low level */
    MD_PIS_HIGH_LEVEL = 1U, /**< Modulate high level */
} md_pis_modu_level_t;

/**
  * @brief PIS modulate source
  */
typedef enum
{
    MD_PIS_SRC_NONE     = 0U,   /**< Stop modulate */
    MD_PIS_SRC_TIMER0   = 1U,   /**< Modulate source is TIMER0 */
    MD_PIS_SRC_TIMER1   = 2U,   /**< Modulate source is TIMER1 */
    MD_PIS_SRC_TIMER2   = 3U,   /**< Modulate source is TIMER2 */
    MD_PIS_SRC_TIMER3   = 4U,   /**< Modulate source is TIMER3 */
    MD_PIS_SRC_TIMER6   = 5U,   /**< Modulate source is TIMER6 */
    MD_PIS_SRC_TIMER7   = 6U,   /**< Modulate source is TIMER7 */
    MD_PIS_SRC_LPTIMER0 = 7U,   /**< Modulate source is LPTIMER0 */
    MD_PIS_SRC_BUZ      = 8U,   /**< Modulate source is buz */
} md_pis_modu_src_t;

/**
  * @brief PIS modulate channel
  */
typedef enum
{
    MD_PIS_TIMER_CH1 = 0U,  /**< Src is TIMERx and choose channel 1 */
    MD_PIS_TIMER_CH2 = 1U,  /**< Src is TIMERx and choose channel 2 */
    MD_PIS_TIMER_CH3 = 2U,  /**< Src is TIMERx and choose channel 3 */
    MD_PIS_TIMER_CH4 = 3U,  /**< Src is TIMERx and choose channel 4 */
} md_pis_modu_channel_t;
/**
  * @}
  */

/** @defgroup MD_PIS_Public_Functions PIS Public Functions
  * @{
  */
/** @defgroup MD_PIS_Public_Functions_Group2 CH_CON
  * @{
  */

/**
  * @brief  Set pis generate level singal.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_level(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    SET_BIT(PISx->CH_CON[channel], PIS_CH0_CON_LEVEL_MSK);
}

/**
  * @brief  Clear pis generate level singal.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval None
  */
__STATIC_INLINE void md_pis_clear_level(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    CLEAR_BIT(PISx->CH_CON[channel], PIS_CH0_CON_LEVEL_MSK);
}

/**
  * @brief  Set pis generate pulse singal.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_pulse(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    SET_BIT(PISx->CH_CON[channel], PIS_CH0_CON_PULSE_MSK);
}

/**
  * @brief  Set pis signal synchronization type.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  signal PIS signal.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_signal_syn_type(PIS_TypeDef *PISx, md_pis_ch_t channel, md_pis_syncsel_t signal)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_SYNCSEL_MSK, signal << PIS_CH0_CON_SYNCSEL_POSS);
}

/**
  * @brief  Get pis signal synchronization type.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis channel signal synchronization type @ref md_pis_syncsel_t
  */
__STATIC_INLINE uint32_t md_pis_get_signal_syn_type(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_SYNCSEL_MSK, PIS_CH0_CON_SYNCSEL_POSS);
}

/**
  * @brief  Set pis edge sampling clock.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  clock PIS sample clock.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_edge_sample_clock(PIS_TypeDef *PISx, md_pis_ch_t channel, md_pis_clock_t clock)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_PULCK_MSK, clock << PIS_CH0_CON_PULCK_POSS);
}

/**
  * @brief  Get pis edge sampling clock.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis edge sampling clock @ref md_pis_clock_t
  */
__STATIC_INLINE uint32_t md_pis_get_edge_sample_clock(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_PULCK_MSK, PIS_CH0_CON_PULCK_POSS);
}

/**
  * @brief  Set pis sample edge.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  edge PIS sample edge.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_sample_edge(PIS_TypeDef *PISx, md_pis_ch_t channel, md_pis_edge_t edge)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_EDGS_MSK, edge << PIS_CH0_CON_EDGS_POSS);
}

/**
  * @brief  Get pis sample edge.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis edge sampling clock @ref md_pis_edge_t
  */
__STATIC_INLINE uint32_t md_pis_get_sample_edge(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_EDGS_MSK, PIS_CH0_CON_EDGS_POSS);
}

/**
  * @brief  Set pis input source.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  src PIS input source.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_input_src(PIS_TypeDef *PISx, md_pis_ch_t channel, md_pis_input_src_t src)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_SRCS_MSK, src << PIS_CH0_CON_SRCS_POSS);
}

/**
  * @brief  Get pis input source.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source @ref md_pis_input_src_t
  */
__STATIC_INLINE uint32_t md_pis_get_input_src(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_SRCS_MSK, PIS_CH0_CON_SRCS_POSS);
}

/**
  * @brief  Set pis input source gpiob.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  pin PIS input source gpiob.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_gpiob(PIS_TypeDef *PISx, md_pis_ch_t channel, uint8_t pin)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, pin << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source gpiob.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source gpiob.
  */
__STATIC_INLINE uint32_t md_pis_get_src_gpiob(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source acmp.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  acmp PIS input source acmp.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_acmp(PIS_TypeDef *PISx, md_pis_ch_t channel, uint8_t acmp)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, acmp << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source acmp.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source acmp.
  */
__STATIC_INLINE uint32_t md_pis_get_src_acmp(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source dac0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  dac0 PIS input source dac0.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_dac0(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t dac0)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, dac0 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source dac0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source dac0.
  */
__STATIC_INLINE uint32_t md_pis_get_src_dac0(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source adc0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  adc0 PIS input source adc0.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_adc0(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t adc0)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, adc0 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source adc0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source adc0.
  */
__STATIC_INLINE uint32_t md_pis_get_src_adc0(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source lvd.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  lvd PIS input source lvd.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_lvd(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t lvd)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, lvd << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source lvd.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source lvd.
  */
__STATIC_INLINE uint32_t md_pis_get_src_lvd(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source uart0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  uart0 PIS input source uart0.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_uart0(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t uart0)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, uart0 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source uart0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source uart0.
  */
__STATIC_INLINE uint32_t md_pis_get_src_uart0(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source uart1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  uart1 PIS input source uart1.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_uart1(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t uart1)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, uart1 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source uart1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source uart1.
  */
__STATIC_INLINE uint32_t md_pis_get_src_uart1(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source uart2.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  uart2 PIS input source uart2.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_uart2(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t uart2)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, uart2 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source uart2.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source uart2.
  */
__STATIC_INLINE uint32_t md_pis_get_src_uart2(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source uart3.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  uart3 PIS input source uart3.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_uart3(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t uart3)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, uart3 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source uart3.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source uart3.
  */
__STATIC_INLINE uint32_t md_pis_get_src_uart3(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source uart4.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  uart4 PIS input source uart4.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_uart4(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t uart4)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, uart4 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source uart4.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source uart4.
  */
__STATIC_INLINE uint32_t md_pis_get_src_uart4(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source uart5.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  uart5 PIS input source uart5.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_uart5(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t uart5)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, uart5 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source uart5.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source uart5.
  */
__STATIC_INLINE uint32_t md_pis_get_src_uart5(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source spi0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  spi0 PIS input source spi0.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_spi0(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t spi0)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, spi0 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source spi0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source spi0.
  */
__STATIC_INLINE uint32_t md_pis_get_src_spi0(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source spi1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  spi1 PIS input source spi1.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_spi1(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t spi1)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, spi1 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source spi1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source spi1.
  */
__STATIC_INLINE uint32_t md_pis_get_src_spi1(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source i2c0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  i2c0 PIS input source i2c0.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_i2c0(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t i2c0)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, i2c0 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source i2c0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source i2c0.
  */
__STATIC_INLINE uint32_t md_pis_get_src_i2c0(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source i2c1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  i2c1 PIS input source i2c1.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_i2c1(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t i2c1)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, i2c1 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source i2c1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source i2c1
  */
__STATIC_INLINE uint32_t md_pis_get_src_i2c1(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source tim0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  tim0 PIS input source tim0.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_tim0(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t tim0)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, tim0 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source tim0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source tim0
  */
__STATIC_INLINE uint32_t md_pis_get_src_tim0(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source tim1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  tim1 PIS input source tim1.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_tim1(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t tim1)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, tim1 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source tim1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source tim1
  */
__STATIC_INLINE uint32_t md_pis_get_src_tim1(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source tim2.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  tim2 PIS input source tim2.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_tim2(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t tim2)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, tim2 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source tim2.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source tim2
  */
__STATIC_INLINE uint32_t md_pis_get_src_tim2(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source tim3.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  tim3 PIS input source tim3.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_tim3(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t tim3)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, tim3 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source tim3.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source tim3
  */
__STATIC_INLINE uint32_t md_pis_get_src_tim3(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source rtc.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  rtc PIS input source rtc.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_rtc(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t rtc)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, rtc << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source rtc.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source rtc
  */
__STATIC_INLINE uint32_t md_pis_get_src_rtc(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source lptim0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  lptim0 PIS input source lptim0.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_lptim0(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t lptim0)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, lptim0 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source lptim0.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source lptim0
  */
__STATIC_INLINE uint32_t md_pis_get_src_lptim0(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source dma.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  dma PIS input source dma.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_dma(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t dma)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, dma << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source dma.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source dma.
  */
__STATIC_INLINE uint32_t md_pis_get_src_dma(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Set pis input source adc1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @param  adc1 PIS input source adc1.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_src_adc1(PIS_TypeDef *PISx, md_pis_ch_t channel, uint32_t adc1)
{
    MODIFY_REG(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, adc1 << PIS_CH0_CON_MSIGS_POSS);
}

/**
  * @brief  Get pis input source adc1.
  * @param  PISx PIS Instance.
  * @param  channel PIS channel.
  * @retval pis input source adc1.
  */
__STATIC_INLINE uint32_t md_pis_get_src_adc1(PIS_TypeDef *PISx, md_pis_ch_t channel)
{
    return READ_BITS(PISx->CH_CON[channel], PIS_CH0_CON_MSIGS_MSK, PIS_CH0_CON_MSIGS_POSS);
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group3 CH_OER
  * @{
  */
/**
  * @brief  Disable pis channel 3 output to port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_ch3_output(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->CH_OER, PIS_CH_OER_CH3OE_MSK);
}

/**
  * @brief  Enable pis channel 3 output to port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_disable_ch3_output(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->CH_OER, PIS_CH_OER_CH3OE_MSK);
}

/**
  * @brief  Indicate if pis channel 3 output to port is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_ch3_output(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->CH_OER, PIS_CH_OER_CH3OE_MSK) == (PIS_CH_OER_CH3OE_MSK));
}

/**
  * @brief  Disable pis channel 2 output to port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_ch2_output(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->CH_OER, PIS_CH_OER_CH2OE_MSK);
}

/**
  * @brief  Enable pis channel 2 output to port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_disable_ch2_output(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->CH_OER, PIS_CH_OER_CH2OE_MSK);
}

/**
  * @brief  Indicate if pis channel 2 output to port is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_ch2_output(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->CH_OER, PIS_CH_OER_CH2OE_MSK) == (PIS_CH_OER_CH2OE_MSK));
}

/**
  * @brief  Disable pis channel 1 output to port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_ch1_output(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->CH_OER, PIS_CH_OER_CH1OE_MSK);
}

/**
  * @brief  Enable pis channel 1 output to port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_disable_ch1_output(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->CH_OER, PIS_CH_OER_CH1OE_MSK);
}

/**
  * @brief  Indicate if pis channel 1 output to port is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_ch1_output(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->CH_OER, PIS_CH_OER_CH1OE_MSK) == (PIS_CH_OER_CH1OE_MSK));
}

/**
  * @brief  Disable pis channel 0 output to port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_ch0_output(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->CH_OER, PIS_CH_OER_CH0OE_MSK);
}

/**
  * @brief  Enable pis channel 0 output to port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_disable_ch0_output(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->CH_OER, PIS_CH_OER_CH0OE_MSK);
}

/**
  * @brief  Indicate if pis channel 0 output to port is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_ch0_output(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->CH_OER, PIS_CH_OER_CH0OE_MSK) == (PIS_CH_OER_CH0OE_MSK));
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group4 TAR_CON0
  * @{
  */
/**
  * @brief  Set pis tim3 input capture ch4 chose ch8.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim3_ch4in_ch8(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH4IN_SEL_MSK);
}

/**
  * @brief  Set pis tim3 input capture ch4 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim3_ch4in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH4IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim3 input capture ch4 chose ch8 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim3_ch4in_ch8(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH4IN_SEL_MSK) == (PIS_TAR_CON0_TIM3_CH4IN_SEL_MSK));
}

/**
  * @brief  Set pis tim3 input capture ch3 chose ch7.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim3_ch3in_ch7(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH3IN_SEL_MSK);
}

/**
  * @brief  Set pis tim3 input capture ch3 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim3_ch3in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH3IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim3 input capture ch3 chose ch7 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim3_ch3in_ch7(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH3IN_SEL_MSK) == (PIS_TAR_CON0_TIM3_CH3IN_SEL_MSK));
}

/**
  * @brief  Set pis tim3 input capture ch2 chose ch6.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim3_ch2in_ch6(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH2IN_SEL_MSK);
}

/**
  * @brief  Set pis tim3 input capture ch2 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim3_ch2in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH2IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim3 input capture ch2 chose ch6 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim3_ch2in_ch6(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH2IN_SEL_MSK) == (PIS_TAR_CON0_TIM3_CH2IN_SEL_MSK));
}

/**
  * @brief  Set pis tim3 input capture ch1 chose ch5.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim3_ch1in_ch5(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH1IN_SEL_MSK);
}

/**
  * @brief  Set pis tim3 input capture ch1 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim3_ch1in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH1IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim3 input capture ch1 chose ch5 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim3_ch1in_ch5(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM3_CH1IN_SEL_MSK) == (PIS_TAR_CON0_TIM3_CH1IN_SEL_MSK));
}

/**
  * @brief  Set pis tim2 input capture ch4 chose ch8.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim2_ch4in_ch8(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH4IN_SEL_MSK);
}

/**
  * @brief  Set pis tim2 input capture ch4 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim2_ch4in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH4IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim2 input capture ch4 chose ch8 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim2_ch4in_ch8(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH4IN_SEL_MSK) == (PIS_TAR_CON0_TIM2_CH4IN_SEL_MSK));
}

/**
  * @brief  Set pis tim2 input capture ch3 chose ch7.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim2_ch3in_ch7(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH3IN_SEL_MSK);
}

/**
  * @brief  Set pis tim2 input capture ch3 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim2_ch3in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH3IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim2 input capture ch3 chose ch7 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim2_ch3in_ch7(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH3IN_SEL_MSK) == (PIS_TAR_CON0_TIM2_CH3IN_SEL_MSK));
}

/**
  * @brief  Set pis tim2 input capture ch2 chose ch6.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim2_ch2in_ch6(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH2IN_SEL_MSK);
}

/**
  * @brief  Set pis tim2 input capture ch2 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim2_ch2in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH2IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim2 input capture ch2 chose ch6 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim2_ch2in_ch6(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH2IN_SEL_MSK) == (PIS_TAR_CON0_TIM2_CH2IN_SEL_MSK));
}

/**
  * @brief  Set pis tim2 input capture ch1 chose ch5.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim2_ch1in_ch5(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH1IN_SEL_MSK);
}

/**
  * @brief  Set pis tim2 input capture ch1 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim2_ch1in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH1IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim2 input capture ch1 chose ch5 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim2_ch1in_ch5(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM2_CH1IN_SEL_MSK) == (PIS_TAR_CON0_TIM2_CH1IN_SEL_MSK));
}

/**
  * @brief  Set pis tim0 input capture ch4 chose ch8.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim0_ch4in_ch8(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH4IN_SEL_MSK);
}

/**
  * @brief  Set pis tim0 input capture ch4 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim0_ch4in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH4IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim0 input capture ch4 chose ch8 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim0_ch4in_ch8(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH4IN_SEL_MSK) == (PIS_TAR_CON0_TIM0_CH4IN_SEL_MSK));
}

/**
  * @brief  Set pis tim0 input capture ch3 chose ch7.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim0_ch3in_ch7(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH3IN_SEL_MSK);
}

/**
  * @brief  Set pis tim0 input capture ch3 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim0_ch3in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH3IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim0 input capture ch3 chose ch7 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim0_ch3in_ch7(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH3IN_SEL_MSK) == (PIS_TAR_CON0_TIM0_CH3IN_SEL_MSK));
}

/**
  * @brief  Set pis tim0 input capture ch2 chose ch6.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim0_ch2in_ch6(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH2IN_SEL_MSK);
}

/**
  * @brief  Set pis tim0 input capture ch2 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim0_ch2in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH2IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim0 input capture ch2 chose ch6 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim0_ch2in_ch6(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH2IN_SEL_MSK) == (PIS_TAR_CON0_TIM0_CH2IN_SEL_MSK));
}

/**
  * @brief  Set pis tim0 input capture ch1 chose ch5.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim0_ch1in_ch5(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH1IN_SEL_MSK);
}

/**
  * @brief  Set pis tim0 input capture ch1 chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_tim0_ch1in_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH1IN_SEL_MSK);
}

/**
  * @brief  Indicate if pis tim0 input capture ch1 chose ch5 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_tim0_ch1in_ch5(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON0, PIS_TAR_CON0_TIM0_CH1IN_SEL_MSK) == (PIS_TAR_CON0_TIM0_CH1IN_SEL_MSK));
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group5 TAR_CON1
  * @{
  */
/**
  * @brief  Set pis spi1 clk input chose ch8.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_spi1_clk_ch8(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI1_CLK_SEL_MSK);
}

/**
  * @brief  Set pis spi1 clk input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_spi1_clk_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI1_CLK_SEL_MSK);
}

/**
  * @brief  Indicate if pis spi1 clk input chose ch8 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_spi1_clk_ch8(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI1_CLK_SEL_MSK) == (PIS_TAR_CON1_SPI1_CLK_SEL_MSK));
}

/**
  * @brief  Set pis spi1 rx input chose ch7.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_spi1_rx_ch7(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI1_RX_SEL_MSK);
}

/**
  * @brief  Set pis spi1 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_spi1_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI1_RX_SEL_MSK);
}

/**
  * @brief  Indicate if pis spi1 rx input chose ch7 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_spi1_rx_ch7(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI1_RX_SEL_MSK) == (PIS_TAR_CON1_SPI1_RX_SEL_MSK));
}

/**
  * @brief  Set pis spi0 clk input chose ch6.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_spi0_clk_ch6(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI0_CLK_SEL_MSK);
}

/**
  * @brief  Set pis spi0 clk input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_spi0_clk_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI0_CLK_SEL_MSK);
}

/**
  * @brief  Indicate if pis spi0 clk input chose ch6 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_spi0_clk_ch6(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI0_CLK_SEL_MSK) == (PIS_TAR_CON1_SPI0_CLK_SEL_MSK));
}

/**
  * @brief  Set pis spi0 rx input chose ch5.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_spi0_rx_ch5(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI0_RX_SEL_MSK);
}

/**
  * @brief  Set pis spi0 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_spi0_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI0_RX_SEL_MSK);
}

/**
  * @brief  Indicate if pis spi0 rx input chose ch5 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_spi0_rx_ch5(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_SPI0_RX_SEL_MSK) == (PIS_TAR_CON1_SPI0_RX_SEL_MSK));
}

/**
  * @brief  Set pis lpuart0 rx input chose ch15.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_lpuart0_rx_ch15(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_LPUART0_RXD_SEL_MSK);
}

/**
  * @brief  Set pis lpuart0 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_lpuart0_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_LPUART0_RXD_SEL_MSK);
}

/**
  * @brief  Indicate if pis lpuart0 rx input chose ch15 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_lpuart0_rx_ch15(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_LPUART0_RXD_SEL_MSK) == (PIS_TAR_CON1_LPUART0_RXD_SEL_MSK));
}

/**
  * @brief  Set pis uart5 rx input chose ch14.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart5_rx_ch14(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART5_RXD_SEL_MSK);
}

/**
  * @brief  Set pis uart5 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart5_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART5_RXD_SEL_MSK);
}

/**
  * @brief  Indicate if pis uart5 rx input chose ch14 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_uart5_rx_ch14(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART5_RXD_SEL_MSK) == (PIS_TAR_CON1_UART5_RXD_SEL_MSK));
}

/**
  * @brief  Set pis uart4 rx input chose ch13.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart4_rx_ch13(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART4_RXD_SEL_MSK);
}

/**
  * @brief  Set pis uart4 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart4_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART4_RXD_SEL_MSK);
}

/**
  * @brief  Indicate if pis uart4 rx input chose ch14 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_uart4_rx_ch13(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART4_RXD_SEL_MSK) == (PIS_TAR_CON1_UART4_RXD_SEL_MSK));
}

/**
  * @brief  Set pis uart3 rx input chose ch12.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart3_rx_ch12(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART3_RXD_SEL_MSK);
}

/**
  * @brief  Set pis uart3 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart3_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART3_RXD_SEL_MSK);
}

/**
  * @brief  Indicate if pis uart3 rx input chose ch12 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_uart3_rx_ch12(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART3_RXD_SEL_MSK) == (PIS_TAR_CON1_UART3_RXD_SEL_MSK));
}

/**
  * @brief  Set pis uart2 rx input chose ch11.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart2_rx_ch11(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART2_RXD_SEL_MSK);
}

/**
  * @brief  Set pis uart2 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart2_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART2_RXD_SEL_MSK);
}

/**
  * @brief  Indicate if pis uart2 rx input chose ch11 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_uart2_rx_ch11(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART2_RXD_SEL_MSK) == (PIS_TAR_CON1_UART2_RXD_SEL_MSK));
}

/**
  * @brief  Set pis uart1 rx input chose ch10.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart1_rx_ch10(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART1_RXD_SEL_MSK);
}

/**
  * @brief  Set pis uart1 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart1_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART1_RXD_SEL_MSK);
}

/**
  * @brief  Indicate if pis uart1 rx input chose ch10 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_uart1_rx_ch10(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART1_RXD_SEL_MSK) == (PIS_TAR_CON1_UART1_RXD_SEL_MSK));
}

/**
  * @brief  Set pis uart0 rx input chose ch9.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart0_rx_ch9(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART0_RXD_SEL_MSK);
}

/**
  * @brief  Set pis uart0 rx input chose port.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_enable_uart0_rx_port(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART0_RXD_SEL_MSK);
}

/**
  * @brief  Indicate if pis uart0 rx input chose ch10 is enabled
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_enabled_uart0_rx_ch9(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->TAR_CON1, PIS_TAR_CON1_UART0_RXD_SEL_MSK) == (PIS_TAR_CON1_UART0_RXD_SEL_MSK));
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group6 UART0_TXMCR
  * @{
  */
/**
  * @brief  Set pis uart0 tx modulation level high.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart0_tx_mod_high(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->UART0_TXMCR, PIS_UART0_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Set pis uart0 tx modulation level low.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart0_tx_mod_low(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->UART0_TXMCR, PIS_UART0_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Indicate if pis uart0 tx modulation level high is set
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_set_uart0_tx_mod_high(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->UART0_TXMCR, PIS_UART0_TXMCR_TXMLVLS_MSK) == (PIS_UART0_TXMCR_TXMLVLS_MSK));
}

/**
  * @brief  Set pis uart0 modulation source.
  * @param  PISx PIS Instance.
  * @param  src PIS uart0 modulation source.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart0_mod_src(PIS_TypeDef *PISx, md_pis_modu_src_t src)
{
    MODIFY_REG(PISx->UART0_TXMCR, PIS_UART0_TXMCR_TXMSS_MSK, src << PIS_UART0_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Get pis uart0 modulation source.
  * @param  PISx PIS Instance.
  * @retval uart0 modulation source @ref md_pis_modu_src_t
  */
__STATIC_INLINE uint32_t md_pis_get_uart0_mod_src(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->UART0_TXMCR, PIS_UART0_TXMCR_TXMSS_MSK, PIS_UART0_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Set pis uart0 modulation timx channel.
  * @param  PISx PIS Instance.
  * @param  channel PIS timx modulation channel.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart0_mod_tim_channel(PIS_TypeDef *PISx, md_pis_modu_channel_t channel)
{
    MODIFY_REG(PISx->UART0_TXMCR, PIS_UART0_TXMCR_TXSIGS_MSK, channel << PIS_UART0_TXMCR_TXSIGS_POSS);
}

/**
  * @brief  Get pis uart0 timx modulation channel.
  * @param  PISx PIS Instance.
  * @retval uart0 timx modulation channel @ref md_pis_modu_channel_t
  */
__STATIC_INLINE uint32_t md_pis_get_uart0_mod_tim_channel(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->UART0_TXMCR, PIS_UART0_TXMCR_TXSIGS_MSK, PIS_UART0_TXMCR_TXSIGS_POSS);
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group7 UART1_TXMCR
  * @{
  */
/**
  * @brief  Set pis uart1 tx modulation level high.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart1_tx_mod_high(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->UART1_TXMCR, PIS_UART1_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Set pis uart1 tx modulation level low.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart1_tx_mod_low(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->UART1_TXMCR, PIS_UART1_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Indicate if pis uart1 tx modulation level high is set
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_set_uart1_tx_mod_high(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->UART1_TXMCR, PIS_UART1_TXMCR_TXMLVLS_MSK) == (PIS_UART1_TXMCR_TXMLVLS_MSK));
}

/**
  * @brief  Set pis uart1 modulation source.
  * @param  PISx PIS Instance.
  * @param  src PIS uart1 modulation source.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart1_mod_src(PIS_TypeDef *PISx, md_pis_modu_src_t src)
{
    MODIFY_REG(PISx->UART1_TXMCR, PIS_UART1_TXMCR_TXMSS_MSK, src << PIS_UART1_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Get pis uart1 modulation source.
  * @param  PISx PIS Instance.
  * @retval uart1 modulation source @ref md_pis_modu_src_t
  */
__STATIC_INLINE uint32_t md_pis_get_uart1_mod_src(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->UART1_TXMCR, PIS_UART1_TXMCR_TXMSS_MSK, PIS_UART1_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Set pis uart1 modulation timx channel.
  * @param  PISx PIS Instance.
  * @param  channel PIS timx modulation channel.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart1_mod_tim_channel(PIS_TypeDef *PISx, md_pis_modu_channel_t channel)
{
    MODIFY_REG(PISx->UART1_TXMCR, PIS_UART1_TXMCR_TXSIGS_MSK, channel << PIS_UART1_TXMCR_TXSIGS_POSS);
}

/**
  * @brief  Get pis uart1 timx modulation channel.
  * @param  PISx PIS Instance.
  * @retval uart1 timx modulation channel @ref md_pis_modu_channel_t
  */
__STATIC_INLINE uint32_t md_pis_get_uart1_mod_tim_channel(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->UART1_TXMCR, PIS_UART1_TXMCR_TXSIGS_MSK, PIS_UART1_TXMCR_TXSIGS_POSS);
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group8 UART2_TXMCR
  * @{
  */
/**
  * @brief  Set pis uart2 tx modulation level high.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart2_tx_mod_high(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->UART2_TXMCR, PIS_UART2_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Set pis uart2 tx modulation level low.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart2_tx_mod_low(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->UART2_TXMCR, PIS_UART2_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Indicate if pis uart2 tx modulation level high is set
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_set_uart2_tx_mod_high(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->UART2_TXMCR, PIS_UART2_TXMCR_TXMLVLS_MSK) == (PIS_UART2_TXMCR_TXMLVLS_MSK));
}

/**
  * @brief  Set pis uart2 modulation source.
  * @param  PISx PIS Instance.
  * @param  src PIS uart2 modulation source.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart2_mod_src(PIS_TypeDef *PISx, md_pis_modu_src_t src)
{
    MODIFY_REG(PISx->UART2_TXMCR, PIS_UART2_TXMCR_TXMSS_MSK, src << PIS_UART2_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Get pis uart2 modulation source.
  * @param  PISx PIS Instance.
  * @retval uart2 modulation source @ref md_pis_modu_src_t
  */
__STATIC_INLINE uint32_t md_pis_get_uart2_mod_src(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->UART2_TXMCR, PIS_UART2_TXMCR_TXMSS_MSK, PIS_UART2_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Set pis uart2 modulation timx channel.
  * @param  PISx PIS Instance.
  * @param  channel PIS timx modulation channel.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart2_mod_tim_channel(PIS_TypeDef *PISx, md_pis_modu_channel_t channel)
{
    MODIFY_REG(PISx->UART2_TXMCR, PIS_UART2_TXMCR_TXSIGS_MSK, channel << PIS_UART2_TXMCR_TXSIGS_POSS);
}

/**
  * @brief  Get pis uart2 timx modulation channel.
  * @param  PISx PIS Instance.
  * @retval uart2 timx modulation channel @ref md_pis_modu_channel_t
  */
__STATIC_INLINE uint32_t md_pis_get_uart2_mod_tim_channel(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->UART2_TXMCR, PIS_UART2_TXMCR_TXSIGS_MSK, PIS_UART2_TXMCR_TXSIGS_POSS);
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group9 UART3_TXMCR
  * @{
  */
/**
  * @brief  Set pis uart3 tx modulation level high.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart3_tx_mod_high(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->UART3_TXMCR, PIS_UART3_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Set pis uart3 tx modulation level low.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart3_tx_mod_low(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->UART3_TXMCR, PIS_UART3_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Indicate if pis uart3 tx modulation level high is set
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_set_uart3_tx_mod_high(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->UART3_TXMCR, PIS_UART3_TXMCR_TXMLVLS_MSK) == (PIS_UART3_TXMCR_TXMLVLS_MSK));
}

/**
  * @brief  Set pis uart3 modulation source.
  * @param  PISx PIS Instance.
  * @param  src PIS uart3 modulation source.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart3_mod_src(PIS_TypeDef *PISx, md_pis_modu_src_t src)
{
    MODIFY_REG(PISx->UART3_TXMCR, PIS_UART3_TXMCR_TXMSS_MSK, src << PIS_UART3_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Get pis uart3 modulation source.
  * @param  PISx PIS Instance.
  * @retval uart3 modulation source @ref md_pis_modu_src_t
  */
__STATIC_INLINE uint32_t md_pis_get_uart3_mod_src(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->UART3_TXMCR, PIS_UART3_TXMCR_TXMSS_MSK, PIS_UART3_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Set pis uart3 modulation timx channel.
  * @param  PISx PIS Instance.
  * @param  channel PIS timx modulation channel.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_uart3_mod_tim_channel(PIS_TypeDef *PISx, md_pis_modu_channel_t channel)
{
    MODIFY_REG(PISx->UART3_TXMCR, PIS_UART3_TXMCR_TXSIGS_MSK, channel << PIS_UART3_TXMCR_TXSIGS_POSS);
}

/**
  * @brief  Get pis uart3 timx modulation channel.
  * @param  PISx PIS Instance.
  * @retval uart3 timx modulation channel @ref md_pis_modu_channel_t
  */
__STATIC_INLINE uint32_t md_pis_get_uart3_mod_tim_channel(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->UART3_TXMCR, PIS_UART3_TXMCR_TXSIGS_MSK, PIS_UART3_TXMCR_TXSIGS_POSS);
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group10 LPUART0_TXMCR
  * @{
  */
/**
  * @brief  Set pis lpuart0 tx modulation level high.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_lpuart0_tx_mod_high(PIS_TypeDef *PISx)
{
    SET_BIT(PISx->LPUART0_TXMCR, PIS_LPUART0_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Set pis lpuart0 tx modulation level low.
  * @param  PISx PIS Instance.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_lpuart0_tx_mod_low(PIS_TypeDef *PISx)
{
    CLEAR_BIT(PISx->LPUART0_TXMCR, PIS_LPUART0_TXMCR_TXMLVLS_MSK);
}

/**
  * @brief  Indicate if pis lpuart0 tx modulation level high is set
  * @param  PISx PIS Instance.
  * @retval State of bit (1 or 0).
  */
__STATIC_INLINE uint32_t md_pis_is_set_lpuart0_tx_mod_high(PIS_TypeDef *PISx)
{
    return (uint32_t)(READ_BIT(PISx->LPUART0_TXMCR, PIS_LPUART0_TXMCR_TXMLVLS_MSK) == (PIS_LPUART0_TXMCR_TXMLVLS_MSK));
}

/**
  * @brief  Set pis lpuart0 modulation source.
  * @param  PISx PIS Instance.
  * @param  src PIS lpuart0 modulation source.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_lpuart0_mod_src(PIS_TypeDef *PISx, md_pis_modu_src_t src)
{
    MODIFY_REG(PISx->LPUART0_TXMCR, PIS_LPUART0_TXMCR_TXMSS_MSK, src << PIS_LPUART0_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Get pis lpuart0 modulation source.
  * @param  PISx PIS Instance.
  * @retval lpuart0 modulation source @ref md_pis_modu_src_t
  */
__STATIC_INLINE uint32_t md_pis_get_lpuart0_mod_src(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->LPUART0_TXMCR, PIS_LPUART0_TXMCR_TXMSS_MSK, PIS_LPUART0_TXMCR_TXMSS_POSS);
}

/**
  * @brief  Set pis lpuart0 modulation timx channel.
  * @param  PISx PIS Instance.
  * @param  channel PIS timx modulation channel.
  * @retval None
  */
__STATIC_INLINE void md_pis_set_lpuart0_mod_tim_channel(PIS_TypeDef *PISx, md_pis_modu_channel_t channel)
{
    MODIFY_REG(PISx->LPUART0_TXMCR, PIS_LPUART0_TXMCR_TXSIGS_MSK, channel << PIS_LPUART0_TXMCR_TXSIGS_POSS);
}

/**
  * @brief  Get pis lpuart0 timx modulation channel.
  * @param  PISx PIS Instance.
  * @retval uart3 timx modulation channel @ref md_pis_modu_channel_t
  */
__STATIC_INLINE uint32_t md_pis_get_lpuart0_mod_tim_channel(PIS_TypeDef *PISx)
{
    return READ_BITS(PISx->LPUART0_TXMCR, PIS_LPUART0_TXMCR_TXSIGS_MSK, PIS_LPUART0_TXMCR_TXSIGS_POSS);
}
/**
  * @}
  */
/** @defgroup MD_PIS_Public_Functions_Group1 Initialization
  * @{
  */
extern void md_pis_reset(void);
extern md_status_t md_pis_init(md_pis_init_t *init);
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif
