/**
  *********************************************************************************
  *
  * @file    md_msc.h
  * @brief   Header file of MSC module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __MD_MSC_H__
#define __MD_MSC_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"


/** @addtogroup Micro_Driver
  * @{
  */

/**
  * @brief  Flash iap unlock.
  */
static void MD_IAP_Unlock(void)
{
    MSC->FLASHKEY = 0x8ACE0246;
    MSC->FLASHKEY = 0x9BDF1357;
}

/**
  * @brief  Flash iap lock.
  */
static void MD_IAP_Lock(void)
{
    MSC->FLASHKEY = 0xFFFFFFFF;
    MSC->FLASHKEY = 0xFFFFFFFF;
}


/** @defgroup MD_MSC MSC
  * @brief MSC module driver
  * @{
  */

/**
  * @defgroup MD_MSC_Private_Macros MSC Private Macros
  * @{
  */
#define MD_MSC_WSP_ADDR 0x10000000U
#define MD_MSC_PE_ADDR  0x10000004U
#define MD_MSC_WP_ADDR  0x10000008U

#define MD_MSC_WSP_DF   0x10000010U
#define MD_MSC_PE_DF    0x10000014U
#define MD_MSC_WP_DF    0x10000018U

/**
  * @}
  */

/** @defgroup MD_MSC_Private_Types MSC Private Types
  * @{
  */
typedef uint32_t (*MD_MSC_PE)(uint32_t addr, uint32_t sec);
typedef uint32_t (*MD_MSC_WP)(uint32_t addr, uint32_t data);
typedef uint32_t (*MD_MSC_WSP)(uint32_t addr, uint8_t *data, uint32_t len, uint32_t erase);
/**
  * @}
  */

/** @defgroup IAP_Public_Functions MSC Public Functions
  * @{
  */
/** @defgroup IAP_Public_Functions_Group1 FLASH Operation Functions
  * @{
  */
/**
  * @brief  Erases a specified page.
  * @param  addr: The beginning address of the page to be erased.
  * @retval The result:
  *           - 0: SUCCESS
  *           - 1: ERROR
  */
__STATIC_INLINE__ uint32_t md_msc_erase_page(uint32_t addr)
{
    uint32_t status;
    MD_MSC_PE pe = (MD_MSC_PE)(*(uint32_t *)MD_MSC_PE_ADDR);

    MD_IAP_Unlock();
    __disable_irq();
    status = (*pe)(addr, 0);
    __enable_irq();
    MD_IAP_Lock();

    return !status;
}

/**
  * @brief  Erases a specified page of dataflash.
  * @param  addr: The beginning address of the page to be erased.
  * @retval The result:
  *           - 0: SUCCESS
  *           - 1: ERROR
  */
__STATIC_INLINE__ uint32_t md_msc_erase_page_df(uint32_t addr)
{
    uint32_t status;
    MD_MSC_PE iap_pe = (MD_MSC_PE)(*(uint32_t *)MD_MSC_PE_DF);

    MD_IAP_Unlock();
    __disable_irq();
    status = (*iap_pe)(addr, 0);
    __enable_irq();
    MD_IAP_Lock();

    return !status;
}

/**
  * @brief  Programs a word at a specified address.
  * @param  addr: Specifies the address to be programmed.
  *         Bit0-1 must be zero.
  * @param  data: Specifies the data to be programmed.
  * @retval The result:
  *           - 0: SUCCESS
  *           - 1: ERROR
  */
__STATIC_INLINE__ uint32_t md_msc_program_word(uint32_t addr, uint32_t data)
{
    uint32_t status;
    MD_MSC_WP wp = (MD_MSC_WP)(*(uint32_t *)MD_MSC_WP_ADDR);

    if (addr & 0x3)
        return 1;

    MD_IAP_Unlock();
    __disable_irq();
    status = (*wp)(addr, data);
    __enable_irq();
    MD_IAP_Lock();

    return !status;
}

/**
  * @brief  Programs a word at a specified address of dataflash.
  * @param  addr: Specifies the address to be programmed.
  *         Bit0-1 must be zero.
  * @param  data: Specifies the data to be programmed.
  * @retval The result:
  *           - 0: SUCCESS
  *           - 1: ERROR
  */
__STATIC_INLINE__ uint32_t md_msc_program_word_df(uint32_t addr, uint32_t data)
{
    uint32_t status;
    MD_MSC_WP iap_wp = (MD_MSC_WP)(*(uint32_t *)MD_MSC_WP_DF);

    if (addr & 0x3)
        return 1;

    MD_IAP_Unlock();
    __disable_irq();
    status = (*iap_wp)(addr, data);
    __enable_irq();
    MD_IAP_Lock();

    return !status;
}

/**
  * @brief  Programs datas at a specified address.
  * @param  addr: Specifies the address to be programmed.
  *         Bit0-1 must be zero.
  * @param  data: Specifies the data to be programmed.
  * @param  len: Specifies the data length to be programmed.
  *         Bit0-1 must be zero.
  * @param  erase: Erase page flag before programming.
  * @retval The result:
  *           - 0: SUCCESS
  *           - 1: ERROR
  */
__STATIC_INLINE__ uint32_t md_msc_program_words(uint32_t addr, uint8_t *data, uint32_t len, uint32_t erase)
{
    uint32_t status;
    MD_MSC_WSP wsp = (MD_MSC_WSP)(*(uint32_t *)MD_MSC_WSP_ADDR);

    if ((addr & 0x3) || (len & 0x3))
        return 1;

    MD_IAP_Unlock();
    __disable_irq();
    status = (*wsp)(addr, data, len, erase);
    __enable_irq();
    MD_IAP_Lock();

    return !status;
}

/**
  * @brief  Programs datas at a specified address of dataflash.
  * @param  addr: Specifies the address to be programmed.
  *         Bit0-1 must be zero.
  * @param  data: Specifies the data to be programmed.
  * @param  len: Specifies the data length to be programmed.
  *         Bit0-1 must be zero.
  * @param  erase: Erase page flag before programming.
  * @retval The result:
  *           - 0: SUCCESS
  *           - 1: ERROR
  */
__STATIC_INLINE__ uint32_t md_msc_program_words_df(uint32_t addr, uint8_t *data, uint32_t len, uint32_t erase)
{
    uint32_t status;
    MD_MSC_WSP iap_wsp = (MD_MSC_WSP)(*(uint32_t *)MD_MSC_WSP_DF);

    if ((addr & 0x3) || (len & 0x3))
        return 1;

    MD_IAP_Unlock();
    __disable_irq();
    status = (*iap_wsp)(addr, data, len, erase);
    __enable_irq();
    MD_IAP_Lock();

    return !status;
}
/**
  * @}
  */
/** @defgroup IAP_Public_Functions_Group2 MSC Control Functions
  * @{
  */
/**
  * @brief  Set flash read wait cycle
  * @param  cycle: Wait cycle, [0-15] sysclock
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_read_wait_cycle_set(uint8_t cycle)
{
    uint32_t tmp;

    tmp = MSC->MEMWAIT;
    MODIFY_REG(tmp, MSC_MEMWAIT_FLASH_W_MSK, (0x30U | cycle) << MSC_MEMWAIT_FLASH_W_POSS);
    MSC->MEMWAIT = tmp;
}

/**
  * @brief  Get flash read wait cycle
  * @retval Wait cycle, [0-15] sysclock
  */
__STATIC_INLINE__ uint32_t md_msc_flash_read_wait_cycle_get(void)
{
    return READ_BITS(MSC->MEMWAIT, MSC_MEMWAIT_FLASH_W_MSK, MSC_MEMWAIT_FLASH_W_POSS);
}

/**
  * @brief  Set sram read wait cycle
  * @param  cycle: Wait cycle, [0-3] sysclock
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_sram_wait_cycle_set(uint8_t cycle)
{
    MODIFY_REG(MSC->MEMWAIT, MSC_MEMWAIT_SRAM_W_MSK, (cycle | 0x30) << MSC_MEMWAIT_SRAM_W_POSS);
}

/**
  * @brief  Get sram read wait cycle
  * @retval Wait cycle, [0-3] sysclock
  */
__STATIC_INLINE__ uint32_t md_msc_sram_read_wait_cycle_get(void)
{
    return READ_BITS(MSC->MEMWAIT, MSC_MEMWAIT_SRAM_W_MSK, MSC_MEMWAIT_SRAM_W_POSS);
}

/**
  * @brief  Get Flash ECCR
  * @retval Flash ECCR value
  */
__STATIC_INLINE__ uint32_t md_msc_feccr_get(void)
{
    return READ_REG(MSC->FECCR);
}

/**
  * @brief  Clear Flash NVI flag
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_eccr_nvi_clear(void)
{
     SET_BIT(MSC->FECCR, MSC_FECCR_NMIFC_MSK);
}

/**
  * @brief  Clear Flash error interrupt flag
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_eccr_interrupt_clear(void)
{
     SET_BIT(MSC->FECCR, MSC_FECCR_IFC_MSK);
}

/**
  * @brief  Flash NVI interrupt enable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_eccr_nvi_enable(void)
{
     SET_BIT(MSC->FECCR, MSC_FECCR_NMIE_MSK);
}

/**
  * @brief  Flash NVI interrupt disable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_eccr_nvi_disable(void)
{
     CLEAR_BIT(MSC->FECCR, MSC_FECCR_NMIE_MSK);
}

/**
  * @brief  Flash error interrupt enable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_eccr_interrupt_enable(void)
{
     SET_BIT(MSC->FECCR, MSC_FECCR_IE_MSK);
}

/**
  * @brief  Flash error interrupt disable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_eccr_interrupt_disable(void)
{
     CLEAR_BIT(MSC->FECCR, MSC_FECCR_IE_MSK);
}

/**
  * @brief  Flash ecc enable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_ecc_enable(void)
{
     SET_BIT(MSC->FECCR, MSC_FECCR_ECEN_MSK);
}

/**
  * @brief  Flash ecc disable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_flash_ecc_disable(void)
{
     CLEAR_BIT(MSC->FECCR, MSC_FECCR_ECEN_MSK);
}

/**
  * @brief  Get Flash ecc status 
  * @retval None
  */
__STATIC_INLINE__ uint32_t md_msc_flash_ecc_stat_get(void)
{
    return READ_REG(MSC->FECSR);
}


/**
  * @brief  SRAM ecc enable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_sram_ecc_enable(void)
{
     SET_BIT(MSC->RPCSR, MSC_RPCSR_PEN_MSK);
}

/**
  * @brief  SRAM ecc disable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_sram_ecc_disable(void)
{
     CLEAR_BIT(MSC->RPCSR, MSC_RPCSR_PEN_MSK);
}

/**
  * @brief  SRAM ecc NVI interrupt enable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_sram_ecc_nvi_enable(void)
{
     SET_BIT(MSC->RPCSR, MSC_RPCSR_NMIE_MSK);
}

/**
  * @brief  SRAM ecc NVI interupt disable
  * @retval None
  */
__STATIC_INLINE__ void md_msc_sram_ecc_nvi_disable(void)
{
     CLEAR_BIT(MSC->RPCSR, MSC_RPCSR_NMIE_MSK);
}

/**
  * @brief  Clear SRAM ecc error NVI flag
  * @retval None
  */
__STATIC_INLINE__ void md_msc_sram_eccr_nvi_clear(void)
{
     SET_BIT(MSC->FECCR, MSC_RPCSR_NMIFC_MSK);
}

/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* __MD_MSC_H__ */
