/**
  *********************************************************************************
  *
  * @file    md_dac.h
  * @brief   Header file of DAC module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */


#ifndef __MD_DAC_H__
#define __MD_DAC_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"

/** @addtogroup Micro_Driver
  * @{
  */

/** @defgroup MD_DAC DAC
  * @brief DAC micro driver
  * @{
  */

/** @defgroup DAC_Pubulic_Types DAC Pubulic Types
  * @{
  */

/**
  * @brief DAC noise filter width
  */
typedef enum
{
    MD_DAC_NWBW_1   = 0U,  /**< DAC nwbw 1 */
    MD_DAC_NWBW_2   = 1U,  /**< DAC nwbw 2 */
    MD_DAC_NWBW_3   = 2U,  /**< DAC nwbw 3 */
    MD_DAC_NWBW_4   = 3U,  /**< DAC nwbw 4 */
    MD_DAC_NWBW_5   = 4U,  /**< DAC nwbw 5 */
    MD_DAC_NWBW_6   = 5U,  /**< DAC nwbw 6 */
    MD_DAC_NWBW_7   = 6U,  /**< DAC nwbw 7 */
    MD_DAC_NWBW_8   = 7U,  /**< DAC nwbw 8 */
    MD_DAC_NWBW_9   = 8U,  /**< DAC nwbw 9 */
    MD_DAC_NWBW_10  = 9U,  /**< DAC nwbw 10 */
    MD_DAC_NWBW_11  = 10U, /**< DAC nwbw 11 */
    MD_DAC_NWBW_12  = 11U, /**< DAC nwbw 12 */
} md_dac_nwbw_t;

/**
  * @brief DAC noise filter mode
  */
typedef enum
{
    MD_DAC_NWM_None      = 0U, /**< DAC no filter */
    MD_DAC_NWM_LFSR      = 1U, /**< DAC LFSR mode */
    MD_DAC_NWM_TRINOISE  = 2U, /**< DAC TRINOISE mode */
} md_dac_nwm_mode_t;

/**
  * @brief DAC pis input channel
  */
typedef enum
{
    MD_DAC_PIS_CH_0  = 0U,     /**< PIS channel 0 triggers DAC channel conversion */
    MD_DAC_PIS_CH_1  = 1U,     /**< PIS channel 1 triggers DAC channel conversion */
    MD_DAC_PIS_CH_2  = 2U,     /**< PIS channel 2 triggers DAC channel conversion */
    MD_DAC_PIS_CH_3  = 3U,     /**< PIS channel 3 triggers DAC channel conversion */
    MD_DAC_PIS_CH_4  = 4U,     /**< PIS channel 4 triggers DAC channel conversion */
    MD_DAC_PIS_CH_5  = 5U,     /**< PIS channel 5 triggers DAC channel conversion */
    MD_DAC_PIS_CH_6  = 6U,     /**< PIS channel 6 triggers DAC channel conversion */
    MD_DAC_PIS_CH_7  = 7U,     /**< PIS channel 7 triggers DAC channel conversion */
    MD_DAC_PIS_CH_8  = 8U,     /**< PIS channel 8 triggers DAC channel conversion */
    MD_DAC_PIS_CH_9  = 9U,     /**< PIS channel 9 triggers DAC channel conversion */
    MD_DAC_PIS_CH_10 = 10U,    /**< PIS channel 10 triggers DAC channel conversion */
    MD_DAC_PIS_CH_11 = 11U,    /**< PIS channel 11 triggers DAC channel conversion */
    MD_DAC_PIS_CH_12 = 12U,    /**< PIS channel 12 triggers DAC channel conversion */
    MD_DAC_PIS_CH_13 = 13U,    /**< PIS channel 13 triggers DAC channel conversion */
    MD_DAC_PIS_CH_14 = 14U,    /**< PIS channel 14 triggers DAC channel conversion */
    MD_DAC_PIS_CH_15 = 15U,    /**< PIS channel 15 triggers DAC channel conversion */
} md_dac_pis_ch_t;

/**
  * @brief DAC trigger mode
  */
typedef enum
{
    MD_DAC_WR_DBUF_TRG  = 0U,  /**< Write DAC value trigger output */
    MD_DAC_WR_BIT_TRG   = 1U,  /**< Set trigger bit to output */
} md_dac_soft_trig_t;

/**
  * @brief DAC output mode
  */
typedef enum
{
    MD_DAC_OUTPUT_NORMAL  = 0U,    /**< DAC NORMAL MODE OUTPUT */
    MD_DAC_OUTPUT_ENHANCE = 1U,    /**< DAC ENHANCED MODE OUTPUT */
} md_dac_out_mode_t;

/**
  * @brief DAC data_alignment
  */
typedef enum
{
    MD_DAC_ALIGN_R12  = 0U,    /**< DAC NORMAL MODE OUTPUT */
    MD_DAC_ALIGN_L12  = 1U,    /**< DAC ENHANCED MODE OUTPUT */
    MD_DAC_ALIGN_R8   = 2U,
} md_dac_data_align_t;

/**
  * @brief DAC init structure definition
  */
typedef struct
{
    md_dac_soft_trig_t  soft_trig;  /**< Trig mode */
    type_func_t  pis_trig;          /**< External PIS trigger */
    md_dac_pis_ch_t  pis_ch;        /**< External pis channel */
    md_dac_out_mode_t out_mode;     /**< Select output mode */
    type_func_t  output_op;         /**< Refresh interval select */
    type_func_t  output_gpio;       /**< DAC output to GPIO */
    md_dac_nwm_mode_t  noise_mode;  /**< Noise filter mode */
    md_dac_nwbw_t  noise_width;     /**< Noise filter width */
    md_dac_data_align_t align_mode; /**< DAC output data align mode */
} md_dac_init_t;
/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions DAC Public Functions
  * @{
  */
/** @defgroup MD_DAC_Public_Functions_Group1 CON
  * @{
  */
/**
  * @brief  Set noise width
  * @param  DACx DAC Instance
  * @param  wdth Noise width, see @ref md_dac_nwbw_t
  * @retval None
  */
__STATIC_INLINE void md_dac_set_noise_width(DAC_TypeDef *DACx, md_dac_nwbw_t wdth)
{
    MODIFY_REG(DACx->CON, DAC_CON_NWBW_MSK, wdth << DAC_CON_NWBW_POSS);
}

/**
  * @brief  Set noise mode
  * @param  DACx DAC Instance
  * @retval mod Noise mode, see @ref md_dac_nwm_mode_t
  */
__STATIC_INLINE void md_dac_get_diff_mode(DAC_TypeDef *DACx, md_dac_nwm_mode_t mod)
{
    MODIFY_REG(DACx->CON, DAC_CON_NWM_MSK, DAC_CON_NWM_POSS);
}

/**
  * @brief  Set external trigger source
  * @param  DACx DAC Instance
  * @param  sel External trigger source, see @ref md_dac_exts_t
  * @retval None
  */
__STATIC_INLINE void md_dac_set_external_trigger_pis(DAC_TypeDef *DACx, md_dac_pis_ch_t sel)
{
    MODIFY_REG(DACx->CON, DAC_CON_EXTS_MSK, sel << DAC_CON_EXTS_POSS);
}

/**
  * @brief  Enable external trigger
  * @param  DACx DAC Instance
  */
__STATIC_INLINE void md_dac_external_trig_enable(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->CON, DAC_CON_EXTEN_MSK);
}

/**
  * @brief  Disable external trigger
  * @param  DACx DAC Instance
  */
__STATIC_INLINE void md_dac_external_trig_disable(DAC_TypeDef *DACx)
{
    CLEAR_BIT(DACx->CON, DAC_CON_EXTEN_MSK);
}

/**
  * @brief  Enable software trigger
  * @param  DACx DAC Instance
  */
__STATIC_INLINE void md_dac_software_trig_enable(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->CON, DAC_CON_SWTEN_MSK);
}

/**
  * @brief  DMA transfer enable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_dma_enalbe(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->CON, DAC_CON_DMAEN_MSK);
}

/**
  * @brief  DMA transfer disable
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_dma_trig_disable(DAC_TypeDef *DACx)
{
    CLEAR_BIT(DACx->CON, DAC_CON_DMAEN_MSK);
}

/**
  * @brief  DAC normal output.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_normal_output(DAC_TypeDef *DACx)
{
    CLEAR_BIT(DACx->CON, DAC_CON_MSEL_MSK);
}

/**
  * @brief  DAC enchanced output.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_enchanced_output(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->CON, DAC_CON_MSEL_MSK);
}

/**
  * @brief  DAC output to GPIO enable.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_to_gpio_enable(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->CON, DAC_CON_DACOE_MSK);
}

/**
  * @brief  DAC output to GPIO disable.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_to_gpio_disable(DAC_TypeDef *DACx)
{
    CLEAR_BIT(DACx->CON, DAC_CON_DACOE_MSK);
}

/**
  * @brief  DAC output amplify enable.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_output_amp_enable(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->CON, DAC_CON_OPOFF_MSK);
}

/**
  * @brief  DAC output amplify disable.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_output_amp_disable(DAC_TypeDef *DACx)
{
    CLEAR_BIT(DACx->CON, DAC_CON_OPOFF_MSK);
}

/**
  * @brief  DAC enable.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_enable(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->CON, DAC_CON_DACEN_MSK);
}

/**
  * @brief  DAC disable.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_disable(DAC_TypeDef *DACx)
{
    CLEAR_BIT(DACx->CON, DAC_CON_DACEN_MSK);
}

/**
  * @brief  DAC software reset.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_software_reset(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->SWTRG, DAC_SWTRG_RST_MSK);
}

/**
  * @brief  DAC software trigger.
  * @param  DACx DAC Instance
  * @retval None
  */
__STATIC_INLINE void md_dac_software_trigger(DAC_TypeDef *DACx)
{
    SET_BIT(DACx->SWTRG, DAC_SWTRG_TRG_MSK);
}

/**
  * @brief  Set DAC right-aligned 12bit value
  * @param  DACx DAC Instance
  * @param  val Value of the dac convertion
  * @retval None
  */
__STATIC_INLINE void md_dac_set_conv_r12value(DAC_TypeDef *DACx, uint16_t val)
{
    MODIFY_REG(DACx->R12BUF, DAC_R12BUF_VAL_MSK, val << DAC_R12BUF_VAL_POSS);
}

/**
  * @brief  Set DAC output left-aligned 12bit value
  * @param  DACx DAC Instance
  * @param  val Value of the dac convertion
  * @retval None
  */
__STATIC_INLINE void md_dac_set_conv_l12value(DAC_TypeDef *DACx, uint16_t val)
{
    MODIFY_REG(DACx->L12BUF, DAC_L12BUF_VAL_MSK, val << DAC_L12BUF_VAL_POSS);
}

/**
  * @brief  Set DAC right-aligned 8bit value
  * @param  DACx DAC Instance
  * @param  val Value of the dac convertion
  * @retval None
  */
__STATIC_INLINE void md_dac_set_conv_r8value(DAC_TypeDef *DACx, uint16_t val)
{
    MODIFY_REG(DACx->R8BUF, DAC_R8BUF_VAL_MSK, val << DAC_R8BUF_VAL_POSS);
}

/**
  * @}
  */
/** @defgroup MD_DAC_Public_Functions_Group1 Initialization
  * @{
  */
extern void md_dac_struct_init(md_dac_init_t *init);
extern md_status_t md_dac_init(md_dac_init_t *init);
/**
  * @}
  */
/**
  * @}
  */

/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif
#endif
