/**
  *********************************************************************************
  *
  * @file    ald_cache.h
  * @brief   Header file of cache module driver.
  *
  * @version V1.0
  * @date    12 Mar 2024
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          12 Mar 2024     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#ifndef __MD_CACHE_H__
#define __MD_CACHE_H__

#ifdef __cplusplus
extern "C" {
#endif

#include "md_utils.h"


/** @addtogroup Micro_Driver
  * @{
  */
  
/** @addtogroup CACHE
  * @{
  */

/** @defgroup MD_CACHE_Public_Types CACHE Public Types
  * @{
  */
/**
  * @brief Cache statistic setting
  */
typedef enum
{
    MD_AUTO_INV = 0x0U,    /**< Automatic cache invalidate when cache enabled */
    MD_MAN_INV  = 0x1U,    /**< Manual cache invalidate mode */
} md_cache_invalidate_t;

/**
  * @brief Cache Power control setting
  */
typedef enum
{
    MD_AUTO_POWER = 0x0U,    /**< Automatic power when cache enabled */
    MD_MAN_POWER  = 0x1U,    /**< Manual power mode */
} md_cache_power_t;

/**
 * @}
 */

/**
  * @brief Cache configuration structure definition
  */
typedef struct
{
    type_func_t statistic;              /**< Cache statistics logic */
    type_func_t prefetch;               /**< Cache prefetch setting */
    md_cache_invalidate_t invlidate;    /**< Cache invalidate setting */
    md_cache_power_t power;             /**< Power control setting */
} md_cache_config_t;

/**
  * @brief Cache power and invalidating status
  */
typedef enum
{
    MD_CACHE_INV_STAT  = (1U << 2),     /**< Indicates if invalidation process is ongoing */
    MD_CACHE_POW_STAT  = (1U << 4),     /**< SRAM power acknowledges */
} md_inv_pow_stat_t;

/**
  * @brief Cache status
  */
typedef enum
{
    MD_CACHE_DISABLED  = 0U,     /**< Cache disabled */
    MD_CACHE_ENABLING  = 1U,     /**< Cache enabling */
    MD_CACHE_ENABLED   = 2U,     /**< Cache enabled */
    MD_CACHE_DISABLING = 3U,     /**< Cache disabling */
} md_cache_stat_t;

/**
  * @brief Cache interrupt 
  */
typedef enum
{
    MD_CACHE_IT_POW_ERR      = (1U << 0),   /**< Power error interrupt */
    MD_CACHE_IT_MAN_INV_ERR  = (1U << 1),   /**< Manual invalidation error interrupt */
} md_cache_it_t;

/**
  * @brief Cache interrupt flag
  */
typedef enum
{
    MD_CACHE_IF_POW_ERR      = (1U << 0),   /**< Power error indication */
    MD_CACHE_IF_MAN_INV_ERR  = (1U << 1),   /**< Manual invalidation error indication */
} md_cache_flag_t;
/**
 * @}
 */


/** @addtogroup MD_CACHE_Public_Functions  CACHE Public Functions
  * @{
  */
/**
  * @brief  Enable statistics logic.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_enable_statistic(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_STATISTIC_EN_MSK);
}

/**
  * @brief  Disable statistics logic.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_disable_statistic(void)
{
    CLEAR_BIT(CACHE->CCR, CACHE_CCR_STATISTIC_EN_MSK);
}

/**
  * @brief  Enable Cache Prefetch Setting.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_enable_prefetch(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_SET_PREFETCH_MSK);
}

/**
  * @brief  Disable Cache Prefetch Setting.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_disable_prefetch(void)
{
    CLEAR_BIT(CACHE->CCR, CACHE_CCR_SET_PREFETCH_MSK);
}

/**
  * @brief  Manual cache invalidate mode.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_enable_manual_invalidate(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_SET_MAN_INV_MSK);
}

/**
  * @brief  Automatic cache invalidate when cache enabled.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_auto_invalidate(void)
{
    CLEAR_BIT(CACHE->CCR, CACHE_CCR_SET_MAN_INV_MSK);
}

/**
  * @brief  Manual Power Control Setting.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_manual_power(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_SET_MAN_POW_MSK);
}

/**
  * @brief  Auto Power Control Setting.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_auto_power(void)
{
    CLEAR_BIT(CACHE->CCR, CACHE_CCR_SET_MAN_POW_MSK);
}

/**
  * @brief  Enable Manual SRAM power request.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_enable_manual_sram_power(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_POW_REQ_MSK);
}

/**
  * @brief  Disable Manual SRAM power request.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_disable_manual_sram_power(void)
{
    CLEAR_BIT(CACHE->CCR, CACHE_CCR_POW_REQ_MSK);
}

/**
  * @brief  Manual invalidate request.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_manual_invalidate_request(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_INV_REQ_MSK);
}

/**
  * @brief  Enable Cache.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_enable(void)
{
    SET_BIT(CACHE->CCR, CACHE_CCR_EN_MSK);
}

/**
  * @brief  Disable Cache.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_disable(void)
{
    CLEAR_BIT(CACHE->CCR, CACHE_CCR_EN_MSK);
}

/**
  * @brief  Read SRAM power acknowledges.
  * @param  None.
  * @retval Real-time registered value of RAMPWRUPACK port.
  */
__STATIC_INLINE uint32_t md_cache_get_sram_power_stat(void)
{
    return READ_BIT(CACHE->SR, CACHE_SR_POW_STAT_MSK);
}

/**
  * @brief  Read Invalidating Status.
  * @param  None.
  * @retval  Indicates if invalidation process is ongoing.
  */
__STATIC_INLINE uint32_t md_cache_get_invalidating_status(void)
{
    return READ_BIT(CACHE->SR, CACHE_SR_INV_STAT_MSK);
}

/**
  * @brief  Read Cache Status.
  * @param  None.
  * @retval  Cache status value:
  *             0: Cache disabled.
  *             1: Cache enabling.
  *             2: Cache enabled.
  *             3: Cache disabling.
  */
__STATIC_INLINE uint32_t md_cache_get_cache_status(void)
{
    return READ_BITS(CACHE->SR, CACHE_SR_CS_MSK, CACHE_SR_CS_POSS);
}

/**
  * @brief  Enable interrupt request on manual invalidation error indication.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_enable_manual_invalidate_error_interrupt(void)
{
    CLEAR_BIT(CACHE->IRQMASK, CACHE_IRQMASK_MAN_INV_ERR_MSK);
}

/**
  * @brief  Disable interrupt request on manual invalidation error indication.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_disable_manual_invalidate_error_interrupt(void)
{
    SET_BIT(CACHE->IRQMASK, CACHE_IRQMASK_MAN_INV_ERR_MSK);
}

/**
  * @brief  Enable interrupt request on Power Error indication.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_enable_power_error_interrupt(void)
{
    CLEAR_BIT(CACHE->IRQMASK, CACHE_IRQMASK_POW_ERR_MSK);
}

/**
  * @brief  Disable interrupt request on Power Error indication.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_disable_power_error_interrupt(void)
{
    SET_BIT(CACHE->IRQMASK, CACHE_IRQMASK_POW_ERR_MSK);
}

/**
  * @brief  Clear Manual invalidation error status.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_clear_manual_error_status(void)
{
    SET_BIT(CACHE->IRQMASK, CACHE_IRQSTAT_MAN_INV_ERR_MSK);
}

/**
  * @brief  Get Manual invalidation error status.
  * @param  None.
  * @retval Set when manual invalidation is requested meanwhile the cache is not disabled.
  */
__STATIC_INLINE void md_cache_get_manual_error_status(void)
{
    READ_BIT(CACHE->IRQMASK, CACHE_IRQSTAT_MAN_INV_ERR_MSK);
}

/**
  * @brief  Clear SRAM power error status.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_clear_power_error_status(void)
{
    SET_BIT(CACHE->IRQMASK, CACHE_IRQSTAT_POW_ERR_MSK);
}

/**
  * @brief  Get Manual invalidation error status.
  * @param  None.
  * @retval Manual power request de-asserted while cache is enabled and operating in manual power mode.
  */
__STATIC_INLINE void md_cache_get_power_error_status(void)
{
    READ_BIT(CACHE->IRQMASK, CACHE_IRQSTAT_POW_ERR_MSK);
}

/**
  * @brief  Get the number of cache hits during cache look up.
  * @param  None.
  * @retval Number of cache hits .
  */
__STATIC_INLINE uint32_t md_cache_get_hits_counts(void)
{
    return READ_REG(CACHE->CSHR);
}

/**
  * @brief  Clear the number of cache hits counter.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_clear_hits_counter(void)
{
    WRITE_REG(CACHE->CSHR, 0x0U);
}

/**
  * @brief  Get the number of cache misses during cache look up.
  * @param  None.
  * @retval Number of cache misses .
  */
__STATIC_INLINE uint32_t md_cache_get_miss_counts(void)
{
    return READ_REG(CACHE->CSMR);
}

/**
  * @brief  Clear the number of cache misses counter.
  * @param  None.
  * @retval None.
  */
__STATIC_INLINE void md_cache_clear_misses_counter(void)
{
    WRITE_REG(CACHE->CSMR, 0x0U);
}

/** @addtogroup CACHE_Public_Functions
  * @{
  */
extern void md_cache_init(md_cache_config_t* config);

/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
#ifdef __cplusplus
}
#endif

#endif /* __ALD_CACHE_H__ */
