/**
  *********************************************************************************
  *
  * @file    md_utils.c
  * @brief   UTILS module driver.
  *
  * @version V1.0
  * @date    19 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          19 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */
#include <string.h>
#include "md_conf.h"

/** @addtogroup Micro_Driver
  * @{
  */
/** @addtogroup MD_UTILS
  * @{
  */

/** @defgroup MD_UTILS_Private_Variables Private Variables
  * @{
  */
static __IO uint32_t __md_tick = 0;
/**
  * @}
  */
/** @defgroup MD_UTILS_Public_Functions  UTILS Public Functions
  * @{
  */
/**
  * @brief  This function configures the source of the time base.
  *         The time source is configured to have 1ms time base with a dedicated
  *         Tick interrupt priority.
  * @retval None
  */
void md_init_1ms_tick(void)
{
	/* Configure the SysTick IRQ */
	NVIC_SetPriority(SysTick_IRQn, 15);
	SysTick_Config(md_cmu_get_clock() / 1000);
	return;
}

/**
  * @brief  This function invoked by Systick ISR each 1ms.
  * @retval None
  */
__isr__ void md_inc_tick(void)
{
	++__md_tick;
	return;
}

/**
  * @brief  Provides a tick value in millisecond.
  * @retval tick value
  */
uint32_t md_get_tick(void)
{
	return __md_tick;
}

/**
  * @brief  This function provides accurate delay (in milliseconds) based
  *         on variable incremented.
  * @param  delay: specifies the delay time length, in milliseconds.
  * @retval None
  */
void md_delay_1ms(__IO uint32_t delay)
{
	uint32_t tick;

	tick  = md_get_tick();
	delay = delay == 0 ? 1 : delay;

	while ((md_get_tick() - tick) < delay)
		;
}

/**
  * @brief  This function provides accurate delay (in microseconds) based
  *         on variable incremented.
  * @param  delay: specifies the delay time length, in microseconds.
  * @retval None
  */
void md_delay_us(uint32_t n)
{
	uint32_t ticks = 0;
	uint32_t told = 0;
	uint32_t tnow = 0;
	uint32_t tcnt = 0;
	uint32_t load = 0;
       
	load = SysTick->LOAD;                
	ticks = n * (md_cmu_get_sys_clock() / 1000000);	
    
	tcnt = 0;
	told = SysTick->VAL;             

	while (1) {
		tnow = SysTick->VAL;    
		if (tnow != told) {
		if (tnow < told) {
			tcnt += told - tnow;    
		} else {
			tcnt += load - tnow + told;    
		}        
		told = tnow;

		if (tcnt >= ticks) 
			break;
        }  
    }
} 

/**
  * @brief  Configure interrupt.
  * @param  irq: Interrunpt type.
  * @param  preempt_prio: preempt priority(0-3).
  * @param  sub_prio: sub-priority(0-3).
  * @param  status: Status.
  *           @arg ENABLE
  *           @arg DISABLE
  * @retval None
  */
void md_mcu_irq_config(IRQn_Type irq, uint8_t preempt_prio, uint8_t sub_prio, type_func_t status)
{
	uint32_t pri;
	uint8_t sub_bw, pre_bw;
	uint8_t sub_mask = 0xF;

	if (status == ENABLE) {
		pre_bw     = 7 - (((SCB->AIRCR) >> 8) & 7);
		sub_bw     = 4 - pre_bw;
		sub_mask >>= pre_bw;

		pri  = preempt_prio << sub_bw;
		pri |= sub_prio & sub_mask;

		NVIC_SetPriority(irq, pri);
		NVIC_EnableIRQ(irq);
	}
	else {
		NVIC_DisableIRQ(irq);
	}

	return;
}

/**
  * @brief  Get the CPU ID.
  * @retval CPU ID.
  */
uint32_t md_mcu_get_cpu_id(void)
{
	return SCB->CPUID;
}

/**
  * @brief  Get the UID.
  * @param  buf: Pointer to UID, len: 12Bytes(96-bits)
  * @retval None
  */
void md_mcu_get_uid(uint8_t *buf)
{
	memcpy(&buf[0], (void *)MD_MCU_UID0_ADDR, 4);
	memcpy(&buf[4], (void *)MD_MCU_UID1_ADDR, 4);
	memcpy(&buf[8], (void *)MD_MCU_UID2_ADDR, 4);

	return;
}

/**
  * @brief  Get the CHIPID
  * @retval CHPID
  */
uint32_t md_mcu_get_chipid(void)
{
	return (uint32_t)*(uint32_t *)MD_MCU_CHIPID_ADDR;
}

/**
  * @brief  Bypass bootroom and set VR1_Ref 0xA
  * @retval None
  */
__weak void sys_config(void)
{
	uint32_t i = 0, tmp = 0;
	uint8_t err = 0, flag = 0;
	uint32_t inf014 = 0, inf0154 = 0, inf0244 = 0;
	uint8_t cnt = 4;

	uint32_t *inf0_addr = (uint32_t *)0x20003C00;
	/* read bootroom cfg register */
	inf014  = *((uint32_t *)(0x80000 + 56));
	/* read VR1_VREF register */
	inf0154 = *((uint32_t *)(0x80000 + 616));
	/* read Chip_v */
	inf0244 = *((uint32_t *)(0x80000 + 0x03D0));
	
	/* if D version ,do nothing */
	if (inf0244 == 0xFFFFFF44) return;

	if (inf0154 == 0xFFFFFFFF)
		while(1);
	
	/* if bypass bootroom */
	if ((0xFFFFFFFF != inf014)) { 
		/* change cfg_boot value = 0xffff */
		inf014 = 0xFFFFFFFF; 
		flag   = 0x1;
	}

	/* change CFG_VR1_VREF value, FLASH ref 0xA */
	tmp = (inf0154 >> 8) & 0xF;
	if (0xA != tmp) {
		inf0154 &= ~(0xF << 8);
		inf0154 |= (0xA << 8);
		inf0154 = (inf0154 & (0x0000FFFF)) | ((~(inf0154 & 0xFFFF)) << 16);
		flag = 0x1;
	}
	
	/* if flag reset, return */
	if (0x0 == flag)
		return;
	
	/* 0x80000, 256words,INFO0 value */
	for (i = 0; i < 256; i++)
		inf0_addr[i] = *((uint32_t *)(0x80000 + i * 4));

	/* refresh value */
	inf0_addr[14]  = inf014;
	inf0_addr[154] = inf0154;

	while(--cnt) {
		err = 0;
		/* unlock */
		*((volatile uint32_t *)(0x40080000)) = 0x55AA6996;
		*((volatile uint32_t *)(0x40080100)) = 0x5A962814;
		*((volatile uint32_t *)(0x40080100)) = 0xE7CB69A5;	

		/* erase */
		if (md_msc_erase_page(0x80000) == 0) {
			/* program 256*4bytes, info0 */
			if (md_msc_program_words(0x80000, (uint8_t *)inf0_addr, 1024, 0) == 0) {
				/* check */
				for (i = 0; i < 256; i++) {
					if (inf0_addr[i] != *((uint32_t *)(0x80000 + i * 4))) {
						err = 1;
						break;;
					}
				}
				if (err == 0) { 
					/* lock */
					*((volatile uint32_t *)(0x40080100)) = 0x123456;
					*((volatile uint32_t *)(0x40080100)) = 0x123456;
					*((volatile uint32_t *)(0x40080000)) = 0x123456;
					return;
				}
			}
			else {
				err = 1;
			}
		}
		else {
			err = 1;
		}
	}

	if (err) {
		md_msc_erase_page(0x80000);
		/* lock */
		*((volatile uint32_t *)(0x40080100)) = 0x123456;
		*((volatile uint32_t *)(0x40080100)) = 0x123456;
		*((volatile uint32_t *)(0x40080000)) = 0x123456;
		while(1);
	}
}

/**
  * @brief  ADC adjust parameter config
  * @retval None
  */
__weak void adc_config(void)
{
	uint32_t inf0176 = 0, inf0178 = 0;
	uint32_t inf0250 = 0, inf0251 = 0;
	uint32_t inf0242 = 0, i = 0;
	uint8_t flag = 0, err = 0;
	uint8_t cnt = 4;
	/* ram store inf0 1k buffer, 15k ~ 16k */
	uint32_t *inf0_addr = (uint32_t *)0x20003C00;

	/* Read ADC_GE */
	inf0242 = *((uint32_t *)(0x80000 + 968));

	if (0xF5230ADC == inf0242)  return;

	/* read Lot ID */
	inf0250 = *((uint32_t *)(0x80000 + 1000));
	inf0251 = *((uint32_t *)(0x80000 + 1004));
	inf0251 = (inf0251 & 0xFFFF0000) >> 16;

	/* read CFG_ADC0DA/CFG_ADC1DA */
	inf0176 = *((uint32_t *)(0x80000 + 704));
	inf0178 = *((uint32_t *)(0x80000 + 712));

	switch(inf0250) {
		case 0x45465537:
			if ((inf0251 == 0x3034) || (inf0251 == 0x3035))
				flag = 1;
			break;
		case 0x45503931:
			if ((inf0251 == 0x3732) || (inf0251 == 0x3734))
				flag = 1;
			break;
		case 0x45503935:
			if ((inf0251 == 0x3837) || (inf0251 == 0x3839))
				flag = 1;
			break;
		default:
			break;
	}

	if (!flag)  return;
        
	inf0176 ^= (0x1 << 15);
	inf0176  = (inf0176 & 0x0000FFFF) | ((~(inf0176 & 0xFFFF)) << 16);

	inf0178 ^= (0x1 << 15);
	inf0178  = (inf0178 & 0x0000FFFF) | ((~(inf0178 & 0xFFFF)) << 16);

	/* 0x80000, 256words,INFO0 value */
	for (i = 0; i < 256; i++)
		inf0_addr[i] = *((uint32_t *)(0x80000 + i * 4));

	inf0_addr[176] = inf0176;
	inf0_addr[178] = inf0178;
	inf0_addr[242] = 0xF5230ADC;

	while(--cnt) {
		err = 0;
		/* unlock */
		*((volatile uint32_t *)(0x40080000)) = 0x55AA6996;
		*((volatile uint32_t *)(0x40080100)) = 0x5A962814;
		*((volatile uint32_t *)(0x40080100)) = 0xE7CB69A5;	

		/* erase */
		if (md_msc_erase_page(0x80000) == 0) {
			/* program 256*4bytes, info0 */
			if (md_msc_program_words(0x80000, (uint8_t *)inf0_addr, 1024, 0) == 0) {
				/* check */
				for (i = 0; i < 256; i++) {
					if (inf0_addr[i] != *((uint32_t *)(0x80000 + i * 4))) {
						err = 1;
						break;;
					}
				}
				if (err == 0) { 
					/* lock */
					*((volatile uint32_t *)(0x40080100)) = 0x123456;
					*((volatile uint32_t *)(0x40080100)) = 0x123456;
					*((volatile uint32_t *)(0x40080000)) = 0x123456;
					return;
				}
			}
			else {
				err = 1;
			}
		}
		else {
			err = 1;
		}
	}

	if (err) {
		md_msc_erase_page(0x80000);
		/* lock */
		*((volatile uint32_t *)(0x40080100)) = 0x123456;
		*((volatile uint32_t *)(0x40080100)) = 0x123456;
		*((volatile uint32_t *)(0x40080000)) = 0x123456;
		while(1);
	}
}

/**
  * @}
  */
/**
  * @}
  */
/**
  * @}
  */
