/**
  *********************************************************************************
  *
  * @file    md_lcd.c
  * @brief   LCD module driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "md_lcd.h"

/** @addtogroup Micro_Driver
  * @{
  */

#ifdef MD_LCD

/** @addtogroup MD_LCD
  * @{
  */

/** @addtogroup MD_LCD_Public_Functions
  * @{
  */
/** @addtogroup MD_LCD_Public_Functions_Group1
  * @{
  */
/**
  * @brief Initializes the LCD Peripheral according to the specified parameters.
  * @note This function can be used only when the LCD is disabled.
  * @param lcd: LCD Instance.
  * @param init: Pointer to a lcd_handle_t structure that contains
  *                 the configuration information for the specified LCD.
  * @retval Status, see @ref md_status_t.
  */
md_status_t md_lcd_init(LCD_TypeDef *lcd, md_lcd_init_t *init)
{
	uint16_t delay = 0;

	MODIFY_REG(LCD->FCR, LCD_FCR_WFS_MSK, init->lcd_wfs << LCD_FCR_WFS_POS);
	MODIFY_REG(LCD->FCR, LCD_FCR_PRS_MSK, init->lcd_prs << LCD_FCR_PRS_POSS);
	for (delay = 0; delay < 3000; delay++);
	MODIFY_REG(LCD->FCR, LCD_FCR_DIV_MSK, init->lcd_div << LCD_FCR_DIV_POSS);
	MODIFY_REG(LCD->FCR, LCD_FCR_DEAD_MSK, init->lcd_dead << LCD_FCR_DEAD_POSS);
	MODIFY_REG(LCD->FCR, LCD_FCR_PON_MSK, init->lcd_pon << LCD_FCR_PON_POSS);
	MODIFY_REG(LCD->FCR, LCD_FCR_VGS_MSK, init->lcd_vgs << LCD_FCR_VGS_POSS);

	MODIFY_REG(LCD->CR, LCD_CR_DUTY_MSK, init->lcd_duty << LCD_CR_DUTY_POSS);
	MODIFY_REG(LCD->CR, LCD_CR_BIAS_MSK, init->lcd_bias << LCD_CR_BIAS_POSS);
	MODIFY_REG(LCD->CR, LCD_CR_VBUFHD_MSK, init->lcd_vbufhd << LCD_CR_VBUFHD_POS);
	MODIFY_REG(LCD->CR, LCD_CR_VBUFLD_MSK, init->lcd_vbufld << LCD_CR_VBUFLD_POS);
	MODIFY_REG(LCD->CR, LCD_CR_DSHD_MSK, init->lcd_dshd << LCD_CR_DSHD_POSS);
	MODIFY_REG(LCD->CR, LCD_CR_DSLD_MSK, init->lcd_dsld << LCD_CR_DSLD_POSS);
	MODIFY_REG(LCD->CR, LCD_CR_RESHD_MSK, init->lcd_reshd << LCD_CR_RESHD_POSS);
	MODIFY_REG(LCD->CR, LCD_CR_RESLD_MSK, init->lcd_resld << LCD_CR_RESLD_POSS);
	MODIFY_REG(LCD->CR, LCD_CR_VSEL_MSK, init->lcd_vsel << LCD_CR_VSEL_POSS);
	MODIFY_REG(LCD->CR, LCD_CR_VCHPS_MSK, init->lcd_vchps << LCD_CR_VCHPS_POSS);

	return MD_OK;
}

/**
  * @brief  Set each @ref md_lcd_init_t field to default value.
  * @param  init: Pointer to a @ref md_lcd_init_t structure
  *                        whose fields will be set to default values.
  * @retval None
  */
void md_lcd_struct_init(md_lcd_init_t *init)
{
	init->lcd_vsel   = MD_LCD_VSEL_VDD;
	init->lcd_vchps  = MD_LCD_VCHPS_3V8;
	init->lcd_vbufhd = ENABLE;
	init->lcd_vbufld = DISABLE;
	init->lcd_dsld   = 0x0;
	init->lcd_dshd   = 0x0;
	init->lcd_resld  = MD_LCD_RES_3MOHM;
	init->lcd_reshd  = MD_LCD_RES_3MOHM;
	init->lcd_bias   = MD_LCD_BIAS_1_4;
	init->lcd_duty   = MD_LCD_DUTY_1_6;    /* COM0 ~ COM5 */
	init->lcd_wfs    = MD_LCD_WAVE_A;
	init->lcd_prs    = MD_LCD_PRS_4;
	init->lcd_div    = MD_LCD_DIV_21;
	init->lcd_dead   = MD_LCD_DEAD_TIME_NONE;
	init->lcd_pon    = MD_LCD_PON_NONE;
	init->lcd_vgs    = MD_LCD_VGS_15;
}
/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
#endif

/**
  * @}
  */
