/**
  *********************************************************************************
  *
  * @file    md_dac.c
  * @brief   DAC module driver.
  *
  * @version V1.0
  * @date    20 Nov 2018
  * @author  AE Team
  * @note
  *          Change Logs:
  *          Date            Author          Notes
  *          20 Nov 2018     AE Team         The first version
  *
  * Copyright (C) Shanghai Eastsoft Microelectronics Co. Ltd. All rights reserved.
  *
  * SPDX-License-Identifier: Apache-2.0
  *
  * Licensed under the Apache License, Version 2.0 (the License); you may
  * not use this file except in compliance with the License.
  * You may obtain a copy of the License at
  *
  * www.apache.org/licenses/LICENSE-2.0
  *
  * Unless required by applicable law or agreed to in writing, software
  * distributed under the License is distributed on an AS IS BASIS, WITHOUT
  * WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
  * See the License for the specific language governing permissions and
  * limitations under the License.
  **********************************************************************************
  */

#include "md_conf.h"

/** @addtogroup Micro_Driver
  * @{
  */
#ifdef MD_DAC

/** @addtogroup MD_DAC
  * @{
  */

/** @addtogroup MD_DAC_Public_Functions
  * @{
  */
/** @addtogroup MD_DAC_Public_Functions_Group1
  * @{
  */
/**
  * @brief  Initializes the DAC peripheral.
  * @param  DACx: DAC Instance
  * @param  init: Pointer to a dac_handle_t structure that contains
  *         the configuration information for the specified DAC module.
  * @retval Status, see @ref md_status_t.
  */
md_status_t md_dac_init(DAC_TypeDef *DACx, md_dac_init_t *init)
{
	uint32_t tmp;

	CLEAR_BIT(DAC0->CH0CTRL, DAC_CH0CTRL_EN_MSK);
	CLEAR_BIT(DAC0->CH1CTRL, DAC_CH1CTRL_EN_MSK);

	MODIFY_REG(ADC0->CCR, ADC_CCR_VREFEN_MSK, 1 << ADC_CCR_VREFEN_POS);

	if (init->p_ref == MD_DAC_POS_REF_VREEFP_BUF || init->p_ref == MD_DAC_POS_REF_2V)
		SET_BIT(ADC0->CCR, (ADC_CCR_IREFEN_MSK | ADC_CCR_VRBUFEN_MSK | ADC_CCR_VCMBUFEN_MSK));

	MODIFY_REG(ADC0->CCR, ADC_CCR_VRNSEL_MSK, init->n_ref << ADC_CCR_VRNSEL_POS);
	MODIFY_REG(ADC0->CCR, ADC_CCR_VRPSEL_MSK, init->p_ref << ADC_CCR_VRPSEL_POSS);

	tmp = ((init->refresh << DAC_CON_RCYCLSEL_POSS) | (init->div << DAC_CON_PRES_POSS) |
	       (init->ch0_reset << DAC_CON_CH0PRESRST_POS) | (init->o_ctrl_pis << DAC_CON_OUTENPIS_POS) |
	       (init->out_mode << DAC_CON_OUTMD_POSS) | (init->conv_mode << DAC_CON_CONVMD_POSS) |
	       (init->sine << DAC_CON_SINEMD_POS) | (init->diff << DAC_CON_DIFEN_POS));
	DACx->CON = tmp;

	/* Automatic calibration */
	SET_BIT(DACx->CAL, DAC_CAL_SELF_CALEN_MSK);
	for (tmp = 0; tmp < 1000; ++tmp);
	CLEAR_BIT(DACx->CAL, DAC_CAL_SELF_CALEN_MSK);

	return MD_OK;
}

/**
  * @brief  Set each @ref md_dac_init_t field to default value.
  * @param  init: Pointer to a @ref md_dac_init_t structure
  *              whose fields will be set to default values.
  * @retval None
  */
void md_dac_struct_init(md_dac_init_t *init)
{
	init->conv_mode  = MD_DAC_CONV_MODE_CONTINUOUS;
	init->out_mode   = MD_DAC_OUTPUT_PIN;
	init->refresh    = MD_DAC_REFRESH_8;
	init->div        = MD_DAC_PRES_DIV_64;
	init->ch0_reset  = DISABLE;
	init->o_ctrl_pis = DISABLE;
	init->sine       = DISABLE;
	init->diff       = DISABLE;
	init->n_ref      = MD_DAC_NEG_REF_VSS;
	init->p_ref      = MD_DAC_POS_REF_VDD;
}

/**
 * @}
 */
/**
 * @}
 */
/**
 * @}
 */
#endif
/**
 * @}
 */
